/*
 * Copyright (c) 2004-2010 The Trustees of Indiana University and Indiana
 *                         University Research and Technology
 *                         Corporation.  All rights reserved.
 * Copyright (c) 2004-2005 The University of Tennessee and The University
 *                         of Tennessee Research Foundation.  All rights
 *                         reserved.
 * Copyright (c) 2004-2005 High Performance Computing Center Stuttgart,
 *                         University of Stuttgart.  All rights reserved.
 * Copyright (c) 2004-2005 The Regents of the University of California.
 *                         All rights reserved.
 * Copyright (c) 2008-2015 Cisco Systems, Inc.  All rights reserved.
 * Copyright (c) 2010-2015 Los Alamos National Security, LLC.
 *                         All rights reserved.
 * Copyright (c) 2013-2020 Intel, Inc.  All rights reserved.
 * Copyright (c) 2016-2019 Research Organization for Information Science
 *                         and Technology (RIST).  All rights reserved.
 * Copyright (c) 2021-2023 Triad National Security, LLC. All rights reserved.
 * Copyright (c) 2021-2026 Nanook Consulting  All rights reserved.
 * $COPYRIGHT$
 *
 * Additional copyrights may follow
 *
 * $HEADER$
 */

/** @file **/

#include "src/include/pmix_config.h"

#include "src/class/pmix_object.h"
#include "src/client/pmix_client_ops.h"
#include "src/common/pmix_attributes.h"
#include "src/hwloc/pmix_hwloc.h"
#include "src/mca/base/pmix_base.h"
#include "src/mca/base/pmix_mca_base_var.h"
#include "src/mca/bfrops/base/base.h"
#include "src/mca/gds/base/base.h"
#include "src/mca/pcompress/base/base.h"
#include "src/mca/pif/base/base.h"
#include "src/mca/pinstalldirs/base/base.h"
#include "src/mca/plog/base/base.h"
#include "src/mca/pnet/base/base.h"
#include "src/mca/preg/base/base.h"
#include "src/mca/psec/base/base.h"
#include "src/mca/ptl/base/base.h"
#include "src/threads/pmix_tsd.h"
#include "src/util/pmix_keyval_parse.h"
#include "src/util/pmix_output.h"
#include "src/util/pmix_show_help.h"
#include "src/util/pmix_net.h"
#include "src/runtime/pmix_init_util.h"
#include <event.h>

#include "src/runtime/pmix_progress_threads.h"
#include "src/runtime/pmix_rte.h"

extern int pmix_initialized;
extern bool pmix_init_called;

void pmix_rte_finalize(void)
{
    int i;
    pmix_notify_caddy_t *cd;
    pmix_iof_req_t *req;

    if (!pmix_init_called) {
        return;
    }

    /* stop the main progress thread - does not release
     * the event base so we don't break the components */
    (void) pmix_progress_thread_pause(NULL);

    /* release the attribute support trackers */
    pmix_release_registered_attrs();

    /* close plog */
    (void) pmix_mca_base_framework_close(&pmix_plog_base_framework);

    /* close preg */
    (void) pmix_mca_base_framework_close(&pmix_preg_base_framework);

    /* cleanup communications */
    (void) pmix_mca_base_framework_close(&pmix_ptl_base_framework);

    /* close the security framework */
    (void) pmix_mca_base_framework_close(&pmix_psec_base_framework);

    /* close bfrops */
    (void) pmix_mca_base_framework_close(&pmix_bfrops_base_framework);

    /* close compress */
    (void) pmix_mca_base_framework_close(&pmix_pcompress_base_framework);

    /* close GDS */
    (void) pmix_mca_base_framework_close(&pmix_gds_base_framework);

    /* Finalize the network helper subsystem. */
    (void)pmix_net_finalize();

    /* finalize the mca */
    /* Clear out all the registered MCA params */
    pmix_deregister_params();
    pmix_mca_base_var_finalize();

    /* keyval lex-based parser */
    pmix_util_keyval_parse_finalize();

    (void) pmix_mca_base_framework_close(&pmix_pinstalldirs_base_framework);
    (void) pmix_mca_base_framework_close(&pmix_pif_base_framework);
    (void) pmix_mca_base_close();

    /* finalize the show_help system */
    pmix_show_help_finalize();

    /* finalize the output system.  This has to come *after* the
       malloc code, as the malloc code needs to call into this, but
       the malloc code turning off doesn't affect pmix_output that
       much */
    pmix_output_finalize();

    /* clean out the globals */
    PMIX_RELEASE(pmix_globals.mypeer);
    PMIX_DESTRUCT(&pmix_globals.events);
    PMIX_LIST_DESTRUCT(&pmix_globals.cached_events);
    /* clear any notifications */
    for (i = 0; i < pmix_globals.max_events; i++) {
        pmix_hotel_checkout_and_return_occupant(&pmix_globals.notifications, i, (void **) &cd);
        if (NULL != cd) {
            PMIX_RELEASE(cd);
        }
    }
    PMIX_DESTRUCT(&pmix_globals.notifications);
    for (i = 0; i < pmix_globals.iof_requests.size; i++) {
        req = (pmix_iof_req_t *) pmix_pointer_array_get_item(&pmix_globals.iof_requests, i);
        if (NULL != req) {
            PMIX_RELEASE(req);
        }
    }
    PMIX_DESTRUCT(&pmix_globals.iof_requests);
    PMIX_LIST_DESTRUCT(&pmix_globals.stdin_targets);
    if (NULL != pmix_globals.hostname) {
        free(pmix_globals.hostname);
        pmix_globals.hostname = NULL;
    }
    if (NULL != pmix_globals.aliases) {
        PMIx_Argv_free(pmix_globals.aliases);
        pmix_globals.aliases = NULL;
    }
    PMIX_LIST_DESTRUCT(&pmix_globals.nspaces);
    PMIX_LIST_DESTRUCT(&pmix_client_globals.groups);
    PMIX_DESTRUCT(&pmix_globals.keyindex);
    free(pmix_globals.myidval.data.proc);

    // release the topology
    pmix_hwloc_finalize();

    /* now safe to release the event base */
    (void) pmix_progress_thread_finalize(NULL);

    for (i = 0; i < PMIX_VAR_DUMP_COLOR_KEY_COUNT; i++) {
        free(pmix_var_dump_color[i]);
        pmix_var_dump_color[i] = NULL;
    }
    pmix_tsd_keys_destruct();

    pmix_finalize_util();
    // release the event base
    pmix_progress_thread_stop(NULL);
}
