;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2013, 2014, 2015, 2016, 2022 Andreas Enge <andreas@enge.fr>
;;; Copyright © 2014 Eric Bavier <bavier@member.fsf.org>
;;; Copyright © 2015 Mark H Weaver <mhw@netris.org>
;;; Copyright © 2016 Roel Janssen <roel@gnu.org>
;;; Copyright © 2016, 2018, 2019, 2020, 2021 Efraim Flashner <efraim@flashner.co.il>
;;; Copyright © 2016 Federico Beffa <beffa@fbengineering.ch>
;;; Copyright © 2016 Thomas Danckaert <post@thomasdanckaert.be>
;;; Copyright © 2016-2023 Ricardo Wurmus <rekado@elephly.net>
;;; Copyright © 2017 Leo Famulari <leo@famulari.name>
;;; Copyright © 2017, 2020-2022 Marius Bakke <marius@gnu.org>
;;; Copyright © 2017, 2018, 2019, 2020 Tobias Geerinckx-Rice <me@tobias.gr>
;;; Copyright © 2018 Danny Milosavljevic <dannym+a@scratchpost.org>
;;; Copyright © 2018, 2020 Arun Isaac <arunisaac@systemreboot.net>
;;; Copyright © 2020 Vincent Legoll <vincent.legoll@gmail.com>
;;; Copyright © 2020, 2021 Paul Garlick <pgarlick@tourbillion-technology.com>
;;; Copyright © 2021, 2022 Maxim Cournoyer <maxim@guixotic.coop>
;;; Copyright © 2021-2025 Nicolas Goaziou <mail@nicolasgoaziou.fr>
;;; Copyright © 2021 Leo Le Bouter <lle-bout@zaclys.net>
;;; Copyright © 2021 Xinglu Chen <public@yoctocell.xyz>
;;; Copyright © 2021 Ivan Gankevich <i.gankevich@spbu.ru>
;;; Copyright © 2021 Julien Lepiller <julien@lepiller.eu>
;;; Copyright © 2021 Thiago Jung Bauermann <bauermann@kolabnow.com>
;;; Copyright © 2022 Simon South <simon@simonsouth.net>
;;; Copyright © 2022 Jack Hill <jackhill@jackhill.us>
;;; Copyright © 2022 Fabio Natali <me@fabionatali.com>
;;; Copyright © 2022 Philip McGrath <philip@philipmcgrath.com>
;;; Copyright © 2023 Thomas Albers Raviola <thomas@thomaslabs.org>
;;; Copyright © 2023 John Kehayias <john.kehayias@protonmail.com>
;;; Copyright © 2023 Dominik Delgado Steuter <d@delgado.nrw>
;;; Copyright © 2023 Timothy Sample <samplet@ngyro.com>
;;; Copyright © 2024, 2025 Janneke Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2025 Hugo Buddelmeijer <hugo@buddelmeijer.nl>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (gnu packages tex)
  #:use-module ((guix licenses) #:prefix license:)
  #:use-module (guix gexp)
  #:use-module (guix packages)
  #:use-module (guix download)
  #:use-module (guix build-system copy)
  #:use-module (guix build-system gnu)
  #:use-module (guix build-system perl)
  #:use-module (guix build-system pyproject)
  #:use-module (guix build-system python)
  #:use-module (guix build-system qt)
  #:use-module (guix build-system trivial)
  #:use-module (guix build-system texlive)
  #:use-module (guix utils)
  #:use-module (guix deprecation)
  #:use-module (guix gexp)
  #:use-module (guix git-download)
  #:use-module (guix svn-download)
  #:use-module (gnu packages)
  #:use-module (gnu packages algebra)
  #:use-module (gnu packages aspell)
  #:use-module (gnu packages autotools)
  #:use-module (gnu packages bash)
  #:use-module (gnu packages boost)
  #:use-module (gnu packages check)
  #:use-module (gnu packages compression)
  #:use-module (gnu packages cpp)
  #:use-module (gnu packages digest)
  #:use-module (gnu packages lisp)
  #:use-module (gnu packages fonts)
  #:use-module (gnu packages fontutils)
  #:use-module (gnu packages gd)
  #:use-module (gnu packages ghostscript)
  #:use-module (gnu packages graphviz)
  #:use-module (gnu packages groff)
  #:use-module (gnu packages gtk)
  #:use-module (gnu packages icu4c)
  #:use-module (gnu packages image)
  #:use-module (gnu packages java)
  #:use-module (gnu packages libreoffice)
  #:use-module (gnu packages lua)
  #:use-module (gnu packages maths)
  #:use-module (gnu packages multiprecision)
  #:use-module (gnu packages pdf)
  #:use-module (gnu packages perl)
  #:use-module (gnu packages perl-check)
  #:use-module (gnu packages pkg-config)
  #:use-module (gnu packages plotutils)
  #:use-module (gnu packages python)
  #:use-module (gnu packages python-xyz)
  #:use-module (gnu packages qt)
  #:use-module (gnu packages readline)
  #:use-module (gnu packages ruby)
  #:use-module (gnu packages ruby-xyz)
  #:use-module (gnu packages shells)
  #:use-module (gnu packages tcl)
  #:use-module (gnu packages time)
  #:use-module (gnu packages tls)
  #:use-module (gnu packages base)
  #:use-module (gnu packages gawk)
  #:use-module (gnu packages web)
  #:use-module (gnu packages xml)
  #:use-module (gnu packages xorg)
  #:use-module (gnu packages xdisorg)
  #:use-module (gnu packages texinfo)
  #:use-module (ice-9 ftw)
  #:use-module (ice-9 match)
  #:use-module ((srfi srfi-1) #:hide (zip)))

;;; Commentary:
;;;
;;; Guix provides two different TeX Live systems: one monolithic, the TEXLIVE
;;; package, and the other modular.  This module is about the latter.  It aims
;;; at being as faithful as possible to TeX Live distribution.  Yet, some of
;;; the packages defined here are Guix specific.  The following paragraphs
;;; describe them.
;;;
;;; Source for all TeX Live binaries is located in TEXLIVE-SOURCE, which is
;;; therefore the starting point of any TeX Live update.  This is first used
;;; to build TEXLIVE-LIBKPATHSEA and TEXLIVE-LIBPTEXENC.
;;;
;;; Both TEXLIVE-SOURCE and TEXLIVE-LIBKPATHSEA---which takes care of creating
;;; a search path for GUIX_TEXMF environment variable---are used to compile
;;; TEXLIVE-BIN.  In turn, TEXLIVE-BIN propagates TEXLIVE-SCRIPTS, which
;;; contains core scripts and related files, including "texlive.tldb"
;;; database.  TEXLIVE-BIN is a mandatory native input in the `texlive' build
;;; system.
;;;
;;; Then, the system builds its way towards regular TEXLIVE-LATEX-BIN package,
;;; which is a convenient native input---that can be ignored using
;;; `texlive-latex-bin?' keyword argument---for most TeX Live packages.  For
;;; obvious reasons, packages used to build TEXLIVE-LATEX-BIN must be set it
;;; to `#f'.
;;;
;;; The TEXLIVE-BIN package provides a few Web2C tools, Metafont and some TeX
;;; engines---i.e, TeX, pdfLaTeX, LuaTeX and LuaHBTeX.  The rest of the
;;; binaries are built in separate packages, from TEXLIVE-SOURCE, and possibly
;;; with the help of TEXLIVE-LIBKPATHSEA and TEXLIVE-LIBPTEXENC.  Those
;;; packages inherit their phases from TEXLIVE-BIN.  As a convention, the
;;; package named "texlive-NAME-bin" provides binaries for "texlive-NAME",
;;; TEXLIVE-LATEX-BIN being the obvious exception to this scheme.
;;;
;;; Unlike font map files, TeX formats are not built from a profile hook, as
;;; the process would be too time-consuming, e.g., when invoking "guix shell".
;;; Instead, those are generated when the corresponding package is built.  It
;;; is therefore not possible for the build system to take into consideration
;;; hyphenation rules installed after a given format has been built.  To work
;;; around this, all hyphenations rules are packed into
;;; TEXLIVE-HYPHEN-COMPLETE, and all formats, being built with it, include all
;;; rules right from the start.
;;;
;;; Any other "texlive-NAME" package matches "NAME" TeX Live package, as
;;; defined in the "texlive.tlpdb" database.
;;;
;;; The following piece of art illustrates the bootstrap process of the
;;; modular Guix TeX Live distribution.  All "texlive-" prefixes have been
;;; dropped for brevity.
;;;
;;;
;;;           ,-- libptexenc-----------------------.
;;;           |                                    |
;;;           |                                    |
;;;  source --|                    ,-- all "*-bin" minus latex-bin
;;;    |      |                    |   ___________________________
;;;    |      |                    |               |
;;;    |      `-- libkpathsea --.  |               |
;;;    |                        |--|    (inherit phases from bin)
;;;    `------------------------’  |
;;;                                |
;;;                                |
;;;                                `-- bin -- hyphen-complete, etc. -- latex-bin
;;;                                     |     __________________________________
;;;  scripts --------(propagated)-------’                     |
;;;                                                           |
;;;                                               (#:texlive-latex-bin? #f)
;;;
;;;
;;; Default font map files and ls-R database are updated in a profile hook
;;; (see `texlive-font-maps' in "profiles.scm").  However, this doesn't happen
;;; when building documentation for a package.  Consequently, this module also
;;; provides TEXLIVE-LOCAL-TREE function, which creates a TeX Live tree with
;;; font map files updates and a dedicated database.  It should be used
;;; exclusively for package definitions, as a native input.  It is possible to
;;; augment that tree with additional TeX Live packages.
;;;
;;;
;;; Notes about updating TeX Live distribution:
;;; -------------------------------------------
;;;
;;; Thanks to the `texlive' updater, the more tedious part of that task, i.e.,
;;; refreshing version strings, hashes, possibly locations and inputs, is
;;; automated.  Yet, the following additional steps are required, and they can
;;; keep one busy for a while:
;;;
;;; 1. First and foremost, run the updater and log the (verbose) output:
;;;
;;;      ./pre-inst-env guix refresh -t texlive -u > update.log 2>&1
;;;
;;;    This takes care of updating all binaries and all TeX Live packages,
;;;    barring "collections" and "schemes", in a single run.
;;;
;;; 2. Then delete packages absent from the new TeX Live version.  Those are
;;;    usually indicated by a failed update, or, more obviously, by their
;;;    outdated version in the module.
;;;
;;;    Since "collections" and "schemes" are not updated automatically, all
;;;    references to deleted packages must also be cleared from their
;;;    propagated inputs.
;;;
;;; 3. Conversely, import newly integrated packages by refreshing all
;;;    "collections" with, e.g.,
;;;
;;;       ./pre-inst-env guix import texlive -r collection-NAME >> \
;;;       new-packages.scm 2>&1
;;;
;;;    It is advisable to iterate the command above over the list of all their
;;;    names:
;;;
;;;      basic bibtexextra binextra context fontextra fontsrecommended
;;;      fontutils formatsextra games humanities langarabic langchinese
;;;      langcjk langcyrillic langczechslovak langenglish langeuropean
;;;      langfrench langgerman langgreek langitalian langjapanese langkorean
;;;      langother langpolish langportuguese langspanish latex latexextra
;;;      latexrecommended luatex mathscience metapost music pictures
;;;      plaingeneric pstricks publishers xetex
;;;
;;;    Propagated inputs of collections must be updated accordingly.
;;;
;;; 4. Handle inputs issues.  No updater is allowed to create an input field
;;;    in a package if it didn't exist beforehand.  `texlive' updater is no
;;;    exception; it cannot update, e.g., propagated inputs if the relative
;;;    package didn't propagate some inputs already.  Double check updater's
;;;    output, such issues are mentioned, and resolve them manually.
;;;
;;;    Likewise, look for `(list)' artifacts the updater may have scattered
;;;    throughout this module.  They usually appear when all elements have
;;;    been removed from an input field.  In this situation, for aesthetics,
;;;    the whole field can be suppressed.
;;;
;;; 5. Last, but not least, test and fix the new packages, progressively.  For
;;;    example, it is possible to consider the following packages as
;;;    milestones:
;;;
;;;      libkpathsea > bin > latex-bin > scheme-basic > scheme-small
;;;
;;;    Also, for good measure, it is advisable to check some common
;;;    "collections" such as "pictures" and (gulp!) "latexextra".
;;;
;;;    At this point, if all is fine, send the update to the ML and let the CI
;;;    sort the rest out.
;;;
;;;    Don't forget to pat yourself on the back!
;;;
;;;
;;; Code:

(define-public texlive-source
  (package
    (name "texlive-source")
    (version "2025.2")
    (source
     (origin
       ;; This could be written as a SVN reference, but with a multi-reference
       ;; the package can benefit from the `texlive' updater.
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (string-append %texlive-repository
                                 "tags/texlive-" version "/Build/source"))
             (revision 74725)
             (locations (list "./"))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "1wcldxrk6kv855nzbwp2j2qlkh64z6dxy3jpfdzas4wjp5rwhgli"))))
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "Source code for all TeX Live programs")
    (description
     "This package fetches the source for all TeX Live programs provided by
the TeX Live repository.  It is meant to be used as a source-only package; it
should not be installed in a profile.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-libkpathsea
  (package
    (name "texlive-libkpathsea")
    (version (package-version texlive-source))
    (source
     (origin
       (inherit (package-source texlive-source))
       (modules '((guix build utils)
                  (ice-9 ftw)))
       (snippet
        #~(let ((delete-other-directories
                 (lambda (root dirs)
                   (with-directory-excursion root
                     (for-each
                      delete-file-recursively
                      (scandir "."
                               (lambda (file)
                                 (and (not (member file (append '("." "..") dirs)))
                                      (eq? 'directory (stat:type (stat file)))))))))))
            (delete-other-directories "libs" '())
            (delete-other-directories "utils" '())
            (delete-other-directories "texk" '("kpathsea"))))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:out-of-source? #t
      #:configure-flags
      #~(list "--disable-static"
              "--disable-native-texlive-build"
              "--enable-shared"
              "--with-banner-add=/GNU Guix"
              "--disable-all-pkgs"
              "--enable-kpathsea")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'customize-texmf.cnf
            ;; The default "texmf.cnf" file is provided by this package.
            ;; Every variable of interest is set relatively to the GUIX_TEXMF
            ;; environment variable defined via a search path below.
            ;;
            ;; This phase must happen before the `configure' phase, because
            ;; the value of the TEXMFCNF variable is used at compile time to
            ;; generate "paths.h" file.
            (lambda _
              (substitute* "texk/kpathsea/texmf.cnf"
                (("^TEXMFROOT = .*") "TEXMFROOT = {$GUIX_TEXMF}/..\n")
                (("^TEXMFDIST = .*") "TEXMFDIST = {$GUIX_TEXMF}\n")
                ;; Use XDG recommendations for local variables.  Also ignore
                ;; system-wide cache, which is not writable; use local one
                ;; instead, i.e., "$XDG_CACHE_HOME/texliveYYYY/texmf-var/".
                (("^TEXMFVAR = ~/\\.")
                 (string-append "XDG_CACHE_HOME = ~/.cache\n"
                                "TEXMFVAR = $XDG_CACHE_HOME/"))
                (("^TEXMFCONFIG = ~/\\.")
                 (string-append "XDG_CONFIG_HOME = ~/.config\n"
                                "TEXMFCONFIG = $XDG_CONFIG_HOME/"))
                (("^TEXMFCACHE = .*") "TEXMFCACHE = $TEXMFVAR\n")
                ;; "ls-R" files are to be expected only in the TEXMFDIST
                ;; directories.  TEXMFLOCAL is not necessary for Guix, but
                ;; could be required anyway by external TeX installations.
                (("^TEXMF = .*")
                 "TEXMF = {$TEXMFCONFIG,$TEXMFVAR,$TEXMFHOME,!!TEXMFLOCAL,TEXMFSYSVAR,TEXMFSYSCONFIG,!!$TEXMFDIST}\n")
                (("^TEXMFDBS = .*") "TEXMFDBS = {!!$TEXMFLOCAL,!!$TEXMFDIST}\n")
                ;; Set TEXMFCNF.  Since earlier values of variables have
                ;; precedence over later ones, insert the desired value first.
                (("^TEXMFCNF =")
                 (string-append
                  "TEXMFCNF = " #$output "/share/texmf-dist/web2c\n"
                  "TEXMFCNF ="))
                ;; Help TeX finding fonts installed on the system.
                (("^OSFONTDIR = .*") "OSFONTDIR = {$XDG_DATA_DIRS}\n")
                ;; Don't truncate lines.
                (("^error_line = .*$") "error_line = 254\n")
                (("^half_error_line = .*$") "half_error_line = 238\n")
                (("^max_print_line = .*$") "max_print_line = 1000\n"))))
          (add-after 'unpack 'patch-directory-traversal
            ;; When ST_NLINK_TRICK is set, kpathsea attempts to avoid work
            ;; when searching files by assuming that a directory with exactly
            ;; two links has no sub-directories.  This assumption does not
            ;; hold in our case, so some directories with symlinked
            ;; sub-directories would not be traversed.
            (lambda _
              (substitute* "texk/kpathsea/config.h"
                (("#define ST_NLINK_TRICK") ""))))
          (replace 'install
            (lambda _
              (with-directory-excursion "texk/kpathsea"
                (invoke "make" "install"))))
          (add-after 'install 'remove-documentation
            ;; Documentation is provided by TEXLIVE-KPATHSEA, in a dedicated
            ;; "doc" output.  Remove duplicates.
            (lambda _
              (with-directory-excursion #$output
                (for-each delete-file-recursively
                          '("share/info" "share/man"))))))))
    (native-search-paths
     (list (search-path-specification
            (variable "GUIX_TEXMF")
            (files '("share/texmf-dist")))))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "Path searching library")
    (description
     "Kpathsea is a library whose purpose is to return a filename from a list
of user-specified directories similar to how shells look up executables.")
    (license license:lgpl2.1)))

(define-public texlive-libptexenc
  (package
    (name "texlive-libptexenc")
    (version (package-version texlive-source))
    (source
     (origin
       (inherit (package-source texlive-source))
       (modules '((guix build utils)
                  (ice-9 ftw)))
       (snippet
        #~(let ((delete-other-directories
                 (lambda (root dirs with-files?)
                   (with-directory-excursion root
                     (for-each
                      delete-file-recursively
                      (scandir
                       "."
                       (lambda (file)
                         (and (not (member file (append '("." "..") dirs)))
                              (or with-files?
                                  (eq? 'directory (stat:type (stat file))))))))))))
            (delete-other-directories "libs" '() #f)
            (delete-other-directories "utils" '() #f)
            ;; Tests require "texmf.cnf" to be present in the tree.  Also test
            ;; data is spread across multiple directories, which need to be
            ;; preserved.
            (delete-other-directories "texk"
                                      '("bibtex-x" "ptexenc" "kpathsea" "tests"
                                        "web2c")
                                      #f)
            (delete-other-directories "texk/web2c" '("tests") #t)
            (delete-other-directories "texk/bibtex-x" '("tests") #t)
            (with-directory-excursion "texk/kpathsea"
              (for-each
               delete-file-recursively
               (scandir "." (lambda (f)
                              (not (member f '("." ".." "texmf.cnf")))))))))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:out-of-source? #t
      #:configure-flags
      #~(list "--disable-static"
              "--disable-native-texlive-build"
              "--enable-shared"
              "--with-banner-add=/GNU Guix"
              "--disable-all-pkgs")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'install-missing-files
            ;; These two files are not installed (on purpose, see ChangeLog),
            ;; but are required nonetheless if this library is meant to be
            ;; used externally.
            (lambda _
              (with-directory-excursion "texk/ptexenc/ptexenc"
                (let ((inc (string-append #$output "/include/ptexenc")))
                  (for-each (lambda (f) (install-file f inc))
                            '("kanjicnv.h" "unicode-jp.h"))))))
          (replace 'check
            (lambda* (#:key tests? #:allow-other-keys)
              (when tests?
                (with-directory-excursion "texk/ptexenc"
                  (invoke "make" "check")))))
          (replace 'install
            (lambda* (#:key inputs #:allow-other-keys)
              (with-directory-excursion "texk/ptexenc"
                (invoke "make" "install")))))))
    (native-inputs (list pkg-config texlive-libkpathsea))
    (inputs (list libiconv))
    (home-page "http://tutimura.ath.cx/ptexlive/?ptexenc%2FDetails")
    (synopsis "Library for Japanese pTeX")
    (description
     "This package provides a library for Japanese pTeX and its surrounding tools.")
    (license license:gpl2)))

(define-public texlive-scripts
  (package
    (name "texlive-scripts")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (string-append %texlive-repository
                                        "tags/texlive-" version "/Master/"))
                    (revision 74725)
                    (locations
                     (list "texmf-dist/doc/man/man1/fmtutil-sys.1"
                           "texmf-dist/doc/man/man1/fmtutil-sys.man1.pdf"
                           "texmf-dist/doc/man/man1/fmtutil-user.1"
                           "texmf-dist/doc/man/man1/fmtutil-user.man1.pdf"
                           "texmf-dist/doc/man/man1/fmtutil.1"
                           "texmf-dist/doc/man/man1/fmtutil.man1.pdf"
                           "texmf-dist/doc/man/man1/install-tl.1"
                           "texmf-dist/doc/man/man1/install-tl.man1.pdf"
                           "texmf-dist/doc/man/man1/mktexfmt.1"
                           "texmf-dist/doc/man/man1/mktexfmt.man1.pdf"
                           "texmf-dist/doc/man/man1/mktexlsr.1"
                           "texmf-dist/doc/man/man1/mktexlsr.man1.pdf"
                           "texmf-dist/doc/man/man1/mktexmf.1"
                           "texmf-dist/doc/man/man1/mktexmf.man1.pdf"
                           "texmf-dist/doc/man/man1/mktexpk.1"
                           "texmf-dist/doc/man/man1/mktexpk.man1.pdf"
                           "texmf-dist/doc/man/man1/mktextfm.1"
                           "texmf-dist/doc/man/man1/mktextfm.man1.pdf"
                           "texmf-dist/doc/man/man1/texhash.1"
                           "texmf-dist/doc/man/man1/texhash.man1.pdf"
                           "texmf-dist/doc/man/man1/tlmgr.1"
                           "texmf-dist/doc/man/man1/tlmgr.man1.pdf"
                           "texmf-dist/doc/man/man1/updmap-sys.1"
                           "texmf-dist/doc/man/man1/updmap-sys.man1.pdf"
                           "texmf-dist/doc/man/man1/updmap-user.1"
                           "texmf-dist/doc/man/man1/updmap-user.man1.pdf"
                           "texmf-dist/doc/man/man1/updmap.1"
                           "texmf-dist/doc/man/man1/updmap.man1.pdf"
                           "texmf-dist/doc/man/man5/fmtutil.cnf.5"
                           "texmf-dist/doc/man/man5/fmtutil.cnf.man5.pdf"
                           "texmf-dist/doc/man/man5/updmap.cfg.5"
                           "texmf-dist/doc/man/man5/updmap.cfg.man5.pdf"
                           "texmf-dist/dvips/tetex/"
                           "texmf-dist/fonts/enc/dvips/tetex/"
                           "texmf-dist/fonts/map/dvips/tetex/"
                           "texmf-dist/scripts/texlive/fmtutil-sys.sh"
                           "texmf-dist/scripts/texlive/fmtutil-user.sh"
                           "texmf-dist/scripts/texlive/fmtutil.pl"
                           "texmf-dist/scripts/texlive/mktexlsr.pl"
                           "texmf-dist/scripts/texlive/mktexmf"
                           "texmf-dist/scripts/texlive/mktexpk"
                           "texmf-dist/scripts/texlive/mktextfm"
                           "texmf-dist/scripts/texlive/tlmgr.pl"
                           "texmf-dist/scripts/texlive/updmap-sys.sh"
                           "texmf-dist/scripts/texlive/updmap-user.sh"
                           "texmf-dist/scripts/texlive/updmap.pl"
                           "texmf-dist/web2c/fmtutil-hdr.cnf"
                           "texmf-dist/web2c/updmap-hdr.cfg"
                           "texmf-dist/web2c/updmap.cfg"
                           "tlpkg/TeXLive/"
                           "tlpkg/gpg/"
                           "tlpkg/installer/config.guess"
                           "tlpkg/installer/curl/curl-ca-bundle.crt"
                           "tlpkg/texlive.tlpdb"))))
              (file-name (git-file-name name version))
              (sha256
               "0i7pf13a2j6k6gkn56h97912gssv7z5im2s0zxygawb1pylahayx")))
    (outputs '("out" "doc"))
    (properties '((updater-extra-inputs "perl")))
    (build-system copy-build-system)
    (arguments
     (list
      #:imported-modules `(,@%copy-build-system-modules
                           (guix build texlive-build-system)
                           (guix build union))
      #:modules '((guix build copy-build-system)
                  ((guix build texlive-build-system) #:prefix tex:)
                  (guix build utils))
      #:install-plan
      #~'(("texmf-dist/dvips/"   "share/texmf-dist/dvips")
          ("texmf-dist/fonts/"   "share/texmf-dist/fonts")
          ("texmf-dist/scripts/" "share/texmf-dist/scripts")
          ("texmf-dist/web2c/"   "share/texmf-dist/web2c")
          ("tlpkg/"              "share/tlpkg"))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'patch-scripts
            (lambda _
              ;; First patch shell scripts with ".sh" extension.
              (with-directory-excursion "texmf-dist"
                ((assoc-ref tex:%standard-phases 'patch-shell-scripts)))
              ;; Then patch scripts without such extension.
              (let ((dirs (map (compose dirname which)
                               (list "awk" "cat" "grep" "sed"))))
                (substitute*
                    (find-files "texmf-dist/scripts/" "^mktex(mf|pk|tfm)$")
                  (("^version=" m)
                   (format #false "PATH=\"~{~a:~}$PATH\"; export PATH~%~a"
                           dirs m))))
              ;; Make sure that fmtutil can find its Perl modules.
              (substitute* "texmf-dist/scripts/texlive/fmtutil.pl"
                (("\\$TEXMFROOT/")
                 (string-append #$output "/share/")))
              ;; Likewise for updmap.pl.
              (substitute* "texmf-dist/scripts/texlive/updmap.pl"
                (("\\$TEXMFROOT/tlpkg")
                 (string-append #$output "/share/tlpkg")))
              ;; Likewise for the tlmgr.
              (substitute* "texmf-dist/scripts/texlive/tlmgr.pl"
                ((".*\\$::installerdir = \\$Master.*" all)
                 (format #f "  $Master = ~s;~%~a"
                         (string-append #$output "/share")
                         all)))))
          (add-after 'install 'install-doc
            (lambda _
              (let ((doc (string-append #$output:doc "/share/texmf-dist/doc")))
                (mkdir-p doc)
                (copy-recursively "texmf-dist/doc/" doc))))
          (add-after 'install-doc 'link-scripts
            (lambda* (#:key outputs #:allow-other-keys)
              (with-directory-excursion "texmf-dist"
                (apply (assoc-ref tex:%standard-phases 'link-scripts)
                       (list #:outputs outputs
                             #:link-scripts
                             (find-files "scripts"))))
              (with-directory-excursion (string-append #$output "/bin")
                (symlink "fmtutil" "mktexfmt")))))))
    (inputs (list perl))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "TeX Live infrastructure programs")
    (description
     "This package provides core TeX Live scripts such as @code{updmap},
@code{fmtutil}, and @code{tlmgr}.  It is automatically installed alongside
@code{texlive-bin}.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-hyphen-complete
  (package
    (name "texlive-hyphen-complete")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/dehyph-exptl/"
                                     "doc/generic/elhyphen/"
                                     "doc/generic/huhyphen/"
                                     "doc/generic/hyph-utf8/"
                                     "doc/generic/ukrhyph/"
                                     "doc/luatex/hyph-utf8/"
                                     "source/generic/hyph-utf8/"
                                     "source/generic/ruhyphen/"
                                     "source/luatex/hyph-utf8/"
                                     "tex/generic/config/"
                                     "tex/generic/dehyph-exptl/"
                                     "tex/generic/dehyph/"
                                     "tex/generic/hyph-utf8/"
                                     "tex/generic/hyphen/"
                                     "tex/generic/ruhyphen/"
                                     "tex/generic/ukrhyph/"
                                     "tex/luatex/hyph-utf8/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1zcqpc3qf3vgfqka3lzj1sfdc5cflfjsjdsn3y359rlrb20k9wms"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list
      #:texlive-latex-bin? #f
      #:tex-engine "tex"
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'remove-ruby-byebug-dependency
            ;; Avoid dependency on byebug to reduce package closure
            ;; significantly, see <https://issues.guix.gnu.org/55997>.
            (lambda _
              (substitute* "source/generic/hyph-utf8/lib/tex/hyphen/language.rb"
                (("require 'byebug'") ""))))
          (add-before 'build 'include-docstrip.tex
            (lambda* (#:key inputs native-inputs #:allow-other-keys)
              (let ((docstrip.tex
                     (search-input-file (or native-inputs inputs)
                                        "tex/latex/base/docstrip.tex")))
                (setenv "TEXINPUTS"
                        (string-append (dirname docstrip.tex) ":")))))
          (add-before 'build 'regenerate-converters
            (lambda _
              (let ((root (getcwd)))
                (for-each delete-file
                          (find-files "tex/generic/hyph-utf8/conversions/"))
                (with-directory-excursion "source/generic/hyph-utf8"
                  (substitute* "generate-converters.rb"
                    (("\\$path_root=File.*")
                     (string-append "$path_root=\"" root "\"\n"))
                    ;; Avoid error with newer Ruby.
                    (("#1\\{%") "#1{%%"))
                  (invoke "ruby" "generate-converters.rb")))))
          (add-before 'build 'regenerate-patterns
            (lambda* (#:key inputs #:allow-other-keys)
              (let* ((root (getcwd))
                     (hyph-utf8 (string-append root "/tex/generic/hyph-utf8"))
                     (loaders (string-append hyph-utf8 "/loadhyph"))
                     (patterns (string-append hyph-utf8 "/patterns/txt"))
                     (ptex (string-append hyph-utf8 "/patterns/ptex"))
                     (quote (string-append hyph-utf8 "/patterns/quote")))
                ;; Initial clean-up.  Some files are not generated and need to
                ;; be preserved.
                (for-each delete-file (find-files loaders))
                (let ((preserved (list "hyph-sr-cyrl.hyp.txt"
                                       "hyph-sr-cyrl.pat.txt"
                                       "hyph-de-1901.ec.tex"
                                       "hyph-de-1996.ec.tex"
                                       "hyph-ru.t2a.tex"
                                       "hyph-uk.t2a.tex"
                                       "hyph-zh-latn-pinyin.ec.tex")))
                  (for-each
                   (lambda (directory)
                     (for-each delete-file
                               (find-files directory
                                           (lambda (f _)
                                             (not (member (basename f)
                                                          preserved))))))
                   (list patterns ptex quote)))
                ;; Generate plain patterns.  Write to the local directory.
                ;; Install phase will take care of moving the files to the
                ;; output.
                (with-directory-excursion "source/generic/hyph-utf8/"
                  (substitute* "lib/tex/hyphen/path.rb"
                    (("^([[:blank:]]+)TeXROOT = .*" _ indent)
                     (string-append indent "TeXROOT = \"" root "\"\n")))
                  (substitute* "generate-plain-patterns.rb"
                    ;; Ruby 2 does not need this.
                    (("require 'unicode'") "")
                    (("File\\.join\\(PATH::TXT")
                     (string-append "File.join(\"" patterns "\""))
                    (("File\\.join\\(PATH::QUOTE")
                     (string-append "File.join(\"" quote "\"")))
                  (invoke "ruby" "generate-plain-patterns.rb")
                  ;; Build pattern loaders.
                  (substitute* "generate-pattern-loaders.rb"
                    (("File\\.join\\(PATH::LOADER")
                     (string-append "File.join(\"" loaders "\"")))
                  (invoke "ruby" "generate-pattern-loaders.rb")
                  ;; Build ptex patterns.
                  (substitute* "generate-ptex-patterns.rb"
                    (("File\\.join\\(PATH::PTEX")
                     (string-append "File.join(\"" ptex "\"")))
                  (invoke "ruby" "generate-ptex-patterns.rb"))))))))
    (native-inputs
     ;; TODO: Update to ruby@3 on next rebuild-cycle.
     (list ruby-2.7
           ruby-hydra-minimal/pinned
           ;; Build phase requires "docstrip.tex" from TEXLIVE-LATEX.
           ;; However, adding this package to native inputs would initiate
           ;; a circular dependency.  To work around this, use TEXLIVE-LATEX
           ;; source, then add "docstrip.tex" to TEXINPUTS before build.
           (package-source texlive-latex)
           texlive-tex))
    (home-page "https://ctan.org/pkg/hyph-utf8")
    (synopsis "Hyphenation patterns expressed in UTF-8")
    (description
     "Modern native UTF-8 engines such as XeTeX and LuaTeX need hyphenation
patterns in UTF-8 format, whereas older systems require hyphenation patterns
in the 8-bit encoding of the font in use (such encodings are codified in the
LaTeX scheme with names like OT1, T2A, TS1, OML, LY1, etc).  The present
package offers a collection of conversions of existing patterns to UTF-8
format, together with converters for use with 8-bit fonts in older systems.

This Guix-specific package provides hyphenation patterns for all languages
supported in TeX Live.  It is a strict super-set of code{hyphen-base} package
and should be preferred to it whenever a package would otherwise depend on
@code{hyph-utf8}.")
    ;; Individual files each have their own license.  Most of these files are
    ;; independent hyphenation patterns.
    (license
     (list license:asl2.0
           license:bsd-3
           license:cc0
           license:expat
           license:gpl2
           license:gpl2+
           license:gpl3+
           license:knuth
           license:lgpl2.1
           license:lgpl2.1+
           license:lgpl3+
           license:lppl
           license:lppl1.0+
           license:lppl1.2+
           license:lppl1.3
           license:lppl1.3+
           license:lppl1.3a+
           license:mpl1.1
           license:public-domain
           license:wtfpl2
           (license:fsf-free
            "/tex/generic/hyph-utf8/patterns/tex/hyph-eu.tex")
           (license:non-copyleft
            "file:///tex/generic/hyph-utf8/patterns/tex/hyph-bg.tex"
            "Ancestral BSD variant")
           (license:non-copyleft
            "file:///tex/generic/hyph-utf8/patterns/tex/hyph-en-us.tex"
            "FSF all permissive license")
           (license:non-copyleft
            "/tex/generic/hyph-utf8/patterns/tex/hyph-no.tex"
            "FSF All permissive license")
           (license:non-copyleft
            "file:///tex/generic/hyph-utf8/patterns/tex/hyph-sa.tex")))))

(define-public texlive-bin
  (package
    (name "texlive-bin")
    (version (package-version texlive-source))
    (source
     (origin
       (inherit (package-source texlive-source))
       (modules '((guix build utils)
                  (ice-9 ftw)))
       (snippet
        #~(let ((delete-other-directories
                 (lambda (root dirs)
                   (with-directory-excursion root
                     (for-each
                      delete-file-recursively
                      (scandir "."
                               (lambda (file)
                                 (and (not (member file
                                                   (append '("." "..") dirs)))
                                      (eq? 'directory
                                           (stat:type (stat file)))))))))))
            ;; Delete bundled software, except Lua which cannot easily be used
            ;; as an external dependency, pplib and xpdf which aren't
            ;; supported as system libraries (see m4/kpse-xpdf-flags.m4).
            ;;
            ;; XXX: Also keep zziplib because 32bit systems fail to build
            ;; using external library, with the following error:
            ;; "src/luazip.c:183:(.text+0x192): undefined reference to
            ;; `zzip_open_ext_io64'".
            (delete-other-directories
             "libs"
             '("lua53" "luajit" "pplib" "xpdf" "zziplib"))
            ;; Remove all packages: they are installed through their
            ;; respective regular TeX Live package.
            (delete-other-directories "utils" '())
            (delete-other-directories "texk" '("kpathsea" "tests" "web2c"))
            ;; Tests require the "texmf.cnf" file to still be present in the
            ;; tree.
            (with-directory-excursion "texk/kpathsea"
              (for-each
               delete-file-recursively
               (scandir "." (lambda (f)
                              (not (member f '("." ".." "texmf.cnf")))))))))))
    (build-system gnu-build-system)
    (arguments
     (list
      #:out-of-source? #t
      #:configure-flags
      #~(let ((kpathsea #$(this-package-input "texlive-libkpathsea")))
          (list "CFLAGS=-g -O2 -Wno-error=incompatible-pointer-types"
                "--with-banner-add=/GNU Guix"
                "--enable-shared"
                "--disable-native-texlive-build"
                "--disable-static"
                "--disable-linked-scripts"
                "--disable-all-pkgs"
                ;; XXX: Use bundled zziplib when building on 32bit.  Other
                ;; systems still use the system one.  See `snippet' field
                ;; above for details.
                #$@(if (target-64bit?)
                       '("--with-system-zziplib")
                       '("--with-system-zziplib=no"))
                ;; Help locating external kpathsea.  For some reason
                ;; PKG-CONFIG is unable to find it.
                "--with-system-kpathsea"
                (format #f "--with-kpathsea-includes=~a/include" kpathsea)
                (format #f "--with-kpathsea-libdir=~a/lib" kpathsea)
                "--enable-web2c"
                ;; Disable web2c parts provided in their own packages.  See
                ;; "texk/web2c/ac/web2c.ac" in source.
                "--disable-aleph"
                "--disable-eptex"
                "--disable-euptex"
                "--disable-hitex"
                "--disable-luajithbtex"
                "--disable-luajittex"
                "--disable-mflua"
                "--disable-mfluajit"
                "--disable-mp"
                "--disable-pmp"
                "--disable-ptex"
                "--disable-upmp"
                "--disable-uptex"
                "--disable-web-progs"
                "--disable-xetex"))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'locate-external-kpathsea
            ;; Despite our best efforts, the configure script below is not
            ;; able to find an external Kpathsea library.  Set include
            ;; directory when appropriate.
            (lambda _
              (when (file-exists? "texk/web2c/configure")
                (substitute* "texk/web2c/configure"
                  (("/usr/include /usr/local/include")
                   (string-append #$(this-package-input "texlive-libkpathsea")
                                  "/include"))))))
          ;; XXX: Fix a failing test where "kpsewhich" location is hard-coded
          ;; in the source tree.
          (add-after 'unpack 'fix-failing-test
            (lambda _
              (substitute* (find-files "texk/" "wcfname\\.test$")
                (("_kpsewhich=.*")
                 (format #f "_kpsewhich=~a/bin/kpsewhich\n"
                         #$(this-package-input "texlive-libkpathsea"))))))
          (add-after 'install 'remove-documentation
            ;; Documentation is provided by specific TeX Live packages, in
            ;; a dedicated "doc" output.  Ignore documentation generated when
            ;; building TEXLIVE-BIN and its derivatives.
            (lambda _
              (with-directory-excursion #$output
                (for-each (lambda (d)
                            (when (file-exists? d)
                              (delete-file-recursively d)))
                          '("share/info"
                            "share/man"
                            "share/texmf-dist/doc"))))))))
    (native-inputs (list perl pkg-config))
    (inputs
     (list fontconfig
           harfbuzz
           icu4c
           libpng
           libxt
           zlib
           zziplib))
    (propagated-inputs (list texlive-libkpathsea texlive-scripts))
    (synopsis "TeX Live, a package of the TeX typesetting system")
    (description
     "TeX Live provides a comprehensive TeX document production system.
It includes all the major TeX-related programs, macro packages, and fonts that
are free software, including support for many languages around the world.

This package contains some Web2C binaries and Metafont.  TeX engines, in
addition to TeX itself, are limited to LuaHBTeX, LuaTeX and pdfTeX.  For
a self-sufficient TeX installation, users are advised to add at least a TeX
Live collection or scheme package to their profile instead of this package.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))
    (home-page "https://www.tug.org/texlive/")))

;; This package must be located before `texlive-local-tree' in the module.
(define-public texlive-scheme-basic
  (package
    (name "texlive-scheme-basic")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs (list texlive-collection-basic texlive-collection-latex))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "Basic scheme (plain and latex)")
    (description
     "This is the basic TeX Live scheme: it is a small set of files sufficient
to typeset plain TeX or LaTeX documents in PostScript or PDF, using the
Computer Modern fonts.  This scheme corresponds to @code{collection-basic} and
@code{collection-latex}.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-scheme-bookpub
  (package
    (name "texlive-scheme-bookpub")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs
     (list texlive-barcodes
           texlive-biber
           texlive-biblatex
           texlive-bookcover
           texlive-caption
           texlive-collection-basic
           texlive-collection-latex
           texlive-enumitem
           texlive-fontspec
           texlive-latexmk
           texlive-lipsum
           texlive-listings
           texlive-markdown
           texlive-memoir
           texlive-microtype
           texlive-minted
           texlive-novel
           texlive-octavo
           texlive-pdfpages
           texlive-pgf
           texlive-qrcode
           texlive-shapes
           texlive-titlesec
           texlive-tocloft
           texlive-tufte-latex
           texlive-willowtreebook))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "Book publishing scheme (core LaTeX and add-ons)")
    (description
     "This is a book publishing scheme, containing core (Lua)LaTeX and
selected additional packages likely to be useful for non-technical book
publication.  It does not contain additional fonts (different books need
different fonts, and the packages are large), nor does it contain additional
mathematical or other technical packages.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-scheme-context
  (package
    (name "texlive-scheme-context")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs
     (list texlive-antt
           texlive-asana-math
           texlive-ccicons
           texlive-collection-context
           texlive-collection-metapost
           texlive-dejavu
           texlive-eulervm
           texlive-iwona
           texlive-kurier
           texlive-ly1
           texlive-manfnt-font
           texlive-marvosym
           texlive-mflogo-font
           texlive-poltawski
           texlive-pxfonts
           texlive-tex-gyre
           texlive-tex-gyre-math
           texlive-txfonts
           texlive-wasy
           texlive-xits))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "ConTeXt scheme")
    (description "This is the TeX Live scheme for installing ConTeXt.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-scheme-full
  (package
    (name "texlive-scheme-full")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs
     (list texlive-collection-basic
           texlive-collection-bibtexextra
           texlive-collection-binextra
           texlive-collection-context
           texlive-collection-fontsextra
           texlive-collection-fontsrecommended
           texlive-collection-fontutils
           texlive-collection-formatsextra
           texlive-collection-games
           texlive-collection-humanities
           texlive-collection-langarabic
           texlive-collection-langchinese
           texlive-collection-langcjk
           texlive-collection-langcyrillic
           texlive-collection-langczechslovak
           texlive-collection-langenglish
           texlive-collection-langeuropean
           texlive-collection-langfrench
           texlive-collection-langgerman
           texlive-collection-langgreek
           texlive-collection-langitalian
           texlive-collection-langjapanese
           texlive-collection-langkorean
           texlive-collection-langother
           texlive-collection-langpolish
           texlive-collection-langportuguese
           texlive-collection-langspanish
           texlive-collection-latex
           texlive-collection-latexextra
           texlive-collection-latexrecommended
           texlive-collection-luatex
           texlive-collection-mathscience
           texlive-collection-metapost
           texlive-collection-music
           texlive-collection-pictures
           texlive-collection-plaingeneric
           texlive-collection-pstricks
           texlive-collection-publishers
           texlive-collection-xetex))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "TeX Live full scheme (everything)")
    (description
     "This is the full TeX Live scheme: it installs everything available.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-scheme-gust
  (package
    (name "texlive-scheme-gust")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs
     (list texlive-amslatex-primer
           texlive-amstex
           texlive-antt
           texlive-bibtex8
           texlive-collection-basic
           texlive-collection-context
           texlive-collection-fontsrecommended
           texlive-collection-fontutils
           texlive-collection-langpolish
           texlive-collection-latex
           texlive-collection-latexrecommended
           texlive-collection-metapost
           texlive-collection-plaingeneric
           texlive-collection-xetex
           texlive-comment
           texlive-comprehensive
           texlive-concrete
           texlive-cyklop
           texlive-dvidvi
           texlive-dviljk
           texlive-fontinstallationguide
           texlive-gustprog
           texlive-impatient
           texlive-iwona
           texlive-metafont-beginners
           texlive-metapost-examples
           texlive-poltawski
           texlive-seetexk
           texlive-seminar
           texlive-tds
           texlive-tex4ht
           texlive-texdoc))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "GUST TeX Live scheme")
    (description
     "This is the GUST TeX Live scheme: it is a set of files sufficient to
typeset Polish plain TeX, LaTeX and ConTeXt documents in PostScript or PDF.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-scheme-medium
  (package
    (name "texlive-scheme-medium")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs
     (list texlive-collection-basic
           texlive-collection-binextra
           texlive-collection-context
           texlive-collection-fontsrecommended
           texlive-collection-fontutils
           texlive-collection-langczechslovak
           texlive-collection-langenglish
           texlive-collection-langeuropean
           texlive-collection-langfrench
           texlive-collection-langgerman
           texlive-collection-langitalian
           texlive-collection-langpolish
           texlive-collection-langportuguese
           texlive-collection-langspanish
           texlive-collection-latex
           texlive-collection-latexrecommended
           texlive-collection-luatex
           texlive-collection-mathscience
           texlive-collection-metapost
           texlive-collection-plaingeneric
           texlive-collection-xetex))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "Small scheme with additional packages and languages")
    (description
     "This is the medium TeX Live collection: it contains plain TeX,
LaTeX, many recommended packages, and support for most European languages.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-scheme-minimal
  (package
    (name "texlive-scheme-minimal")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs (list texlive-collection-basic))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "Minimal scheme (plain only)")
    (description
     "This is the minimal TeX Live scheme, with support for only plain
TeX. (No LaTeX macros.) LuaTeX is included because Lua scripts are used in TeX
Live infrastructure.  This scheme corresponds exactly to
@code{collection-basic}.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-scheme-small
  (package
    (name "texlive-scheme-small")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs
     (list texlive-babel-basque
           texlive-babel-czech
           texlive-babel-danish
           texlive-babel-dutch
           texlive-babel-english
           texlive-babel-finnish
           texlive-babel-french
           texlive-babel-german
           texlive-babel-hungarian
           texlive-babel-italian
           texlive-babel-norsk
           texlive-babel-polish
           texlive-babel-portuges
           texlive-babel-spanish
           texlive-babel-swedish
           texlive-collection-basic
           texlive-collection-latex
           texlive-collection-latexrecommended
           texlive-collection-metapost
           texlive-collection-xetex
           texlive-ec
           texlive-eurosym
           texlive-hyphen-complete
           texlive-lm
           texlive-lualibs
           texlive-luaotfload
           texlive-luatexbase
           texlive-revtex
           texlive-synctex
           texlive-times
           texlive-tipa
           texlive-ulem
           texlive-upquote
           texlive-zapfding))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "Small scheme (basic, XeTeX, Metapost, and a few languages)")
    (description
     "This is a small TeX Live scheme, corresponding to MacTeX's BasicTeX
variant.  It adds XeTeX, MetaPost, and some recommended packages to
@code{scheme-basic}.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

(define-public texlive-scheme-tetex
  (package
    (name "texlive-scheme-tetex")
    (version (package-version texlive-source))
    (source #f)
    (build-system trivial-build-system)
    (arguments (list #:builder #~(mkdir #$output)))
    (propagated-inputs
     (list texlive-acronym
           texlive-amslatex-primer
           texlive-bbm
           texlive-bbm-macros
           texlive-bbold
           texlive-bibtex8
           texlive-cmbright
           texlive-collection-basic
           texlive-collection-context
           texlive-collection-fontsrecommended
           texlive-collection-fontutils
           texlive-collection-formatsextra
           texlive-collection-langcjk
           texlive-collection-langcyrillic
           texlive-collection-langczechslovak
           texlive-collection-langenglish
           texlive-collection-langeuropean
           texlive-collection-langfrench
           texlive-collection-langgerman
           texlive-collection-langgreek
           texlive-collection-langitalian
           texlive-collection-langother
           texlive-collection-langpolish
           texlive-collection-langportuguese
           texlive-collection-langspanish
           texlive-collection-latex
           texlive-collection-latexrecommended
           texlive-collection-mathscience
           texlive-collection-metapost
           texlive-collection-pictures
           texlive-collection-plaingeneric
           texlive-collection-pstricks
           texlive-ctie
           texlive-cweb
           texlive-detex
           texlive-dtl
           texlive-dvi2tty
           texlive-dvicopy
           texlive-dvidvi
           texlive-dviljk
           texlive-eplain
           texlive-eulervm
           texlive-gentle
           texlive-lshort-english
           texlive-mltex
           texlive-multirow
           texlive-nomencl
           texlive-patgen
           texlive-pst-pdf
           texlive-rsfs
           texlive-seetexk
           texlive-siunits
           texlive-subfigure
           texlive-supertabular
           texlive-tamethebeast
           texlive-tds
           texlive-tie
           texlive-web
           texlive-xpdfopen))
    (home-page "https://www.tug.org/texlive/")
    (synopsis "teTeX scheme (more than medium, but nowhere near full)")
    (description
     "Larger than medium, this TeX Live scheme is nearly equivalent to the
teTeX distribution that was maintained by Thomas Esser.")
    (license (license:fsf-free "https://www.tug.org/texlive/copying.html"))))

;; This package must be located before any package adding it to its native
;; inputs.
(define-public texlive-local-tree
  (lambda* (#:optional (packages '()))
    "Return a 'texlive-local-tree' package which contains the fonts map and
mktex scripts configuration, along with a base set of packages plus additional
PACKAGES.

This function is meant to be used in packages as a native input, to build
documentation in the TeX format."
    (let ((default-packages (list texlive-scheme-basic)))
      (package
        (version (package-version texlive-source))
        (source (package-source texlive-scripts))
        (name "texlive-local-tree")
        (build-system copy-build-system)
        (arguments
         (list
          #:imported-modules `(,@%copy-build-system-modules
                               (guix build union))
          #:modules '((guix build copy-build-system)
                      (guix build union)
                      (guix build utils)
                      (ice-9 match)
                      (ice-9 popen)
                      (ice-9 textual-ports)
                      (srfi srfi-1))
          #:install-plan
          #~'(("texmf-dist/web2c/updmap.cfg" "share/texmf-dist/web2c/"))
          #:phases
          #~(modify-phases %standard-phases
              (add-before 'unpack 'initialize-tree
                (lambda* (#:key inputs #:allow-other-keys)
                  ;; Build complete TeX Live tree in #$output, barring the
                  ;; files going to be regenerated.
                  (let ((texlive-outputs
                         (filter-map
                          (match-lambda
                            (`(,label . ,dir)
                             (and (string-prefix? "texlive-" label)
                                  dir)))
                          inputs)))
                    (union-build #$output texlive-outputs
                                 #:create-all-directories? #t
                                 #:log-port (%make-void-port "w")))
                  ;; Remove files that are going to be regenerated.
                  (with-directory-excursion
                      (string-append #$output "/share/texmf-dist")
                    (for-each (lambda (file)
                                (when (file-exists? file) (delete-file file)))
                              (list "fonts/map/dvipdfmx/updmap/kanjix.map"
                                    "fonts/map/dvips/updmap/builtin35.map"
                                    "fonts/map/dvips/updmap/download35.map"
                                    "fonts/map/dvips/updmap/ps2pk.map"
                                    "fonts/map/dvips/updmap/psfonts.map"
                                    "fonts/map/dvips/updmap/psfonts_pk.map"
                                    "fonts/map/dvips/updmap/psfonts_t1.map"
                                    "fonts/map/pdftex/updmap/pdftex.map"
                                    "fonts/map/pdftex/updmap/pdftex_dl14.map"
                                    "fonts/map/pdftex/updmap/pdftex_ndl14.map"
                                    "web2c/updmap.cfg")))))
              (add-before 'install 'regenerate-updmap.cfg
                (lambda _
                  ;; Set TEXMFSYSVAR to a sane and writable value; updmap
                  ;; fails if it cannot create its log file.
                  (setenv "TEXMFSYSVAR" (getcwd))
                  ;; Limit scope of TEXMF to the current tree, and skip ls-R
                  ;; database usage since it has not been generated yet.
                  (setenv "TEXMF" "{$TEXMFSYSVAR,$TEXMFDIST}")

                  (with-directory-excursion "texmf-dist/web2c"
                    ;; Disable unavailable map files.
                    (let ((port (open-pipe* OPEN_WRITE "updmap-sys"
                                            "--syncwithtrees"
                                            "--nohash"
                                            "--cnffile" "updmap.cfg")))
                      (display "Y\n" port)
                      (when (not (zero? (status:exit-val (close-pipe port))))
                        (error "failed to filter updmap.cfg")))
                    ;; Generate maps.
                    (let ((root (string-append #$output
                                               "/share/texmf-dist/fonts/map/")))
                      (invoke "updmap-sys"
                              "--cnffile" "updmap.cfg"
                              "--dvipdfmxoutputdir"
                              (string-append root "dvipdfmx/updmap/")
                              "--dvipsoutputdir"
                              (string-append root "dvips/updmap/")
                              "--pdftexoutputdir"
                              (string-append root "pdftex/updmap/"))))))
              (add-after 'regenerate-updmap.cfg 'regenerate-ls-R
                (lambda _
                  ;; Generate ls-R database for local tree.  Unfortunately,
                  ;; "mktexlsr" doesn't preserve alphabetic order, probably
                  ;; because it is used on symlinks.  Use a lower level
                  ;; equivalent of that command.
                  (with-directory-excursion
                      (string-append #$output "/share/texmf-dist")
                    (with-output-to-file "ls-R"
                      (lambda ()
                        (invoke "ls" "-1LAR" "./")))))))))
        (native-inputs (list texlive-scripts))
        (inputs (map (lambda (package)
                       (list (package-name package) package))
                     (append default-packages packages)))
        ;; Propagate libkpathsea in order to populate GUIX_TEXMF when
        ;; building the package using this one as an input.
        (propagated-inputs (list texlive-libkpathsea))
        (home-page (package-home-page texlive-bin))
        (synopsis "TeX Live autonomous tree")
        (description
         "This package contains an autonomous TeX Live consisting of base
packages as well as, optionally, user-provided ones.  It is meant to be added
as a package native input, in order to build TeX documentation.")
        (license (delete-duplicates
                  (fold (lambda (package result)
                          (match (package-license package)
                            ((lst ...)
                             (append lst result))
                            ((? license:license? license)
                             (cons license result))))
                        '()
                        (append default-packages packages))))))))

(define-deprecated/public-alias texlive-updmap.cfg texlive-local-tree)

(define-public texlive-12many
  (package
    (name "texlive-12many")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/12many/" "source/latex/12many/"
                                     "tex/latex/12many/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "104vqgl5kqdbwfa5wxj7gpxfrqa3k4ky4806mg6zscd6palf2gi5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/one2many")
    (synopsis "Generalising mathematical index sets")
    (description
     "In the discrete branches of mathematics and the computer sciences, it
will only take some seconds before you're faced with a set like
@samp{@{1,...,m@}}.  Some people write @samp{$1\\ldotp\\ldotp m$}, others
@samp{$\\{j:1\\leq j\\leq m\\}$}, and the journal you're submitting to might
want something else entirely.  The @code{12many} package provides an interface
that makes changing from one to another a one-line change.")
    (license license:lppl)))

(define-public texlive-a2ping
  (package
    (name "texlive-a2ping")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/a2ping.1"
                                     "doc/man/man1/a2ping.man1.pdf"
                                     "doc/support/a2ping/" "scripts/a2ping/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1lhmh1rnykmi6i5mklj7fwhndw1xxmwy2xsy8j1px8ishf5czhvq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "a2ping.pl")))
    (inputs (list perl))
    (home-page "https://ctan.org/pkg/a2ping")
    (synopsis "Advanced PS, PDF, EPS converter")
    (description
     "@command{a2ping} is a Perl script command line utility written for Unix
that converts many raster image and vector graphics formats to EPS or PDF and
other page description formats.  Accepted input file formats are:
PS (PostScript), EPS, PDF, PNG, JPEG, TIFF, PNM, BMP, GIF, LBM, XPM, PCX,
TGA. Accepted output formats are: EPS, PCL5, PDF, PDF1, PBM, PGM, PPM, PS,
markedEPS, markedPS, PNG, XWD, BMP, TIFF, JPEG, GIF, XPM.")
    (license license:gpl3+)))

(define-public texlive-aaai-named
  (package
    (name "texlive-aaai-named")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/aaai-named/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xs2iki0v1mh0kbqjxnv9r24bcfmgj6myfw7h368h8g0s1n8d5v6"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aaai-named")
    (synopsis "BibTeX style for AAAI")
    (description
     "This package provides a BibTeX style derived from the standard master,
presumably for use with the @code{aaai} package.")
    (license license:knuth)))

(define-public texlive-aalok
  (package
    (name "texlive-aalok")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/aalok/" "source/latex/aalok/"
                                     "tex/latex/aalok/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0vpvqv60p5v3frdqa5r8k9k4zxk0icma5blw9dr6jhqfzf6lc3jy"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aalok")
    (synopsis "LaTeX class file for the Marathi journal @emph{Aalok}")
    (description
     "This package provides the class file for typesetting @emph{Aalok},
a Marathi journal, with LaTeX.")
    (license (list license:gpl3+
                   ;; Aalok (आलोक) copyleft license v1.0+.
                   (license:fsf-free "file://doc/latex/aalok/README.txt")
                   license:fdl1.3+))))

(define-public texlive-aastex
  (package
    (name "texlive-aastex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/aastex/" "doc/latex/aastex/"
                                     "tex/latex/aastex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07kbrv0vhlpdaxqkiwn2q4psldn5p2x5mmpaw70g6pgcbhk7r74h"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aastex")
    (synopsis "Macros for manuscript preparation for AAS journals")
    (description
     "The bundle provides a document class for preparing papers for
@acronym{AAS, American Astronomical Society} publications.  Authors who wish
to submit papers to AAS journals are strongly urged to use this class in
preference to any of the alternatives available.")
    (license license:lppl1.3+)))

(define-public texlive-abc
  (package
    (name "texlive-abc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/abc/" "source/latex/abc/"
                                     "tex/latex/abc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1vywrv35h65gkxipc7d26k32r04fk61kfkkk9bj3zsmxsw1v3r8d"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/abc")
    (synopsis "Support ABC music notation in LaTeX")
    (description
     "The abc package lets you include lines of music written in the ABC Plus
language.  The package will then employ the @code{\\write18} facility to
convert your notation to PostScript (using the established utility
@command{abcm2ps}) and hence to the format needed for inclusion in your
document.")
    (license license:lppl1.2+)))

(define-public texlive-abnt
  (package
    (name "texlive-abnt")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/abnt/" "tex/latex/abnt/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1zjsvp6ng6vf87mmnv18q435h6p3ygs512pk0m05yd8gyrmhmrxr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/abnt")
    (synopsis "Typesetting academic works according to ABNT rules")
    (description
     "The ABNT package provides a clean and practical implementation of the
@acronym{ABNT, Associacao Brasileira de Normas Tecnicas} rules for academic
texts.")
    (license license:lppl1.3c)))

(define-public texlive-abntex2
  (package
    (name "texlive-abntex2")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bib/abntex2/" "bibtex/bst/abntex2/"
                                     "doc/latex/abntex2/" "tex/latex/abntex2/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07752n6s2nl4h8j1wg7hjx2wisn505asb1r1i38z7qjbzg7lwqq5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/abntex2")
    (synopsis
     "Typeset technical and scientific Brazilian documents based on ABNT rules")
    (description
     "The bundle provides support for typesetting technical and scientific
Brazilian documents (like academic thesis, articles, reports, research project
and others) based on the @acronym{ABNT, Associacao Brasileira de Normas
Tecnicas} rules.  It replaces the old @code{abntex}.")
    (license license:lppl1.3+)))

(define-public texlive-abntexto
  (package
    (name "texlive-abntexto")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/abntexto/" "tex/latex/abntexto/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0mmjbn5vnqpq1l7mkpw5zrd84k3vvfxbj9xv0ndpb1cw4iadka7k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/abntexto")
    (synopsis "LaTeX class for formatting academic papers in ABNT standards")
    (description
     "This is a LaTeX class created for Brazilian students to facilitate the
use of standards from the @acronym{ABNT, Associacao Brasileira de Normas
Tecnicas} in academic works like TCCs, dissertations, theses.")
    (license license:public-domain)))

(define-public texlive-aboensis
  (package
    (name "texlive-aboensis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/aboensis/"
                                     "fonts/opentype/public/aboensis/"
                                     "tex/latex/aboensis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14333k6bkfywdka3h2a2yd2nw0521q0lwl3x38612j7cb82izyi3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aboensis")
    (synopsis "Late medieval OpenType cursive font")
    (description
     "The package contains the OpenType medieval cursive font Aboensis and
a style file to use it in XeLaTeX documents.  The font is based on @emph{Codex
Aboensis}, that is a law book written in Sweden in the 1430s.  Since medieval
cursive is very difficult to read for modern people, the font is not suitable
for use as an ordinary book font, but is intended for emulating late medieval
manuscripts.

The font contains two sets of initials: lombardic and cursive to go with the
basic alphabet, and there is support for writing two-colored initials and
capitals.  There are also a large number of abbreviation sigla that can be
accessed as ligature substitutions.  The style file contains macros that help
to use the extended features of the font such as initials and two-colored
capitals.  There are also macros to help achieve even pages with consistent
line spacing.")
    (license
     (list license:silofl1.1
           license:lppl1.3c
           license:cc-by4.0
           license:public-domain))))

(define-public texlive-academicons
  (package
    (name "texlive-academicons")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/academicons/"
                                     "fonts/truetype/public/academicons/"
                                     "tex/latex/academicons/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gxgj0ppyrf6pj9qvhk02ybwf68yzlh2110r41iwsai5by2iczhm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/academicons")
    (synopsis "Font containing high quality icons of online academic profiles")
    (description
     "The @code{academicons} package provides access in (La)TeX to 124 high
quality icons of online academic profiles included in the free Academicons
font.  This package requires either the Xe(La)TeX or Lua(La)TeX engine to load
the Academicons font from the system.  The @code{academicons} package provides
the generic @code{\\aiicon} command to access icons, which takes as mandatory
argument the name of the desired icon.  It also provides individual direct
commands for each specific icon.")
    (license (list license:lppl1.3c license:silofl1.1))))

(define-public texlive-accanthis
  (package
    (name "texlive-accanthis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/accanthis/"
                                     "fonts/enc/dvips/accanthis/"
                                     "fonts/map/dvips/accanthis/"
                                     "fonts/opentype/arkandis/accanthis/"
                                     "fonts/tfm/arkandis/accanthis/"
                                     "fonts/type1/arkandis/accanthis/"
                                     "fonts/vf/arkandis/accanthis/"
                                     "tex/latex/accanthis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xpy3bs8wwb6ncw4c8kc5vvddl77rmjwfjq66xw12sc8mgg9grmv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/accanthis")
    (synopsis "Accanthis fonts, with LaTeX support")
    (description
     "Accanthis No.@: 3, designed by Hirwin Harendal, is suitable as an
alternative to fonts such as Garamond, Galliard, Horley old style, Sabon, and
Bembo.  The package provides support files for use with all LaTeX engines.")
    (license (list license:gpl2+ license:lppl))))

(define-public texlive-accents
  (package
    (name "texlive-accents")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/accents/" "tex/latex/accents/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0p8g86xbfladz52dwbssmw3w9mbl987d1b740d25miphvj4ia9y9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/accents")
    (synopsis "Multiple mathematical accents")
    (description
     "This package provides a package for multiple accents in mathematics,
with nice features concerning the creation of accents and placement of
scripts.")
    (license license:expat)))

(define-public texlive-accfonts
  (package
    (name "texlive-accfonts")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/accfonts/" "scripts/accfonts/"
                                     "tex/latex/accfonts/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kja24rh0ysljwgkyg9mf47h64ayi2kmb1jb3dmvb42ywvg0w9i7"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-inputs "perl")))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "mkt1font" "vpl2ovp" "vpl2vpl")))
    (inputs (list perl))
    (home-page "https://ctan.org/pkg/accfonts")
    (synopsis "Utilities to derive new fonts from existing ones")
    (description
     "The @code{accfonts} package contains three utilities to permit easy
manipulation of fonts, in particular the creation of unusual accented
characters. @command{mkt1font} works on Adobe Type 1 fonts, @command{vpl2vpl}
works on TeX virtual fonts and @command{vpl2ovp} transforms a TeX font to an
Omega one.")
    (license license:gpl3+)))

(define-public texlive-accsupp
  (package
    (name "texlive-accsupp")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/accsupp/" "source/latex/accsupp/"
                                     "tex/latex/accsupp/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10xdg3gk5madpb131lhc9x2sxzby4rvqycbfzb5kl2zkwr1qw5v7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/accsupp")
    (synopsis "Better accessibility support for PDF files")
    (description
     "Since PDF 1.5 portions of a page can be marked for better accessibility
support.  For example, replacement texts or expansions of abbreviations can be
provided.  This package starts with providing a minimal low-level interface
for programmers; its status is experimental.")
    (license license:lppl1.3c)))

(define-public texlive-abbr
  (package
    (name "texlive-abbr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/abbr/" "tex/generic/abbr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0zi5qkgr2nnb3jp5adv42d84w7yav1dw0lgg2vdxfsz03j1bmlsb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/abbr")
    (synopsis "Simple macros supporting abbreviations for Plain and LaTeX")
    (description
     "The package provides some simple macros to support abbreviations in
Plain TeX or LaTeX.  It allows writing, e.g., @samp{\\<TEX>} instead of
@samp{\\TeX}, hence frees users from having to escape space after
parameterless macros.")
    (license license:public-domain)))

(define-public texlive-abstyles
  (package
    (name "texlive-abstyles")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bib/abstyles/"
                                     "bibtex/bst/abstyles/"
                                     "doc/bibtex/abstyles/"
                                     "tex/generic/abstyles/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ij4cp3pydfs4lch0f93gi2q7msafxyqfvv44whw1ppczidicnn2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/abstyles-orig")
    (synopsis "Adaptable BibTeX styles")
    (description
     "This package provides a family of modifications of the standard BibTeX
styles whose behaviour may be changed by changing the user document, without
change to the styles themselves.  The package is largely used nowadays in its
adaptation for working with Babel.")
    (license
     (list license:isc                  ;apreamble
           license:knuth                ;bxabst
           license:public-domain))))    ;docmac

(define-public texlive-acmconf
  (package
    (name "texlive-acmconf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/acmconf/" "source/latex/acmconf/"
                                     "tex/latex/acmconf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0j30krvxzps7ka3xzv0qifrzxhxzkr3b3f9gska6najp1dvygs3a"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/acmconf")
    (synopsis "Class for ACM conference proceedings")
    (description
     "This class may be used to typeset articles to be published in the
proceedings of @acronym{ACM, Association for Computing Machinery} conferences
and workshops.  The layout produced by the @code{acmconf} class is based on
the ACM's own specification.")
    (license license:lppl)))

(define-public texlive-active-conf
  (package
    (name "texlive-active-conf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/active-conf/"
                                     "source/latex/active-conf/"
                                     "tex/latex/active-conf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12krmwspwkhcwk5q0qf2hldnsk44vd2jfssfdiwm97alicc3cpj1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/active-conf")
    (synopsis "Class for typesetting ACTIVE conference papers")
    (description
     "Active-conf is a class for typesetting papers for the Active conference
on noise and vibration control.  The class is based on @code{article} with
more flexible front-matter, and can be customised for conferences in future
years with a header file.")
    (license license:lppl)))

(define-public texlive-adfathesis
  (package
    (name "texlive-adfathesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/adfathesis/"
                                     "doc/latex/adfathesis/"
                                     "source/latex/adfathesis/"
                                     "tex/latex/adfathesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0zi91xc3sbdjvp87zgrb7g3l5hxqig33fchlhg2i8gjya9ij67p6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/adfathesis")
    (synopsis "Australian Defence Force Academy thesis format")
    (description "The package provides the Australian Defence Force Academy
thesis format.  The bundle also includes a BibTeX style file.")
    (license license:public-domain)))

(define-public texlive-adforn
  (package
    (name "texlive-adforn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/adforn/"
                                     "fonts/afm/arkandis/adforn/"
                                     "fonts/enc/dvips/adforn/"
                                     "fonts/map/dvips/adforn/"
                                     "fonts/tfm/arkandis/adforn/"
                                     "fonts/type1/arkandis/adforn/"
                                     "source/fonts/adforn/"
                                     "tex/latex/adforn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1dw2033lr0j9ag6cn1sa9qvj4awi43vgapcwam2rzj0hhy7r1yd0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/adforn")
    (synopsis "OrnementsADF font with TeX and LaTeX support")
    (description
     "The bundle provides the Ornements ADF font in PostScript Type 1 format
with TeX and LaTeX support files.")
    (license (list license:lppl license:gpl2+))))

(define-public texlive-adfsymbols
  (package
    (name "texlive-adfsymbols")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/adfsymbols/"
                                     "fonts/afm/arkandis/adfsymbols/"
                                     "fonts/enc/dvips/adfsymbols/"
                                     "fonts/map/dvips/adfsymbols/"
                                     "fonts/tfm/arkandis/adfsymbols/"
                                     "fonts/type1/arkandis/adfsymbols/"
                                     "source/fonts/adfsymbols/"
                                     "tex/latex/adfsymbols/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12la4i9cv9azr4ldqawf7jmvmkcramj95ck298my356dsfmqpmyv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/adfsymbols")
    (synopsis "SymbolsADF with TeX and LaTeX support")
    (description
     "The package provides Arkandis foundry's ArrowsADF and BulletsADF fonts
in Adobe Type 1 format, together with TeX and LaTeX support files.")
    (license (list license:lppl license:gpl2))))

(define-public texlive-adhocfilelist
  (package
    (name "texlive-adhocfilelist")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/support/adhocfilelist/"
                                     "scripts/adhocfilelist/"
                                     "source/support/adhocfilelist/"
                                     "tex/support/adhocfilelist/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "097yy38571fa2hzp4s3p125xknqgjmkhv1vgi0q9vz83b6175hc4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "adhocfilelist.sh")))
    (home-page "https://ctan.org/pkg/adhocfilelist")
    (synopsis "@code{\\listfiles} entries from the command line")
    (description
     "The package provides a Unix shell script to display a list of LaTeX
@code{\\Provides}...-command contexts on screen.  Provision is made for
controlling the searches that the package does.")
    (license license:lppl)))

(define-public texlive-adtrees
  (package
    (name "texlive-adtrees")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/adtrees/" "tex/latex/adtrees/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ypb609wvw3mvs7bf2mwa7r7v4frxjhfll8d1sqfry3dv3kr13pq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/adtrees")
    (synopsis "Macros for drawing adpositional trees")
    (description
     "This package provides a means to write adpositional trees, a formalism
devoted to representing natural language expressions.")
    (license license:gpl3+)))

(define-public texlive-adobemapping
  (package
    (name "texlive-adobemapping")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "fonts/cmap/adobemapping/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1yvf74mlmxmnq9m0wvda9xcna8xnn382bbrzizzjm5w7bk8c4g98"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/adobemapping")
    (synopsis "Adobe CMap and PDF mapping files")
    (description
     "The package comprises the collection of CMap and PDF mapping files made
available for distribution by Adobe.")
    (license license:bsd-3)))

(define-public texlive-aesupp
  (package
    (name "texlive-aesupp")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/aesupp/"
                                     "fonts/enc/dvips/aesupp/"
                                     "fonts/map/dvips/aesupp/"
                                     "fonts/opentype/public/aesupp/"
                                     "fonts/tfm/public/aesupp/"
                                     "fonts/type1/public/aesupp/"
                                     "source/fonts/aesupp/"
                                     "tex/latex/aesupp/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1sx5802d1ib72i0ssjpr0c54pgq1s980wg3q1685pjniw68ns1vj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aesupp")
    (synopsis "Special support for the @samp{ae} character")
    (description
     "This package provides special support for the italic @samp{ae} character
in some fonts, due to design flaws (in the author's opinion) regarding this
character.  At the moment only the fonts TeX Gyre Bonum, TeX Gyre Schola, TeX
Gyre Pagella, and the Latin Modern fonts are supported.  The other fonts in
the TeX Gyre bundle do not need this support.")
    (license (list license:gfl1.0 license:gpl3+))))

(define-public texlive-affilauthor
  (package
    (name "texlive-affilauthor")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/affilauthor/"
                              "tex/latex/affilauthor/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0nig3fz7pa8360y9nrc7xlvsyhv2g5pvcl2pp25qj3v88nbrv0r5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/affilauthor")
    (synopsis "Tag author and affiliation information in a key-value style")
    (description
     "This package provides key-value style author and affiliation information
tagging in a structured format.  Each field has a specific name within
@code{\\author} and @code{\\affil} commands similar to BibTeX format, and can
be customized individually.")
    (license license:lppl1.3c)))

(define-public texlive-afm2pl
  (package
    (name "texlive-afm2pl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/afm2pl.1"
                                     "doc/man/man1/afm2pl.man1.pdf"
                                     "fonts/enc/dvips/afm2pl/"
                                     "fonts/lig/afm2pl/" "tex/fontinst/afm2pl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "19llzzr4kmmyf7l18ngx1rhaqaqvgm3md924m4dxcv7nmrvga2b2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-afm2pl-bin))
    (home-page "https://ctan.org/pkg/afm2pl")
    (synopsis "Convert AFM to TeX property list (@file{.pl}) metrics")
    (description
     "@command{afm2pl} converts a @file{.afm} (Adobe Font Metric) file into
a @file{.pl} (Property List) file, which in its turn can be converted to
a @file{.tfm} (TeX Font Metric) file.  It normally preserves kerns and
ligatures, but also offers additional control over them.")
    (license license:gpl2)))

(define-public texlive-afm2pl-bin
  (package
    (inherit texlive-bin)
    (name "texlive-afm2pl-bin")
    (source
     (origin
       (inherit (package-source texlive-source))
       (modules '((guix build utils)
                  (ice-9 ftw)))
       (snippet
        #~(let ((delete-other-directories
                 (lambda (root dirs)
                   (with-directory-excursion root
                     (for-each
                      delete-file-recursively
                      (scandir "."
                               (lambda (file)
                                 (and (not (member file (append '("." "..") dirs)))
                                      (eq? 'directory (stat:type (stat file)))))))))))
            (delete-other-directories "libs" '())
            (delete-other-directories "utils" '())
            (delete-other-directories "texk" '("afm2pl"))))))
    (arguments
     (substitute-keyword-arguments (package-arguments texlive-bin)
       ((#:configure-flags flags)
        #~(cons "--enable-afm2pl" (delete "--enable-web2c" #$flags)))
       ((#:phases phases)
        #~(modify-phases #$phases
            (replace 'check
              (lambda* (#:key tests? #:allow-other-keys)
                (when tests?
                  (with-directory-excursion "texk/afm2pl"
                    (invoke "make" "check")))))
            (replace 'install
              (lambda _
                (with-directory-excursion "texk/afm2pl"
                  (invoke "make" "install"))))))))
    (native-inputs (list pkg-config))
    (inputs (list texlive-libkpathsea))
    (propagated-inputs '())
    (synopsis "Binary for @code{texlive-afm2pl}")
    (description
     "This package provides the binary for @code{texlive-afm2pl}.")
    (license (package-license texlive-afm2pl))))

(define-public texlive-afthesis
  (package
    (name "texlive-afthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/afthesis/" "doc/latex/afthesis/"
                                     "tex/latex/afthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1db9fzs4rf1d86n2gcn0zs255nlwc8i1yl5n9qzsn6xf5as7zz1i"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/afthesis")
    (synopsis "Air Force Institute of Technology thesis class")
    (description
     "This is a LaTeX thesis and dissertation class for US Air Force Institute
Of Technology.")
    (license license:public-domain)))

(define-public texlive-aguplus
  (package
    (name "texlive-aguplus")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/aguplus/" "doc/latex/aguplus/"
                                     "tex/latex/aguplus/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qpi5r4y70cmxinr9y46rw6dfd337l5xhnqfvbkz27nw98j87fi3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aguplus")
    (synopsis "Styles for American Geophysical Union")
    (description
     "This bundle started as an extension to the @acronym{AGU, American
Geophysical Union}'s own published styles, providing extra facilities and
improved usability.  The AGU now publishes satisfactory LaTeX materials of its
own; it is recommended to switch to the official distribution.")
    (license license:lppl)))

(define-public texlive-aiaa
  (package
    (name "texlive-aiaa")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/aiaa/" "doc/latex/aiaa/"
                                     "source/latex/aiaa/" "tex/latex/aiaa/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0l6sp7q53nr1ppxaynfc0q3kf1hg5v22yxyvzjdg028rf1nvbhww"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aiaa")
    (synopsis "Typeset AIAA conference papers")
    (description
     "This package provides a bundle of LaTeX and BibTeX files and sample
documents to aid those producing papers and journal articles according to the
guidelines of the @acronym{AIAA, American Institute of Aeronautics and
Astronautics}.")
    (license license:lppl)))

(define-public texlive-aichej
  (package
    (name "texlive-aichej")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/aichej/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pr92pq3yjsrbax6ah01cqg7131fp2rkqqjnxwcrm4119z2ic0gf"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aichej")
    (synopsis "Bibliography style file for the AIChE Journal")
    (description
     "The style was generated using custom-bib, and implements the style of
the American Institute of Chemical Engineers Journal (or @code{AIChE} Journal
or @code{AIChE} J or @code{AIChEJ).}")
    (license license:lppl)))

(define-public texlive-aiplans
  (package
    (name "texlive-aiplans")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/aiplans/"
                              "tex/latex/aiplans/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0sx5qs8wyhhmrzz0jfnsrxm5xh6m18ifl2v0z2qcv699w41n0zaq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aiplans")
    (synopsis "TikZ-based library for drawing POCL plans")
    (description
     "This TikZ library is designed for generating diagrams related to
Automated Planning, a sub-discipline of Artificial Intelligence.  It allows
users to define a ``domain model'' for actions, similar to PDDL and HDDL used
in hierarchical planning.  The package is useful for researchers and students
to create diagrams that represent sequential action sequences or partially
ordered plans, including causal links and ordering constraints (e.g., POCL
plans).  It is particularly suited for presentations and scientific
publications.")
    (license license:lppl1.3c)))

(define-public texlive-ajl
  (package
    (name "texlive-ajl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/ajl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xkrvy06lw4ngj94zybs14ysal2pg7g71h864gq9bjw0zwq2yjzs"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ajl")
    (synopsis "BibTeX style for AJL")
    (description
     "This package provides bibliographic style references in style of
Australian Journal of Linguistics.")
    (license license:lppl)))

(define-public texlive-akshar
  (package
    (name "texlive-akshar")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/akshar/" "source/latex/akshar/"
                                     "tex/latex/akshar/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17xiqalqpw64pgvmpsa41qf6y5fhg720rpbr57kv8ipvvm1fn3n5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/akshar")
    (synopsis "Support for syllables in the Devanagari script")
    (description
     "This LaTeX3 package provides macros and interfaces to work with
Devanagari characters and syllables in a more correct way.")
    (license license:lppl1.3c)))

(define-public texlive-albatross
  (package
    (name "texlive-albatross")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/albatross.1"
                                     "doc/man/man1/albatross.man1.pdf"
                                     "doc/support/albatross/"
                                     "scripts/albatross/"
                                     "source/support/albatross/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0gg7abaixfxmwhdj3z84sg1vkglh8a8q2p3ql1hy74f5rpkyf3yg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "albatross.sh")))
    (home-page "https://ctan.org/pkg/albatross")
    (synopsis "Find fonts that contain a given glyph")
    (description
     "This is a command line tool for finding fonts that contain
a given (Unicode) glyph.  It relies on Fontconfig.")
    (license license:bsd-3)))

(define-public texlive-alegreya
  (package
    (name "texlive-alegreya")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/alegreya/"
                                     "fonts/enc/dvips/alegreya/"
                                     "fonts/map/dvips/alegreya/"
                                     "fonts/opentype/huerta/alegreya/"
                                     "fonts/tfm/huerta/alegreya/"
                                     "fonts/type1/huerta/alegreya/"
                                     "fonts/vf/huerta/alegreya/"
                                     "tex/latex/alegreya/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1n09fq89nxw3jaglmvjfl7ysflqi0rh6mmp6q0zn57m72sza9rra"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/alegreya")
    (synopsis "Alegreya fonts with LaTeX support")
    (description
     "Alegreya, designed by Juan Pablo del Peral, is a typeface originally
intended for literature.  It conveys a dynamic and varied rhythm which
facilitates the reading of long texts.  Bold, black, small caps and five
number styles are available.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-aleph
  (package
    (name "texlive-aleph")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/aleph/base/" "doc/man/man1/aleph.1"
                                     "doc/man/man1/aleph.man1.pdf"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0nx8nrjjavkqrvp7nmf3nc6yvxlab58q8a0c7xfb1qh08pzr8gn2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:create-formats #~(list "aleph")))
    (native-inputs (list (if (target-64bit?) libfaketime datefudge)))
    (propagated-inputs
     (list texlive-aleph-bin
           texlive-cm
           texlive-hyphen-base
           texlive-knuth-lib
           texlive-lambda
           texlive-latex
           texlive-plain))
    (home-page "https://ctan.org/pkg/aleph")
    (synopsis "Extended TeX")
    (description
     "This package provides a development of Omega, using most of the
extensions of TeX, itself developed for e-TeX.")
    (license license:gpl3+)))

(define-public texlive-aleph-bin
  (package
    (inherit texlive-bin)
    (name "texlive-aleph-bin")
    (arguments
     (substitute-keyword-arguments (package-arguments texlive-bin)
       ((#:configure-flags flags)
        #~(cons* "--disable-web2c"
                 "--enable-aleph"
                 (delete "--disable-aleph"
                         (delete "--enable-web2c" #$flags))))
       ((#:phases phases)
        #~(modify-phases #$phases
            (replace 'install
              (lambda _
                (with-directory-excursion "texk/web2c"
                  (invoke "make" "aleph")
                  (install-file "aleph" (string-append #$output "/bin")))))))))
    (native-inputs (list pkg-config))
    (home-page (package-home-page texlive-aleph))
    (synopsis "Binary for @code{texlive-aleph}")
    (description
     "This package provides the binary for @code{texlive-aleph}.")
    (license (package-license texlive-aleph))))

(define-public texlive-alfaslabone
  (package
    (name "texlive-alfaslabone")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/alfaslabone/"
                                     "fonts/enc/dvips/alfaslabone/"
                                     "fonts/map/dvips/alfaslabone/"
                                     "fonts/opentype/public/alfaslabone/"
                                     "fonts/tfm/public/alfaslabone/"
                                     "fonts/type1/public/alfaslabone/"
                                     "fonts/vf/public/alfaslabone/"
                                     "tex/latex/alfaslabone/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wpn96dd1rca52dqwngvyyvnmvak32sjz2id6lcra2jnz5jhk01g"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/alfaslabone")
    (synopsis "Alfa Slab One font face with support for LaTeX and pdfLaTeX")
    (description
     "The @code{alfaslabone} package supports the Alfa Slab One font face for
LaTeX.  There is only a Regular font face.  It's useful for book-chapter
headlines.")
    (license license:silofl1.1)))

(define-public texlive-alg
  (package
    (name "texlive-alg")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/alg/" "source/latex/alg/"
                                     "tex/latex/alg/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jqk9sgw2m4c6hssfc5c3qfxqvpyppvcmy8i5crm7fsmkv0y6wkg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/alg")
    (synopsis "LaTeX environments for typesetting algorithms")
    (description
     "The package defines two environments for typesetting algorithms in LaTeX2e.
The @code{algtab} environment is used to typeset an algorithm with
automatically numbered lines.  The @code{algorithm} environment can be used to
encapsulate the @code{algtab} environment algorithm in a floating body
together with a header, a caption, etc.  @code{\\listofalgorithms} is
defined.")
    (license license:lppl)))

(define-public texlive-algobox
  (package
    (name "texlive-algobox")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/algobox/" "source/latex/algobox/"
                                     "tex/latex/algobox/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03z5s4v1asfp6rm51wfkmax69rlggfdyb3p1acdy8cx1gwd10f2w"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/algobox")
    (synopsis "Typeset Algobox programs")
    (description
     "This LaTeX package can typeset Algobox programs almost exactly as
displayed when editing with Algobox itself, using an input syntax very similar
to the actual Algobox program text.  It gives better results than Algobox's
own LaTeX export which does not look like the editor rendition, produces
standalone documents cumbersome to customize, and has arbitrary and
inconsistent differences between the input syntax and the program text.")
    (license license:gpl3+)))

(define-public texlive-algolrevived
  (package
    (name "texlive-algolrevived")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/algolrevived/"
                                     "fonts/enc/dvips/algolrevived/"
                                     "fonts/map/dvips/algolrevived/"
                                     "fonts/opentype/public/algolrevived/"
                                     "fonts/tfm/public/algolrevived/"
                                     "fonts/type1/public/algolrevived/"
                                     "fonts/vf/public/algolrevived/"
                                     "tex/latex/algolrevived/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1x56hgb8cxavg1yfbsi3k8bvlx06xsx7fl08ps8gxg4vj57gkcyh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/algolrevived")
    (synopsis "Revival of Frutiger's Algol alphabet")
    (description
     "The package revives Frutiger's Algol alphabet, designed in 1963 for the
code segments in an ALGOL manual.  It provides OpenType and Type 1, regular
and medium weights, upright and slanted variations.  Albeit not monospaced,
this font is good for listings if you don't need code to be aligned with
specific columns.  It also makes a passable but limited text font.")
    (license (list license:silofl1.1 license:lppl1.3c))))

(define-public texlive-algorithm2e
  (package
    (name "texlive-algorithm2e")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/algorithm2e/"
                                     "tex/latex/algorithm2e/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0bf4fc9p84zy2zdl0hiklcbbxn17j3qyl02kyj5ya00lrf6sv93n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/algorithm2e")
    (synopsis "Floating algorithm environment with algorithmic keywords")
    (description
     "Algorithm2e is an environment for writing algorithms.  An algorithm
becomes a floating object (like figure, table, etc.).  The package provides
macros that allow you to create different keywords, and a set of predefined
key words is provided; you can change the typography of the keywords.  The
package allows vertical lines delimiting a block of instructions in an
algorithm, and defines different sorts of algorithms such as @samp{Procedure}
or @samp{Function}; the name of these functions may be reused in the text or
in other algorithms.")
    (license license:lppl)))

(define-public texlive-algorithmicx
  (package
    (name "texlive-algorithmicx")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/algorithmicx/"
                                     "tex/latex/algorithmicx/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0q0qkzc88479y5dqifnpfxhclk48gf1c4shxi1xhgqbvjp2gqgza"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/algorithmicx")
    (synopsis "The algorithmic style you always wanted")
    (description
     "Algorithmicx provides a flexible, yet easy to use, way for inserting
good looking pseudocode or source code in your papers.  It has built in
support for pseudocode, Pascal and C, and offers powerful means to create
definitions for any programming language.  The user can adapt a pseudocode
style to his native language.")
    (license license:lppl)))

(define-public texlive-algorithms
  (package
    (name "texlive-algorithms")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/algorithms/"
                                     "source/latex/algorithms/"
                                     "tex/latex/algorithms/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0yw0alkigq77a2ha4hdrvpagqvyy0cv61ddhvf227wlwmvw5ch2w"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/algorithms")
    (synopsis "Suite of tools for typesetting algorithms in pseudo-code")
    (description
     "The package consists of two environments: @code{algorithm} and
@code{algorithmic}.  The @code{algorithm} package defines a floating
@code{algorithm} environment designed to work with the @code{algorithmic}
style.  Within an @code{algorithmic} environment a number of commands for
typesetting popular algorithmic constructs are available.")
    (license license:lgpl2.1)))

(define-public texlive-algpseudocodex
  (package
    (name "texlive-algpseudocodex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/algpseudocodex/"
                                     "tex/latex/algpseudocodex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03k0ynz63gqgmm6ylikpaq28cd70qfq3advv8n1nbi8bjgz7xrd4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/algpseudocodex")
    (synopsis "Package for typesetting pseudocode")
    (description
     "This package allows typesetting pseudocode in LaTeX.  It is based on
@code{algpseudocode} from the @code{algorithmicx} package and uses the same
syntax, but adds several new features and improvements.  Notable features
include customizable indent guide lines and the ability to draw boxes around
parts of the code for highlighting differences.  This package also has better
support for long code lines spanning several lines and improved comments.")
    (license license:lppl1.3c)))

(define-public texlive-algxpar
  (package
    (name "texlive-algxpar")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/algxpar/"
                                     "tex/latex/algxpar/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1mikzc63n8kv0kaw6l1rfasxqi9a4hfzgz1xhd4rx4dzalhjhxzg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/algxpar")
    (synopsis "Support multiple lines pseudocode")
    (description
     "This package extends the package @code{algorithmicx} to support long
text which spans over multiple lines.")
    (license license:lppl1.3+)))

(define-public texlive-aligned-overset
  (package
    (name "texlive-aligned-overset")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/aligned-overset/"
                                     "source/latex/aligned-overset/"
                                     "tex/latex/aligned-overset/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1awawz0ypq15r16mx949g7r7pp9a60xhi8hciy1xsa6dvvj47h5x"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aligned-overset")
    (synopsis "Fix alignment at @code{\\overset} or @code{\\underset}")
    (description
     "This package allows the base character of @code{\\underset} or
@code{\\overset} to be used as the alignment position for the @code{aligned}
math environments.")
    (license license:lppl1.3c)))

(define-public texlive-alkalami
  (package
    (name "texlive-alkalami")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/alkalami/"
                                     "fonts/truetype/public/alkalami/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0979xhsjar0gwjp9yqp0kaf4gp4z715gr2kdb8dkqg8zzlkqhgzk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/alkalami")
    (synopsis "Font for Arabic-based writing systems in Nigeria and Niger")
    (description
     "This font is designed for Arabic-based writing systems in the Kano
region of Nigeria and Niger.")
    (license license:silofl1.1)))

(define-public texlive-allrunes
  (package
    (name "texlive-allrunes")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/allrunes/"
                                     "fonts/map/dvips/allrunes/"
                                     "fonts/source/public/allrunes/"
                                     "fonts/type1/public/allrunes/"
                                     "source/fonts/allrunes/"
                                     "tex/latex/allrunes/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14a1qaa30s4dzmg4dlrjd3faa3bcrl2gdpp700i8pghj3pjk7gp4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/allrunes")
    (synopsis "Fonts and LaTeX package for almost all runes")
    (description
     "This large collection of fonts (in Adobe Type 1 format), with the LaTeX
package gives access to almost all runes ever used in Europe.  The bundle
covers not only the main forms but also a lot of varieties.")
    (license license:lppl)))

(define-public texlive-almendra
  (package
    (name "texlive-almendra")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/almendra/"
                                     "fonts/enc/dvips/almendra/"
                                     "fonts/map/dvips/almendra/"
                                     "fonts/tfm/public/almendra/"
                                     "fonts/truetype/public/almendra/"
                                     "fonts/type1/public/almendra/"
                                     "fonts/vf/public/almendra/"
                                     "tex/latex/almendra/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1mkj0cjpry0hdynlpf719haczm1csv0yvla6nhlgp3dq5k28hvp2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/almendra")
    (synopsis "Almendra fonts with LaTeX support")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX, and LuaLaTeX support for
the Almendra family of fonts, designed by Ana Sanfelippo.  Almendra is
a typeface design based on calligraphy.  Its style is related to the Chancery
and Gothic hands.  There are regular and bold weights with matching italics.
There is also a regular-weight small-caps.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-almfixed
  (package
    (name "texlive-almfixed")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/almfixed/"
                                     "fonts/opentype/public/almfixed/"
                                     "fonts/truetype/public/almfixed/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ihyldbdz89swj8lm75kb3n3nqqgzkydwlppsndfpib74pvp6s97"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/almfixed")
    (synopsis "Extend TeX-Gyre Latin Modern to full Arabic Unicode support")
    (description
     "Arabic-Latin Modern Fixed is an extension of TeX-Gyre Latin Modern Mono
10 Regular.  Every glyph and OpenType feature of the Latin Modern Mono has
been retained, with minor improvements.  On the other hand, we have changed
the vertical metrics of the font.

The unique feature of Arabic-Latin Modern is its treatment of vowels and
diacritics.  Each vowel and diacritic (ALM Fixed contains a total of 68 such
glyphs) may now be edited horizontally within any text editor or processor.
Editing complex Arabic texts will now be much easier to input and to
proofread.")
    (license license:gfl1.0)))

(define-public texlive-alpha-persian
  (package
    (name "texlive-alpha-persian")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/alpha-persian/"
                                     "doc/bibtex/alpha-persian/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15ynmgh3fp6s40c200kgr6v0q84bkzp9cv45yg90mr9d7mq72djv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/alpha-persian")
    (synopsis "Persian version of @file{alpha.bst}")
    (description
     "The package provides a Persian version of the alpha BibTeX style and
offers several enhancements.  It is compatible with the @code{hyperref},
@code{url}, @code{natbib}, and @code{cite} packages.")
    (license license:lppl1.3c)))

(define-public texlive-amscdx
  (package
    (name "texlive-amscdx")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/amscdx/" "source/latex/amscdx/"
                                     "tex/latex/amscdx/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13gxllncd46w4kscb7isivw6x82pw9yfamcv3sq6wnx68z03n2q3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/amscdx")
    (synopsis "Enhanced commutative diagrams")
    (description
     "The original @code{amscd} package provides a @code{CD} environment that
emulates the commutative diagram capabilities of AMS-TeX version 2.x.  This
means that only simple rectangular diagrams are supported, with no diagonal
arrows or more exotic features.  This enhancement package implements double,
dashed, and bidirectional arrows (left-right and up-down), and color
attributes for arrows and their annotations.  The restriction to rectangular
geometry remains.  This nevertheless allows the drawing of a much broader
class of commutative diagrams and alike.")
    (license license:lppl1.3c)))

(define-public texlive-amscls-doc
  (package
    (name "texlive-amscls-doc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/amscls-doc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0allim05cp20zhn480df2mivd3p9gnc069d7hbjlzv660bw7mapx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/amscls-doc")
    (synopsis "User documentation for AMS document classes")
    (description
     "This collection comprises a set of four manuals, or Author Handbooks,
each documenting the use of a class of publications based on one of the AMS
document classes @code{amsart}, @code{amsbook}, @code{amsproc} and one hybrid,
as well as a guide to the generation of the four manuals from a coordinated
set of LaTeX source files.  The Handbooks comprise the user documentation for
the pertinent document classes.  As the source for the Handbooks consists of
a large number of files, and the intended output is multiple different
documents, the principles underlying this collection can be used as a model
for similar projects.  The manual @emph{Compiling the AMS Author Handbooks}
provides information about the structure of and interaction between the
various components.")
    (license license:lppl1.3c)))

(define-public texlive-amslatex-primer
  (package
    (name "texlive-amslatex-primer")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/amslatex-primer/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1kzayyh1bycmq43s2xn81jf05r18azidbk3gv6igf2vaq37fmxil"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/amslatex-primer")
    (synopsis "Getting up and running with AMS-LaTeX")
    (description
     "The document aims to get you up and running with AMS-LaTeX as quickly as
possible.  These instructions are not a substitute for the full documentation,
but they may get you started quickly enough so that you will only need to
refer to the main documentation occasionally.  In addition to AMS-LaTeX out of
the box, the document contains a section describing how to draw commutative
diagrams using Xy-pic and a section describing how to use @code{amsrefs} to
create a bibliography.")
    (license license:lppl)))

(define-public texlive-amsldoc-it
  (package
    (name "texlive-amsldoc-it")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/amsldoc-it/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0d4hwb7hywy56d6934448lcr6fdx7qchkfzs806dr7wfzfy36yix"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/amsldoc-it")
    (synopsis "Italian translation of @code{amsldoc}")
    (description
     "This package provides an Italian translation of @code{amsldoc}.")
    ;; Use same license as `amsmath'.
    (license license:lppl1.3c)))

(define-public texlive-amsldoc-vn
  (package
    (name "texlive-amsldoc-vn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/amsldoc-vn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1iz0zjn1v7izwbsq0zb6cvpnkbvk0nxw1b24j7dzqwk3m9j43i6x"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/amslatexdoc-vietnamese")
    (synopsis "Vietnamese translation of AMSLaTeX documentation")
    (description
     "This is a Vietnamese translation of @code{amsldoc}, the users guide to
@code{amsmath}.")
    (license license:lppl1.3+)))

(define-public texlive-amstex
  (package
    (name "texlive-amstex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/amstex/base/" "doc/man/man1/amstex.1"
                                     "doc/man/man1/amstex.man1.pdf"
                                     "tex/amstex/base/" "tex/amstex/config/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "16ivmxl924cchgss7jkix7fihbfw1z62avhk5lpi81x3xiz71f6f"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:create-formats #~(list "amstex")
           #:phases
           #~(modify-phases %standard-phases
               (add-after 'install 'symlink-binaries
                 (lambda* (#:key inputs #:allow-other-keys)
                   (let ((pdftex (search-input-file inputs "bin/pdftex"))
                         (bin (string-append #$output "/bin")))
                     (mkdir-p bin)
                     (with-directory-excursion bin
                       (symlink pdftex "amstex"))))))))
    (native-inputs (list (if (target-64bit?) libfaketime datefudge)))
    (propagated-inputs
     (list texlive-amsfonts
           texlive-cm
           texlive-hyphen-base
           texlive-knuth-lib
           texlive-pdftex
           texlive-plain
           texlive-tex))
    (home-page "https://ctan.org/pkg/amstex")
    (synopsis "American Mathematical Society plain TeX macros")
    (description
     "AMS-TeX is a TeX macro package based on Plain TeX: it provides many
features for producing more professional-looking maths formulas with less
burden on authors.

This is the final archival distribution of AMS-TeX.  AMS-TeX is no longer
supported by the AMS, nor is it used by the AMS publishing program.  The AMS
does not recommend creating any new documents using AMS-TeX; this distribution
will be left on CTAN to facilitate processing of legacy documents and as
a historical record of a pioneering TeX macro collection that played a key
role in popularizing TeX and revolutionizing mathematics publishing.  AMS-TeX
is the historical basis of @code{amslatex}, which should now be used to
prepare submissions for the AMS.")
    (license license:lppl)))

(define-public texlive-amsthdoc-it
  (package
    (name "texlive-amsthdoc-it")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/amsthdoc-it/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ic88gs89m3d9ys40c4k7sgx6wy82c8isg2qkmd4snw5yms6fpaz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/amsthdoc-it")
    (synopsis "Italian translation of @code{amsthdoc}")
    (description
     "This package provides an Italian translation of @code{amsthdoc}.")
    ;; Use same license as `amscls'.
    (license license:lppl1.3c)))

(define-public texlive-andika
  (package
    (name "texlive-andika")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/andika/"
                                     "fonts/enc/dvips/andika/"
                                     "fonts/map/dvips/andika/"
                                     "fonts/tfm/SIL/andika/"
                                     "fonts/truetype/SIL/andika/"
                                     "fonts/type1/SIL/andika/"
                                     "fonts/vf/SIL/andika/"
                                     "tex/latex/andika/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gjahg3xwzwrj9cp6l0zmhfqxwzl4m89vvngmpq4ym0f6rg5pb4i"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/andika")
    (synopsis "Andika fonts with support for all LaTeX engines")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX and LuaLaTeX support for
the Andika family of fonts designed by SIL International especially for
literacy use, taking into account the needs of beginning readers.  The focus
is on clear, easy-to-perceive letterforms that will not be readily confused
with one another.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-annee-scolaire
  (package
    (name "texlive-annee-scolaire")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/annee-scolaire/"
                                     "source/latex/annee-scolaire/"
                                     "tex/latex/annee-scolaire/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1nwm67p4s822d7r4kr3rbvnm9vlkvzjbikqy8hq2fqsc9bdx34ib"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/annee-scolaire")
    (synopsis "Automatically typeset the academic year (French way)")
    (description
     "This package provides a macro @code{\\anneescolaire} to automatically
write the academic year in the French way, according to the date of
compilation, two other macros to obtain the first and the second calendar year
of the academic year, a macro to be redefined to change the presentation of
the years.")
    (license license:lppl1.3c)))

(define-public texlive-annotate
  (package
    (name "texlive-annotate")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/annotate/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12q3xk08ycz57vsrcr2glk7y48w8sbc4n8lrqaicd46cxfd4jdqm"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/annotate")
    (synopsis "Bibliography style with annotations")
    (description
     "The style is a derivative of the standard @code{alpha} style, which
processes an entry's annotate field as part of the printed output.")
    ;; There is no explicit license in the "annotate.bst" file, but it
    ;; mentions being derived from "alpha.bst", which is released under
    ;; Knuth's terms.  We therefore use the same license.
    (license license:knuth)))

(define-public texlive-annotate-equations
  (package
    (name "texlive-annotate-equations")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/annotate-equations/"
                                     "tex/latex/annotate-equations/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07cdj0sqx796gsjmrpq8bzkmiqbli24593y4axcmrr8kngfki9f2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/annotate-equations")
    (synopsis "Easily annotate math equations using TikZ")
    (description
     "This package provides commands that make it easy to highlight terms in
equations and add annotation labels using TikZ.  It should work with pdfLaTeX
as well as LuaLaTeX.")
    (license license:expat)))

(define-public texlive-anonymous-acm
  (package
    (name "texlive-anonymous-acm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/anonymous-acm/"
                                     "tex/latex/anonymous-acm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05csmx485sjky64d2hwlb5f7mrkixhw1gn3nl7p7lng9sd8qh25k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/anonymous-acm")
    (synopsis "Typeset anonymous versions for ACM articles")
    (description
     "Academics often need to submit anonymous versions of their papers for
peer-review.  This often requires anonymization which at some future date
needs to be reversed.  However de-anonymizing an anonymized paper can be
laborious and error-prone.  This LaTeX package allows anonymization options to
be specified at the time of writing for authors using @code{acmart.cls}, the
official @acronym{ACM, Association of Computing Machinery} master
@code{article} template.  Anonymization or deanonymization is carried out by
simply changing one option and recompiling.")
    (license license:lppl1.3+)))

(define-public texlive-anonymouspro
  (package
    (name "texlive-anonymouspro")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/anonymouspro/"
                                     "fonts/afm/public/anonymouspro/"
                                     "fonts/enc/dvips/anonymouspro/"
                                     "fonts/map/dvips/anonymouspro/"
                                     "fonts/tfm/public/anonymouspro/"
                                     "fonts/truetype/public/anonymouspro/"
                                     "fonts/type1/public/anonymouspro/"
                                     "fonts/vf/public/anonymouspro/"
                                     "source/fonts/anonymouspro/"
                                     "tex/latex/anonymouspro/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1qfhsbq3ab72k9d023pa70vs1807djijg84l888nwsf3ylfwg7mf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/anonymouspro")
    (synopsis "Use AnonymousPro fonts with LaTeX")
    (description
     "The fonts are a monowidth set, designed for use by coders.  They appear
as a set of four TrueType, or Adobe Type 1 font files, and LaTeX support is
also provided.")
    (license license:lppl1.3c)))

(define-public texlive-antanilipsum
  (package
    (name "texlive-antanilipsum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/antanilipsum/"
                                     "source/latex/antanilipsum/"
                                     "tex/latex/antanilipsum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1yighrv3nxb266949m8sqvvl5yp8bhjq478h2abfrs1z0r65gpjr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/antanilipsum")
    (synopsis "Generate sentences in the style of @emph{Amici miei}")
    (description
     "This package is an italian blind text generator that outputs supercazzole,
mocking nonsense phrases from the movie series @emph{Amici Miei} (``My
friends'', in English), directed by Mario Monicelli.")
    (license license:lppl1.3c)))

(define-public texlive-antiqua
  (package
    (name "texlive-antiqua")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/antiqua/"
                                     "fonts/afm/urw/antiqua/"
                                     "fonts/map/dvips/antiqua/"
                                     "fonts/map/vtex/antiqua/"
                                     "fonts/tfm/urw/antiqua/"
                                     "fonts/type1/urw/antiqua/"
                                     "fonts/vf/urw/antiqua/"
                                     "tex/latex/antiqua/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0agnibj98zx2f2r40gqcfm2f1x06389wd2qrlf6b4pl8pgay42z2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/urw-antiqua")
    (synopsis "URW Antiqua condensed font, for use with TeX")
    (description
     "The package contains a copy of the Type 1 font URW Antiqua 2051 Regular
Condensed, with supporting files for use with (La)TeX.")
    (license license:gpl3+)))

(define-public texlive-antique-spanish-units
  (package
    (name "texlive-antique-spanish-units")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/generic/antique-spanish-units/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "09w95jv60qy5fc8b121f69b2bj8gj9cqz77x4rgp68xpnm125y0z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/antique-spanish-units")
    (synopsis "Short document about antique Spanish units")
    (description
     "This short document is about antique Spanish units used in Spain and
their colonies between the sixteenth and nineteenth centuries.  The next step
will be to develop a LaTeX package similar to @code{siunitx}.  The document
could be interesting for historians, economists, metrologists and others, as
a reference and detailed compendium about this old system of units.")
    (license license:cc-by4.0)))

(define-public texlive-antomega
  (package
    (name "texlive-antomega")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/omega/antomega/" "omega/ocp/antomega/"
                                     "omega/otp/antomega/"
                                     "source/lambda/antomega/"
                                     "tex/lambda/antomega/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "02pfjm9y33mjggn9w2lrk1fxfz3m72xgbvyvrq2iri9yf0hk33pf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-build
            ;; This phase is necessary because the build phase is reluctant to
            ;; generate "hyphen.cfg" since there is another one among the
            ;; inputs already.
            (lambda _
              (substitute* "source/lambda/antomega/antomega.ins"
                (("\\\\generateFile\\{hyphen\\.cfg\\}\\{t\\}")
                 "\\generateFile{hyphen.cfg}{f}")))))))
    (propagated-inputs (list texlive-omega))
    (home-page "https://ctan.org/pkg/antomega")
    (synopsis "Alternative language support for Omega and Lambda")
    (description
     "This package provides a language support package for Omega and Lambda.
This replaces the original Omega package for use with Lambda, and provides
extra facilities (including Babel-like language switching, which eases porting
of LaTeX documents to Lambda).")
    (license license:lppl)))

(define-public texlive-anufinalexam
  (package
    (name "texlive-anufinalexam")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/anufinalexam/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1v1k74vxidgxn5zzqz6v9zga468kcf7hwdrnvw44cd318221y396"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/anufinalexam")
    (synopsis "LaTeX document shell for ANU final exam")
    (description
     "This LaTeX document shell is created for the standard formatting of
final exams in the @acronym{ANU, Australian National University}.")
    (license license:gpl3+)))

(define-public texlive-anyfontsize
  (package
    (name "texlive-anyfontsize")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/anyfontsize/"
                                     "tex/latex/anyfontsize/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0wr4brhggmkb1rwzmcc2r5ygzqp6090z0bp3sfbarwvwz903wpdn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/anyfontsize")
    (synopsis "Select any font size in LaTeX")
    (description
     "The package allows the to user select any font size , even those sizes
that are not listed in the @file{.fd} file.  If such a size is requested,
LaTeX will search for and select the nearest listed size; @code{anyfontsize}
will then scale the font to the size actually requested.  Similar
functionality is available for the CM family, for the EC family, or for either
Computer Modern encoding; the present package generalises the facility.")
    (license license:lppl)))

(define-public texlive-aomart
  (package
    (name "texlive-aomart")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/aomart/"
                                     "doc/latex/aomart/"
                                     "doc/man/man1/aom-fullref.1"
                                     "doc/man/man1/aom-fullref.man1.pdf"
                                     "scripts/aomart/"
                                     "source/latex/aomart/"
                                     "tex/latex/aomart/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0lqy3ydm798iivirdm2gxdp0lfz2im74qiywq4zpmzcr56q6zyjh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:link-scripts #~(list "aom-fullref.pl")))
    (inputs (list perl))
    (home-page "https://ctan.org/pkg/aomart")
    (synopsis "Typeset articles for the @emph{Annals of Mathematics}")
    (description
     "The package provides a class for typesetting articles for the
@emph{Annals of Mathematics}.")
    (license license:lppl1.3+)))

(define-public texlive-apa
  (package
    (name "texlive-apa")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/apa/" "tex/latex/apa/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0slrvw7awva4d2zf7a6qyvh85qhrp8ml4ncdzjfizr5xz4ki6gs2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/apa")
    (synopsis "American Psychological Association format")
    (description
     "This package provides a LaTeX class to format text according to the
@emph{American Psychological Association Publication Manual} (5th ed.)
specifications for manuscripts or to the @acronym{APA, Americal Psychological
Association} journal look found in journals like the @emph{Journal of
Experimental Psychology} etc.  In addition, it provides regular LaTeX-like
output with a few enhancements and APA-motivated changes.  Note that the
@code{apa7} class (covering the 7th edition of the manual) and
@code{apa6} (covering the 6th edition of the manual) are now commonly in use.
Apacite, which used to work with this class, has been updated for use with
@code{apa6}.")
    (license license:lppl)))

(define-public texlive-apa6e
  (package
    (name "texlive-apa6e")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/apa6e/" "source/latex/apa6e/"
                                     "tex/latex/apa6e/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1dksjpdywwvyj9mfhaghsrf935p061gsnx4lf5s6pkv5nxn2pbln"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/apa6e")
    (synopsis "Format manuscripts to APA 6th edition guidelines")
    (description
     "This is a minimalist class file for formatting manuscripts in the style
described in the @acronym{APA, American Psychological Association} 6th edition
guidelines.  The @code{apa6} class provides better coverage of the
requirements.")
    (license license:bsd-2)))

(define-public texlive-apa7
  (package
    (name "texlive-apa7")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/apa7/" "source/latex/apa7/"
                                     "tex/latex/apa7/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kifjg61s16cn7zvmix2yqdz26spr7f9gsxkiv822qs3y337l6ry"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               ;; ".ins" file writes files to "./config" and "./build".
               ;; Create these directories first to prevent an error.
               (add-before 'build 'prepare-build
                 (lambda _
                   (for-each mkdir-p (list "build/config" "build/samples")))))))
    (home-page "https://ctan.org/pkg/apa7")
    (synopsis "Format documents in APA style (7th edition)")
    (description
     "This class formats documents in @acronym{APA, American Psychological
Association} style (7th Edition).  It provides a full set of facilities in
four different output modes (journal-like appearance, double-spaced
manuscript, double-spaced student manuscript, LaTeX-like document).  The class
can mask author identity for copies for use in masked peer review.  It is
a development of the @code{apa6} class.")
    (license license:lppl1.3c)))

(define-public texlive-apalike-ejor
  (package
    (name "texlive-apalike-ejor")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/apalike-ejor/"
                                     "doc/bibtex/apalike-ejor/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qq7c81hqi46596zw322w1qd8z341zksi5720f9s75nj1hmjj12g"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/apalike-ejor")
    (synopsis
     "BibTeX style file for the European Journal of Operational Research")
    (description
     "This package contains a BibTeX style file, @file{apalike-ejor.bst},
made to follow the European Journal of Operational Research reference style
guidelines.")
    (license license:lppl1.3c)))

(define-public texlive-apalike2
  (package
    (name "texlive-apalike2")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/apalike2/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0mn84ahwjg8gvw5lifvcrnch11q79d4bgw3nmhrh4v8slwgip5l7"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/apalike2")
    (synopsis "Bibliography style that approaches APA requirements")
    (description
     "This package is an adaptation of @code{apalike}, which is part of the
base BibTeX distribution.")
    (license license:knuth)))

(define-public texlive-apnum
  (package
    (name "texlive-apnum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/apnum/" "tex/generic/apnum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0s6nklimzbqc1z1vhgffzp76d1kl2shz1xkf119bck482rvn3x61"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/apnum")
    (synopsis "Arbitrary precision numbers implemented by TeX macros")
    (description
     "The basic operations (addition, subtraction, multiplication,
division, power to an integer) are implemented by TeX macros in this package.
Operands may be numbers with arbitrary numbers of digits; scientific notation
is allowed.  The expression scanner is also provided.")
    (license license:public-domain)))

(define-public texlive-apprendre-a-programmer-en-tex
  (package
    (name "texlive-apprendre-a-programmer-en-tex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/plain/apprendre-a-programmer-en-tex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05779lk9v849k712wfjv0mhyzahwpl4n892ydamfdc5yg05bsnyv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/apprendre-a-programmer-en-tex")
    (synopsis "@emph{Apprendre à programmer en TeX} book")
    (description
     "This book explains the basic concepts required for programming in TeX
and explains the programming methods, providing many examples.  The package
makes the compilable source code as well as the compiled PDF file accessible
to everyone.")
    (license license:lppl1.2+)))

(define-public texlive-apprends-latex
  (package
    (name "texlive-apprends-latex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/apprends-latex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xzy7svb2xz6bdfg0f1r3whwda118pl7qdwygx1l7h4d1vqm2rcq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/apprends-latex")
    (synopsis "@emph{Apprends LaTeX!} book")
    (description
     "@emph{Apprends LaTeX!} (``Learn LaTeX'', in English) is French
documentation for LaTeX beginners.")
    (license license:lppl)))

(define-public texlive-apxproof
  (package
    (name "texlive-apxproof")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/apxproof/"
                                     "source/latex/apxproof/"
                                     "tex/latex/apxproof/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1m4nvqxvnsllh25qmms6szcqqpb8wffk1z0v2p507fkgsdjz9jln"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/apxproof")
    (synopsis "Proofs in appendix")
    (description
     "The package makes it easier to write articles where proofs and other
material are deferred to the appendix.  The appendix material is written in
the LaTeX code along with the main text which it naturally complements, and it
is automatically deferred.  The package can automatically send proofs to the
appendix, can repeat in the appendix the theorem environments stated in the
main text, can section the appendix automatically based on the sectioning of
the main text, and supports a separate bibliography for the appendix
material.")
    (license license:lppl1.3+)))

(define-public texlive-arabi
  (package
    (name "texlive-arabi")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/arabi/"
                                     "fonts/afm/arabi/arabeyes/"
                                     "fonts/enc/dvips/arabi/"
                                     "fonts/map/dvips/arabi/"
                                     "fonts/tfm/arabi/arabeyes/"
                                     "fonts/tfm/arabi/farsiweb/"
                                     "fonts/type1/arabi/arabeyes/"
                                     "fonts/type1/arabi/farsiweb/"
                                     "tex/latex/arabi/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "19js5lw5r51n97gnf7ggvnvdkaamd0aagx73hnpyfzgzj9nb8pjr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arabi")
    (synopsis "(La)TeX support for Arabic and Farsi, compliant with Babel")
    (description
     "The package provides an Arabic and Farsi script support for TeX without
the need of any external pre-processor, and in a way that is compatible with
Babel.  The bi-directional capability supposes that the user has a TeX engine
that knows the four primitives @code{\\beginR}, @code{\\endR}, @code{\\beginL}
and @code{\\endL}.  That is the case in both the TeX--XeT and e-TeX engines.

Arabi will accept input in several 8-bit encodings, including UTF-8.  Arabi
can make use of a wide variety of Arabic and Farsi fonts, and provides one of
its own.  PDF files generated using Arabi may be searched, and text may be
copied from them and pasted elsewhere.")
    (license license:lppl)))

(define-public texlive-arabi-add
  (package
    (name "texlive-arabi-add")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/arabi-add/"
                                     "tex/latex/arabi-add/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1lrr3vr9sv554bvad70rc32x0jgsam2jaflgxw0nid0629rmigdj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arabi-add")
    (synopsis
     "Using @code{hyperref} and @code{bookmark} packages with Arabic and Farsi languages")
    (description
     "This package takes advantage of some of the possibilities that
@code{hyperref} and @code{bookmark} packages offer when you create a table of
contents for Arabic texts created by the @code{arabi} package.")
    (license license:lppl1.3)))

(define-public texlive-arabic-book
  (package
    (name "texlive-arabic-book")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/xelatex/arabic-book/"
                                     "tex/xelatex/arabic-book/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "070agzwpgdfvjqn2klxw2m1bavy0l9b9pqqikc7il3wx107m4ni1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arabic-book")
    (synopsis "Arabic @code{book} class")
    (description
     "This document class provides both Arabic and English support for TeX and
LaTeX.  Input may be in ASCII transliteration or other encodings
(including UTF-8), and output may be Arabic, Hebrew, or any of several
languages that use the Arabic script, as can be specified by the Polyglossia
package.  The Arabic font is presently available in any Arabic fonts style.
In order to use Amiri font style, the user needs to install the amiri package.
This document class runs with the XeTeX engine.  PDF files generated using
this class can be searched, and text can be copied from them and pasted
elsewhere.")
    (license license:lppl1.3+)))

(define-public texlive-arabluatex
  (package
    (name "texlive-arabluatex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/lualatex/arabluatex/"
                                     "source/lualatex/arabluatex/"
                                     "tex/lualatex/arabluatex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1g1ai0inai3jniq49avfswyaysxlk1yp6n97l94pq8v3vszx0k8i"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arabluatex")
    (synopsis "ArabTeX for LuaLaTeX")
    (description
     "This package provides for LuaLaTeX an ArabTeX-like interface to generate
Arabic writing from an ascii transliteration.  It is particularly well-suited
for complex documents such as technical documents or critical editions where
a lot of left-to-right commands intertwine with Arabic writing.
@code{arabluatex} is able to process any ArabTeX input notation.  Its output
can be set in the same modes of vocalization as ArabTeX, or in different roman
transliterations.  It further allows many typographical refinements.  It will
eventually interact with some other packages yet to come to produce from
@file{.tex} source files, in addition to printed books, TEI XML compliant
critical editions and/or lexicons that can be searched, analyzed and
correlated in various ways.")
    (license (list license:gpl3+ license:cc-by-sa4.0))))

(define-public texlive-arabtex
  (package
    (name "texlive-arabtex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/arabtex/"
                                     "fonts/map/dvips/arabtex/"
                                     "fonts/source/public/arabtex/"
                                     "fonts/tfm/public/arabtex/"
                                     "fonts/type1/public/arabtex/"
                                     "tex/latex/arabtex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1z1iaj59lawilydp31d3zxbd5jbny4szpdnkbwl6lrnki4c0s51c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/arabtex")
    (synopsis "Macros and fonts for typesetting Arabic")
    (description
     "ArabTeX is a package extending the capabilities of TeX and LaTeX to
generate Arabic and Hebrew text.  Input may be in ASCII transliteration or
other encodings (including UTF-8); output may be Arabic, Hebrew, or any of
several languages that use the Arabic script.  ArabTeX consists of a TeX macro
package and Arabic and Hebrew fonts (provided both in Metafont format and
Adobe Type 1).  The Arabic font is presently only available in the Naskhi
style.  ArabTeX will run with Plain TeX and also with LaTeX.")
    (license license:lppl)))

(define-public texlive-arara
  (package
    (name "texlive-arara")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/arara.1"
                                     "doc/man/man1/arara.man1.pdf"
                                     "doc/support/arara/" "scripts/arara/"
                                     "source/support/arara/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0w0c9vbyd0gs9nm7z8aphszrf6nj3ckn7afixaqlx2pbz7bscvn8"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-inputs "icedtea")))
    (build-system texlive-build-system)
    (arguments
     (list
      #:link-scripts #~(list "arara.sh")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'locate-java
            (lambda* (#:key inputs #:allow-other-keys)
              (substitute* "scripts/arara/arara.sh"
                (("java") (search-input-file inputs "/bin/java"))))))))
    (inputs (list icedtea))
    (home-page "https://ctan.org/pkg/arara")
    (synopsis "Automation of LaTeX compilation")
    (description
     "Arara is comparable with other well-known compilation tools like
@command{latexmk} and @command{rubber}.  The key difference is that Arara
determines its actions from metadata in the source code, rather than relying
on indirect resources, such as log file analysis.  Arara requires a Java
virtual machine.")
    (license license:bsd-3)))

(define-public texlive-aramaic-serto
  (package
    (name "texlive-aramaic-serto")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/aramaic-serto/"
                                     "fonts/afm/public/aramaic-serto/"
                                     "fonts/map/dvips/aramaic-serto/"
                                     "fonts/source/public/aramaic-serto/"
                                     "fonts/tfm/public/aramaic-serto/"
                                     "fonts/type1/public/aramaic-serto/"
                                     "tex/latex/aramaic-serto/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "154bicknhsj8nypnyzr999wj34xli0kfpic9vym805abxbpwrmsw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/aramaic-serto")
    (synopsis "Fonts and LaTeX for Syriac written in Serto")
    (description
     "This package enables (La)TeX users to typeset words or phrases (e-TeX
extensions are needed) in Syriac (Aramaic) using the Serto-alphabet.  The
package includes a preprocessor written in Python in order to deal with
right-to-left typesetting for those who do not want to use e-LaTeX and to
choose the correct letter depending on word context (initial/medial/final
form).")
    (license license:lppl1.3+)))

(define-public texlive-archaeologie
  (package
    (name "texlive-archaeologie")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bib/archaeologie/"
                                     "doc/latex/archaeologie/"
                                     "source/latex/archaeologie/"
                                     "tex/latex/archaeologie/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17wffhg034ja4iy6sikyhfh97gvjhk2020bpk3zqmh13wkh2p6cg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/archaeologie")
    (synopsis "Citation-style for the German Archaeological Institute")
    (description
     "This citation-style covers the citation and bibliography rules of the
German Archaeological Institute (DAI).  Various options are available to
change and adjust the outcome according to one's own preferences.")
    (license license:lppl1.3+)))

(define-public texlive-archaic
  (package
    (name "texlive-archaic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/archaic/"
                                     "fonts/afm/public/archaic/"
                                     "fonts/map/dvips/archaic/"
                                     "fonts/source/public/archaic/"
                                     "fonts/tfm/public/archaic/"
                                     "fonts/type1/public/archaic/"
                                     "source/fonts/archaic/"
                                     "tex/latex/archaic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0a6lp4w9b76q1w30a58205b1n7jf61n58ps2hdxrhpg9nxqwy1w0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/archaic")
    (synopsis "Collection of archaic fonts")
    (description
     "The collection contains fonts to represent Aramaic, Cypriot,
Etruscan, Greek of the 6th and 4th centuries BCE, Egyptian hieroglyphics,
Linear A, Linear B, Nabatean old Persian, the Phaistos disc, Phoenician,
proto-Semitic, runic, South Arabian Ugaritic and Viking scripts.  The bundle
also includes a small font for use in phonetic transcription of the archaic
writings.")
    (license license:lppl)))

(define-public texlive-archivo
  (package
    (name "texlive-archivo")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/archivo/"
                                     "fonts/enc/dvips/archivo/"
                                     "fonts/map/dvips/archivo/"
                                     "fonts/opentype/public/archivo/"
                                     "fonts/tfm/public/archivo/"
                                     "fonts/type1/public/archivo/"
                                     "fonts/vf/public/archivo/"
                                     "tex/latex/archivo/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wdvr7ch2ydj41yd0xniwppb5wh1m9w97d839fhmp21lgyhss0jl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/archivo")
    (synopsis "Archivo font face with support for LaTeX and pdfLaTeX")
    (description
     "This package provides the Archivo family of fonts designed by
Omnibus-Type, with support for LaTeX and pdfLaTeX.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-argumentation
  (package
    (name "texlive-argumentation")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/argumentation/"
                              "tex/latex/argumentation/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "13lr0g5wiwbgdzl79hwcn7i9nqas0b8xj5b9w0mmy53zqn5gjzmi"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/argumentation")
    (synopsis "Create abstract argumentation frameworks via TikZ")
    (description
     "This package provides commands for creating abstract argumentation
 frameworks via TikZ.")
    (license license:lppl1.3c)))

(define-public texlive-arimo
  (package
    (name "texlive-arimo")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/arimo/"
                                     "fonts/enc/dvips/arimo/"
                                     "fonts/map/dvips/arimo/"
                                     "fonts/tfm/google/arimo/"
                                     "fonts/truetype/google/arimo/"
                                     "fonts/type1/google/arimo/"
                                     "fonts/vf/google/arimo/"
                                     "tex/latex/arimo/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06m23slsvcp0pm2ds2rxmgj0qrrvy8qknda4yjhvcz9almwb4syx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arimo")
    (synopsis "Arimo sans serif fonts with LaTeX support")
    (description
     "The Arimo family, designed by Steve Matteson, is a refreshing sans serif
design which is metrically compatible with Arial.")
    (license license:asl2.0)))

(define-public texlive-armtex
  (package
    (name "texlive-armtex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/armenian/"
                                     "fonts/afm/public/armenian/"
                                     "fonts/map/dvips/armenian/"
                                     "fonts/source/public/armenian/"
                                     "fonts/tfm/public/armenian/"
                                     "fonts/type1/public/armenian/"
                                     "tex/latex/armenian/"
                                     "tex/plain/armenian/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00y0kb6kmfa8d2sqsya93b312170p3vgcd0q9b9l96vys45npx8n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/armtex")
    (synopsis "System for writing in Armenian with TeX and LaTeX")
    (description
     "ArmTeX is a system for typesetting Armenian text with Plain TeX or LaTeX(2e).
It may be used with input: from a standard Latin keyboard without any special
encoding or support for Armenian letters, from any keyboard which uses an
encoding that has Armenian letters in the second half (characters 128-255) of
the extended ASCII table (for example ArmSCII8 Armenian standard), or from an
Armenian keyboard using UTF-8 encoding.")
    (license license:lppl)))

(define-public texlive-around-the-bend
  (package
    (name "texlive-around-the-bend")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/around-the-bend/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00nc6kx4ylz9g6vmjlgyjvmy1ls86pjblbcpm5z40wh5xl7l184f"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/around-the-bend")
    (synopsis "Typeset exercises in TeX, with answers")
    (description
     "This is a typeset version of the files of the @code{aro-bend}, plus
three extra questions (with their answers) that Michael Downes didn't manage
to get onto CTAN.")
    (license license:lppl)))

(define-public texlive-arphic
  (package
    (name "texlive-arphic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/arphic/"
                                     "dvips/arphic/"
                                     "fonts/afm/arphic/bkaiu/"
                                     "fonts/afm/arphic/bsmiu/"
                                     "fonts/afm/arphic/gbsnu/"
                                     "fonts/afm/arphic/gkaiu/"
                                     "fonts/map/dvips/arphic/"
                                     "fonts/tfm/arphic/bkaimp/"
                                     "fonts/tfm/arphic/bkaiu/"
                                     "fonts/tfm/arphic/bsmilp/"
                                     "fonts/tfm/arphic/bsmiu/"
                                     "fonts/tfm/arphic/gbsnlp/"
                                     "fonts/tfm/arphic/gbsnu/"
                                     "fonts/tfm/arphic/gkaimp/"
                                     "fonts/tfm/arphic/gkaiu/"
                                     "fonts/type1/arphic/bkaiu/"
                                     "fonts/type1/arphic/bsmiu/"
                                     "fonts/type1/arphic/gbsnu/"
                                     "fonts/type1/arphic/gkaiu/"
                                     "fonts/vf/arphic/bkaimp/"
                                     "fonts/vf/arphic/bsmilp/"
                                     "fonts/vf/arphic/gbsnlp/"
                                     "fonts/vf/arphic/gkaimp/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1sqsgw9xjgd3ss2q7hwnb3vv38sg7l9i8bvsy79m5hyg29wn973x"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arphic")
    (synopsis "Arphic (Chinese) font packages")
    (description
     "These are font bundles for the Chinese Arphic fonts which work with the
CJK package.  TrueType versions of these fonts for use with XeLaTeX and
LuaLaTeX are provided by the @code{arphic-ttf} package.")
    ;; The files are generated from `chinese-fonts-truetype', located in
    ;; non-GNU FTP site, using the license below.
    (license license:arphic-1999)))

(define-public texlive-arphic-ttf
  (package
    (name "texlive-arphic-ttf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/arphic-ttf/"
                                     "fonts/truetype/public/arphic-ttf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "141hsz7lhqpbz3k10kl57bg21yslrb562alk1a6q129i56zb7jby"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arphic-ttf")
    (synopsis "TrueType version of Chinese Arphic fonts")
    (description
     "This package provides TrueType versions of the Chinese Arphic fonts for
use with XeLaTeX and LuaLaTeX.  Type1 versions of these fonts, for use with
pdfLaTeX and the @code{cjk} package, are provided by the @code{arphic}
package.")
    (license license:arphic-1999)))

(define-public texlive-arsclassica
  (package
    (name "texlive-arsclassica")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/arsclassica/"
                                     "tex/latex/arsclassica/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0cxmp68na3ww565r7ixqbhsy2mp063wwq5p6gwvq5fdrk34ci2a5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arsclassica")
    (synopsis "Different view of the ClassicThesis package")
    (description
     "The package changes some typographical points of the ClassicThesis
style.  It enables the user to reproduce the look of the guide @emph{The art
of writing with LaTeX} (the web page is in Italian).")
    (license license:lppl)))

(define-public texlive-arsenal
  (package
    (name "texlive-arsenal")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/fonts/arsenal/"
                              "fonts/opentype/public/arsenal/"
                              "source/fonts/arsenal/"
                              "tex/latex/arsenal/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1gm0vrcs7dzv4lryaw0phgzhy98sd9y33vwfzlm9a8149fjagh6l"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arsenal")
    (synopsis "Open Type font by Andrij Shevchenko")
    (description
     "In 2011 Andrij's typeface became a winner of the Ukrainian Type Design
Competition ``Mystetsky Arsenal'' in which three main criteria were sought
for: being zeitgeist, practical, and Ukrainian.  Andrij's winning entry was
crowned Arsenal and made publicly available.  Arsenal is a semi-grotesque with
traditional forms.  It is primarily designed for body text and intended for
various professional communication.  Its special qualities of letter shapes
and subtle contrast modulation articulate grace and expressiveness.  Arsenal's
somewhat lyrical sentiment abides to the Ukrainian nature of the font.  This
package provides the fonts and LaTeX support for them with matching math.  It
needs LuaLaTeX or XeLaTeX.")
    (license (list license:silofl1.1 license:lppl1.3c))))

(define-public texlive-articleingud
  (package
    (name "texlive-articleingud")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/articleingud/"
                                     "source/latex/articleingud/"
                                     "tex/latex/articleingud/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1iprjpjr3yav8afmgirrfmdipxb1fmdwa3b0ss5y8l7wspijmmp8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/articleingud")
    (synopsis "LaTeX class for articles published in INGENIERIA review")
    (description
     "The package provides a class for articles published in INGENIERIA
review.  This class is derived from the standard LaTeX class @code{article}.")
    (license license:lppl1.2+)))

(define-public texlive-arvo
  (package
    (name "texlive-arvo")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/arvo/"
                                     "fonts/enc/dvips/arvo/"
                                     "fonts/map/dvips/arvo/"
                                     "fonts/tfm/public/arvo/"
                                     "fonts/truetype/public/arvo/"
                                     "fonts/vf/public/arvo/"
                                     "tex/latex/arvo/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1dc52n81wxz06vfbxzmfdp9j6a217v8yf8bczqlnx5vgzy8wlmi3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/arvo")
    (synopsis "Arvo font face with support for LaTeX and pdfLaTeX")
    (description
     "This package provides the Arvo family of fonts, designed by Anton
Koovit, with support for LaTeX and pdfLaTeX.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-asaetr
  (package
    (name "texlive-asaetr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/asaetr/" "doc/latex/asaetr/"
                                     "tex/latex/asaetr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1yms34l9nx151vgjrjsa94801q0bgp2khjgk933iy1bpkmm7jf2d"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asaetr")
    (synopsis "Transactions of the ASAE")
    (description
     "This package provides a class and BibTeX style for submissions to the
@emph{Transactions of the American Society of Agricultural Engineers}.  Also
included is the Metafont source of a slanted Computer Modern Caps and Small
Caps font.")
    (license license:public-domain)))

(define-public texlive-asapsym
  (package
    (name "texlive-asapsym")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/asapsym/"
                                     "fonts/opentype/omnibus-type/asapsym/"
                                     "source/fonts/asapsym/"
                                     "tex/generic/asapsym/"
                                     "tex/latex/asapsym/"
                                     "tex/plain/asapsym/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "038adcagkzy1bj5y9i5a4i5kbyr3wg7w0lfhbbg9z7glqmivilic"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asapsym")
    (synopsis "Using the ASAP Symbol font with LaTeX and Plain TeX")
    (description
     "The package provides macros (usable with LaTeX or Plain TeX) for using
the ASAP Symbol font, which is also included.  The font is distributed in
OpenType format, and makes extensive use of OpenType features.  Therefore, at
this time, only XeTeX and LuaTeX are supported.")
    (license license:lppl1.3+)))

(define-public texlive-ascelike
  (package
    (name "texlive-ascelike")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/ascelike/" "doc/latex/ascelike/"
                                     "tex/latex/ascelike/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14n4z4agsnnvms51r142q0r5bv9yqp4gchrr7zvakkp2r3232dai"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ascelike")
    (synopsis "Bibliography style for the ASCE")
    (description
     "This package provides a document class and bibliographic style that
prepares documents in the style required by the @acronym{ASCE, American
Society of Civil Engineers}.  These are unofficial files, not sanctioned by
that organization.")
    (license license:lppl)))

(define-public texlive-ascii-chart
  (package
    (name "texlive-ascii-chart")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/support/ascii-chart/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1m2x7iwz80plq6hbhp9xia6bp8wbi03iifs0pkr7ji3gn8hqgh1r"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ascii-chart")
    (synopsis "ASCII wall chart")
    (description
     "This package contains an ASCII wall chart.  The document may be
converted between Plain TeX and LaTeX by a simple editing action.")
    (license license:lppl1.3+)))

(define-public texlive-ascii-font
  (package
    (name "texlive-ascii-font")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/ascii-font/"
                                     "fonts/map/dvips/ascii-font/"
                                     "fonts/tfm/public/ascii-font/"
                                     "fonts/type1/public/ascii-font/"
                                     "source/fonts/ascii-font/"
                                     "tex/latex/ascii-font/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0cbzvbszxyv552z1274m3bf5k40xr8w12da17db0w2w02y47nyq5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ascii-font")
    (synopsis "Use the ASCII font in LaTeX")
    (description
     "The package provides glyph and font access commands so that LaTeX users
can use the ASCII glyphs in their documents.  The ASCII font is encoded
according to the IBM PC Code Page 437 C0 Graphics.")
    (license license:lppl)))

(define-public texlive-ascmac
  (package
    (name "texlive-ascmac")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ascmac/"
                                     "fonts/map/dvips/ascmac/"
                                     "fonts/source/public/ascmac/"
                                     "fonts/tfm/public/ascmac/"
                                     "fonts/type1/public/ascmac/"
                                     "source/latex/ascmac/"
                                     "tex/latex/ascmac/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1d388nqyjipqsgd0pd4zrskaf3wddacg6rrmy3q1n8pvmna4gmz1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/ascmac")
    (synopsis "Boxes and picture macros with Japanese vertical writing support")
    (description
     "The bundle provides boxes and picture macros with Japanese vertical
writing support.  It uses only native picture macros and fonts for drawing
boxes and is thus driver independent.")
    (license license:bsd-3)))

(define-public texlive-asmeconf
  (package
    (name "texlive-asmeconf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/asmeconf/" "doc/latex/asmeconf/"
                                     "tex/latex/asmeconf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0y3s4qvpp1dsh1k30xmwcjcjk2x12w9809wqfr1dpwx64abcf9c3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asmeconf")
    (synopsis "LaTeX template for ASME conference papers")
    (description
     "The @code{asmeconf} class provides a LaTeX template for ASME conference
papers, following ASME's guidelines for margins, fonts, headings, captions,
and reference formats as of 2022.  This LaTeX template is intended to be used
with the @file{asmeconf.bst} BibTeX style, for reference formatting, which is
part of this distribution.  The code is compatible with pdfLaTeX or LuaLaTeX.

This LaTeX template is not a publication of ASME.")
    (license license:expat)))

(define-public texlive-asmejour
  (package
    (name "texlive-asmejour")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/asmejour/" "doc/latex/asmejour/"
                                     "tex/latex/asmejour/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1s3x0g7hjjrdrxk1sk35j27i3w0lgkk2k10hfngs5jk1qkakbn1d"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asmejour")
    (synopsis "Template for ASME journal papers")
    (description
     "The @code{asmejour} class provides a template to format preprints
submitted to ASME journals.  The layout and reference formats closely follow
the style that is currently being used for published papers.  The class is
intended to be used with the @file{asmejour.bst} BibTeX style, which is part
of this distribution. The class is compatible with pdfLaTeX or LuaLaTeX.

This package is not a publication of ASME.")
    (license license:expat)))

(define-public texlive-aspectratio
  (package
    (name "texlive-aspectratio")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/aspectratio/"
                                     "fonts/map/dvips/aspectratio/"
                                     "fonts/source/public/aspectratio/"
                                     "fonts/tfm/public/aspectratio/"
                                     "fonts/type1/public/aspectratio/"
                                     "tex/latex/aspectratio/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1vm1aj4mm2rn2p032nakg16gmyzv43149h2v1l18dw72sayqqg2s"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/aspectratio")
    (synopsis "Capital @samp{A} and capital @samp{R} ligature for Aspect Ratio")
    (description
     "The package provides fonts (both as Adobe Type 1 format, and as Metafont
source) for the AR symbol (for Aspect Ratio) used by aeronautical scientists
and engineers.  Note that the package supersedes the package @code{ar}.")
    (license license:lppl)))

(define-public texlive-asternote
  (package
    (name "texlive-asternote")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/asternote/"
                                     "tex/latex/asternote/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1a53pmw237lm1yjx55qwq3fq6jq34vlw03w62vrq3ksw9346dg5r"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asternote")
    (synopsis "Annotation symbols enclosed in square brackets with an asterisk")
    (description
     "This LaTeX package can output annotation symbols enclosed in square
brackets and marked with an asterisk.")
    (license license:expat)))

(define-public texlive-astro
  (package
    (name "texlive-astro")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/astro/"
                                     "fonts/source/public/astro/"
                                     "fonts/tfm/public/astro/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pdmn93wiarwplha6r59ap9nvhdm9ir4aa4az93w3b189g316g2c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/astro")
    (synopsis "Astronomical (planetary) symbols")
    (description
     "Astrosym is a font containing astronomical symbols, including those used
for the planets, four planetoids, the phases of the moon, the signs of the
zodiac, and some additional symbols.  The font is distributed as Metafont
source.")
    (license license:lppl)))

(define-public texlive-asy-overview
  (package
    (name "texlive-asy-overview")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/asy-overview/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1ykzhy85gca230jf5w4c2i4kxjdvx3phbqa1ci6vv0v2f2lnb9kj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asy-overview")
    (synopsis
     "Overview of the Asymptote language for drawing mathematical graphics")
    (description
     "Asymptote is a programming language for creating mathematical graphics.
This document gives you a quick overview, illustrating with a few familiar
Calculus examples.  Readers can work through it in a couple of hours to get
a feel for the system's strengths, and if they are interested then go on to
a full tutorial or the official reference.")
    (license license:cc-by-sa4.0)))

(define-public texlive-asymptote-by-example-zh-cn
  (package
    (name "texlive-asymptote-by-example-zh-cn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/support/asymptote-by-example-zh-cn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0854pbh3rm21rzzbhai7xijzw26qawffj4wd3mrwbva1406106ff"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asymptote-by-example-zh-cn")
    (synopsis "Asymptote by example")
    (description "This is an Asymptote tutorial written in Simplified
Chinese.")
    (license license:gpl3+)))

(define-public texlive-asymptote-faq-zh-cn
  (package
    (name "texlive-asymptote-faq-zh-cn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/support/asymptote-faq-zh-cn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0z09f5as68dz66dlj370xfx593bv50ihxx10zxmm9lk27ymr52hb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asymptote-faq-zh-cn")
    (synopsis "Asymptote FAQ (Chinese translation)")
    (description "This is a Chinese translation of the Asymptote FAQ.")
    (license license:lppl)))

(define-public texlive-asymptote-manual-zh-cn
  (package
    (name "texlive-asymptote-manual-zh-cn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/support/asymptote-manual-zh-cn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "047l3bh8vvk185wkazyfw4dm6893xpa2ksn5vfc57r5rv4x16wmb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/asymptote-manual-zh-cn")
    (synopsis "Chinese translation of the Asymptote manual")
    (description
     "This is an (incomplete, simplified) Chinese translation of the Asymptote
manual.")
    (license license:lgpl3+)))

(define-public texlive-atableau
  (package
    (name "texlive-atableau")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/atableau/"
                              "tex/latex/atableau/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0h2sx9h5ra9bp109lvs1r22abxhjdybsdb9196f81i66vfdvc6zx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/atableau")
    (synopsis "LaTeX package for symmetric group combinatorics")
    (description
     "This is a LaTeX package for symmetric group combinatorics,
with commands for Young diagrams, tableaux, tabloids, skew tableaux, shifted
tableaux, ribbon tableaux, multitableaux, and abacuses.")
    (license license:lppl1.3c)))

(define-public texlive-atkinson
  (package
    (name "texlive-atkinson")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/atkinson/"
                                     "fonts/enc/dvips/atkinson/"
                                     "fonts/map/dvips/atkinson/"
                                     "fonts/opentype/public/atkinson/"
                                     "fonts/tfm/public/atkinson/"
                                     "fonts/type1/public/atkinson/"
                                     "fonts/vf/public/atkinson/"
                                     "tex/latex/atkinson/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0vas4g5mlkrgnnl5q0i4y313cjl0xa7b89jy7z93m9zc64jrk1fi"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/atkinson")
    (synopsis "Support for the Atkinson Hyperlegible family of fonts")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX and LuaLaTeX support for
the Atkinson Hyperlegible family of fonts.  What makes it different from
traditional typography design is that it focuses on letterform distinction to
increase character recognition, ultimately improving readability.")
    (license
     (list license:lppl
           (license:fsdg-compatible
            "file://doc/Atkinson-Hyperlegible-Font-License-2020-1104.pdf")))))

(define-public texlive-aucklandthesis
  (package
    (name "texlive-aucklandthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/aucklandthesis/"
                                     "tex/latex/aucklandthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1bdm6crn17a0rrszx28r6bn8v9w6z4rwfz2r8avd5fc3byl5dix8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aucklandthesis")
    (synopsis "Memoir-based class for formatting University of Auckland theses")
    (description
     "This package provides a @code{memoir}-based class for formatting
University of Auckland masters and doctors thesis dissertations in any
discipline.  The title page does not handle short dissertations for
diplomas.")
    (license license:lppl1.3+)))

(define-public texlive-augie
  (package
    (name "texlive-augie")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/augie/"
                                     "fonts/afm/public/augie/"
                                     "fonts/map/dvips/augie/"
                                     "fonts/tfm/public/augie/"
                                     "fonts/type1/public/augie/"
                                     "fonts/vf/public/augie/"
                                     "tex/latex/augie/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fvl04lcvi4vsx547gx349dbld4hf7k2j08adqlnszs23p95yj3f"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/augie")
    (synopsis "Calligraphic font for typesetting handwriting")
    (description
     "This package provides a calligraphic font for simulating American-style
informal handwriting.  The font is distributed in Adobe Type 1 format.")
    (license license:lppl)))

(define-public texlive-auncial-new
  (package
    (name "texlive-auncial-new")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/auncial-new/"
                                     "fonts/afm/public/auncial-new/"
                                     "fonts/map/dvips/auncial-new/"
                                     "fonts/tfm/public/auncial-new/"
                                     "fonts/type1/public/auncial-new/"
                                     "source/fonts/auncial-new/"
                                     "tex/latex/auncial-new/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1644ank81l1qlijmld88k1qq3qk4x6an0qvyqz1da2dgwdhjap00"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/auncial-new")
    (synopsis "Artificial Uncial font and LaTeX support macros")
    (description
     "The @code{auncial-new} bundle provides packages and fonts for a script
based on the Artificial Uncial manuscript book-hand used between the 6th &
10th century AD.  The script consists of minuscules and digits, with some
appropriate period punctuation marks.  Both normal and bold versions are
provided, and the font is distributed in Adobe Type 1 format.")
    (license license:lppl)))

(define-public texlive-aurical
  (package
    (name "texlive-aurical")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/aurical/"
                                     "fonts/afm/public/aurical/"
                                     "fonts/map/dvips/aurical/"
                                     "fonts/source/public/aurical/"
                                     "fonts/tfm/public/aurical/"
                                     "fonts/type1/public/aurical/"
                                     "tex/latex/aurical/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1jaqr4gdywnjn7hclinjyydndxsa1f2j7382ygl0jnj35kxz0pkr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/aurical")
    (synopsis "Calligraphic fonts for use with LaTeX in T1 encoding")
    (description
     "The package that implements a set (AuriocusKalligraphicus) of three
calligraphic fonts derived from the author's handwriting in Adobe Type
1 Format, T1 encoding for use with LaTeX: Auriocus Kalligraphicus; Lukas
Svatba; and Jana Skrivana.  Each font features old style digits
and (machine-generated) boldface and slanted versions.  A variant of Lukas
Svatba offers a long @samp{s}.")
    (license license:lppl)))

(define-public texlive-authordate
  (package
    (name "texlive-authordate")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/authordate/"
                                     "doc/bibtex/authordate/"
                                     "tex/latex/authordate/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0d3cd63b80vfd8vzbpn6dnhy47fpq5rxm95i52id2j6606gi1skf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/authordate")
    (synopsis "Author/date style citation styles")
    (description
     "The bundle provides four BibTeX styles (@code{authordate1}, ...,
@code{authordate4}), and a LaTeX package, for citation in author/date style.
The BibTeX styles differ in how they format names and titles; one of them is
necessary for the LaTeX package to work.")
    (license license:knuth)))

(define-public texlive-auto-pst-pdf
  (package
    (name "texlive-auto-pst-pdf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/auto-pst-pdf/"
                                     "source/latex/auto-pst-pdf/"
                                     "tex/latex/auto-pst-pdf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1lpjwqd0rhdzz3kywl54pjlpj1qsj7kflj0336vj2zb20rxl0hqp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-ifplatform texlive-iftex texlive-xkeyval))
    (home-page "https://ctan.org/pkg/auto-pst-pdf")
    (synopsis "Wrapper for @code{pst-pdf} (with some PSfrag features)")
    (description
     "The package uses @samp{--shell-escape} to execute @code{pst-pdf} when necessary.
Wrappers are provided for various psfrag-related features so that Matlab
figures via @code{laprint}, Mathematica figures via MathPSfrag, and regular
PSfrag figures can all be input consistently and easily.")
    (license license:lppl1.3c)))

(define-public texlive-autoaligne
  (package
    (name "texlive-autoaligne")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/autoaligne/"
                                     "tex/generic/autoaligne/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "138d030zgzpdwqxx55gjw35y0c3v794fpa3c6znc9kv1vj8cs9d3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/autoaligne")
    (synopsis "Align terms and members in math expressions")
    (description
     "This package aligns terms and members between lines containing
math expressions.")
    (license license:lppl1.3c)))

(define-public texlive-autobreak
  (package
    (name "texlive-autobreak")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/autobreak/"
                                     "source/latex/autobreak/"
                                     "tex/latex/autobreak/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xw4m1wyv17xi6yjfcamb1gjdwzbhrjjdpyc8fydq1qbki1yvv1k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/autobreak")
    (synopsis "Simple line breaking of long formulae")
    (description
     "This package implements a simple mechanism of line or page breaking
within the @code{align} environment of the @code{amsmath} package; new line
characters are considered as possible candidates for the breaks and the
package tries to put breaks at adequate places.  It is suitable for
computer-generated long formulae with many terms.")
    (license license:lppl1.3+)))

(define-public texlive-autosp
  (package
    (name "texlive-autosp")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/autosp/"
                                     "doc/man/man1/autosp.1"
                                     "doc/man/man1/autosp.man1.pdf"
                                     "doc/man/man1/tex2aspc.1"
                                     "doc/man/man1/tex2aspc.man1.pdf"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1n9vpb0cmi0b9gbi3f70zafvhj9znvfhd9614xcn49jwss2bwdk9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-autosp-bin))
    (home-page "https://ctan.org/pkg/autosp")
    (synopsis
     "Preprocessor generating note-spacing commands for MusiXTeX scores")
    (description
     "This program simplifies the creation of MusiXTeX scores by converting
(non-standard) commands of the form @code{\\anotes ... \\en} into one or more
conventional note-spacing commands, as determined by the note values
themselves, with @code{\\sk} spacing commands inserted as necessary.  The
coding for an entire measure can be entered one part at a time, without
concern for note-spacing changes within the part or spacing requirements of
other parts.")
    (license license:gpl2+)))

(define-public texlive-autosp-bin
  (package
    (inherit texlive-bin)
    (name "texlive-autosp-bin")
    (source
     (origin
       (inherit (package-source texlive-source))
       (modules '((guix build utils)
                  (ice-9 ftw)))
       (snippet
        #~(let ((delete-other-directories
                 (lambda (root dirs)
                   (with-directory-excursion root
                     (for-each
                      delete-file-recursively
                      (scandir "."
                               (lambda (file)
                                 (and (not (member file (append '("." "..") dirs)))
                                      (eq? 'directory (stat:type (stat file)))))))))))
            (delete-other-directories "libs" '())
            (delete-other-directories "utils" '("autosp"))
            (delete-other-directories "texk" '())))))
    (arguments
     (substitute-keyword-arguments (package-arguments texlive-bin)
       ((#:configure-flags flags)
        #~(cons "--enable-autosp" (delete "--enable-web2c" #$flags)))
       ((#:phases phases)
        #~(modify-phases #$phases
            (replace 'check
              (lambda* (#:key tests? #:allow-other-keys)
                (when tests?
                  (with-directory-excursion "utils/autosp"
                    (invoke "make" "check")))))
            (replace 'install
              (lambda* (#:key inputs native-inputs #:allow-other-keys)
                (with-directory-excursion "utils/autosp"
                  (invoke "make" "install"))))))))
    (native-inputs '())
    (inputs '())
    (propagated-inputs '())
    (home-page (package-home-page texlive-autosp))
    (synopsis "Binaries for @code{texlive-autosp}")
    (description
     "This package provides binaries for @code{texlive-autosp}.")
    (license (package-license texlive-autosp))))

(define-public texlive-axodraw2
  (package
    (name "texlive-axodraw2")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/axodraw2/"
                                     "doc/man/man1/axohelp.1"
                                     "doc/man/man1/axohelp.man1.pdf"
                                     "tex/latex/axodraw2/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0r52al3ya302ps8hnn1hqi8nhvva46p81dv68qp1abfpaax0s227"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-axodraw2-bin))
    (home-page "https://ctan.org/pkg/axodraw2")
    (synopsis "Feynman diagrams in a LaTeX document")
    (description
     "This package defines macros for drawing Feynman graphs in LaTeX documents.
It is an important update of the @code{axodraw} package, but since it is not
completely backwards compatible, we have given the style file a changed name.

Many new features have been added, with new types of line, and much more
flexibility in their properties.  In addition, it is now possible to use
@code{axodraw2} with pdfLaTeX, as well as with the LaTeX-dvips method.
However with pdfLaTeX (and also LuaLaTeX and XeLaTeX), an external program,
@command{axohelp}, is used to perform the geometrical calculations needed for
the pdf code inserted in the output file.  The processing involves a run of
@command{pdflatex}, a run of @command{axohelp}, and then another run of
@command{pdflatex}.")
    (license license:gpl3+)))

(define-public texlive-axodraw2-bin
  (package
    (inherit texlive-bin)
    (name "texlive-axodraw2-bin")
    (source
     (origin
       (inherit (package-source texlive-source))
       (modules '((guix build utils)
                  (ice-9 ftw)))
       (snippet
        #~(let ((delete-other-directories
                 (lambda (root keep)
                   (with-directory-excursion root
                     (for-each
                      delete-file-recursively
                      (scandir
                       "."
                       (lambda (file)
                         (and (not (member file (append keep '("." ".."))))
                              (eq? 'directory (stat:type (stat file)))))))))))
            (delete-other-directories "libs" '())
            (delete-other-directories "utils" '("axodraw2"))
            (delete-other-directories "texk" '())))))
    (arguments
     (substitute-keyword-arguments (package-arguments texlive-bin)
       ((#:configure-flags flags)
        #~(cons "--enable-axodraw2" (delete "--enable-web2c" #$flags)))
       ((#:phases phases)
        #~(modify-phases #$phases
            (replace 'check
              (lambda* (#:key tests? #:allow-other-keys)
                (when tests?
                  (with-directory-excursion "utils/axodraw2"
                    (invoke "make" "check")))))
            (replace 'install
              (lambda _
                (with-directory-excursion "utils/axodraw2"
                  (invoke "make" "install"))))))))
    (native-inputs '())
    (inputs '())
    (propagated-inputs '())
    (home-page (package-home-page texlive-axodraw2))
    (synopsis "Binary for @code{texlive-axodraw2}")
    (description
     "This package provides the binary for @code{texlive-axodraw2}.")
    (license (package-license texlive-axodraw2))))

(define-public texlive-b1encoding
  (package
    (name "texlive-b1encoding")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/b1encoding/"
                                     "fonts/enc/dvips/b1encoding/"
                                     "source/latex/b1encoding/"
                                     "tex/latex/b1encoding/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1lzsa0k5rr0n50d31dh4ghww329qjpg39w843zmh05z0jir6rg79"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/b1encoding")
    (synopsis "LaTeX encoding tools for Bookhands fonts")
    (description
     "The package characterises and defines the author's B1 encoding for use
with LaTeX when typesetting things using his Bookhands fonts.")
    (license license:lppl1.3+)))

(define-public texlive-backnaur
  (package
    (name "texlive-backnaur")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/backnaur/"
                                     "source/latex/backnaur/"
                                     "tex/latex/backnaur/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "168ng265vdl74l9c2jmp34ba6ir5i6c1jb7jpyzc31g9snqmyr09"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/backnaur")
    (synopsis "Typeset Backus Naur Form definitions")
    (description
     "The package typesets Backus-Naur Form (BNF) definitions.  It prints
formatted lists of productions, with numbers if required.  It can also print
in-line BNF expressions using math mode.")
    (license license:lppl1.3c)))

(define-public texlive-baekmuk
  (package
    (name "texlive-baekmuk")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/baekmuk/"
                                     "fonts/truetype/public/baekmuk/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0nc6sfjbimh1g8zhcl58ag6kqskhwd3h0m68y7k2v3v5i9n5g5yr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/baekmuk")
    (synopsis "Baekmuk Korean TrueType fonts")
    (description
     "This bundle consists of four Korean fonts: @file{batang.ttf} (serif),
@file{dotum.ttf} (sans-serif), @file{gulim.ttf} (sans-serif rounded) and
@file{hline.ttf} (headline).")
    (license license:expat)))

(define-public texlive-bagpipe
  (package
    (name "texlive-bagpipe")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/bagpipe/"
                                     "tex/generic/bagpipe/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0zybnikkj00hmlzfaaljcq1ml5vi6qfhl1hfrjq9vjc409vzzvka"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bagpipe")
    (synopsis "Support for typesetting bagpipe music")
    (description
     "Typesetting bagpipe music in MusixTeX is needlessly tedious.  This
package provides specialized and re-defined macros to simplify this task.")
    (license license:lppl1.3+)))

(define-public texlive-bangla
  (package
    (name "texlive-bangla")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bangla/"
                                     "fonts/truetype/public/bangla/"
                                     "tex/latex/bangla/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0basibrsbns9s2ny6ny95hl6dj4rjf6q4z9bqn2kc9945akxg8af"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-charissil texlive-doulossil))
    (home-page "https://ctan.org/pkg/bangla")
    (synopsis "Comprehensive Bangla LaTeX package")
    (description
     "This package provides all the necessary LaTeX frontends for the Bangla
language and comes with some fonts of its own.")
    (license (list license:lppl1.3c license:silofl1.1))))

(define-public texlive-bangorcsthesis
  (package
    (name "texlive-bangorcsthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bangorcsthesis/"
                                     "source/latex/bangorcsthesis/"
                                     "tex/latex/bangorcsthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0lnrqlzgl8ayycx4by7lygf4jbsghb22ymlzl0h3fg6jf50032ag"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bangorcsthesis")
    (synopsis "Typeset a thesis at Bangor University")
    (description
     "The class typesets thesis or dissertation documents for all
levels (i.e., both undergraduate and graduate students may use the class).  It
also provides macros designed to optimise the process of producing a thesis.")
    (license license:lppl1.3+)))

(define-public texlive-bangorexam
  (package
    (name "texlive-bangorexam")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bangorexam/"
                                     "source/latex/bangorexam/"
                                     "tex/latex/bangorexam/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09pp88415mba65icf6cx4vqn3an54hrgkd8s677ndlf2ml7n0lhs"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bangorexam")
    (synopsis "Typeset an examination at Bangor University")
    (description
     "The package allows typesetting of Bangor Univesity's exam style.  It
currently supports a standard A/B choice, A-only compulsory and n from m exam
styles.  Marks are totalled and checked automatically.")
    (license license:lppl1.3+)))

(define-public texlive-bangtex
  (package
    (name "texlive-bangtex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bangtex/"
                                     "fonts/source/public/bangtex/"
                                     "fonts/tfm/public/bangtex/"
                                     "tex/latex/bangtex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1hmv27wpnwv70gipvfvxs1rwfjqbcy0lgn09hs2ah6ayqgr740im"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/bangtex")
    (synopsis "Writing Bangla and Assamese with LaTeX")
    (description
     "The bundle provides class files for writing Bangla and Assamese with LaTeX,
and Metafont sources for fonts.")
    (license license:lppl)))

(define-public texlive-barr
  (package
    (name "texlive-barr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/barr/" "tex/generic/barr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1l532g0nxigbllv2lf74bwx78sf4jsfr1cad0hpq7i7kq820wbii"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/diagxy")
    (synopsis "Diagram macros by Michael Barr")
    (description
     "Diagxy is a general diagramming package, useful for diagrams in a number
of mathematical disciplines.")
    (license license:lppl1.3+)))

(define-public texlive-bartel-chess-fonts
  (package
    (name "texlive-bartel-chess-fonts")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/bartel-chess-fonts/"
                                     "fonts/source/public/bartel-chess-fonts/"
                                     "fonts/tfm/public/bartel-chess-fonts/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gg7g2gb5j0g0ig4190260zlpnyfmdzcqn7dsw5kp9p5pbn5hbhf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    ;; FIXME: Font metrics generation fails with "! Strange path (turning
    ;; number is zero)." error.
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               (delete 'generate-font-metrics))))
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/bartel-chess-fonts")
    (synopsis "Set of fonts supporting chess diagrams")
    (description "This package provides fonts supporting chess diagrams.")
    (license license:gpl3+)))

(define-public texlive-baskervaldadf
  (package
    (name "texlive-baskervaldadf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/baskervaldadf/"
                                     "fonts/afm/public/baskervaldadf/"
                                     "fonts/enc/dvips/baskervaldadf/"
                                     "fonts/map/dvips/baskervaldadf/"
                                     "fonts/tfm/public/baskervaldadf/"
                                     "fonts/type1/public/baskervaldadf/"
                                     "fonts/vf/public/baskervaldadf/"
                                     "source/fonts/baskervaldadf/"
                                     "tex/latex/baskervaldadf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32 "05ishkqwxgv482hcpzbkbkb5cwv1xxg98r7bymfnfxqcqz1zd5fx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:tex-format "latex"
           #:phases
           #~(modify-phases %standard-phases
               (add-after 'unpack 'remove-pre-generated-files
                 (lambda _
                   (for-each delete-file
                             (find-files "source/fonts/baskervaldadf/"
                                         "\\.etx$")))))))
    (home-page "https://ctan.org/pkg/baskervaldadf")
    (synopsis "Baskervald ADF fonts collection with TeX and LaTeX support")
    (description
     "Baskervald ADF is a serif family with lining figures designed as
a substitute for Baskerville.  The family currently includes upright and
italic or oblique shapes in each of regular, bold and heavy weights.  All
fonts include the slashed zero and additional non-standard ligatures.")
    (license (list license:lppl1.3c license:gpl2))))

(define-public texlive-baskervaldx
  (package
    (name "texlive-baskervaldx")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/baskervaldx/"
                                     "fonts/afm/public/baskervaldx/"
                                     "fonts/enc/dvips/baskervaldx/"
                                     "fonts/map/dvips/baskervaldx/"
                                     "fonts/opentype/public/baskervaldx/"
                                     "fonts/tfm/public/baskervaldx/"
                                     "fonts/type1/public/baskervaldx/"
                                     "fonts/vf/public/baskervaldx/"
                                     "tex/latex/baskervaldx/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "008w1ripan6xxn290scsf0dhba0k9kpd0dylxzn3gy4zxdvsm3jn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/baskervaldx")
    (synopsis "Extension and modification of BaskervaldADF with LaTeX support")
    (description
     "This package extends and modifies the BaskervaldADF font (a Baskerville
substitute) with more accented glyphs, with small caps and oldstyle figures in
all shapes.  It includes OpenType and PostScript fonts, as well as LaTeX
support files.")
    (license (list license:gpl2+ license:lppl1.3+))))

(define-public texlive-baskervillef
  (package
    (name "texlive-baskervillef")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/baskervillef/"
                                     "fonts/enc/dvips/baskervillef/"
                                     "fonts/map/dvips/baskervillef/"
                                     "fonts/opentype/public/baskervillef/"
                                     "fonts/tfm/public/baskervillef/"
                                     "fonts/type1/public/baskervillef/"
                                     "fonts/vf/public/baskervillef/"
                                     "tex/latex/baskervillef/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "04wczv0fp8lg1wg6w84hikwsbzzd8mpjq1p3qzjdwzqr53pxq710"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/baskervillef")
    (synopsis "Fry's Baskerville look-alike, with math support")
    (description
     "BaskervilleF is a fork from the Libre Baskerville fonts (Roman, Italic,
Bold only).  Their fonts are optimized for web usage, while BaskervilleF is
optimized for traditional TeX usage, normally destined for production of PDF
files.  A bold italic style was added and mathematical support is offered as
an option to @code{newtxmath}.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-basque-book
  (package
    (name "texlive-basque-book")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/basque-book/"
                                     "source/latex/basque-book/"
                                     "tex/latex/basque-book/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "16jgn4pxqbmq2dx5gfyy4p04ykaijkjyy9banp3l3a6ykm2xn8w7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/basque-book")
    (synopsis "Class for book-type documents written in Basque")
    (description
     "The class is derived from the LaTeX @code{book} class.  The extensions
solve grammatical and numeration issues that occur when book-type documents
are written in Basque.  The class is useful for writing books, PhD and Master
Theses, etc., in Basque.")
    (license license:lppl1.2+)))

(define-public texlive-basque-date
  (package
    (name "texlive-basque-date")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/basque-date/"
                                     "source/latex/basque-date/"
                                     "tex/latex/basque-date/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0n1nc2nrwl3ajk1n7xbnf49mvg8dlqlyndsyywgs46dnv55b7gs5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/basque-date")
    (synopsis "Print the date in Basque")
    (description
     "The package provides two LaTeX commands to print the current date in
Basque according to the correct forms ruled by The Basque Language Academy
(Euskaltzaindia).  The commands automatically solve the complex declination
issues of numbers in Basque.")
    (license license:lppl1.2+)))

(define-public texlive-bath-bst
  (package
    (name "texlive-bath-bst")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/bath-bst/"
                                     "doc/bibtex/bath-bst/"
                                     "source/bibtex/bath-bst/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06v5icy6537d69g5in7ps322wgnkf0imd93gnmh2y7fladsmvp06"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bath-bst")
    (synopsis
     "Harvard referencing style recommended by the University of Bath Library")
    (description
     "This package provides a BibTeX style to format reference lists in the
Harvard style recommended by the University of Bath Library.  It should be
used in conjunction with @code{natbib} for citations.")
    (license license:lppl1.3c)))

(define-public texlive-bbding
  (package
    (name "texlive-bbding")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bbding/"
                                     "fonts/source/public/bbding/"
                                     "fonts/tfm/public/bbding/"
                                     "source/latex/bbding/" "tex/latex/bbding/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0l507nxnnh880h5kg8q0ww6sl93k441l0lsplr5ldrngxvx1vrsc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/bbding")
    (synopsis "Symbol (dingbat) font and LaTeX macros for its use")
    (description
     "This package provides a symbol font (distributed as Metafont source)
that contains many of the symbols of the Zapf dingbats set, together with an
NFSS interface for using the font.  An Adobe Type 1 version of the fonts is
available in the @code{niceframe} fonts bundle.")
    (license license:lppl)))

(define-public texlive-bbold-type1
  (package
    (name "texlive-bbold-type1")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/bbold-type1/"
                                     "fonts/afm/public/bbold-type1/"
                                     "fonts/map/dvips/bbold-type1/"
                                     "fonts/type1/public/bbold-type1/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1flccfh58w977j9w62mcn7xzspmg68gb6h5nwh2mfbz7l0015fsz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bbold-type1")
    (synopsis "Adobe Type 1 format version of the @code{bbold} font")
    (description
     "The files offer an Adobe Type 1 format version of the 5pt, 7pt and 10pt
versions of the @code{bbold} fonts.")
    ;; The README states: "You may freely use, modify and/or
    ;; distribute this file".
    (license (license:fsf-free "file:doc/fonts/bbold-type1/README"))))

(define-public texlive-bboldx
  (package
    (name "texlive-bboldx")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/bboldx/"
                                     "fonts/afm/public/bboldx/"
                                     "fonts/enc/dvips/bboldx/"
                                     "fonts/map/dvips/bboldx/"
                                     "fonts/tfm/public/bboldx/"
                                     "fonts/type1/public/bboldx/"
                                     "tex/latex/bboldx/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1r4s3yyah0iqwhzss52wjv3wml5r4bywp3fbanjzkv16f2pk51yb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bboldx")
    (synopsis "Extend the @code{bbold} package with a Blackboard Bold alphabet")
    (description
     "This is an extension of @code{bbold} to a package with three weights,
of which the original is considered as light and the additions as regular and
bold.")
    ;; The README states: "You may freely use, modify and/or
    ;; distribute this file".
    (license (license:fsf-free "file://doc/fonts/bboldx/README"))))

(define-public texlive-bclogo
  (package
    (name "texlive-bclogo")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bclogo/" "metapost/bclogo/"
                                     "tex/latex/bclogo/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "087d4cgv42vf8hh1d5n09zjbd0rl7rs961gzqmp1ha8n8fymadi2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bclogo")
    (synopsis "Creating colourful boxes with logos")
    (description
     "The package facilitates the creation of colorful boxes with a title and logo.
It may use either TikZ or PSTricks as graphics engine.")
    (license license:lppl1.3+)))

(define-public texlive-beamer-fuberlin
  (package
    (name "texlive-beamer-fuberlin")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/beamer-fuberlin/"
                                     "tex/latex/beamer-fuberlin/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03zp6gpzilcfgrbz1lh8lpldvkf9g98f45fhsb72ikgjnka6211c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/beamer-fuberlin")
    (synopsis "Beamer, using the style of FU Berlin")
    (description
     "The bundle provides a @code{beamer}-derived class and a theme style file
for the corporate design of the Free University in Berlin.  Users may use the
class itself (FUbeamer) or use the theme in the usual way with
@code{\\usetheme@{BerlinFU@}}.")
    (license license:lppl)))

(define-public texlive-beamer-verona
  (package
    (name "texlive-beamer-verona")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/beamer-verona/"
                                     "tex/latex/beamer-verona/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0q79bdsc98a3y5h0yb3qwpsd0yyfyaxfjfhn4xkxf2qzyfp1mgd5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/beamer-verona")
    (synopsis "Theme for the @code{beamer} class")
    (description
     "This package provides the Verona theme for the @code{beamer} class.")
    (license license:lppl1.3+)))

(define-public texlive-beebe
  (package
    (name "texlive-beebe")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bib/beebe/" "tex/generic/beebe/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1w8f2zc5rghjpqrp8klq65prqblyam3wzhwp9yas5xfmy7hlsf52"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblio")
    (synopsis "Collection of bibliographies")
    (description
     "This package provides a collection of BibTeX bibliographies on
TeX-related topics (including, for example, spell-checking and SGML).  Each
includes a LaTeX wrapper file to typeset the bibliography.")
    (license license:public-domain)))

(define-public texlive-begingreek
  (package
    (name "texlive-begingreek")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/begingreek/"
                                     "source/latex/begingreek/"
                                     "tex/latex/begingreek/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1amzzc96pqxjicrgl4fnl2j7b65vvpxl32ckl308nbwfhd56fz6l"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/begingreek")
    (synopsis "Greek environment to be used with pdfLaTeX only")
    (description
     "This simple package defines a greek environment to be used with pdfLaTeX
only, that accepts an optional Greek font family name to type its contents
with.  A similar @code{\\greektxt} command does a similar action for shorter
texts.")
    (license license:lppl1.3c)))

(define-public texlive-beilstein
  (package
    (name "texlive-beilstein")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/beilstein/"
                                     "doc/latex/beilstein/"
                                     "source/latex/beilstein/"
                                     "tex/latex/beilstein/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0gr4fb0g3449jx6mbwfl2g00anss6xbj9xiak83hkc7797jzmg2b"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/beilstein")
    (synopsis
     "Support for submissions to the @emph{Beilstein Journal of Nanotechnology}")
    (description
     "The package provides a LaTeX class file and a BibTeX style file in
accordance with the requirements of submissions to the @emph{Beilstein Journal
of Nanotechnology}.")
    (license license:lppl1.3c)))

(define-public texlive-belleek
  (package
    (name "texlive-belleek")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/belleek/"
                                     "fonts/map/dvips/belleek/"
                                     "fonts/truetype/public/belleek/"
                                     "fonts/type1/public/belleek/"
                                     "source/latex/belleek/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12jigkk2xlnfllwaj6qsisgvl5b2kcwnv5j921cvm7gjn9x1i941"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/belleek")
    (synopsis "Replacement for basic MathTime fonts")
    (description "This package replaces the original MathTime fonts.")
    (license license:public-domain)))

(define-public texlive-begriff
  (package
    (name "texlive-begriff")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/begriff/" "tex/latex/begriff/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0vi7i35fqi32y26crk5aj93mivix61ppgmpz0djshizwrd9pn9xy"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/begriff")
    (synopsis "Typeset Begriffschrift")
    (description
     "The package defines maths mode commands for typesetting Frege's
@emph{Begriffschrift}.")
    (license license:gpl3+)))

(define-public texlive-bengali
  (package
    (name "texlive-bengali")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/bengali/"
                                     "fonts/source/public/bengali/"
                                     "fonts/tfm/public/bengali/"
                                     "source/latex/bengali/" "tex/latex/bengali/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "077rs7rcx592g5m53hy73w6jd39qgb83z5sg2jbymjfwkk7y593h"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/bengali-pandey")
    (synopsis "Support for the Bengali language")
    (description
     "The package is based on Velthuis transliteration scheme, with extensions
to deal with the Bengali letters that are not in Devanagari.  The package also
supports Assamese.")
    (license license:lppl)))

(define-public texlive-berenisadf
  (package
    (name "texlive-berenisadf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/berenisadf/"
                                     "fonts/afm/arkandis/berenisadf/"
                                     "fonts/enc/dvips/berenisadf/"
                                     "fonts/map/dvips/berenisadf/"
                                     "fonts/opentype/arkandis/berenisadf/"
                                     "fonts/tfm/arkandis/berenisadf/"
                                     "fonts/type1/arkandis/berenisadf/"
                                     "source/fonts/berenisadf/"
                                     "tex/latex/berenisadf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "106gxzdcfi3rkrg1ylrdmybwn9b0kra85a9d4f1qd3q0i28f1z2b"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:tex-format "latex"
           #:phases
           #~(modify-phases %standard-phases
               (add-after 'unpack 'remove-pre-generated-file
                 (lambda _
                   (delete-file "source/fonts/berenisadf/Makefile.make"))))))
    (home-page "https://ctan.org/pkg/berenisadf")
    (synopsis "Berenis ADF fonts and TeX/LaTeX support")
    (description
     "The bundle provides the BerenisADF Pro font collection, in OpenType and
PostScript Type 1 formats, together with support files to use the fonts in
TeXnANSI (LY1) and LaTeX standard T1 and TS1 encodings.")
    ;; (La)TeX support is released under LPPL 1.3+ terms.  Fonts themselves
    ;; use GPL 2+ with exceptions.
    (license (list license:lppl1.3+ license:gpl2+))))

(define-public texlive-besjournals
  (package
    (name "texlive-besjournals")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/besjournals/"
                                     "doc/bibtex/besjournals/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1mcc5xa56j17h6wv9b111qypw6ff4mg0skapa4wq79lb8p3l1cmv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/besjournals-bst")
    (synopsis
     "Bibliographies suitable for British Ecological Society journals")
    (description
     "The package provides a BibTeX style for use with journals published by
the British Ecological Society.  The style was produced independently of the
Society, and has no formal approval by the BES.")
    (license license:lppl)))

(define-public texlive-bestpapers
  (package
    (name "texlive-bestpapers")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/bestpapers/"
                                     "doc/bibtex/bestpapers/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0c0lp51z5z2707r1hig993gbx78v7zb84wmk3z9iyw08m3bbq34s"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bestpapers")
    (synopsis "BibTeX package to produce lists of authors' best papers")
    (description
     "Many people preparing their resumes find the requirement ``please list
five (or six, or ten) papers authored by you''.  The same requirement is often
stated for reports prepared by professional teams.  The creation of such lists
may be a cumbersome task.  Even more difficult is it to support such lists
over the time, when new papers are added.  The BibTeX style
@file{bestpapers.bst} is intended to facilitate this task.  It is based on the
idea that it is easier to score than to sort: we can assign a score to a paper
and then let the computer select the papers with highest scores.")
    (license license:public-domain)))

(define-public texlive-betababel
  (package
    (name "texlive-betababel")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/betababel/"
                                     "tex/latex/betababel/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1lck0wighndd3qrhcks6amc6gm5y50f6a61nfcz31j087r3a7liv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/betababel")
    (synopsis "Insert ancient greek text coded in Beta Code")
    (description
     "The @code{betababel} package extends the Babel @samp{polutonikogreek}
option to provide a simple way to insert ancient Greek texts with diacritical
characters into your document using the commonly used Beta Code
transliteration.")
    (license license:lppl)))

(define-public texlive-beuron
  (package
    (name "texlive-beuron")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/beuron/"
                                     "fonts/map/dvips/beuron/"
                                     "fonts/opentype/public/beuron/"
                                     "fonts/source/public/beuron/"
                                     "fonts/tfm/public/beuron/"
                                     "fonts/type1/public/beuron/"
                                     "tex/latex/beuron/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06hk60z0s2rdbj1iard10zxw7zvvm2xds755r4mqh0p70qhkpjzx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/beuron")
    (synopsis "Script of the Beuronese art school")
    (description
     "This package provides the script used in the works of the Beuron art
school for use with TeX and LaTeX.  It is a monumental script consisting of
capital letters only.  The fonts are provided as Metafont sources, in the
Type1 and in the OpenType format.  The package includes suitable font
selection commands for use with LaTeX.")
    (license license:lppl1.3+)))

(define-public texlive-bfh-ci
  (package
    (name "texlive-bfh-ci")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bfh-ci/" "tex/latex/bfh-ci/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1f1ygicy2i23z7ilz3cpv856nccplah0996nv3wc0p2k1xw29rpk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-adjustbox
                             texlive-amsfonts
                             texlive-amsmath
                             texlive-anyfontsize
                             texlive-beamer
                             texlive-fontawesome
                             texlive-fontspec
                             texlive-geometry
                             texlive-graphics
                             texlive-handoutwithnotes
                             texlive-hyperref
                             texlive-iftex
                             texlive-koma-script
                             texlive-l3kernel
                             texlive-l3packages
                             texlive-listings
                             texlive-nunito
                             texlive-pgf
                             texlive-qrcode
                             texlive-sourceserifpro
                             texlive-tcolorbox
                             texlive-tools
                             texlive-translations
                             texlive-url
                             texlive-xcolor
                             texlive-zref))
    (home-page "https://ctan.org/pkg/bfh-ci")
    (synopsis "Corporate design for Bern University of Applied Sciences")
    (description
     "This bundle provides possibilities to use the corporate design of Bern
University of Applied Sciences (BFH) with LaTeX.  To this end it contains
classes as well as some helper packages and config files.")
    (license license:lppl1.3c)))

(define-public texlive-bgteubner
  (package
    (name "texlive-bgteubner")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/bgteubner/"
                                     "doc/latex/bgteubner/"
                                     "makeindex/bgteubner/"
                                     "source/latex/bgteubner/"
                                     "tex/latex/bgteubner/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1plc42glcq2pxyns6lm6fygicjm8whrls1qlfw00fccsw9v1hgxv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'fix-build
            ;; This phase is necessary because the build phase is reluctant to
            ;; generate "ltxdoc.cfg" since there is another one among the
            ;; inputs already.
            (lambda _
              (substitute* "source/latex/bgteubner/hhsubfigure.ins"
                (("\\\\generateFile\\{ltxdoc\\.cfg\\}\\{t\\}")
                 "\\generateFile{ltxdoc.cfg}{f}")))))))
    (home-page "https://ctan.org/pkg/bgteubner")
    (synopsis "Class for producing books for the publisher Teubner Verlag")
    (description
     "The @code{bgteubner} document class has been programmed by order of the
Teubner Verlag, Wiesbaden, Germany, to ensure that books of this publisher
have a unique layout.  Unfortunately, most of the documentation is only
available in German.  Since the document class is intended to generate
a unique layout, many things (layout etc.) are fixed and cannot be altered by
the user.")
    (license license:lppl)))

(define-public texlive-bguq
  (package
    (name "texlive-bguq")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/bguq/"
                                     "fonts/map/dvips/bguq/"
                                     "fonts/source/public/bguq/"
                                     "fonts/tfm/public/bguq/"
                                     "fonts/type1/public/bguq/"
                                     "source/fonts/bguq/"
                                     "tex/latex/bguq/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0d87ihzwq3lxhr28yaj1wpkvl4jgcfw8859g9mamrprkm7klm2rb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/bguq")
    (synopsis "Improved quantifier stroke for Begriffsschrift packages")
    (description
     "The font contains a single character: the Begriffsschrift quantifier (in
several sizes), as used to set the Begriffsschrift (concept notation) of
Frege.  The font is not intended for end users; instead it is expected that it
will be used by other packages which implement the Begriffsschrift.")
    (license license:lppl)))

(define-public texlive-bib-fr
  (package
    (name "texlive-bib-fr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/bib-fr/" "doc/bibtex/bib-fr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "11af1p52vr16l5gf69ql2zz0328yqd75nbcvik7b0g05pbqc12nv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bib-fr")
    (synopsis "French translation of classical BibTeX styles")
    (description
     "These files are French translations of the classical BibTeX style
files.")
    (license license:lppl)))

(define-public texlive-bib2gls
  (package
    (name "texlive-bib2gls")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/bib2gls.1"
                                     "doc/man/man1/bib2gls.man1.pdf"
                                     "doc/man/man1/convertgls2bib.1"
                                     "doc/man/man1/convertgls2bib.man1.pdf"
                                     "doc/man/man1/datatool2bib.1"
                                     "doc/man/man1/datatool2bib.man1.pdf"
                                     "doc/support/bib2gls/"
                                     "scripts/bib2gls/"
                                     "source/support/bib2gls/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1r77zf0h1zxdapfwxdwd8ljvj70r7zll819c0614aympnnl4slp4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "bib2gls.sh" "convertgls2bib.sh")))
    (propagated-inputs (list texlive-glossaries-extra))
    (home-page "https://ctan.org/pkg/bib2gls")
    (synopsis
     "Convert @file{.bib} files to @file{glossaries-extra.sty} resource files")
    (description
     "This Java command line application may be used to extract glossary
information stored in a @file{.bib} file and convert it into glossary entry
definition commands.  This application should be used with
@file{glossaries-extra.sty}'s record package option.  It performs two
functions in one: it selects entries according to records found in the
@file{.aux} file (similar to BibTeX), and hierarchically sorts entries and
collates location lists (similar to MakeIndex or Xindy).  The glossary entries
can then be managed in a system such as JabRef, and only the entries that are
actually required will be defined, reducing the resources required by TeX.

The supplementary application @command{convertgls2bib} can be used to convert
existing @file{.tex} files containing definitions (@code{\\newglossaryentry}
etc.)#: to the @file{.bib} format required by @command{bib2gls}.")
    (license license:gpl3+)))

(define-public texlive-bib2qr
  (package
    (name "texlive-bib2qr")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/bib2qr/"
                              "source/latex/bib2qr/"
                              "tex/latex/bib2qr/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0z1ln5y7d6z5dwsd2y4s5n9yb4npzndh0297l3ph5gks656cdv5j"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bib2qr")
    (synopsis "Cite BibTeX entries with QR codes")
    (description
     "This package provides functionality to cite BibTeX entries with QR codes
for easy sharing and referencing.  The target of the QR code is the entry's
digital object identifier (DOI), or URL if no DOI exists.  It is realised via
the @code{LaTeX} packages @code{biblatex} and @code{qrcode}.")
    (license license:lppl1.3c)))

(define-public texlive-bibarts
  (package
    (name "texlive-bibarts")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibarts/" "source/latex/bibarts/"
                                     "tex/latex/bibarts/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ymjp12l8d6yg9n1gypy8lcw2fijr66cmr0xr3my2dfbnkjgd4g8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               (add-after 'build 'build-and-install'bibsort
                 (lambda _
                   (mkdir-p "build")
                   (invoke "gcc" "-o" "build/bibsort"
                           "source/latex/bibarts/bibsort.c")
                   (install-file "build/bibsort"
                                 (string-append #$output "/bin")))))))
    (home-page "https://ctan.org/pkg/bibarts")
    (synopsis "Arts-style bibliographical information")
    (description
     "@code{BibArts} is a LaTeX package to assist in making bibliographical
features common in the arts and the humanities (history, political science,
philosophy, etc.).  @file{bibarts.sty} provides commands for quotations,
abbreviations, and especially for a formatted citation of literature,
journals (periodicals), edited sources, and archive sources.

It will also copy all citation information, abbreviations, and register key
words into lists for an automatically generated appendix.  These lists may
refer to page and footnote numbers. BibArts has nothing to do with BibTeX.
The lists are created by @command{bibsort}.  This program creates the
bibliography without using MakeIndex or BibTeX.")
    (license license:gpl3+)))

(define-public texlive-bibcop
  (package
    (name "texlive-bibcop")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/bibtex/bibcop/"
                                     "doc/man/man1/bibcop.1"
                                     "doc/man/man1/bibcop.man1.pdf"
                                     "scripts/bibcop/"
                                     "source/bibtex/bibcop/"
                                     "tex/latex/bibcop/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1z6lfd5sqg5nn8gnm0kpqhl169wpdnhvbdlsin0x0i1s6kc7g2xs"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "bibcop.pl")))
    (inputs (list perl))
    (propagated-inputs (list texlive-iexec texlive-pgfopts))
    (home-page "https://ctan.org/pkg/bibcop")
    (synopsis "Style checker for .bib files")
    (description
     "This LaTeX package checks the quality of your @file{.bib} file and emits
warning messages if any issues are found.  For this, the TeX processor must be
run with the @samp{--shell-escape} option.  @command{bibcop} can also be used
as a standalone command line tool.")
    (license license:expat)))

(define-public texlive-biber-ms
  (package
    (name "texlive-biber-ms")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/bibtex/biber-ms/"
                                     "source/bibtex/biber-ms/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10r9jgarrbvqrz9nkjfkzsy3niggg5w6sjr7zi7bqy8zhkwf3hsf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (properties '((updater-extra-native-inputs "perl-config-autoconf"
                                               "perl-extutils-libbuilder"
                                               "perl-file-which"
                                               "perl-module-build"
                                               "perl-test-differences")
                  (updater-extra-inputs "perl"
                                        "perl-autovivification"
                                        "perl-class-accessor"
                                        "perl-data-dump"
                                        "perl-data-compare"
                                        "perl-data-uniqid"
                                        "perl-datetime-format-builder"
                                        "perl-datetime-calendar-julian"
                                        "perl-file-slurper"
                                        "perl-io-string"
                                        "perl-ipc-cmd"
                                        "perl-ipc-run3"
                                        "perl-list-allutils"
                                        "perl-list-moreutils"
                                        "perl-mozilla-ca"
                                        "perl-regexp-common"
                                        "perl-log-log4perl"
                                        "perl-parse-recdescent"
                                        "perl-unicode-collate"
                                        "perl-unicode-normalize"
                                        "perl-unicode-linebreak"
                                        "perl-encode-eucjpascii"
                                        "perl-encode-jis2k"
                                        "perl-encode-hanextra"
                                        "perl-xml-libxml"
                                        "perl-xml-libxml-simple"
                                        "perl-xml-libxslt"
                                        "perl-xml-writer"
                                        "perl-sort-key"
                                        "perl-text-csv"
                                        "perl-text-csv-xs"
                                        "perl-text-roman"
                                        "perl-uri"
                                        "perl-text-bibtex"
                                        "perl-libwww"
                                        "perl-lwp-protocol-https"
                                        "perl-business-isbn"
                                        "perl-business-issn"
                                        "perl-business-ismn"
                                        "perl-lingua-translit")))
    (arguments
     (list
      #:tests? #true
      #:imported-modules `(,@%texlive-build-system-modules
                           (guix build perl-build-system))
      #:modules '((guix build texlive-build-system)
                  ((guix build perl-build-system) #:prefix perl:)
                  (guix build utils))
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'unpack 'unpack-biber-ms-source
            (lambda _
              (mkdir-p "build")
              (with-directory-excursion "build"
                (invoke "tar" "xvf"
                        "../source/bibtex/biber-ms/biblatex-biber-ms.tar.gz"
                        "--strip-components=1"))))
          (add-after 'unpack-biber-ms-source 'build-biber-ms
            (lambda args
              (with-directory-excursion "build"
                ;; The following test is expected to fail on 32 bits systems.
                #$@(if (target-x86-32?)
                       '((delete-file "t/dateformats.t"))
                       '())
                ;; FIXME: The following tests fail without a good reason.
                (for-each delete-file '("t/multiscript.t" "t/utils.t"))
                (for-each (lambda (phase)
                            (apply (assoc-ref perl:%standard-phases phase) args))
                          '(configure build check install)))))
          (add-after 'install 'wrap-programs
            (lambda _
              (with-directory-excursion (string-append #$output "/bin")
                (rename-file "biber" "biber-ms")
                (let ((perl5lib (getenv "PERL5LIB")))
                  (wrap-program "biber-ms"
                    `("PERL5LIB" ":" prefix
                      (,(string-append perl5lib ":"
                                       #$output "/lib/perl5/site_perl")))))))))))
    (native-inputs
     (list perl-config-autoconf
           perl-extutils-libbuilder
           perl-file-which
           perl-module-build
           perl-test-differences))
    (inputs
     (list perl
           perl-autovivification
           perl-business-isbn
           perl-business-ismn
           perl-business-issn
           perl-class-accessor
           perl-data-compare
           perl-data-dump
           perl-data-uniqid
           perl-datetime-calendar-julian
           perl-datetime-format-builder
           perl-encode-eucjpascii
           perl-encode-hanextra
           perl-encode-jis2k
           perl-file-slurper
           perl-io-string
           perl-ipc-cmd
           perl-ipc-run3
           perl-libwww
           perl-lingua-translit
           perl-list-allutils
           perl-list-moreutils
           perl-log-log4perl
           perl-lwp-protocol-https
           perl-mozilla-ca
           perl-parse-recdescent
           perl-regexp-common
           perl-sort-key
           perl-text-bibtex
           perl-text-csv
           perl-text-csv-xs
           perl-text-roman
           perl-unicode-collate
           perl-unicode-linebreak
           perl-unicode-normalize
           perl-uri
           perl-xml-libxml
           perl-xml-libxml-simple
           perl-xml-libxslt
           perl-xml-writer))
    (home-page "https://ctan.org/pkg/biber-ms")
    (synopsis
     "BibTeX replacement for users of BibLaTeX (multiscript version)")
    (description
     "This is the multiscript version of Biber.  It must be used with the
multiscript version of @code{biblatex-ms}.")
    (license license:artistic2.0)))

(define-public texlive-bibexport
  (package
    (name "texlive-bibexport")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/bibexport/"
                                     "doc/bibtex/bibexport/" "scripts/bibexport/"
                                     "source/bibtex/bibexport/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "161056627w1lazfpld3lyjwfrl8j8gc4b6dzml46bzwf7mk9ifln"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list
      #:link-scripts #~(list "bibexport.sh")
      #:phases
      #~(modify-phases %standard-phases
          (add-after 'build 'fix-bash-shebang
            (lambda _
              (substitute* "scripts/bibexport/bibexport.sh"
                (("/bin/bash") (which "bash"))))))))
    (home-page "https://ctan.org/pkg/bibexport")
    (synopsis "Extract a BibTeX file based on a @file{.aux} file")
    (description
     "This package provides a Bourne shell script that uses BibTeX to extract
bibliography entries that are @code{\\cite}'d in a document.  It can also
expand a BibTeX file, expanding the abbreviations (other than the built-in
ones like month names) and following the cross-references.")
    (license license:lppl1.3+)))

(define-public texlive-bibhtml
  (package
    (name "texlive-bibhtml")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/bibhtml/" "doc/bibtex/bibhtml/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0fjpipxc885hk9pvjf3f0wsp84lr1d7flvs5c56jmlvryvc6mv0w"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibhtml")
    (synopsis "BibTeX support for HTML files")
    (description
     "Bibhtml consists of a Perl script and a set of BibTeX style files, which
together allow you to output a bibliography as a collection of HTML files.
The references in the text are linked directly to the corresponding
bibliography entry, and if a URL is defined in the entry within the BibTeX
database file, then the generated bibliography entry is linked to this.  The
package provides three different style files derived from each of the standard
@file{plain.bst} and @file{alpha.bst}, as well as two style files derived from
@file{abbrv.bst} and unsrt.bst (i.e., eight in total).")
    (license license:gpl3+)))

(define-public texlive-biblatex-abnt
  (package
    (name "texlive-biblatex-abnt")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-abnt/"
                                     "tex/latex/biblatex-abnt/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14v6203y1xi10fv0lmdk46pnrvd489ag982jlkmyxqvy6rfhw9lg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-abnt")
    (synopsis "BibLaTeX style for Brazil's ABNT rules")
    (description
     "This package offers a BibLaTeX style for Brazil's ABNT (Brazilian
Association of Technical Norms) rules.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-accursius
  (package
    (name "texlive-biblatex-accursius")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/biblatex-accursius/"
                              "source/latex/biblatex-accursius/"
                              "tex/latex/biblatex-accursius/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0i71sd94045qgxyhzcrfr6qm7fiibbibzzz7hwaprj6phac0zzky"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-local-tree")))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "lualatex"))
    (native-inputs
     (list (texlive-local-tree
            (list texlive-babel-french
                  texlive-babel-german
                  texlive-babel-italian
                  texlive-biblatex
                  texlive-biblatex-ext
                  texlive-caption
                  texlive-csquotes
                  texlive-emptypage
                  texlive-enumitem
                  texlive-fmtcount
                  texlive-fontsize
                  texlive-fontspec
                  texlive-hypdoc
                  texlive-itnumpar
                  texlive-lettrine
                  texlive-libertine
                  texlive-libertinus
                  texlive-libertinus-fonts
                  texlive-libertinus-otf
                  texlive-listings
                  texlive-lualatex-math
                  texlive-mdframed
                  texlive-metalogo
                  texlive-needspace
                  texlive-pgf
                  texlive-quoting
                  texlive-titlesec
                  texlive-unicode-math
                  texlive-xcolor
                  texlive-xkeyval
                  texlive-zref))))
    (home-page "https://ctan.org/pkg/biblatex-accursius")
    (synopsis "Citing features for Italian jurists")
    (description
     "This style is primarily aimed at Italian legal jurists and provides them
with the ability to cite legal materials, such as legislative acts,
regulations, soft law, treaties and case law.  Additionally, the style
codifies the most prevalent citation practices among Italian legal scholars.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-ajc2020unofficial
  (package
    (name "texlive-biblatex-ajc2020unofficial")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-ajc2020unofficial/"
                                     "tex/latex/biblatex-ajc2020unofficial/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ngh68a7ihhvpdfz09c087pfqnnpj3mnm8c4jmw8krq52b79fr87"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-ajc2020unofficial")
    (synopsis "BibLaTeX style for the Australasian Journal of Combinatorics")
    (description
     "This is an unofficial BibLaTeX style for the Australasian Journal of
Combinatorics.  Note that the journal (as for 01 March 2020) does not accept
BibLaTeX, so you probably want to use @code{biblatex2bibitem}.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-anonymous
  (package
    (name "texlive-biblatex-anonymous")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-anonymous/"
                                     "tex/latex/biblatex-anonymous/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1q24xrr80i6xsq65gfvvi1s5qsp78g52qgn2nf2523gjnnzp4p7n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-anonymous")
    (synopsis "Tool to manage anonymous work with BibLaTeX")
    (description
     "The package provides tools to help manage anonymous work with BibLaTeX.
It will be useful, for example, in history or classical philology.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-apa6
  (package
    (name "texlive-biblatex-apa6")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-apa6/"
                                     "tex/latex/biblatex-apa6/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14xkqca4l4sb8jwlz6k28hmlya4m9474sv79icyzzq1v0vbqmgbq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-apa6")
    (synopsis "BibLaTeX citation and reference style for APA 6th Edition")
    (description
     "This is a fairly complete BibLaTeX style (citations and references) for
APA (American Psychological Association) 6th Edition conformant publications.
It implements and automates most of the guidelines in the APA 6th edition
style guide for citations and references.  An example document is also given
which typesets every citation and reference example in the APA 6th edition
style guide.  This is a legacy style for 6th Edition documents.  Please use
the BibLaTeX-apa style package for the latest APA edition conformance.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-archaeology
  (package
    (name "texlive-biblatex-archaeology")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-archaeology/"
                                     "source/latex/biblatex-archaeology/"
                                     "tex/latex/biblatex-archaeology/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "059mp6c1wgxkdjqrpwyx9yafw8dma75c3653rjc3sp66xx3akr8k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-archaeology")
    (synopsis "Collection of BibLaTeX styles for German prehistory")
    (description
     "This package provides additional BibLaTeX styles for German humanities.
Its core purpose is to enable the referencing rules of the Romano-Germanic
Commission (@emph{Romisch-Germanische Kommission}), the department of
prehistory of the German Archaeological Institute (@emph{Deutsches
Archaologisches Institut}), since these are referenced by most guidelines in
German prehistory and medieval archaeology and serve as a kind of template.
@code{biblatex-archaeology} provides verbose, numeric and author date styles
as well and adaptions to specific document types like exhibition and auction
catalogues.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-arthistory-bonn
  (package
    (name "texlive-biblatex-arthistory-bonn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-arthistory-bonn/"
                                     "tex/latex/biblatex-arthistory-bonn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0iwms7w1xqghdf3s0m91xzw4y7d20lvpv69ibl92cpjj5fggp0x8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-arthistory-bonn")
    (synopsis "BibLaTeX citation style for art historians")
    (description
     "This citation style covers the citation and bibliography guidelines of
the Kunsthistorisches Institut der Universitat Bonn for undergraduates.  It
introduces bibliography entry types for catalogs and features a tabular
bibliography, among other things.  Various options are available to change and
adjust the outcome according to one's own preferences.  The style is
compatible with English and German.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-bath
  (package
    (name "texlive-biblatex-bath")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-bath/"
                                     "source/latex/biblatex-bath/"
                                     "tex/latex/biblatex-bath/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1c640csqvq0f9fd5d7xx4apllbwvr4cmi2x38863wlrvlnxa9kd7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-bath")
    (synopsis "Style recommended by the University of Bath Library")
    (description
     "This package provides a BibLaTeX style to format reference lists in the
Harvard style recommended by the University of Bath Library.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-bookinarticle
  (package
    (name "texlive-biblatex-bookinarticle")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-bookinarticle/"
                                     "tex/latex/biblatex-bookinarticle/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0mhj3adqlirylhjqj8y0m8pinmark07zqvx00zl9rrdpagj6q4y8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-bookinarticle")
    (synopsis "Manage book edited in article")
    (description
     "This package provides three new BibLaTeX entry types --- @code{@@bookinarticle},
@code{@@bookinincollection} and @code{@@bookinthesis} --- to refer to a modern
edition of an old book, where this modern edition is provided in
a @code{@@article}, @code{@@incollection} or in a @code{@@thesis}.  The
package is now superseded by @code{biblatex-bookinother}.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-bookinother
  (package
    (name "texlive-biblatex-bookinother")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-bookinother/"
                                     "tex/latex/biblatex-bookinother/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "062ri08pwr5fh8v500dbgrc7nbxgi0jd864xfmvf601lwpwqn9c7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-bookinother")
    (synopsis "Manage book edited in other entry type")
    (description
     "This package provides new BibLaTeX entry types and fields for book
edited in other types, like for instance @code{@@bookinarticle}.  It offers
more types than the older package @code{biblatex-bookinarticle}, which it
supersedes.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-bwl
  (package
    (name "texlive-biblatex-bwl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-bwl/"
                                     "tex/latex/biblatex-bwl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "122qz05rc7c3pys6adg38xq0r123f4hspc3yyw8l2mzlpbbwm9y5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-bwl")
    (synopsis "BibLaTeX citations for FU Berlin")
    (description
     "The bundle provides a set of BibLaTeX implementations of bibliography
and citation styles for the Business Administration Department of the Free
University of Berlin.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-caspervector
  (package
    (name "texlive-biblatex-caspervector")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-caspervector/"
                                     "tex/latex/biblatex-caspervector/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0m1x3r52i6z06gy3jqlhaqnc876mi0qwjywmwyapvm8kfa2my0p3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-caspervector")
    (synopsis "Simple citation style for Chinese users")
    (description
     "The package provides a simple and easily extensible biblography/citation
style for Chinese LaTeX users, using BibLaTeX.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-chem
  (package
    (name "texlive-biblatex-chem")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-chem/"
                                     "tex/latex/biblatex-chem/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0y4fgnykh5x48cnbgyhgj04iqz05mvnzqvjn84vbr3mqmwjrndbl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-chem")
    (synopsis
     "BibLaTeX implementations of chemistry-related bibliography styles")
    (description
     "The bundle offers a set of styles to allow chemists to use BibLaTeX.
The package has complete styles for: all ACS journals; RSC journals using
standard (Chem.@: Commun.) style; and Angewandte Chem.@: style, (thus covering
a wide range of journals).")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-chicago
  (package
    (name "texlive-biblatex-chicago")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-chicago/"
                                     "tex/latex/biblatex-chicago/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0wa4i84hkladgz1j9aq5yiazpryxh3sykmkvsf7xpw5qlaay1psj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-chicago")
    (synopsis "Chicago style files for BibLaTeX")
    (description
     "This is a BibLaTeX style that implements the Chicago author-date and
notes with bibliography style specifications given in the @emph{Chicago Manual
of Style}, 17th edition (with continuing support for the 16th edition, too).
The style implements entry types for citing audio-visual materials, among many
others.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-claves
  (package
    (name "texlive-biblatex-claves")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-claves/"
                                     "tex/latex/biblatex-claves/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "08ydcv7qlbxaysh7jm3d4glrhp1fyd4fvvxc3jc4k50ga77ksqj0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-claves")
    (synopsis "Tool to manage claves of old literature with BibLaTeX")
    (description
     "When studying antic and medieval literature, we may find many different
texts published with the same title, or, in contrary, the same text published
with different titles.  To avoid confusion, scholars have published claves,
which are books listing ancient texts, identifying them by an identifier ---
a number or a string of text.  For example, for early Christianity, we have
the Bibliotheca Hagiographica Graeca, the Clavis Apocryphorum Novi Testamenti
and other claves.  It could be useful to print the identifier of a texts in
one specific clavis, or in many claves.  The package allows us to create new
field for different claves, and to present all these fields in a consistent
way.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-cse
  (package
    (name "texlive-biblatex-cse")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/biblatex-cse/"
                              "tex/latex/biblatex-cse/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "03habhgn696z8zaar0m0l8yws82p6ds5bv0nskhfz5sqc2ddjpmc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-cse")
    (synopsis "Council of Science Editors (CSE) style file for BibLaTeX")
    (description
     "This is a BibLaTeX style that implements the bibliography style of the
Council of Science Editors (CSE) for BibLaTeX.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-cv
  (package
    (name "texlive-biblatex-cv")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-cv/"
                                     "tex/latex/biblatex-cv/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17m6lqnr35y6vcfrrhlhxjf55b85py9miksswhkipyiaj3kphhb8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-cv")
    (synopsis "Create a CV from BibTeX files")
    (description
     "This package creates an academic curriculum vitae (CV) from a BibTeX
@file{.bib} file.  The package makes use of BibLaTeX and Biber to
automatically format, group, and sort the entries on a CV.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-dw
  (package
    (name "texlive-biblatex-dw")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-dw/"
                                     "tex/latex/biblatex-dw/bbx/"
                                     "tex/latex/biblatex-dw/cbx/"
                                     "tex/latex/biblatex-dw/lbx/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1vw008djg535hdwbmk5dqvaw0v6cm3lqmlvk8sfzybpqffw1crlr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-dw")
    (synopsis "Humanities styles for BibLaTeX")
    (description
     "This package provides a small collection of styles for the BibLaTeX
package.  It was designed for citations in the humanities and offers some
features that are not provided by the standard BibLaTeX styles.  The styles
are dependent on BibLaTeX and cannot be used without it.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-enc
  (package
    (name "texlive-biblatex-enc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-enc/"
                                     "tex/latex/biblatex-enc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0w8qahm6msmgjsbgz3x67wsd0p4xvwlfzqmnxicjkgghzx6x0br1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-enc")
    (synopsis "BibLaTeX style for the Ecole nationale des chartes (Paris)")
    (description
     "This package provides a citation and bibliography style for use with
BibLaTeX.  It conforms to the bibliographic standards used at the Ecole
nationale des chartes (Paris), and may be suitable for a more general use in
historical and philological works.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-ext
  (package
    (name "texlive-biblatex-ext")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-ext/"
                                     "tex/latex/biblatex-ext/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0i9n6hj81ma3w67hhfc59wv01h3kna4lwcn652hslbg4x03bprl2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-ext")
    (synopsis "Extended BibLaTeX standard styles")
    (description
     "The BibLaTeX-ext bundle provides styles that slightly extend the
standard styles that ship with BibLaTeX.  The styles offered in this bundle
provide a simple interface to change some of the stylistic decisions made in
the standard styles.  At the same time they stay as close to their standard
counterparts as possible, so that most customisation methods can be applied
here as well.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-fiwi
  (package
    (name "texlive-biblatex-fiwi")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-fiwi/"
                                     "tex/latex/biblatex-fiwi/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "080m1xj1g8v6aavp2i0hipjk3iikjqd2i36by2fiq0ys5vl5vhnq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-fiwi")
    (synopsis "BibLaTeX styles for use in German humanities")
    (description
     "The package provides a collection of styles for BibLaTeX.  It was
designed for citations in German Humanities, especially film studies, and
offers some features that are not provided by the standard BibLaTeX styles.
The style is highly optimized for documents written in German, and the main
documentation is only available in German.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-gb7714-2015
  (package
    (name "texlive-biblatex-gb7714-2015")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-gb7714-2015/"
                                     "tex/latex/biblatex-gb7714-2015/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "01m60n88k7hy3dkhbc07in57f2h98q9phkbxd7c04cch82ci1mlg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-gb7714-2015")
    (synopsis
     "BibLaTeX implementation of the GBT7714-2015 bibliography style for
Chinese users")
    (description
     "This package provides an implementation of the GBT7714-2015 bibliography
style.  This implementation follows the GBT7714-2015 standard and can be used
by simply loading BibLaTeX with the appropriate option.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-german-legal
  (package
    (name "texlive-biblatex-german-legal")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-german-legal/"
                                     "tex/latex/biblatex-german-legal/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14njaba2gvz338f8z4w3vi1qyf7yxx3l9c9nla6ggrw4rnicjpzs"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-german-legal")
    (synopsis "Comprehensive citation style for German legal texts")
    (description
     "This package aims to provide citation styles (for footnotes and
bibliographies) for German legal texts.  It is currently focused on citations
in books (style german-legal-book), but may be extended to journal articles in
the future.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-gost
  (package
    (name "texlive-biblatex-gost")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-gost/"
                                     "tex/latex/biblatex-gost/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "08zrbh6ipqg4r0bnjgh9sd5z7aismyi7s8fhkbgdmniw99c4qdc5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-gost")
    (synopsis "BibLaTeX support for GOST standard bibliographies")
    (description
     "The package provides BibLaTeX support for Russian bibliography style
GOST 7.0.5-2008")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-historian
  (package
    (name "texlive-biblatex-historian")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-historian/"
                                     "tex/latex/biblatex-historian/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0y9jvjyji39ly5pf72qhnnlsrixaxjxdx529lyav8bhd2zz3c5s7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-historian")
    (synopsis "BibLaTeX style based on the @emph{Turabian Manual}")
    (description
     "This package provides a BibLaTeX style, based on the @emph{Turabian
Manual} (a version of Chicago).")
    (license license:lppl)))

(define-public texlive-biblatex-ieee
  (package
    (name "texlive-biblatex-ieee")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-ieee/"
                                     "tex/latex/biblatex-ieee/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ld360la3h4j3whg89lx9aw6s28sb4g91c0vk1m5dpvhrsdrnnqi"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-ieee")
    (synopsis "IEEE style files for BibLaTeX")
    (description
     "This is a BibLaTeX style that implements the bibliography style of the
IEEE for BibLaTeX.  The implementation follows standard BibLaTeX conventions,
and can be used simply by loading BibLaTeX with the appropriate @samp{ieee}
option.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-ijsra
  (package
    (name "texlive-biblatex-ijsra")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-ijsra/"
                                     "tex/latex/biblatex-ijsra/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0phgbrp47f6pgayws29acsacm7dcb5w3asvwswx5a2rg30wj2fln"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-ijsra")
    (synopsis
     "BibLaTeX style for the International Journal of Student Research in
Archaeology")
    (description
     "This is the BibLaTeX style used for the journal International Journal of
Student Research in Archaeology.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-iso690
  (package
    (name "texlive-biblatex-iso690")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-iso690/"
                                     "tex/latex/biblatex-iso690/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07h16m8n8lydph2n567fnx2hzdw15mwyrcgs30x8anv9qxh0z6jf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-iso690")
    (synopsis "BibLaTeX style for ISO 690 standard")
    (description
     "The package provides a bibliography and citation style which conforms to
the latest revision of the international standard ISO 690:2010.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-jura2
  (package
    (name "texlive-biblatex-jura2")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-jura2/"
                                     "tex/latex/biblatex-jura2/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1cnsmk46j98z7wk7qs6v9f5m42x124v2nxrnq8n1jcs1jd03q6ry"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-jura2")
    (synopsis "Citation style for the German legal profession")
    (description
     "The package offers BibLaTeX support for citations in German legal texts.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-juradiss
  (package
    (name "texlive-biblatex-juradiss")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-juradiss/"
                                     "tex/latex/biblatex-juradiss/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13rwvj1ibhj0c46xr13wjyh68abkrvh2jcg2ccichh33crrxaql0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-juradiss")
    (synopsis "BibLaTeX stylefiles for German law theses")
    (description
     "The package provides a custom citation-style for typesetting a German
law thesis with LaTeX.")
    (license license:lppl)))

(define-public texlive-biblatex-license
  (package
    (name "texlive-biblatex-license")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-license/"
                                     "tex/latex/biblatex-license/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xz6zms984v5r8hq01f7ap245lfhqj577rc9ww57ccv4kgvgicqd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-license")
    (synopsis "Add license data to the bibliography")
    (description
     "This package is for adding license license:data to bibliography entries
via BibLaTeX's built-in related mechanism.  It provides a new related type
@code{license} and some bibmacros for typesetting these related entries.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-lncs
  (package
    (name "texlive-biblatex-lncs")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-lncs/"
                                     "tex/latex/biblatex-lncs/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0agr8dnyfyclj2h3yw8z23jqpvrh8jrm3bdxr9fwvb3l7fazq4mv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-lncs")
    (synopsis "BibLaTeX style for Springer Lecture Notes in Computer Science")
    (description
     "This is a BibLaTeX style for Springer Lecture Notes in Computer Science
(LNCS).  It extends the standard BiBTeX model by an acronym entry.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-lni
  (package
    (name "texlive-biblatex-lni")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-lni/"
                                     "tex/latex/biblatex-lni/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0038pa7pzd2hf9hh43nd7iq7761hrhq445ggi6b3ibw24s8mwif5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-lni")
    (synopsis "LNI style for BibLaTeX")
    (description
     "BibLaTeX style for the Lecture Notes in Informatics, which is published
by the Gesellschaft fur Informatik (GI e.V.).")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-luh-ipw
  (package
    (name "texlive-biblatex-luh-ipw")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-luh-ipw/"
                                     "tex/latex/biblatex-luh-ipw/bbx/"
                                     "tex/latex/biblatex-luh-ipw/cbx/"
                                     "tex/latex/biblatex-luh-ipw/lbx/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "18fnq2xjwsp5lq7vywbgzppbijqn4xarqga0g1rpj89g9q0zjs30"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-luh-ipw")
    (synopsis "BibLaTeX styles for social sciences")
    (description
     "The bundle is a small collection of styles for BibLaTeX.  It was
designed for citations in the Humanities, following the guidelines of style of
the institutes for the social sciences of the Leibniz University Hannover/LUH
(especially the Institute of Political Science).")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-manuscripts-philology
  (package
    (name "texlive-biblatex-manuscripts-philology")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-manuscripts-philology/"
                                     "tex/latex/biblatex-manuscripts-philology/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05d2z1w7wdzdinfw1j72y2d8f0nid0a6wsrvkwqkryl3nykyh4z5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-manuscripts-philology")
    (synopsis "Manage classical manuscripts with BibLaTeX")
    (description
     "The package adds a new entry type: @code{@@manuscript} to manage
manuscript in classical philology, for example to prepare a critical
edition.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-mla
  (package
    (name "texlive-biblatex-mla")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-mla/"
                                     "tex/latex/biblatex-mla/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09qlx067vhc4aflx9rjdbyq99xh3nzwlxj81cxp07mm2mykhxf39"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-mla")
    (synopsis "MLA style files for BibLaTeX")
    (description
     "The package provides BibLaTeX support for citations in the format
specified by the MLA handbook.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-morenames
  (package
    (name "texlive-biblatex-morenames")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-morenames/"
                                     "tex/latex/biblatex-morenames/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gwzxz1yqnq3q2fjrb4nf0370a90fbi6b20zvplj3lfiqmdkaiak"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-morenames")
    (synopsis "New names for standard BibLaTeX entry type")
    (description
     "This package adds new fields of ``name'' type to the standard entry
types of BibLaTeX.  For example, @samp{maineditor}, for a @code{@@collection},
means the editor of @code{@@mvcollection}, and not the editor of the
@code{@@collection}.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-ms
  (package
    (name "texlive-biblatex-ms")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bib/biblatex-ms/biblatex/"
                                     "bibtex/bst/biblatex-ms/"
                                     "doc/latex/biblatex-ms/"
                                     "tex/latex/biblatex-ms/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00xib8xvxl78qzxs66qmfyp4jdkcs3qx4ray2nwv1fffhj69aw3n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-etoolbox texlive-kvoptions texlive-logreq
                             texlive-pdftexcmds texlive-url))
    (home-page "https://ctan.org/pkg/biblatex-ms")
    (synopsis "Sophisticated bibliographies in LaTeX (multiscript version)")
    (description
     "This package is the multiscript' version of the BibLaTeX package
intended to solve the issues faced by those wishing to create multilingual
bibliographies.  It is intended to be backwards-compatible with the standard
BibLaTeX package and includes significantly enhanced optional functionality.

It requires the use of the multiscript version of Biber (biber-ms).")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-multiple-dm
  (package
    (name "texlive-biblatex-multiple-dm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-multiple-dm/"
                                     "tex/latex/biblatex-multiple-dm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07dp3ppz8kma5gh0bs98jnk7gvqqcp4l1ag60941myjvfsyqplxv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-multiple-dm")
    (synopsis "Load multiple datamodels in BibLaTeX")
    (description
     "The package adds the possibility to BibLaTeX to load data models from
multiple sources.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-musuos
  (package
    (name "texlive-biblatex-musuos")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-musuos/"
                                     "tex/latex/biblatex-musuos/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0iqp02yk8b9s18k8v78q2y5g507fsh2zk68dj0yc77gwa6gsmrmh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-musuos")
    (synopsis "BibLaTeX style for citations in @file{musuos.cls}")
    (description
     "The style is designed for use with the musuos class, but it should be
usable with other classes, too.")
    (license license:lppl)))

(define-public texlive-biblatex-nature
  (package
    (name "texlive-biblatex-nature")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-nature/"
                                     "tex/latex/biblatex-nature/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1bpgcwpd8sw28dn7q8bkxpnpqb741216gaasvqdpah9kj53zfypg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-nature")
    (synopsis "BibLaTeX support for @emph{Nature}")
    (description
     "The bundle offers styles that allow authors to use BibLaTeX when
preparing papers for submission to the journal @emph{Nature}.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-nejm
  (package
    (name "texlive-biblatex-nejm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-nejm/"
                                     "source/latex/biblatex-nejm/"
                                     "tex/latex/biblatex-nejm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fa12hbzrqd7brz90zhn3lklbqvvn5sw5l3y8pnm5xx604dkdiiw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-nejm")
    (synopsis "BibLaTeX style for the New England Journal of Medicine (NEJM)")
    (description
     "This is a BibLaTeX numeric style based on the design of the New England
Journal of Medicine (NEJM).")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-nottsclassic
  (package
    (name "texlive-biblatex-nottsclassic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-nottsclassic/"
                                     "tex/latex/biblatex-nottsclassic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13rvxykjyah5vn44sfagf1ggqhh1hd1qhz9m6jb87yhwz5snkrxh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-nottsclassic")
    (synopsis "Citation style for the University of Nottingham")
    (description
     "This citation-style covers the citation and bibliography rules of the
University of Nottingham.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-opcit-booktitle
  (package
    (name "texlive-biblatex-opcit-booktitle")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-opcit-booktitle/"
                                     "tex/latex/biblatex-opcit-booktitle/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "078fzk4i7sknaxkgn2lr54bcslqbjjsr1nb0z7y46v7kpg2lpcrf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-opcit-booktitle")
    (synopsis "Use @samp{op.@: cit.}@: for the booktitle of a subentry")
    (description
     "The default citation styles use the @samp{op.@: cit.}@: form in order to
have a shorter reference when a title has already been cited.  However, when
you cite two entries which share the same booktitle but not the same title,
the @samp{op.@: cit.}@: mechanism does not work.  This package fixes this.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-oxref
  (package
    (name "texlive-biblatex-oxref")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-oxref/"
                                     "source/latex/biblatex-oxref/"
                                     "tex/latex/biblatex-oxref/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "185swvbhpdicsgbxg3bpxkqkydl72vyl8k1mqxnxfi9ki9z7mbz9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-oxref")
    (synopsis "BibLaTeX styles inspired by the @emph{Oxford Guide to Style}")
    (description
     "This bundle provides four BibLaTeX styles that implement (many of) the
stipulations and examples provided by the 2014 @emph{New Hart's Rules} and the
2002 @emph{Oxford Guide to Style}:
@itemize

@item @code{oxnotes} is a style similar to the standard @code{verbose},
intended for use with footnotes;

@item @code{oxnum} is a style similar to the standard @code{numeric}, intended
for use with numeric in-text citations;

@item @code{oxalph} is a style similar to the standard @code{alphabetic},
intended for use with alphabetic in-text citations;

@item @code{oxyear} is a style similar to the standard @code{author-year},
intended for use with parenthetical in-text citations.

@end itemize

The bundle provides support for a wide variety of content types, including
manuscripts, audiovisual resources, social media and legal references.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-philosophy
  (package
    (name "texlive-biblatex-philosophy")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-philosophy/"
                                     "source/latex/biblatex-philosophy/"
                                     "tex/latex/biblatex-philosophy/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05fapyb6wwyv4mwjhgg3gasvqkwpwd6jxv095hird9011n6drrzm"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-local-tree")))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (native-inputs
     (list (texlive-local-tree
            (list texlive-biblatex
                  texlive-cochineal
                  texlive-csquotes
                  texlive-etoolbox
                  texlive-fontaxes
                  texlive-fontsize
                  texlive-guitlogo
                  texlive-hologo
                  texlive-hypdoc
                  texlive-inconsolata
                  texlive-libertine
                  texlive-listings
                  texlive-ltxdockit
                  texlive-manfnt
                  texlive-mdframed
                  texlive-metalogo
                  texlive-microtype
                  texlive-needspace
                  texlive-newtx
                  texlive-parskip
                  texlive-pgf
                  texlive-sectsty
                  texlive-upquote
                  texlive-xcolor
                  texlive-xkeyval
                  texlive-xstring
                  texlive-zref))))
    (home-page "https://ctan.org/pkg/biblatex-philosophy")
    (synopsis "Styles for using BibLaTeX for work in philosophy")
    (description
     "The bundle offers two styles --- @code{philosophy-classic} and
@code{philosophy-modern} --- that facilitate the production of two different
kinds of bibliography, based on the author-year style, with options and
features to manage the information about the translation of foreign texts or
their reprints.  Though the package's default settings are based on the
conventions used in Italian publications, these styles can be used with every
language recognized by Babel, possibly with some simple redefinitions.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-phys
  (package
    (name "texlive-biblatex-phys")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-phys/"
                                     "tex/latex/biblatex-phys/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "04qbfcxqp2gvsspvay35vwis8vdpsd86yiylf9k6rmjqk7f3s4px"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-phys")
    (synopsis
     "BibLaTeX implementation of the AIP and APS bibliography style")
    (description
     "The package provides an implementation of the bibliography styles of
both the AIP and the APS for BibLaTeX.  This implementation follows standard
BibLaTeX conventions, and can be used simply by loading BibLaTeX with the
appropriate option: @samp{\\usepackage[style=phys]@{biblatex@}}.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-publist
  (package
    (name "texlive-biblatex-publist")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-publist/"
                                     "tex/latex/biblatex-publist/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "01by2sb53568m5c5qg4psx74lyj45cj9c86j6ma7bk3312yijhsc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-publist")
    (synopsis "BibLaTeX bibliography support for publication lists")
    (description
     "The package provides a BibLaTeX bibliography style file (@file{.bbx})
for publication lists.  The style file draws on BibLaTeX's @code{authoryear}
style, but provides some extra features often desired for publication lists,
such as the omission of the author's own name from author or editor data.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-readbbl
  (package
    (name "texlive-biblatex-readbbl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-readbbl/"
                                     "tex/latex/biblatex-readbbl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0fr0p5c2v66vy86qx2m8i4p3p75d6qm506gv8xbfqckzyfzi9x6z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-readbbl")
    (synopsis "Read a @file{.bbl} file created by biber")
    (description
     "This small package modifies the BibLaTeX macro which reads a @file{.bbl}
file created by Biber.  It is thus possible to include a @file{.bbl} file into
the main document with the @code{environment} and send it to a publisher who
does not need to run the Biber program.  However, when the bibliography
changes one has to create a new @file{.bbl} file.")
    (license license:lppl)))

(define-public texlive-biblatex-realauthor
  (package
    (name "texlive-biblatex-realauthor")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-realauthor/"
                                     "tex/latex/biblatex-realauthor/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1av0vd7lwg5yad75b9fbi09s4bcaqd8bdz43kzyj2r7cimj8dpl0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-realauthor")
    (synopsis "Indicate the real author of a work")
    (description
     "This package allows using a new field @code{realauthor}, which
indicates the real author of a work, when published in a pseudepigraphic
name.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-sbl
  (package
    (name "texlive-biblatex-sbl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-sbl/"
                                     "makeindex/biblatex-sbl/"
                                     "tex/latex/biblatex-sbl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fhxvcyafvk1xxsiqhqi1lpvcfpjxmssdvrq3vj69qvbdc4zqp9d"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-sbl")
    (synopsis "Society of Biblical Literature (SBL) style files for BibLaTeX")
    (description
     "The package provides BibLaTeX support for citations in the format
specified by the second edition of the Society of Biblical Literature (SBL)
Handbook of Style.  A style file for writing SBL student papers is also
included.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-science
  (package
    (name "texlive-biblatex-science")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-science/"
                                     "tex/latex/biblatex-science/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0sw2bzhbgbfg5gajbm61x4243qrfna1ifbp9bl1swdmvw9g3hycd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-science")
    (synopsis "BibLaTeX implementation of the @emph{Science} bibliography style")
    (description
     "The bundle offers styles that allow authors to use BibLaTeX when
preparing papers for submission to the journal @emph{Science}.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-shortfields
  (package
    (name "texlive-biblatex-shortfields")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-shortfields/"
                                     "tex/latex/biblatex-shortfields/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14s6ykbnb6b2i811rjn44chv8f73yhnnywczn78zwbij5z6jgcjv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-shortfields")
    (synopsis "Use short forms of fields with BibLaTeX")
    (description
     "The BibLaTeX package provides @code{shortseries} and @code{shortjournal}
field, but the default styles don't use them.  It also provides a mechanism to
print the equivalence between short forms of fields and long
fields (@code{\\printbiblist}), but this mechanism does not allow mixing
between different type of short fields, for example, between short forms of
journal title and short forms of series titles.

This package provides a solution to these two problems.  If
a @code{shortjournal} field is defined, it prints it instead of the
@code{\\journal} field.  If a @code{shortseries} field is defined, it prints
it instead of the @code{\\series} field.  It provides
a @code{\\printbibshortfields} command to print a list of the sort forms of
the fields.  This list also includes the claves defined with the
@code{biblatex-claves} package.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-socialscienceshuberlin
  (package
    (name "texlive-biblatex-socialscienceshuberlin")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-socialscienceshuberlin/"
                                     "tex/latex/biblatex-socialscienceshuberlin/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1klcvjf8vz3bg0q7pmci6hhih52ph0z0hhkbv3iippyy33jhijrc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-socialscienceshuberlin")
    (synopsis "BibLaTeX-style for the social sciences at HU Berlin")
    (description
     "This is a BibLaTeX style for the social sciences at the
Humboldt-Universitat zu Berlin.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-software
  (package
    (name "texlive-biblatex-software")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-software/"
                                     "source/latex/biblatex-software/"
                                     "tex/latex/biblatex-software/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1s5d11gdzqr4q04x49sx0l5vf23yayfqcs1vdpbz2miqah1dm5iw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-software")
    (synopsis "BibLaTeX stylefiles for software products")
    (description
     "This package implements software entry types for BibLaTeX in the form of
a bibliography style extension.  It requires the Biber backend.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-source-division
  (package
    (name "texlive-biblatex-source-division")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-source-division/"
                                     "tex/latex/biblatex-source-division/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07y25624fpmx9hfmyyga8wh5cmvl14lqbmflglpl60jqsy622mgi"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-source-division")
    (synopsis "References by division in classical sources")
    (description
     "The package enables the user to make reference to division marks (such
as book, chapter, section), in the document being referenced, in addition to
the page-based references that BibTeX-based citations have always had.  The
citation is made in the same way as the LaTeX standard, but what's inside the
square brackets may include the division specification.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-spbasic
  (package
    (name "texlive-biblatex-spbasic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-spbasic/"
                                     "tex/latex/biblatex-spbasic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "071d1iqy1scz3wncsfv1backp5b7ly28rn5k0wy67wwz1vxa0d17"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-spbasic")
    (synopsis "BibLaTeX style emulating Springer's old @file{spbasic.bst}")
    (description
     "This package provides a bibliography and citation style for BibLaTeX and
Biber for typesetting articles for Springer's journals.  It is the same as the
old BibTeX style @file{spbasic.bst}.")
    (license license:lppl)))

(define-public texlive-biblatex-subseries
  (package
    (name "texlive-biblatex-subseries")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-subseries/"
                                     "tex/latex/biblatex-subseries/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09kqqccn8dlaydz1v2szllryaxrj0c36z63ynhrv9dvd1sk0vdhd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-subseries")
    (synopsis "Manages subseries with BibLaTeX")
    (description
     "Some publishers organize book series with subseries.  In this case, two
numbers are associated with one volume: the number inside the series and the
number inside the subseries.  This package provides new fields to manage such
system.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-swiss-legal
  (package
    (name "texlive-biblatex-swiss-legal")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-swiss-legal/"
                                     "tex/latex/biblatex-swiss-legal/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1x4dy1vxrbx0xkw3vysvcds0s1jz9w1pwri5ypiidj517sdsnff4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-swiss-legal")
    (synopsis
     "Bibliography and citation styles following Swiss legal practice")
    (description
     "The package provides BibLaTeX bibliography and citation styles for
documents written in accordance with Swiss legal citation standards in either
French or German.  However, the package is at present outdated and does not
work properly with newer versions of BibLaTeX.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-trad
  (package
    (name "texlive-biblatex-trad")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-trad/"
                                     "tex/latex/biblatex-trad/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "19ma61dsdpsm52zm7wnkqccmx54g9gvh05d10c59q86ffv1dnlmx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-trad")
    (synopsis "Traditional BibTeX styles with BibLaTeX")
    (description
     "The bundle provides implementations of the traditional BibTeX styles
(@code{plain}, @code{abbrev}, @code{unsrt} and @code{alpha}) with BibLaTeX.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-true-citepages-omit
  (package
    (name "texlive-biblatex-true-citepages-omit")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-true-citepages-omit/"
                                     "tex/latex/biblatex-true-citepages-omit/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1mfkmn5g6r61mlyf3vx98q18a042yjkmh3drid9vs0kaj66di4n5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-true-citepages-omit")
    (synopsis
     "Correct some limitations of BibLaTeX @samp{citepages=omit} option")
    (description
     "This package deals with a limitation of the @samp{citepages=omit} option
of the verbose family of BibLaTeX citestyles.  The option works when you
@code{\\cite[xx]@{key@}}, but not when you @code{\\cite[\\pno~xx, some
text]@{key@}}.  The package corrects this problem.")
    (license license:lppl1.3+)))

(define-public texlive-biblatex-unified
  (package
    (name "texlive-biblatex-unified")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-unified/"
                                     "tex/latex/biblatex-unified/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0d9b20m10dlkii2zxj00xgascqarf8qv1vfz28v7hnm9vd54h3rw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-unified")
    (synopsis "BibLaTeX unified stylesheet for linguistics journals")
    (description
     "BibLaTeX-unified is an opinionated BibLaTeX implementation of the
Unified Stylesheet for Linguistics Journals.")
    (license license:lppl1.3c)))

(define-public texlive-biblatex-vancouver
  (package
    (name "texlive-biblatex-vancouver")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex-vancouver/"
                                     "tex/latex/biblatex-vancouver/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1sji214mi5garp5h3if1fh1rvck573cxg61rgdlyy543883nrxv7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex-vancouver")
    (synopsis "Vancouver style for BibLaTeX")
    (description
     "This package provides the Vancouver reference style for BibLaTeX.  It is
based on the @code{numeric} style and requires Biber.")
    (license license:gpl3+)))

(define-public texlive-biblatex2bibitem
  (package
    (name "texlive-biblatex2bibitem")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblatex2bibitem/"
                                     "tex/latex/biblatex2bibitem/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1cd9b7vhmlx2clqr2gpwqs4hjga9mjqlxj7m74kq555rn5ppkyyn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblatex2bibitem")
    (synopsis "Convert BibLaTeX-generated bibliography to bibitems")
    (description
     "Some journals accept the reference list only as @code{\\bibitems}.  If
you use BibTeX, there is no problem: just paste the content of the @file{.bbl}
file into your document.  However, there was no out-of-the-box way to do the
same for BibLaTeX, and you had to struggle with searching appropriate
@file{.bst} files, or formatting your reference list by hand, or something
like that.  Using the workaround provided by this package solves the
problem.")
    (license license:lppl1.3c)))

(define-public texlive-bibleref
  (package
    (name "texlive-bibleref")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibleref/"
                                     "source/latex/bibleref/"
                                     "tex/latex/bibleref/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fgb6lljdsdvkbs4i7yvwvgjbpbc8vnki0wv3gdi9yq6s5j6lzv5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibleref")
    (synopsis "Format @emph{Bible} citations")
    (description
     "The bibleref package offers consistent formatting of references to parts
of the Christian @emph{Bible}, in a number of well-defined formats.")
    (license license:lppl1.3+)))

(define-public texlive-bibleref-french
  (package
    (name "texlive-bibleref-french")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibleref-french/"
                                     "source/latex/bibleref-french/"
                                     "tex/latex/bibleref-french/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12y9xdsz0swh3rndmzk7ch9fbszbhcp0ixb21jyy36xnb3x80wa5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibleref-french")
    (synopsis "French translations for @code{bibleref}")
    (description
     "The package provides translations and alternative typesetting
conventions for use of @code{bibleref} in French.")
    (license license:lppl1.3+)))

(define-public texlive-bibleref-german
  (package
    (name "texlive-bibleref-german")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibleref-german/"
                                     "tex/latex/bibleref-german/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0nqky0x9j6w3xg2vaj29db7l0zjw88d7bl6vs8864hqz2i6h2pcw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibleref-german")
    (synopsis "German adaptation of @code{bibleref}")
    (description
     "The package provides translations and various formats for the use of
@code{bibleref} in German documents.  The German naming of the Bible books
complies with the @emph{Loccumer Richtlinien} (Locum guidelines).  In
addition, the Vulgate (Latin Bible) is supported.")
    (license license:lppl1.3+)))

(define-public texlive-bibleref-lds
  (package
    (name "texlive-bibleref-lds")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibleref-lds/"
                                     "source/latex/bibleref-lds/"
                                     "tex/latex/bibleref-lds/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0wqmpzml3yyhzv7bx5wkfl2ni0qinz0kjzdanh6rx0bczp29dlfh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibleref-lds")
    (synopsis
     "Bible references, including scriptures of the Church of Jesus Christ of
Latter Day Saints")
    (description
     "The package extends the @code{bibleref-mouth} package to support
references to the scriptures of The Church of Jesus Christ of Latter-day
Saints (LDS).")
    (license license:lppl1.3+)))

(define-public texlive-bibleref-mouth
  (package
    (name "texlive-bibleref-mouth")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibleref-mouth/"
                                     "source/latex/bibleref-mouth/"
                                     "tex/latex/bibleref-mouth/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0dakbx8rnjs9yjnn1xal9y0bmh165lqhbjj3bns974vkyb7nla4m"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibleref-mouth")
    (synopsis "Consistent formatting of @emph{Bible} references")
    (description
     "The package allows @emph{Bible} references to be formatted in
a consistent way.  It is similar to the @code{bibleref} package, except that
the formatting macros are all purely expandable --- that is, they are all
implemented in TeX's mouth.  This means that they can be used in any
expandable context, such as an argument to a @code{\\url} command.")
    (license license:lppl1.3+)))

(define-public texlive-bibleref-parse
  (package
    (name "texlive-bibleref-parse")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibleref-parse/"
                                     "tex/latex/bibleref-parse/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "01jhic1idgjgv9hsx5gih4230krfqm8pc3j7f4kn22w2jnw7fs68"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibleref-parse")
    (synopsis "Specify @emph{Bible} passages in human-readable format")
    (description
     "The package parses @emph{Bible} passages that are given in human
readable format.  It accepts a wide variety of formats.  This allows for
a simpler and more convenient interface to the functionality of the
@code{bibleref} package.")
    (license license:lppl1.3+)))

(define-public texlive-biblist
  (package
    (name "texlive-biblist")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biblist/" "tex/latex/biblist/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1r7cd68yr8pnmqnr39qzp42kxqcx89cqzz81ig9y7qgicznji0i8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biblist")
    (synopsis "Print a BibTeX database")
    (description
     "The package provides the means of listing an entire BibTeX database,
avoiding the potentially large (macro) impact associated with
@code{\\nocite@{*@}}.")
    (license license:gpl3+)))

(define-public texlive-bidihl
  (package
    (name "texlive-bidihl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/xelatex/bidihl/" "tex/xelatex/bidihl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "021q2qhfl2ww95nbp63p3j7lv9wjj5s9wvn2sj5zyj1pcw75l8ir"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bidihl")
    (synopsis "Experimental Bidi-aware text highlighting")
    (description
     "This package provides experimental Bidi-aware text highlighting.")
    (license license:lppl1.3+)))

(define-public texlive-bibtexperllibs
  (package
    (name "texlive-bibtexperllibs")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/ltx2unitxt.1"
                                     "doc/man/man1/ltx2unitxt.man1.pdf"
                                     "scripts/bibtexperllibs/"
                                     "source/support/bibtexperllibs/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14v3gr9cs5iyg40cwfnifhq478chsc6ar4jsywrqixgs9psbj8fk"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibtexperllibs")
    (synopsis "BibTeX Perl Libraries")
    (description
     "This package provides BibTeX related Perl libraries.")
    (license
     (list license:artistic2.0
           license:gpl1
           license:public-domain))))

(define-public texlive-bibtools
  (package
    (name "texlive-bibtools")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "bibtex/bst/bibtools/"
                              "source/bibtex/bibtools/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0ma8zdk8wr1hniax1f6mqps2ppc14awikwh78qgb7v5s4js0scr1"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibtools")
    (synopsis "Bib management tools")
    (description
     "This package provides a set of bibliography tools.  It includes:
@itemize

@item @code{aux2bib}, a Perl script which will take an @file{.aux} file and
make a portable @file{.bib} file to go with it;

@item @code{bibify}, a shell script that will optimize away one pass of the
LaTeX-BibTeX cycle, in some cases;

@item @code{bibkey}, a shell script that finds entries whose @samp{keyword}
field matches the given keys (uses @command{sed} and @command{awk});

@item @code{cleantex}, a shell script to tidy up after a LaTeX run;

@item @code{looktex}, a shell script to list entries that match a given regexp;

@item @code{makebib}, a shell script to make an exportable @file{.bib} file
from an existing (set of) @file{.bib} file(s) and an optional set of
citations (uses @command{sed});

@item @code{printbib}, a shell script to make a DVI file from a @file{.bib}
file, sorted by cite key, and including fields like @samp{keyword},
@samp{abstract}, and @samp{comment};

@item @code{bib2html}, a Perl script that makes a browsable HTML version of
a bibliography (several @file{.bst} files are supplied);

@item and @code{citekeys}, a shell script that lists the citation keys of
a @file{.bib} file.
@end itemize")
    (license license:lppl)))

(define-public texlive-bibtopic
  (package
    (name "texlive-bibtopic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibtopic/"
                                     "source/latex/bibtopic/"
                                     "tex/latex/bibtopic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1b4lhlw29y37c7pxmqvy4ahgb85bmxrgnr2jhzvdhzdm5y9md7a3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (home-page "https://ctan.org/pkg/bibtopic")
    (synopsis "Include multiple bibliographies in a document")
    (description
     "The package allows the user to include several bibliographies covering
different topics or bibliographic material into a document (e.g., one
bibliography for primary literature and one for secondary literature).  The
package provides commands to include either all references from a @file{.bib}
file, only the references actually cited or those not cited in your document.
The user has to construct a separate @file{.bib} file for each bibliographic
topic, each of which will be processed separately by BibTeX.  If you want to
have bibliographies specific to one part of a document, see the packages
@code{bibunits} or @code{chapterbib}.")
    (license license:gpl3+)))

(define-public texlive-bibtopicprefix
  (package
    (name "texlive-bibtopicprefix")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibtopicprefix/"
                                     "source/latex/bibtopicprefix/"
                                     "tex/latex/bibtopicprefix/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fadq51adii1453v31xj7p328h4rvans1ynmxjcmnwhlj4hdjp5n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibtopicprefix")
    (synopsis "Prefix references to bibliographies produced by
@code{bibtopic}")
    (description
     "The package permits users to apply prefixes (fixed strings) to
references to entries in bibliographies produced by the @code{bibtopic}
package.")
    (license license:lppl)))

(define-public texlive-bibunits
  (package
    (name "texlive-bibunits")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bibunits/"
                                     "source/latex/bibunits/"
                                     "tex/latex/bibunits/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0hlz277lskhcs5xr0qn2rram957himj67hkwqqsjjap8khf7fz6v"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bibunits")
    (synopsis "Multiple bibliographies in one document")
    (description
     "The package provide a mechanism to generate separate bibliographies for
different units (chapters, sections or bibunit-environments) of a text.  The
package separates the citations of each unit of text into a separate file to
be processed by BibTeX.  The global bibliography section produced by LaTeX may
also appear in the document and citations can be placed in both the local unit
and the global bibliographies at the same time.")
    (license license:lppl)))

(define-public texlive-binomexp
  (package
    (name "texlive-binomexp")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/binomexp/"
                                     "source/latex/binomexp/"
                                     "tex/latex/binomexp/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wmhsqqz2hy75x20srs1237yba4s00yrdhr52hkrh3nldzqwhqcc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/binomexp")
    (synopsis "Calculate Pascal's triangle")
    (description
     "The package calculates and prints rows of Pascal's triangle.  It may be
used to print successive rows of the triangle, or to print the rows inside an
@code{array} or @code{tabular} environment.")
    (license license:lppl)))

(define-public texlive-biocon
  (package
    (name "texlive-biocon")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/biocon/" "tex/latex/biocon/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0m3qqrl5z071w4zs8fs3wv0b4ix0s6cqh4ixzqj28d74aaf9r1y1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biocon")
    (synopsis "Typesetting biological species names")
    (description
     "The biocon--biological conventions--package aids the typesetting of some
biological conventions.  At the moment, it makes a good job of typesetting
species names (and ranks below the species level).  A distinction is made
between the Plant, Fungi, Animalia and Bacteria kingdoms.  There are default
settings for the way species names are typeset, but they can be customized.
Different default styles are used in different situations.")
    (license license:gpl3+)))

(define-public texlive-biolett-bst
  (package
    (name "texlive-biolett-bst")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/biolett-bst/"
                                     "doc/bibtex/biolett-bst/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xfr79xw0ih61s5qapx2adiyjj0d149nslpmkd1fw29g1v4zrjd2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/biolett-bst")
    (synopsis "BibTeX style for the journal @emph{Biology Letters}")
    (description
     "This package provides a BibTeX style (@file{.bst}) file for the journal
@emph{Biology Letters} published by the Royal Society.")
    (license license:lppl1.0+)))

(define-public texlive-bitelist
  (package
    (name "texlive-bitelist")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/bitelist/"
                                     "source/generic/bitelist/"
                                     "tex/generic/bitelist/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1jv9w2jgx8q2k9xx0n4azlngsh26zs37ihavq0xy4n62ajf64ps6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bitelist")
    (synopsis "Split list, in TeX's mouth")
    (description
     "The package provides commands for splitting a token list at the first
occurrence of another (specified) token list.  The package's mechanism differs
from those of packages providing similar features, in the following ways: the
method uses TeX's mechanism of reading delimited macro parameters; splitting
macros work by pure expansion, without assignments; the operation is carried
out in a single macro call.  A variant of the operation is provided, that
retains outer braces.")
    (license license:lppl1.3+)))

(define-public texlive-bitpattern
  (package
    (name "texlive-bitpattern")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bitpattern/"
                                     "source/latex/bitpattern/"
                                     "tex/latex/bitpattern/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00f423jran7qqg9fwy3p7w1lhi20cxzlvn4hzsrz5pwd87bmvxxb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bitpattern")
    (synopsis "Typeset bit pattern diagrams")
    (description
     "This package provides a package to typeset bit pattern diagrams such as those
used to describe hardware, data format or protocols.")
    (license license:lppl)))

(define-public texlive-bitter
  (package
    (name "texlive-bitter")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/bitter/"
                                     "fonts/enc/dvips/bitter/"
                                     "fonts/map/dvips/bitter/"
                                     "fonts/tfm/huerta/bitter/"
                                     "fonts/truetype/huerta/bitter/"
                                     "fonts/type1/huerta/bitter/"
                                     "fonts/vf/huerta/bitter/"
                                     "tex/latex/bitter/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0xh3vssfrfkdrwzsmid7frm4n9gjb8x7czh8zyybdcvv9vb8n3d9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bitter")
    (synopsis "Bitter family of fonts with LaTeX support")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX, and LuaLaTeX support for
the Bitter family of fonts.  Bitter is a contemporary slab-serif typeface for
text.  There are regular and bold weights and an italic, but no bold italic.")
    (license (list license:lppl license:silofl1.1))))

(define-public texlive-bjfuthesis
  (package
    (name "texlive-bjfuthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bjfuthesis/"
                                     "tex/latex/bjfuthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qhb9kighs4ljmnn94qaihpllhwvmi76j3p8yylyjw2hzip102wr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bjfuthesis")
    (synopsis "Thesis class for Beijing Forestry University")
    (description
     "This is a class file for producing dissertations and theses according to
the @acronym{BJFU, Beijing Forestry University} guidelines for undergraduate
theses and dissertations.")
    (license license:gpl3)))

(define-public texlive-blacklettert1
  (package
    (name "texlive-blacklettert1")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/blacklettert1/"
                                     "fonts/tfm/public/blacklettert1/"
                                     "fonts/vf/public/blacklettert1/"
                                     "source/fonts/blacklettert1/"
                                     "tex/latex/blacklettert1/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0fvrz3rbcdgzrg3rg3nfvdb52zxn9vhsx0xd8rc6xryzrqaib7q0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/blacklettert1")
    (synopsis "T1-encoded versions of Haralambous old German fonts")
    (description
     "This package contains virtual fonts that offer T1-alike encoded variants
of old German fonts Gothic, Schwabacher and Fraktur (which are also available
in Adobe type 1 format).  The package includes LaTeX macros to embed the fonts
into the LaTeX font selection scheme.")
    (license license:lppl1.2+)))

(define-public texlive-bmstu
  (package
    (name "texlive-bmstu")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bmstu/" "tex/latex/bmstu/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ipdwd78b5w4v85nqlbxf8gi42mdz19r5mdjj3agwlgb0z6icjra"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bmstu")
    (synopsis "LaTeX class for Bauman Moscow State Technical University")
    (description
     "The class defines commands and environments for creating reports and
explanatory notes in Bauman Moscow State Technical University (Russia).")
    (license
     (list license:lppl1.3+
           ;; File "bmstu-logo.pdf" is a logo of Bauman Moscow ;; State
           ;; Technical University and complies with fair use (article 1274 of
           ;; the Russian Civil Code).
           (license:fsdg-compatible "file://doc/latex/bmstu/README.md")))))

(define-public texlive-bmstu-iu8
  (package
    (name "texlive-bmstu-iu8")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bmstu-iu8/"
                                     "tex/latex/bmstu-iu8/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jl0h6f8v1v4afkvlcyqbhv9jrjxi2wj7j9lfgzbj7y2487lpcgg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs
     (list texlive-algorithm2e
           texlive-amscls
           texlive-anyfontsize
           texlive-babel
           texlive-biblatex
           texlive-bookmark
           texlive-caption
           texlive-chngcntr
           texlive-csquotes
           texlive-enumitem
           texlive-fancyhdr
           texlive-float
           texlive-fontspec
           texlive-geometry
           texlive-glossaries
           texlive-glossaries-extra
           texlive-graphics
           texlive-ifoddpage
           texlive-koma-script
           texlive-lastpage
           texlive-lineno
           texlive-listings
           texlive-ltablex
           texlive-multirow
           texlive-nowidow
           texlive-oberdiek
           texlive-relsize
           texlive-setspace
           texlive-stackengine
           texlive-tabto-ltx
           texlive-titlesec
           texlive-tools
           texlive-totcount
           texlive-ulem
           texlive-xassoccnt
           texlive-xcolor
           texlive-xltabular
           texlive-xstring))
    (home-page "https://ctan.org/pkg/bmstu-iu8")
    (synopsis "Class for IU8 reports")
    (description
     "This package consists of a class file and style files for writing
reports at the IU8 department of IU faculty of @acronym{BMSTU, Bauman Moscow
State Technical University}.  The class defines all headings, structure
elements and other things in respect of Russian standard GOST 7.32-2017.  But
there are correctives to be compatible with our local IU8 department
requirements.")
    (license license:expat)))

(define-public texlive-bodeplot
  (package
    (name "texlive-bodeplot")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bodeplot/"
                                     "source/latex/bodeplot/"
                                     "tex/latex/bodeplot/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0d90y36pfs11yyn7gjjv6nmd09p618qy33wagpwnrqlcw2f0za22"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bodeplot")
    (synopsis "Draw Bode, Nyquist and Nichols plots with Gnuplot or
 @code{pgfplots}")
    (description
     "This is a LaTeX package to plot Bode, Nichols, and Nyquist diagrams.  It
provides added functionality over the similar @code{bodegraph} package:

@itemize
@item new @code{\\BodeZPK} and @code{\\BodeTF} commands to generate Bode plots
of any transfer function given either poles, zeros, gain, and delay, or
numerator and denominator coefficients and delay;
@item support for unstable poles and zeros;
@item support for complex poles and zeros;
@item support for general stable and unstable second order transfer functions;
@item support for both Gnuplot (default) and @code{pgfplots};
@item support for linear and asymptotic approximation of magnitude and phase
plots of any transfer function given poles, zeros, and gain.
@end itemize")
    (license license:lppl1.3c)))

(define-public texlive-bohr
  (package
    (name "texlive-bohr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bohr/" "tex/latex/bohr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "02slnkmxha906y7z04dffamy0lywq95l9yvklxjlm2kvjhqv2wnw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bohr")
    (synopsis "Simple atom representation according to the Bohr model")
    (description
     "The package provides means for the creation of simple Bohr models of
atoms up to the atomic number 112.  In addition, commands are provided to
convert atomic numbers to element symbols or element names and vice versa.")
    (license license:lppl1.3+)))

(define-public texlive-boisik
  (package
    (name "texlive-boisik")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/boisik/"
                                     "fonts/source/public/boisik/"
                                     "fonts/tfm/public/boisik/"
                                     "tex/latex/boisik/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xg00nb28dgc8ch85n1mgj5n0cshkvyvgv98qawxn2xvx14vqw89"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     ;; FIXME: Font metrics generation fails for "bskmab10" font, with error:
     ;; "! Strange path (turning number is zero)".
     (list #:phases
           #~(modify-phases %standard-phases
               (delete 'generate-font-metrics))))
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/boisik")
    (synopsis "Font inspired by Baskerville design")
    (description
     "Boisik is a serif font set (inspired by the Baskerville typeface),
written in Metafont.  The set comprises roman and italic text fonts and maths
fonts.  LaTeX support is offered for use with OT1, IL2 and OM* encodings.")
    (license license:gpl2)))

(define-public texlive-boldtensors
  (package
    (name "texlive-boldtensors")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/boldtensors/"
                                     "tex/latex/boldtensors/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06c0wa4qai0qdbad44464nc5zap9bc03yjn89yagsr4dzxif0jrg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/boldtensors")
    (synopsis
     "Bold latin and greek characters through simple prefix characters")
    (description
     "This package provides bold latin and greek characters within
@code{\\mathversion@{normal@}}, by using @samp{~} and @samp{\"} as prefix
characters.")
    (license license:gpl3+)))

(define-public texlive-bookdb
  (package
    (name "texlive-bookdb")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/bookdb/" "doc/bibtex/bookdb/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0df6bzmc90af3v4nzv6hp9padpc7h680vm2hy8wzb268z3ld268h"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bookdb")
    (synopsis "BibTeX style file for cataloguing a home library")
    (description
     "This package provides an extended book entry for use in cataloguing
a home library.  The extensions include fields for binding, category,
collator, condition, copy, illustrations, introduction, location, pages, size,
value, volumes.")
    (license license:lppl1.3+)))

(define-public texlive-bookhands
  (package
    (name "texlive-bookhands")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/bookhands/"
                                     "fonts/afm/public/bookhands/"
                                     "fonts/map/dvips/bookhands/"
                                     "fonts/source/public/bookhands/"
                                     "fonts/tfm/public/bookhands/"
                                     "fonts/type1/public/bookhands/"
                                     "source/fonts/bookhands/"
                                     "tex/latex/bookhands/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ssz80jwmd00x7d1x1xc6i61kbs30bc875bvakd5gymdw2kv7s3z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/bookhands")
    (synopsis "Collection of book-hand fonts")
    (description
     "This is a set of book-hand (Metafont) fonts and packages covering
manuscript scripts from the 1st century until Gutenberg and Caxton.  The
included hands are: Square Capitals (1st century onwards); Roman
Rustic (1st-6th centuries); Insular Minuscule (6th cenury onwards);
Carolingian Minuscule (8th-12th centuries); Early Gothic (11th-12th
centuries); Gothic Textura Quadrata (13th-15th centuries); Gothic Textura
Prescisus vel sine pedibus (13th century onwards); Rotunda (13-15th
centuries); Humanist Minuscule (14th century onwards); Uncial (3rd-6th
centuries); Half Uncial (3rd-9th centuries); Artificial Uncial (6th-10th
centuries); and Insular Majuscule (6th-9th centuries).")
    (license license:lppl)))

(define-public texlive-booktabs-de
  (package
    (name "texlive-booktabs-de")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/booktabs-de/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ld7zdr88xjhnnl2x734rxr7h0fz84awapmv50p1kzn6fczrwyam"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/booktabs-de")
    (synopsis "German version of @code{booktabs}")
    (description
     "This is a German translation of the @code{booktabs} package
documentation.")
    (license license:gpl3+)))

(define-public texlive-booktabs-fr
  (package
    (name "texlive-booktabs-fr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/booktabs-fr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xkqh6r2q835xaa92b24fzf61jsm85280570hivvwnch96i4fnh3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/booktabs-fr")
    (synopsis "French translation of @code{booktabs} documentation")
    (description
     "This package provides a French translation of @code{booktabs}
documentation.")
    (license license:lppl)))

(define-public texlive-boondox
  (package
    (name "texlive-boondox")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/boondox/"
                                     "fonts/map/dvips/boondox/"
                                     "fonts/tfm/public/boondox/"
                                     "fonts/type1/public/boondox/"
                                     "fonts/vf/public/boondox/"
                                     "tex/latex/boondox/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0rsid25qlsbj3z60h8n22m8l6gyk4nbr598c6z9azldpxn88dz91"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/boondox")
    (synopsis "Mathematical alphabets derived from the STIX fonts")
    (description
     "The package contains a number of PostScript fonts derived from the STIX
OpenType fonts that may be used in maths mode in regular and bold weights for
Calligraphic, Fraktur and Double-struck alphabets.  Virtual fonts with metrics
suitable for maths mode are provided, as are LaTeX support files.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-borceux
  (package
    (name "texlive-borceux")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/borceux/"
                                     "tex/generic/borceux/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fil90jsiqg7b0if3yq0jxv4rh7vanj2yj9pv8pnfkhvqii4p821"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/borceux")
    (synopsis "Diagram macros by Francois Borceux")
    (description
     "The macros support the construction of diagrams, such as those that
appear in category theory texts.  The user gives the list of vertices and
arrows to be included, just as when composing a matrix, and the program takes
care of computing the dimensions of the arrows and realizing the page setting.
All the user has to do about the arrows is to specify their
type (monomorphism, pair of adjoint arrows, etc.) and their direction (north,
south-east, etc.); 12 types and 32 directions are available.")
    ;; "You may freely use, modify, and/or distribute each of the
    ;; files in this package without limitation."
    (license (license:fsf-free "file://doc/generic/borceux/README"))))

(define-public texlive-bosisio
  (package
    (name "texlive-bosisio")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bosisio/" "source/latex/bosisio/"
                                     "tex/latex/bosisio/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1z4s0dgvwffy4356zj0m6kkk9lqzq1180km5sg5syvk11m7mxffn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (home-page "https://ctan.org/pkg/bosisio")
    (synopsis "Collection of packages by Francesco Bosisio")
    (description
     "This package provides a collection of packages containing: @code{accenti},
@code{dblfont}, @code{envmath}, @code{evenpage}, @code{graphfig},
@code{mathcmd}, @code{quotes}, and @code{sobolev}.")
    (license license:lppl)))

(define-public texlive-bpchem
  (package
    (name "texlive-bpchem")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bpchem/" "source/latex/bpchem/"
                                     "tex/latex/bpchem/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qm0w43iyh42l3srhqsgqq5yjxx3ghxamq6nh68qq6x4izlyv51z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bpchem")
    (synopsis "Typeset chemical names, formulae, etc")
    (description
     "The package provides support for typesetting simple chemical formulae,
those long IUPAC compound names, and some chemical idioms.  It also supports
the labelling of compounds and reference to labelled compounds.")
    (license license:lppl)))

(define-public texlive-br-lex
  (package
    (name "texlive-br-lex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/br-lex/" "tex/latex/br-lex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "154mjkywvyif57zvqbq5wxs84357148km2x0lqqdx6nanr3vwmxb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/br-lex")
    (synopsis "Class for typesetting Brazilian legal texts")
    (description
     "This class implements rules to typeset Brazilian legal texts.  Its
purpose is to be an easy-to-use implementation for the end-user.")
    (license license:lppl1.3+)))

(define-public texlive-braille
  (package
    (name "texlive-braille")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/braille/" "tex/latex/braille/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "021nxjz23gn8zjrrzxggjaqq3gv894c512x15kffywm8h5k0bzy5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/braille")
    (synopsis "Support for Braille")
    (description
     "This package allows the user to produce Braille documents on paper for
the blind without knowing Braille.  Python scripts @file{grade1.py} and
@file{grade2.py} convert ordinary text to grade 1 and 2 Braille tags; then,
the LaTeX package takes the tags and prints out corresponding Braille
symbols.")
    (license license:lppl1.3+)))

(define-public texlive-brandeis-dissertation
  (package
    (name "texlive-brandeis-dissertation")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/brandeis-dissertation/"
                                     "source/latex/brandeis-dissertation/"
                                     "tex/latex/brandeis-dissertation/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0dsg285wcpfxd0p1fs1dac87c0csbv21zz76cdjgishasmqrxdmg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/brandeis-dissertation")
    (synopsis "Class for Brandeis University dissertations")
    (description
     "The class will enable the user to typeset a dissertation which adheres
to the formatting guidelines of Brandeis University @acronym{GSAS, Graduate
School of Arts and Sciences}.")
    (license license:lppl1.2+)))

(define-public texlive-brandeis-problemset
  (package
    (name "texlive-brandeis-problemset")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/brandeis-problemset/"
                                     "tex/latex/brandeis-problemset/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "172l1jmvacjg9j5fkyrmsrhsj5pyaaq8g9pfvkahcnhk3jfc3pnp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/brandeis-problemset")
    (synopsis "Document class for COSI Problem sets at Brandeis University")
    (description
     "Brandeis University's computer science courses often assign problem sets
which require fairly rigorous formatting.  This document class, which extends
@code{article}, provides a simple way to typeset these problem sets in LaTeX.")
    (license license:lppl1.3c)))

(define-public texlive-brandeis-thesis
  (package
    (name "texlive-brandeis-thesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/brandeis-thesis/"
                                     "source/latex/brandeis-thesis/"
                                     "tex/latex/brandeis-thesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03m5m5l8jw5jkrqc4pzmk0ghijlaw25r1bn9mhva6m9kxz1sj9q4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/brandeis-thesis")
    (synopsis "Class for Brandeis University M.A. theses")
    (description
     "@file{brandeis-thesis.cls} provides the structures and formatting
information for an M.A.@: thesis for the Brandeis University Graduate School
of Arts and Sciences.")
    (license license:lppl1.3c)))

(define-public texlive-breakcites
  (package
    (name "texlive-breakcites")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/breakcites/"
                                     "tex/latex/breakcites/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12jiw135xi12k0f46zmza4jw141gq919c6q0ijv8xlic0wra4gyy"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/breakcites")
    (synopsis "Ensure that multiple citations may break at line end")
    (description
     "This package makes a very minor change to the operation of the
@code{\\cite} command so that multiple citations may break at line end.  Note
that the change is not necessary in unmodified LaTeX; however, there remain
packages that restore the undesirable behaviour of the command as provided in
LaTeX 2.09.  Neither @code{cite} nor @code{natbib} make this mistake.")
    ;; The file below states: "You may freely use, modify, and/or
    ;; distribute this package without limitation."
    (license (license:fsf-free "file://doc/latex/breakcites/README"))))

(define-public texlive-bropd
  (package
    (name "texlive-bropd")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bropd/" "source/latex/bropd/"
                                     "tex/latex/bropd/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1cyyadfvrcym4vvxl9p9zb88692m0578nqljip12xxahb4srcyb9"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-local-tree")))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (native-inputs (list (texlive-local-tree
                          (list texlive-etoolbox
                                texlive-hypdoc))))
    (home-page "https://ctan.org/pkg/bropd")
    (synopsis "Simplified brackets and differentials in LaTeX")
    (description
     "The package simplifies the process of writing differential operators and
brackets in LaTeX.  The commands facilitate the easy manipulation of equations
involving brackets and allow partial differentials to be expressed in an
alternate form.")
    (license license:lppl1.3+)))

(define-public texlive-brushscr
  (package
    (name "texlive-brushscr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/brushscr/"
                                     "dvips/brushscr/"
                                     "fonts/afm/public/brushscr/"
                                     "fonts/map/dvips/brushscr/"
                                     "fonts/tfm/public/brushscr/"
                                     "fonts/type1/public/brushscr/"
                                     "fonts/vf/public/brushscr/"
                                     "tex/latex/brushscr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kdikrlhbqfcnb42y5pz93l8wkzbrn7hx5gjf05vkchcxq24nj5x"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/brushscr")
    (synopsis "Handwriting script font")
    (description
     "The BrushScript font simulates hand-written characters; it is
distributed in Adobe Type 1 format (but is available in italic shape only).
The package includes the files needed by LaTeX in order to use that font.")
    (license license:public-domain)))

(define-public texlive-buctthesis
  (package
    (name "texlive-buctthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/xelatex/buctthesis/"
                                     "source/xelatex/buctthesis/"
                                     "tex/xelatex/buctthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0h8djmkc6jcn10g0w1qmw0mw8v42dsb7kijfv87qxw78i1cfsbax"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/buctthesis")
    (synopsis "Beijing University of Chemical Technology thesis template")
    (description
     "This package provides a LaTeX class and template for Beijing University
of Chemical Technology, supporting bachelor, master, and doctor theses.")
    (license license:lppl1.3c)))

(define-public texlive-burmese
  (package
    (name "texlive-burmese")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/burmese/"
                                     "fonts/map/dvips/burmese/"
                                     "fonts/tfm/public/burmese/"
                                     "fonts/type1/public/burmese/"
                                     "source/fonts/burmese/"
                                     "tex/latex/burmese/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "04d022k7bqc7092xhsda0h5ma18b24hkmn0b7mlblpd3zf4qhs79"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-inputs "perl")))
    (build-system texlive-build-system)
    (arguments
     (list
      #:phases
      #~(modify-phases %standard-phases
          ;; The "birm.pl" script is located in the "source" part, and
          ;; therefore will not appear in any output.  This phase forces its
          ;; installation in "bin/" directory.
          (add-after 'install 'install-script
            (lambda _
              (install-file "source/fonts/burmese/birm.pl"
                            (string-append #$output "/bin")))))))
    (inputs (list perl))
    (home-page "https://ctan.org/pkg/burmese")
    (synopsis "Basic support for writing Burmese")
    (description
     "This package provides basic support for writing Burmese.  The package
provides a preprocessor (written in Perl), an Adobe Type 1 font, and LaTeX
macros.")
    ;; The package itself is under LPPL terms, but the preprocessor relies on
    ;; Knuth's.
    (license (list license:lppl license:knuth))))

(define-public texlive-bussproofs-extra
  (package
    (name "texlive-bussproofs-extra")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bussproofs-extra/"
                                     "source/latex/bussproofs-extra/"
                                     "tex/latex/bussproofs-extra/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "08n6ww2fcijkisldrhbvvm52n3r4fy10bz1i4jyac7wnnpw8dd2p"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bussproofs-extra")
    (synopsis "Extra commands for @file{bussproofs.sty}")
    (description
     "This package provides additional functionality for
@file{bussproofs.sty}; specifically, it allows for typesetting of
entire (sub)deductions.")
    (license license:lppl1.3c)))

(define-public texlive-bxbase
  (package
    (name "texlive-bxbase")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxbase/" "tex/latex/bxbase/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "19zhvwj55lihlj11mk322735z595al847wsg3p0s3a9zm17b0skk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxbase")
    (synopsis "BX bundle base components")
    (description
     "The main purpose of this bundle is to serve as an underlying library for
other packages created by the same author.  However @code{bxbase} package
contains a few user-level commands and is of some use by itself.")
    (license license:expat)))

(define-public texlive-bxcjkjatype
  (package
    (name "texlive-bxcjkjatype")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxcjkjatype/"
                                     "tex/latex/bxcjkjatype/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jhjhrfmabfdwzk2gv2zsaadj3907ajxic4dqzwaz43cip05iy3z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxcjkjatype")
    (synopsis "Typeset Japanese with pdfLaTeX and CJK")
    (description
     "The package provides a working configuration of the CJK package,
suitable for Japanese typesetting of moderate quality.  Moreover, it
facilitates use of the CJK package for pLaTeX users, by providing commands
that are similar to those used by the pLaTeX kernel and some other packages
used with it.")
    (license license:expat)))

(define-public texlive-bxcoloremoji
  (package
    (name "texlive-bxcoloremoji")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/bxcoloremoji/"
                              "tex/latex/bxcoloremoji/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0h4pa9h683rds27ni762hg0arrddzpcb5pmr7r8pvr129cy98rvj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxcoloremoji")
    (synopsis "Use color emojis more conveniently")
    (description
     "This package lets users output color emojis in LaTeX documents.
Emojis can be entered as the characters themselves, as their Unicode code
values, or as their short names.

There are already many emoji packages in TeX Live.  To avoid uploading a large
amount of emoji image data that are essentially identical, the package
delegates the image output to the @code{twmojis} package and therefore
contains no image data.")
    (license license:expat)))

(define-public texlive-bxghost
  (package
    (name "texlive-bxghost")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxghost/" "tex/latex/bxghost/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "11hnnyd6h3r5fl9p23qj1zspwzgnamfah7xqj7l1prp6ygb322mn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxghost")
    (synopsis "Ghost insertion for proper @code{xkanjiskip}")
    (description
     "The package provides two commands to help authors for documents in
Japanese to insert proper @code{xkanjiskips}.  It supports LuaTeX, XeTeX,
pTeX, upTeX, and ApTeX (pTeX-ng).")
    (license license:expat)))

(define-public texlive-bxjaholiday
  (package
    (name "texlive-bxjaholiday")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxjaholiday/"
                                     "tex/latex/bxjaholiday/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1n0h9g5lgaz00gn8wjai8w3pbg7bs02aaw7zi7h8l01i4lc8l1r1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxjaholiday")
    (synopsis "Support for Japanese holidays")
    (description
     "This LaTeX package provides a command to convert dates to names of
Japanese holidays.  Another command, converting dates to the day of the week
in Japanese, is available as a free gift.")
    (license license:expat)))

(define-public texlive-bxjalipsum
  (package
    (name "texlive-bxjalipsum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxjalipsum/"
                                     "tex/latex/bxjalipsum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00lmr20lb7yd98iblg0i3pqvxh80arivdgyvzkj5mj2ivil7vw9k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxjalipsum")
    (synopsis "Dummy text in Japanese")
    (description
     "This package enables users to print some Japanese text that can be used
as dummy text.  It is a Japanese counterpart of the @code{lipsum} package.
Since there is no well-known nonsense text like Lipsum in the Japanese
language, the package uses some real text in public domain.")
    (license license:expat)))

(define-public texlive-bxjaprnind
  (package
    (name "texlive-bxjaprnind")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxjaprnind/"
                                     "tex/latex/bxjaprnind/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0j2k06y63c7dgpcflrfrjy82g2746fnmpf8dk445lbqkbn1jypr5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxjaprnind")
    (synopsis "Adjust the position of parentheses at paragraph head")
    (description
     "In Japanese typesetting, opening parentheses placed at the beginning of
paragraphs or lines are treated specially.  This package adjusts the position
of parentheses accordingly.")
    (license license:expat)))

(define-public texlive-bxjatoucs
  (package
    (name "texlive-bxjatoucs")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxjatoucs/"
                                     "fonts/tfm/public/bxjatoucs/"
                                     "tex/latex/bxjatoucs/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03f6vmfwlyawn79vlz09rzziky4435n7b0ax97c5s1kmkc8bv0ly"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxjatoucs")
    (synopsis "Convert Japanese character code to Unicode")
    (description
     "This package is meant for macro or package developers: it provides
function-like macros that convert a character code value in one of several
Japanese encodings to a Unicode value.  Supported source encodings are:
ISO-2022-JP (jis), EUC-JP (euc), Shift_JIS (sjis), and the Adobe-Japan1 glyph
set.")
    (license license:expat)))

(define-public texlive-bxjscls
  (package
    (name "texlive-bxjscls")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxjscls/" "source/latex/bxjscls/"
                                     "tex/latex/bxjscls/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "006sq8p88sxsv54g81kv4yv2vy1nbwj2lk2by53wykrjhjjc8lrg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxjscls")
    (synopsis "Japanese document class collection for all major engines")
    (description
     "This package provides an extended version of the Japanese document class
collection provided by @code{jsclasses}.  While the original version supports
only pLaTeX and upLaTeX, the extended version also supports pdfLaTeX, XeLaTeX
and LuaLaTeX, with the aid of suitable packages that provide capability of
Japanese typesetting.")
    (license license:bsd-2)))

(define-public texlive-bxorigcapt
  (package
    (name "texlive-bxorigcapt")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxorigcapt/"
                                     "tex/latex/bxorigcapt/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gj61hjmyxsbfkrs1sh604b6dfl00kp39fcamlfzb8s197wsl9fv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxorigcapt")
    (synopsis "Retain the original caption names when using Babel")
    (description
     "This package forces the caption names declared by the document class in
use to be used as the caption names for a specific language introduced by the
Babel package.")
    (license license:expat)))

(define-public texlive-bxwareki
  (package
    (name "texlive-bxwareki")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bxwareki/" "tex/latex/bxwareki/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "105fx28gy2f73127cpy3fdsmggw5a49bbr1b99kd5ydmwckwkp5g"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bxwareki")
    (synopsis "Convert dates from Gregorian to Japanese calendar")
    (description
     "This LaTeX package provides commands to convert from the Gregorian
calendar to the Japanese rendering of the Japanese calendar.  You can choose
whether the numbers are written in Western numerals or kanji numerals.  Note
that the package only deals with dates in the year 1873 or later, where the
Japanese calendar is really a Gregorian calendar with a different notation of
years.")
    (license license:expat)))

(define-public texlive-bytefield
  (package
    (name "texlive-bytefield")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/bytefield/"
                                     "source/latex/bytefield/"
                                     "tex/latex/bytefield/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0b2qik4pkqyaqk00aynl5k6vzzgwdr5ir4x72364v59wn97cx1bz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/bytefield")
    (synopsis "Create illustrations for network protocol specifications")
    (description
     "The @code{bytefield} package helps the user create illustrations for
network protocol specifications and anything else that utilizes fields of
data.  These illustrations show how the bits and bytes are laid out in
a packet or in memory.")
    (license license:lppl1.3a)))

(define-public texlive-c-pascal
  (package
    (name "texlive-c-pascal")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/c-pascal/"
                                     "tex/generic/c-pascal/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kmxki1yclk7lzlidcvsdry7mzm58zzwq3rxgvmpawq2hs8hpcra"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/c-pascal")
    (synopsis "Typeset Python, C and Pascal programs")
    (description
     "This package provides a TeX macro package for easy typesetting programs
in Python, C and Pascal.  Program source files may also be input.")
    (license license:public-domain)))

(define-public texlive-c90
  (package
    (name "texlive-c90")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/enc/c90/" "fonts/enc/dvips/c90/"
                                     "source/fonts/enc/c90/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0g4rwimlqqzbbs8ar15nsf8qcr8bbyjacmklbgv4pmsls5ka9n3n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/c90")
    (synopsis "C90 font encoding for Thai")
    (description "This package provides C90 font encoding for Thai.")
    (license license:gpl2+)))

(define-public texlive-cahierprof
  (package
    (name "texlive-cahierprof")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/cahierprof/"
                              "tex/latex/cahierprof/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "07zbwnbk2pdzb6rkkivlrmfzyx312p1h6qb9sgjxggnq03zifcmh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cahierprof")
    (synopsis "Schedule and grade books for French teachers")
    (description
     "This package provide tools to help teachers in France to produce weekly
schedules and grade books.")
    (license license:lppl1.3c)))

(define-public texlive-caladea
  (package
    (name "texlive-caladea")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/caladea/"
                                     "fonts/enc/dvips/caladea/"
                                     "fonts/map/dvips/caladea/"
                                     "fonts/tfm/huerta/caladea/"
                                     "fonts/truetype/huerta/caladea/"
                                     "fonts/type1/huerta/caladea/"
                                     "fonts/vf/huerta/caladea/"
                                     "tex/latex/caladea/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "02pw132njvk23iwfwlfq5cbamqmgxap2h5pq23x3r8dbym1zpk9f"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/caladea")
    (synopsis "Support for the Caladea family of fonts")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX and LuaLaTeX support for
the Caladea family of fonts.")
    (license (list license:asl2.0 license:lppl))))

(define-public texlive-calculation
  (package
    (name "texlive-calculation")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/calculation/"
                                     "source/latex/calculation/"
                                     "tex/latex/calculation/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15sk3v1cwpjln4rp9k877awxaz9js5gbazlzjknmv5fv1cb4d09c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/calculation")
    (synopsis
     "Typesetting reasoned calculations, also called calculational proofs")
    (description
     "The @code{calculation} environment formats reasoned calculations, also
called calculational proofs.  The package allows steps and expressions to be
numbered (by LaTeX equation numbers, obeying the LaTeX @code{\\label} command
to refer to these numbers), and a step doesn't take vertical space if its hint
is empty.  An expression in a calculation can be given a comment; it is placed
at the side opposite to the equation numbers.  Calculations are allowed inside
hints although numbering and commenting is then disabled.")
    (license license:lppl1.3+)))

(define-public texlive-calligra
  (package
    (name "texlive-calligra")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/calligra/"
                                     "fonts/source/public/calligra/"
                                     "fonts/tfm/public/calligra/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1h9w2qifsd0sjr2czy6zbyvgvdl72nd1sn2hmxv0jsmbcjqak1ib"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/calligra")
    (synopsis "Calligraphic font")
    (description
     "This package provides a calligraphic font in the handwriting style.
The font is supplied as Metafont source.  LaTeX support of the font is
provided in the @code{calligra} package in the @code{fundus} bundle.")
    (license license:expat)))

(define-public texlive-calligra-type1
  (package
    (name "texlive-calligra-type1")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/calligra-type1/"
                                     "fonts/afm/public/calligra-type1/"
                                     "fonts/map/dvips/calligra-type1/"
                                     "fonts/type1/public/calligra-type1/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1v6zwghf8q70lls3hmwj8gc0n6d4ffjf1xjvq1k6w71shiga8bqi"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/calligra-type1")
    (synopsis "Type 1 version of Calligra")
    (description
     "This is a Type 1 conversion of Peter Vanroose's Calligra handwriting
font.")
    (license license:expat)))

(define-public texlive-cantarell
  (package
    (name "texlive-cantarell")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cantarell/"
                                     "fonts/enc/dvips/cantarell/"
                                     "fonts/map/dvips/cantarell/"
                                     "fonts/opentype/gnome/cantarell/"
                                     "fonts/tfm/gnome/cantarell/"
                                     "fonts/type1/gnome/cantarell/"
                                     "fonts/vf/gnome/cantarell/"
                                     "tex/latex/cantarell/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "11cj8z6wyzq92x66alas88qgrs4ap0xgvf0yr7h62vg5ym3618c8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cantarell")
    (synopsis "LaTeX support for the Cantarell font family")
    (description
     "Cantarell is a contemporary Humanist sans serif designed by Dave
Crossland and Jakub Steiner.  The present package provides support for this
font in LaTeX. It includes Type 1 versions of the fonts, converted for this
package using FontForge from its sources, for full support with Dvips.")
    (license (list license:silofl1.1 license:lppl1.3c))))

(define-public texlive-carlito
  (package
    (name "texlive-carlito")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/carlito/"
                                     "fonts/enc/dvips/carlito/"
                                     "fonts/map/dvips/carlito/"
                                     "fonts/tfm/google/carlito/"
                                     "fonts/truetype/google/carlito/"
                                     "fonts/type1/google/carlito/"
                                     "fonts/vf/google/carlito/"
                                     "tex/latex/carlito/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14hk51bgwpn4wr843g7w8wjwmadq0ca86gviayhwsz4zax68jhx6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/carlito")
    (synopsis "Support for Carlito sans-serif fonts")
    (description
     "The package provides LaTeX, pdfLaTeX, XeLaTeX and LuaLaTeX support for
the Carlito family of sans serif fonts, designed by Lukasz Dziedzic.")
    (license license:silofl1.1)))

(define-public texlive-carolmin-ps
  (package
    (name "texlive-carolmin-ps")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/carolmin-ps/"
                                     "fonts/afm/public/carolmin-ps/"
                                     "fonts/map/dvips/carolmin-ps/"
                                     "fonts/type1/public/carolmin-ps/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1867h17l0ms2schmbrsjm4gajfmckm2x7ny2ijd9vf232l7s1b53"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/carolmin-t1")
    (synopsis "Adobe Type 1 format of Carolingian Minuscule fonts")
    (description
     "The bundle offers Adobe Type 1 format versions of Peter Wilson's
Carolingian Minuscule font set (part of the @code{bookhands} collection).  The
fonts in the bundle are ready-to-use replacements for the Metafont
originals.")
    (license license:lppl)))

(define-public texlive-cartonaugh
  (package
    (name "texlive-cartonaugh")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cartonaugh/"
                                     "source/latex/cartonaugh/"
                                     "tex/latex/cartonaugh/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05f6xslfv3d1n48j2ivl7vdgs12lzgzx8ccfv6l49ddgq0dssnir"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cartonaugh")
    (synopsis
     "LuaLaTeX package for drawing karnaugh maps with up to 6 variables")
    (description
     "This package, a fork of @code{karnaugh-map} package, draws karnaugh maps
with 2, 3, 4, 5, and 6 variables.  It also contains commands for filling the
karnaugh map with terms semi-automatically or manually.  Last but not least it
contains commands for drawing implicants on top of the map.")
    (license license:cc-by-sa3.0)))

(define-public texlive-cascade
  (package
    (name "texlive-cascade")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cascade/" "source/latex/cascade/"
                                     "tex/latex/cascade/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13950pv70hr21jn7f0hg4hyai3b28bwcnrd23l44mhynv4hbs5yz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cascade")
    (synopsis
     "Constructions with braces to present mathematical demonstrations")
    (description
     "The LaTeX package cascade provides a command @code{\\Cascade} to do
constructions to present mathematical demonstrations with successive braces
for the deductions.")
    (license license:lppl1.3+)))

(define-public texlive-cascadia-code
  (package
    (name "texlive-cascadia-code")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cascadia-code/"
                                     "fonts/enc/dvips/cascadia-code/"
                                     "fonts/map/dvips/cascadia-code/"
                                     "fonts/opentype/public/cascadia-code/"
                                     "fonts/tfm/public/cascadia-code/"
                                     "fonts/type1/public/cascadia-code/"
                                     "fonts/vf/public/cascadia-code/"
                                     "tex/latex/cascadia-code/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "015kjdr9lgbznmzz1pw57jw4dy4v4fk1qb3w6nizf1z3xza9b49r"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cascadia-code")
    (synopsis "Cascadia Code font with support for LaTeX and pdfLaTeX")
    (description
     "Cascadia Code is a monospaced font by Microsoft.  This package provides
the Cascadia Code family of fonts with support for LaTeX and pdfLaTeX.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-cascadiamono-otf
  (package
    (name "texlive-cascadiamono-otf")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/fonts/cascadiamono-otf/"
                              "tex/fonts/cascadiamono-otf/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1v27fz7cx68hfcl2ypgs9ddgyac1mnxv10rv82wrqbcd39vxm14q"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cascadiamono-otf")
    (synopsis "Fontspec support for the OpenType font CascadiaMono")
    (description
     "This package provides support for the OpenType font CascadiaMono --- i.e.,
with LuaLaTeX or XeTeX and Fontspec --- which is a variant of CascadiaCode,
but without ligatures.")
    (license license:lppl1.3+)))

(define-public texlive-cascadilla
  (package
    (name "texlive-cascadilla")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/cascadilla/"
                                     "doc/latex/cascadilla/"
                                     "tex/latex/cascadilla/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0sl29gxjhkakaz7r48wfbvcxngnxdyixmh5awxnlsmrajxgdzh9l"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cascadilla")
    (synopsis "Conform to the stylesheet of the Cascadilla Proceedings
Project")
    (description
     "The class provides an extension of the standard LaTeX @code{article}
class that may be used to typeset papers conforming to the stylesheet of the
Cascadilla Proceedings Project, which is used by a number of linguistics
conference proceedings.")
    (license license:lppl)))

(define-public texlive-catcodes
  (package
    (name "texlive-catcodes")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/catcodes/"
                                     "source/generic/catcodes/"
                                     "tex/generic/catcodes/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0di6rd5c0viiwvc6ncwndah38q318l3s1pf47ddmzjdq6rys9s33"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/catcodes")
    (synopsis "Generic handling of TeX category codes")
    (description
     "The bundle deals with category code switching; the packages of the
bundle should work with any TeX format (with the support of the
@code{plainpkg} package).  The bundle provides:
@itemize
@item @file{stacklet.sty}, which supports stacks that control the use of
different catcodes;
@item @file{actcodes.sty}, which deals with active characters;
@item @file{catchdq.sty}, which provides a simple quotation character control
mechanism.
@end itemize")
    (license license:lppl1.3+)))

(define-public texlive-causets
  (package
    (name "texlive-causets")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/causets/" "tex/latex/causets/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15wfsg5vpkx25hx5d9scvyxbq1qwanjzinkxbwrib36pdblv237k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/causets")
    (synopsis "Draw causal set (Hasse) diagrams")
    (description
     "This LaTeX package uses TikZ to generate (Hasse) diagrams for causal sets
(causets) to be used inline with text or in mathematical expressions.  The
macros can also be used in the @code{tikzpicture} environment to annotate or
modify a diagram.")
    (license license:lppl1.3+)))

(define-public texlive-ccfonts
  (package
    (name "texlive-ccfonts")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ccfonts/" "source/latex/ccfonts/"
                                     "tex/latex/ccfonts/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1r319fv1n9fsnnycza4z1j23dp03fww98a03hmcapkyr9pp9lvv4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (home-page "https://ctan.org/pkg/ccfonts")
    (synopsis "Support for Concrete text and math fonts in LaTeX")
    (description
     "This package provides LaTeX font definition files for the Concrete fonts
and a LaTeX package for typesetting documents using Concrete as the default
font family.  The files support OT1, T1, TS1, and Concrete mathematics
including AMS fonts (Ulrik Vieth's @code{concmath}).")
    (license license:lppl)))

(define-public texlive-ccool
  (package
    (name "texlive-ccool")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ccool/" "source/latex/ccool/"
                                     "tex/latex/ccool/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17xkwd8mrvl751rn6zvxaznac27rpvmi53rq8nwwlp6j1bdnaj94"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ccool")
    (synopsis "Key-value document command parser")
    (description
     "This package provides a key-value interface, @code{\\Ccool}, on top of
@code{xparse}'s document command parser.  Global options control input
processing and its expansion.  By default, they are set to meet likely
requirements, depending on context: the selected language, and which of text
and math mode is active.  These options can be overridden inline.  Polymorphic
commands can be generated by parameterizing the keys (for instance, one
parameter value for style, another for a property).  User input to
@code{\\Ccool} can optionally be serialized.  This can useful for typesetting
documents sharing the same notation.")
    (license license:lppl1.3c)))

(define-public texlive-cell
  (package
    (name "texlive-cell")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/cell/" "doc/latex/cell/"
                                     "tex/latex/cell/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0x2dw80z0kzisbwnx2xrmsk0biqbanmydzygg9zdymhcx7w5rz52"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cell")
    (synopsis "Bibliography style for Cell")
    (description
     "This is an APA-like style (cf.@: @file{apalike.bst} in the BibTeX
distribution), developed from the same author's JMB style.  A supporting LaTeX
package is also provided.")
    (license license:public-domain)))

(define-public texlive-cesenaexam
  (package
    (name "texlive-cesenaexam")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cesenaexam/"
                                     "source/latex/cesenaexam/"
                                     "tex/latex/cesenaexam/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05n61y7lqycnj22a80p5pa0ajr8c4zc5b73cc3c2x40w8yx85xcf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cesenaexam")
    (synopsis "Class file to typeset exams")
    (description
     "This LaTeX document class has been designed to typeset exams.")
    (license license:lppl1.3c)))

(define-public texlive-cfr-initials
  (package
    (name "texlive-cfr-initials")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cfr-initials/"
                                     "tex/latex/cfr-initials/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03isx6wck9q2nmn2gm6zmd1830d2v9bps5g1k9qw6y02wdac8az9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cfr-initials")
    (synopsis "LaTeX packages for use of initials")
    (description
     "This is a set of 23 tiny packages designed to make it easier to use
fonts from the initials package in LaTeX, e.g., with the @code{lettrine}
package.")
    (license license:lppl1.3+)))

(define-public texlive-cfr-lm
  (package
    (name "texlive-cfr-lm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cfr-lm/"
                                     "fonts/enc/dvips/cfr-lm/"
                                     "fonts/map/dvips/cfr-lm/"
                                     "fonts/tfm/public/cfr-lm/"
                                     "fonts/vf/public/cfr-lm/"
                                     "source/fonts/cfr-lm/"
                                     "tex/latex/cfr-lm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "169wav78cr14d2zrrsfndszbixg4vyvfw3i8rcg9sa7d7821zmmp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:tex-format "latex"
           #:phases
           #~(modify-phases %standard-phases
               (add-after 'unpack 'remove-pre-generated-files
                 (lambda _
                   (for-each delete-file
                             (find-files "source/fonts/cfr-lm/" "\\.etx$")))))))
    (home-page "https://ctan.org/pkg/cfr-lm")
    (synopsis "Enhanced support for the Latin Modern fonts")
    (description
     "The package supports a number of features of the Latin Modern fonts
which are not easily accessible via the default (La)TeX support.  In
particular, the package supports the use of the various styles of digits
available, small-caps and upright italic shapes, and alternative weights and
widths.  It also supports variable width typewriter and the @code{quotation}
font.

By default, the package uses proportional oldstyle digits and variable width
typewriter but this can be changed by passing appropriate options to the
package.  The package also supports using (for example) different styles of
digits within a document so it is possible to use proportional oldstyle digits
by default, say, but tabular lining digits within a particular table.")
    (license license:lppl1.3+)))

(define-public texlive-chbibref
  (package
    (name "texlive-chbibref")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chbibref/" "tex/latex/chbibref/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1k4cwy4z0kacikpvxkk3y68apw2x33sh06yccb1s894r1bn3c1qi"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chbibref")
    (synopsis "Change the Bibliography and References title")
    (description
     "This package defines a single command, @code{\\setbibref}, which sets
whichever of @code{\\bibname} and @code{\\refname} is in use.")
    (license license:lppl)))

(define-public texlive-chem-journal
  (package
    (name "texlive-chem-journal")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/chem-journal/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "01s4sbmc4g96cmnbhx9a6bqjskkrr6saq91cly2pfy5gr256vmnq"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chem-journal")
    (synopsis "Various BibTeX formats for journals in chemistry")
    (description
     "This package provides various BibTeX formats for journals in chemistry,
including @emph{Reviews in Computational Chemistry}, @emph{Journal of Physical
Chemistry}, @emph{Journal of Computational Chemistry}, and @emph{Physical
Chemistry Chemical Physics}.")
    (license license:gpl3+)))

(define-public texlive-chemarrow
  (package
    (name "texlive-chemarrow")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/chemarrow/"
                                     "fonts/afm/public/chemarrow/"
                                     "fonts/map/dvips/chemarrow/"
                                     "fonts/source/public/chemarrow/"
                                     "fonts/tfm/public/chemarrow/"
                                     "fonts/type1/public/chemarrow/"
                                     "source/fonts/chemarrow/"
                                     "tex/latex/chemarrow/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ky0v2psxamzilmijih88zxanmq2hnpv1yc18mcy9mpymcmvrhwr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/chemarrow")
    (synopsis "Arrows for use in chemistry")
    (description
     "This bundle consists of a font (available as Metafont source, MetaPost
source, and generated type 1 versions), and a package to use it.  The arrows
in the font are designed to look more like those in chemistry text-books than
do Knuth's originals.")
    (license license:public-domain)))

(define-public texlive-chembst
  (package
    (name "texlive-chembst")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/chembst/" "doc/latex/chembst/"
                                     "source/latex/chembst/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "026j8lv3kbxmbq6sc1n8rdrr47njgn5mn9im1mmppzk5bi7w6yzz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chembst")
    (synopsis "Collection of BibTeX files for chemistry journals")
    (description
     "The package offers a collection of advanced BibTeX style files suitable
for publications in chemistry journals.  Currently, style files for journals
published by the American Chemical Society, Wiley-VCH and The Royal Society of
Chemistry are available.  The style files support advanced features such as
automatic formatting of errata or creating an appropriate entry for
publications in @emph{Angewandte Chemie} where both English and German should
be cited simultaneously.")
    (license license:lppl)))

(define-public texlive-chemcompounds
  (package
    (name "texlive-chemcompounds")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemcompounds/"
                                     "source/latex/chemcompounds/"
                                     "tex/latex/chemcompounds/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0bk625vdd6i1dc2569ypcq27hin1yggvadf18463qb6xhyndplvf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemcompounds")
    (synopsis "Simple consecutive numbering of chemical compounds")
    (description
     "The chemcompounds package allows for a simple consecutive numbering of
chemical compounds.  Optionally, it is possible to supply a custom name for
each compound.  The package differs from the @code{chemcono} package by not
generating an odd-looking list of compounds inside the text.")
    (license license:lppl)))

(define-public texlive-chemcono
  (package
    (name "texlive-chemcono")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemcono/" "tex/latex/chemcono/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1lwsizrypzn6vnhf7szg47b1synafihihq14wfc0ffrksq1c730c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemcono")
    (synopsis "Support for compound numbers in chemistry documents")
    (description
     "This package provides a LaTeX package for using compound numbers in
chemistry documents.  It works like @code{\\cite} and the
@code{\\thebibliography}, using @code{\\fcite} and @code{\\theffbibliography}
instead.  It allows compound names in documents to be numbered and does not
affect the normal citation routines.")
    (license license:lppl)))

(define-public texlive-chemexec
  (package
    (name "texlive-chemexec")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemexec/" "tex/latex/chemexec/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1al19xf107qizd7nr7297szqd2k8sf1i3ngpnw6kl6fkak5h7qdg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemexec")
    (synopsis "Creating (chemical) exercise sheets")
    (description
     "The package provides environments and commands that the author needed
when preparing exercise sheets and other teaching material.  In particular,
the package supports the creation of exercise sheets, with separating printing
of solutions.")
    (license license:lppl1.3+)))

(define-public texlive-chemformula
  (package
    (name "texlive-chemformula")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemformula/"
                                     "tex/latex/chemformula/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14w98hnhljfm62cxg2hml3razrxw9hkzm54v7ybnbhcp5mm7wvdm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-units))
    (home-page "https://ctan.org/pkg/chemformula")
    (synopsis "Command for typesetting chemical formulas and reactions")
    (description
     "The package provides a command to typeset chemical formulas and
reactions in support of other chemistry packages (such as
@code{chemmacros}).")
    (license license:lppl1.3c)))

(define-public texlive-chemformula-ru
  (package
    (name "texlive-chemformula-ru")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/chemformula-ru/"
                              "tex/latex/chemformula-ru/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "08frbj3swbf57l3kcljaj82kk1r5xsb0zwx995i6f07d7gkzsr97"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemformula-ru")
    (synopsis "Fix @code{chemformula} package and @code{babel-russian} conflict")
    (description
     "The @code{chemformula} package and @code{babel-russian} settings both
define macros named @code{\\ch}.  This package un-defines Babel's macro to
prevent an error when both packages are loaded together.  Optionally, it
redefines the @code{\\cosh} macro to print the hyperbolic cosine in Russian
notation or defines a new macro @code{\\Ch} for that purpose.")
    (license license:lppl1.3c)))

(define-public texlive-chemgreek
  (package
    (name "texlive-chemgreek")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemgreek/"
                                     "tex/latex/chemgreek/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0rpcywz5w2im5ck0kp56dywiy1his01s817pngdf9xdn02qk517n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemgreek")
    (synopsis "Upright Greek letters in chemistry")
    (description
     "The package provides upright Greek letters in support of other chemistry
packages (such as @code{chemmacros}).")
    (license license:lppl1.3+)))

(define-public texlive-chemmacros
  (package
    (name "texlive-chemmacros")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemmacros/"
                                     "tex/latex/chemmacros/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1jj7i00h83mhajwayn48x4qf0xz526jrjlj5zclqkgrb6kxmx978"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemmacros")
    (synopsis "Collection of macros to support typesetting chemistry documents")
    (description
     "The bundle offers a collection of macros and commands which are intended
to make typesetting chemistry documents faster and more convenient.  Coverage
includes some nomenclature commands, oxidation numbers, thermodynamic data,
newman projections, etc.")
    (license license:lppl1.3c)))

(define-public texlive-chemnum
  (package
    (name "texlive-chemnum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemnum/" "tex/latex/chemnum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ld2ja2by5y48v5k8x1mgmcpr75vkf8r8ni0fmvrpxh8xrv7mq4f"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemnum")
    (synopsis "Method for numbering chemical compounds")
    (description
     "The package defines a @code{\\label}- and @code{\\ref}-like commands for
compound numbers.")
    (license license:lppl1.3c)))

(define-public texlive-chemobabel
  (package
    (name "texlive-chemobabel")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemobabel/"
                                     "source/latex/chemobabel/"
                                     "tex/latex/chemobabel/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ky7xcyln4j40b18anld77mxc2i4af75y9xdjisj2ixl3vkg5a1x"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemobabel")
    (synopsis "Convert chemical structures from ChemDraw, MDL molfile or SMILES")
    (description
     "This package provides a way to convert and include chemical structure
graphics from various chemical formats, such as ChemDraw files, MDL molfile or
SMILES notations using Open Babel.  To use this LaTeX package, it is necessary
to enable execution of the following external commands via
@samp{latex -shell-escape}: @command{obabel} (Open Babel) @command{inkscape}
or @command{rsvg-convert} (for SVG -> PDF/EPS conversion), @command{pdfcrop}
or @command{ps2eps} (optional; for cropping large margins of PDF/EPS).")
    (license license:bsd-2)))

(define-public texlive-chemplants
  (package
    (name "texlive-chemplants")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemplants/"
                                     "tex/latex/chemplants/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "164g4wy0x7jqf5l7hzb05ir7yz130bw34b7pwr8bp4vcyp4wy3ai"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemplants")
    (synopsis "Symbology to draw chemical plants with TikZ")
    (description
     "This package offers tools to draw simple or barely complex schemes of
chemical processes.  The package defines several standard symbols and styles
to draw process units and streams.  The guiding light of the package is the
UNICHIM regulation.")
    (license license:lppl1.3c)))

(define-public texlive-chemschemex
  (package
    (name "texlive-chemschemex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemschemex/"
                                     "source/latex/chemschemex/"
                                     "tex/latex/chemschemex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09mjxj1gwddbpkriivn8x98z9hls4pcmpdqf24a7gz19whgs1qyk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemschemex")
    (synopsis "Typeset and cross-reference chemical schemes based on TikZ code")
    (description
     "The package provides a comfortable means of typesetting chemical
schemes, and also offers automatic structure referencing.")
    (license license:lppl1.2+)))

(define-public texlive-chemsec
  (package
    (name "texlive-chemsec")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemsec/" "source/latex/chemsec/"
                                     "tex/latex/chemsec/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1q41mfhnvh4ni7dn0krrrcr9sr6ma855j1hckbg8fl56dl2h7zlp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemsec")
    (synopsis "Automated creation of numeric entity labels")
    (description
     "Packages provides creation of sequential numeric labels for entities in
a document.  The motivating example is chemical structures in a scientific
document.  The package can automatically output a full object name and label
on the first occurrence in the document and just labels only on subsequent
references.")
    (license license:lppl1.3+)))

(define-public texlive-chemstyle
  (package
    (name "texlive-chemstyle")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chemstyle/"
                                     "source/latex/chemstyle/"
                                     "tex/latex/chemstyle/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1s2k4h153yh0f50q9da390f3ihykvrhsd1zjgfsdb6404059in3p"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chemstyle")
    (synopsis "Writing chemistry with style")
    (description
     "The package @code{chemstyle} provides an extensible system for
formatting chemistry documents according to the conventions of a number of
leading journals.  It also provides some handy chemistry-related macros.")
    (license license:lppl1.3+)))

(define-public texlive-cherokee
  (package
    (name "texlive-cherokee")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cherokee/"
                                     "fonts/source/public/cherokee/"
                                     "fonts/tfm/public/cherokee/"
                                     "tex/latex/cherokee/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0cganm5d4gnkkhandancyd915w2x9kskj9qg81y54jcfyjz8gp51"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/cherokee")
    (synopsis "Font for the Cherokee script")
    (description
     "The Cherokee script was designed in 1821 by Segwoya.  The alphabet is
essentially syllabic, only 6 characters (a e i o s u) correspond to Roman
letters: the font encodes these to the corresponding roman letter.  The
remaining 79 characters have been arbitrarily encoded in the range 38-122; the
@code{cherokee} package provides commands that map each such syllable to the
appropriate character.")
    ;; "You may freely use, modify, and/or distribute this file, without
    ;; limitation.  If you make significant additions or improvements,
    ;; I request that you email me an updated version."
    (license
     (license:fsf-free "file://fonts/source/public/cherokee/cherokee.mf"))))

(define-public texlive-chess
  (package
    (name "texlive-chess")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/chess/"
                                     "fonts/source/public/chess/"
                                     "fonts/tfm/public/chess/" "tex/latex/chess/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "079naqw9bd5da3c2bsa0322kdjrfpwa35inrfznzgrfdk1w0irs7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/chess")
    (synopsis "Fonts for typesetting chess boards")
    (description
     "This is the original, and somewhat dated, TeX chess font package.
Potential users should consider @code{skak} (for alternative fonts, and
notation support), @code{texmate} (for alternative notation support), or
@code{chessfss} (for flexible font choices).")
    (license license:public-domain)))

(define-public texlive-chess-problem-diagrams
  (package
    (name "texlive-chess-problem-diagrams")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chess-problem-diagrams/"
                                     "source/latex/chess-problem-diagrams/"
                                     "tex/latex/chess-problem-diagrams/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1yikr896j3n7qslhlpb4qyi960icw163qgpj2ql8yij10la5rfg3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chess-problem-diagrams")
    (synopsis "Package for typesetting chess problem diagrams")
    (description
     "This package provides macros to typeset chess problem diagrams including
fairy chess problems (mostly using rotated images of pieces) and other
boards.")
    (license license:lppl1.2+)))

(define-public texlive-chessboard
  (package
    (name "texlive-chessboard")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chessboard/"
                                     "source/latex/chessboard/"
                                     "tex/latex/chessboard/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jgxi51lshb82xx0ck34zxansgi2c852kb3y7fdahi9r08kvjv46"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chessboard")
    (synopsis "Print chess boards")
    (description
     "This package offers commands to print chessboards.  It can print partial
boards, hide pieces and fields, color the boards and put various marks on the
board.  It has a lot of options to place pieces on the board.  Using exotic
pieces (e.g., for fairy chess) is possible.")
    (license license:lppl)))

(define-public texlive-chessfss
  (package
    (name "texlive-chessfss")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chessfss/"
                                     "fonts/enc/dvips/chessfss/"
                                     "source/latex/chessfss/"
                                     "tex/latex/chessfss/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1l51famz3zx9v4v0mdxwk51xhaidwgfplf268q2f1ipif9h6ma5d"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chessfss")
    (synopsis "Package to handle chess fonts")
    (description
     "This package offers commands to use and switch between chess fonts.
It uses the LaTeX font selection scheme (nfss).  The package doesn't parse,
format and print PGN input like e.g., the packages @code{skak} or
@code{texmate}; the aim of the package is to offer writers of chess packages
a bundle of commands for fonts, so that they don't have to implement all these
commands for themselves.  A normal user can use the package to print
e.g,. single chess symbols and simple diagrams.")
    (license license:lppl)))

(define-public texlive-chhaya
  (package
    (name "texlive-chhaya")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chhaya/" "source/latex/chhaya/"
                                     "tex/latex/chhaya/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1if47icrsy56sjza170cxcpcczwfh9k8jl7j22gwwam4msa4d1if"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chhaya")
    (synopsis "Linguistic glossing in Marathi language")
    (description
     "This package provides macros for linguistic glossing as per the rules
given by Mumbai University.")
    (license (list license:gpl3+
                   ;; Aalok (आलोक) copyleft license v1.0+.
                   (license:fsf-free "file://doc/latex/aalok/README.txt")
                   license:fdl1.3+))))

(define-public texlive-chicago
  (package
    (name "texlive-chicago")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/chicago/" "tex/latex/chicago/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0mwrphf2g7v5yc0qij76dkzalgm3fhcm8zs7akgaypyk98cvxw8k"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chicago")
    (synopsis "Chicago bibliography style")
    (description
     "Chicago is a BibTeX style that follows the @samp{B} reference style of
the 13th Edition of the Chicago manual of style; a LaTeX package is also
provided.  The style was derived from the @code{newapa} style.")
    (license license:knuth)))

(define-public texlive-chicago-annote
  (package
    (name "texlive-chicago-annote")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/chicago-annote/"
                                     "doc/bibtex/chicago-annote/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "16hqz4xf7mbnrnx8420ynykxr1l3slfvjq8d8s0mrw42v88965ni"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chicago-annote")
    (synopsis "Chicago-based annotated BibTeX style")
    (description
     "This is a revision of chicagoa.bst, using the commonly-used annote field
in place of the original's annotation.")
    (license license:lppl)))

(define-public texlive-chicagoa
  (package
    (name "texlive-chicagoa")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/chicagoa/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1za4fqlpirbr3qs894abnlwmrhsqbahniyyimy1dxal0cp61cfil"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chicagoa")
    (synopsis "Chicago bibliography style with annotations")
    (description
     "This is a modification of the author's @code{chicago} style, to support
an annotation field in bibliographies.")
    (license license:knuth)))

(define-public texlive-chicagolinks
  (package
    (name "texlive-chicagolinks")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "bibtex/bst/chicagolinks/"
                              "doc/bibtex/chicagolinks/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0zflsbj46q3vlx03np7xyafn8xfnjamm5lmqayigi6ag2abj1w16"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chicagolinks")
    (synopsis "Chicago bibliography style that allows annotations")
    (description
     "This bibliography style is intended to extend the Chicago bibliography
style so that it can be annotated and at the same allowing DOI and URL
fields.")
    (license license:expat)))

(define-public texlive-chifoot
  (package
    (name "texlive-chifoot")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chifoot/" "tex/latex/chifoot/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07hxfiasfbnmvq367nnjg56n21wxdw0r6i7bl5l4gxa0j9jvmpf0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chifoot")
    (synopsis "Chicago-style footnote formatting")
    (description
     "This package provides a very short snippet that sets the footnotes to be
conformant with the Chicago style, so the footnotes at the bottom of the page
are now marked with a full-sized number, rather than with a superscript
number.")
    (license license:lppl1.3+)))

(define-public texlive-chinesechess
  (package
    (name "texlive-chinesechess")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chinesechess/"
                                     "tex/latex/chinesechess/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15sszrrv1viaa74i57a81xhybhjq2vaxb188wl728hjzm8d0n0wm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chinesechess")
    (synopsis "Typeset Chinese chess with @code{l3draw}")
    (description
     "This LaTeX3 package based on @code{l3draw} provides macros and an
environment for Chinese chess manual writing.")
    (license license:lppl1.3c)))

(define-public texlive-chivo
  (package
    (name "texlive-chivo")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/chivo/"
                                     "fonts/enc/dvips/chivo/"
                                     "fonts/map/dvips/chivo/"
                                     "fonts/opentype/public/chivo/"
                                     "fonts/tfm/public/chivo/"
                                     "fonts/type1/public/chivo/"
                                     "fonts/vf/public/chivo/"
                                     "source/fonts/chivo/"
                                     "tex/latex/chivo/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0dvcz42rw2y33amrjikgicy9r4c6m0c4cvv2h0sm7k5bympnj8w0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chivo")
    (synopsis "Using the Chivo fonts with LaTeX")
    (description
     "This work provides the necessary files to use the Chivo fonts with
LaTeX.  Chivo is a set of eight fonts provided by Hector Gatti and Omnibus
Team.")
    (license (list license:silofl1.1 license:lppl1.3c))))

(define-public texlive-chordbars
  (package
    (name "texlive-chordbars")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chordbars/"
                                     "tex/latex/chordbars/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qnggp8m2r7z5qmgpnqza0a2d3gcszdcypg6i78vr6p0zcmbadvh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chordbars")
    (synopsis "Print chord grids for pop/jazz tunes")
    (description
     "This Tikz-based music-related package is targeted at pop/jazz
guitar/bass/piano musicians.  They usually need only the chords and the song
structure.  This package produces rectangular song patterns with one square
per bar, with the chord shown inside the square.  It also handles the song
structure by showing the bar count and the repetitions of the patterns.")
    (license license:lppl1.3+)))

(define-public texlive-chordbox
  (package
    (name "texlive-chordbox")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chordbox/" "tex/latex/chordbox/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "01css8dnzss7s711ry2rs2fi3bw70s0lgj0cwwdicjgi331zbq70"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chordbox")
    (synopsis "Draw chord diagrams")
    (description
     "This package provides two macros for drawing chord diagrams, as may be
found for example in chord charts/books and educational materials.  They are
composed as TikZ pictures and have several options to modify their
appearance.")
    (license license:lppl1.3+)))

(define-public texlive-chronos
  (package
    (name "texlive-chronos")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/chronos/"
                              "tex/latex/chronos/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0fyz2m8iqsna70lds6y2b278b1zn4wxqj3pmpnbnr3q68fw3hpf7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chronos")
    (synopsis "TikZ-based package for drawing configurable timelines")
    (description
     "The package supports drawing timelines and aims to be highly configurable.
Custom color schemes and styles are supported, and the package includes
documented examples of each.  Memoization works out-of-the-box if the
@code{memoize} package is loaded.")
    (license license:lppl1.3c)))

(define-public texlive-chronosys
  (package
    (name "texlive-chronosys")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/chronosys/"
                                     "tex/generic/chronosys/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "145ap67ykdkp2f0ahasp9sjccrv1wzxdfl05nq6hbi6z65pk56s5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chronosys")
    (synopsis "Drawing time-line diagrams")
    (description
     "This package provides macros to produce time line diagrams.
Interfaces for Plain TeX, ConTeXt and LaTeX are provided.")
    (license license:lppl1.3+)))

(define-public texlive-chs-physics-report
  (package
    (name "texlive-chs-physics-report")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/chs-physics-report/"
                                     "tex/latex/chs-physics-report/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "195f66yffi96qkrxb044mcaycyds0mp3fg3c3lhz1mpvbz0calpl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chs-physics-report")
    (synopsis "Physics lab reports for Carmel High School")
    (description
     "This package may optionally be used by students at Carmel High School in
Indiana in the United States to write physics lab reports for FW physics
courses.  As many students are beginners at LaTeX, it also attempts to
simplify the report-writing process by offering macros for commonly used
notation and by automatically formatting the documents for students who will
only use TeX for mathematics and not typesetting.")
    (license (list license:public-domain license:cc-by-sa3.0))))

(define-public texlive-chscite
  (package
    (name "texlive-chscite")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/chscite/" "doc/latex/chscite/"
                                     "source/latex/chscite/" "tex/latex/chscite/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0q4mgvrbljbw7pmwjs4qcr91sfhbnbshhmygxqrkriyrfnqy6yfq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chscite")
    (synopsis "Bibliography style for Chalmers University of Technology")
    (description
     "The package, heavily based on the harvard package for Harvard-style
citations, provides a citation suite for students at Chalmers University of
Technology that follows given recommendations.")
    (license license:lppl1.2+)))

(define-public texlive-chuushaku
  (package
    (name "texlive-chuushaku")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/chuushaku/"
                              "tex/latex/chuushaku/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "03np78w18m3qcp0aw134vxzcqi14j1qrasdixyfajgr5gd0sh9zs"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/chuushaku")
    (synopsis "Flexible book notes in Japanese")
    (description
     "This style file is designed for compiling book notes in Japanese as part
of the body text.  The ``remember picture'' feature automatically calculates
coordinates, eliminating the need for manual adjustment of note positions.")
    (license license:lppl1.3c)))

(define-public texlive-churchslavonic
  (package
    (name "texlive-churchslavonic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/churchslavonic/"
                                     "tex/latex/churchslavonic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1m9sf9nwnlribfhh5lks4lfnkc5fglcr4x3y30g03hcqrwwi15ks"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs
     (list texlive-etoolbox
           texlive-fonts-churchslavonic
           texlive-hyphen-complete
           texlive-oberdiek
           texlive-xcolor))
    (home-page "https://ctan.org/pkg/churchslavonic")
    (synopsis "Typeset documents in Church Slavonic language using Unicode")
    (description
     "The package provides fonts, hyphenation patterns, and supporting macros
to typeset Church Slavonic texts.")
    (license license:expat)))

(define-public texlive-cinzel
  (package
    (name "texlive-cinzel")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cinzel/"
                                     "fonts/enc/dvips/cinzel/"
                                     "fonts/map/dvips/cinzel/"
                                     "fonts/tfm/ndiscovered/cinzel/"
                                     "fonts/truetype/ndiscovered/cinzel/"
                                     "fonts/type1/ndiscovered/cinzel/"
                                     "fonts/vf/ndiscovered/cinzel/"
                                     "tex/latex/cinzel/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qik1ji23w35350ajjxz5w39w1cy4lg582343nrhr9y8jl18kfap"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cinzel")
    (synopsis "LaTeX support for Cinzel and Cinzel Decorative fonts")
    (description
     "Cinzel and Cinzel Decorative fonts, designed by Natanael Gama,
find their inspiration in first century roman inscriptions, and are based on
classical proportions.  Cinzel is all-caps (similar to Trajan and
Michelangelo), but is available in three weights (Regular, Bold, Black).
There are no italic fonts, but there are Decorative variants, which can be
selected by the usual italic-selection commands in the package's LaTeX
support.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-citation-style-language
  (package
    (name "texlive-citation-style-language")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/citation-style-language/"
                                     "doc/man/man1/citeproc-lua.1"
                                     "doc/man/man1/citeproc-lua.man1.pdf"
                                     "scripts/citation-style-language/"
                                     "tex/latex/citation-style-language/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gd5flkd1ixn7p1xwp6wz258ndv919y3x9xr5dwhjlwqj5w92ax0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "citeproc-lua.lua")))
    (propagated-inputs
     (list texlive-l3kernel
           texlive-l3packages
           texlive-lua-tinyyaml
           texlive-lua-uca
           texlive-lualibs
           texlive-luatex
           texlive-luaxml
           texlive-url))
    (home-page "https://ctan.org/pkg/citation-style-language")
    (synopsis "Bibliography formatting with Citation Style Language")
    (description
     "The Citation Style Language (CSL) is an XML-based language that defines
the formats of citations and bibliography.  There are currently thousands of
styles in CSL including the most widely used APA, Chicago, Vancouver, etc.
The citation-style-language package is aimed to provide another reference
formatting method for LaTeX that utilizes the CSL styles.  It contains
a citation processor implemented in pure Lua (@code{citeproc-lua}) which reads
bibliographic metadata and performs sorting and formatting on both citations
and bibliography according to the selected CSL style.  A LaTeX package
(@file{citation-style-language.sty}) is provided to communicate with the
processor.")
    (license (list license:expat license:cc-by-sa3.0))))

(define-public texlive-citeall
  (package
    (name "texlive-citeall")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/citeall/" "tex/latex/citeall/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0sap2kns1b51zj0dmy70q77rw7dnhdjqab0rmyvpp2hgq5yk3mhq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/citeall")
    (synopsis "Cite all entries of a @file{.bbl} created with BibLaTeX")
    (description
     "This small package allows citing all entries of
a BibLaTeX (@file{.bbl}) file.")
    (license license:lppl1.3+)))

(define-public texlive-citeref
  (package
    (name "texlive-citeref")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/citeref/" "tex/latex/citeref/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0k76msdfsf968cv3m1z8lcf6mlmy90gaaphrgkr7cs67pgyvq6jq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/citeref")
    (synopsis "Add reference-page-list to bibliography-items")
    (description
     "The package adds reference-page-list to bibliography-items.  It does its
job without using the indexing facilities, and needs no special
@code{\\cite}-replacement package.")
    (license license:bsd-4)))

(define-public texlive-cje
  (package
    (name "texlive-cje")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/cje/" "doc/latex/cje/"
                                     "tex/latex/cje/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0n24knrb0ayh9dnqg4l776p5wzh5j1bbqxvwpf7pcljcml94iaa7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cje")
    (synopsis "LaTeX document class for CJE articles")
    (description
     "The @code{cje} article class allows authors to format their papers to
@emph{Canadian Journal of Economics} style with minimum effort.  The class
includes options for two other formats: @code{review} (double spaced, for use
at the submission stage) and @code{proof} (used by the typesetters to prepare
the proof authors will receive for approval).")
    (license license:lppl1.3c)))

(define-public texlive-cjhebrew
  (package
    (name "texlive-cjhebrew")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cjhebrew/"
                                     "fonts/afm/public/cjhebrew/"
                                     "fonts/enc/dvips/cjhebrew/"
                                     "fonts/map/dvips/cjhebrew/"
                                     "fonts/tfm/public/cjhebrew/"
                                     "fonts/type1/public/cjhebrew/"
                                     "fonts/vf/public/cjhebrew/"
                                     "tex/latex/cjhebrew/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00rf55ck8y1ny1v242sbhcfchnq7q5mlsmmqppgqspd892ymlxh0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cjhebrew")
    (synopsis "Typeset Hebrew with LaTeX")
    (description
     "The @code{cjhebrew} package provides Adobe Type 1 fonts for Hebrew, and
LaTeX macros to support their use.  Hebrew text can be vocalised, and a few
accents are also available.  The package makes it easy to include Hebrew text
in other-language documents.")
    (license license:lppl)))

(define-public texlive-cjk
  (package
    (name "texlive-cjk")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cjk/"
                                     "source/latex/cjk/contrib/wadalab/"
                                     "source/latex/cjk/texinput/KS/HLaTeX/"
                                     "source/latex/cjk/utils/"
                                     "tex/latex/cjk/contrib/wadalab/"
                                     "tex/latex/cjk/texinput/"
                                     "tex/latex/cjk/utils/pyhyphen/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "044r1mmszq3hjmlck87vmqp62j2z02bm7lda8wb17phs0grkws5z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs
     (list texlive-arphic
           texlive-cns
           texlive-garuda-c90
           texlive-norasi-c90
           texlive-uhc
           texlive-wadalab))
    (home-page "https://ctan.org/pkg/cjk")
    (synopsis "CJK language support")
    (description
     "CJK is a macro package for LaTeX, providing simultaneous support for
various Asian scripts in many encodings (including Unicode): Chinese (both
traditional and simplified), Japanese, Korean and Thai.  A special add-on
feature is an interface to the Emacs editor (@file{cjk-enc.el}) which gives
simultaneous, easy-to-use support to a bunch of other scripts in addition to
the above -- Cyrillic, Greek, Latin-based scripts, Russian and Vietnamese are
supported.")
    (license license:gpl2)))

(define-public texlive-cjk-gs-integrate
  (package
    (name "texlive-cjk-gs-integrate")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cjk-gs-integrate/"
                                     "fonts/misc/cjk-gs-integrate/"
                                     "scripts/cjk-gs-integrate/"
                                     "source/fonts/cjk-gs-integrate/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ml35bs1pr3kbdqnbclyy31ln8yigy609rsm9rc2y3j5ngmw2y8y"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "cjk-gs-integrate.pl")))
    (inputs (list perl))
    (home-page "https://ctan.org/pkg/cjk-gs-integrate")
    (synopsis "Tools to integrate CJK fonts into Ghostscript")
    (description
     "This script searches a list of directories for CJK fonts, and makes them
available to an installed Ghostscript.  In the simplest case, with sufficient
privileges, a run without arguments should result in a complete setup of
Ghostscript.")
    (license license:gpl3)))

(define-public texlive-cjk-ko
  (package
    (name "texlive-cjk-ko")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cjk-ko/" "tex/latex/cjk-ko/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1yj4j0xbdchi56xyk1fq3xj33xwr9v58yfq4d8798mmzi9bd00m9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-cjk))
    (home-page "https://ctan.org/pkg/cjk-ko")
    (synopsis "Extension of the CJK package for Korean typesetting")
    (description
     "The package supports typesetting UTF-8-encoded modern Korean documents
with the help of the LaTeX2e CJK package.  It provides some enhanced features
focused on Korean typesetting culture, one of them being allowing line-break
between Latin and CJK characters.")
    (license (list license:gpl3+ license:lppl license:public-domain))))

(define-public texlive-cjkpunct
  (package
    (name "texlive-cjkpunct")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cjkpunct/"
                                     "source/latex/cjkpunct/"
                                     "tex/latex/cjkpunct/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0l4jnawgcsq7lzsrky7259diswxm06mn9mvgxgw8j0za10kvv4vb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cjkpunct")
    (synopsis "Adjust locations and kerning of CJK punctuation marks")
    (description "The package serves as a companion package for CJK.  It
adjust locations and kerning of CJK punctuation marks.")
    (license license:lppl1.3+)))

(define-public texlive-cjkutils
  (package
    (name "texlive-cjkutils")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/bg5conv.1"
                                     "doc/man/man1/bg5conv.man1.pdf"
                                     "doc/man/man1/cef5conv.1"
                                     "doc/man/man1/cef5conv.man1.pdf"
                                     "doc/man/man1/cefconv.1"
                                     "doc/man/man1/cefconv.man1.pdf"
                                     "doc/man/man1/cefsconv.1"
                                     "doc/man/man1/cefsconv.man1.pdf"
                                     "doc/man/man1/extconv.1"
                                     "doc/man/man1/extconv.man1.pdf"
                                     "doc/man/man1/hbf2gf.1"
                                     "doc/man/man1/hbf2gf.man1.pdf"
                                     "doc/man/man1/sjisconv.1"
                                     "doc/man/man1/sjisconv.man1.pdf"
                                     "hbf2gf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0by2g05xv5dndnd78jz9y73fyswqhfvcbzcw8rzhvpvd6inrcdq8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-cjkutils-bin))
    (home-page "https://ctan.org/pkg/cjk")
    (synopsis "CJK language support")
    (description
     "CJK is a macro package for LaTeX, providing simultaneous support for
various Asian scripts in many encodings (including Unicode): Chinese (both
traditional and simplified), Japanese, Korean and Thai.  A special add-on
feature is an interface to the Emacs editor (cjk-enc.el) which gives
simultaneous, easy-to-use support to a bunch of other scripts in addition to
the above --- Cyrillic, Greek, Latin-based scripts, Russian and Vietnamese are
supported.")
    (license license:gpl2)))

(define-public texlive-cjkutils-bin
  (package
    (inherit texlive-bin)
    (name "texlive-cjkutils-bin")
    (source
     (origin
       (inherit (package-source texlive-source))
       (modules '((guix build utils)
                  (ice-9 ftw)))
       (snippet
        #~(let ((delete-other-directories
                 (lambda (root dirs)
                   (with-directory-excursion root
                     (for-each
                      delete-file-recursively
                      (scandir "."
                               (lambda (file)
                                 (and (not (member file (append '("." "..") dirs)))
                                      (eq? 'directory (stat:type (stat file)))))))))))
            (delete-other-directories "libs/" '())
            (delete-other-directories "utils/" '())
            (delete-other-directories "texk/" '("cjkutils"))))))
    (arguments
     (substitute-keyword-arguments (package-arguments texlive-bin)
       ((#:configure-flags flags)
        #~(cons "--enable-cjkutils-x" (delete "--enable-web2c" #$flags)))
       ((#:phases phases)
        #~(modify-phases #$phases
            (replace 'check
              (lambda* (#:key tests? #:allow-other-keys)
                (when tests?
                  (with-directory-excursion "texk/cjkutils"
                    (invoke "make" "check")))))
            (replace 'install
              (lambda _
                (with-directory-excursion "texk/cjkutils"
                  (invoke "make" "install"))))))))
    (native-inputs (list pkg-config))
    (inputs (list texlive-libkpathsea))
    (propagated-inputs '())
    (home-page (package-home-page texlive-cjkutils))
    (synopsis "Binaries for @code{texlive-cjkutils}")
    (description
     "This package provides the binaries for @code{texlive-cjkutils}.")
    (license (package-license texlive-cjkutils))))

(define-public texlive-cjs-rcs-article
  (package
    (name "texlive-cjs-rcs-article")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list
                         "bibtex/bst/cjs-rcs-article/"
                         "doc/latex/cjs-rcs-article/"
                         "source/latex/cjs-rcs-article/"
                         "tex/latex/cjs-rcs-article/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0a5d1dkzcz4vqh9vb3lyv7rxjj8ahj48mx453gx99arvcwdw2fm3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cjs-rcs-article")
    (synopsis "Article class for The Canadian Journal of Statistics")
    (description
     "The document class @code{cjs-rcs-article} and its companion
bibliographic styles @code{cjs-rcs-en} and @code{cjs-rcs-fr} typeset
manuscripts immediately in accordance with the presentation rules of The
Canadian Journal of Statistics.")
    (license (list license:lppl1.3c license:cc-by-sa4.0))))

(define-public texlive-clara
  (package
    (name "texlive-clara")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/clara/"
                                     "fonts/enc/dvips/clara/"
                                     "fonts/map/dvips/clara/"
                                     "fonts/opentype/public/clara/"
                                     "fonts/tfm/public/clara/"
                                     "fonts/type1/public/clara/"
                                     "fonts/vf/public/clara/"
                                     "tex/latex/clara/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "182iw362irf247frsy0dzrswnh1341jlwrlwhvx1hcrbrvhpbz42"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/clara")
    (synopsis "Clara serif font family")
    (description
     "Clara is a type family created specially by Seamas O Brogain.  The
family includes italic, bold, bold italic, and small capitals, while the
character set includes (monotonic) Greek, Cyrillic, ogham, phonetic and
mathematical ranges, scribal abbreviations and other specialist characters.
The fonts also include some OpenType features (such as ligature substitution,
small capitals, and old-style numerals) and variant forms for particular
languages.")
    (license (list license:silofl1.1 license:gpl2+))))

(define-public texlive-classicthesis
  (package
    (name "texlive-classicthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/classicthesis/"
                                     "tex/latex/classicthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0y6l2rdmpw6m86r1ji2picnkl5lf0cdn89ajm5f14m0lp2prlk2q"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/classicthesis")
    (synopsis "Classically styled thesis package")
    (description
     "This package provides an elegant layout designed in homage to
Bringhurst's @emph{The Elements of Typographic Style}.  It makes use of
a range of techniques to get the best results achievable using TeX.  Included
in the bundle are templates to make thesis writing easier.")
    (license license:gpl2+)))

(define-public texlive-cleanthesis
  (package
    (name "texlive-cleanthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cleanthesis/"
                                     "tex/latex/cleanthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "11fv49w1nydm5fvkp1p2375ywdfd5yh8hayqfnr6wdidzdycpg85"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cleanthesis")
    (synopsis "Clean LaTeX style for thesis documents")
    (description
     "The package offers a clean, simple, and elegant LaTeX style for thesis
documents.")
    (license license:lppl1.3+)))

(define-public texlive-clearsans
  (package
    (name "texlive-clearsans")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/clearsans/"
                                     "fonts/enc/dvips/clearsans/"
                                     "fonts/map/dvips/clearsans/"
                                     "fonts/tfm/intel/clearsans/"
                                     "fonts/truetype/intel/clearsans/"
                                     "fonts/type1/intel/clearsans/"
                                     "fonts/vf/intel/clearsans/"
                                     "tex/latex/clearsans/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1sf84xxvq9rxb3n5nsggfswvdagynnz6clazk86w4k2a5r0qf8yw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/clearsans")
    (synopsis "Clear Sans fonts with LaTeX support")
    (description
     "Clear Sans was designed by Daniel Ratighan.  It is available in three
weights (regular, medium, and bold) with corresponding italics, plus light and
thin upright (without italics).

It has minimized, unambiguous characters and slightly narrow proportions,
making it ideal for UI design.  Its strong, recognizable forms avoid
distracting ambiguity, making Clear Sans comfortable for reading short UI
labels and long passages in both screen and print.  The fonts are available in
both TrueType and Type 1 formats.")
    (license (list license:asl2.0 license:lppl))))

(define-public texlive-cleveref
  (package
    (name "texlive-cleveref")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cleveref/"
                                     "source/latex/cleveref/"
                                     "tex/latex/cleveref/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "18r8g6ipn1ly60gnp0yi0m409gm8lssbpmka5813i4l45281fk43"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cleveref")
    (synopsis "Intelligent cross-referencing")
    (description
     "The package enhances LaTeX's cross-referencing features, allowing the
format of references to be determined automatically according to the type of
reference.  The formats used may be customised in the preamble of a document;
Babel support is available (though the choice of languages remains limited:
currently Danish, Dutch, English, French, German, Italian, Norwegian, Russian,
Spanish and Ukrainian).

The package also offers a means of referencing a list of references, each
formatted according to its type.  In such lists, it can collapse sequences of
numerically-consecutive labels to a reference range.")
    (license license:lppl1.2+)))

(define-public texlive-cleveref-forward
  (package
    (name "texlive-cleveref-forward")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list
                         "doc/latex/cleveref-forward/"
                         "tex/latex/cleveref-forward/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "11y50jcjnzqlqg4sbjdmxqphdnpcgalnkb5h1frl3hi02q3x3whd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cleveref-forward")
    (synopsis "Forward-referencing functionality for @code{cleveref}")
    (description
     "This package aims at providing the forward-referencing functionality for
the package @code{cleveref}.  It is derived from the package
@code{cleveref-usedon}, with several fixes and enhancements.")
    (license license:lppl1.3c)))

(define-public texlive-cleveref-usedon
  (package
    (name "texlive-cleveref-usedon")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/cleveref-usedon/"
                              "source/latex/cleveref-usedon/"
                              "tex/latex/cleveref-usedon/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "15s0r7p86jnkqsyqwb3ds7acapdj72chb5q40kc166cribaqzwc2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cleveref-usedon")
    (synopsis "Adds forward-referencing functionality to @code{cleveref}")
    (description
     "This package adds forward-referencing to the @code{cleveref} package.
Any label can be referenced with the new optional argument @code{UsedOn}
passed to @code{\\cref}.  Doing so, it will print an info message at the
original label location (in a @code{theorem} environment, say) which reads
``Used on pages ⟨list of pages⟩.'.  This functionality is complementary to
@code{pagebackref} option from @code{hyperref} or @code{backref} option from
@code{biblatex} for the bibliography.  It might be useful for authors of
longer texts such as textbooks or theses, where a lot of supplementary results
and information are given in early chapters, appendices or exercises.  The
message on which pages these results will be used can be a helpful information
for the reader of the final text.")
    (license license:lppl1.3+)))

(define-public texlive-clrscode
  (package
    (name "texlive-clrscode")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/clrscode/" "tex/latex/clrscode/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gl2ap3w7zanbrn1k2bcf3f75lfpakp5dd0w8m1jy46hyvha7jpd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/clrscode")
    (synopsis "Typesets pseudocode as in Introduction to Algorithms")
    (description
     "This package allows you to typeset pseudocode in the style of
@emph{Introduction to Algorithms}, Second edition, by Cormen, Leiserson,
Rivest, and Stein.  The package was written by the authors.  You use the
commands the same way the package's author did when writing the book, and your
output will look just like the pseudocode in the text.")
    (license license:lppl)))

(define-public texlive-clrscode3e
  (package
    (name "texlive-clrscode3e")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/clrscode3e/"
                                     "tex/latex/clrscode3e/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1kcsrlvqs83fc9ianim3drqvz60xp2cfl6a554icnm39p1fjf80c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/clrscode3e")
    (synopsis "Typesets pseudocode as in @emph{Introduction to Algorithms}")
    (description
     "This package allows you to typeset pseudocode in the style of
@emph{Introduction to Algorithms}, Third edition, by Cormen, Leiserson,
Rivest, and Stein.  The package was written by the authors.  Use the commands
the same way the package's author did when writing the book, and your output
will look just like the pseudocode in the text.")
    (license license:lppl)))

(define-public texlive-cm-mf-extra-bold
  (package
    (name "texlive-cm-mf-extra-bold")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "fonts/source/public/cm-mf-extra-bold/"
                                     "fonts/tfm/public/cm-mf-extra-bold/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1l1cm32z8wcpj0cl7czqkhnwil3ail7a2bbzn70dnswpj0ka8k9n"))))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/cm-mf-extra-bold")
    (synopsis "Extra Metafont files for CM")
    (description
     "The bundle provides bold versions of @code{cmcsc}, @code{cmex},
@code{cmtex} and @code{cmtt} fonts (all parts of the standard Computer Modern
font distribution), as Metafont base files.")
    (license (list license:gpl3+ license:public-domain))))

(define-public texlive-cm-unicode
  (package
    (name "texlive-cm-unicode")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cm-unicode/"
                                     "fonts/afm/public/cm-unicode/"
                                     "fonts/enc/dvips/cm-unicode/"
                                     "fonts/map/dvips/cm-unicode/"
                                     "fonts/opentype/public/cm-unicode/"
                                     "fonts/type1/public/cm-unicode/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15ysic6vah1wircybhpm9915mr1bbc8b2jalia86ajiqbxn2l9h0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cm-unicode")
    (synopsis "Computer Modern Unicode font family")
    (description
     "This package provides Computer Modern Unicode fonts.  Some characters in
several fonts are copied from Blue Sky Type 1 fonts released by AMS.
Currently the fonts contain glyphs from Latin, Cyrillic, Greek code sets and
IPA extensions.  This font set contains 33 fonts.  This archive contains AFM,
PFB and OTF versions; the OTF version of the Computer Modern Unicode fonts
works with TeX engines that directly support OpenType features, such as XeTeX
and LuaTeX.")
    (license license:silofl1.1)))

(define-public texlive-cmathbb
  (package
    (name "texlive-cmathbb")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cmathbb/"
                                     "fonts/enc/dvips/cmathbb/"
                                     "fonts/map/dvips/cmathbb/"
                                     "fonts/tfm/public/cmathbb/"
                                     "fonts/type1/public/cmathbb/"
                                     "fonts/vf/public/cmathbb/"
                                     "tex/latex/cmathbb/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pmzsaskwcbyjihidy7hvz7s2cixhl7ccy0c9cbh2cm3md6gdrqh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cmathbb")
    (synopsis "Computer modern mathematical blackboard bold font")
    (description
     "This font contains all digits and latin letters uppercase and lowercase
for the Computer Modern font family in blackboard bold.")
    (license license:lppl1.3c)))

(define-public texlive-cmll
  (package
    (name "texlive-cmll")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cmll/"
                                     "fonts/map/dvips/cmll/"
                                     "fonts/source/public/cmll/"
                                     "fonts/tfm/public/cmll/"
                                     "fonts/type1/public/cmll/"
                                     "source/latex/cmll/"
                                     "tex/latex/cmll/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13q5ar2zaf7h6f2mzgxxqzaraziwl6asicbnyq6anfv2ky38rd72"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-euxm")))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (native-inputs (list texlive-euxm texlive-metafont))
    (home-page "https://ctan.org/pkg/cmll")
    (synopsis "Symbols for linear logic")
    (description
     "This is a very small font set that contain some symbols useful in linear
logic, which are apparently not available elsewhere.  Variants are included
for use with Computer Modern serif and sans-serif and with the AMS Euler
series.  The font is provided both as Metafont source, and in Adobe Type
1 format.  LaTeX support is provided.")
    (license license:lppl)))

(define-public texlive-cmpica
  (package
    (name "texlive-cmpica")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cmpica/"
                                     "fonts/source/public/cmpica/"
                                     "fonts/tfm/public/cmpica/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0gpj2i588h6d0b5cl4xaik9nyx7w1sd8wik5q4mgmzspbkdk5a04"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/cmpica")
    (synopsis "Computer Modern Pica variant")
    (description
     "This is an approximate equivalent of the Xerox Pica typeface; the font
is optimised for submitting fiction manuscripts to mainline publishers.  The
font is a fixed-width one, rather less heavy than Computer Modern typewriter.
Emphasis for bold-face comes from a wavy underline of each letter.  The two
fonts are supplied as Metafont source.")
    (license license:public-domain)))

(define-public texlive-cmpj
  (package
    (name "texlive-cmpj")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/cmpj/" "doc/latex/cmpj/"
                                     "tex/latex/cmpj/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1drmil2vba85wa7ykrma452hwh7ggzsq33zrhch7rjnqaw7xihwv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cmpj")
    (synopsis "Style for the journal @emph{Condensed Matter Physics}")
    (description
     "The package contains macros and some documentation for typesetting
papers for submission to the @emph{Condensed Matter Physics} journal published
by the Institute for Condensed Matter Physics of the National Academy of
Sciences of Ukraine.")
    (license license:lppl)))

(define-public texlive-cmsrb
  (package
    (name "texlive-cmsrb")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cmsrb/"
                                     "fonts/afm/public/cmsrb/"
                                     "fonts/enc/dvips/cmsrb/"
                                     "fonts/map/dvips/cmsrb/"
                                     "fonts/tfm/public/cmsrb/"
                                     "fonts/type1/public/cmsrb/"
                                     "fonts/vf/public/cmsrb/"
                                     "tex/latex/cmsrb/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0f6ylk8n881sl0njbz7yqy98pbk51s6m73zldcsz2w62qnghd1x0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cmsrb")
    (synopsis "Computer Modern for Serbian and Macedonian")
    (description
     "This package provides provides Adobe Type 1 Computer Modern fonts for
the Serbian and Macedonian languages.  Although the @code{cm-super} package
provides great support for Cyrillic script in various languages, there remains
a problem with italic variants of some letters for Serbian and Macedonian.
This package includes the correct shapes for italic letters @code{\\cyrb},
@code{\\cyrg}, @code{\\cyrd}, @code{\\cyrp}, and @code{\\cyrt}.  It also
offers some improvements in letters and accents used in the Serbian language.
Supported encodings are: T1, T2A, TS1, X2 and OT2.  The OT2 encoding is
modified so that it is now easy to transcribe Latin text to Cyrillic.")
    (license license:gpl3+)))

(define-public texlive-cmtiup
  (package
    (name "texlive-cmtiup")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cmtiup/"
                                     "fonts/source/public/cmtiup/"
                                     "fonts/tfm/public/cmtiup/"
                                     "fonts/vf/public/cmtiup/"
                                     "tex/latex/cmtiup/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0cdm10aybgc8ifi11fahl3hb18blmibk08b4b5m8q88150ic563r"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/cmtiup")
    (synopsis "Upright punctuation with CM italic")
    (description
     "The @code{cmtiup} fonts address a problem with the appearance of
punctuation in italic text in mathematical documents.  To achieve this, all
punctuation characters are upright, and kerning between letters and
punctuation is adjusted to allow for the italic correction.  The fonts are
implemented as a set of @file{.vf} files; a package for support in LaTeX is
provided.")
    (license license:lppl1.3+)))

(define-public texlive-cmupint
  (package
    (name "texlive-cmupint")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cmupint/"
                                     "fonts/afm/public/cmupint/"
                                     "fonts/map/dvips/cmupint/"
                                     "fonts/source/public/cmupint/"
                                     "fonts/tfm/public/cmupint/"
                                     "fonts/type1/public/cmupint/"
                                     "tex/latex/cmupint/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1acgzx5b02900vzwvaq0j8w6x8jc1650kfhp992f3bcyd658aj62"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/cmupint")
    (synopsis "Upright integral symbols for Computer Modern")
    (description
     "This package contains various upright integral symbols to match the
Computer Modern font.")
    (license license:lppl)))

(define-public texlive-cochineal
  (package
    (name "texlive-cochineal")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cochineal/"
                                     "fonts/afm/public/cochineal/"
                                     "fonts/enc/dvips/cochineal/"
                                     "fonts/map/dvips/cochineal/"
                                     "fonts/opentype/public/cochineal/"
                                     "fonts/tfm/public/cochineal/"
                                     "fonts/type1/public/cochineal/"
                                     "fonts/vf/public/cochineal/"
                                     "tex/latex/cochineal/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1lr07badgq7glvjbpvifif15y86kr58xf0ag937jlr5f7vbmk1ax"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cochineal")
    (synopsis "Cochineal fonts with LaTeX support")
    (description
     "Cochineal is a fork from the Crimson fonts (Roman, Italic, Bold,
@code{BoldItalic} only), which contain roughly 4200 glyphs in the four styles
mentioned above.  Cochineal adds more than 1500 glyphs in those styles so that
it is possible to make a TeX support collection that contains essentially all
glyphs in all styles.  The fonts are provided in OpenType and PostScript
formats.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-codeanatomy
  (package
    (name "texlive-codeanatomy")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/codeanatomy/"
                                     "source/latex/codeanatomy/"
                                     "tex/latex/codeanatomy/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1frkyj86kf9b3n08xlwfndiygllymp2lp7ra82zsnx4xzw1ff4as"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/codeanatomy")
    (synopsis "Typeset code with annotations")
    (description
     "The idea of this Package is to typeset illustrations of pieces of code
with annotations on each single part of code (Code Anatomy).  The origin of
this idea are code illustrations from the book @emph{Computer Science: An
Interdisciplinary Approach} from Robert Sedgewick and Kevin Wayne.")
    (license license:lppl1.3c)))

(define-public texlive-codicefiscaleitaliano
  (package
    (name "texlive-codicefiscaleitaliano")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/codicefiscaleitaliano/"
                                     "source/latex/codicefiscaleitaliano/"
                                     "tex/latex/codicefiscaleitaliano/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ij3vcikhwx1d55z2gsx4yb9fin9dhm8yjwc6apiwg353ipxwibw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/codicefiscaleitaliano")
    (synopsis "Test the consistency of the Italian personal Fiscal Code")
    (description
     "The alphanumeric string that forms the Italian personal Fiscal Code is
prone to be misspelled thus rendering a legal document invalid.  The package
quickly verifies the consistency of the fiscal code string, and can therefore
be useful for lawyers and accountants that use fiscal codes very frequently.")
    (license license:lppl1.3+)))

(define-public texlive-coelacanth
  (package
    (name "texlive-coelacanth")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/coelacanth/"
                                     "fonts/enc/dvips/coelacanth/"
                                     "fonts/map/dvips/coelacanth/"
                                     "fonts/opentype/public/coelacanth/"
                                     "fonts/tfm/public/coelacanth/"
                                     "fonts/type1/public/coelacanth/"
                                     "fonts/vf/public/coelacanth/"
                                     "tex/latex/coelacanth/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1cwk36rm1m2qaphwlpnw6rhmsmkvsdpzqs7sv1c39qcyq853q4yj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/coelacanth")
    (synopsis "Coelacanth fonts with LaTeX support")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX, and LuaLaTeX support for
Coelecanth fonts, designed by Ben Whitmore.  Coelacanth is inspired by the
classic Centaur type design of Bruce Rogers, described by some as the most
beautiful typeface ever designed.  It aims to be a professional quality type
family for general book typesetting.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-collargs
  (package
    (name "texlive-collargs")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/generic/collargs/"
                              "source/generic/collargs/"
                              "tex/context/third/collargs/"
                              "tex/latex/collargs/"
                              "tex/plain/collargs/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "075scmn3r20ycfbf0j9zpsw84irg7j1fmpyyhkm374ixlmsin17j"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-easydtx")))
    (build-system texlive-build-system)
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               (add-after 'unpack 'edtx-to-dtx
                 (lambda _
                   (with-directory-excursion "source/generic/collargs"
                     (with-output-to-file "collargs.dtx"
                       (lambda ()
                         (invoke "edtx2dtx" "collargs.edtx")))))))))
    (native-inputs (list texlive-easydtx))
    (home-page "https://ctan.org/pkg/collargs")
    (synopsis "Collect arguments of any command")
    (description
     "This is a cross-format package providing a command which can determine
the argument scope of any command whose argument structure conforms to
@code{xparse} argument specification.")
    (license license:lppl1.3c)))

(define-public texlive-collref
  (package
    (name "texlive-collref")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/collref/" "source/latex/collref/"
                                     "tex/latex/collref/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0gvla01fypbdqvjg0zziy4cl2s6z441pvn3s8b45dvz4hy1hkfji"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/collref")
    (synopsis "Collect blocks of references into a single reference")
    (description
     "The package automatically collects multiple @code{\\bibitem} references,
which always appear in the same sequence in @code{\\cite}, into a single
@code{\\bibitem} block.")
    (license license:lppl1.3+)))

(define-public texlive-colorblind
  (package
    (name "texlive-colorblind")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/colorblind/"
                              "tex/latex/colorblind/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0w21drrpmal8iw50kcvik0d2q7sgj7ngqif01iq9xky5xiy1ywx6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/colorblind")
    (synopsis "Easy colorblind-safe typesetting")
    (description
     "In colorblind-safe documents, the contents are presented in a way that
the same information is conveyed to readers regardless of a potential color
vision deficiency.  This package provides the tools necessary for
colorblind-safe typesetting in LaTeX.  It provides color schemes for a wide
range of applications.  The most commonly used schemes are qualitative
schemes, providing easily distinguishable colors for use in graphics, but also
for text coloring or highlighting.  Additionally, diverging and sequential
schemes are included which can be used for encoding quantitative information
using colors.  This package incorporates colorblind-safeness into the writing
process, making it both less cumbersome and less error-prone.")
    (license license:lppl1.3c)))

(define-public texlive-coloredbelts
  (package
    (name "texlive-coloredbelts")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/coloredbelts/"
                              "tex/latex/coloredbelts/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0yc50fy2bgl8w76bwb818nxi1ls4llhpq1rfx18s6xvdycpynvci"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/coloredbelts")
    (synopsis "Insert colored belts in documents")
    (description
     "The package provides commands (English and French version) to insert
colored belts to present skills, for example.")
    (license (list license:lppl1.3c license:cc-by-sa3.0))))

(define-public texlive-coloredtheorem
  (package
    (name "texlive-coloredtheorem")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/coloredtheorem/"
                              "tex/latex/coloredtheorem/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "11dmvqxs1gij8hrrp5rrwlcfj6vc68j5492r4qqg8nif2aphmhqh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/coloredtheorem")
    (synopsis "Colorful boxed theorem environment")
    (description
     "This packages provides a colorful boxed theorem environment, combining
@code{tcolorbox} and breakable boxes.")
    (license license:lppl1.3c)))

(define-public texlive-colorsep
  (package
    (name "texlive-colorsep")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "dvips/colorsep/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0nf72pf2bjn8pcps45sn5dcjf1dkrww9wlpp1xbzl1h9i68p3h3h"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/colorsep")
    (synopsis "Color separation")
    (description
     "This package provides support for colour separation when using Dvips.")
    (license license:public-domain)))

(define-public texlive-comfortaa
  (package
    (name "texlive-comfortaa")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/comfortaa/"
                                     "fonts/enc/dvips/comfortaa/"
                                     "fonts/map/dvips/comfortaa/"
                                     "fonts/tfm/aajohan/comfortaa/"
                                     "fonts/truetype/aajohan/comfortaa/"
                                     "fonts/type1/aajohan/comfortaa/"
                                     "fonts/vf/aajohan/comfortaa/"
                                     "tex/latex/comfortaa/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ayh1f0f1a0qkngsk84zwzyx5kidb2n5wq2ksk0d7r68m8cbs981"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/comfortaa")
    (synopsis "Sans serif font, with LaTeX support")
    (description
     "Comfortaa is a sans-serif font, comfortable in every aspect, designed by
Johan Aakerlund.  This package provides support for this font in LaTeX, and
includes both the TrueType fonts, and conversions to Adobe Type 1 format.")
    (license (list license:silofl1.1 license:lppl1.3c))))

(define-public texlive-comicneue
  (package
    (name "texlive-comicneue")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/comicneue/"
                                     "fonts/enc/dvips/comicneue/"
                                     "fonts/map/dvips/comicneue/"
                                     "fonts/opentype/rozynski/comicneue/"
                                     "fonts/tfm/rozynski/comicneue/"
                                     "fonts/type1/rozynski/comicneue/"
                                     "fonts/vf/rozynski/comicneue/"
                                     "tex/latex/comicneue/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0q9ckbypx5hn9004pgfyqnf1ddniqwq0fixps0acmifz8qspjih6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/comicneue")
    (synopsis "Use Comic Neue with TeX(-alike) systems")
    (description
     "Comic Neue is a well-known redesign of the (in)famous Comic Sans font.
The package provides the original OpenType font for XeTeX and LuaTeX users,
and also has converted Type1 files for pdfTeX users.")
    (license license:silofl1.1)))

(define-public texlive-commath
  (package
    (name "texlive-commath")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/commath/" "tex/latex/commath/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0mvlqw78183mp8fdvcmrwxvf4cpqdb2sr6jif16rad0av602lnfr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/commath")
    (synopsis "Mathematics typesetting support")
    (description
     "This package provides a range of differential, partial differential and
delimiter commands, together with a @code{\\fullfunction} (function, with both
domain and range, and function operation) and various reference commands.")
    (license license:lppl)))

(define-public texlive-commutative-diagrams
  (package
    (name "texlive-commutative-diagrams")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/commutative-diagrams/"
                                     "tex/context/third/commutative-diagrams/"
                                     "tex/generic/commutative-diagrams/"
                                     "tex/latex/commutative-diagrams/"
                                     "tex/plain/commutative-diagrams/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0p2icaq0znji162g11yqvjma83y2wg5011dkgbvwlvxawk5rr7dc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/commutative-diagrams")
    (synopsis "CoDi: Commutative Diagrams for TeX")
    (description
     "This package provides a TikZ library for making commutative diagrams
easy to design, parse and tweak.")
    (license license:expat)))

(define-public texlive-compactbib
  (package
    (name "texlive-compactbib")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "tex/latex/compactbib/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "077l73vb0rcxy1n51r3wmcb7gma6nn3xrl543a67n96lpp5xvnnk"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/compactbib")
    (synopsis "Multiple thebibliography environments")
    (description
     "This package allows a second bibliography, optionally with a different
title, after the main bibliography.")
    (license license:lppl)))

(define-public texlive-compare
  (package
    (name "texlive-compare")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "tex/generic/compare/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0isr7gmskv55cr4f1fa7s478v6fh65q9ijxkmbpxj0448g9817w4"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/compare")
    (synopsis "Compare two strings")
    (description
     "The file defines a macro @code{\\compare}, which takes two arguments;
the macro expands to @samp{-1}, @samp{0}, @samp{1}, according as the first
argument is less than, equal to, or greater than the second argument.  Sorting
is alphabetic, using ASCII collating order.")
    (license license:public-domain)))

(define-public texlive-complexity
  (package
    (name "texlive-complexity")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/complexity/"
                                     "tex/latex/complexity/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0yqv0qp4i6jkjkrsvjzq5r2x282y5maq6w5cbvkspp7d0g4sg0ph"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/complexity")
    (synopsis "Computational complexity class names")
    (description
     "Complexity is a LaTeX package that defines commands to typeset
Computational Complexity Classes such as @samp{$\\P$} and @samp{$\\NP$} (as
well as hundreds of others).  It also offers several options including which
font classes are typeset in and how many are defined (all of them or just the
basic, most commonly used ones).")
    (license license:lppl1.3c)))

(define-public texlive-complexpolylongdiv
  (package
    (name "texlive-complexpolylongdiv")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list
                         "doc/latex/complexpolylongdiv/"
                         "source/latex/complexpolylongdiv/"
                         "tex/latex/complexpolylongdiv/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "12kgdk7b90i672ch8hlfnz62bpia689nfi6xih8k6r6fvraf60qf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/complexpolylongdiv")
    (synopsis "Typesetting (complex) polynomial long division")
    (description
     "This package provides a simple interface for typesetting (complex)
polynomial long division.")
    (license license:lppl1.3c)))

(define-public texlive-computational-complexity
  (package
    (name "texlive-computational-complexity")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/computational-complexity/"
                                     "doc/latex/computational-complexity/"
                                     "source/latex/computational-complexity/"
                                     "tex/latex/computational-complexity/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12205p6i611vcywsvkq4qnkbgfyyrkxgrqhj93bfdy4n8y2802ir"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/computational-complexity")
    (synopsis "Class for the journal Computational Complexity")
    (description
     "The LaTeX2e class @code{cc} was written for the journal Computational
Complexity, and it can also be used for a lot of other articles.  You may like
it since it contains a lot of features such as more intelligent references,
a set of theorem definitions, an algorithm environment, and more.")
    (license license:lppl1.2+)))

(define-public texlive-concmath
  (package
    (name "texlive-concmath")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/concmath/"
                                     "source/latex/concmath/"
                                     "tex/latex/concmath/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0lc1bj2yqbyn13bq4fwiqf1hijbj5kwxadifzbg1riscwal3z5vw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/concmath")
    (synopsis "Concrete Math fonts")
    (description
     "This package provides a LaTeX package and font definition files to
access the Concrete mathematics fonts, which were derived from Computer Modern
math fonts using parameters from Concrete Roman text fonts.")
    (license license:lppl)))

(define-public texlive-concmath-fonts
  (package
    (name "texlive-concmath-fonts")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/concmath-fonts/"
                                     "fonts/source/public/concmath-fonts/"
                                     "fonts/tfm/public/concmath-fonts/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09frgpcwpfkj9j9aaj5psb8kr12vgbr10plcvcipn0mjahdw9nnz"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-amsfonts")))
    (build-system texlive-build-system)
    (native-inputs (list texlive-amsfonts texlive-metafont))
    (home-page "https://ctan.org/pkg/concmath-fonts")
    (synopsis "Concrete mathematics fonts")
    (description
     "The fonts are derived from the Computer Modern Mathematics fonts and
from Knuth's Concrete Roman fonts; they are distributed as Metafont source.
LaTeX support is offered by the @code{concmath} package.")
    (license license:lppl)))

(define-public texlive-concmath-otf
  (package
    (name "texlive-concmath-otf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/concmath-otf/"
                                     "fonts/opentype/public/concmath-otf/"
                                     "tex/latex/concmath-otf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1s7bzc14jmhd9zsmicz69yzj2kmzhkrb4x4v93xjdfd6fg9wbpkl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/concmath-otf")
    (synopsis "Concrete based OpenType Math font")
    (description
     "This package provides an OpenType version of the Concrete Math font
created by Ulrik Vieth in Metafont.  @file{concmath-otf.sty} is a replacement
for the original @file{concmath.sty} package to be used with LuaTeX or XeTeX
engines.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-concrete
  (package
    (name "texlive-concrete")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/concrete/"
                                     "fonts/source/public/concrete/"
                                     "fonts/tfm/public/concrete/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0xras2ybr33xm27vl0ym3lyd954gizgyd6h84ivxg0zjpqpjanb1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/concrete")
    (synopsis "Concrete Roman fonts")
    (description
     "This package provides Concrete Roman fonts, designed by Donald Knuth,
originally for use with Euler mathematics fonts.  Alternative mathematics
fonts, based on the concrete parameter set are available as the
@code{concmath} fonts bundle.  LaTeX support is offered by the @code{beton},
@code{concmath} and @code{ccfonts} packages.  T1- and TS1-encoded versions of
the fonts are available in the @code{ecc} bundle, and Adobe Type 1 versions of
the @code{ecc} fonts are part of the @code{cm-super} bundle.")
    (license license:knuth)))

(define-public texlive-confproc
  (package
    (name "texlive-confproc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/confproc/" "doc/latex/confproc/"
                                     "makeindex/confproc/"
                                     "source/latex/confproc/"
                                     "tex/latex/confproc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1v3ppwxss1ans62j1pss8mcfjqmx708za7pmcm1wic7z7byc6xdb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (home-page "https://ctan.org/pkg/confproc")
    (synopsis "Set of tools for generating conference proceedings")
    (description
     "The @code{confproc} collection comprises a class, a BibTeX style, and
some scripts for generating conference proceedings.")
    (license license:lppl)))

(define-public texlive-conteq
  (package
    (name "texlive-conteq")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/conteq/" "source/latex/conteq/"
                                     "tex/latex/conteq/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14lj914i1h29pvg7knfn3ym9nsq6qxngbmkks4hircj0082m86pz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/conteq")
    (synopsis "Typeset multiline continued equalities")
    (description
     "The package provides an environment @code{conteq}, which will lay out
systems of continued equalities (or inequalities).  Several variant layouts of
the equalities are provided, and the user may define their own.")
    (license license:lppl1.3+)))

(define-public texlive-convbkmk
  (package
    (name "texlive-convbkmk")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/support/convbkmk/" "scripts/convbkmk/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0w4fm4az1smrw002sqjkiiz94x01z6jkrb0mf9wb8qxdwrxriql9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "convbkmk.rb")))
    (inputs (list ruby))
    (home-page "https://ctan.org/pkg/convbkmk")
    (synopsis
     "Correct pLaTeX/upLaTeX bookmarks in PDF created with @code{hyperref}")
    (description
     "The package provides a small Ruby script that corrects bookmarks in PDF
files created by pLaTeX or upLaTeX, using @code{hyperref}.")
    (license license:expat)))

(define-public texlive-cookingsymbols
  (package
    (name "texlive-cookingsymbols")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cookingsymbols/"
                                     "fonts/source/public/cookingsymbols/"
                                     "fonts/tfm/public/cookingsymbols/"
                                     "source/latex/cookingsymbols/"
                                     "tex/latex/cookingsymbols/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1cqz5hvb1vcf4mgk7i960kjdbg9cbxkik62riy6l2n1ld2cnl81x"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/cookingsymbols")
    (synopsis "Symbols for recipes")
    (description
     "The package provides 11 symbols for typesetting recipes: oven, gasstove,
topheat, fanoven, gloves and dish symbol (among others).  The symbols are
defined using Metafont.")
    (license license:lppl)))


(define-public texlive-cooperhewitt
  (package
    (name "texlive-cooperhewitt")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cooperhewitt/"
                                     "fonts/enc/dvips/cooperhewitt/"
                                     "fonts/map/dvips/cooperhewitt/"
                                     "fonts/opentype/public/cooperhewitt/"
                                     "fonts/tfm/public/cooperhewitt/"
                                     "fonts/type1/public/cooperhewitt/"
                                     "fonts/vf/public/cooperhewitt/"
                                     "tex/latex/cooperhewitt/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1m81sxrwwsmzsan4iln2lsf16pijay7w9k29gjwlwikivvy21kj7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cooperhewitt")
    (synopsis "Cooper Hewitt family of sans serif fonts")
    (description
     "Cooper Hewitt is a contemporary sans serif, with characters composed of
modified-geometric curves and arches, by Chester Jenkins.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-cora-macs
  (package
    (name "texlive-cora-macs")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/cora-macs/"
                              "tex/latex/cora-macs/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0lgmsykfivvq6ixs9frjclih3jhq7qjd03wf20zkgb2rwafl8cgx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cora-macs")
    (synopsis
     "Macros for continuous sets and neural networks in cyber-physical systems")
    (description
     "This LaTeX package has been designed to assist in the representation and
manipulation of continuous sets, operations, neural networks, and color
schemes tailored for use in the context of cyber-physical systems.  It
provides a comprehensive set of macros that streamline the process of
documenting complex mathematical objects and operations.")
    (license license:expat)))

(define-public texlive-cormorantgaramond
  (package
    (name "texlive-cormorantgaramond")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cormorantgaramond/"
                                     "fonts/enc/dvips/cormorantgaramond/"
                                     "fonts/map/dvips/cormorantgaramond/"
                                     "fonts/tfm/catharsis/cormorantgaramond/"
                                     "fonts/truetype/catharsis/cormorantgaramond/"
                                     "fonts/type1/catharsis/cormorantgaramond/"
                                     "fonts/vf/catharsis/cormorantgaramond/"
                                     "tex/latex/cormorantgaramond/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "198zvgi68cndshyrq82bi3lgfcz7agkknrp9gckm68haji45h1jp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cormorantgaramond")
    (synopsis "Cormorant Garamond family of fonts")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX and LuaLaTeX support for
the Cormorant Garamond family of fonts, designed by Christian Thalman.  The
family includes light, regular, medium, semi-bold, and bold weights, with
italics.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-correctmathalign
  (package
    (name "texlive-correctmathalign")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/correctmathalign/"
                                     "tex/latex/correctmathalign/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1zqla0gjv1b3x3z6g6fdb89c66ny7agshfmcsjjg42vlkz17ianm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/correctmathalign")
    (synopsis "Correct spacing of the alignment in expressions")
    (description
     "This package realigns the horizontal spacing of the alignments in some
mathematical environments.")
    (license license:bsd-2)))

(define-public texlive-couleurs-fr
  (package
    (name "texlive-couleurs-fr")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/couleurs-fr/"
                              "tex/latex/couleurs-fr/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0nkp76z0ar3hjyrw4ydq177bm27giz69f68jf1gcrb246aid0bj4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/couleurs-fr")
    (synopsis "French version of color definitions from @code{xcolor}")
    (description
     "This package provides provides colors with French names, based on
@code{xcolor} and @code{xkcdcolors}.")
    (license license:lppl1.3c)))

(define-public texlive-count1to
  (package
    (name "texlive-count1to")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/count1to/"
                              "source/latex/count1to/"
                              "tex/latex/count1to/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0gihx2qydnwksjm751b3wi1bh083k66sjn6gfsz1b0jijbljrk0i"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (home-page "https://ctan.org/pkg/count1to")
    (synopsis "Make use of @code{count1} to @code{count9}")
    (description
     "This package provides a LaTeX package which sets @code{count1} to
absolute page number, @code{count2-8} to the numbers of the current
@code{\\part}, @code{\\chapter}, ... @code{\\subparagraph}, and @code{count9}
to @code{1} or @code{0}, according to whether the page is odd or even.  These
values can be used to select pages with some drivers.")
    (license license:lppl)))

(define-public texlive-counterz
  (package
    (name "texlive-counterz")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/counterz/"
                              "source/latex/counterz/"
                              "tex/latex/counterz/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "03gs5jv0spm5yfy2rgndlxi9fv0aa21azzyamkq54ifmpnva2zxv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/counterz")
    (synopsis "Additional tools for counters")
    (description
     "The package facilitates the use of stealth prefixes for counter names in
order to help distinguish between counters from multiple input files.  The
package also provides a means to generate random counters and save such
counter values for future typesetting.")
    (license license:lppl1.3c)))

(define-public texlive-countriesofeurope
  (package
    (name "texlive-countriesofeurope")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/countriesofeurope/"
                                     "fonts/afm/public/countriesofeurope/"
                                     "fonts/enc/dvips/countriesofeurope/"
                                     "fonts/map/dvips/countriesofeurope/"
                                     "fonts/opentype/public/countriesofeurope/"
                                     "fonts/tfm/public/countriesofeurope/"
                                     "fonts/type1/public/countriesofeurope/"
                                     "tex/latex/countriesofeurope/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07q0hrrlvadcc3vcl97gsgp1pbkmfhy60zm8ib65qd79m81kyxs0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/countriesofeurope")
    (synopsis "Font with the images of the countries of Europe")
    (description
     "The bundle provides a font CountriesOfEurope (in Adobe Type 1 format)
and the necessary metrics, together with LaTeX macros for its use.  The font
provides glyphs with a filled outline of the shape of each country; each glyph
is at the same cartographic scale.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-courier-scaled
  (package
    (name "texlive-courier-scaled")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/courier-scaled/"
                                     "tex/latex/courier-scaled/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1d61afhvx9s70mg9d97m0zyzqfdlwbgljwgkv0z87khxy4z33wy6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/courier-scaled")
    (synopsis "Provides a scaled Courier font")
    (description
     "This package sets the default typewriter font to Courier with a possible
scale factor (in the same way as the @code{helvet} package for Helvetica works
for sans serif).")
    (license license:lppl1.2+)))

(define-public texlive-courierten
  (package
    (name "texlive-courierten")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/courierten/"
                                     "fonts/enc/dvips/courierten/"
                                     "fonts/map/dvips/courierten/"
                                     "fonts/opentype/public/courierten/"
                                     "fonts/tfm/public/courierten/"
                                     "fonts/type1/public/courierten/"
                                     "fonts/vf/public/courierten/"
                                     "tex/latex/courierten/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0xfra9x03zjx7jsk48kj7mmjng3rs45ydj258c4s6inl91m7i9w0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/courierten")
    (synopsis "Courier 10 Pitch BT with LaTeX support")
    (description
     "This is the font Courier 10 Pitch BT, with LaTeX support and an OpenType
conversion as well.")
    (license license:expat)))

(define-public texlive-covington
  (package
    (name "texlive-covington")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/covington/"
                                     "tex/latex/covington/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "16ba3mzahkaza69a92mqrnrr7jx6r9alcwxgma03pppblfkz3k3q"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/covington")
    (synopsis "LaTeX macros for Linguistics")
    (description
     "This package provides numerous minor LaTeX enhancements for linguistics,
including multiple accents on the same letter, interline glosses (word-by-word
translations), Discourse Representation Structures, and example numbering.")
    (license license:lppl1.3+)))

(define-public texlive-cquthesis
  (package
    (name "texlive-cquthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/cquthesis/"
                                     "doc/latex/cquthesis/"
                                     "source/latex/cquthesis/"
                                     "tex/latex/cquthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0591wd88zp4sgnv2avwqv8127c2g5zbhjr7y15xaahiy7s404hn2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cquthesis")
    (synopsis "LaTeX thesis template for Chongqing University")
    (description
     "CQUThesis stands for Chongqing University thesis template for LaTeX,
bearing the ability to support bachelor, master, doctor dissertations with
grace and speed.")
    (license license:lppl1.3+)))

(define-public texlive-create-theorem
  (package
    (name "texlive-create-theorem")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/create-theorem/"
                                     "tex/latex/create-theorem/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "082nyap4kp7mrl06qbl9lar1x3gs9m4f2zjjbzwkn3fjyng40pfa"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-crefthe))
    (home-page "https://ctan.org/pkg/create-theorem")
    (synopsis "Multilingual support for @code{theorem}-like environments")
    (description
     "This package provides commands for naming, initializing and configuring
@code{theorem}-like environments.  These commands have key-value based
interfaces and are especially useful in multilingual documents, allowing the
easy declaration of @code{theorem}-like environments that can automatically
adapt to the language settings.")
    (license license:lppl1.3c)))

(define-public texlive-crefthe
  (package
    (name "texlive-crefthe")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/crefthe/" "tex/latex/crefthe/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xszhn0bg077297w981miwjnsqylxam9x5bkpli2k0vj73xqmsqs"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/crefthe")
    (synopsis "Cross referencing with proper definite articles")
    (description
     "By default, when using @code{cleveref}'s @code{\\cref} to reference
@code{theorem}-like environments, the names do not contain definite articles.
In languages such as French, Italian, Portuguese, Spanish, etc., this results
in incorrect grammar.  For this purpose, the current package offers
@code{\\crefthe}, which handles the definite articles properly (especially for
the article contractions in many European languages).")
    (license license:lppl1.3c)))

(define-public texlive-creationboites
  (package
    (name "texlive-creationboites")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/creationboites/"
                              "tex/latex/creationboites/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0d207i32niy8shlq0f2ap5yc4agh4zhyh88rjqk887zkzxlxffa5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/creationboites")
    (synopsis "Macros to create simple, yet configurable, color boxes")
    (description
     "This package provides provides some macros to create simple boxes:
@code{\\CreationBoite} to create the box and @code{\\ParamBoites} to modify
some of its parameters.")
    (license license:lppl1.3c)))

(define-public texlive-crimson
  (package
    (name "texlive-crimson")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/crimson/"
                                     "fonts/enc/dvips/crimson/"
                                     "fonts/map/dvips/crimson/"
                                     "fonts/opentype/kosch/crimson/"
                                     "fonts/tfm/kosch/crimson/"
                                     "fonts/type1/kosch/crimson/"
                                     "fonts/vf/kosch/crimson/"
                                     "tex/latex/crimson/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10b2bclm2xs1f0pm5w7j8nb1dn23d4gnlp76416rqzvcp2c08qc9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/crimson")
    (synopsis "Crimson fonts with LaTeX support")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX, and LuaLaTeX support for
the Crimson family of fonts, designed by Sebastian Kosch.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-crimsonpro
  (package
    (name "texlive-crimsonpro")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/crimsonpro/"
                                     "fonts/enc/dvips/crimsonpro/"
                                     "fonts/map/dvips/crimsonpro/"
                                     "fonts/tfm/public/crimsonpro/"
                                     "fonts/truetype/public/crimsonpro/"
                                     "fonts/type1/public/crimsonpro/"
                                     "fonts/vf/public/crimsonpro/"
                                     "tex/latex/crimsonpro/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pzsdm4i32pgwf726q5waga8lr37xfj1qj417kkblw3q3fgb4jsd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/crimsonpro")
    (synopsis "CrimsonPro fonts with LaTeX support")
    (description
     "The CrimsonPro fonts are designed by Jacques Le Bailly and derived
from the Crimson Text fonts designed by Sebastian Kosch.  The family
includes eight weights and italics for each weight.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-crossrefenum
  (package
    (name "texlive-crossrefenum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/crossrefenum/"
                                     "tex/generic/crossrefenum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0vv963kfibrnlq0lr88hl6iziyw0b74a15fhjg88lrifyzdiv281"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/crossrefenum")
    (synopsis
     "Smart typesetting of enumerated cross-references for various TeX formats")
    (description
     "@code{crossrefenum} lets TeX manage the formatting of bunches of
cross-references for you.  It features:

@itemize
@item automatic collapsing of references;
@item support for references by various criteria, including page and note
number, line number in ConTeXt, and @code{edpage} and @code{edline} when used
in conjunction with @code{reledmac};
@item handling of references combining two criteria (e.g., by page and note
number);
@item extension mechanisms to add support to other types of references without
modifying the internal macros.  Note that sorting is not supported.
@end itemize

It is written in Plain TeX as much as possible in order to make it compatible
with a wide array of formats.  For the moment, it works out of the box with
ConTeXt and LaTeX.")
    (license (list license:gpl3+ license:fdl1.3+))))

(define-public texlive-crossrefware
  (package
    (name "texlive-crossrefware")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/bbl2bib.1"
                                     "doc/man/man1/bbl2bib.man1.pdf"
                                     "doc/man/man1/bibdoiadd.1"
                                     "doc/man/man1/bibdoiadd.man1.pdf"
                                     "doc/man/man1/bibmradd.1"
                                     "doc/man/man1/bibmradd.man1.pdf"
                                     "doc/man/man1/biburl2doi.1"
                                     "doc/man/man1/biburl2doi.man1.pdf"
                                     "doc/man/man1/bibzbladd.1"
                                     "doc/man/man1/bibzbladd.man1.pdf"
                                     "doc/man/man1/ltx2crossrefxml.1"
                                     "doc/man/man1/ltx2crossrefxml.man1.pdf"
                                     "doc/support/crossrefware/"
                                     "scripts/crossrefware/"
                                     "tex/latex/crossrefware/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09rqvsf2lgb75z83295qf7h0pnlbgvm3l8p41qlbgymm9505jid0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:link-scripts
           #~(list "bbl2bib.pl"
                   "bibdoiadd.pl"
                   "bibmradd.pl"
                   "biburl2doi.pl"
                   "bibzbladd.pl"
                   "ltx2crossrefxml.pl")))
    (inputs (list perl))
    (home-page "https://ctan.org/pkg/crossrefware")
    (synopsis "Scripts for working with @url{crossref.org}")
    (description
     "This bundle contains the following scripts:
@itemize

@item @file{bibdoiadd.pl}: add DOI numbers to papers in a given @file{.bib}
file,

@item @file{bibzbladd.pl}: add Zbl numbers to papers in a given @file{.bib}
file,

@item @file{bibmradd.pl}: add MR numbers to papers in a given @file{.bib}
file,

@item @file{bbl2bib.pl}: convert @code{thebibliography} environment to
a @file{.bib} file,

@item @file{biburl2doi.pl}: convert URLs pointing to @url{doi.org} to DOIs,

@item @file{ltx2crossrefxml.pl}: tool for the creation of XML files for
submitting to @url{crossref.org}.

@end itemize")
    (license license:gpl3+)))

(define-public texlive-crossword
  (package
    (name "texlive-crossword")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/crossword/"
                                     "source/latex/crossword/"
                                     "tex/latex/crossword/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "134103kkrill6dlql3j9lyx4k7avmv1gmdcri4zrqdrwgh6bv6bd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/crossword")
    (synopsis "Typeset crossword puzzles")
    (description
     "This is an extended grid-based puzzle package, designed to take all
input (both grid and clues) from the same file.  The package can
typeset grids with holes in them, and can deal with several sorts of
puzzle:
@itemize

@item the classical puzzle contains numbers for the words and clues
for the words to be filled in;

@item the numbered puzzle contains numbers in each cell where
identical numbers represent identical letters; the goal is to find out
which number corresponds to which letter;

@item the fill-in type of puzzle consists of a grid and a list of
words; the goal is to place all words in the grid;

@item Sudoku and Kakuro puzzles involve filling in grids of numbers
according to their own rules; format may be block-separated, or
separated by thick lines.

@end itemize")
    (license license:expat)))

(define-public texlive-crosswrd
  (package
    (name "texlive-crosswrd")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/crosswrd/"
                                     "source/latex/crosswrd/"
                                     "tex/latex/crosswrd/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0yhsrfn49wj579ms3smd1z97rjqnsi1wrsgrjs570bllgf09bcir"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/crosswrd")
    (synopsis "Macros for typesetting crossword puzzles")
    (description
     "The package provides a LaTeX method of typesetting crosswords, and
assists the composer ensure that the grid all goes together properly.")
    (license license:lppl)))

(define-public texlive-cryptocode
  (package
    (name "texlive-cryptocode")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cryptocode/"
                                     "source/latex/cryptocode/"
                                     "tex/latex/cryptocode/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0i22bdg2mn305vlmpy32yqbsp7kf5ld54vdvjzq49n8v00qxy286"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cryptocode")
    (synopsis
     "Pseudocode, protocols, game-based proofs, black-box reductions in cryptography")
    (description
     "The @code{cryptocode} package provides a set of macros to ease the
typesetting of pseudocode, algorithms and protocols.  In addition it comes
with a wide range of tools to typeset cryptographic papers.  This includes
simple predefined commands for concepts such as a security parameter or
advantage terms but also flexible and powerful environments to layout
game-based proofs or black-box reductions.")
    (license license:lppl1.3+)))

(define-public texlive-cryst
  (package
    (name "texlive-cryst")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cryst/" "fonts/afm/public/cryst/"
                                     "fonts/source/public/cryst/"
                                     "fonts/tfm/public/cryst/"
                                     "fonts/type1/public/cryst/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05lbldwghpdrmw0ffncs86k5pn04zi0shkk40ycca3kb0h3fa3h8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/cryst")
    (synopsis "Font for graphical symbols used in crystallography")
    (description
     "This package provides the Cryst font, which contains graphical symbols
used in crystallography.  It provided as an Adobe Type 1 font, and as Metafont
source.")
    (license license:lppl)))

(define-public texlive-csassignments
  (package
    (name "texlive-csassignments")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/csassignments/"
                                     "source/latex/csassignments/"
                                     "tex/latex/csassignments/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gdrsh0iyg8dz0pnb9ak099qrznp54x37qqwd7kllw8jjzdcaqg1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/csassignments")
    (synopsis "Extend @code{article} class for computer science assignments")
    (description
     "This class wraps the default @code{article} and extends it for
a homogeneous look of hand-in assignments at university (RWTH Aachen
University, Computer Science Department), specifically in the field of
computer science, but easily extensible to other fields.  It provides macros
for structuring exercises, aggregating points, and displaying a grading table,
as well as several macros for easier math mode usage.")
    (license license:expat)))

(define-public texlive-culmus
  (package
    (name "texlive-culmus")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/fonts/culmus/"
                              "fonts/afm/public/culmus/"
                              "fonts/enc/dvips/culmus/"
                              "fonts/map/dvips/culmus/"
                              "fonts/opentype/public/culmus/"
                              "fonts/tfm/public/culmus/"
                              "fonts/truetype/public/culmus/"
                              "fonts/type1/public/culmus/"
                              "fonts/type3/culmus/"
                              "fonts/vf/public/culmus/"
                              "source/fonts/culmus/"
                              "tex/latex/culmus/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0amrpywyffs124lv3w91gy060s5ikhnjw0i3wjb0frsd79nni8f3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/culmus")
    (synopsis "Hebrew fonts from the Culmus project")
    (description
     "This package provides Hebrew fonts from the Culmus Project.  Both Type1
and Open/TrueType versions of the fonts are provided, as well as font
definition files.  It is recomended to use these fonts with the NHE8 font
encoding, from the @code{hebrew-fonts} package.")
    (license (list license:lppl1.3+ license:gpl2))))

(define-public texlive-cuprum
  (package
    (name "texlive-cuprum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/cuprum/"
                                     "fonts/map/dvips/cuprum/"
                                     "fonts/tfm/public/cuprum/"
                                     "fonts/truetype/public/cuprum/"
                                     "tex/latex/cuprum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wimmwjpx6dg41ncpjbrsxfv41ayppy87f1b8r38vyg0vw6vcsz5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cuprum")
    (synopsis "Cuprum font family support for LaTeX")
    (description
     "This package provides support for the Cuprum font family.")
    (license license:silofl1.1)))

(define-public texlive-custom-bib
  (package
    (name "texlive-custom-bib")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/custom-bib/"
                                     "source/latex/custom-bib/"
                                     "tex/latex/custom-bib/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1hzy0j9k55ygkzvgs3y2jz435267l3g4isqynr64978zhyqybqpd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               ;; The installation process requires ".mbs" files to be
               ;; available when generating the package.  Extend TEXINPUTS to
               ;; include their location.
               (add-before 'build 'extend-texinputs
                 (lambda _
                   (setenv "TEXINPUTS"
                           (string-append (getcwd)
                                          "/tex/latex/custom-bib/:")))))))
    (home-page "https://ctan.org/pkg/custom-bib")
    (synopsis "Customised BibTeX styles")
    (description
     "This package generates customized BibTeX bibliography styles from
a generic file using @code{docstrip} driven by parameters generated by a menu
application.  It includes support for the Harvard style of citations.")
    (license license:lppl)))

(define-public texlive-customdice
  (package
    (name "texlive-customdice")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/customdice/"
                                     "source/latex/customdice/"
                                     "tex/latex/customdice/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jwhvg13rla5pav0z4wns4s0x25myiqcinv592g6kqnbgwzj4q7g"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/customdice")
    (synopsis "Simple commands for drawing customisable dice")
    (description
     "The @code{customdice} package for LaTeX, LuaLaTeX and XeTeX that
provides functionality for drawing dice.  The aim is to provide
highly-customisable but simple-to-use commands, allowing: adding custom text
to dice faces; control over colouring; control over sizing.")
    (license license:cc-by-sa4.0)))

(define-public texlive-customenvs
  (package
    (name "texlive-customenvs")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/customenvs/"
                              "tex/latex/customenvs/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "12mjzmj21fg2rgy1sqyd1v1bdxy47zdsmssmxmschgv6d7f4yhx4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/customenvs")
    (synopsis "Custom environments (MCQ, list with picked items, ...)")
    (description
     "The package provides some custom environments (multiple choice, list
with chosen items, ...) based on existing environments.")
    (license (list license:lppl1.3c license:cc-by-sa4.0))))

(define-public texlive-cvss
  (package
    (name "texlive-cvss")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/cvss/" "source/latex/cvss/"
                                     "tex/latex/cvss/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0hs931xcj7jai901nvfvi942x403sda7xslq14bfjls0v74lkhbd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cvss")
    (synopsis "Compute and display CVSS base scores")
    (description
     "The Common Vulnerability Scoring System (CVSS) is an open framework for
communicating the characteristics and severity of software vulnerabilities.
CVSS consists of three metric groups: Base, Temporal, and Environmental.  This
package allows the user to compute CVSS3.1 base scores and use them in
documents, i.e., it only deals with the Base score.  Temporal and
Environmental scores will be part of a future release.")
    (license license:lppl1.3c)))

(define-public texlive-cweb-old
  (package
    (name "texlive-cweb-old")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "tex/plain/cweb-old/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0vx235zpflqpnrfa9kqq7wmc1rylg5bw1r26knfzvh3w1swbp4ai"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/cweb-old")
    (synopsis "Obsolete files from CWEB")
    (description
     "This package contains parts of CWEB that are no longer useful.")
    (license license:knuth)))

(define-public texlive-dad
  (package
    (name "texlive-dad")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dad/"
                                     "fonts/afm/public/dad/"
                                     "fonts/map/dvips/dad/"
                                     "fonts/ofm/public/dad/"
                                     "fonts/ovf/public/dad/"
                                     "fonts/tfm/public/dad/"
                                     "fonts/type1/public/dad/"
                                     "tex/lualatex/dad/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0mkfhdai5m1fwj9x0rkpklvcrp3synfxl67dp6cb0bfz3w8nmis0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dad")
    (synopsis "Simple typesetting system for mixed Arabic/Latin documents")
    (description
     "This package allows simple typesetting in Arabic script, intended for
mixed Arabic/Latin script usage in situations where heavy-duty solutions are
discouraged.  The system operates with both Unicode and transliterated input,
allowing the user to choose the most appropriate approach for every
situation.")
    (license license:lppl)))

(define-public texlive-dancers
  (package
    (name "texlive-dancers")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "fonts/source/public/dancers/"
                                     "fonts/tfm/public/dancers/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0nni21f6y9gynx1lsymb3pmh6w761q21idq60fib90hvv9jjd85q"))))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/dancers")
    (synopsis "Font for Conan Doyle's @emph{The Dancing Men}")
    (description
     "The (Sherlock Holmes) book contains a code which uses dancing men as
glyphs.  The alphabet as given is not complete, lacking @samp{f}, @samp{j},
@samp{k}, @samp{q}, @samp{u}, @samp{w}, @samp{x} and @samp{z}, so those
letters in the font are not due to Conan Doyle.

The code required word endings to be marked by the dancing man representing
the last letter to be holding a flag: these are coded as A-Z.  In some cases,
the man has no arms, making it impossible for him to hold a flag.  In these
cases, he is wearing a flag on his hat in the character.  The font is
distributed as Metafont source.")
    ;; "This font may be freely used, modified and distributed."
    (license
     (license:fsf-free "file://fonts/source/public/dancers/dancers.mf"))))

(define-public texlive-dashrule
  (package
    (name "texlive-dashrule")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/dashrule/"
                                     "source/latex/dashrule/"
                                     "tex/latex/dashrule/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0glnb77dmhl9svgxpz1s8ivsbmr0cy1a0sz68vfx1plfkxfglbyl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dashrule")
    (synopsis "Draw dashed rules")
    (description
     "The @code{dashrule} package makes it easy to draw a huge variety of
dashed rules (i.e., lines) in LaTeX.  It provides a command,
@code{\\hdashrule}, which draws horizontally dashed rules using the same
syntax as @code{\\rule}, but with an additional parameter that specifies the
pattern of dash segments and the space between those segments.  Those rules
are fully compatible with every @code{LaTeX} back-end processor.")
    (license license:lppl)))

(define-public texlive-dashrulex
  (package
    (name "texlive-dashrulex")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/dashrulex/"
                              "tex/latex/dashrulex/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1isk2qyz9nmcc1p03cp94lrxnkhw5xcdaq7z0fqgcq91l3q9s18l"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dashrulex")
    (synopsis "Draw dashed rules")
    (description
     "This package package provides a flexible solution for drawing dashed
rules in the body.  It currently provides two commands, @code{\\hdashrule} and
@code{\\hanyrule}.  It can be used as an alternative to the @code{dashrule}
package.")
    (license license:cc-by4.0)))

(define-public texlive-dantelogo
  (package
    (name "texlive-dantelogo")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dantelogo/"
                                     "fonts/enc/dvips/dantelogo/"
                                     "fonts/map/dvips/dantelogo/"
                                     "fonts/opentype/public/dantelogo/"
                                     "fonts/tfm/public/dantelogo/"
                                     "fonts/type1/public/dantelogo/"
                                     "fonts/vf/public/dantelogo/"
                                     "tex/latex/dantelogo/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "19wz5qyr4bhd0dwp0y38ql87s3103yimg5b6r2mjz4mz0dsiyy1b"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dantelogo")
    (synopsis "Font for DANTE's logo")
    (description
     "The DANTE font for the logo of @url{https://www.dante.de, DANTE}, the
German speaking TeX users group.  The font includes only the five characters
@samp{d}, @samp{a}, @samp{n}, @samp{t}, and @samp{e}.  @file{dantelogo.sty}
provides an interface for LuaLaTeX, XeLaTeX, and pdfLaTeX.")
    (license license:lppl)))

(define-public texlive-dccpaper
  (package
    (name "texlive-dccpaper")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/dccpaper/"
                                     "source/latex/dccpaper/"
                                     "tex/latex/dccpaper/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fqdf56pmcfcl2zcr711qky4anl0pplc3m28fjfbbc3zbll7pm9m"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dccpaper")
    (synopsis
     "Typeset papers for the @emph{International Journal of Digital Curation}")
    (description
     "The LaTeX class @code{ijdc-v14} produces camera-ready papers and
articles suitable for inclusion in the @emph{International Journal of Digital
Curation}, with applicability from volume 14 onwards; a legacy class
@code{ijdc-v9} is provided for papers and articles written for volumes 9-13.
The similar @code{idcc} class can be used for submissions to the International
Digital Curation Conference, beginning with the 2015 conference.")
    (license (list license:lppl1.3c license:cc-by4.0))))

(define-public texlive-decision-table
  (package
    (name "texlive-decision-table")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/decision-table/"
                                     "source/latex/decision-table/"
                                     "tex/latex/decision-table/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14cgf142wmi2qlrxaa8dmyr7grb7d33ngfpg8sbrm82a78cvl1yc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/decision-table")
    (synopsis "Create Decision Model and Notation decision tables")
    (description
     "The decision-table package allows for an easy way to generate decision
tables in the Decision Model and Notation (DMN) format.  This package ensures
consistency in the tables (i.e., fontsize), and is thus a better alternative
to inserting tables via images.

The decision-table package adds the @code{\\dmntable} command, with which
tables can be created.  This command expands into a tabular, so it can be used
within a @code{table} or @code{figure} environment.  Furthermore, this allows
labels and captions to be added seamlessly.  It is also possible to place
multiple DMN tables in one table/figure environment.")
    (license license:lppl1.3c)))

(define-public texlive-defoldfonts
  (package
    (name "texlive-defoldfonts")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/defoldfonts/"
                              "source/latex/defoldfonts/"
                              "tex/latex/defoldfonts/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0arzvd8dqnybp9mph1kbpv6nbvc7bzyqprncf29g60vmzcslv80m"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/defoldfonts")
    (synopsis "Define old font commands")
    (description
     "In 1994, with LaTeX2e, the old font commands @code{\\rm}, @code{\\sf},
@code{\\tt}, @code{\\bf}, @code{\\it}, @code{\\sl}, and @code{\\sc} became
obsolete.  This package defines them, and also the deprecated KOMA-Script
command @code{\\sfb}.")
    (license license:lppl1.3c)))

(define-public texlive-dejavu-otf
  (package
    (name "texlive-dejavu-otf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dejavu-otf/"
                                     "tex/latex/dejavu-otf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "171mnygvm5984v5cpgh91vaw2fv2rh42098bvbx9i9k01348ry4d"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dejavu-otf")
    (synopsis "Support for the TTF and OTF DejaVu fonts")
    (description
     "The @code{dejavu-otf} package supports the TTF fonts from the DejaVu
project and the OpenType version of the TeXGyre Math.")
    (license license:lppl1.3+)))

(define-public texlive-delim
  (package
    (name "texlive-delim")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/delim/" "source/latex/delim/"
                                     "tex/latex/delim/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12h0zqn46yrqf079a6k939g3pv843rs417va4g2r39rhb2fvh6r8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/delim")
    (synopsis "Simplify typesetting mathematical delimiters")
    (description
     "The package permits simpler control of delimiters without excessive use
of @code{\\big} commands and the like.")
    (license license:lppl1.2+)))

(define-public texlive-delimseasy
  (package
    (name "texlive-delimseasy")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/delimseasy/"
                                     "tex/latex/delimseasy/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0afgcnwqcbw34wgfryzk1rjw82qakl6nq98gl2w1aw5lkgi4fb9p"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/delimseasy")
    (synopsis "Delimiter commands that are easy to use and resize")
    (description
     "This package provides commands to give an easy way to control the size
and blackness of delimiters: append 1-4 @samp{b}s to command for larger sizes;
prepend @samp{B} for for boldface.  These commands reduce the likelihood of
incomplete delimiter pairs and typically use fewer characters than the LaTeX
default.")
    (license license:lppl1.3+)))

(define-public texlive-delimset
  (package
    (name "texlive-delimset")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/delimset/"
                                     "source/latex/delimset/"
                                     "tex/latex/delimset/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xygc60m27x81id2mv8v9ixrwwx3zqrn5k1piz5vqiry6xjqfikp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/delimset")
    (synopsis
     "Typeset and declare sets of delimiters with convenient size control")
    (description
     "@code{delimset} is a LaTeX2e package to typeset and declare sets of
delimiters in math mode whose size can be adjusted conveniently.")
    (license license:lppl1.3+)))

(define-public texlive-denisbdoc
  (package
    (name "texlive-denisbdoc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/denisbdoc/"
                                     "source/latex/denisbdoc/"
                                     "tex/latex/denisbdoc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ypd5xl3psg11jss9n7g1qihd7rag480j63wxkvgifhi4kainypc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/denisbdoc")
    (synopsis "Personal dirty package for documenting packages")
    (description
     "This package provides a personal dirty package for documenting packages.")
    (license license:lppl1.3c)))

(define-public texlive-derivative
  (package
    (name "texlive-derivative")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/derivative/"
                                     "tex/latex/derivative/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06c1ha4w8i3f6k77cjr82cn1rl7jbk82q1n24dd3l7ha4d7qncjp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/derivative")
    (synopsis "Nice and easy derivatives")
    (description
     "Typesetting derivatives and differentials in a consistent way are clumsy
and require care to ensure the preferred formatting.  Several packages have
been developed for this purpose, each with its own features and drawbacks,
with the most ambitious one being @code{diffcoeff}.  While this package is
comparable to diffcoeff in terms of features, it takes a different approach.
One difference is this package provides more options to tweak the format of
the derivatives and differentials.  However, the automatic calculation of the
total order isn't as developed as the one in @code{diffcoeff}.  This package
makes it easy to write derivatives and differentials consistently with its
predefined commands.  It also provides a set of commands that can define
custom derivatives and differential operators.  The options follow
a consistent naming scheme making them easy to use and understand.")
    (license license:lppl1.3+)))

(define-public texlive-diadia
  (package
    (name "texlive-diadia")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/diadia/" "scripts/diadia/"
                                     "tex/latex/diadia/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1sjnjabgqsv6v45sffkkzm78fnx89vmk836dajsvck5pd5nvkzxv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "diadia.lua")))
    (home-page "https://ctan.org/pkg/diadia")
    (synopsis "Package to keep a diabetes diary")
    (description
     "The @code{diadia} package allows you to keep a diabetes diary.  Usually,
this means keeping record of certain medical values like blood sugar, blood
pressure, pulse or weight.  It might also include other medical,
pharmaceutical or nutritional data (HbA1c, insulin doses, carbohydrate units).

The @code{diadia} package supports all of this plus more --- simply by adding
more columns to the data file!  It is able to evaluate the data file and
typesets formatted tables and derived plots.  Furthermore, it supports
medication charts and info boxes.")
    (license license:lppl)))

(define-public texlive-diagbox
  (package
    (name "texlive-diagbox")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/diagbox/" "source/latex/diagbox/"
                                     "tex/latex/diagbox/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0zr7ih8jr56j727gf3l1q87rlrqn3zfz8cm7hs2q5l1f3y1gmcgh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/diagbox")
    (synopsis "Table heads with diagonal lines")
    (description
     "The package's principal command, @code{\\diagbox}, takes two
arguments (texts for the slash-separated parts of the box), and an optional
argument with which the direction the slash will go, the box dimensions, etc.,
may be controlled.  The package also provides @code{\\slashbox} and
@code{\\backslashbox} commands for compatibility with the now removed slashbox
package, which it supersedes.")
    (license license:lppl1.3+)))

(define-public texlive-dice
  (package
    (name "texlive-dice")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dice/"
                                     "fonts/source/public/dice/"
                                     "fonts/tfm/public/dice/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ky33hg66d0x8bcjmdq4ilynpb25mm82x93wzwnz11y59nv4cvsl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/dice")
    (synopsis "Font for die faces")
    (description
     "This package provides a Metafont font that can produce die faces in 2D
or with various 3D effects.")
    (license license:lppl)))

(define-public texlive-dictsym
  (package
    (name "texlive-dictsym")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dictsym/"
                                     "fonts/afm/public/dictsym/"
                                     "fonts/map/dvips/dictsym/"
                                     "fonts/map/vtex/dictsym/"
                                     "fonts/tfm/public/dictsym/"
                                     "fonts/type1/public/dictsym/"
                                     "tex/latex/dictsym/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ygnnmg2m57wzj1iz2wznvdbsi8323804s74a2ic7q1akpy5n2ac"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dictsym")
    (synopsis "DictSym font and macro package")
    (description
     "This directory contains the DictSym Type1 font designed by Georg
Verweyen and all files required to use it with LaTeX.  The font provides
a number of symbols commonly used in dictionaries.  The accompanying macro
package makes the symbols accessible as LaTeX commands.")
    (license license:lppl)))

(define-public texlive-didactic
  (package
    (name "texlive-didactic")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/didactic/"
                              "source/latex/didactic/"
                              "tex/latex/didactic/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0li1f0lfzmhf6ac8vsszsyxl2ch7j6g9zk2b1vmnsilzhwlkaw6y"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/didactic")
    (synopsis "Tools for writing teaching material")
    (description
     "This package provides macros and environments useful for writing
teaching material.  It provides more semantic environments on top of the
standard definition, theorem, and friends: for instance, exercise, activity
and question.  These are suitably color-coded when used with Beamer.  They
occur as normal text in handouts produced by @code{beamerarticle} (same style
as definition usually has).  It also provides macros for typesetting code
listings and output side by side.  Finally, it modifies the appearance of
Beamer (Berlin-based theme) and Memoir (Tufte style layout), if loaded.  It is
designed to be used with Beamer to produce slides and @code{beamerarticle}
with @code{memoir} to produce notes and handouts from the same source.")
    (license license:lppl1.3c)))

(define-public texlive-didec
  (package
    (name "texlive-didec")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/didec/"
                              "tex/latex/didec/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0c3z7bn812gn5mx2mfk3lkrn8v1j2kkmm2ir2s27kk8ajx0q7i49"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/didec")
    (synopsis "Fixed-point arithmetic with two decimal places")
    (description
     "This package supports fixed-point arithmetic with two decimal places
(di-decimal) which is typical for financial transactions in many currencies.
The intended use case is (personal) bookkeeping.")
    (license license:lppl1.3+)))

(define-public texlive-din1505
  (package
    (name "texlive-din1505")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/din1505/" "doc/latex/din1505/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ggi58kra06k8r4drkhnlap9khvscpji78j8v92s3gzh8qmsjhp4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/din1505")
    (synopsis "Bibliography styles for German texts")
    (description
     "This package provides a set of bibliography styles that conform to DIN
1505, and match the original BibTeX standard set (@code{plain}, @code{unsrt},
@code{alpha} and @code{abbrv}), together with a style @code{natdin} to work with
@code{natbib}.")
    (license license:knuth)))

(define-public texlive-dingbat
  (package
    (name "texlive-dingbat")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dingbat/"
                                     "fonts/source/public/dingbat/"
                                     "fonts/tfm/public/dingbat/"
                                     "source/latex/dingbat/" "tex/latex/dingbat/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12vhykmz1xzvrdzp61qy0q69fnxjjvc9m8v8kras92v49f6m4z7a"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/dingbat")
    (synopsis "Two dingbat symbol fonts")
    (description
     "The package provides the fonts (@code{ark10} and @code{dingbat}),
specified in Metafont; support macros are also provided for use in LaTeX.  An
Adobe Type 1 version of the fonts is available in the @code{niceframe} fonts
bundle.")
    (license license:lppl)))

(define-public texlive-diffcoeff
  (package
    (name "texlive-diffcoeff")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/diffcoeff/"
                                     "tex/latex/diffcoeff/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "01x16jh9d69pa085ib0di6x3vbgmz722byw73d0kjj8fpr4z4pnj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/diffcoeff")
    (synopsis "Write differential coefficients easily and consistently")
    (description
     "This package allows the easy and consistent writing of ordinary, partial
and other derivatives of arbitrary (algebraic or numeric) order.  For mixed
partial derivatives, the total order of differentiation is calculated by the
package.  Optional arguments allow specification of points of
evaluation (ordinary derivatives), or variables held constant (partial
derivatives), and the placement of the differentiand (numerator or appended).
The package is built on xtemplate and the configurability it enables,
extending to differentials (including simple line elements) and jacobians.")
    (license license:lppl1.3c)))

(define-public texlive-digiconfigs
  (package
    (name "texlive-digiconfigs")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/digiconfigs/"
                                     "tex/latex/digiconfigs/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13682as94lpy3qws03ymgz380fkkihwppzdwijig85j1yq098wqg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/digiconfigs")
    (synopsis "Writing ``configurations''")
    (description
     "In Stochastic Geometry and Digital Image Analysis some problems can be
solved in terms of so-called ``configurations''.  A configuration is basically
a square matrix of @code{\\circ} and @code{\\bullet} symbols.  This package
provides a convenient and compact mechanism for displaying these
configurations.")
    (license license:lppl)))

(define-public texlive-dijkstra
  (package
    (name "texlive-dijkstra")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/dijkstra/" "tex/latex/dijkstra/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xjq51x4g1xngp8npicgnpmbkzk0rnj992pmhizjv0j92zmaikqm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dijkstra")
    (synopsis "Dijkstra algorithm for LaTeX")
    (description
     "This small package uses the Dijkstra algorithm for weighted
graphs,directed or not: the search table of the shortest path can be
displayed, the minimum distance between two vertices and the corresponding
path are stored in macros.")
    (license license:lppl1.3c)))

(define-public texlive-dinat
  (package
    (name "texlive-dinat")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/dinat/" "doc/bibtex/dinat/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05ab7aq8h08a8pcl3si2lki569sf201bygf3ra95dpqhs1ijxaag"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dinat")
    (synopsis "Bibliography style for German texts")
    (description
     "This package provides bibliography style files intended for texts in
german.  They draw up bibliographies in accordance with the german DIN 1505,
parts 2 and 3.")
    (license license:public-domain)))

(define-public texlive-dirtree
  (package
    (name "texlive-dirtree")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/dirtree/"
                                     "source/generic/dirtree/"
                                     "tex/generic/dirtree/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1righlip75ry0yc6kbpijq423k4m625byzgb55d32s7c0imvj41v"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dirtree")
    (synopsis "Display trees in the style of Windows Explorer")
    (description
     "This package is designed to emulate the way Windows Explorer displays
directory and file trees, with the root at top left, and each level of subtree
displaying one step in to the right.  The macros work equally well with Plain
TeX and with LaTeX.")
    (license license:lppl)))

(define-public texlive-disser
  (package
    (name "texlive-disser")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/disser/" "makeindex/disser/"
                                     "source/latex/disser/" "tex/latex/disser/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0sxvj4cka9xqzl2s3c465fm19lc1b8hyar1chjb51y42q4mx2bmg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/disser")
    (synopsis "Class and templates for typesetting dissertations in Russian")
    (description
     "Disser comprises a document class and set of templates for typesetting
dissertations in Russian.  One of its primary advantages is a simplicity of
format specification for titlepage, headers and elements of automatically
generated lists (table of contents, list of figures, etc).  Bibliography
styles, that conform to the requirements of the Russian standard GOST
R 7.0.11-2011, are provided.")
    (license license:lppl1.3+)))

(define-public texlive-dithesis
  (package
    (name "texlive-dithesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/dithesis/" "tex/latex/dithesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1yz53jabca56k65pddhl68d3384k2f2jn4zrfkg2d4x8aa1dnvvx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dithesis")
    (synopsis "Class for undergraduate theses at the University of Athens")
    (description
     "The class conforms to the requirements of the Department of Informatics
and Telecommunications at the University of Athens regarding the preparation
of undergraduate theses, as of Sep 1, 2011.  The class is designed for use
with XeLaTeX.")
    (license license:lppl)))

(define-public texlive-dk-bib
  (package
    (name "texlive-dk-bib")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bib/dk-bib/"
                                     "bibtex/bst/dk-bib/"
                                     "bibtex/csf/dk-bib/"
                                     "doc/latex/dk-bib/"
                                     "source/latex/dk-bib/"
                                     "tex/latex/dk-bib/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1bwmaja41mivsrcx3j9p0yplqccw62fd9wn6v1yx138kg0ayib1n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dk-bib")
    (synopsis "Danish variants of standard BibTeX styles")
    (description
     "Dk-bib is a translation of the four standard BibTeX style files (@code{abbrv},
@code{alpha}, @code{plain} and @code{unsrt}) and the @code{apalike} style file
into Danish.  The files have been extended with URL, ISBN, ISSN, annote and
printing fields which can be enabled through a LaTeX style file.  Dk-bib also
comes with a couple of Danish sorting order files for BibTeX8.")
    (license (list license:knuth license:gpl2+))))

(define-public texlive-dnp
  (package
    (name "texlive-dnp")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "fonts/sfd/dnp/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jlvb0nps1ij4sgbg3clgbk34p80la1fhh9zihn9fhl9nrqk637r"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dnp")
    (synopsis "Subfont numbers for DNP font encoding")
    (description "This package provides subfont numbers for DNP font encoding.")
    ;; This is part of the CJK package, use the same GPL license.
    (license license:gpl2)))

(define-public texlive-docbytex
  (package
    (name "texlive-docbytex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/docbytex/"
                                     "tex/generic/docbytex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09x4xpyq89jjjsp4yzrifcazz0p2f5w7785g8pvss8v0wwsb0lav"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/docbytex")
    (synopsis "Creating documentation from source code")
    (description
     "The package creates documentation from C source code, or other
programming languages.")
    ;; README states: "You can do anything with the files from DocBy.TeX
    ;; package without any limit".
    (license (license:fsf-free "file://doc/generic/docbytex/README"))))

(define-public texlive-doipubmed
  (package
    (name "texlive-doipubmed")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/doipubmed/"
                                     "source/latex/doipubmed/"
                                     "tex/latex/doipubmed/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1432hh1pr5r6izfcqlbvl1lxpradidrwpfkvjr25ds1rlh240y66"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/doipubmed")
    (synopsis "Special commands for use in bibliographies")
    (description
     "The package provides the commands @code{\\doi}, @code{\\pubmed} and
@code{\\citeurl}.  These commands are primarily designed for use in
bibliographies.  A @code{LaTeX2HTML} style file is also provided.")
    (license license:lppl)))

(define-public texlive-domaincoloring
  (package
    (name "texlive-domaincoloring")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list
                         "doc/lualatex/domaincoloring/"
                         "tex/lualatex/domaincoloring/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0k69ri56hbrlmkwmzq95agmvkzn7gqg1wzrrninq9ac3aykj3ji7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/domaincoloring")
    (synopsis "Draw colored representations of complex functions")
    (description
     "Domain coloring is a technique to visualize complex functions by
assigning a color to each point of the complex plane.  This package calculates
with the help of Lua any complex function to visualize its behavior.  The
value of the complex function can be described by radius and angle which can
be two values of the HSV-color model, which then defines the color of each
point in the complex plane.")
    (license license:lppl1.3+)))

(define-public texlive-domitian
  (package
    (name "texlive-domitian")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/domitian/"
                                     "fonts/enc/dvips/domitian/"
                                     "fonts/map/dvips/domitian/"
                                     "fonts/opentype/public/domitian/"
                                     "fonts/tfm/public/domitian/"
                                     "fonts/type1/public/domitian/"
                                     "fonts/vf/public/domitian/"
                                     "tex/latex/domitian/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10pg90kdwklz2rsbcafhcld4zdchmxzr6cszrajwrky8hfz67dkg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/domitian")
    (synopsis "Drop-in replacement for Palatino")
    (description
     "The Domitian fonts are an OpenType font family, based on the Palatino
design by Hermann Zapf (1918-2015), as implemented in Palladio.  Domitian is
meant as a drop-in replacement for Adobe's version of Palatino.  It extends
Palladio with small capitals, old-style figures and scientific inferiors.  The
metrics have been adjusted to more closely match Adobe Palatino, and hinting
has been improved.")
    ;; Use any of the three licenses.
    (license (list license:lppl1.3c license:silofl1.1 license:agpl3))))

(define-public texlive-dosepsbin
  (package
    (name "texlive-dosepsbin")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/man/man1/dosepsbin.1"
                                     "doc/man/man1/dosepsbin.man1.pdf"
                                     "doc/support/dosepsbin/"
                                     "scripts/dosepsbin/"
                                     "source/support/dosepsbin/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "117crwcdpirvvm9srrzjn40marg00q16rzyqipm3xxbmwv52i93c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:link-scripts #~(list "dosepsbin.pl")))
    (inputs (list perl))
    (home-page "https://ctan.org/pkg/dosepsbin")
    (synopsis "Deal with DOS binary EPS files")
    (description
     "This package provides a Encapsulated PostScript (EPS) file may given in
a special binary format to support the inclusion of a thumbnail.  This file
format, commonly known as DOS EPS format, starts with a binary header that
contains the positions of the possible sections: PostScript (PS); Windows
Metafile Format (WMF); and Tag Image File Format (TIFF).  The PS section must
be present and either the WMF file or the TIFF file should be given.  The
package provides a Perl program that will extract any of the sections of such
a file, in particular providing a text'-form EPS file for use with (La)TeX.")
    (license license:artistic2.0)))

(define-public texlive-doublestroke
  (package
    (name "texlive-doublestroke")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/doublestroke/"
                                     "fonts/map/dvips/doublestroke/"
                                     "fonts/source/public/doublestroke/"
                                     "fonts/tfm/public/doublestroke/"
                                     "fonts/type1/public/doublestroke/"
                                     "tex/latex/doublestroke/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0v9g025l0qfw4zrjkm9yypcsramwl2di997jgnznxpxms0v6ib7c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/doublestroke")
    (synopsis "Typeset mathematical double stroke symbols")
    (description
     "This package provides a font based on Computer Modern Roman useful for
typesetting the mathematical symbols for the natural numbers, whole numbers,
rational numbers, real numbers and complex numbers; coverage includes all
Roman capital letters, @samp{1}, @samp{h} and @samp{k}.  The font is available
both as Metafont source and in Adobe Type 1 format, and LaTeX macros for its
use are provided.")
    (license license:knuth)))

(define-public texlive-dowith
  (package
    (name "texlive-dowith")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/dowith/"
                                     "source/generic/dowith/"
                                     "tex/generic/dowith/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1qlrcjac1qszgrap5v4g0d27g9k5ajqgyy493hmc9yh5wa9wmkmp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dowith")
    (synopsis "Apply a command to a list of items")
    (description
     "The package provides macros for applying a command to all elements of
a list without separators, and also for extending and reducing macros storing
such lists.  Applications in mind belonged to LaTeX, but the package should
work with other formats as well.")
    (license license:lppl1.3+)))

(define-public texlive-dozenal
  (package
    (name "texlive-dozenal")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dozenal/"
                                     "fonts/afm/public/dozenal/"
                                     "fonts/map/dvips/dozenal/"
                                     "fonts/source/public/dozenal/"
                                     "fonts/tfm/public/dozenal/"
                                     "fonts/type1/public/dozenal/"
                                     "source/fonts/dozenal/"
                                     "tex/latex/dozenal/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0fhdcnpjpfxaqshddw6vxy0f8c1c3yn8p11aab8fkhnfg3pnllck"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/dozenal")
    (synopsis "Typeset documents using base twelve numbering")
    (description
     "The package supports typesetting documents whose counters are
represented in base twelve, also called @dfn{dozenal}.  It includes a macro
for converting positive whole numbers to dozenal from decimal (base ten)
representation.  The package also includes a few other macros and redefines
all the standard counters to produce dozenal output.  Fonts, in Roman, italic,
slanted, and boldface versions, provide ten and eleven.  The fonts were
designed to blend well with the Computer Modern fonts, and are available both
as Metafont source and in Adobe Type 1 format.")
    (license license:lppl1.3+)))

(define-public texlive-dramatist
  (package
    (name "texlive-dramatist")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/dramatist/"
                                     "source/latex/dramatist/"
                                     "tex/latex/dramatist/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17x10jaa7f9rnqfrnvgw225n5ypqrmacbpsfpl5zdhx9abbjh793"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dramatist")
    (synopsis "Typeset dramas, both in verse and in prose")
    (description
     "This package is intended for typesetting drama of any length.  It
provides two environments for typesetting dialogues in prose or in verse; new
document divisions corresponding to acts and scenes; macros that control the
appearance of characters and stage directions; and automatic generation of
a ``dramatis personae'' list.")
    (license license:gpl3+)))

(define-public texlive-drawing-with-metapost
  (package
    (name "texlive-drawing-with-metapost")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/metapost/drawing-with-metapost/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1w6icqqqx0h5byzmfwjdra523khj9sfgbxwxa2jpy6xy5n03gdlj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/drawing-with-metapost")
    (synopsis "How to draw technical diagrams with MetaPost")
    (description
     "This project provides a document that discusses how to draw technical
diagrams with MetaPost language.  It includes over 200 illustrations created
with MetaPost, complete with source code as inspiration and examples.")
    (license license:opl1.0+)))

(define-public texlive-drawmatrix
  (package
    (name "texlive-drawmatrix")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/drawmatrix/"
                                     "source/latex/drawmatrix/"
                                     "tex/latex/drawmatrix/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14y9xah06ya8krg7ckmhbmxs113g1vw0x2ryldapww7qi08i78yx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/drawmatrix")
    (synopsis "Draw visual representations of matrices in LaTeX")
    (description
     "The package provides macros to visually represent matrices.  Various
options allow changing the visualizations, e.g., drawing rectangular,
triangular, or banded matrices.")
    (license license:expat)))

(define-public texlive-drawstack
  (package
    (name "texlive-drawstack")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/drawstack/"
                                     "tex/latex/drawstack/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1qv2j7crg4b1ggxvmjb6zf0cxmr4mmpk382l7w3dqhywf9lr722v"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/drawstack")
    (synopsis "Draw execution stacks")
    (description
     "This simple LaTeX package provides support for drawing execution stack
(typically to illustrate assembly language notions).  The code is written on
top of TikZ.")
    (license license:lppl1.3+)))

(define-public texlive-droit-fr
  (package
    (name "texlive-droit-fr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/droit-fr/" "tex/latex/droit-fr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0lx4k3w7mzldhwykpfvhv05v7fhd1jibra80kz0zf9s0jga8sq4a"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/droit-fr")
    (synopsis "Document class and bibliographic style for French law")
    (description
     "The bundle provides a toolkit intended for students writing a thesis in
French law.  It features a LaTeX document class, a bibliographic style for
BibLaTeX package, a practical example of french thesis document, and
documentation.  The class assumes use of Biber and BibLaTeX.")
    (license license:lppl1.3+)))

(define-public texlive-drm
  (package
    (name "texlive-drm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/drm/"
                                     "fonts/afm/public/drm/"
                                     "fonts/map/dvips/drm/"
                                     "fonts/opentype/public/drm/"
                                     "fonts/source/public/drm/"
                                     "fonts/tfm/public/drm/"
                                     "fonts/type1/public/drm/"
                                     "source/fonts/drm/"
                                     "tex/latex/drm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0fxmwakgy7inin85pjad5rdqg11flrw8mlsh86ga6ghqcw69zhwq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/drm")
    (synopsis "Complete family of fonts written in Metafont")
    (description
     "The package provides access to the DRM (Don's Revised Modern) family of
fonts, which includes a variety of optical sizes in Roman (in four weights),
italic, and small caps, among other shapes, along with a set of symbols and
ornaments.  It is intended to be a full-body text font, but its larger sizes
can also be used for simple display purposes, and its significant body of
symbols can stand on its own.  It comes complete with textual (old-style) and
lining figures, and even has small-caps figures.  It also comes with
extensible decorative rules to be used with ornaments from itself or other
fonts, along with an extremely flexible ellipsis package.")
    (license license:lppl1.3+)))

(define-public texlive-droid
  (package
    (name "texlive-droid")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/droid/"
                                     "fonts/enc/dvips/droid/"
                                     "fonts/map/dvips/droid/"
                                     "fonts/tfm/ascender/droid/droidsans/"
                                     "fonts/tfm/ascender/droid/droidsansmono/"
                                     "fonts/tfm/ascender/droid/droidserif/"
                                     "fonts/truetype/ascender/droid/droidsans/"
                                     "fonts/truetype/ascender/droid/droidsansmono/"
                                     "fonts/truetype/ascender/droid/droidserif/"
                                     "fonts/type1/ascender/droid/droidsans/"
                                     "fonts/type1/ascender/droid/droidsansmono/"
                                     "fonts/type1/ascender/droid/droidserif/"
                                     "fonts/vf/ascender/droid/droidsans/"
                                     "fonts/vf/ascender/droid/droidsansmono/"
                                     "fonts/vf/ascender/droid/droidserif/"
                                     "tex/latex/droid/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10lgw5yv0s48rwydycq6lxj5dfyna7iflnhz1zq42akg21n595vx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/droid")
    (synopsis "LaTeX support for the Droid font families")
    (description
     "The Droid typeface family was designed by Steve Matteson.  The Droid
family consists of Droid Serif, Droid Sans and Droid Sans Mono fonts.  The
bundle includes the fonts in both TrueType and Adobe Type 1 formats.")
    (license (list license:lppl1.3c license:asl2.0))))

(define-public texlive-dsptricks
  (package
    (name "texlive-dsptricks")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/dsptricks/"
                                     "tex/latex/dsptricks/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0pjc78c195nkj493qyc0dxbr9d1488dc528vzkpgcgrf0g5pxm96"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dsptricks")
    (synopsis "Macros for Digital Signal Processing (DSP) plots")
    (description
     "The package provides a set of @code{LaTeX} macros (based on PSTricks)
for plotting the kind of graphs and figures that are usually employed in
digital signal processing publications.  DSPTricks provides facilities for
standard discrete-time lollipop plots, continuous-time and frequency plots,
and pole-zero plots.  The companion package
DSPFunctions (@file{dspfunctions.sty}) provides macros for computing frequency
responses and DFTs, while the package DSPBlocks (@file{dspblocks.sty})
supports DSP block diagrams.")
    (license license:lppl)))

(define-public texlive-dsserif
  (package
    (name "texlive-dsserif")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dsserif/"
                                     "fonts/afm/public/dsserif/"
                                     "fonts/map/dvips/dsserif/"
                                     "fonts/tfm/public/dsserif/"
                                     "fonts/type1/public/dsserif/"
                                     "source/fonts/dsserif/"
                                     "tex/latex/dsserif/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0dl8380lyv8r5kfrlhi5r4iwadb52m1397z68qwa65c5p2px1y3i"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dsserif")
    (synopsis "Double-struck serifed font for mathematical use")
    (description
     "DSSerif is a mathematical font package with double struck serifed
digits, upper and lower case letters, in regular and bold weights.  The design
was inspired by the STIX double struck fonts, which are sans serif, but
starting from a Courier-like base.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-duerer
  (package
    (name "texlive-duerer")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/duerer/"
                                     "fonts/source/public/duerer/"
                                     "fonts/tfm/public/duerer/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1x5s9qnmhhpj54l6cd0pzd9gmd0ipr95j60rcs73c66wwfjw0xki"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/duerer")
    (synopsis "Computer Duerer fonts")
    (description
     "These fonts are designed for titling use, and consist of capital roman
letters only.  Together with the normal set of base shapes, the family also
offers an informal shape.  The distribution is as Metafont source.  LaTeX
support is available in the @code{duerer-latex} bundle.")
    (license license:public-domain)))

(define-public texlive-duerer-latex
  (package
    (name "texlive-duerer-latex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/duerer-latex/"
                                     "tex/latex/duerer-latex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17ni6hrcblkbzn1f8cn2mvrc01sjqmi8qi2dxy82z14llr23qmbh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/duerer-latex")
    (synopsis "LaTeX support for the Duerer fonts")
    (description
     "This package provides LaTeX support for Hoenig's Computer Duerer fonts,
using their standard fontname names.")
    (license license:gpl3+)))

(define-public texlive-dutchcal
  (package
    (name "texlive-dutchcal")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/dutchcal/"
                                     "fonts/afm/public/dutchcal/"
                                     "fonts/map/dvips/dutchcal/"
                                     "fonts/tfm/public/dutchcal/"
                                     "fonts/type1/public/dutchcal/"
                                     "fonts/vf/public/dutchcal/"
                                     "tex/latex/dutchcal/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "173c5k6q35ljbqdi00kplxss0n1aiss8nvigdv33mlkqpg7i74qg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dutchcal")
    (synopsis "Reworking of ESSTIX13, adding a bold version")
    (description
     "This package reworks the mathematical calligraphic font ESSTIX13,
adding a bold version.  LaTeX support files are included.")
    (license license:lppl)))

(define-public texlive-dvgloss
  (package
    (name "texlive-dvgloss")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/dvgloss/" "source/latex/dvgloss/"
                                     "tex/latex/dvgloss/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1fs5ghbyl3qhg1v0iljjqn723a93m4l7f873h3h81pcga75mng4s"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dvgloss")
    (synopsis "Facilities for setting interlinear glossed text")
    (description
     "The package provides extensible macros for setting interlinear glossed
text --- useful, for instance, for typing linguistics papers.")
    (license license:lppl)))

(define-public texlive-dyntree
  (package
    (name "texlive-dyntree")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/dyntree/" "source/latex/dyntree/"
                                     "tex/latex/dyntree/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1j891cswgcyv5b4l6an0sa6vfwkn6lxswx55hsv2sdkkfbpb7dnd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/dyntree")
    (synopsis "Construct Dynkin tree diagrams")
    (description
     "The package is intended for users needing to typeset a Dynkin Tree
Diagram---a group theoretical construct consisting of cartan coefficients in
boxes connected by a series of lines.  Such a diagram is a tool for working
out the states and their weights in terms of the fundamental weights and the
simple roots.")
    ;; Package mentions LGPL without any reference to the version.  Assuming
    ;; LGPL 2.1+.
    (license license:lgpl2.1+)))

(define-public texlive-e-french
  (package
    (name "texlive-e-french")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/e-french/"
                                     "makeindex/e-french/"
                                     "tex/generic/e-french/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14qxxfjlxygprs1ndhnn9mhz9hkb4j2dqikni5cvva457bivm4l9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/e-french")
    (synopsis "Comprehensive LaTeX support for French-language typesetting")
    (description
     "E-french is a distribution that keeps alive the work of Bernard
Gaulle (now deceased), under a free licence.  It replaces the old full
@code{frenchpro} (the professional distribution) and the light-weight
@code{frenchle} packages.")
    (license license:lppl1.3+)))

(define-public texlive-ean
  (package
    (name "texlive-ean")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/ean/" "tex/generic/ean/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1zyxjpc0ggas43lpvl1l1mknqmd94q0cqgw6w2by29w3r8wafjh9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ean")
    (synopsis "Macros for making EAN barcodes")
    (description "This package provides EAN-8 and EAN-13 forms.")
    (license license:gpl3+)))

(define-public texlive-easing
  (package
    (name "texlive-easing")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/easing/"
                                     "source/latex/easing/"
                                     "tex/latex/easing/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1j6y1i6cz7pcgrd6pisk6f11r30lrr4cnbhhdshl48wp97byjnw4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/easing")
    (synopsis "Easing functions for @code{pgfmath}")
    (description
     "This library implements a collection of easing functions and adds them
to the PGF mathematical engine.")
    (license license:lppl1.3+)))

(define-public texlive-ebgaramond
  (package
    (name "texlive-ebgaramond")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/ebgaramond/"
                                     "fonts/enc/dvips/ebgaramond/"
                                     "fonts/map/dvips/ebgaramond/"
                                     "fonts/opentype/public/ebgaramond/"
                                     "fonts/tfm/public/ebgaramond/"
                                     "fonts/type1/public/ebgaramond/"
                                     "fonts/vf/public/ebgaramond/"
                                     "tex/latex/ebgaramond/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1k7hfxl8i3b2wcpwv5dgnxsa0n6mdwgpyaygklkgqd46095x4xwq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ebgaramond")
    (synopsis "LaTeX support for EBGaramond fonts")
    (description
     "EB Garamond is a revival by Georg Duffner of the 16th century fonts
designed by Claude Garamond.  The LaTeX support package works for (pdf)LaTeX,
XeLaTeX and LuaLaTeX users; configuration files for use with @code{microtype}
are provided.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-ebgaramond-maths
  (package
    (name "texlive-ebgaramond-maths")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/ebgaramond-maths/"
                                     "fonts/enc/dvips/ebgaramond-maths/"
                                     "fonts/map/dvips/ebgaramond-maths/"
                                     "fonts/tfm/public/ebgaramond-maths/"
                                     "source/fonts/ebgaramond-maths/"
                                     "tex/latex/ebgaramond-maths/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0zqvd17m33czb3dni35iw04v0vzclw5zwgbssygclzh4vhmnb0qn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ebgaramond-maths")
    (synopsis "LaTeX support for EBGaramond fonts in mathematics")
    (description
     "This package provides some LaTeX support for the use of EBGaramond12 in
mathematics.")
    (license license:lppl1.3+)))

(define-public texlive-ebook
  (package
    (name "texlive-ebook")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ebook/" "tex/latex/ebook/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1i9zixl8wiwmfrhrzwm2adgwm5kzqk5xaq5nq9bks9i2ayvmiij2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ebook")
    (synopsis "Helps creating an ebook by providing an @code{ebook} class")
    (description
     "The package defines a command @code{\\ebook} that defines page layout,
fonts, and font-sizes for documents to be rendered as PDF-ebooks on small
ebook-readers.")
    (license license:public-domain)))

(define-public texlive-ebsthesis
  (package
    (name "texlive-ebsthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ebsthesis/"
                                     "source/latex/ebsthesis/"
                                     "tex/latex/ebsthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0vmnkjnpg840rdgbim7gz2frzgr9dycmzqyj9kylicz5nvc4nk5x"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ebsthesis")
    (synopsis "Typesetting theses for economics")
    (description
     "The @code{ebsthesis} class and @code{ebstools} package facilitate the
production of camera-ready manuscripts in conformance with the guidelines of
Gabler Verlag and typographical rules established by the European Business
School.")
    (license license:lppl)))

(define-public texlive-ecc
  (package
    (name "texlive-ecc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/ecc/" "fonts/source/public/ecc/"
                                     "fonts/tfm/public/ecc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06mznqh1k9ff8rpifxkphv3yk9ym6zcvz3x1ksk677n6fp5a5jkh"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-ec")))
    (build-system texlive-build-system)
    (native-inputs (list texlive-ec texlive-metafont))
    (home-page "https://ctan.org/pkg/ecc")
    (synopsis "Sources for the European Concrete fonts")
    (description
     "The Metafont sources and TFM files of the European Concrete Fonts.
This is the T1-encoded extension of Knuth's Concrete fonts, including also the
corresponding text companion fonts.  Adobe Type 1 versions of the fonts are
available as part of the @code{cm-super} font bundle.")
    (license license:lppl)))

(define-public texlive-ecltree
  (package
    (name "texlive-ecltree")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ecltree/" "tex/latex/ecltree/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ki5zzpl8dkz7zwyccmsxfnfq7vl5vlx3gi1py64mgqxfxp0rivw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ecltree")
    (synopsis "Trees using Epic and Eepic macros")
    (description
     "The package recursively draws trees: each subtree is defined in
a @code{bundle} environment, with a set of leaves described by @code{\\chunk}
macros.  A chunk may have a @code{bundle} environment inside it.")
    (license license:lppl)))

(define-public texlive-eco
  (package
    (name "texlive-eco")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/eco/" "fonts/tfm/public/eco/"
                                     "fonts/vf/public/eco/" "source/fonts/eco/"
                                     "tex/latex/eco/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0b2g3bmldad4vfx9qd3sakhayl9knijwn26xwysxrzc3rnfnz5w2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eco")
    (synopsis "Old style numerals using EC fonts")
    (description
     "This package provides a set of font metric files and virtual fonts for
using the EC fonts with old-style numerals.  The style file @file{eco.sty} is
sufficient to use the @code{eco} fonts but if you intend to use other font
families as well, e.g., PostScript fonts, try @code{altfont}.")
    (license license:gpl3+)))

(define-public texlive-ecobiblatex
  (package
    (name "texlive-ecobiblatex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ecobiblatex/"
                                     "tex/latex/ecobiblatex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09569x2dx1h6a63ji44ipkx0nvapc9mgkgrqvc552x62c97zcf21"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ecobiblatex")
    (synopsis
     "Global Ecology and Biogeography BibLaTeX styles for the Biber backend")
    (description
     "This bundle provides a set of styles for creating bibliographies using
BibLaTeX in the style of the @emph{Global Ecology and Biogeography} journal.")
    (license license:lppl1.3+)))

(define-public texlive-econ-bst
  (package
    (name "texlive-econ-bst")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/econ-bst/"
                                     "doc/bibtex/econ-bst/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1zs07qznvz8hwlinacaas6lpyc6l2zmpcc1m3xxdkszl95xfc3md"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/econ-bst")
    (synopsis "BibTeX style for economics papers")
    (description
     "This is a BibTeX style file for papers in economics.  It provides the
following features: author-year type citation reference style used in
economics papers highly customizable use of certified random order, as
proposed by Ray Robson (2018)")
    (license license:lppl1.3+)))

(define-public texlive-econometrics
  (package
    (name "texlive-econometrics")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/econometrics/"
                                     "tex/latex/econometrics/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09sjgdsa93pfz7iha335z4xdh80939iqrwr3jxdmdxjyracz61vs"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/econometrics")
    (synopsis
     "Simplified mathematic notation in economic and econometric writing")
    (description
     "Econometrics is a package that defines some commands that simplify
mathematic notation in economic and econometrics writing.  The commands are
related to the notation of vectors, matrices, sets, calligraphic and roman
letters statistical distributions constants and symbols matrix operators and
statistical operators.")
    (license license:lppl1.3+)))

(define-public texlive-economic
  (package
    (name "texlive-economic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/economic/"
                                     "doc/bibtex/economic/" "tex/latex/economic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1b530sy5bf7f6xa4bakm5agnzx0dckafxkqsy9gv2rk2803qsf6c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/economic")
    (synopsis "BibTeX support for submitting to economics journals")
    (description
     "The bundle offers macros and BibTeX styles for the @emph{American
Economic Review} (AER), the @emph{American Journal of Agricultural
Economics} (AJAE), the @emph{Canadian Journal of Economics} (CJE), the
@emph{European Review of Agricultural Economics} (ERAE), the
@emph{International Economic Review} (IER) and @emph{Economica}.

The macro sets are based on (and require) the @code{harvard} package, and all
provide variations of author-date styles of presentation.")
    (license license:lppl)))

(define-public texlive-ecothesis
  (package
    (name "texlive-ecothesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ecothesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1avnq2y5l2bc2lkf2g16p2vh77dabv52jr6jrm4jfm1wdapc6fmq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ecothesis")
    (synopsis "LaTeX thesis template for the Universidade Federal de Vicosa")
    (description
     "The package provides a LaTeX thesis template for the @acronym{UFV,
Universidade Federal de Vicosa}, Brazil.")
    (license license:lppl1.3c)))

(define-public texlive-eczar
  (package
    (name "texlive-eczar")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/eczar/"
                                     "fonts/opentype/public/eczar/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1r7alay4g12gn7sr6d584y04qzi77qiialmsq1wfxay7a2h8c3rn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eczar")
    (synopsis "Font family supporting Devanagari and Latin script")
    (description
     "Eczar is a type family designed by Vaibhav Singh.  The fonts support
over 45+3 languages in Latin and Devanagari scripts in 5 weights.")
    (license license:silofl1.1)))

(define-public texlive-edfnotes
  (package
    (name "texlive-edfnotes")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/edfnotes/"
                                     "source/latex/edfnotes/"
                                     "tex/latex/edfnotes/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "074db5fanasjzk7clj0l4ka3x1qpmvczwxix0l2v6sjjmyb4xfz4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/edfnotes")
    (synopsis "Critical annotations to footnotes with @code{ednotes}")
    (description
     "The package modifies the annotation commands and label-test mechanism of
the ednotes package so that critical notes appear on the pages and in the
order that one would expect.")
    (license license:lppl1.3+)))

(define-public texlive-edmac
  (package
    (name "texlive-edmac")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/edmac/"
                                     "tex/generic/edmac/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03jqv8ar7m6l1ywp86piwm2cvxj9yrq4mncwkz3qwlky7lykqvlk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/edmac")
    (synopsis "Typeset critical editions")
    (description
     "This is the type example package for typesetting scholarly critical
editions.")
    (license license:gpl2)))

(define-public texlive-edmaths
  (package
    (name "texlive-edmaths")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/edmaths/"
                              "tex/latex/edmaths/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "19ymw7377h968ggc1kcbs5kgk17953s3h2kbl4jys1pg5s6pp11z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/edmaths")
    (synopsis "Report and thesis class file for University of Edinburgh")
    (description
     "This package provides a report and thesis class file for The School of
Mathematics, Scotland at the University of Edinburgh (UoE).  When working on
a report or thesis, an easy way to implement the University's typesetting
rules in LaTeX is provided by @file{edmaths.sty}.  It sets the page margins as
required and defines commands to create the correct cover page and standard
declaration.")
    (license license:lppl1.3c)))

(define-public texlive-egameps
  (package
    (name "texlive-egameps")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/egameps/" "tex/latex/egameps/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wlki6y54czvvq7cvs7pvsvl1fhd8laaj5j52jv1v8w1msh5mlpr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/egameps")
    (synopsis "LaTeX package for typesetting extensive games")
    (description
     "The style is intended to have enough features to draw any extensive game
with relative ease.  The facilities of PSTricks are used for graphics.")
    (license license:lppl)))

(define-public texlive-eiad
  (package
    (name "texlive-eiad")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/eiad/"
                                     "fonts/source/public/eiad/"
                                     "fonts/tfm/public/eiad/" "tex/latex/eiad/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0il058v1x79w5faxyalysdmfsgbxc3a1nq8kraffawpqw6wjcrbj"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-sauter")))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont texlive-sauter))
    (home-page "https://ctan.org/pkg/eiad")
    (synopsis "Traditional style Irish fonts")
    (description
     "This package provides traditional style Irish fonts, in both lower and
upper case 32 letters are defined (18 plain ones, 5 long vowels and
9 aspirated consonants).  The ligature agus is also made available.  The
remaining characters (digits, punctuation and accents) are inherited from the
Computer Modern family of fonts.")
    (license license:public-domain)))

(define-public texlive-eiad-ltx
  (package
    (name "texlive-eiad-ltx")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/eiad-ltx/"
                                     "fonts/source/public/eiad-ltx/"
                                     "source/latex/eiad-ltx/"
                                     "tex/latex/eiad-ltx/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qqwzl345v6hn0gp46flvhhzacgv810qjn575bhigqq726zl4kz7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eiad-ltx")
    (synopsis "LaTeX support for the @code{eiad} font")
    (description
     "The package provides macros to support use of the @code{eiad} fonts in
OT1 encoding.  Also offered are a couple of Metafont files described in the
font package, but not provided there.")
    (license license:lppl)))

(define-public texlive-einfart
  (package
    (name "texlive-einfart")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/einfart/" "tex/latex/einfart/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0hlj00jp7s9b22r4a8l5fc98hnsblmn05v1ryqxknkcj6qdvrj9d"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-minimalist))
    (home-page "https://ctan.org/pkg/einfart")
    (synopsis "Write your articles in a simple and clear way")
    (description
     "This package provides a LaTeX class for typesetting articles with a
simple and clear design.  Currently, it has native support for Chinese
(simplified and traditional), English, French, German, Italian, Japanese,
Portuguese (European and Brazilian), Russian and Spanish typesetting.  It
compiles with either XeLaTeX or LuaLaTeX.  This is part of the
@code{minimalist} class series and depends on that package.")
    (license license:lppl1.3c)))

(define-public texlive-ejpecp
  (package
    (name "texlive-ejpecp")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ejpecp/" "source/latex/ejpecp/"
                                     "tex/latex/ejpecp/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "083abggngg89nj4ra0a33j3s6wf0jba814lc4zp5sh64pc2x8h0j"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ejpecp")
    (synopsis "Class for EJP and ECP")
    (description
     "The class is designed for typesetting articles for the mathematical
research periodicals @emph{Electronic Journal of Probability} (EJP) and
@emph{Electronic Communications in Probability} (ECP).")
    (license license:lppl1.2+)))

(define-public texlive-ekaia
  (package
    (name "texlive-ekaia")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ekaia/" "source/latex/ekaia/"
                                     "tex/latex/ekaia/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1n6zinzcg3win9nn86ljf8fcwsxxn6dq1gjnnw1si8ca01zdyx45"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ekaia")
    (synopsis
     "Article format for the Basque Country Science and Technology Journal
@emph{Ekaia}")
    (description
     "The package provides the article format for publishing the Basque
Country Science and Technology Journal @emph{Ekaia} at the University of the
Basque Country.")
    (license license:lppl1.2+)))

(define-public texlive-ektype-tanka
  (package
    (name "texlive-ektype-tanka")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/ektype-tanka/"
                                     "fonts/truetype/public/ektype-tanka/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05jyx7dgbi19n6g8f4a5f8lkalld2p7lrlrfy1pj961hx65zbj80"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ektype-tanka")
    (synopsis "Devanagari fonts by EkType")
    (description
     "This package provides a collection of some Devanagari fonts by EkType:
Mukta, Baloo, Modak, and Jaini.")
    (license license:lppl1.3c)))

(define-public texlive-elbioimp
  (package
    (name "texlive-elbioimp")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/elbioimp/"
                                     "source/latex/elbioimp/"
                                     "tex/latex/elbioimp/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1iwsahllfw975cs11llr0ghx80wf5kmw5zrbs4iqviai12ipnkd2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/elbioimp")
    (synopsis
     "LaTeX document class for the @emph{Journal of Electrical Bioimpedance}")
    (description
     "This package provides a document class for writing articles to the
@emph{Journal of Electrical Bioimpedance}.")
    (license license:lppl)))

(define-public texlive-electrumadf
  (package
    (name "texlive-electrumadf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/electrumadf/"
                                     "fonts/afm/public/electrumadf/"
                                     "fonts/enc/dvips/electrumadf/"
                                     "fonts/map/dvips/electrumadf/"
                                     "fonts/tfm/public/electrumadf/"
                                     "fonts/type1/public/electrumadf/"
                                     "fonts/vf/public/electrumadf/"
                                     "source/fonts/electrumadf/"
                                     "tex/latex/electrumadf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32 "1mjhwzaryxq2b9g05apz9190wrc1hsxyakdzfl1rn5zfvnb3gx4j"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:tex-format "latex"
           #:phases
           #~(modify-phases %standard-phases
               (add-after 'unpack 'remove-pre-generated-files
                 (lambda _
                   (for-each delete-file
                             (find-files "source/fonts/electrumadf/"
                                         "\\.(e|m)tx$")))))))
    (home-page "https://ctan.org/pkg/electrumadf")
    (synopsis "Electrum ADF fonts collection")
    (description
     "Electrum ADF is a slab-serif font featuring optical and italic
small-caps; additional ligatures and an alternate Q; lining, hanging, inferior
and superior digits; and four weights.  The fonts are provided in Adobe Type
1 format and the support material enables use with LaTeX.")
    (license (list license:lppl1.3c license:gpl2))))

(define-public texlive-eledform
  (package
    (name "texlive-eledform")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/eledform/"
                                     "source/latex/eledform/"
                                     "tex/latex/eledform/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1sslbwabzpvx314bfmr6gy1p8nrh89v5mnkgdcby9w42hnfqiw2n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eledform")
    (synopsis "Define textual variants")
    (description
     "The package provides commands to formalize textual variants in critical
editions typeset using @code{eledmac}.")
    (license license:lppl1.3+)))

(define-public texlive-eledmac
  (package
    (name "texlive-eledmac")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/eledmac/" "source/latex/eledmac/"
                                     "tex/latex/eledmac/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1a4579g3rykcgbwjqbky260lx6x7mc41i6cqhi83anf3zilsii0r"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eledmac")
    (synopsis "Typeset scholarly editions")
    (description
     "This package provides a package for typesetting scholarly critical editions,
replacing the established @code{ledmac} package.  The package supports
indexing by page and by line numbers, and simple @code{tabular}- and
@code{array}-style environments.  The package is distributed with the related
@code{eledpar} package.  The package is now superseded by @code{reledmac}.")
    (license license:lppl1.3+)))

(define-public texlive-els-cas-templates
  (package
    (name "texlive-els-cas-templates")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/els-cas-templates/"
                                     "doc/latex/els-cas-templates/"
                                     "tex/latex/els-cas-templates/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jpwwn4m54zj5rl9ylc3rsjs30rf7hka0qz70wd45405y8mdh5ji"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/els-cas-templates")
    (synopsis "Elsevier updated LaTeX templates")
    (description
     "This bundle provides two class and corresponding template files for
typesetting journal articles supposed to go through Elsevier's updated
workflow.  One of the sets is meant for one-column, the other for two-column
layout.  These are now accepted for submitting articles both in Elsevier's
electronic submission system and elsewhere.")
    (license license:lppl1.2+)))

(define-public texlive-elsarticle
  (package
    (name "texlive-elsarticle")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/elsarticle/"
                                     "doc/latex/elsarticle/"
                                     "source/latex/elsarticle/"
                                     "tex/latex/elsarticle/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1hkqfwn8417irdndxazdpzimc2vr2gdb5bmh6q1f1zrx1ixjks3c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/elsarticle")
    (synopsis "Class for articles for submission to Elsevier journals")
    (description
     "This class for typesetting journal articles is accepted for submitted
articles both in Elsevier's electronic submission system and elsewhere.")
    (license license:lppl1.2+)))

(define-public texlive-elteiktdk
  (package
    (name "texlive-elteiktdk")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/elteiktdk/"
                              "tex/latex/elteiktdk/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0mpy60kszwgr5ckn3xhrfwmqlzh6xzzsdys80ssbcnzjji36cgym"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/elteiktdk")
    (synopsis
     "TDK-thesis template for Hungarian TDK conferences, Section of Computer Science")
    (description
     "The National Conference of Scientific Students Associations (OTDK) of
Hungary is a scientific event for Bachelor and Master students in the country,
where students compete with their research papers in all field of science.
The conference/competition has two rounds: a university level and a country
level (for the best papers).  This class template enforces the required
formatting rules for TDK theses and generates the cover and title page given
on the provided metadata.  The formatting rules are defined to meet the
requirements for TDK theses submitted at the Eotvos Lorand University, Faculty
of Informatics (Budapest, Hungary).  This also fits the formatting
requirements of the Computer Science Section of the country level round.  With
sufficient modifications the template could be usable for TDK theses at other
national and faculty level sections, too.  The template supports producing
both Hungarian and English theses.")
    (license license:expat)))

(define-public texlive-elteikthesis
  (package
    (name "texlive-elteikthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/elteikthesis/"
                                     "tex/latex/elteikthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1apd8ag55s1hmfm99sd1wvpvippry43v3k5yjqvgqf4azb4x9r8d"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/elteikthesis")
    (synopsis "Thesis template for Eotvos Lorand University (Informatics)")
    (description
     "This package provides a bachelor and master thesis template for the
Eotvos Lorand University, Faculty of Informatics (Budapest, Hungary).  The
template supports producing both Hungarian and English theses.")
    (license license:expat)))

(define-public texlive-eltex
  (package
    (name "texlive-eltex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/eltex/" "tex/latex/eltex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1p6wy61n5q93mp07kv7bkp5nnsshshs6wrrhixicn0vankcgnm7w"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eltex")
    (synopsis "Simple circuit diagrams in LaTeX picture mode")
    (description
     "The macros enable the user to draw simple circuit diagrams in the
picture environment, with no need of special resources.  The macros are
appropriate for drawing for school materials.  The circuit symbols accord to
the various parts of the standard IEC 617.")
    (license license:lppl)))

(define-public texlive-elvish
  (package
    (name "texlive-elvish")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/elvish/"
                                     "fonts/source/public/elvish/"
                                     "fonts/tfm/public/elvish/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1vmvkc97a7j93jhdp0wbz4mqvk5q2kxjy5dlklwl5jawggagm7r4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/elvish")
    (synopsis "Fonts for typesetting Tolkien Elvish scripts")
    (description
     "The bundle provides fonts for Cirth and for Tengwar.  The Tengwar fonts
are supported by macros in @file{teng.tex}, or by the (better documented)
@code{tengtex} package.")
    ;; "If you produce modifications that seem useful, you are asked to send
    ;; them to me, so that they can be incorporated into the master files."
    (license (license:non-copyleft "file://doc/fonts/elvish/README"))))

(define-public texlive-emf
  (package
    (name "texlive-emf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/emf/" "tex/latex/emf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1w1wrvb2agq8vw2n8qg0mj3as09nygaxvjj6ji90qd865fj7lwzl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/emf")
    (synopsis "Support for the EMF symbol")
    (description
     "This package provides LaTeX support for the symbol for the EMF in
electric circuits and electrodynamics.  It provides support for multiple
symbols but does not provide any fonts.  The fonts themselves must be acquired
otherwise.")
    (license license:gpl3)))

(define-public texlive-emisa
  (package
    (name "texlive-emisa")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/emisa/" "source/latex/emisa/"
                                     "tex/latex/emisa/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "02f8blixqqbn7k3h2sn59b9d8i2lxqnh2zwdk4gv7wyjr9l0sc1x"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               ;; "emisa.ins" apparently wants to generate itself!  Fix that.
               (add-after 'unpack 'fix-ins
                 (lambda _
                   (substitute* "source/latex/emisa/emisa.ins"
                     (("\\\\file\\{\\\\jobname\\.ins\\}.*") "")))))))
    (home-page "https://ctan.org/pkg/emisa")
    (synopsis "LaTeX package for preparing manuscripts for the journal
@emph{EMISA}")
    (description
     "The EMISA LaTeX package is provided for preparing manuscripts for
submission to @acronym{EMISA, Enterprise Modelling and Information Systems
Architectures}, and for preparing accepted submissions for publication as well
as for typesetting the final document by the editorial office.")
    (license license:lppl1.3c)))

(define-public texlive-endiagram
  (package
    (name "texlive-endiagram")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/endiagram/"
                                     "tex/latex/endiagram/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07vm7ka7651mp5kk7m4ipk6agh4afmnq5ns1mjxvssb2wxfli3f0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/endiagram")
    (synopsis "Easy creation of potential energy curve diagrams")
    (description
     "The package provides the facility of drawing potential energy curve
diagrams with just a few simple commands.")
    (license license:lppl1.3+)))

(define-public texlive-engtlc
  (package
    (name "texlive-engtlc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/engtlc/" "tex/latex/engtlc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xl0x6yanf2933p4ajlwzlrxjrn36kkdy6bm7cy10s1nys3y3pqk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/engtlc")
    (synopsis "Support for users in telecommunications engineering")
    (description
     "The package provides a wide range of abbreviations for terms used in
telecommunications engineering.")
    (license license:lppl1.3+)))

(define-public texlive-iexec
  (package
    (name "texlive-iexec")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/iexec/" "source/latex/iexec/"
                                     "tex/latex/iexec/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13nymi6sk06jwgmc9a24qh892a5gi1wpzly7s53rp9bq41450z3k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-pgf))
    (home-page "https://ctan.org/pkg/iexec")
    (synopsis "Execute shell commands and input their output")
    (description
     "With the help of the @code{\\iexec} command, you can execute a shell
command and then input its output into your document.  This package also lets
you use any special symbols inside your command.")
    (license license:expat)))

(define-public texlive-eolang
  (package
    (name "texlive-eolang")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/eolang/"
                                     "doc/man/man1/eolang.1"
                                     "doc/man/man1/eolang.man1.pdf"
                                     "scripts/eolang/"
                                     "source/latex/eolang/"
                                     "tex/latex/eolang/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0r4ci6dxr0hln0iq23ry7s012iglpkfdz3k0svcq0kqx7jgl72rg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:link-scripts #~(list "eolang.pl")))
    (inputs (list perl))
    (propagated-inputs
     (list texlive-amsfonts
           texlive-amsmath
           texlive-fancyvrb
           texlive-iexec
           texlive-pgf
           texlive-pgfopts
           texlive-stmaryrd))
    (home-page "https://ctan.org/pkg/eolang")
    (synopsis "Formulas and graphs for the EO programming language")
    (description
     "This LaTeX package helps you write φ-calculus formulas and
@url{https://github.com/objectionary/sodg, SODG} graphs for the
@url{https://www.eolang.org, EO} programming language.")
    (license license:expat)))

(define-public texlive-epcqrcode
  (package
    (name "texlive-epcqrcode")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/epcqrcode/"
                              "source/latex/epcqrcode/"
                              "tex/latex/epcqrcode/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "192crw7d0arsw4hl2dqi8pfsbdp37cnmcxj5yq8wngargkm83zsh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/epcqrcode")
    (synopsis "Generate valid EPC QR codes")
    (description
     "This package is a wrapper for the @code{qrcode} package.  It can be used
to generate valid EPC QR codes by providing syntax and validity checks, e.g.,
verifying the correctness of the IBAN.")
    (license license:lppl1.3+)))

(define-public texlive-epigrafica
  (package
    (name "texlive-epigrafica")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/epigrafica/"
                                     "fonts/afm/public/epigrafica/"
                                     "fonts/enc/dvips/epigrafica/"
                                     "fonts/map/dvips/epigrafica/"
                                     "fonts/tfm/public/epigrafica/"
                                     "fonts/type1/public/epigrafica/"
                                     "fonts/vf/public/epigrafica/"
                                     "tex/latex/epigrafica/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1s18ykapm50xyvrw5ls9gzc3m8r6pnjaplryr7q469k69455fh6z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/epigrafica")
    (synopsis "Greek and Latin font")
    (description
     "Epigrafica is a Greek and Latin font, forked from the development of the
Cosmetica font, which is a similar design to Optima and includes Greek.")
    (license license:gpl3+)))

(define-public texlive-eplain
  (package
    (name "texlive-eplain")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/eplain/"
                                     "doc/info/eplain.info"
                                     "doc/man/man1/eplain.1"
                                     "doc/man/man1/eplain.man1.pdf"
                                     "source/eplain/"
                                     "tex/eplain/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0r89l1xbixs4hwc56ya3irlcac87ffmmb374x85g34rm258xrbz3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:create-formats #~(list "eplain")
           #:phases
           #~(modify-phases %standard-phases
               (add-after 'install 'symlink-binaries
                 (lambda* (#:key inputs #:allow-other-keys)
                   (let ((pdftex (search-input-file inputs "bin/pdftex"))
                         (bin (string-append #$output "/bin")))
                     (mkdir-p bin)
                     (with-directory-excursion bin
                       (symlink pdftex "eplain"))))))))
    (native-inputs (list (if (target-64bit?) libfaketime datefudge)))
    (propagated-inputs
     (list texlive-babel
           texlive-cm
           texlive-hyphen-complete
           texlive-knuth-lib
           texlive-latex-fonts
           texlive-pdftex
           texlive-plain))
    (home-page "https://ctan.org/pkg/eplain")
    (synopsis "Extended plain TeX macros")
    (description
     "This package provides an extended version of the plain TeX format,
adding support for bibliographies, tables of contents, enumerated lists,
verbatim input of files, numbered equations, tables, two-column output,
footnotes, hyperlinks in PDF output and commutative diagrams.  Eplain can also
load some of the more useful LaTeX packages, notably @code{graphics},
@code{graphicx} (an extended version of graphics), @code{color},
@code{autopict} (a package instance of the LaTeX picture code), @code{psfrag},
and @code{url}.")
    (license license:gpl2+)))

(define-public texlive-epsdice
  (package
    (name "texlive-epsdice")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/epsdice/" "source/latex/epsdice/"
                                     "tex/latex/epsdice/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09h3jqb44vl1jpb3hf6gbpfpbvfv6lvdvjmp4xpzhl1zs672wccj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/epsdice")
    (synopsis "Scalable dice font")
    (description
     "The @code{epsdice} package defines a single command @code{\\epsdice}
that takes a numeric argument (in the range 1-6), and selects a face image
from a file that contains each of the 6 possible die faces.  The graphic file
is provided in both Encapsulated PostScript and PDF formats.")
    (license license:lppl)))

(define-public texlive-epslatex-fr
  (package
    (name "texlive-epslatex-fr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/epslatex-fr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1xs9977g9g3i6sipkf7i9jdl8sdm69ci3161a4p7k66qbizi7zvb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fepslatex")
    (synopsis "French version of @emph{Graphics in LaTeX}")
    (description
     "This is the French translation of @code{epslatex}, and describes how to
use imported graphics in LaTeX(2e) documents.")
    (license license:gpl3+)))

(define-public texlive-eqexpl
  (package
    (name "texlive-eqexpl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/eqexpl/" "tex/latex/eqexpl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0crbq63m7ra20a1s1yka2v24spfvi86ls1v8asswdrf83zkdpibi"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eqexpl")
    (synopsis "Align explanations for formulas")
    (description
     "This package tries to create perfectly formatted explanation of
components of a formula.")
    (license license:cc-by-sa4.0)))

(define-public texlive-eqnarray
  (package
    (name "texlive-eqnarray")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/eqnarray/"
                                     "source/latex/eqnarray/"
                                     "tex/latex/eqnarray/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0w01rh4a1yglfm9fqkhry1d6bs47caj0dsna9j648z0pxvdf1v38"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eqnarray")
    (synopsis "More generalised equation arrays with numbering")
    (description
     "This package defines an @code{equationarray} environment, that allows
more than three columns, but otherwise behaves like LaTeX's @code{eqnarray}
environment.  This environment is similar, in some ways, to the @code{align}
environment of @code{amsmath}.")
    (license license:gpl3)))

(define-public texlive-eqnnumwarn
  (package
    (name "texlive-eqnnumwarn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/eqnnumwarn/"
                                     "tex/latex/eqnnumwarn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "06z6rb0q5qln5knamwxljzw5h373r2xh159ml9akw4kb2cdk3bri"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/eqnnumwarn")
    (synopsis "Warn for a displaced equation number")
    (description
     "Sometimes an equation is too long that an equation number will be
typeset below the equation itself, but yet not long enough to yield an
@samp{overfull \\hbox} warning.  The @code{eqnnumwarn} package modifies the
standard @code{amsmath} numbered equation environments to throw a warning
whenever this occurs.")
    (license license:lppl1.3+)))

(define-public texlive-erdc
  (package
    (name "texlive-erdc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/erdc/" "source/latex/erdc/"
                                     "tex/latex/erdc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0sqq0zd89jx2l4n1hpas4c4y4p3mh1mbnfigjbal9xg2h5pijkv1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/erdc")
    (synopsis "Style for reports by US Army Corps of Engineers")
    (description
     "This package provides a class for typesetting Technical Information
Reports of the Engineer Research and Development Center, US Army Corps of
Engineers.")
    (license license:lppl)))

(define-public texlive-erewhon
  (package
    (name "texlive-erewhon")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/erewhon/"
                                     "fonts/afm/public/erewhon/"
                                     "fonts/enc/dvips/erewhon/"
                                     "fonts/map/dvips/erewhon/"
                                     "fonts/opentype/public/erewhon/"
                                     "fonts/tfm/public/erewhon/"
                                     "fonts/type1/public/erewhon/"
                                     "fonts/vf/public/erewhon/"
                                     "tex/latex/erewhon/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0vgam0s0kawr69vgykd9b38gc1djjlashbniz877ffw1rsf8p07q"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/erewhon")
    (synopsis "Font package derived from Heuristica and Utopia")
    (description
     "Erewhon is based on the Heuristica package, which is based in turn on
Utopia.  Erewhon adds a number of new features --- small caps in all styles
rather than just regular, added figure styles (proportional, inferior,
numerator, denominator) and superior letters.  The size is 6% smaller than
Heuristica, matching that of UtopiaStd.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-erewhon-math
  (package
    (name "texlive-erewhon-math")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/erewhon-math/"
                                     "fonts/opentype/public/erewhon-math/"
                                     "tex/latex/erewhon-math/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1lsnk2rszqp8snibyx2lf9grvmnjg0y7ys07z4xfrplc04swaw30"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/erewhon-math")
    (synopsis "Utopia based OpenType math font")
    (description
     "Erewhon Math is an OpenType version of the Fourier Type1 fonts designed
by Michel Bovani.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-esrelation
  (package
    (name "texlive-esrelation")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/esrelation/"
                                     "fonts/map/dvips/esrelation/"
                                     "fonts/source/public/esrelation/"
                                     "fonts/tfm/public/esrelation/"
                                     "fonts/type1/public/esrelation/"
                                     "source/fonts/esrelation/"
                                     "tex/latex/esrelation/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "168pjrn3pgyqb79nvwk4rykb214jj1fvygnlb18rx7nrnzdmagjw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/esrelation")
    (synopsis "Symbol set for describing relations between ordered pairs")
    (description
     "This package provides an math symbol font, by Tauba Auerbach, for
describing relations between ordered pairs, using Metafont.")
    (license license:lppl1.3+)))

(define-public texlive-esstix
  (package
    (name "texlive-esstix")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/esstix/"
                                     "fonts/afm/esstix/"
                                     "fonts/map/dvips/esstix/"
                                     "fonts/tfm/public/esstix/"
                                     "fonts/type1/public/esstix/"
                                     "fonts/vf/public/esstix/"
                                     "tex/latex/esstix/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wfl7q6yx7pflzy402ldx7p3zw3xgvgvilrdzv7pm0w937a3qkm9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/esstix")
    (synopsis "PostScript versions of the ESSTIX, with macro support")
    (description
     "These fonts represent translation to PostScript Type 1 of the ESSTIX
fonts.  ESSTIX seem to have been a precursor to the STIX project.  The
accompanying virtual fonts with customized metrics and LaTeX support files
allow their use as @code{calligraphic}, @code{fraktur} and
@code{double-struck} (blackboard bold) in maths mode.")
    (license license:silofl1.1)))

(define-public texlive-estcpmm
  (package
    (name "texlive-estcpmm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/estcpmm/" "source/latex/estcpmm/"
                                     "tex/latex/estcpmm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09fp8vidb9wzdsm05nira4p8dyjn75i1jhhwnr6k17pdrad1sqrj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/estcpmm")
    (synopsis "Style for Munitions Management Project Reports")
    (description
     "This package provides a class which supports typesetting Cost and
Performance Reports and Final Reports for Munitions Management Reports, US
Environmental Security Technology Certification Program.")
    (license license:lppl)))

(define-public texlive-esvect
  (package
    (name "texlive-esvect")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/esvect/"
                                     "fonts/map/dvips/esvect/"
                                     "fonts/source/public/esvect/"
                                     "fonts/tfm/public/esvect/"
                                     "fonts/type1/public/esvect/"
                                     "source/latex/esvect/"
                                     "tex/latex/esvect/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15sr1rm86aamfy186v2zvvcjsw5rhcq6ph5ckpfn275vjp3fazrf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/esvect")
    (synopsis "Vector arrows")
    (description
     "This package can be used to write vectors using an arrow which differs
from the Computer Modern one.  You have the choice between several kinds of
arrows.  The package consists of the relevant Metafont code and a package to
use it.")
    (license license:gpl3+)))

(define-public texlive-etbb
  (package
    (name "texlive-etbb")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/etbb/"
                                     "fonts/afm/public/etbb/"
                                     "fonts/enc/dvips/etbb/"
                                     "fonts/map/dvips/etbb/"
                                     "fonts/opentype/public/etbb/"
                                     "fonts/tfm/public/etbb/"
                                     "fonts/type1/public/etbb/"
                                     "fonts/vf/public/etbb/"
                                     "tex/latex/etbb/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0f07wz9qgvp9m3s4n7rzh2r5fq4h03ffxqpw5wv17imz2bvr1dr3"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/etbb")
    (synopsis "Expansion of Edward Tufte's ET-Bembo family")
    (description
     "The Bembo-like font family, ETbb, expands ET-Bembo features to include
a full set of figure styles, small caps in all styles, superior letters and
figures, inferior figures, a new capital Sharp @samp{S} with small caps
version, along with macros to activate these features in LaTeX.")
    (license (list license:expat license:lppl1.3+))))

(define-public texlive-etsvthor
  (package
    (name "texlive-etsvthor")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/etsvthor/" "tex/latex/etsvthor/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0lrvxr2lrfq1vkfkx7sjrm8fsxv8aalkpi0x0wmkdizc6n2wlhx4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/etsvthor")
    (synopsis "Some useful abbreviations for members of e.t.s.v. Thor")
    (description
     "``e.t.s.v.  Thor'' stands for Elektrotechnische Studievereniging Thor,
a study association of Electrical Engeering at the Eindhoven University of
Technology.  This package provides abbreviations useful for creating meeting
notes or other documents within the association.")
    (license license:lppl1.3c)))

(define-public texlive-euclidean-lattice
  (package
    (name "texlive-euclidean-lattice")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list
                         "doc/latex/euclidean-lattice/"
                         "source/latex/euclidean-lattice/"
                         "tex/latex/euclidean-lattice/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1gfnqmsqnsrfd6pphly1n5hwzkhqqvarzdabdz944xvlpwhvfq6c"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/euclidean-lattice")
    (synopsis "Draw two-dimensional Euclidean lattices with TikZ")
    (description
     "This package provides an efficient and configurable way to draw
two-dimensional Euclidean lattices using TikZ.")
    (license license:lppl1.3c)))

(define-public texlive-euclideangeometry
  (package
    (name "texlive-euclideangeometry")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/euclideangeometry/"
                                     "source/latex/euclideangeometry/"
                                     "tex/latex/euclideangeometry/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pzqagrkr0w67n4v3z3q37p1rk3zk8sp2z2kl86rcqq5zi1rf9cl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/euclideangeometry")
    (synopsis "Draw geometrical constructions")
    (description
     "This package provides tools to draw most of the geometrical
constructions that a high school instructor or bachelor degree professor might
need to teach geometry.  The connection to Euclide depends on the fact that in
his times calculations were made with ruler, compass and also with
ellipsograph.  This package extends the functionalities of the @code{curve2e}
package.")
    (license license:lppl1.3c)))

(define-public texlive-euler-math
  (package
    (name "texlive-euler-math")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/euler-math/"
                                     "fonts/opentype/public/euler-math/"
                                     "tex/latex/euler-math/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0r4ygvgvmib88d4b0pkzi9s84lbc8dm423vr0y75cy8szxlnb7kj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/euler-math")
    (synopsis "OpenType version of Hermann Zapf's Euler maths font")
    (description
     "@file{Euler-Math.otf} is an OpenType version of Hermann Zapf's Euler
maths font.  A style file @file{euler-math.sty} is provided as a replacement
of the @code{eulervm} package for LuaLaTeX and XeLaTeX users.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-euxm
  (package
    (name "texlive-euxm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "fonts/source/public/euxm/"
                                     "fonts/tfm/public/euxm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10dqknqlqrga9m2lgd74zx5hk4qcbkfvlzjgr304p94w3k7pclqc"))))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/euxm")
    (synopsis "Extended Euler")
    (description
     "This package extends Euler font to include two additional characters
needed for Concrete Math.")
    (license license:knuth)))

(define-public texlive-evangelion-jfm
  (package
    (name "texlive-evangelion-jfm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/luatex/evangelion-jfm/"
                                     "source/luatex/evangelion-jfm/"
                                     "tex/luatex/evangelion-jfm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1g174ck75h43gppv2c7ihbf815zc07fchc56c5bidywfv09c4y3s"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/evangelion-jfm")
    (synopsis "Japanese font metric supporting many advanced features")
    (description
     "This package provides a Japanese font metric supporting vertical and
horizontal typesetting, linegap punctuations, extended fonts, and more
interesting and helpful features using traditional and simplified Chinese or
Japanese fonts under LuaTeX-ja.  It also makes full use of the priority
feature, meeting the standards, and allows easy customisation.")
    (license license:expat)))

(define-public texlive-expex
  (package
    (name "texlive-expex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/expex/" "tex/generic/expex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0is4cdpsqycaw8xzc6x1gkg4ldk8v6a1i52qmxw13q8k10qh6b95"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/expex")
    (synopsis "Linguistic examples and glosses, with reference capabilities")
    (description
     "The package provides macros for typesetting linguistic examples and
glosses, with a refined mechanism for referencing examples and parts of
examples.  The package can be used with LaTeX or with Plain TeX.")
    (license license:lppl)))

(define-public texlive-expex-glossonly
  (package
    (name "texlive-expex-glossonly")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/expex-glossonly/"
                              "tex/latex/expex-glossonly/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "16x99ypmfpqyac07760d7jdxjnndw4q6z0m2hjy8qfvwm2ik24gx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/expex-glossonly")
    (synopsis
     "Help gb4e, linguex, and covington users use the ExPex glossing macros")
    (description
     "The ExPex package provides very fine-grained control over glossing and
example formatting, including unlimited gloss lines and various ways of
formatting multiline glosses.  By contrast the @code{cgloss4e} glossing macros
provided with @code{gb4e}, @code{linguex}, and @code{covington}, although very
capable at basic glossing, lack the degree of customization that is sometimes
needed for more complex glossing.  This package is an attempt to have the best
of both worlds: it allows @code{gb4e}, @code{linguex} and @code{covington}
users to keep using those packages for basic example numbering and formatting,
but also allows them to use the glossing macros that ExPex provides.")
    (license license:lppl1.3+)))

(define-public texlive-expose-expl3-dunkerque-2019
  (package
    (name "texlive-expose-expl3-dunkerque-2019")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/expose-expl3-dunkerque-2019/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0xij1ycxnc3hfzkl0bfs03s0zzzv6xsj3myvykb6n836j1g66g2y"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/expose-expl3-dunkerque-2019")
    (synopsis "Using @code{expl3} to implement some numerical algorithms")
    (description
     "This is an article, in French, based on a presentation made in Dunkerque
for the @emph{Stage LaTeX} on 12 June 2019.  The article gives three examples
of code in @code{expl3} with (lots of) comments: Knuth's algorithm to create
a list of primes, the sieve of Eratosthenes, Kaprekar sequences.  The package
contains the code itself, the documentation as a PDF file, and all the files
needed to produce it.")
    (license license:lppl1.3+)))

(define-public texlive-extarrows
  (package
    (name "texlive-extarrows")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/extarrows/"
                                     "tex/latex/extarrows/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0fspc9vgz1vwi5lamv30czh6pp1vqsrk6f2ynqygq166rfxxm1gz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/extarrows")
    (synopsis "Extra arrows beyond those provided in @code{amsmath}")
    (description
     "This package provides arrows to supplement @code{\\xleftarrow} and
@code{\\xrightarrow} of the @code{amsath} package.")
    ;; Package mentions LGPL without any reference to the version.  Assuming
    ;; LGPL 2.1+.
    (license license:lgpl2.1+)))

(define-public texlive-extpfeil
  (package
    (name "texlive-extpfeil")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/extpfeil/"
                                     "source/latex/extpfeil/"
                                     "tex/latex/extpfeil/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "11vri2kij4y3da2v1gsps66jf4r529w4nqbgbfajrxh2nhfb60iv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/extpfeil")
    (synopsis "Extensible arrows in mathematics")
    (description
     "The package provides some more extensible arrows (usable in the same way
as @code{\\xleftarrow} from @code{amsmath}), and a simple command to create
new ones.")
    (license license:lppl1.3+)))

(define-public texlive-ezedits
  (package
    (name "texlive-ezedits")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/ezedits/"
                              "tex/latex/ezedits/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0zrwlcd54gipxsg9pfdxyjfwpb3mc7gpcymc70gyb9d3kgwcrwnm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ezedits")
    (synopsis "Commands for tracking document changes and notes")
    (description
     "This is a package built for collaboratively editing LaTeX documents and
tracking changes.  Through highly configurable commands, the user can choose
how their and their collaborators modifications appear in the document.
Additional tools are provided to help keep track of where the edits are made
within the PDF.")
    (license license:lppl1.3c)))

(define-public texlive-facture
  (package
    (name "texlive-facture")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/xelatex/facture/"
                                     "source/xelatex/facture/"
                                     "tex/xelatex/facture/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "04a9h99vqrb25pqwl3gn24kw7pm87i3bwq24inhfvac3sj7wq97k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/facture")
    (synopsis "Generate an invoice")
    (description
     "This package provides a simple class that allows production of an invoice,
with or without VAT; different addresses for delivery and for billing are
permitted.")
    (license license:cc-by-sa2.0)))

(define-public texlive-facture-belge-simple-sans-tva
  (package
    (name "texlive-facture-belge-simple-sans-tva")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/xelatex/facture-belge-simple-sans-tva/"
                                     "tex/xelatex/facture-belge-simple-sans-tva/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1zs2hcc5lhzvfkv83wym1rax58mryylq9b7i358ln0pklnm2gncx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/facture-belge-simple-sans-tva")
    (synopsis "Simple Belgian invoice without VAT")
    (description
     "This package can be used to generate invoices for Belgian individuals
who do not have a VAT number and who wish to do occasional work, or to carry
out paid additional activities during their free time up to 6,000 euros per
calendar year (amount indexed annually) without having to pay tax or social
security contributions (see the website Activites complementaires).  The
package can also generate expense reports.  All totals are calculated
automatically, in the invoice and in the expense report.")
    (license license:lppl1.3c)))

(define-public texlive-fail-fast
  (package
    (name "texlive-fail-fast")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/fail-fast/"
                              "source/latex/fail-fast/"
                              "tex/latex/fail-fast/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "16hbqsyxiw1c8lbvzygh9yd8m63s0i6n50wj41bh9q04vdm9nnld"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fail-fast")
    (synopsis "Turn warnings into errors")
    (description
     "This LaTeX package helps you make the build more fragile by turning
warnings into errors.  This may be a good practice if you care about the
quality of your documents.")
    (license license:expat)))

(define-public texlive-faktor
  (package
    (name "texlive-faktor")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/faktor/" "source/latex/faktor/"
                                     "tex/latex/faktor/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1sridf0sn775n9ivrbfxz2fgrcm0wl7ajbhr4509ck0pq2n5hq17"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/faktor")
    (synopsis "Typeset quotient structures with LaTeX")
    (description
     "The package provides the means to typeset factor structures, as are used
in many areas of algebraic notation.  The structure is similar to the A/B that
is provided by the @code{nicefrac} package (part of the @code{units}
distribution), and by the @code{xfrac} package; the most obvious difference is
that the numerator and denominator's sizes do not change in the
@code{\\faktor} command.")
    (license license:lppl)))

(define-public texlive-fascicules
  (package
    (name "texlive-fascicules")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fascicules/"
                                     "source/latex/fascicules/"
                                     "tex/latex/fascicules/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0hsap900psylamhn3my7gsn7n06glqczcff0yw53zs9gjwyh4s13"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fascicules")
    (synopsis "Create mathematical manuals for schools")
    (description
     "This package enables LaTeX users to create math books for middle and
high schools.  It provides commands to create the front page of the manual and
the chapters.  Each chapter can consist of three sections: the lesson, the
exercises and the activities.")
    (license license:lppl)))

(define-public texlive-fbb
  (package
    (name "texlive-fbb")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fbb/"
                                     "fonts/enc/dvips/fbb/"
                                     "fonts/map/dvips/fbb/"
                                     "fonts/opentype/public/fbb/"
                                     "fonts/tfm/public/fbb/"
                                     "fonts/type1/public/fbb/"
                                     "fonts/vf/public/fbb/"
                                     "tex/latex/fbb/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "012zfxq6hdz9m4pn6dxqv3lsbr809r10mj795w96cijzrys4zww1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fbb")
    (synopsis "Bembo-like font")
    (description
     "The package provides a Bembo-like font package based on Cardo but with
many modifications, adding Bold Italic, small caps in all styles, six figure
choices in all styles, updated kerning tables, added figure tables and
corrected f-ligatures.  Both OpenType and Adobe Type 1 versions are provided;
all necessary support files are provided.  The font works well with
@code{newtxmath}'s @code{libertine} option.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-fbithesis
  (package
    (name "texlive-fbithesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fbithesis/"
                                     "source/latex/fbithesis/"
                                     "tex/latex/fbithesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1650ilscm47x606piwx7a8z08lhkq87g8k0z40ii3ig0c3p9isgl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:tex-format "latex"))
    (home-page "https://ctan.org/pkg/fbithesis")
    (synopsis "Computer Science thesis class for University of Dortmund")
    (description
     "At the department of computer science at the University of Dortmund
there are cardboard cover pages for research or internal reports like master
and phd theses.  The main function of this LaTeX2e document-class is
a replacement for the @code{\\maketitle} command to typeset a title page that
is adjusted to these cover pages.")
    (license license:lppl1.3+)))

(define-public texlive-fbs
  (package
    (name "texlive-fbs")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/fbs/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "18h6n8azrnk2qabv30kpa8gmj2ljkvx0nqs8qb9r8dxxnhng78ch"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fbs")
    (synopsis "BibTeX style for @emph{Frontiers in Bioscience}")
    (description
     "This package provides a BibTeX style file made with @code{custom}-bib to
fit @emph{Frontiers in Bioscience} requirements.")
    (license license:lppl)))

(define-public texlive-fcavtex
  (package
    (name "texlive-fcavtex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/fcavtex/" "doc/latex/fcavtex/"
                                     "tex/latex/fcavtex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0l87wlh9mzh0hysfw35mlx3asqw7lvwc55yiqr01jbmv1hwgfnl6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fcavtex")
    (synopsis "Thesis class for the FCAV/UNESP, Brazil")
    (description
     "This package provides a class and a bibliography style for the
@acronym{FCAV-UNESP, Faculdade de Ciencias Agrarias e Veterinarias de
Jaboticabal UNESP} Brazilian university, written based on the institution
rules for thesis publications.")
    (license license:lppl1.3+)))

(define-public texlive-fcltxdoc
  (package
    (name "texlive-fcltxdoc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fcltxdoc/"
                                     "source/latex/fcltxdoc/"
                                     "tex/latex/fcltxdoc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0qb0l622svj1bqs0p05mcf30f4xy432aa4qzk6w4rlckssdfiwva"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fcltxdoc")
    (synopsis "Macros for use in the author's documentation")
    (description
     "The package is not advertised for public use, but is necessary for the
support of others of the author's packages (which are compiled under the
@code{ltxdoc} class).")
    (license license:lppl1.3+)))

(define-public texlive-fdsymbol
  (package
    (name "texlive-fdsymbol")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fdsymbol/"
                                     "doc/latex/fdsymbol/"
                                     "fonts/enc/dvips/fdsymbol/"
                                     "fonts/map/dvips/fdsymbol/"
                                     "fonts/opentype/public/fdsymbol/"
                                     "fonts/source/public/fdsymbol/"
                                     "fonts/tfm/public/fdsymbol/"
                                     "fonts/type1/public/fdsymbol/"
                                     "source/latex/fdsymbol/"
                                     "tex/latex/fdsymbol/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14abmd3pa8icmf8mvfaq8i6il9w9n6dq7g0y73rwn17917zkx2ii"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/fdsymbol")
    (synopsis "Maths symbol font")
    (description
     "FdSymbol is a maths symbol font, designed as a companion to the Fedra
family, but it might also fit other contemporary typefaces.")
    (license license:silofl1.1)))

(define-public texlive-fduthesis
  (package
    (name "texlive-fduthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fduthesis/"
                                     "source/latex/fduthesis/"
                                     "tex/latex/fduthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1w8dqld77pjcv5si3d1xmvphy908g5sqndy8rrbbi5rm2cqw6a5s"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fduthesis")
    (synopsis "LaTeX thesis template for Fudan University")
    (description
     "This package is a LaTeX thesis template package for Fudan University.
It can make it easy to write theses both in Chinese and English.")
    (license license:lppl1.3c)))

(define-public texlive-fei
  (package
    (name "texlive-fei")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fei/" "source/latex/fei/"
                                     "tex/latex/fei/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13f2sn9pazq6ak37025wr20mi11069paw8hd7qlrcy10bn7bkbl7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fei")
    (synopsis "Class for academic works at FEI University Center, Brazil")
    (description
     "@code{fei} is a class created by graduate students and LaTeX enthusiasts
that allows students from FEI University Center to create their academic
works, be it a monograph, masters dissertation or PhD thesis, under the
typographic rules of the institution.  The class makes it possible to create
a full academic work, supporting functionalities such as cover, title page,
catalog entry, dedication, summary, lists of figures, tables, algorithms,
acronyms and symbols, multiple authors, index, references, appendices and
attachments.  @code{fei} is loosely based in the Brazilian National Standards
Organization (@acronym{ABNT, Associacao Brasileira de Normas Tecnicas}).")
    (license license:lppl1.3c)))

(define-public texlive-fetamont
  (package
    (name "texlive-fetamont")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fetamont/"
                                     "fonts/afm/public/fetamont/"
                                     "fonts/map/dvips/fetamont/"
                                     "fonts/opentype/public/fetamont/"
                                     "fonts/source/public/fetamont/"
                                     "fonts/tfm/public/fetamont/"
                                     "fonts/type1/public/fetamont/"
                                     "metapost/fetamont/"
                                     "source/fonts/fetamont/"
                                     "tex/latex/fetamont/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0bs911vyn54i61vv3rhr0xcp6myap2s8ww6p1fd30ld1ha6nsrf6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    ;; FIXME: Generating font metrics require unpackaged "mf2outline.py"
    ;; script.
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               (delete 'generate-font-metrics))))
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/fetamont")
    (synopsis "Extended version of Knuth's Logo typeface")
    (description
     "The @code{fetamont} typeface was designed in Metafont and extends the
Logo fonts to complete the Type 1 encoding.")
    (license license:lppl1.3+)))

(define-public texlive-feyn
  (package
    (name "texlive-feyn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/feyn/"
                                     "fonts/source/public/feyn/"
                                     "fonts/tfm/public/feyn/"
                                     "source/fonts/feyn/" "tex/latex/feyn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kd5xs1xffgplpapj6rjv4aaz7pfjfh8rspy2xrabbf03npkdqz5"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-sauter")))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont texlive-sauter))
    (home-page "https://ctan.org/pkg/feyn")
    (synopsis "Font for in-text Feynman diagrams")
    (description
     "Feyn may be used to produce relatively simple Feynman diagrams within
equations in a LaTeX document.  While the @code{feynmf} package is good at
drawing large diagrams for figures, the present package and its fonts allow
diagrams within equations or text, at a matching size.  The fonts are
distributed as Metafont source, and macros for their use are also provided.")
    (license license:bsd-2)))

(define-public texlive-fge
  (package
    (name "texlive-fge")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fge/"
                                     "fonts/map/dvips/fge/"
                                     "fonts/source/public/fge/"
                                     "fonts/tfm/public/fge/"
                                     "fonts/type1/public/fge/"
                                     "source/fonts/fge/"
                                     "tex/latex/fge/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00q9rf8jrb6jl5c10kn6axnlza6q02xan2gf88rd736v0wpi4936"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/fge")
    (synopsis "Font for Frege's @emph{Grundgesetze der Arithmetik}")
    (description
     "The package provides fonts for Frege's @emph{Grundgesetze der
Arithmetik}.  The fonts are provided as Metafont source and Adobe Type 1 (pfb)
files.  A small LaTeX package (@code{fge}) is included.")
    (license license:lppl)))

(define-public texlive-figbas
  (package
    (name "texlive-figbas")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/figbas/"
                                     "fonts/afm/public/figbas/"
                                     "fonts/map/dvips/figbas/"
                                     "fonts/tfm/public/figbas/"
                                     "fonts/type1/public/figbas/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07dc19989sx968hbn3h2m1bbplp8zqh1yj9v92mcp54876rk8bbp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/figbas")
    (synopsis "Mini-fonts for figured-bass notation in music")
    (description
     "This package consists of three mini-fonts (and associated metrics) of
conventional ligatures for the figured-bass notations 2+, 4+, 5+, 6+ and 9+ in
music manuscripts.  The fonts are usable with Computer Modern Roman and Sans,
and Palatino/Palladio, respectively.")
    (license license:lppl)))

(define-public texlive-figbib
  (package
    (name "texlive-figbib")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/figbib/" "doc/latex/figbib/"
                                     "tex/latex/figbib/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wvglj3z86xjz1bi5931hxm9b2kn6r4j7391di26wmsyhdajs2yz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/figbib")
    (synopsis "Organize figure databases with BibTeX")
    (description
     "FigBib lets you organize your figures in BibTeX databases.  Some FigBib
features are:
@itemize

@item store and manage figures in a BibTeX database;

@item Include figures in your LaTeX document with one short command;

@item generate a list of figures containing more or other information than the
figure captions;

@item control with one switch where to output the figures, either as usual
float objects or in a separate part at the end of your document.

@end itemize")
    (license license:lppl)))

(define-public texlive-filecontentsdef
  (package
    (name "texlive-filecontentsdef")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/filecontentsdef/"
                                     "source/latex/filecontentsdef/"
                                     "tex/latex/filecontentsdef/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0pmjvxjsfnyzx3bl10n8is7l27l11mrw22g3w41f72kric0vfd65"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/filecontentsdef")
    (synopsis "@code{filecontents}, @code{macro}, and @code{verbatim}")
    (description
     "The package provides two environments called @code{filecontentsdef} and
@code{filecontentshere}.  They are derived from the LaTeX @code{filecontents}
environment.  In addition to the file creation they either store
the (verbatim) contents in a macro (@code{filecontentsdef}) or typeset
them (verbatim) on the spot (@code{filecontentshere}).  The author developed
the package to display TeX code verbatim in documentation and the same time
produce the corresponding files during the LaTeX run in order to embed them in
the PDF as file attachment annotations (by using Scott Pakin's package
@code{attachfile}).")
    (license license:lppl1.3+)))

(define-public texlive-firamath
  (package
    (name "texlive-firamath")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/firamath/"
                                     "fonts/opentype/public/firamath/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0sc3ip9mvyqlbsj7vfcb1yaxlc32w18kqrqm051v5y2bbvp53s0p"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/firamath")
    (synopsis "Fira sans serif font with Unicode math support")
    (description
     "Fira Math is a sans-serif font with Unicode math support.  The design of
this font is based on Fira Sans and FiraGO.  Fira Math is distributed in
OpenType format and can be used with the @code{unicode-math} package under
XeLaTeX or LuaLaTeX.  More support is offered by the @code{firamath-otf}
package.")
    (license license:silofl1.1)))

(define-public texlive-firamath-otf
  (package
    (name "texlive-firamath-otf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/firamath-otf/"
                                     "tex/latex/firamath-otf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "094b56is939li9cadqz2b7mz86ji5dnryi8k54nym8qmjsdx2gd1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-firamath))
    (home-page "https://ctan.org/pkg/firamath-otf")
    (synopsis "Use OpenType math font Fira Math")
    (description
     "The package offers XeTeX and LuaTeX support for the sans serif OpenType
Fira Math font.")
    (license license:lppl1.3+)))

(define-public texlive-fixdif
  (package
    (name "texlive-fixdif")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fixdif/" "source/latex/fixdif/"
                                     "tex/latex/fixdif/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0l15mkwbnk0k2qqdnkady5n3iqyj0w7f8k23rxnbyrk5835crl67"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fixdif")
    (synopsis "Macros for typesetting differential operators")
    (description
     "This package redefines the @code{\\d} command in LaTeX and provides an
interface to define new commands for differential operators.  It is compatible
with pdfTeX, XeTeX and LuaTeX, and can also be used with the
@code{unicode-math} package.")
    (license license:lppl1.3c)))

(define-public texlive-fixfoot
  (package
    (name "texlive-fixfoot")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fixfoot/" "tex/latex/fixfoot/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1anyswchln7y7a6x72ay6hm3ck36byxk2ciia0w6y100iqcpq2sl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fixfoot")
    (synopsis "Multiple use of the same footnote text")
    (description
     "This package provides a @code{\\DeclareFixedFootnote} command to provide
a single command for a frequently-used footnote.  The package ensures that
only one instance of the footnote text appears on each page (LaTeX needs to be
run several times to achieve this).")
    (license license:lppl)))

(define-public texlive-fixjfm
  (package
    (name "texlive-fixjfm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/generic/fixjfm/" "tex/generic/fixjfm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0x3wf7v03jqczvxaslqw422bvccvna31jl5sgld6llvfh7b0ziyx"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fixjfm")
    (synopsis "Fix JFM (for *pTeX)")
    (description
     "This package fixes several bugs in the JFM format.  Both LaTeX and plain
TeX are supported.")
    (license license:knuth)))

(define-public texlive-fixltxhyph
  (package
    (name "texlive-fixltxhyph")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fixltxhyph/"
                                     "source/latex/fixltxhyph/"
                                     "tex/latex/fixltxhyph/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1csznkhxfwpfmnjl97bynsj3rcq7zyb5ixlkx4mb9ycm3yzfcaal"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fixltxhyph")
    (synopsis "Allow hyphenation of partially-emphasised substrings")
    (description
     "The package fixes the problem of TeX failing to hyphenate letter strings
that seem (to TeX} to be words, but which are followed by an apostrophe and
then an emphasis command.  The cause of the problem is not the apostrophe, but
the font change in the middle of the string.  The problem arises in Catalan,
French, Italian and Romansh.")
    (license license:lppl1.3+)))

(define-public texlive-fixmath
  (package
    (name "texlive-fixmath")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fixmath/" "source/latex/fixmath/"
                                     "tex/latex/fixmath/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15vwikvxxd8wagbfxjb1am3savciyppkq0bdk3zdxp09zg0dx2ra"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fixmath")
    (synopsis "Make maths comply with ISO 31-0:1992 to ISO 31-13:1992")
    (description
     "LaTeX's default style of typesetting mathematics does not comply with
the International Standards ISO 31-0:1992 to ISO 31-13:1992 which require that
uppercase Greek letters always be typset upright, as opposed to italic (even
though they usually represent variables) and allow for typsetting of variables
in a boldface italic style (even though the required fonts are available).
This package ensures that uppercase Greek be typeset in italic style, that
upright @samp{$\\Delta$} and @samp{$\\Omega$} symbols are available through
the commands @code{\\upDelta} and @code{\\upOmega}; and provides a new math
alphabet @code{\\mathbold} for boldface italic letters, including Greek.")
    (license license:lppl1.3c)))

(define-public texlive-fnspe
  (package
    (name "texlive-fnspe")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fnspe/" "tex/latex/fnspe/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wvr0jp8qyrrx907diphgjzvgq0alc922bmx905sndnwqzfl8fhd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fnspe")
    (synopsis "Macros for supporting mainly students of FNSPE CTU in Prague")
    (description
     "This package is primary intended for students of FNSPE CTU in Prague but
many other students or scientists can found this package as useful.  This
package implements different standards of tensor notation, interval notation
and complex notation.  Further many macros and shortcuts are added, e.g., for
spaces, operators, physics unit, etc.")
    (license license:lppl)))

(define-public texlive-foekfont
  (package
    (name "texlive-foekfont")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/foekfont/"
                                     "fonts/map/dvips/foekfont/"
                                     "fonts/tfm/public/foekfont/"
                                     "fonts/type1/public/foekfont/"
                                     "tex/latex/foekfont/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "01l6mj5dh52cdaqdkv74j8qw2pfd6w7x53n856vsjgdha4r8362y"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/foekfont")
    (synopsis "Title font of the @emph{Mads Fok} magazine")
    (description
     "The bundle provides the Foek font, an Adobe Type 1 font, and LaTeX
support for its use.")
    (license license:gpl3+)))

(define-public texlive-fonetika
  (package
    (name "texlive-fonetika")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fonetika/"
                                     "fonts/afm/public/fonetika/"
                                     "fonts/map/dvips/fonetika/"
                                     "fonts/tfm/public/fonetika/"
                                     "fonts/truetype/public/fonetika/"
                                     "fonts/type1/public/fonetika/"
                                     "tex/latex/fonetika/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1d19a9v3innfq9602w8rd2ffn0yd52fz4wmdva6qnix0jgadc1ka"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fonetika")
    (synopsis "Support for the Danish Dania phonetic system")
    (description
     "Fonetika Dania is a font bundle with a serif font and a sans serif font
for the Danish phonetic system Dania.  Both fonts exist in regular and bold
weights.  LaTeX support is provided.")
    ;; The license is GPL3+, except for the fonts and metrics of the Fonetika
    ;; Dania Iwonae, which are under GUST Font Nosource License.
    (license (list license:gpl3+ license:gfl1.0))))

(define-public texlive-fontawesome
  (package
    (name "texlive-fontawesome")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fontawesome/"
                                     "fonts/enc/dvips/fontawesome/"
                                     "fonts/map/dvips/fontawesome/"
                                     "fonts/opentype/public/fontawesome/"
                                     "fonts/tfm/public/fontawesome/"
                                     "fonts/type1/public/fontawesome/"
                                     "tex/latex/fontawesome/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0m3wl0jc00h8r4w3fa5vkf062hmaadb2rvf3x9lm4pb0c99ia5x9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fontawesome")
    (synopsis "Font containing web-related icons")
    (description
     "The package offers access to the large number of web-related icons
provided by the FontAwesome font.")
    (license license:lppl1.3+)))

(define-public texlive-fontawesome5
  (package
    (name "texlive-fontawesome5")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fontawesome5/"
                                     "fonts/enc/dvips/fontawesome5/"
                                     "fonts/map/dvips/fontawesome5/"
                                     "fonts/opentype/public/fontawesome5/"
                                     "fonts/tfm/public/fontawesome5/"
                                     "fonts/type1/public/fontawesome5/"
                                     "tex/latex/fontawesome5/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17cfqa2jjg3lp0damwy8kfdihmchx55994rrc897kcdba865rrwg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fontawesome5")
    (synopsis "Font Awesome 5 with LaTeX support")
    (description
     "This package provides LaTeX support for the included Font Awesome 5 icon
set.")
    (license (list license:silofl1.1 license:lppl1.3c))))

(define-public texlive-fontmfizz
  (package
    (name "texlive-fontmfizz")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fontmfizz/"
                                     "fonts/truetype/public/fontmfizz/"
                                     "tex/latex/fontmfizz/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1z0k7n1yyisv3igw77bj3r8hywyz55ciylb3fjx9dpdcrwv3kngd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fontmfizz")
    (synopsis "Font Mfizz icons for use in LaTeX")
    (description
     "The MFizz font provides scalable vector icons representing programming
languages, operating systems, software engineering, and technology.  It can be
seen as an extension to FontAwesome.")
    (license license:expat)))

(define-public texlive-fontsize
  (package
    (name "texlive-fontsize")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fontsize/"
                                     "source/latex/fontsize/"
                                     "tex/latex/fontsize/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14kibnibj201fnhc00mgq4spl941mhrpl4l8mmkdz3fc6b06acc8"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-local-tree")))
    (build-system texlive-build-system)
    (arguments
     (list #:tex-format "latex"
           #:phases
           #~(modify-phases %standard-phases
               (add-after 'unpack 'fix-circular-dependency
                 ;; Documentation doesn't build, and generates a circular
                 ;; dependency with BIBLATEX-PHILOSOPHY package.  Since we're
                 ;; not interested in re-building the whole documentation,
                 ;; just skip that part.
                 (lambda _
                   (substitute* "source/latex/fontsize/fontsize.dtx"
                     (("\\[style=philosophy-classic\\]") "")
                     (("\\\\DocInput\\{fontsize\\.dtx\\}") "")))))))
    (native-inputs
     (list (texlive-local-tree
            (list texlive-biblatex
                  texlive-booktabs
                  texlive-caption
                  texlive-cochineal
                  texlive-csquotes
                  texlive-etoolbox
                  texlive-fontaxes
                  texlive-guitlogo
                  texlive-hologo
                  texlive-hypdoc
                  texlive-inconsolata
                  texlive-libertine
                  texlive-listings
                  texlive-ltxdockit
                  texlive-manfnt
                  texlive-mathalpha
                  texlive-mdframed
                  texlive-metalogo
                  texlive-microtype
                  texlive-needspace
                  texlive-newtx
                  texlive-parskip
                  texlive-pgf
                  texlive-sectsty
                  texlive-siunitx
                  texlive-tabu
                  texlive-upquote
                  texlive-xcolor
                  texlive-xkeyval
                  texlive-xstring
                  texlive-zref))))
    (home-page "https://ctan.org/pkg/fontsize")
    (synopsis
     "Small package to set arbitrary sizes for the main font of the document")
    (description
     "The package allows you to set arbitrary sizes for the main font of the
document, through the @emph{fontsize=<size>} option.")
    (license license:lppl1.3+)))

(define-public texlive-fonttable
  (package
    (name "texlive-fonttable")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fonttable/"
                                     "source/latex/fonttable/"
                                     "tex/latex/fonttable/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1mb83zzx5wk92x6zz28k42wmi82hyd1h7fb3p9imjssn6qb73356"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fonttable")
    (synopsis "Print font tables from a LaTeX document")
    (description
     "This is a package version of @file{nfssfont.tex}; it enables you to
print a table of the characters of a font and/or some text (for demonstration
or testing purposes), from within a document.")
    (license license:lppl1.3c)))

(define-public texlive-footbib
  (package
    (name "texlive-footbib")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/footbib/" "source/latex/footbib/"
                                     "tex/latex/footbib/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kv8nh82b16bmhxh2291ahznhjgm5g4qskqwdba66w48wkhzr32j"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/footbib")
    (synopsis "Bibliographic references as footnotes")
    (description
     "The package makes bibliographic references appear as footnotes.  It
defines a command @code{\\footcite} which is similar to the LaTeX
@code{\\cite} command but the references cited in this way appear at the
bottom of the pages.  This @code{foot} bibliography does not conflict with the
standard one and both may exist simultaneously in a document.  The command
@code{\\cite} may still be used to produce the standard bibliography.  The
@code{foot} bibliography uses its own style and bibliographic database which
may be specified independently of the standard one.  Any standard bibliography
style may be used.")
    (license license:lppl)))

(define-public texlive-formal-grammar
  (package
    (name "texlive-formal-grammar")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/formal-grammar/"
                                     "source/latex/formal-grammar/"
                                     "tex/latex/formal-grammar/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00vr0gdz6jikdycsbfn7n8ddb8391h9fl314vjf8yxz25rn75qz4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/formal-grammar")
    (synopsis "Typeset formal grammars")
    (description
     "This package provides a new environment and associated commands to
typeset BNF grammars.  It allows easily writing formal grammars.  Its
original motivation was to typeset grammars for beamer presentations,
therefore, there are macros to emphasize or downplay some parts of the
grammar (which is the main novelty compared to other BNF packages).")
    (license license:lppl1.3+)))

(define-public texlive-formation-latex-ul
  (package
    (name "texlive-formation-latex-ul")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/formation-latex-ul/"
                                     "source/latex/formation-latex-ul/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1c8c7sdzfrqld0nq8s3xmdxlskws37mjg9z4ag9i82z4kr1ffb56"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/formation-latex-ul")
    (synopsis "Introductory LaTeX course in French")
    (description
     "This package contains the supporting documentation, slides, exercise files,
and templates for an introductory LaTeX course (in French) prepared for
Universite Laval, Quebec, Canada.")
    (license license:cc-by4.0)))

(define-public texlive-forum
  (package
    (name "texlive-forum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/forum/"
                                     "fonts/enc/dvips/forum/"
                                     "fonts/map/dvips/forum/"
                                     "fonts/opentype/public/forum/"
                                     "fonts/tfm/public/forum/"
                                     "fonts/type1/public/forum/"
                                     "fonts/vf/public/forum/"
                                     "tex/latex/forum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07sbzd2da3i5adlbgrrxd7igyailzjwxvr3pcwmb9wl9jry1gg9m"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/forum")
    (synopsis "Forum fonts with LaTeX support")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX and LuaLaTeX support for
the Forum font, designed by Denis Masharov.  Forum has antique, classic Roman
proportions.  It can be used to set body texts and works well in titles and
headlines too.  It is truly multilingual, with glyphs for Central and Eastern
Europe, Baltics, Cyrillic and Asian Cyrillic communities.  There is currently
just a regular weight and an artificially emboldened bold.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-fouridx
  (package
    (name "texlive-fouridx")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/fouridx/" "source/latex/fouridx/"
                                     "tex/latex/fouridx/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0mlinh47gcazi3icyvkixhdkwd1wkpffq8snabhh4bjsvh6m8sxz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fouridx")
    (synopsis "Left sub- and superscripts in maths mode")
    (description
     "The package enables left subscripts and superscripts in maths mode.  The
sub- and superscripts are raised for optimum fitting to the symbol indexed, in
such a way that left and right sub- and superscripts are set on the same
level, as appropriate.  The package provides an alternative to the use of the
@code{\\sideset} command in the @code{amsmath} package.")
    (license license:lppl)))

(define-public texlive-fouriernc
  (package
    (name "texlive-fouriernc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/fouriernc/"
                                     "fonts/afm/public/fouriernc/"
                                     "fonts/tfm/public/fouriernc/"
                                     "fonts/vf/public/fouriernc/"
                                     "tex/latex/fouriernc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gg9l6zvyi98jg1w7z33pxsnmh7lqjcs2zakhj3pbv2q7ysxnppw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/fouriernc")
    (synopsis "Use New Century Schoolbook text with Fourier maths fonts")
    (description
     "This package provides a LaTeX mathematics font setup for use with New
Century Schoolbook text.")
    (license license:lppl)))

(define-public texlive-francais-bst
  (package
    (name "texlive-francais-bst")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/francais-bst/"
                                     "doc/bibtex/francais-bst/"
                                     "source/bibtex/francais-bst/"
                                     "tex/latex/francais-bst/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07a0kqdwg9bzf1kgw11xn2344qdj52kgjc1iqz61hpirqlwfgw8s"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/francais-bst")
    (synopsis "Bibliographies conforming to French typographic standards")
    (description
     "The package provides bibliographies (in French) conforming to the rules
in @emph{Guide de la communication ecrite} (Malo, M., Quebec Amerique, 1996).
The BibTeX styles were generated using @code{custom-bib} and they are
compatible with @code{natbib}.")
    (license license:lppl1.3+)))

(define-public texlive-frcursive
  (package
    (name "texlive-frcursive")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/frcursive/"
                                     "fonts/map/dvips/frcursive/"
                                     "fonts/source/public/frcursive/"
                                     "fonts/tfm/public/frcursive/"
                                     "fonts/type1/public/frcursive/"
                                     "tex/latex/frcursive/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "04pqic407b0pfri5sydhbani0fxw0bxy3fl6x4isg2m2z73p4qqa"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/frcursive")
    (synopsis "French cursive hand fonts")
    (description
     "This package provides a hand-writing font in the style of the French
academic running-hand.  The font was written in Metafont and has been
converted to Adobe Type 1 format.  LaTeX support is provided.")
    (license license:lppl1.2+)))

(define-public texlive-frederika2016
  (package
    (name "texlive-frederika2016")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/frederika2016/"
                                     "fonts/opentype/public/frederika2016/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10pvx4n4ahn8b251fzz8crg9pijpclan5w4b6jc7m7jjsykzw8xj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/frederika2016")
    (synopsis "OpenType Greek calligraphy font")
    (description
     "Frederika2016 is an attempt to digitize Hermann Zapf's Frederika font.
The font is the Greek companion of Virtuosa by the same designer.  This font
is a calligraphy font and this is an initial release.")
    (license license:silofl1.1)))

(define-public texlive-freealign
  (package
    (name "texlive-freealign")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/freealign/"
                              "tex/latex/freealign/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0569wfag41n81cjpynylzgjp8lafm2lkq5ngfzqkqgicppq42b9p"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/freealign")
    (synopsis "Align math formulas in different lines")
    (description
     "This package provides several commands for aligning math formulas in
different lines.")
    (license license:lppl1.3c)))

(define-public texlive-frenchmath
  (package
    (name "texlive-frenchmath")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/frenchmath/"
                                     "source/latex/frenchmath/"
                                     "tex/latex/frenchmath/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1y1hqaw1ss37nzp5zas8axhbygzxr7qb06qqmqs51v8kbsdv8dbc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/frenchmath")
    (synopsis "Typesetting mathematics according to French rules")
    (description
     "The package provides:

@itemize
@item capital letters in roman (upright shape) in mathematical mode according
to French rule (can be optionally disabled),
@item optionally lowercase Greek letters in upright shape,
@item correct spacing in math mode after commas, before a semicolon and around
square brackets,
@item some useful macros and aliases for symbols used in France such as
@code{\\infeg}, @code{\\supeg}, @code{\\paral}, @item several macros for
writing french operator names like @emph{pgcd}, @emph{ppcm}, @emph{Card},
@emph{rg}, @emph{Vect}.
@end itemize")
    (license license:lppl1.3+)))

(define-public texlive-frimurer
  (package
    (name "texlive-frimurer")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/frimurer/"
                                     "fonts/afm/public/frimurer/"
                                     "fonts/enc/dvips/frimurer/"
                                     "fonts/tfm/public/frimurer/"
                                     "fonts/type1/public/frimurer/"
                                     "source/fonts/frimurer/"
                                     "tex/latex/frimurer/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "072mbcn6272rs1sw85xznk0f7psqh3dkz8y0hayn8y7a83vcrfgr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/frimurer")
    (synopsis "Access to the @emph{frimurer} cipher for use with LaTeX")
    (description
     "This package provides access to the @emph{frimurer} cipher for use with
LaTeX.")
    (license license:gpl3)))

(define-public texlive-frletter
  (package
    (name "texlive-frletter")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/frletter/" "tex/latex/frletter/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0h8lsfmv0icb616kwhkqqbfh64l43vwc63p4bl8sw9xy2lfal5xm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/frletter")
    (synopsis "Typeset letters in the French style")
    (description
     "This package provides a small class for typesetting letters in France.
No assumption is made about the language in use.  The class represents a small
modification of the @code{beletter} class, which is itself a modification of
the standard LaTeX @code{letter} class.")
    (license license:public-domain)))

(define-public texlive-frontespizio
  (package
    (name "texlive-frontespizio")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/frontespizio/"
                                     "source/latex/frontespizio/"
                                     "tex/latex/frontespizio/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17xqq0n2b0acpqq2hd5c5qi5fsrhyjwm1c452i3rbdmf1xkqvkvy"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/frontespizio")
    (synopsis "Create a frontispiece for Italian theses")
    (description
     "Typesetting a frontispiece independently of the layout of the main
document is difficult.  This package provides a solution by producing an
auxiliary TeX file to be typeset on its own and the result is automatically
included at the next run.  The markup necessary for the frontispiece is
written in the main document in a @code{frontespizio} environment.
Documentation is mainly in Italian, as the style is probably apt only to
theses in Italy.")
    (license license:lppl)))

(define-public texlive-frpseudocode
  (package
    (name "texlive-frpseudocode")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/frpseudocode/"
                                     "tex/latex/frpseudocode/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1jsrx1slbbqx9hg0a3f7xylxfly2cinw2g5vhhvvgxpp356v6ick"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/frpseudocode")
    (synopsis "French translation for the @code{algorithmicx} package")
    (description
     "This package is intended for use alongside @code{algorithmicx} package.
Its aim is to provide a French translation of terms and words used in
algorithms to make it integrate seamlessly in a French written document.")
    (license license:lppl1.3c)))

(define-public texlive-ftc-notebook
  (package
    (name "texlive-ftc-notebook")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ftc-notebook/"
                                     "tex/latex/ftc-notebook/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "068fc2wisryvgr8zqmkiwnyrrh5rmmw5njl41rn70bvv9qdgh31y"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ftc-notebook")
    (synopsis "Typeset FIRST Tech Challenge (FTC) notebooks")
    (description
     "This LaTeX package will greatly simplify filling entries for your FIRST
Tech Challenge (FTC) engineering or outreach notebook.  We developed this
package to support most frequently used constructs encountered in an FTC
notebook: meetings, tasks, decisions with pros and cons, tables, figures with
explanations, team stories and bios, and more.")
    (license license:lppl1.3+)))

(define-public texlive-functan
  (package
    (name "texlive-functan")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/functan/" "source/latex/functan/"
                                     "tex/latex/functan/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "077jfm145bnan63r1rlqlmy15zqj89fl4nwksdpyd67zcda3k1vf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/functan")
    (synopsis "Macros for functional analysis and PDE theory")
    (description
     "This package provides a convenient and coherent way to deal with name of
functional spaces (mainly Sobolev spaces) in functional analysis and PDE
theory.  It also provides a set of macros for dealing with norms, scalar
products and convergence with some object oriented flavor (it gives the
possibility to override the standard behavior of norms, ...).")
    (license license:lppl)))

(define-public texlive-gaceta
  (package
    (name "texlive-gaceta")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gaceta/" "tex/latex/gaceta/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17w2i3qmks9qpdwpx0hy0mjrl5xhrc8hmz9s0229xnhf6h12yb1r"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gaceta")
    (synopsis "Class to typeset @code{La Gaceta de la RSME}")
    (description
     "The class will typeset papers for @emph{La Gaceta de la Real Sociedad
Matematica Espanola}.")
    (license license:lppl)))

(define-public texlive-gamebook
  (package
    (name "texlive-gamebook")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gamebook/"
                                     "source/latex/gamebook/"
                                     "tex/latex/gamebook/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ksyr0hb1bfhc1lbbnzlj7ih8xw516djkn0lddnn07sb6hpzl8x4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gamebook")
    (synopsis "Typeset gamebooks and other interactive novels")
    (description
     "This package provides the means in order to lay-out gamebooks with LaTeX.
A simple gamebook example is included with the package, and acts as
a tutorial.")
    (license license:lppl1.3+)))

(define-public texlive-gamebooklib
  (package
    (name "texlive-gamebooklib")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gamebooklib/"
                                     "source/latex/gamebooklib/"
                                     "tex/latex/gamebooklib/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0j85xsvsw6xlkjr6z1qrbsld4kddhbcrbbsw9ci6azr6w47bh5vd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gamebooklib")
    (synopsis "Macros for setting numbered entries in shuffled order")
    (description
     "This package provides macros and environments to allow the user to
typeset a series of cross-referenced, numbered entries, shuffled into random
order, to produce an interactive novel or @dfn{gamebook}.  This allows entries
to be written in natural order and shuffled automatically into a repeatable
non-linear order.  Limited support is provided for footnotes to appear at the
natural position: the end of each entry, or the end of each page, whichever is
closest to the footnote mark.

This is unrelated to the @code{gamebook} package which is more concerned with
the formatting of entries rather than their order.  The two packages can be
used together or separately.")
    (license license:lppl1.3+)))

(define-public texlive-gammas
  (package
    (name "texlive-gammas")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/gammas/" "doc/latex/gammas/"
                                     "tex/latex/gammas/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jj8lbwlydsih01d637zxpj97ncd44qmf7azgpzn77s8fi760p8n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gammas")
    (synopsis "Template for the @emph{GAMM Archive for Students}")
    (description
     "This is the official document class for typesetting journal articles for
@emph{GAMM Archive for Students}, the open-access online journal run by the
@acronym{GAMM, Gesellschaft fur angewandte Mathematik und Mechanik} Juniors.")
    (license license:lppl1.3c)))

(define-public texlive-garamond-libre
  (package
    (name "texlive-garamond-libre")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/garamond-libre/"
                                     "fonts/enc/dvips/garamond-libre/"
                                     "fonts/map/dvips/garamond-libre/"
                                     "fonts/opentype/public/garamond-libre/"
                                     "fonts/tfm/public/garamond-libre/"
                                     "fonts/type1/public/garamond-libre/"
                                     "fonts/vf/public/garamond-libre/"
                                     "tex/latex/garamond-libre/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1djx6ill3yznkw2xbqshm9xy54fii2i71b9xalml0f7yb5fsbr6l"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/garamond-libre")
    (synopsis "Garamond Libre font face")
    (description
     "Garamond Libre is an old-style font family.  It is a true Garamond,
i.e., it is based off the designs of 16th-century French engraver Claude
Garamond (also spelled Garamont).  The Roman design is Garamond's; the italics
are from a design by Robert Granjon.  The upright Greek font is after a design
by Firmin Didot; the italic Greek font is after a design by Alexander Wilson.
The font family includes support for Latin, Greek (monotonic and polytonic)
and Cyrillic scripts, as well as small capitals, old-style figures, superior
and inferior figures, historical ligatures, Byzantine musical symbols, the IPA
and swash capitals.")
    (license (list license:expat license:lppl))))

(define-public texlive-garamond-math
  (package
    (name "texlive-garamond-math")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/garamond-math/"
                                     "fonts/opentype/public/garamond-math/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1ay8qmcn8glzn0bpg4qqs3k6dr9h7lljal239qvxfvg1fzj76h4n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/garamond-math")
    (synopsis "OTF math font matching EB Garamond")
    (description
     "Garamond-Math is an OpenType math font matching EB Garamond (Octavio
Pardo) and EB Garamond (Georg Mayr-Duffner).  Many mathematical symbols are
derived from other fonts, others are made from scratch.")
    (license license:silofl1.1)))

(define-public texlive-gastex
  (package
    (name "texlive-gastex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gastex/" "dvips/gastex/"
                                     "tex/latex/gastex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0l56wi6z5zm7sks2pa3yvamkfxsz53ziprnpr0ay141z0l8bd3rl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gastex")
    (synopsis "Graphs and Automata Simplified in TeX")
    (description
     "GasTeX is a set of LaTeX macros which enable the user to draw graphs, automata,
nets, diagrams, etc., very easily, in the LaTeX @code{picture} environment.")
    (license license:lppl)))

(define-public texlive-gb4e
  (package
    (name "texlive-gb4e")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gb4e/" "tex/latex/gb4e/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pi6n59v8vrrm2qaff8z28kwb0xa6nykgrfi9r0lxc8f1d2r5k7f"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gb4e")
    (synopsis "Linguistic tools")
    (description
     "This package provides an environment for linguistic examples, tools for
glosses, and various other goodies.")
    (license license:lppl1.2+)))

(define-public texlive-gb4e-next
  (package
    (name "texlive-gb4e-next")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/gb4e-next/"
                              "tex/latex/gb4e-next/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1386r7cylsm3kq8g3mvr77933gn1j6xvv6zdz737yr9wm75a8jpv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gb4e-next")
    (synopsis "Linguistic tools")
    (description
     "The package provides @code{gb4e} users two relative example reference
commands: @code{\\Next} refers to the next example in the document and
@code{\\Prev} refers to the previous example.  No explicit label command is
required.")
    (license license:lppl1.3+)))

(define-public texlive-gbt7714
  (package
    (name "texlive-gbt7714")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/gbt7714/" "doc/bibtex/gbt7714/"
                                     "source/bibtex/gbt7714/"
                                     "tex/latex/gbt7714/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0pmkbh0axh0qnc08fjn5g7pbw59n543siy4l9q3bihnyid11i5qp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-bibtex texlive-natbib texlive-url))
    (home-page "https://ctan.org/pkg/gbt7714")
    (synopsis "China's bibliography style standard GB/T 7714-2015 for BibTeX")
    (description
     "The package provides a BibTeX implementation for the Chinese national
bibliography style standard GB/T 7714-2015.  It consists of two @file{.bst}
files for numerical and author-year styles as well as a LaTeX package which
provides the citation style defined in the standard.

The package is compatible with @code{natbib} and supports language
detection (Chinese and English) for each biblilography entry.")
    (license license:lppl1.3c)))

(define-public texlive-gchords
  (package
    (name "texlive-gchords")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gchords/" "tex/latex/gchords/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0wchfk25v6rh5nffjyn8bypwjsqc9pi2rjrw3np65a0sxxc0sl1v"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gchords")
    (synopsis "Typeset guitar chords")
    (description
     "This package provides a LaTeX package for typesetting of guitar chord
diagrams, including options for chord names, finger numbers and typesetting
above lyrics.  The bundle also includes a TCL script (@file{chordbox.tcl})
that provides a graphical application which creates LaTeX files that use
@file{gchords.sty}.")
    (license license:gpl3+)))

(define-public texlive-gelasio
  (package
    (name "texlive-gelasio")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/fonts/gelasio/"
                              "fonts/enc/dvips/gelasio/"
                              "fonts/map/dvips/gelasio/"
                              "fonts/opentype/sorkin/gelasio/"
                              "fonts/tfm/sorkin/gelasio/"
                              "fonts/type1/sorkin/gelasio/"
                              "fonts/vf/sorkin/gelasio/"
                              "tex/latex/gelasio/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0kj91rpn4iw3cws4iq0cyd5vf38jg1893r3nc5mwm3kc5wd24s4g"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gelasio")
    (synopsis "LaTeX support for the Gelasio family of fonts")
    (description
     "This package provides (pdf)LaTeX, XeLaTeX and LuaLaTeX support for the
Gelasio family of fonts designed by Eben Sorkin to be metric-compatible with
Georgia in its Regular and Bold weights.  Medium and SemiBold weights have
been added.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-gelasiomath
  (package
    (name "texlive-gelasiomath")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/fonts/gelasiomath/"
                              "fonts/map/dvips/gelasiomath/"
                              "fonts/tfm/public/gelasiomath/"
                              "fonts/type1/public/gelasiomath/"
                              "fonts/vf/public/gelasiomath/"
                              "tex/latex/gelasiomath/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1zp9q6f3az9b373dlvdw2avmbl9bdjqypky6k0vgh5ppl6w2qmhp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gelasiomath")
    (synopsis "Math and small cap additions to Gelasio fonts")
    (description
     "The package offers math support for the @code{gelasio} package, using
symbols from @code{newtxmath}, Roman math letters from Gelasio and Greek math
letters based on XCharter Greek.  It also adds small caps based on XCharter
small caps and other minor features to Gelasio.")
    (license (list license:silofl1.1 license:lppl1.3+))))

(define-public texlive-gene-logic
  (package
    (name "texlive-gene-logic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gene-logic/"
                                     "tex/latex/gene-logic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13nibrf1f1w92ia7fj8yfa44w9c5wiyy1ndaby5p6bbjdjcxhyv9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gene-logic")
    (synopsis "Typeset logic formulae, etc")
    (description
     "The package provides a facility to typeset certain logic formulae.  It
provides an environment like @code{eqnarray}, a @code{newtheorem}-like
environment (NewTheorem), and several macros.")
    (license license:isc)))

(define-public texlive-genealogy
  (package
    (name "texlive-genealogy")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/genealogy/"
                                     "fonts/source/public/genealogy/"
                                     "fonts/tfm/public/genealogy/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "155ifff77s5cwjk1gkib7gcdx1mcrzzbca35mm8f9w5rzg1d5x7k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/genealogy")
    (synopsis "Compilation of genealogy fonts")
    (description
     "This package provides a simple compilation of the genealogical symbols
found in the @code{wasy} and @code{gen} fonts, adding the male and female
symbols to Knuth's @code{gen} font, and so avoiding loading two fonts when you
need only genealogical symbols.  The font is distributed as Metafont source.")
    (license license:lppl)))

(define-public texlive-genealogy-profiles
  (package
    (name "texlive-genealogy-profiles")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list
                         "doc/latex/genealogy-profiles/"
                         "tex/latex/genealogy-profiles/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0076nn6l5nl7vznzgxxd6xibipi60nzhp77xf5md8g84h4d4ybc7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/genealogy-profiles")
    (synopsis "Genealogical profiles for LaTeX")
    (description
     "This package enables the presentation of individual profiles, which may
be useful for genealogical or local history treatises.  Each profile is
typeset using key/value-configurable environments, and a number of macros are
provided to enable references and name formatting.")
    (license license:cc-by-sa4.0)))

(define-public texlive-geradwp
  (package
    (name "texlive-geradwp")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/geradwp/" "source/latex/geradwp/"
                                     "tex/latex/geradwp/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1g2kh2wdlf6y8wvs7r1zl2ws8ysrjzsk1k0032jci8ik9kx9wlg9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/geradwp")
    (synopsis "Document class for the @emph{Cahiers du GERAD} series")
    (description
     "This package provides the @code{geradwp} class, a class based on
@code{article} and compatible with LaTeX.  With this class, researchers at
GERAD will be able to write their working paper while complying to all the
presentation standards required by the @emph{Cahiers du GERAD} series.")
    (license license:lppl1.3c)))

(define-public texlive-geschichtsfrkl
  (package
    (name "texlive-geschichtsfrkl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/geschichtsfrkl/"
                                     "source/latex/geschichtsfrkl/"
                                     "tex/latex/geschichtsfrkl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1mplym0rkfl8pxzw31m1p117pbfz1s59180bhbwi2wgj4225p4w7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/geschichtsfrkl")
    (synopsis "BibLaTeX style for historians")
    (description
     "The package provides a BibLaTeX style, (mostly) meeting the requirements
of the History Faculty of the University of Freiburg (Germany).")
    (license license:lppl)))

(define-public texlive-gfdl
  (package
    (name "texlive-gfdl")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gfdl/" "source/latex/gfdl/"
                                     "tex/latex/gfdl/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1yiyhks3097sz686sar1jnkska669hp7v2sv1fm7n0d4j00zm816"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfdl")
    (synopsis "Support for using GFDL in LaTeX")
    (description
     "The @acronym{GFDL, GNU Free Documentation License} is a popular license
used for programming manuals, documentations and various other textual works
too, but using this license with LaTeX is not very convenient.  This package
aims to help users in easily using the license without violating any rules of
the license.  With a handful of commands, users can rest assured that their
document will be perfectly licensed under GFDL.")
    (license (list license:gpl3+ license:fdl1.3+))))

(define-public texlive-gfsartemisia
  (package
    (name "texlive-gfsartemisia")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gfsartemisia/"
                                     "fonts/afm/public/gfsartemisia/"
                                     "fonts/enc/dvips/gfsartemisia/"
                                     "fonts/map/dvips/gfsartemisia/"
                                     "fonts/opentype/public/gfsartemisia/"
                                     "fonts/tfm/public/gfsartemisia/"
                                     "fonts/type1/public/gfsartemisia/"
                                     "fonts/vf/public/gfsartemisia/"
                                     "tex/latex/gfsartemisia/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09kk65sn5wj70dksrywszdbwg2gg9wzdy9m25xj09d8k7hdirayn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfsartemisia")
    (synopsis "Modern Greek font design")
    (description
     "GFS Artemisia is a relatively modern font, designed as a general purpose
font in the same sense as Times is nowadays treated.  The font supports the
Greek and Latin alphabets.  LaTeX support is provided, using the OT1, T1 and
LGR encodings.")
    (license (list license:silofl1.1 license:lppl1.0+))))

(define-public texlive-gfsbodoni
  (package
    (name "texlive-gfsbodoni")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gfsbodoni/"
                                     "fonts/afm/public/gfsbodoni/"
                                     "fonts/enc/dvips/gfsbodoni/"
                                     "fonts/map/dvips/gfsbodoni/"
                                     "fonts/opentype/public/gfsbodoni/"
                                     "fonts/tfm/public/gfsbodoni/"
                                     "fonts/type1/public/gfsbodoni/"
                                     "fonts/vf/public/gfsbodoni/"
                                     "tex/latex/gfsbodoni/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0a4l6ijf7lqb8g7cz1fly3c73aq14pybr0cvpn86lsw3sg7cy9jk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfsbodoni")
    (synopsis "Greek and Latin font based on Bodoni")
    (description
     "Bodoni's Greek fonts in the 18th century broke, for the first time,
with the Byzantine cursive tradition of Greek fonts.  GFS Bodoni resurrects
his work for general use.  The font family supports both Greek and Latin
letters.  LaTeX support of the fonts is provided, offering OT1, T1 and LGR
encodings.  The fonts themselves are provided in Adobe Type 1 and OpenType
formats.")
    (license license:silofl1.1)))

(define-public texlive-gfscomplutum
  (package
    (name "texlive-gfscomplutum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gfscomplutum/"
                                     "fonts/afm/public/gfscomplutum/"
                                     "fonts/enc/dvips/gfscomplutum/"
                                     "fonts/map/dvips/gfscomplutum/"
                                     "fonts/opentype/public/gfscomplutum/"
                                     "fonts/tfm/public/gfscomplutum/"
                                     "fonts/type1/public/gfscomplutum/"
                                     "fonts/vf/public/gfscomplutum/"
                                     "tex/latex/gfscomplutum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "018hpaijnky23brhqmqsq2zm7h5w4y42vdfawba62qlz5m1zijn1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfscomplutum")
    (synopsis "Greek font with a long history")
    (description
     "GFS Complutum derives from a minuscule-only font cut in the 16th
century.  An unsatisfactory set of majuscules were added in the early 20th
century, but its author died before he could complete the revival of the font.
The Greek Font Society has released this version, which has a new set of
majuscules.")
    (license license:silofl1.1)))

(define-public texlive-gfsdidot
  (package
    (name "texlive-gfsdidot")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gfsdidot/"
                                     "fonts/afm/public/gfsdidot/"
                                     "fonts/enc/dvips/gfsdidot/"
                                     "fonts/map/dvips/gfsdidot/"
                                     "fonts/opentype/public/gfsdidot/"
                                     "fonts/tfm/public/gfsdidot/"
                                     "fonts/type1/public/gfsdidot/"
                                     "fonts/vf/public/gfsdidot/"
                                     "tex/latex/gfsdidot/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1sh1xv2fvb0y9vfzl0wl8fps4474fkvvh82fbqgfm83ri8h8p4ck"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfsdidot")
    (synopsis "Greek font based on Didot's work")
    (description
     "The design of Didot's 1805 Greek typeface was influenced by the
neoclassical ideals of the late 18th century.  The font was brought to Greece
at the time of the 1821 Greek Revolution, by Didot's son, and was very widely
used.  The font supports the Greek alphabet, and is accompanied by a matching
Latin alphabet based on Zapf's Palatino.  LaTeX support is provided, using the
OT1, T1, TS1, and LGR encodings.")
    (license license:silofl1.1)))

(define-public texlive-gfsdidotclassic
  (package
    (name "texlive-gfsdidotclassic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gfsdidotclassic/"
                                     "fonts/opentype/public/gfsdidotclassic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1cpsmgrq2x6ym9xb63msvm6pc4p2qyi2k40hmxmzhi979ch8gkaz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfsdidotclassic")
    (synopsis "Classic version of GFSDidot")
    (description
     "This is the classic version of GFSDidot provided for Unicode TeX
engines.")
    (license license:silofl1.1)))

(define-public texlive-gfsneohellenic
  (package
    (name "texlive-gfsneohellenic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gfsneohellenic/"
                                     "fonts/afm/public/gfsneohellenic/"
                                     "fonts/enc/dvips/gfsneohellenic/"
                                     "fonts/map/dvips/gfsneohellenic/"
                                     "fonts/opentype/public/gfsneohellenic/"
                                     "fonts/tfm/public/gfsneohellenic/"
                                     "fonts/type1/public/gfsneohellenic/"
                                     "fonts/vf/public/gfsneohellenic/"
                                     "tex/latex/gfsneohellenic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "109zgqj63ffryy42zi64p6r37jmzj5ma3d8bgn7mz3gc2scr1bpc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfsneohellenic")
    (synopsis "Font in the Neo-Hellenic style")
    (description
     "The Neo-Hellenic style evolved in academic circles in the 19th and 20th
century; the present font follows a cut commissioned from Monotype in 1927.
The font supports both Greek and Latin characters, and has been adjusted to
work well with the @code{cmbright} fonts for mathematics support.  LaTeX
support of the fonts is provided, offering OT1, T1 and LGR encodings.")
    (license license:silofl1.1)))

(define-public texlive-gfsneohellenicmath
  (package
    (name "texlive-gfsneohellenicmath")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gfsneohellenicmath/"
                                     "fonts/opentype/public/gfsneohellenicmath/"
                                     "tex/latex/gfsneohellenicmath/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "08yzv6qww3qvbg8bp4k8ibd46j1a41ysw4k8fi5nazchf3py7qb0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfsneohellenicmath")
    (synopsis "Math font in the Neo-Hellenic style")
    (description
     "The GFSNeohellenic font, a historic font first designed by Victor
Scholderer, now has native support for Mathematics.  A useful application is
in Beamer documents since this is a sans math font.")
    (license license:silofl1.1)))

(define-public texlive-gfssolomos
  (package
    (name "texlive-gfssolomos")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gfssolomos/"
                                     "fonts/afm/public/gfssolomos/"
                                     "fonts/enc/dvips/gfssolomos/"
                                     "fonts/map/dvips/gfssolomos/"
                                     "fonts/opentype/public/gfssolomos/"
                                     "fonts/tfm/public/gfssolomos/"
                                     "fonts/type1/public/gfssolomos/"
                                     "fonts/vf/public/gfssolomos/"
                                     "tex/latex/gfssolomos/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0zjbmnkfnaiq7pmpv0xhp5vag2dww70049knf759lf2s0ygnmaaa"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gfssolomos")
    (synopsis "Greek-alphabet font")
    (description
     "Solomos is a font which traces its descent from
a calligraphically-inspired font of the mid-19th century.  LaTeX support, for
use with the LGR encoding only, is provided.")
    (license license:silofl1.1)))

(define-public texlive-ghab
  (package
    (name "texlive-ghab")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ghab/"
                                     "fonts/source/public/ghab/"
                                     "tex/latex/ghab/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jknnlcigk6aa48xqvxd67015fxjlmmp36p52c95xrii4lv3m0zk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ghab")
    (synopsis "Typeset ghab boxes in LaTeX")
    (description
     "The package defines a command @code{\\darghab} that will typeset its
argument in a box with a decorated frame.  The width of the box may be set
using an optional argument.")
    (license license:lppl)))

(define-public texlive-ghsystem
  (package
    (name "texlive-ghsystem")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ghsystem/" "tex/latex/ghsystem/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1a8plly9zcbym67yzk6rp5mlk12lngmhdx6c3riar2w86nlvq55l"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ghsystem")
    (synopsis "Globally harmonised system of chemical (etc) naming")
    (description
     "The package provides the means to typeset all the hazard and
precautionary statements and pictograms in a straightforward way.  The
statements are taken from EU regulation 1272/2008.")
    (license license:lppl1.3+)))

(define-public texlive-gillcm
  (package
    (name "texlive-gillcm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gillcm/"
                                     "fonts/map/dvips/gillcm/"
                                     "fonts/tfm/public/gillcm/"
                                     "fonts/vf/public/gillcm/"
                                     "tex/latex/gillcm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0cz6aqcq2fv6hb2yv2ymc3j6myb8y73gyqbyigabwc909fmh5qdm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gillcm")
    (synopsis "Alternative unslanted italic Computer Modern fonts")
    (description
     "This is a demonstration of the use of virtual fonts for unusual effects:
the package implements unslanted italic Computer Modern fonts.")
    (license license:bsd-3)))

(define-public texlive-gillius
  (package
    (name "texlive-gillius")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gillius/"
                                     "fonts/enc/dvips/gillius/"
                                     "fonts/map/dvips/gillius/"
                                     "fonts/opentype/arkandis/gillius/"
                                     "fonts/tfm/arkandis/gillius/"
                                     "fonts/type1/arkandis/gillius/"
                                     "fonts/vf/arkandis/gillius/"
                                     "tex/latex/gillius/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0f2jr70ab5qnmla1n7iwhiw9x9q5fdzw9lcgjzji8rdqlsgq1451"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gillius")
    (synopsis "Gillius fonts with LaTeX support")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX and LuaLaTeX support for
the Gillius and Gillius No.@: 2 families of sans serif fonts and condensed
versions of them, designed by Hirwen Harendal.")
    (license (list license:gpl2+ license:lppl))))

(define-public texlive-glosmathtools
  (package
    (name "texlive-glosmathtools")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/glosmathtools/"
                                     "tex/latex/glosmathtools/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0rw194qxxf38fp1xczzjvsa6vb0jxz88cgc2s5fqx2k4fwg989kh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/glosmathtools")
    (synopsis
     "Mathematical nomenclature tools based on the @code{glossaries} package")
    (description
     "This package can be used to generate a mathematical nomenclature (also
called list of symbols or notation).  It is based on the @code{glossaries}
package.  Its main features are:

@itemize
@item symbol categories (e.g., latin, greek),
@item automatic but customizable symbol sorting,
@item easy subscript management,
@item easy accentuation management,
@item abbreviation support (with first use definition),
@item bilingual nomenclatures (for bilingual documents),
@item bilingual abbreviations.
@end itemize")
    (license license:lppl1.3c)))

(define-public texlive-gloss-occitan
  (package
    (name "texlive-gloss-occitan")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gloss-occitan/"
                                     "source/latex/gloss-occitan/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0h9w7gk7klwhackx30c7wa2xv23jy2r75zvgs22crhv40h8blj3f"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gloss-occitan")
    (synopsis "Polyglossia support for Occitan")
    (description
     "This package provides Occitan language description file for
Polyglossia.")
    (license license:lppl1.3+)))

(define-public texlive-glossaries
  (package
    (name "texlive-glossaries")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/glossaries/"
                                     "doc/man/man1/makeglossaries-lite.1"
                                     "doc/man/man1/makeglossaries-lite.man1.pdf"
                                     "doc/man/man1/makeglossaries.1"
                                     "doc/man/man1/makeglossaries.man1.pdf"
                                     "scripts/glossaries/"
                                     "source/latex/glossaries/"
                                     "tex/latex/glossaries/base/"
                                     "tex/latex/glossaries/expl/"
                                     "tex/latex/glossaries/rollback/"
                                     "tex/latex/glossaries/styles/"
                                     "tex/latex/glossaries/test-entries/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0q82hmvivly0g02pfwx49fvnrg0v7xf2b4ly6lccc1h1fym124vh"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-inputs "perl")))
    (build-system texlive-build-system)
    (arguments
     (list #:link-scripts #~(list "makeglossaries" "makeglossaries-lite.lua")))
    (inputs (list perl))
    (propagated-inputs
     (list texlive-amsmath
           texlive-datatool
           texlive-etoolbox
           texlive-mfirstuc
           texlive-tracklang
           texlive-xfor
           texlive-xkeyval))
    (home-page "https://ctan.org/pkg/glossaries")
    (synopsis "Create glossaries and lists of acronyms")
    (description
     "The glossaries package supports acronyms and multiple glossaries, and has
provision for operation in several languages (using the facilities of either
Babel or Polyglossia).  New entries are defined to have a name and description
(and optionally an associated symbol).  Support for multiple languages is
offered, and plural forms of terms may be specified.  An additional package,
@code{glossaries-accsupp}, can make use of the @code{accsupp} package
mechanisms for accessibility support for PDF files containing glossaries.  The
user may define new glossary styles, and preambles and postambles can be
specified.  There is provision for loading a database of terms, but only terms
used in the text will be added to the relevant glossary.

The package uses an indexing program to provide the actual glossary; either
MakeIndex or Xindy may serve this purpose, and a Perl script is provided to
serve as interface.  The package supersedes @code{glossary} package (which is
now obsolete).")
    (license license:lppl1.3+)))

(define-public texlive-gmverse
  (package
    (name "texlive-gmverse")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gmverse/" "tex/latex/gmverse/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13vs1w9pfl4is5f5papwqqm83kmb17d4z7gv8nxkswnav9v5cl56"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gmverse")
    (synopsis "Package for typesetting (short) poems")
    (description
     "This package provides a redefinition of the @code{verse} environment to
make the @code{\\\\} command optional for line ends and to give it
a possibility of optical centering and right-hanging alignment of lines broken
because of length.")
    (license license:lppl)))

(define-public texlive-gnu-freefont
  (package
    (name "texlive-gnu-freefont")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gnu-freefont/"
                                     "fonts/opentype/public/gnu-freefont/"
                                     "fonts/truetype/public/gnu-freefont/"
                                     "source/fonts/gnu-freefont/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14bq8i7n6zhy8352pmhd5d9l4p152c6bgarz17wfp2p00kzs47aw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gnu-freefont")
    (synopsis "Unicode font, with rather wide coverage")
    (description
     "The package provides a set of outline (i.e., OpenType} fonts covering as
much as possible of the Unicode character set.  The set consists of three
typefaces: one monospaced and two proportional (one with uniform and one with
modulated stroke).")
    (license license:gpl3)))

(define-public texlive-go
  (package
    (name "texlive-go")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/go/" "fonts/source/public/go/"
                                     "fonts/tfm/public/go/" "source/fonts/go/"
                                     "tex/latex/go/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1cvxfz9m7fx62iiz00f7qlywrmwwnpk0xzlyv63c90ji9xzfawcv"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/go")
    (synopsis "Fonts and macros for typesetting go games")
    (description
     "The macros provide for nothing more complicated than the standard
19x19 board; the fonts are written in Metafont.")
    (license license:public-domain)))

(define-public texlive-gofonts
  (package
    (name "texlive-gofonts")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gofonts/"
                                     "fonts/enc/dvips/gofonts/"
                                     "fonts/map/dvips/gofonts/"
                                     "fonts/tfm/bh/gofonts/"
                                     "fonts/truetype/bh/gofonts/"
                                     "fonts/type1/bh/gofonts/"
                                     "fonts/vf/bh/gofonts/"
                                     "tex/latex/gofonts/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07ysxm10xwg6cc781gdppidq1s87bqli4x6shkarjl29pxcncn61"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gofonts")
    (synopsis "GoSans and GoMono fonts with LaTeX support")
    (description
     "This package provides LaTeX, pdfLaTeX, XeLaTeX and LuaLaTeX support for
the GoSans and GoMono families of fonts designed by the Bigelow & Holmes
foundry.  GoSans is available in three weights: Regular, Medium, and
Bold (with corresponding italics).  GoMono is available in regular and bold,
with italics.")
    (license (list license:bsd-3 license:lppl))))

(define-public texlive-gost
  (package
    (name "texlive-gost")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/gost/" "bibtex/csf/gost/"
                                     "doc/bibtex/gost/" "source/bibtex/gost/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0rsqk4r1r741ggvpgg7g51knlaqrrdq9g8yiix66vx3n5v1arp26"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gost")
    (synopsis "BibTeX styles to format according to GOST")
    (description
     "This package provides BibTeX styles to format bibliographies in English,
Russian or Ukrainian according to GOST 7.0.5-2008 or GOST 7.1-2003.  Both
8-bit and Unicode (UTF-8) versions of each @code{BibTeX} style, in each case
offering a choice of sorted and unsorted.  Further, a set of three
styles (which do not conform to current standards) are retained for backwards
compatibility.")
    (license license:lppl1.3c)))

(define-public texlive-gothic
  (package
    (name "texlive-gothic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gothic/"
                                     "fonts/source/public/gothic/"
                                     "fonts/tfm/public/gothic/"
                                     "source/fonts/gothic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "08hqp2a0ch67d38p2jca6cvx72vc6dagb04a9w6ff7dniqp6gps2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    ;; FIXME: Font metrics generation fails with "! Strange path (turning
    ;; number is zero)." error.
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               (delete 'generate-font-metrics))))
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/gothic")
    (synopsis "Collection of old German-style fonts")
    (description
     "This package provides a collection of fonts that reproduce those used in
old German printing and handwriting.  The set comprises Gothic, Schwabacher
and Fraktur fonts, a pair of handwriting fonts, Sutterlin and Schwell, and
a font containing decorative initials.  In addition, there are two re-encoding
packages for Haralambous's fonts, providing T1, using virtual fonts, and OT1
and T1, using Metafont.")
    ;; This is a collection of packages, with various licenses.
    (license (list license:lppl1.2+ license:lppl1.3c license:public-domain))))

(define-public texlive-gotoh
  (package
    (name "texlive-gotoh")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gotoh/" "source/latex/gotoh/"
                                     "tex/latex/gotoh/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0h08ygzlhv1mpd27yfv8slvgfwa250z7alpv99p6768jafysp3k9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gotoh")
    (synopsis "Implementation of the Gotoh sequence alignment algorithm")
    (description
     "This package calculates biological sequence alignment with the Gotoh algorithm.
The package also provides an interface to control various settings including
algorithm parameters.")
    (license license:expat)))

(define-public texlive-gradstudentresume
  (package
    (name "texlive-gradstudentresume")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gradstudentresume/"
                                     "tex/latex/gradstudentresume/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0jimvivx5vf1jy7yszvx2zbf13r0brqc0avz8lcx1rsqa3d7rjcn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gradstudentresume")
    (synopsis "Generic template for graduate student resumes")
    (description
     "The package offers a template for graduate students writing an academic
CV.  The goal is to create a flexible template that can be customized based on
each specific individual's needs.")
    (license license:lppl1.3+)))

(define-public texlive-grant
  (package
    (name "texlive-grant")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/grant/" "source/latex/grant/"
                                     "tex/latex/grant/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0cihhr3fqjbn2grkps5wl6aqx19s78vgklmriw2jqg27fqgv5q2p"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/grant")
    (synopsis "Classes for formatting federal grant proposals")
    (description
     "This package provides LaTeX classes for formatting federal grant
proposals:
@itemize
@item grant: base class for formatting grant proposals;
@item grant-arl: Army Research Laboratory;
@item grant-darpa: Defense Advanced Research Projects Agency;
@item grant-doe: Department of Energy;
@item grant-nih: National Institutes of Health;
@item grant-nrl: Naval Research Laboratory;
@item grant-nsf: National Science Foundation;
@item grant-onr: Office of Naval Research.
@end itemize")
    (license license:expat)))

(define-public texlive-greenpoint
  (package
    (name "texlive-greenpoint")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/greenpoint/"
                                     "fonts/source/public/greenpoint/"
                                     "fonts/tfm/public/greenpoint/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1y5an7lgx975ppb8s1abx494s4m115k137f82hy08iwdx9l1plj5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/greenpoint")
    (synopsis "Green Point logo")
    (description
     "This package provides a Metafont-implementation of the logo commonly
known as @emph{Der Grune Punkt} (``The Green Point'').  In Austria, it can be
found on nearly every bottle.  It should not be confused with the Recycle
logo.")
    (license license:gpl3+)))

(define-public texlive-gregoriotex
  (package
    (name "texlive-gregoriotex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/luatex/gregoriotex/"
                                     "fonts/source/gregoriotex/"
                                     "fonts/truetype/public/gregoriotex/"
                                     "scripts/gregoriotex/"
                                     "source/luatex/gregoriotex/"
                                     "tex/lualatex/gregoriotex/"
                                     "tex/luatex/gregoriotex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0bs1fkdn1mikiam52m4bql3mcvbv6x7g9qy19qkmf4ycd46zx25m"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (propagated-inputs (list texlive-gregoriotex-bin))
    (home-page "https://ctan.org/pkg/gregoriotex")
    (synopsis "Engraving gregorian chant scores")
    (description
     "Gregorio is a software application for engraving gregorian chant scores
on a computer.  Gregorio's main job is to convert a gabc file (simple text
representation of a score) into a GregorioTeX file, which makes TeX able to
create a PDF of your score.")
    (license license:gpl3)))

(define-public texlive-gregoriotex-bin
  (package
    (inherit texlive-bin)
    (name "texlive-gregoriotex-bin")
    (source
     (origin
       (inherit (package-source texlive-source))
       (modules '((guix build utils)
                  (ice-9 ftw)))
       (snippet
        #~(let ((delete-other-directories
                 (lambda (root dirs)
                   (with-directory-excursion root
                     (for-each
                      delete-file-recursively
                      (scandir "."
                               (lambda (file)
                                 (and (not (member file (append '("." "..") dirs)))
                                      (eq? 'directory (stat:type (stat file)))))))))))
            (delete-other-directories "libs" '())
            (delete-other-directories "utils" '())
            (delete-other-directories "texk" '("gregorio"))))))
    (arguments
     (substitute-keyword-arguments (package-arguments texlive-bin)
       ((#:configure-flags flags)
        #~(cons "--enable-gregorio" (delete "--enable-web2c" #$flags)))
       ((#:phases phases)
        #~(modify-phases #$phases
            (replace 'check
              (lambda* (#:key tests? #:allow-other-keys)
                (when tests?
                  (with-directory-excursion "texk/gregorio"
                    (invoke "make" "check")))))
            (replace 'install
              (lambda _
                (with-directory-excursion "texk/gregorio"
                  (invoke "make" "install"))))))))
    (native-inputs (list pkg-config))
    (inputs (list texlive-libkpathsea))
    (propagated-inputs '())
    (home-page (package-home-page texlive-gregoriotex))
    (synopsis "Binary for @code{texlive-gregoriotex}")
    (description
     "This package provides the binary for @code{texlive-gregoriotex}.")
    (license (package-license texlive-gregoriotex))))

(define-public texlive-grotesq
  (package
    (name "texlive-grotesq")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/grotesq/"
                                     "fonts/afm/urw/grotesq/"
                                     "fonts/map/dvips/grotesq/"
                                     "fonts/tfm/urw/grotesq/"
                                     "fonts/type1/urw/grotesq/"
                                     "fonts/vf/urw/grotesq/"
                                     "tex/latex/grotesq/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12q0n6z442j725drp3919k2qk7kg9fas4cqz415a8lj1nmvg60i4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/urw-grotesq")
    (synopsis "URW Grotesq font pack for LaTeX")
    (description
     "The directory contains a copy of the Type 1 font URW Grotesq 2031
Bold, with supporting files for use with (La)TeX.")
    (license license:gpl3+)))

(define-public texlive-grundgesetze
  (package
    (name "texlive-grundgesetze")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/grundgesetze/"
                                     "source/latex/grundgesetze/"
                                     "tex/latex/grundgesetze/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0hpl336bn33qjmq2bhqn94fjdxlcxs0lxm4sdr124dnagdnksl4n"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/grundgesetze")
    (synopsis "Typeset Frege's @emph{Grundgesetze der Arithmetik}")
    (description
     "The package defines maths mode commands for typesetting Gottlob Frege's
concept-script in the style of his @emph{Grundgesetze der Arithmetik} (Basic
Laws of Arithmetic).")
    (license license:gpl2)))

(define-public texlive-gsemthesis
  (package
    (name "texlive-gsemthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gsemthesis/"
                                     "source/latex/gsemthesis/"
                                     "tex/latex/gsemthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0wwd6pddxb91gj6jr6zc0xy0sr0s0mbdjzyw856w9ds0lp0k6cl9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gsemthesis")
    (synopsis "Geneva School of Economics and Management PhD thesis format")
    (description
     "The class provides a PhD thesis template for the @acronym{GSEM,
Geneva School of Economics and Management}, University of Geneva, Switzerland.
The class provides utilities to easily set up the cover page, the front matter
pages, the page headers, etc., conformant to the official guidelines of the
GSEM Faculty for writing PhD dissertations.")
    (license license:lppl1.3+)))

(define-public texlive-gtrcrd
  (package
    (name "texlive-gtrcrd")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gtrcrd/" "tex/latex/gtrcrd/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0ndnzkm84ynn3wcnfg6j6fcl87wmd3g9w5d17g1z6qp340asrkp9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gtrcrd")
    (synopsis "Add chords to lyrics")
    (description
     "The package provides the means to specify guitar chords to be played
with each part of the lyrics of a song.  The syntax of the macros reduces the
chance of failing to provide a chord where one is needed, and the structure of
the macros ensures that the chord specification appears immediately above the
start of the lyric.")
    (license license:lppl1.3+)))

(define-public texlive-gu
  (package
    (name "texlive-gu")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gu/" "tex/latex/gu/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13gw8gk03s3ha7xwn6vfix7w50if3gvlqywn1l6z5zi3qbfaapvr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gu")
    (synopsis "Typeset crystallographic group-subgroup-schemes")
    (description
     "The package simplifies typesetting of simple crystallographic
group-subgroup-schemes in the Barnighausen formalism.  It defines a new
environment @code{stammbaum}, wherein all elements of the scheme are defined.
Afterwards all necessary dimensions are calculated and the scheme is drawn.
Currently two steps of symmetry reduction are supported.")
    (license license:lppl)))

(define-public texlive-gudea
  (package
    (name "texlive-gudea")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/gudea/"
                                     "fonts/enc/dvips/gudea/"
                                     "fonts/map/dvips/gudea/"
                                     "fonts/tfm/public/gudea/"
                                     "fonts/type1/public/gudea/"
                                     "fonts/vf/public/gudea/"
                                     "tex/latex/gudea/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kj53idgxyqwishg0vibhhrj443016z61c0jl7rx91a508vbwbn2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gudea")
    (synopsis "Gudea font face with support for LaTeX and pdfLaTeX")
    (description
     "This package provides the Gudea family of fonts designed by Agustina
Mingote, with support for LaTeX and pdfLaTeX.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-guide-to-latex
  (package
    (name "texlive-guide-to-latex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/guide-to-latex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wc5pclv27af2zr6kjjahjzxxlab31d8970jvxr9lxiqllvkxyab"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/guide-to-latex")
    (synopsis "Examples and more from @emph{Guide to LaTeX}, by Kopka and Daly")
    (description
     "This package contains material presented in the book @emph{Guide to LaTeX},
4th edition, by Helmut Kopka and Patrick W. Daly as code, sample figures,
processed files, as well as solutions to the exercises.")
    (license license:lppl1.3c+)))

(define-public texlive-guitar
  (package
    (name "texlive-guitar")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/guitar/" "source/latex/guitar/"
                                     "tex/latex/guitar/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0rywmlz59mjm59n4607qk9fa62w1c9qv9iyyja8k9vb4pc9yijrc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/guitar")
    (synopsis "Guitar chords and song texts")
    (description
     "This package provides (La)TeX macros for typesetting guitar chords over
song texts.  Note that this package only places arbitrary TeX code over the
lyrics.  To typeset the chords graphically (and not only by name), the author
recommends use of an additional package such as @code{gchords}.")
    (license license:lppl1.3+)))

(define-public texlive-guitarchordschemes
  (package
    (name "texlive-guitarchordschemes")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/guitarchordschemes/"
                                     "tex/latex/guitarchordschemes/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "189xx0y549ss4ip8x2f1rpcqyad5njrni785sxmy2cghqbdcgzzn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/guitarchordschemes")
    (synopsis "Guitar chord and scale tablatures")
    (description
     "This package provides two commands (@code{\\chordscheme} and
@code{\\scales}).  With those commands it is possible to draw schematic
diagrams of guitar chord tablatures and scale tablatures.  Both commands know
a range of options that allow wide customization of the output.")
    (license license:lppl1.3+)))

(define-public texlive-guitartabs
  (package
    (name "texlive-guitartabs")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/guitartabs/"
                                     "tex/latex/guitartabs/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1hk5yl7za7mm6cpg9g4z5wnq4m2nwm41sq8w7m8ic38sr85n67zz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/guitartabs")
    (synopsis "Class for drawing guitar tablatures easily")
    (description
     "This package provides is a simple LaTeX2e class that allows guitarists
to create basic guitar tablatures using LaTeX.")
    (license license:lppl1.3c)))

(define-public texlive-guitlogo
  (package
    (name "texlive-guitlogo")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/guitlogo/"
                                     "source/latex/guitlogo/"
                                     "tex/latex/guitlogo/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0fwp3w5b51qs9jr9xq0hl0hsqbx7dkj7qgwjpay88sflycv3qpba"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/guitlogo")
    (synopsis "Macros for typesetting the GuIT logo")
    (description
     "This package provides some commands useful to correctly write the logo
of @emph{Gruppo Utilizzatori Italiani di TeX} (Italian TeX User Group), using
the default document color or any other color the user may ever choose, in
conformity with the logo's scheme as seen on the group's website
@url{https://www.guitex.org}.  Likewise, commands are provided that simplify
the writing of the GuIT acronym's complete expansion, of the addresses of the
group's internet site and public forum, and the meeting GuITmeeting and the
magazine @emph{Ars TeXnica}'s logo.  Optionally, using @code{hyperref}, the
outputs of the above cited commands can become hyperlinks to the group's
website @url{https://www.guitex.org}.  The Documentation is available in
Italian only.")
    (license license:lppl1.3a)))

(define-public texlive-gustlib
  (package
    (name "texlive-gustlib")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bib/gustlib/" "bibtex/bst/gustlib/"
                                     "doc/plain/gustlib/" "tex/plain/gustlib/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1jfmsx5zw9yb8fkhw4fad82m48n7fs1inmx471mr7ys0i7y1l92v"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gustlib")
    (synopsis "Plain macros for much core and extra functionality, from GUST")
    (description
     "This package includes Plain TeX macros adding extra functionalities.
This comprises bibliography support, token manipulation, cross-references,
verbatim, determining length of a paragraph's last line, multicolumn output,
Polish bibliography and index styles, prepress and color separation, graphics
manipulation, and tables.")
    (license (list license:knuth license:public-domain))))

(define-public texlive-gustprog
  (package
    (name "texlive-gustprog")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/support/gustprog/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07qx4xvyqb4p0cx0macjgyg0pcxgcmdxmzfsx9ah5s0dqvminwrc"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gustprog")
    (synopsis "Utility programs for Polish users of TeX")
    (description
     "This package provides utility programs for Polish users of TeX.  These
programs are provided as sources, not installed in the @file{bin}
directories.")
    (license license:public-domain)))

(define-public texlive-gzt
  (package
    (name "texlive-gzt")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/gzt/" "source/latex/gzt/"
                                     "tex/latex/gzt/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "01vzqnh7fxpsd23q98n9x828lyjrvjlfkb9zv0sxydamp5qy3pn5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/gzt")
    (synopsis "Bundle of classes for @emph{La Gazette des Mathematiciens}")
    (description
     "This bundle provides two classes and BibLaTeX styles for the French
journal @emph{La Gazette des Mathematiciens}: @code{gzt} for the complete
issues of the journal, aimed at the Gazette's team, @code{gztarticle},
intended for authors who wish to publish an article in the Gazette.  This
class's goals are to faithfully reproduce the layout of the Gazette, thus
enabling the authors to be able to work their document in actual conditions,
and provide a number of tools (commands and environments) to facilitate the
drafting of documents, in particular those containing mathematical formulas.")
    (license license:lppl1.3c)))

(define-public texlive-h2020proposal
  (package
    (name "texlive-h2020proposal")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/h2020proposal/"
                                     "tex/latex/h2020proposal/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0fq43vpq3ixj6fc99fbmj129487xxzcmz22gisqmn469bam5qxj5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/h2020proposal")
    (synopsis "LaTeX class and template for EU H2020 RIA proposal")
    (description
     "This package consists of a class file as well as FET and ICT proposal
templates for writing EU H2020 RIA proposals and generating automatically the
many cross-referenced tables that are required.")
    (license license:gpl3)))

(define-public texlive-hacm
  (package
    (name "texlive-hacm")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/hacm/"
                                     "fonts/map/dvips/hacm/"
                                     "fonts/tfm/public/hacm/"
                                     "fonts/type1/public/hacm/"
                                     "fonts/vf/public/hacm/"
                                     "tex/latex/hacm/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "19n0mlb96ix4vlqaw95fgwah7mbn80l6jm1dim8sigagmhh3bimd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hacm")
    (synopsis "Font support for the Arka language")
    (description
     "The package supports typesetting @emph{hacm}, the alphabet of the
constructed language Arka.  The bundle provides nine official fonts, in Adobe
Type 1 format.")
    (license license:lppl1.3+)))

(define-public texlive-hagenberg-thesis
  (package
    (name "texlive-hagenberg-thesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hagenberg-thesis/"
                                     "tex/latex/hagenberg-thesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gjx3ii0gxpjaqga65qn7xacpfjbg8cimpw1qd507zbly6gqsa52"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hagenberg-thesis")
    (synopsis
     "LaTeX classes, style files and example documents for academic
manuscripts")
    (description
     "This package provides a collection of modern LaTeX classes, style files
and example documents for authoring Bachelor, master or diploma theses and
related academic manuscripts in English and German.  It includes comprehensive
tutorials (in English and German) with detailed instructions and authoring
guidelines.")
    (license license:cc-by4.0)))

(define-public texlive-hamnosys
  (package
    (name "texlive-hamnosys")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/hamnosys/"
                                     "fonts/truetype/public/hamnosys/"
                                     "source/fonts/hamnosys/"
                                     "tex/latex/hamnosys/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1iilp1npy23azk7gjb3m6g9p9f853lbpn9cvply90g2s9jdw0bd9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hamnosys")
    (synopsis "Font for sign languages")
    (description
     "The Hamburg Notation System, HamNoSys for short, is a system for the
phonetic transcription of signed languages.  This package makes HamNoSys
available in XeLaTeX and LuaLaTeX.  The package provides a Unicode font for
rendering HamNoSys symbols as well as three methods for entering them.")
    (license license:lppl1.3c)))

(define-public texlive-hands
  (package
    (name "texlive-hands")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "fonts/source/public/hands/"
                                     "fonts/tfm/public/hands/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0x0vdn7hq6k7wr7yxn8pnkvjhjq3mfl781ijrs2mpvjqygqffzag"))))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/hands")
    (synopsis "Pointing hand font")
    (description
     "This package provides right- and left-pointing hands in both
black-on-white and white-on-black realisation.  The font is distributed as
Metafont source.")
    (license license:public-domain)))

(define-public texlive-hanoi
  (package
    (name "texlive-hanoi")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "tex/plain/hanoi/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "09a7cv76naxzdach5507wdqnjp12amvlia7kw0jh224ydmkzfx9x"))))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hanoi")
    (synopsis "Tower of Hanoi in TeX")
    (description
     "The Plain TeX program (typed in the shape of the towers of Hanoi) serves
both as a game and as a TeX programming exercise.  As a game, it will solve
the towers with (up to) 15 discs.")
    (license license:public-domain)))

(define-public texlive-happy4th
  (package
    (name "texlive-happy4th")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/plain/happy4th/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1x950scxbvcgwycpakflpklc775pknjab620g099dnsfrpb76f4a"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/happy4th")
    (synopsis "Firework display in obfuscated TeX")
    (description
     "The output PDF file gives an amusing display, as the reader pages
through it.")
    (license license:public-domain)))

(define-public texlive-har2nat
  (package
    (name "texlive-har2nat")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/har2nat/" "tex/latex/har2nat/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "13akhwjx48lkch46pk2syzizham3rk81ihrzbwxsvapa1fw74dzg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/har2nat")
    (synopsis "Replace the @code{harvard} package with @code{natbib}")
    (description
     "This small package allows a LaTeX document containing the citation
commands provided by the @code{harvard} package to be compiled using the
@code{natbib} package.")
    (license license:lppl)))

(define-public texlive-harmony
  (package
    (name "texlive-harmony")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/harmony/" "tex/latex/harmony/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0xh0h8wpypbi0z7x1i30b0fm85ivyh2mvmmqjawfxrixw7mczpns"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/harmony")
    (synopsis "Typeset harmony symbols, etc., for musicology")
    (description
     "The package @file{harmony.sty} helps typesetting harmony symbols for
musicology.")
    (license license:lppl)))

(define-public texlive-hanzibox
  (package
    (name "texlive-hanzibox")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/xelatex/hanzibox/"
                                     "source/xelatex/hanzibox/"
                                     "tex/xelatex/hanzibox/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1bgbybzz5h79i2l3mbjxm11x9nsxm366c6d2mha9j5qabjghakya"))))
    (outputs '("out" "doc"))
    (properties '((updater-extra-native-inputs "texlive-ctex")))
    (build-system texlive-build-system)
    (native-inputs (list texlive-ctex))
    (home-page "https://ctan.org/pkg/hanzibox")
    (synopsis
     "Boxed Chinese characters with Pinyin above and translation below")
    (description
     "This is a LaTeX package written to simplify the input of Chinese with
Hanyu Pinyin and translation.  Hanyu Pinyin is placed above Chinese with the
@code{xpinyin} package, and the translation is placed below.  The package can
be used as a utility for learning to write and pronounce Chinese characters,
for Chinese character learning plans, presentations, exercise booklets and
other documentation work.")
    (license license:lppl1.3c)))

(define-public texlive-harvard
  (package
    (name "texlive-harvard")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bib/harvard/" "bibtex/bst/harvard/"
                                     "doc/latex/harvard/" "source/latex/harvard/"
                                     "tex/latex/harvard/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1qcw1rrvcgcz26pwcmsfd7gs4bzlq1n4zws92xqsr5fi95pr8q7k"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/harvard")
    (synopsis "Harvard citation package for use with LaTeX")
    (description
     "This is a re-implementation, for LaTeX, of the original Harvard package.
The bundle contains the LaTeX package, several BibTeX styles, and a Perl
package for use with LaTeX2HTML.

Harvard is an author-year citation style (all but the first author are
suppressed in second and subsequent citations of the same entry); the package
defines several variant styles: @file{apsr.bst} for the @emph{American
Political Science Review}; @file{agsm.bst} for Australian government
publications; @file{dcu.bst} from the Design Computing Unit of the University
of Sydney; @file{kluwer.bst}, which aims at the format preferred in Kluwer
publications; @file{nederlands.bst} which deals with sorting Dutch names with
prefixes (such as van) according to Dutch rules, together with several styles
whose authors offer no description of their behaviour.")
    (license license:lppl)))

(define-public texlive-harvmac
  (package
    (name "texlive-harvmac")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/plain/harvmac/" "tex/plain/harvmac/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0bxa9vvkg79xk3nakdc5an2py6gwbv9ml66xmz1hp1llbg3562vf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/harvmac")
    (synopsis "Macros for scientific articles")
    (description "This package provides macros for scientific articles.")
    (license license:cc-by3.0)))

(define-public texlive-havannah
  (package
    (name "texlive-havannah")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/havannah/"
                                     "source/latex/havannah/"
                                     "tex/latex/havannah/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kw71rr5jhn0gx89jynwxxgd6ddzpmr0wb5qnsh0drljmacq49ai"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/havannah")
    (synopsis "Diagrams of board positions in the games of Havannah and Hex")
    (description
     "This package defines macros for typesetting diagrams of board positions
in the games of Havannah and Hex.")
    (license license:lppl1.2+)))

(define-public texlive-hduthesis
  (package
    (name "texlive-hduthesis")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/hduthesis/"
                              "source/latex/hduthesis/"
                              "tex/latex/hduthesis/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "10d2hw98k0gf6kp3mnsrlwi5dz5gi40845n8y82g5bwgnvwpb09s"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hduthesis")
    (synopsis
     "LaTeX class for bachelor and MPhil theses in Hangzhou Dianzi University")
    (description
     "This package provides a LaTeX template for graduation theses from
Hangzhou Dianzi University.  It supports the formatting of bachelor and MPhil
degree theses.")
    (license license:lppl1.3c)))

(define-public texlive-hebdomon
  (package
    (name "texlive-hebdomon")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/hebdomon/"
                              "tex/latex/hebdomon/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1jlcf1qhxd7wmv808kfysipsa6ni41hcpnhbrhr8zpxcg6iskncj"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hebdomon")
    (synopsis "Document class for report writing in university settings")
    (description
     "This package provides a document class for streamlining document
creation in LaTeX.  It does not overwrite any TeX or LaTeX commands so the
user could use their own macros or other commands as they wish.")
    (license license:lppl1.3c)))

(define-public texlive-hebrew-fonts
  (package
    (name "texlive-hebrew-fonts")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/hebrew-fonts/"
                              "source/latex/hebrew-fonts/"
                              "tex/latex/hebrew-fonts/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1ka8lrlvb48yz7p8ypj7d4kk8ga4afdqv5n8gqamkwbdf27aqcby"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hebrew-fonts")
    (synopsis
     "Input encodings, font encodings and font definition files for Hebrew")
    (description
     "This package provides a collection of input encodings, font encodings
and font definition files for the Hebrew language.")
    (license license:lppl1.3+)))

(define-public texlive-hecthese
  (package
    (name "texlive-hecthese")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hecthese/"
                                     "source/latex/hecthese/"
                                     "tex/latex/hecthese/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "167bbbfnzjdw6rxvqb49szn9gf2x9v4m47yxs32ragbhzn4rvixw"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hecthese")
    (synopsis "Class for dissertations and theses at HEC Montreal")
    (description
     "This package provides the @code{hecthese} class, a class based on
@code{memoir} and compatible with LaTeX.  Using this class, postgraduate
students at HEC Montreal will be able to write their dissertation or thesis
while complying with all the presentation standards required by the
University.  This class is meant to be as flexible as possible; in particular,
there are very few hard-coded features except those that take care of the
document's layout.

Dissertations and theses at HEC Montreal can be written on a per-chapter or
per-article basis.  Documents that are written on a per-article basis require
a bibliography for each of the included articles and a general bibliography
for the entire document.  The @code{hecthese} class takes care of these
requirements.")
    (license license:lppl1.3c)))

(define-public texlive-helmholtz-ellis-ji-notation
  (package
    (name "texlive-helmholtz-ellis-ji-notation")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/helmholtz-ellis-ji-notation/"
                                     "fonts/opentype/public/helmholtz-ellis-ji-notation/"
                                     "source/fonts/helmholtz-ellis-ji-notation/"
                                     "tex/latex/helmholtz-ellis-ji-notation/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0d2l682v0qsh5x6ab6f0swnbb70niahx145szzhm92ls1lsmkzck"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/helmholtz-ellis-ji-notation")
    (synopsis "In-line microtonal just intonation accidentals")
    (description
     "The Helmholtz-Ellis JI Pitch Notation (HEJI), devised in the early 2000s
by Marc Sabat and Wolfgang von Schweinitz, explicitly notates the raising and
lowering of the untempered diatonic Pythagorean notes by specific microtonal
ratios defined for each prime.  It provides visually distinctive logos
distinguishing families of justly tuned intervals that relate to the harmonic
series.  These take the form of strings of additional accidental symbols based
on historical precedents, extending the traditional sharps and flats.  Since
its 2020 update, HEJI version 2 (HEJI2) provides unique microtonal symbols
through the 47-limit.  This package is a simple LaTeX implementation of HEJI2
that allows for in-line typesetting of microtonal accidentals for use within
theoretical texts, program notes, symbol legends, etc.  Documents must be
compiled using XeLaTeX.")
    (license license:cc-by4.0)))

(define-public texlive-hep
  (package
    (name "texlive-hep")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hep/" "tex/latex/hep/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1p7phgv3d4xch9c9qwkpgrpc34nzijxfxiqkpnc9b9a3xfwh71sg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hep")
    (synopsis "Convenience wrapper for High Energy Physics packages")
    (description
     "This package loads the author's @code{hepunits} and @code{hepnicenames}
packages, and a selection of others that are useful in High Energy Physics
papers, etc.")
    (license license:lppl)))

(define-public texlive-hep-bibliography
  (package
    (name "texlive-hep-bibliography")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hep-bibliography/"
                                     "source/latex/hep-bibliography/"
                                     "tex/latex/hep-bibliography/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1g4cgdrpl7jqm19k4z7ykmyqj807d3lkaah07ws7jsrfh120jrys"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hep-bibliography")
    (synopsis "Acronym extension for glossaries")
    (description
     "The @code{hep-bibliography} package extends the BibLaTeX package with
some functionality mostly useful for high energy physics.  In particular it
makes full use of all BibTeX fields provided by Discover High-Energy
Physics.")
    (license license:lppl1.3c)))

(define-public texlive-hep-font
  (package
    (name "texlive-hep-font")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/hep-font/"
                                     "source/fonts/hep-font/"
                                     "tex/latex/hep-font/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1a2vcapvpa6qlbnw5zqmwnn8gil4gnlldpqbhnc9vlv8ahx05i3q"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hep-font")
    (synopsis "Latin modern extended by Computer Modern")
    (description
     "The @code{hep-font} package loads standard font packages and extends the
usual Latin Modern implementations by replacing missing fonts with Computer
Modern counterparts.")
    (license license:lppl1.3c)))

(define-public texlive-hep-graphic
  (package
    (name "texlive-hep-graphic")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/hep-graphic/"
                              "source/latex/hep-graphic/"
                              "tex/latex/hep-graphic/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "12s7hm6f4znjnb9465y58dg79xcr54nry9k33i0ra5znd4h9c49m"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hep-graphic")
    (synopsis
     "Extensions for graphics, plots and Feynman graphs in high energy physics")
    (description
     "The @code{hep-graphic} package collects convenience macros that modify
the behavior of the TikZ, PGFPlots, and TikZ-Feynman packages and ensure that
the generated graphics look consistent.")
    (license license:lppl1.3c)))

(define-public texlive-hep-math-font
  (package
    (name "texlive-hep-math-font")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/hep-math-font/"
                                     "source/fonts/hep-math-font/"
                                     "tex/latex/hep-math-font/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1n8ii2a72lbdqbvz5dmp9jrz6pyyswrvjx8z29343vphfb3in4hd"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hep-math-font")
    (synopsis "Extended Greek and sans-serif math")
    (description
     "The @code{hep-math-font} package adjust the math fonts to be sans-serif
if the document is sans-serif.  Additionally Greek letters are redefined to be
always italic and upright in math and text mode respectively.  Some math font
macros are adjusted to give more consistently the naively expected results.")
    (license license:lppl1.3c)))

(define-public texlive-hep-paper
  (package
    (name "texlive-hep-paper")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hep-paper/"
                                     "source/latex/hep-paper/"
                                     "tex/latex/hep-paper/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1qdylnhr4fhsrqrxz4c4f25lcfjgwabn6yrxn3bswmgbhb1my5r6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hep-paper")
    (synopsis "Publications in @emph{High Energy Physics}")
    (description
     "This package aims to provide a single style file containing most
configurations and macros necessary to write appealing publications in
@emph{High Energy Physics}.  Instead of reinventing the wheel by introducing
newly created macros, hep-paper preferably loads third party packages as long
as they are light-weight enough.  For usual publications it suffices to load
the @code{hep-paper} package, without optional arguments, in addition to the
@code{article} class.")
    (license license:lppl1.3c)))

(define-public texlive-hep-reference
  (package
    (name "texlive-hep-reference")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hep-reference/"
                                     "source/latex/hep-reference/"
                                     "tex/latex/hep-reference/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1jps82gx0hw1g0gwf33rl6llvpw6s4iwdyky98ghzn2vlcsvd0ny"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hep-reference")
    (synopsis "Adjustments for publications in High Energy Physics")
    (description
     "This package makes some changes to the reference, citation and footnote
macros to improve the default behavior of LaTeX for High Energy Physics
publications.")
    (license license:lppl1.3c)))

(define-public texlive-hepnames
  (package
    (name "texlive-hepnames")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hepnames/" "tex/latex/hepnames/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1wvqf4r7p8bkvxwp7mvqnngxillbpii38i413q8acz0777bpassl"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hepnames")
    (synopsis "Pre-defined high energy particle names")
    (description
     "Hepnames provides a pair of LaTeX packages, @code{heppennames} and @code{hepnicenames},
providing a large set of pre-defined high energy physics particle names built
with the @code{hepparticles} package.  The packages are based on
@file{pennames.sty} by Michel Goosens and Eric van Herwijnen.  Heppennames
re-implements the particle names in @file{pennames.sty}, with some additions
and alterations and greater flexibility and robustness due to the
@code{hepparticles} structures, which were written for this purpose.
Hepnicenames provides the main non-resonant particle names from
@code{heppennames} with more friendly names.")
    (license license:lppl)))

(define-public texlive-hepparticles
  (package
    (name "texlive-hepparticles")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hepparticles/"
                                     "tex/latex/hepparticles/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0k8nzw5py23zvp4r4hgddjd2xypcpw85h4slad1yk43zxivb33n7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hepparticles")
    (synopsis "Macros for typesetting high energy physics particle names")
    (description
     "HEPparticles is a set of macros for typesetting high energy particle names,
to meet the following criteria:

@enumerate
@item The main particle name is a Roman or Greek symbol, to be typeset in
upright font in normal contexts.
@item Additionally a superscript or subscript may follow the main symbol.
@item Particle resonances may also have a resonance specifier which is typeset
in parentheses following the main symbol.  In general the parentheses may also
be followed by sub- and superscripts.
@item The particle names are expected to be used both in and out of
mathematical contexts.
@item If the surrounding text is bold or italic then the particle name should
adapt to that context as best as possible (this may not be possible for Greek
symbols).  As a consequence, well-known problems with boldness of particle
names in section titles, headers and tables of contents automatically
disappear if these macros are used.
@end enumerate")
    (license license:lppl)))

(define-public texlive-hepthesis
  (package
    (name "texlive-hepthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hepthesis/"
                                     "tex/latex/hepthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "10yrdsrvnb259fi2qyjldwc6h55s5z8vadr5pf6sbaihyqnv9igg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hepthesis")
    (synopsis "Class for academic reports, especially PhD theses")
    (description
     "@code{hepthesis} is a LaTeX class for typesetting large academic
reports, in particular PhD theses.  In particular, @code{hepthesis} offers:

@itemize
@item attractive semantic environments for various rubric sections;
@item extensive options for draft production, screen viewing and binding-ready
output;
@item helpful extensions of existing environments, including equation and
tabular;
@item support for quotations at the start of the thesis and each chapter.
@end itemize

The class is based on @code{scrbook}, from the KOMA-Script bundle.")
    (license license:lppl)))

(define-public texlive-hepunits
  (package
    (name "texlive-hepunits")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hepunits/"
                                     "tex/latex/hepunits/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0k89jdw5hpav5wr2imrb5wcasi53qswnqqn6jb89kh5crbrywrlz"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hepunits")
    (synopsis "Set of units useful in high energy physics applications")
    (description
     "@code{hepunits} is a LaTeX package built on the SIunits package which
adds a collection of useful @acronym{HEP, High Energy Physics} units to the
existing SIunits set.")
    (license license:lppl)))

(define-public texlive-heria
  (package
    (name "texlive-heria")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/heria/"
                              "source/latex/heria/"
                              "tex/latex/heria/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "041r0p5bs174h3a1mf7b4sd4rlbp2jvfj665z7hzn5mshaihzyq7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/heria")
    (synopsis "LaTeX class for Horizon Europe RIA and IA grant proposals")
    (description
     "This class facilitates the preparation of Research and Innovation
Action (RIA) and Innovation Action (IA) funding proposals for the European
Commission's Horizon Europe program.  The class is a conversion of the
official Part B template into LaTeX; it preserves the formatting and most of
the instructions of the original version, and has the additional feature that
tables (listing the participants, work packages, deliverables, etc.) are
generated according to data supplied by the user.")
    (license license:lppl1.3c)))

(define-public texlive-heros-otf
  (package
    (name "texlive-heros-otf")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/heros-otf/"
                                     "tex/latex/heros-otf/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "17ac8a8r5hvf76lifap070c4746jdhva9arcn1s0bqrs6kydfm56"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/heros-otf")
    (synopsis "Using the OpenType fonts TeX Gyre Heros")
    (description
     "This package can only be used with LuaLaTeX or XeLaTeX.  It does the
font setting for the OpenType font TeX Gyre Heros.  The condensed versions of
the fonts are also supported.  The missing typefaces for slanted text are
defined.")
    (license license:lppl1.3+)))

(define-public texlive-heuristica
  (package
    (name "texlive-heuristica")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/heuristica/"
                                     "fonts/enc/dvips/heuristica/"
                                     "fonts/map/dvips/heuristica/"
                                     "fonts/opentype/public/heuristica/"
                                     "fonts/tfm/public/heuristica/"
                                     "fonts/type1/public/heuristica/"
                                     "fonts/vf/public/heuristica/"
                                     "tex/latex/heuristica/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0bpf3iikl3yv35bkihrnr7hjhvb0jgjpz01flllkp9zvbfl3kcb6"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/heuristica")
    (synopsis "Fonts extending Utopia, with LaTeX support files")
    (description
     "The fonts extend the Utopia set with Cyrillic glyphs, additional figure
styles, ligatures and Small Caps in Regular style only.  Macro support, and
maths fonts that match the Utopia family, are provided by the Fourier and the
Mathdesign font packages.")
    (license license:silofl1.1)))

(define-public texlive-hexboard
  (package
    (name "texlive-hexboard")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hexboard/"
                                     "source/latex/hexboard/"
                                     "tex/latex/hexboard/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "04z0qhajbjn55mqax4kaw53h7s6g84iy1yh0pfhzj3ib7gd4cpw4"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hexboard")
    (synopsis "For drawing Hex boards and games")
    (description
     "@code{hexboard} is a package for LaTeX that should also work with LuaTeX
and XeTeX, that provides functionality for drawing Hex boards and games.")
    (license license:cc-by-sa4.0)))

(define-public texlive-hexgame
  (package
    (name "texlive-hexgame")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hexgame/" "tex/latex/hexgame/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1qr9v7225k6xzykw3rdsxf2sa3b5asvmd767i88jwimmacwi2cp1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hexgame")
    (synopsis "Provide an environment to draw a hexgame-board")
    (description
     "Hex is a mathematical game invented by the Danish mathematician Piet
Hein and independently by the mathematician John Nash.  This package defines
an environment that enables the user to draw such a game in a trivial way.")
    (license license:lppl)))

(define-public texlive-hfbright
  (package
    (name "texlive-hfbright")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/hfbright/"
                                     "fonts/afm/public/hfbright/"
                                     "fonts/enc/dvips/hfbright/"
                                     "fonts/map/dvips/hfbright/"
                                     "fonts/type1/public/hfbright/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1aw4h2law9q7sn2ppf3pxdazsk9kypss3chvca3741w8bzhxq5qr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hfbright")
    (synopsis "@code{hfbright} fonts")
    (description
     "These are Adobe Type 1 versions of the OT1-encoded and maths parts of
the Computer Modern Bright fonts.")
    (license license:lppl)))

(define-public texlive-hfoldsty
  (package
    (name "texlive-hfoldsty")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/hfoldsty/"
                                     "fonts/tfm/public/hfoldsty/"
                                     "fonts/vf/public/hfoldsty/"
                                     "source/fonts/hfoldsty/"
                                     "tex/latex/hfoldsty/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0skzw845i1b3lq76m4an4mpkm2w82hlpirnx4pckz3138nr1rlpf"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hfoldsty")
    (synopsis "Old style numerals with EC fonts")
    (description
     "The @code{hfoldsty} package provides virtual fonts for using old-style
figures with the European Computer Modern fonts.  It does a similar job as the
@code{eco} package but includes a couple of improvements, i.e., better kerning
with guillemets, and support for character protruding using the
@code{pdfcprot} package.")
    (license license:gpl3+)))

(define-public texlive-hfutexam
  (package
    (name "texlive-hfutexam")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hfutexam/" "tex/latex/hfutexam/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "14zvjkj0icdzb8bq31c5gv3nmw52s0969nlcbacbaxhgmkix9lfk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hfutexam")
    (synopsis "Exam class for Hefei University of Technology, China")
    (description
     "This package provides an exam class for Hefei University of
Technology (China).")
    (license license:lppl1.3c)))

(define-public texlive-hfutthesis
  (package
    (name "texlive-hfutthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/xelatex/hfutthesis/"
                                     "tex/xelatex/hfutthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1i7ljf5521f9dynrcnim0m4jzz2qkvpsb4pjvxbj5c7pr85gc34r"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hfutthesis")
    (synopsis "LaTeX thesis template for Hefei University of Technology")
    (description
     "This project is based on the HFUT_Thesis LaTeX template of Hefei
University of Technology compiled on the basis of @code{ustctug} and
@code{ustcthesis}, in accordance with the latest version of @emph{Hefei
University of Technology Graduate Dissertation Writing Specifications} and
@emph{Hefei University of Technology Undergraduate Graduation Project (Thesis)
Work Implementation Rules}.")
    (license license:lppl1.3c)))

(define-public texlive-hithesis
  (package
    (name "texlive-hithesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/hithesis/"
                                     "doc/xelatex/hithesis/"
                                     "makeindex/hithesis/"
                                     "source/xelatex/hithesis/"
                                     "tex/xelatex/hithesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0w701f0ivf7k8jb2jiy5lhns4qiflyrslyiplm6aca67g2rcqgwn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments
     (list #:phases
           #~(modify-phases %standard-phases
               ;; "hithesis.ins" writes files to "./figures" Create these
               ;; directories first to prevent an error.
               (add-before 'build 'prepare-build
                 (lambda _
                   (mkdir-p "build/figures"))))))
    (home-page "https://ctan.org/pkg/hithesis")
    (synopsis "Harbin Institute of Technology thesis template")
    (description
     "@code{hithesis} is a LaTeX thesis template package for Harbin Institute
of Technology supporting bachelor, master, doctor dissertations.")
    (license license:lppl1.3a)))

(define-public texlive-hindmadurai
  (package
    (name "texlive-hindmadurai")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/hindmadurai/"
                                     "fonts/enc/dvips/hindmadurai/"
                                     "fonts/map/dvips/hindmadurai/"
                                     "fonts/opentype/public/hindmadurai/"
                                     "fonts/tfm/public/hindmadurai/"
                                     "fonts/type1/public/hindmadurai/"
                                     "fonts/vf/public/hindmadurai/"
                                     "tex/latex/hindmadurai/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15bf8hka7f04l7zpkaav1azbailjv21w15iidbaxx9n2d4plf2kk"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hindmadurai")
    (synopsis "HindMadurai font face with support for LaTeX and pdfLaTeX")
    (description
     "This package provides the HindMadurai family of fonts designed by the
Indian Type Foundry, with support for LaTeX and pdfLaTeX.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-historische-zeitschrift
  (package
    (name "texlive-historische-zeitschrift")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/historische-zeitschrift/"
                                     "tex/latex/historische-zeitschrift/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1w8zhk1darw39lrgs4i1p8zk78a1q7skc93ac8qczbrw5pln27k5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/historische-zeitschrift")
    (synopsis "BibLaTeX style for the journal @emph{Historische Zeitschrift}")
    (description
     "The package provides citations according with the house style of the
@emph{Historische Zeitschrift}, a German historical journal.")
    (license license:lppl)))

(define-public texlive-hitex
  (package
    (name "texlive-hitex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/hitex/base/"
                                     "doc/man/man1/hishrink.1"
                                     "doc/man/man1/hishrink.man1.pdf"
                                     "doc/man/man1/histretch.1"
                                     "doc/man/man1/histretch.man1.pdf"
                                     "doc/man/man1/hitex.1"
                                     "doc/man/man1/hitex.man1.pdf"
                                     "doc/man/man1/texprof.1"
                                     "doc/man/man1/texprof.man1.pdf"
                                     "doc/man/man1/texprofile.1"
                                     "doc/man/man1/texprofile.man1.pdf"
                                     "makeindex/hitex/"
                                     "tex/hitex/base/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1v8lislaz090phbnkx76r0z4qbjdk5zlmjb1xqarm4kp8hl1q535"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (arguments (list #:create-formats #~(list "hilatex" "hitex")))
    (native-inputs (list (if (target-64bit?) libfaketime datefudge)))
    (propagated-inputs
     (list texlive-babel
           texlive-cm
           texlive-etex
           texlive-firstaid
           texlive-hitex-bin
           texlive-hyphen-base
           texlive-knuth-lib
           texlive-l3backend
           texlive-l3kernel
           texlive-latex
           texlive-latex-fonts
           texlive-plain
           texlive-tex-ini-files
           texlive-unicode-data))
    (home-page "https://ctan.org/pkg/hitex")
    (synopsis "TeX extension writing HINT output for on-screen reading")
    (description
     "This package provides a TeX extension that generates HINT output.  The
HINT file format is an alternative to the DVI and PDF formats which was
designed specifically for on-screen reading of documents.  Especially on
mobile devices, reading DVI or PDF documents can be cumbersome.  Mobile
devices are available in a large variety of sizes but typically are not large
enough to display documents formatted for a4/letter-size paper.  To compensate
for the limitations of a small screen, users are used to alternating between
landscape (few long lines) and portrait (more short lines) mode.  The HINT
format supports variable and varying screen sizes, leveraging the ability of
TeX to format a document for nearly-arbitrary values of @code{\\hsize} and
@code{\\vsize}.")
    (license license:x11)))

(define-public texlive-hitex-bin
  (package
    (inherit texlive-bin)
    (name "texlive-hitex-bin")
    (arguments
     (substitute-keyword-arguments (package-arguments texlive-bin)
       ((#:configure-flags flags)
        #~(cons* "--disable-web2c"
                 "--enable-hitex"
                 (delete "--disable-hitex"
                         (delete "--enable-web2c" #$flags))))
       ((#:phases phases)
        #~(modify-phases #$phases
            (replace 'install
              (lambda _
                (with-directory-excursion "texk/web2c"
                  (let ((bin (string-append #$output "/bin"))
                        (files '("hishrink" "histretch" "hitex")))
                    (for-each (lambda (f) (invoke "make" f)) files)
                    (for-each (lambda (f) (install-file f bin)) files)
                    (with-directory-excursion bin
                      (symlink "hitex" "hilatex"))))))))))
    (native-inputs (list pkg-config))
    (home-page (package-home-page texlive-hitex))
    (synopsis "Binaries for @code{texlive-hitex}")
    (description
     "This package provides the binaries for @code{texlive-hitex}.")
    (license (package-license texlive-hitex))))

(define-public texlive-hitszbeamer
  (package
    (name "texlive-hitszbeamer")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/hitszbeamer/"
                                     "doc/latex/hitszbeamer/"
                                     "source/latex/hitszbeamer/"
                                     "tex/latex/hitszbeamer/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "00c23sdhkvr79lzag8v1j5hyzf5iwmkr07xnz0s0n3haicbfzz44"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hitszbeamer")
    (synopsis "Beamer theme for Harbin Institute of Technology, ShenZhen")
    (description
     ;; XXX: Cannot use @acronym and @comma here, because Guile Texinfo does
     ;; not support this combination.
     "This is a Beamer theme designed for HITSZ (Harbin Institute of
Technology, ShenZhen).")
    (license license:lppl1.3c)))

(define-public texlive-hitszthesis
  (package
    (name "texlive-hitszthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/hitszthesis/"
                                     "doc/latex/hitszthesis/"
                                     "makeindex/hitszthesis/"
                                     "source/latex/hitszthesis/"
                                     "tex/latex/hitszthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0m1wr3iq1nzcymvnamgna0iiqb3ndllflgawzjjv85aay9jszra0"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hitszthesis")
    (synopsis
     "Dissertation template for Harbin Institute of Technology, ShenZhen")
    (description
     ;; XXX: Cannot use @acronym and @comma here, because Guile Texinfo does
     ;; not support this combination.
     "This package provides a dissertation template for HITSZ (Harbin
Institute of Technology, ShenZhen), including bachelor, master and doctor
dissertations.")
    (license license:lppl1.3c)))

(define-public texlive-hmtrump
  (package
    (name "texlive-hmtrump")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/lualatex/hmtrump/"
                                     "fonts/truetype/public/hmtrump/"
                                     "tex/lualatex/hmtrump/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03r1f784ipr2j38y2xy8agl94xwcmyv4pxd0l42iclmx08rczb9q"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hmtrump")
    (synopsis "Describe card games")
    (description
     "This package provides a font with LuaLaTeX support for describing
card games.")
    (license
     (list (license:fsf-free
            "doc/lualatex/hmtrump/nkd04_playing_cards_index/LICENSE")
           license:cc-by-sa4.0))))

(define-public texlive-hobete
  (package
    (name "texlive-hobete")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hobete/" "tex/latex/hobete/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1gccpxh3bfj4sbnag8rjrz8hyrx3107mwxpydl0bcdn8dxyjkfi2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hobete")
    (synopsis "Unofficial Beamer theme for the University of Hohenheim")
    (description
     "The package provides a Beamer theme which features the Ci colors of the
University of Hohenheim.  Please note that this is not an official theme, and
that there will be no support for it from the University.")
    (license license:lppl)))

(define-public texlive-horoscop
  (package
    (name "texlive-horoscop")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/horoscop/"
                                     "source/latex/horoscop/"
                                     "tex/latex/horoscop/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "08acv1sg37qzq3h14kxv62xhrzrv4psgpychshj3gmzvp4vz0jsn"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/horoscop")
    (synopsis "Generate astrological charts in LaTeX")
    (description
     "The @code{horoscop} package provides a unified interface for
astrological font packages; typesetting with @code{pict2e} of standard wheel
charts and some variations, in PostScript- and PDF-generating TeX engines; and
access to external calculation software (Astrolog and Swiss Ephemeris) for
computing object positions.")
    (license license:public-domain)))

(define-public texlive-hrlatex
  (package
    (name "texlive-hrlatex")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hrlatex/" "source/latex/hrlatex/"
                                     "tex/latex/hrlatex/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "16npkf18gy8clwkzvm0qysfv90wb6979cppj5rykn1x0icsvw2ix"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hrlatex")
    (synopsis "LaTeX support for Croatian documents")
    (description
     "This package simplifies creation of new documents for the (average)
Croatian user.  As an example, a class file @code{hrdipl.cls} (designed for
the graduation thesis at the University of Zagreb) and sample thesis documents
are included.")
    (license license:lppl)))

(define-public texlive-hu-berlin-bundle
  (package
    (name "texlive-hu-berlin-bundle")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/lualatex/hu-berlin-bundle/"
                                     "source/lualatex/hu-berlin-bundle/"
                                     "tex/lualatex/hu-berlin-bundle/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0y9qldwr231qh06djvck5zmvqva9hxppn8mx9a7isikr9d5152y9"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hu-berlin-bundle")
    (synopsis "LaTeX classes for the Humboldt-Universitat zu Berlin")
    (description
     "This package provides files according to the corporate design of the
Humboldt-Universitat zu Berlin.  This is not an official package by the
university itself, and not officially approved by it.")
    (license (list license:lppl1.3c license:gpl2 license:bsd-3))))

(define-public texlive-huaz
  (package
    (name "texlive-huaz")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/huaz/" "tex/latex/huaz/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1qn698n2fvk6wcxbz4ail36vslgy042nwbscn1zr32986mrnvlyr"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/huaz")
    (synopsis "Automatic Hungarian definite articles")
    (description
     "In Hungarian there are two definite articles, @samp{a} and @samp{az},
which are determined by the pronunciation of the subsequent word.  The
@code{huaz} package helps the user to insert automatically the correct
definite article for cross-references and other commands containing text.")
    (license license:lppl1.3+)))

(define-public texlive-huffman
  (package
    (name "texlive-huffman")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/metapost/huffman/"
                              "metapost/huffman/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1lspi1ma388aaaanbmnnwcsajzcjnx48444gqcqadhd7wd6k5kz7"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/huffman")
    (synopsis "Drawing binary Huffman trees with MetaPost and METAOBJ")
    (description
     "This MetaPost package allows to draw binary Huffman trees from two
arrays: an array of strings, and an array of weights (numeric).  It is based
on the METAOBJ package which provides many tools for building trees in
general.")
    (license license:lppl1.3c)))

(define-public texlive-hulipsum
  (package
    (name "texlive-hulipsum")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hulipsum/"
                                     "source/latex/hulipsum/"
                                     "tex/latex/hulipsum/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "15xhphgry6y68h1020fwpsz774kn0fici7xzj5xj04p975nbp1qg"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hulipsum")
    (synopsis "Hungarian dummy text (Lorum ipse)")
    (description
     "Lorem ipsum is an improper Latin filler dummy text, cf.@: the
@code{lipsum} package.  It is commonly used for demonstrating the textual
elements of a document template.  Lorum ipse is a Hungarian variation of Lorem
ipsum. (Lorum is a Hungarian card game, and ipse is a Hungarian slang word
meaning bloke.)  With this package you can typeset 150 paragraphs of Lorum
ipse.")
    (license license:lppl1.3+)))

(define-public texlive-hustthesis
  (package
    (name "texlive-hustthesis")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hustthesis/"
                                     "source/latex/hustthesis/"
                                     "tex/latex/hustthesis/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "12r8kg7pwwy3kbg1w993p9wnir8vifr1gvhgvv9w675c46qbr422"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hustthesis")
    (synopsis "Unofficial thesis template for Huazhong University")
    (description
     "The package provides an unofficial thesis template in LaTeX for Huazhong
University of Science and Technology.")
    (license license:lppl1.3+)))

(define-public texlive-hvarabic
  (package
    (name "texlive-hvarabic")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/hvarabic/" "tex/latex/hvarabic/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0kjx2x3fnpqw6hybw7ifdlqad4mmbacba290qwlhhggpjrxsvp39"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/hvarabic")
    (synopsis "Macros for RTL typesetting")
    (description
     "This package provides some macros for right-to-left typesetting.  It
uses by default the Arabic fonts Scheherazade and ALM fixed, the only
monospaced Arabic font.  The package only works with LuaLaTeX or XeLaTeX.")
    (license license:lppl1.3c)))

(define-public texlive-iaria
  (package
    (name "texlive-iaria")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/iaria/"
                              "source/latex/iaria/"
                              "tex/latex/iaria/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "17y72vpvfv8balr776vfykrv8m9k17vlni2117wfcy7821kkw5sb"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/iaria")
    (synopsis "Write documents for the IARIA publications")
    (description
     "This package contains templates for the creation of documents for IARIA
publications (International Academy, Research, and Industry Association) and
implements the specifications for the IARIA citation style.")
    (license license:lppl1.3c)))

(define-public texlive-iaria-lite
  (package
    (name "texlive-iaria-lite")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/latex/iaria-lite/"
                              "source/latex/iaria-lite/"
                              "tex/latex/iaria-lite/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0cv091zyf731p67bjfhgl2ah9gyan7y29nd8mpx394nv3rh00vf5"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/iaria-lite")
    (synopsis "Write documents for the IARIA publications")
    (description
     "This package provides a convenient environment for writing
IARIA (International Academy, Research, and Industry Association) scholar
publications.  It does not implement the specifications for the IARIA citation
style, for which you have to use the @code{iaria} class.")
    (license license:lppl1.3c)))

(define-public texlive-ibarra
  (package
    (name "texlive-ibarra")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/ibarra/"
                                     "fonts/enc/dvips/ibarra/"
                                     "fonts/map/dvips/ibarra/"
                                     "fonts/tfm/public/ibarra/"
                                     "fonts/truetype/public/ibarra/"
                                     "fonts/type1/public/ibarra/"
                                     "fonts/vf/public/ibarra/"
                                     "tex/latex/ibarra/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "157x2gnrnkx6v0px8pcn29f3v720wxdyvpmrz6hyllvc857g08k8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ibarra")
    (synopsis "LaTeX support for the Ibarra Real Nova family of fonts")
    (description
     "The Ibarra Real Nova is a revival of a typeface designed by Geronimo Gil
for the publication of @emph{Don Quixote} for the Real Academia de la Lengua
in 1780.")
    (license (list license:silofl1.1 license:lppl))))

(define-public texlive-ibrackets
  (package
    (name "texlive-ibrackets")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/ibrackets/"
                                     "source/latex/ibrackets/"
                                     "tex/latex/ibrackets/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "037c2g7dfr2c5l99ry0fvmwxh15jx9lq1hh4fwgcpnglh73lmkhp"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ibrackets")
    (synopsis "Intelligent brackets")
    (description
     "This small package provides a new definition of brackets @samp{[} and
@samp{]} as active characters to get correct blank spaces in mathematical mode
when using for open intervals.")
    (license license:lppl1.3+)))

(define-public texlive-icite
  (package
    (name "texlive-icite")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/icite/" "source/latex/icite/"
                                     "tex/latex/icite/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0a3gqnjbyn33ld3j66pfrrsv502zdrsag0glhar82fbba21qlj9w"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/icite")
    (synopsis "Indices locorum citatorum")
    (description
     "The package is designed to produce from BibTeX or BibLaTeX
bibliographical databases the different indices of authors and works cited
which are called indices locorum citatorum.  It relies on a specific
@code{\\icite} command and can operate with either BibTeX or BibLaTeX.")
    (license (list license:gpl3+ license:cc-by-sa4.0))))

(define-public texlive-ietfbibs
  (package
    (name "texlive-ietfbibs")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/bibtex/ietfbibs/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "18ypa96z1gclq4amka28sv26pm7lycak68zf4b3sf1bx4ldnzmxq"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ietfbibs")
    (synopsis "Generate BibTeX entries for various IETF index files")
    (description
     "The package provides scripts to translate IETF index files to BibTeX
files.")
    (license license:expat)))

(define-public texlive-ifis-macros
  (package
    (name "texlive-ifis-macros")
    (version "2025.2")
    (source
     (origin
       (method svn-multi-fetch)
       (uri (svn-multi-reference
             (url (texlive-packages-repository version))
             (revision 74725)
             (locations (list "doc/plain/ifis-macros/"
                              "tex/plain/ifis-macros/"))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "12cmpg4r065qvhjnm272iclih3lyk0vqqyk9zjyxdynpyhg042k1"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ifis-macros")
    (synopsis "Check if a given input string is a number or dimension for TeX")
    (description
     "This package provides two macros: @code{\\ifisint} and @code{\\ifisdim}.
They test if a given input string represents either a valid integer or a valid
dimension for TeX.")
    (license license:gpl3+)))

(define-public texlive-ifsym
  (package
    (name "texlive-ifsym")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/ifsym/"
                                     "fonts/source/public/ifsym/"
                                     "fonts/tfm/public/ifsym/" "tex/latex/ifsym/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "07r16sgqc65wf8c8ijgmnmwq50yrlvjlmzbnk0czgnwqrbdz642g"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (native-inputs (list texlive-metafont))
    (home-page "https://ctan.org/pkg/ifsym")
    (synopsis "Collection of symbols")
    (description
     "This package provides a set of symbol fonts, written in Metafont, offering
(respectively) clock-face symbols, geometrical symbols, weather symbols,
mountaineering symbols, electronic circuit symbols and a set of miscellaneous
symbols.  A LaTeX package is provided, that allows the user to load only those
symbols needed in a document.")
    (license license:lppl1.0+)))

(define-public texlive-ijqc
  (package
    (name "texlive-ijqc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "bibtex/bst/ijqc/" "doc/bibtex/ijqc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "18sqc8k21l15zlplpaimdcg8g75z3lrlc0qcb6wx838980gn961r"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/ijqc")
    (synopsis "BibTeX style file for the @emph{Intl. J. Quantum Chem}")
    (description
     "@file{ijqc.bst} is a BibTeX style file to support publication in Wiley's
@emph{International Journal of Quantum Chemistry}.")
    (license license:lppl)))

(define-public texlive-imfellenglish
  (package
    (name "texlive-imfellenglish")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/fonts/imfellenglish/"
                                     "fonts/enc/dvips/imfellenglish/"
                                     "fonts/map/dvips/imfellenglish/"
                                     "fonts/opentype/iginomarini/imfellenglish/"
                                     "fonts/tfm/iginomarini/imfellenglish/"
                                     "fonts/type1/iginomarini/imfellenglish/"
                                     "fonts/vf/iginomarini/imfellenglish/"
                                     "tex/latex/imfellenglish/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0pw4nsw4pl1nf99j58sy37064712by6kqcb40xkrcl5d9czx45bh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/imfellenglish")
    (synopsis "IM Fell English fonts with LaTeX support")
    (description
     "Igino Marini has implemented digital revivals of fonts bequeathed to
Oxford University by Dr.@: John Fell, Bishop of Oxford and Dean of Christ
Church in 1686.  This package provides the English family, consisting of
Roman, Italic and Small-Cap fonts.")
    (license license:silofl1.1)))

(define-public texlive-impatient
  (package
    (name "texlive-impatient")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/plain/impatient/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "03cjl3lg7k7p4h8drw7vcbw6ymgmm2clv6sgfs3hdixs8dmab1b2"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/impatient")
    (synopsis "@emph{TeX for the Impatient} book")
    (description
     "@emph{TeX for the Impatient} is a book on TeX, Plain TeX and Eplain.")
    (license license:fdl1.3+)))

(define-public texlive-impatient-cn
  (package
    (name "texlive-impatient-cn")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/plain/impatient-cn/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1pc20pb0las3slam1d9hmqigipmr6r98wwif12m33mm750vmq65i"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/impatient")
    (synopsis "Chinese translation of @emph{TeX for the Impatient} book")
    (description
     "@emph{TeX for the Impatient} is a book (of around 350 pages) on TeX,
Plain TeX and Eplain.  This is its Chinese translation.")
    (license license:fdl1.3+)))

(define-public texlive-impatient-fr
  (package
    (name "texlive-impatient-fr")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/plain/impatient-fr/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1qwdllmi0ci304smf4yl6g8ah083jrch9clkyav7r5qsxnhwxbxh"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/impatient")
    (synopsis "French translation of @emph{TeX for the Impatient} book")
    (description
     "@emph{TeX for the Impatient} is a book (of around 350 pages) on TeX,
Plain TeX and Eplain.  This is its French translation.")
    (license license:fdl1.3+)))

(define-public texlive-impnattypo
  (package
    (name "texlive-impnattypo")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/impnattypo/"
                                     "source/latex/impnattypo/"
                                     "tex/latex/impnattypo/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "05ddbpxvybr0a7sg1ximsifvgjg83qwx5pd9xcc22bliz3j5ixvm"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/impnattypo")
    (synopsis "Support typography of l'Imprimerie Nationale Francaise")
    (description
     "The package provides useful macros implementing recommendations by the
French Imprimerie Nationale.")
    (license license:lppl1.3+)))

(define-public texlive-import
  (package
    (name "texlive-import")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/import/" "tex/latex/import/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "0wlzs31li6nvzigkxw59bbpmyqrkzpdangvjqq3z7wl6y79sic6g"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/import")
    (synopsis "Establish input relative to a directory")
    (description
     "The commands @code{\\import@{full_path@}@{file@}} and
@code{\\subimport@{path_extension@}@{file@}} set up input through standard
LaTeX mechanisms (@code{\\input}, @code{\\include} and
@code{\\includegraphics}) to load files relative to the imported directory.
There are also @code{\\includefrom}, @code{\\subincludefrom}, and starred
variants of the commands.")
    (license license:public-domain)))

(define-public texlive-imsproc
  (package
    (name "texlive-imsproc")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/xelatex/imsproc/"
                                     "tex/xelatex/imsproc/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1akzdc4nas4fkmlhm6xp97xw65vm5bif73jq2ki5if5kwssxc38z"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/imsproc")
    (synopsis "Typeset IMS conference proceedings")
    (description
     "The class typesets papers for IMS (Iranian Mathematical Society)
conference proceedings.  The class uses the XePersian package.")
    (license license:lppl1.3+)))

(define-public texlive-includernw
  (package
    (name "texlive-includernw")
    (version "2025.2")
    (source (origin
              (method svn-multi-fetch)
              (uri (svn-multi-reference
                    (url (texlive-packages-repository version))
                    (revision 74725)
                    (locations (list "doc/latex/includernw/"
                                     "tex/latex/includernw/"))))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "1kc14nq0f3jybidifvr6gjfy4ggs5qkbs1syv1k7bw3xrdmpplb8"))))
    (outputs '("out" "doc"))
    (build-system texlive-build-system)
    (home-page "https://ctan.org/pkg/includernw")
    (synopsis "Include .Rnw inside .tex")
    (description
     "This package is fo