
///////////////////////////////////////////////////////////
//                                                       //
//                         SAGA                          //
//                                                       //
//      System for Automated Geoscientific Analyses      //
//                                                       //
//                    Tool Library                       //
//                     Grid_Tools                        //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//               Grid_Table_Conversion.cpp               //
//                                                       //
//                 Copyright (C) 2025 by                 //
//                      Olaf Conrad                      //
//                                                       //
//-------------------------------------------------------//
//                                                       //
// This file is part of 'SAGA - System for Automated     //
// Geoscientific Analyses'. SAGA is free software; you   //
// can redistribute it and/or modify it under the terms  //
// of the GNU General Public License as published by the //
// Free Software Foundation, either version 2 of the     //
// License, or (at your option) any later version.       //
//                                                       //
// SAGA is distributed in the hope that it will be       //
// useful, but WITHOUT ANY WARRANTY; without even the    //
// implied warranty of MERCHANTABILITY or FITNESS FOR A  //
// PARTICULAR PURPOSE. See the GNU General Public        //
// License for more details.                             //
//                                                       //
// You should have received a copy of the GNU General    //
// Public License along with this program; if not, see   //
// <http://www.gnu.org/licenses/>.                       //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//    e-mail:     oconrad@saga-gis.org                   //
//                                                       //
//    contact:    Olaf Conrad                            //
//                Institute of Geography                 //
//                University of Hamburg                  //
//                Germany                                //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
#include "Grid_Table_Conversion.h"


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
CGrid_from_Table::CGrid_from_Table(void)
{
	Set_Name		(_TL("Grid from Table"));

	Set_Author		("O.Conrad (c) 2025");

	Set_Description	(_TW(
		""
	));

	Parameters.Add_Table      ("", "TABLE"      , _TL("Table"    ), _TL(""), PARAMETER_INPUT);
	Parameters.Add_Grid_Output("", "GRID"       , _TL("Grid"     ), _TL(""));
	Parameters.Add_Data_Type  ("", "DATATYPE"   , _TL("Data Type"), _TL(""), SG_DATATYPES_Numeric, SG_DATATYPE_Double);
	Parameters.Add_Double     ("", "GEOREF_SIZE", _TL("Cell Size"), _TL(""), 1., 0., true);
	Parameters.Add_Double     ("", "GEOREF_XMIN", _TL("Left"     ), _TL("Easting of lower left cell's center." ), 0.);
	Parameters.Add_Double     ("", "GEOREF_YMIN", _TL("Bottom"   ), _TL("Northing of lower left cell's center."), 0.);
}


///////////////////////////////////////////////////////////
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
bool CGrid_from_Table::On_Execute(void)
{
	CSG_Table &Table = *Parameters["TABLE"].asTable();

	if( !Table.is_Valid() || Table.Get_Count() < 1 || Table.Get_Field_Count() < 1 )
	{
		Error_Set(_TL("Table seems to be empty!"));

		return( false );
	}

	//-----------------------------------------------------
	CSG_Grid *pGrid = Parameters["GRID"].asGrid(); if( !pGrid ) { Parameters.Set_Parameter("GRID", pGrid = SG_Create_Grid()); }

	if( !pGrid->Create(Parameters["DATATYPE"].asDataType()->Get_Data_Type(),
		Table.Get_Field_Count(), (int)Table.Get_Count(),
		Parameters["GEOREF_SIZE"].asDouble(),
		Parameters["GEOREF_XMIN"].asDouble(),
		Parameters["GEOREF_YMIN"].asDouble()) )
	{
		Error_Set(_TL("Failed to instantiate target grid object!"));

		return( false );
	}

	pGrid->Set_Name(Table.Get_Name());

	//-----------------------------------------------------
	#pragma omp parallel for
	for(int y=0; y<Table.Get_Count(); y++)
	{
		CSG_Table_Record &Row = Table[Table.Get_Count() - 1 - y];

		for(int x=0; x<Table.Get_Field_Count(); x++)
		{
			if( Row.is_NoData(x) )
			{
				pGrid->Set_NoData(x, y);
			}
			else
			{
				pGrid->Set_Value(x, y, Row.asDouble(x));
			}
		}
	}

	//-----------------------------------------------------
	return( true );
}


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
CGrid_to_Table::CGrid_to_Table(void)
{
	Set_Name		(_TL("Grid to Table"));

	Set_Author		("O.Conrad (c) 2025");

	Set_Description	(_TW(
		""
	));

	Parameters.Add_Grid ("", "GRID" , _TL("Grid" ), _TL(""), PARAMETER_INPUT );
	Parameters.Add_Table("", "TABLE", _TL("Table"), _TL(""), PARAMETER_OUTPUT);

	Parameters.Add_Bool ("", "FLOAT", _TL("Floating Point Number"),
		_TL("Force table's value types to be floating point numbers, else take grid's data type."),
		false
	);
}


///////////////////////////////////////////////////////////
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
bool CGrid_to_Table::On_Execute(void)
{
	CSG_Grid &Grid = *Parameters["GRID"].asGrid();

	if( !Grid.is_Valid() )
	{
		Error_Set(_TL("Invalid input grid!"));

		return( false );
	}

	//-----------------------------------------------------
	CSG_Table &Table = *Parameters["TABLE"].asTable();

	Table.Create();
	Table.Set_Name(Grid.Get_Name());
	Table.Set_NoData_Value(Grid.Get_NoData_Value());

	bool bFloat = Parameters["FLOAT"].asBool();

	for(int x=0; x<Get_NX(); x++)
	{
		Table.Add_Field(CSG_String::Format("%d", 1 + x), bFloat ? SG_DATATYPE_Double : Grid.Get_Type());
	}

	//-----------------------------------------------------
	for(int y=0; y<Get_NY(); y++)
	{
		CSG_Table_Record &Row = *Table.Add_Record(); int yy = Get_NY() - 1 - y;

		for(int x=0; x<Get_NX(); x++)
		{
			if( Grid.is_NoData(x, yy) )
			{
				Row.Set_NoData(x);
			}
			else
			{
				Row.Set_Value(x, Grid.asDouble(x, yy, bFloat));
			}
		}
	}

	//-----------------------------------------------------
	return( true );
}


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
