/*
 *  Scilab ( https://www.scilab.org/ ) - This file is part of Scilab
 *  Copyright (C) 2010-2010 - DIGITEO - Bruno JOFRET
 *
 * Copyright (C) 2012 - 2016 - Scilab Enterprises
 *
 * This file is hereby licensed under the terms of the GNU GPL v2.0,
 * pursuant to article 5.3.4 of the CeCILL v.2.1.
 * This file was originally licensed under the terms of the CeCILL v2.1,
 * and continues to be available under such terms.
 * For more information, see the COPYING file which you should have received
 * along with this program.
 *
 */
/*--------------------------------------------------------------------------*/
extern "C"
{
#include "storeCommand.h"
}

#include "parser.hxx"
#include "runner.hxx"
#include "scilabWrite.hxx"
#include "scilabexception.hxx"
#include "localization.hxx"
#include "threadmanagement.hxx"
#include "timer.hxx"
#include "configvariable.hxx"
/*--------------------------------------------------------------------------*/
struct CommandRec
{
    char*     m_command;              /* command info one string two integers */
    ast::Exp* m_tree;                 /* AST generated by the parser using the command string*/  
    int       m_isPrioritary;         /* 1 if the command is prioritary */
    int       m_isInterruptible;      /* 1 if the command execution can be interrupted */
    command_origin_t     m_iCommandOrigin;       /* Indicate who have stored the command (ie: console, tcl) */
    CommandRec(char* command, ast::Exp* tree, int isPrioritary, int isInterruptible, command_origin_t iCmdOrigin) :
        m_command(command), m_tree(tree), m_isPrioritary(isPrioritary), m_isInterruptible(isInterruptible), m_iCommandOrigin(iCmdOrigin) {}
};

static std::list<CommandRec> commandQueue;
static std::list<CommandRec> commandQueuePrioritary;
static Timer _timer;
/*--------------------------------------------------------------------------*/
static parse_state parseCommand(const char* command, ast::Exp** tree)
{
    bool timed = ConfigVariable::getTimed();
    if (command == NULL || strlen(command) == 0)
    {
        return Failed;
    }

    ThreadManagement::LockParser();
    Parser parser;
    if (timed)
    {
        _timer.start();
    }

    parser.parse(command);

    if (timed)
    {
        _timer.check(L"Parsing");
    }

    if (parser.getControlStatus() != Parser::AllControlClosed)
    {
        delete parser.getTree();
        parser.setTree(NULL);
        ThreadManagement::UnlockParser();
        return NotAllControlClosed;
    }
    else if (parser.getExitStatus() == Parser::Failed)
    {
        scilabErrorW(parser.getErrorMessage());
        ConfigVariable::setLastErrorNumber(999);
        ConfigVariable::setLastErrorMessage(parser.getErrorMessage());

        ThreadManagement::UnlockParser();
        return Failed;
    }

    *tree = parser.getTree();
    ThreadManagement::UnlockParser();
    return Succeded;
}

int StoreCommandWithFlags(const char* command, int iPrioritary, int iInterruptible, command_origin_t iCmdOrigin)
{
    ThreadManagement::LockStoreCommand();
    ast::Exp* tree = nullptr;
    parse_state status = parseCommand(command, &tree);
    if(status != Succeded)
    {
        ThreadManagement::UnlockStoreCommand();
        return status;
    }

    if (iPrioritary)
    {
        commandQueuePrioritary.emplace_back(os_strdup(command), tree, iPrioritary, iInterruptible, iCmdOrigin);
    }
    else
    {
        commandQueue.emplace_back(os_strdup(command), tree, iPrioritary, iInterruptible, iCmdOrigin);
    }

    // Awake Scilab to execute a new command
    ThreadManagement::SendCommandStoredSignal();

    ThreadManagement::UnlockStoreCommand();

    return Succeded;
}

int StoreCommand(const char *command)
{
    ThreadManagement::LockStoreCommand();
    ast::Exp* tree = nullptr;
    parse_state status = parseCommand(command, &tree);
    if(status != Succeded)
    {
        ThreadManagement::UnlockStoreCommand();
        return status;
    }

    commandQueue.emplace_back(os_strdup(command),
                              tree,
                              /*is prioritary*/ 0,
                              /* is interruptible*/ 1,
                              /* cmd origin */ NONE);

    // Awake Scilab to execute a new command
    ThreadManagement::SendCommandStoredSignal();

    ThreadManagement::UnlockStoreCommand();

    return Succeded;
}

int StoreConsoleCommand(const char *command, int iWaitFor)
{
    ThreadManagement::LockStoreCommand();
    ast::Exp* tree = nullptr;
    parse_state status = parseCommand(command, &tree);
    if(status != Succeded)
    {
        ThreadManagement::UnlockStoreCommand();
        return status;
    }

    commandQueuePrioritary.emplace_back(os_strdup(command),
                                        tree,
                                        /*is prioritary*/ 1,
                                        /* is interruptible*/ 1,
                                        /* cmd origin */ CONSOLE);

    // Awake Scilab to execute a new command
    ThreadManagement::SendCommandStoredSignal();

    if (iWaitFor)
    {
        // make this wait before unlock the Store Command will prevent
        // dead lock in case where another thread get this command
        // and execute it before this thread is waiting for.
        ThreadManagement::WaitForConsoleExecDoneSignal();
    }
    else
    {
        ThreadManagement::UnlockStoreCommand();
    }

    return Succeded;
}

int StoreDebuggerCommand(const char *command, int iWaitFor)
{
    ThreadManagement::LockStoreCommand();
    ast::Exp* tree = nullptr;
    parse_state status = parseCommand(command, &tree);
    if(status != Succeded)
    {
        ThreadManagement::UnlockStoreCommand();
        return status;
    }

    commandQueuePrioritary.emplace_back(os_strdup(command),
                                        tree,
                                        /*is prioritary*/ 1,
                                        /* is interruptible*/ 1,
                                        /* cmd origin */ DEBUGGER);

    // Awake Scilab to execute a new command
    ThreadManagement::SendCommandStoredSignal();

    if (iWaitFor)
    {
        // make this wait before unlock the Store Command will prevent
        // dead lock in case where another thread get this command
        // and execute it before this thread is waiting for.
        ThreadManagement::WaitForDebuggerExecDoneSignal();
    }
    else
    {
        ThreadManagement::UnlockStoreCommand();
    }

    return Succeded;
}

int StorePrioritaryCommand(const char *command)
{
    ThreadManagement::LockStoreCommand();
    ast::Exp* tree = nullptr;
    parse_state status = parseCommand(command, &tree);
    if(status != Succeded)
    {
        ThreadManagement::UnlockStoreCommand();
        return status;
    }

    commandQueuePrioritary.emplace_back(os_strdup(command),
                                        tree,
                                        /*is prioritary*/ 1,
                                        /* is interruptible*/ 0,
                                        /* cmd origin */ NONE);

    // Awake Scilab to execute a new command
    ThreadManagement::SendCommandStoredSignal();
    ThreadManagement::UnlockStoreCommand();

    return Succeded;
}

int isEmptyCommandQueue(void)
{
    ThreadManagement::LockStoreCommand();
    bool isEmpty = commandQueue.empty() && commandQueuePrioritary.empty();
    ThreadManagement::UnlockStoreCommand();
    return isEmpty;
}

int isEmptyCommandQueuePrioritary(void)
{
    //ThreadManagement::LockStoreCommand();
    bool isEmpty = commandQueuePrioritary.empty();
    //ThreadManagement::UnlockStoreCommand();
    return isEmpty;
}

/*
 * Gets the next command to execute
 * and remove it from the queue
 */
void* GetCommand()
{
    Runner* runMe = nullptr;

    ThreadManagement::LockStoreCommand();
    if (commandQueuePrioritary.empty() == false)
    {
        runMe = new Runner(commandQueuePrioritary.front().m_tree,
            commandQueuePrioritary.front().m_iCommandOrigin,
            commandQueuePrioritary.front().m_isInterruptible);

        // std::cout << commandQueuePrioritary.front().m_command << std::endl;
        FREE(commandQueuePrioritary.front().m_command);
        commandQueuePrioritary.pop_front();
    }
    else if (commandQueue.empty() == false)
    {
        runMe = new Runner(commandQueue.front().m_tree, 
            commandQueue.front().m_iCommandOrigin,
            commandQueue.front().m_isInterruptible);

        // std::cout << commandQueue.front().m_command << std::endl;
        FREE(commandQueue.front().m_command);
        commandQueue.pop_front();
    }
    ThreadManagement::UnlockStoreCommand();

    return (void*)runMe;
}

/*--------------------------------------------------------------------------*/

