# Chapter 2: Quick Overview {#tutorial-overview}

Let's start by taking a concise look at how Louvre works.

## Interface Methods

First and foremost, it's crucial to understand that Louvre employs virtual methods as interfaces to let you respond to signals or client application requests. 

> If you're not familiar with virtual methods, I recommend checking out this excellent video explanation by The Cherno: [Watch the video](https://www.youtube.com/watch?v=oIV2KchSyGQ).

Consider the example of the Louvre::LSurface class, which typically represents a "client window":

```cpp

using namespace Louvre;

class YourCustomSurface : public LSurface
{
	// ... 
    
    void bufferSizeChanged() override
    {
    	// Here your handle this signal
    }
    
    void mappingChanged() override
    {
    	// Here your handle this signal
    }
    
    // ... 
};
```

Both Louvre::LSurface::bufferSizeChanged() and Louvre::LSurface::mappingChanged() are virtual methods of the Louvre::LSurface class, and they are not intended for direct invocation by you. Instead, the library triggers these methods to notify you of changes, allowing you to handle them effectively.

It's important to note that not all methods in the library are virtual. Just a few or none per class, and you can easily identify them because they often end with suffixes such as *Changed*, *Request*, *Signal*, and so on. If you're not sure, you can always check the [API documentation](annotated.html).

Also, you don't need to reimplement every class and override all of its methods. You should selectively focus on the classes and methods that are relevant to your specific use case, as the library provides default ways of handling them.

To see the default implementation of each virtual method, please refer to the API documentation, such as Louvre::LSurface::bufferSizeChanged() and Louvre::LSurface::mappingChanged().

## Understanding Wayland

In case you're not aware, Wayland's sole purpose is to provide a client/server architecture and mechanisms through which a compositor can communicate with applications, nothing more than that. All other functionalities, such as rendering and accessing system input, are implemented by the compositor itself, in this case, by Louvre. Messages in Wayland and their parameters are defined in an `.xml` file, which can be converted into C code using the [wayland-scanner](https://wayland-book.com/libwayland/wayland-scanner.html) tool to be used by clients and the compositor. Each XML file is considered a protocol, and there are several, such as the Wayland protocol itself, XDG Shell, Viewporter, etc. You can even create your own protocols. Each protocol has one or more interfaces, which are considered "objects" that have various events (messages from the compositor to the client) and requests (messages from the client to the compositor).

For example, the Louvre::LSurface class represents the [wl_surface](https://wayland.app/protocols/wayland#wl_surface) interface of the Wayland protocol, and you can use it both to listen to client requests and to send them events. Now, as the name suggests, protocols have rules on how the compositor should react to requests. A significant portion of these requests, especially the more tedious ones, are handled directly by Louvre internally. Others are preprocessed and notified to the user through a simpler API, and some are simply identical to the original ones. 

For a compositor to be functional, it must correctly handle all these requests. For this reason, Louvre also provides a default implementation for each of them, which you can gradually override to create your own unique compositor. This allows you to see results on the screen from day one and validate each change progressively.

## Main Classes

Louvre provides a diverse range of classes, each serving specific purposes. Here's a quick overview of some crucial classes that will guide you in navigating the documentation effectively:

### LCompositor

This is the heart of every Louvre compositor, responsible for tasks such as creating other classes, loading the graphic and input backends, processing events, handling client requests, and much more. One particularly significant aspect to understand is that it includes multiple virtual constructors and destructors for various other classes.

For instance, when a client wishes to create a new surface, the Louvre::LCompositor::createSurfaceRequest() virtual method is invoked:

```cpp

using namespace Louvre;

class YourCustomCompositor : public LCompositor
{
	// ... 
    
    LSurface *createSurfaceRequest(const void *params) override
    {
    	return new YourCustomSurface(params);
    }
    
    void destroySurfaceRequest(LSurface *surface) override
    {
    	// Only notifies; you must not call 'delete surface;'
    }
    
    // ... 
};
```

As you might deduce, each virtual constructor/destructor has a default implementation provided by the library as well. For example, the default implementation of Louvre::LCompositor::createSurfaceRequest() returns a new Louvre::LSurface instance, rather than an instance of `YourCustomSurface` class.

Therefore, if you intend to use your own subclasses, you should override their respective virtual constructors in Louvre::LCompositor.

> It's essential to note that not all classes in Louvre are created in this manner, this approach is specifically applicable to client, backend, and internal compositor resources that provide an interface (virtual methods) you can override.

### LOutput

The Louvre::LOutput class both represents and manages rendering for a specific display. It offers a range of essential virtual methods, including Louvre::LOutput::initializeGL(), Louvre::LOutput::paintGL(), Louvre::LOutput::resizeGL(), and others, which you can override to perform your own painting operations.

### LPointer

Louvre::LPointer is a singleton class responsible for capturing input events from devices like a mouse or touchpad. It offers various virtual methods for event listening and enables forwarding these events to client applications.

### LKeyboard

Louvre::LKeyboard is similar to Louvre::LPointer but for system keyboard events. It allows you to listen to keyboard events, forward them to clients, and configure keyboard aspects such as the keymap, key press repeat rate, and more.

In the upcoming chapters, we will explore the practical usage of these classes, and many more.

<a href="01.md">◀ Chapter 1: Introduction </a> || <a href="03.md"> Chapter 3: Setting Up the Project ▶</a>