# Chapter 4: Compositor Initialization {#tutorial-initialization}

In this chapter, we will explore how to customize the initialization of Louvre::LCompositor. This includes setting up parameters like the keyboard map and output arrangements. This customization will serve as a foundation for overriding the behavior of other classes in upcoming chapters.

To begin, let's create a new Louvre::LCompositor subclass named `ECompositor` (with the "E" prefix denoting "Example Compositor"):

#### src/ECompositor.h

```cpp
#ifndef ECOMPOSITOR_H
#define ECOMPOSITOR_H

#include <LCompositor.h>

using namespace Louvre;

class ECompositor : public LCompositor
{
public:
    ECompositor();

    void initialized() override;
};

#endif // ECOMPOSITOR_H
```

Here, we are overriding the Louvre::LCompositor::initialized() virtual method (click to see its default implementation).\n
Louvre calls this method once the compositor has completed its initialization process after we invoke Louvre::LCompositor::start() in `src/main.cpp`.\n
This method is an ideal place to configure and load all the resources required by your compositor.

> Certain objects, such as textures, can only be created within or after the invocation of Louvre::LCompositor::initialized(). This is imperative as they depend on the graphic backend to be fully initialized.

#### src/ECompositor.cpp

```cpp
#include <LOutput.h>
#include <LSeat.h>
#include <LKeyboard.h>
#include "ECompositor.h"
 
ECompositor::ECompositor() {}
 
void ECompositor::initialized()
{
    // Set a keyboard map with "latam" layout
    seat()->keyboard()->setKeymap(nullptr, nullptr, "latam", nullptr);
 
    Int32 totalWidth { 0 };
 
    // Initialize all avaliable outputs
    for (LOutput *output : seat()->outputs())
    {
        // Set double scale to outputs with DPI >= 200
        output->setScale(output->dpi() >= 200 ? 2.f : 1.f);

        // Assuming your outputs are not rotated or flipped
        output->setTransform(LFramebuffer::Normal);
 
        output->setPos(LPoint(totalWidth, 0));
        totalWidth += output->size().w();
 
        addOutput(output);
        output->repaint();
    }
}
```

Here, we're first configuring the keyboard map with Louvre::LKeyboard::setKeymap(). This keymap helps the compositor interpret raw input key codes as XKB key symbols and ensures clients can correctly understand them as well. 

The method searches for available keymaps on your system based on the provided parameters. If no matches are found, it loads your system's default keymap. In this case, I'm using the Latin American keyboard layout since I'm Chilean. To load your system's default keymap, simply pass `nullptr` for all arguments.

Secondly, we iterate through all available outputs on your machine, those could be your laptop screen, HDMI connector, or others. If an output is considered High DPI (with a DPI greater than or equal to 200), we set its scale to 2 (the default is 1). This adjustment ensures that the content is properly scaled, otherwise, application windows and UI elements would appear tiny.

The Louvre::LOutput::setTransform() method allows you to define a custom transform, particularly useful when dealing with physically rotated or flipped outputs. In this case, we are assigning the Louvre::LFramebuffer::Normal (no transform) as the default value.

> Starting from Louvre version 1.2, Louvre::LOutput::setScale() now supports fractional scales. For a comprehensive understanding of the output global coordinate space, fractional scaling, and transforms, refer to the @ref Scaling "LOutput" documentation.

Outputs can also be virtually positioned, as you may have done in your system's display configuration panel:

<center>
<img style="max-width:45vw" src="https://lh3.googleusercontent.com/pw/AIL4fc8mtBg1aWk8d_9hSa9_sp8V7KYVk6ZskkKgLZ6YP1ouaf-ku7zNndMoASmn2tNsD_2xW7hnHjZgaiLYfFJ-Kbv3SXZhdlGCm-sZamFHsb-Atvo0Kyg=w2400"/><br>
<div style="margin-top:10px;font-style:italic;">Example of a system's display configuration panel.</div>
</center>

In this case, we arrange them from left to right using the `totalWidth` variable. Finally, we initialize these outputs using Louvre::LCompositor::addOutput(), allowing us to render content onto them.

> You can access all available outputs in your system via Louvre::LSeat::outputs(), and for the initialized outputs only, you can use Louvre::LCompositor::outputs().\n As the compositor and seat object instances are both static and unique, these can be accessed from any object using `object->compositor()` and `object->seat()`.

Lastly, in `src/main.cpp`, we should replace `LCompositor` with our custom `ECompositor` class.

#### src/main.cpp

```cpp
#include <LLauncher.h>
#include <LLog.h>
#include "ECompositor.h"

using namespace Louvre;

int main()
{
    setenv("LOUVRE_DEBUG", "4", 0);

    LLauncher::startDaemon();

    ECompositor compositor;

    if (!compositor.start())
    {
        LLog::fatal("[louvre-example] Failed to start compositor.");
        return 1;
    }

    while (compositor.state() != LCompositor::Uninitialized)
        compositor.processLoop(-1);

    return 0;
}
```

### Rebuilding

As we've added two new files, we'll need to reconfigure the project and then rebuild it with the following commands:

```bash
$ cd project_dir
$ meson setup build --wipe
$ cd build
$ meson compile
```

> Executing `$ meson setup build --wipe` is only required when you add or remove source files in your project.

If you run the compositor again, you may not notice significant changes since Louvre::LCompositor::initialized() by default already performs similar tasks. However, if you've modified the keyboard map or the outputs transform, you should now see the desired improvements.

> Louvre::LOutput::name() returns a unique and consistent name for each output, enabling you to identify them even after the system reboots.

### Hotplugging Events

After learning how to configure outputs during startup, what happens if we connect or disconnect an output while the compositor is already running?

Louvre handles this automatically by initializing any newly plugged output and de-initializing it when unplugged. This functionality is managed within the Louvre::LSeat class, specifically through its Louvre::LSeat::outputPlugged() and Louvre::LSeat::outputUnplugged() virtual methods. While it's possible to customize these methods, it's not essential to do so at this stage of the tutorial. We will explore this further in [Chapter 15: Output Hotplugging and Seat](15.md) once you're comfortable working with your own subclasses.

In the following chapters, things are about to get more thrilling as we dive into different ways for rendering content to the outputs.

<a href="03.md">◀ Chapter 3: Setting Up the Project</a> || <a href="05.md"> Chapter 5: Rendering with LPainter ▶</a>
