# Chapter 5: Rendering with LPainter {#tutorial-lpainter}

In this chapter, we will delve into the process of rendering content onto the outputs that we have previously initialized.\n
Each Louvre::LOutput operates within its own thread and provides a set of events that can be customized to suit our rendering needs.

#### initializeGL()

The Louvre::LOutput::initializeGL() event is invoked once, right after we initialize the output using Louvre::LCompositor::addOutput(). Within this event, your focus should be on setting up and loading any necessary OpenGL resources, such as shaders, programs, and textures. However, it's important to refrain from performing any painting operations at this stage, as they will not be visible on the screen. Instead, use Louvre::LOutput::repaint() to schedule a new Louvre::LOutput::paintGL() event.

#### paintGL()

The Louvre::LOutput::paintGL() event is triggered asynchronously when we call Louvre::LOutput::repaint() from any thread. Within this event, you should handle all the painting operations necessary for the current output frame.

#### moveGL()

The Louvre::LOutput::moveGL() event is asynchronously triggered each time we alter the position of the output using Louvre::LOutput::setPos(). It's crucial to avoid performing any painting operations within this event, as they will not be displayed on the screen. Instead, use Louvre::LOutput::repaint() to schedule a new Louvre::LOutput::paintGL() event.

#### resizeGL()

Similarly, the Louvre::LOutput::resizeGL() event is called asynchronously whenever we modify the output mode, transform or scale using Louvre::LOutput::setMode(), Louvre::LOutput::setTransform() or Louvre::LOutput::setScale(). Each output has one or more modes (Louvre::LOutputMode), each of which dictates its refresh rate and resolution. The default output mode typically offers the highest refresh rate and resolution. To access all available output modes, you can use Louvre::LOutput::modes(). Like the other events, it's essential not to perform any painting operations within Louvre::LOutput::resizeGL(), as they will not be visible on the screen. Instead, use Louvre::LOutput::repaint() to schedule a new Louvre::LOutput::paintGL() event.

#### uninitializeGL()

The Louvre::LOutput::uninitializeGL() event is invoked once after we uninitialize an output using Louvre::LCompositor::removeOutput() or when the output is unplugged. We will explore how to listen to output hotplugging events later on. Within this event, your responsibility is to release all the resources created during the Louvre::LOutput::initializeGL() event. It's crucial to avoid any painting operations within Louvre::LOutput::uninitializeGL() because they will not be displayed on the screen.

> It's important to note that Louvre prevents you from initializing/uninitializing or changing the mode of an output from within any of these events, as doing so could lead to a deadlock. So calling Louvre::LCompositor::addOutput(), Louvre::LCompositor::removeOutput() or Louvre::LOutput::setMode() in such cases is a no-op.

### Custom LOutput

As you might expect, Louvre provides a default mechanism for handling all these events, which is why you were able to see content rendered on your screens previously. Now, let's take control by overriding these virtual methods and creating our custom Louvre::LOutput subclass named `EOutput`.

#### src/EOutput.h

```cpp
#ifndef EOUTPUT_H
#define EOUTPUT_H

#include <LOutput.h>

using namespace Louvre;

class EOutput : public LOutput
{
public:
    EOutput(const void *params);

    void initializeGL() override;
    void moveGL() override;
    void resizeGL() override;
    void paintGL() override;
    void uninitializeGL() override;

    LTexture *wallpaper { nullptr };
};

#endif // EOUTPUT_H
```

Here, we are simply overriding all Louvre::LOutput events. Additionally, we have defined a null pointer for a texture that we will use later as our background wallpaper.

> The reason why we will allocate the texture on the heap is that we will only use it while the output is initialized, and we will free it when uninitialized. An Louvre::LOutput is not destroyed when uninitialized, it only gets destroyed when it is no longer available/plugged or during compositor uninitialization.

#### src/EOutput.cpp

```cpp
#include <LPainter.h>
#include "EOutput.h"

EOutput::EOutput(const void *params) : LOutput(params) {}

void EOutput::initializeGL()
{
    painter()->setClearColor(0.1f, 0.3f, 0.8f, 1.f);
    repaint();
}

void EOutput::moveGL()
{
    repaint();
}

void EOutput::resizeGL()
{
    repaint();
}

void EOutput::paintGL()
{
    painter()->clearScreen();
}

void EOutput::uninitializeGL() {}
```

Within the code, we:

1. Modify the clear color in Louvre::LOutput::initializeGL().
2. Trigger Louvre::LOutput::repaint() when the output position or size changes.
3. In Louvre::LOutput::paintGL(), we simply clear the screen every frame without additional operations.
4. Finally, in Louvre::LOutput::uninitializeGL(), no actions are taken as we haven't allocated any resources in Louvre::LOutput::initializeGL() to this point.

### Overriding the Virtual Constructor

To let Louvre use our custom `EOutput` class, we must complete one remaining task: overriding its virtual constructor, specifically Louvre::LCompositor::createOutputRequest().

#### src/ECompositor.h

```cpp
#ifndef ECOMPOSITOR_H
#define ECOMPOSITOR_H

#include <LCompositor.h>

using namespace Louvre;

class ECompositor : public LCompositor
{
public:
    ECompositor();

    void initialized() override;

    // Virtual constructors
    LOutput *createOutputRequest(const void *params) override;
};

#endif // ECOMPOSITOR_H
```

The graphic backend invokes this virtual constructor whenever it necessitates generating a new output instance. Subsequently, it employs this freshly generated output as an interface for invoking the `...GL()` events previously discussed.

#### src/ECompositor.cpp

```cpp
#include "ECompositor.h"
#include "EOutput.h"

ECompositor::ECompositor() {}

void ECompositor::initialized() 
{ 
    // ... 
}

LOutput *ECompositor::createOutputRequest(const void *params)
{
    return new EOutput(params);
}
```

> Whenever you see `// ...` in a code snippet, it means that code has been omitted to reduce boilerplate.

Instead of returning a new Louvre::LOutput instance as the defaults do, we return our custom `EOutput` class.

If you recompile and execute the compositor now, you will only see a bluish background color, probably the cursor as well, but no application windows (surfaces). 

<img src="https://lh3.googleusercontent.com/pw/ADCreHffyDKxkn-ciELGt5uBedV-47WD43mWV2HhFsf1TGT6PbSDiJussQ7Wr01A0-Xm-HYlqPobvKufFkFoFOLhWXrWIcEdmT6FMaHH6lk5BztJ_XnCuBU=w2400"/>

Let's proceed to learn how to render textures.

## Painting

Louvre offers various ways to help you render content on outputs. You can use your own OpenGL shaders, the Louvre::LPainter functions (which use an internal shader), or work with the Louvre::LScene and Louvre::LView classes. In this section, we will begin by exploring how to render using Louvre::LPainter, and later, we will cover the use of the Louvre::LScene and Louvre::LView classes. We will also discuss why the latter option may be your preferred choice.

### Rendering with LPainter

To begin, let's render our background wallpaper. I'll be using this nice Louvre-themed wallpaper, typically located on your system at `/usr/local/share/Louvre/assets/wallpaper.png`, although the path may vary depending on your system. Feel free to choose any image of your preference for this purpose.

<img src="https://github.com/CuarzoSoftware/Louvre/blob/devel/src/examples/louvre-views/assets/wallpaper.png?raw=true"/>

#### src/EOutput.cpp

```cpp
#include <LCompositor.h>
#include <LPainter.h>
#include <LTexture.h>
#include <LOpenGL.h>
#include <LLog.h>
#include "EOutput.h"

// ...

void EOutput::initializeGL()
{
    wallpaper = LOpenGL::loadTexture(compositor()->defaultAssetsPath() / "wallpaper.png");

    if (!wallpaper)
        LLog::error("[louvre-example] Failed to load wallpaper.");

    repaint();
}

// ...

void EOutput::paintGL()
{
    LPainter *p { painter() };

    // Draw the wallpaper
    if (wallpaper)
        p->drawTexture(wallpaper,
                       LRect(0, 0, wallpaper->sizeB().w(), wallpaper->sizeB().h()),
                       rect());
}

void EOutput::uninitializeGL()
{
    if (wallpaper)
        delete wallpaper;
}
```

Now, in Louvre::LOutput::initializeGL(), we load our wallpaper image using the Louvre::LOpenGL::loadTexture() method. This method supports all image formats that [STB Image](https://github.com/nothings/stb) supports and returns `nullptr` if it fails. 

> Just in case you weren't aware, in C/C++, when you use `if (variable)`, the statement is executed only when the variable is anything other than 0. In C++, `nullptr` is a value indicating a null pointer, essentially representing 0. So in this situation, we check whether the wallpaper texture is not equal to `nullptr` before attempting to render or delete it, as failing to do so would lead to a segfault.

In Louvre::LOutput::paintGL(), we draw the wallpaper using the Louvre::LPainter::drawTexture() method. This method takes three (or more) arguments: the first is the texture we want to render, the second is a rect within the texture that we want to render (in this case, we're using the entire texture), and the third argument defines where we want to draw our texture, in this case we pass the position and size of the output (Louvre::LOutput::rect() which is equivalent to `(Louvre::LOutput::pos(), Louvre::LOutput::size())`) to fully cover it.

It's important to note that Louvre uses both buffer and surface coordinates. 

### Buffer Coordinates

Buffer coordinates represent the maximum granularity within an object, independently of the scale associated with it. For example, if the wallpaper image was 1000x500px, its size in buffer coordinates would be the same, 1000x500. Methods and properties that use buffer coordinates have the "B" suffix, such as Louvre::LTexture::sizeB().

### Surface Coordinates

Surface coordinates are a concept from the Wayland protocol but are not exclusive to surfaces. For instance, if one of your outputs has a resolution of 1000x1000px and you assigned it a scale factor of 2 (because it is HiDPI), the size of the output in buffer coordinates (Louvre::LOutput::sizeB()) would be the same (1000x1000), but in surface coordinates (Louvre::LOutput::size()), it would be 500x500. Surface coordinates are essentially defined as buffer coordinates divided by the scale factor of the object. Using surface coordinates simplifies continuous transformations across different scales and allows you to work in an consistent system while rendering.

In the draw operation, for instance, we use buffer coordinates to define the texture's subrect we want to draw and surface coordinates to define the destination rectangle on the output where we want to draw it. This approach ensures that if you're working with multiple outputs with different scales, the image gets automatically scaled by Louvre::LPainter, eliminating the need to handle transformations manually each time. 

If you were using your own shaders, you would need to manage scaling transformations yourself. In that case, you can access the OpenGL texture ID with Louvre::LTexture::id() and target with Louvre::LTexture::target().

> In Louvre::LPainter::drawTexture(), there is a fourth parameter known as `srcScale`, which defaults to 1. This default value is why we defined the `src` rect parameter in buffer coordinates. Change this parameter if you have the source rect defined in surface coordinates at a specific scale and wish to avoid the need to transform it back to buffer coordinates.

Lastly, in Louvre::LOutput::uninitializeGL(), we destroy the texture if it was successfully loaded previously.

If you recompile and run, you should now see your wallpaper displayed, but you won't see any surface yet since we haven't drawn them. Let's proceed to implement the rendering of surfaces.

### Rendering Surfaces

In Wayland, surfaces represent what you would typically think of as application windows. So, in the context of Louvre, each surface has a texture that we can draw.\n
Surfaces, on their own, aren't very useful, they need to have a role assigned before the compositor can use them, which provide rules of how to stack them, position them and more. For instance, a client may assign the role of a cursor to a surface (Louvre::LCursorRole), allowing the compositor to use its texture as a cursor (duh), or as a drag & drop icon (Louvre::LDNDIconRole), or as a toplevel window (Louvre::LToplevelRole), as a popup (Louvre::LPopupRole), etc. We will delve deeper into surfaces later, but for now, let's focus on how to render them.

The Louvre::LCompositor class contains a list of all surfaces created by all clients (Louvre::LCompositor::surfaces()). This list always maintains the correct rendering order, from bottom to topmost surfaces. Surface roles usually define a specific stacking order that must be respected by the compositor to function properly. Therefore, by using this list, you can ensure that the order is respected.

So, let's modify Louvre::LOutput::paintGL() to draw the available surfaces.

#### src/EOutput.cpp

```cpp
#include <LPainter.h>
#include <LTexture.h>
#include <LOpenGL.h>
#include <LLog.h>
#include <LDNDManager.h>
#include <LDNDIconRole.h>
#include <LCompositor.h>
#include <LCursor.h>
#include <LSeat.h>
#include "EOutput.h"

// ...

void EOutput::paintGL()
{
    LPainter *p { painter() };
    LPainter::TextureParams texParams;
    LRect surfaceRect;
 
    // Draw the wallpaper
    if (wallpaper)
        p->drawTexture(wallpaper,
                       LRect(0, 0, wallpaper->sizeB().w(), wallpaper->sizeB().h()),
                       LRect(pos(), size()));
 
    // Raise a drag & drop icon
    if (seat()->dndManager()->icon())
        seat()->dndManager()->icon()->surface()->raise();
 
    // Draw every surface
    for (LSurface *s : compositor()->surfaces())
    {
        // Skip some surfaces
        if (!s->mapped() || s->minimized() || s->cursorRole())
        {
            if (s->cursorRole())
                s->requestNextFrame();
            continue;
        }
 
        // Current surface rect
        surfaceRect = LRect(s->rolePos(), s->size());
 
        // Calculate which outputs the surface intersects
        for (LOutput *o : compositor()->outputs())
        {
            if (o->rect().intersects(surfaceRect))
                s->sendOutputEnterEvent(o);
            else
                s->sendOutputLeaveEvent(o);
        }
 
        texParams = 
        {
            .texture = s->texture(),
            .pos = surfaceRect.pos(),
            .srcRect = s->srcRect(),
            .dstSize = surfaceRect.size(),
            .srcTransform = s->bufferTransform(),
            .srcScale = (Float32)s->bufferScale()
        };

        // Bind the surface texture params
        p->bindTextureMode(texParams);

        // Draw the surface
        p->drawRect(surfaceRect);

        // Notify the client it can now render a new surface frame
        s->requestNextFrame();
    }
 
    // Manually draw the cursor if hardware composition is not supported
    if (!cursor()->hasHardwareSupport(this))
    {
        p->drawTexture(
            cursor()->texture(),
            LRect(0, cursor()->texture()->sizeB()),
            cursor()->rect());
    }
}

// ...
```

There's a lot to cover, so let's break it down step by step:

#### src/EOutput.cpp

```cpp
// Raise a drag & drop icon
if (seat()->dndManager()->icon())
    seat()->dndManager()->icon()->surface()->raise();
```
In this section, we are checking if there is an ongoing drag & drop session with an icon. If that's the case, we use Louvre::LSurface::raise() to elevate the icon surface. This action places the icon at the end of the surfaces list, ensuring that it will be rendered above the rest of the surfaces.

#### src/EOutput.cpp

```cpp
// Skip some surfaces
if (!s->mapped() || s->minimized() || s->cursorRole())
{
    if (s->cursorRole())
        s->requestNextFrame();
    continue;
}
```
In the for loop, we have several conditions to determine whether or not to render a particular surface. Let's break down these conditions:

1. We use the Louvre::LSurface::mapped() property to decide if the surface should be rendered. This property returns `false` when the surface does not have a buffer, does not have a role, or if the client wants to hide it.

2. We skip rendering surfaces that are minimized or have the Louvre::LCursorRole since we use the Louvre::LCursor class to render them using hardware composition (we'll cover rendering cursors with Louvre::LCursor later).

3. We call Louvre::LSurface::requestNextFrame(), which allows us to control the rate at which clients repaint their surfaces. It notifies the client that it's a good moment to render new content for the given surface. We should avoid calling it if the surface is obscured by other surfaces or if it's not visible. In this case, we are always letting clients update their cursor surfaces.

#### src/EOutput.cpp

```cpp
// Current surface rect
surfaceRect = LRect(s->rolePos(), s->size());

// Calculate which outputs the surface intersects
for (LOutput *o : compositor()->outputs())
{
    if (o->rect().intersects(surfaceRect))
        s->sendOutputEnterEvent(o);
    else
        s->sendOutputLeaveEvent(o);
}
```

Next, we determine on which outputs the surface is currently visible and notify the client using Louvre::LSurface::sendOutputEnterEvent() and Louvre::LSurface::sendOutputLeaveEvent(). This informs clients about the outputs where their surfaces are currently visible, allowing them to adjust the surface scale to match the scale of the output (surfaces also have a scale factor).

#### src/EOutput.cpp

```cpp
texParams = 
{
    .texture = s->texture(),
    .pos = surfaceRect.pos(),
    .srcRect = s->srcRect(),
    .dstSize = surfaceRect.size(),
    .srcTransform = s->bufferTransform(),
    .srcScale = (Float32)s->bufferScale()
};

// Bind the surface texture params
p->bindTextureMode(texParams);

// Draw the surface
p->drawRect(surfaceRect);

// Notify the client it can now render a new surface frame
s->requestNextFrame();
```

Following this, we proceed to render the surface, but this time using Louvre::LPainter::bindTextureMode(), allowing us to virtually map a texture on the screen based on parameters set in the Louvre::LPainter::TextureParams struct. We then render it or portions of it using Louvre::LPainter::drawRect(), Louvre::LPainter::drawBox(), or Louvre::LPainter::drawRegion(). This approach minimizes transfers to the GPU when rendering regions of textures, such as the Louvre::LSurface::damage() region of a surface, which we will discuss later.

Unlike Louvre::LPainter::drawTexture(), this approach also enables us to specify custom buffer transforms. When applying a custom transform to an output, clients are notified and encouraged to render their content in a buffer with the same transform, allowing for direct scan-out (refer to Louvre::LSurface::bufferTransform()). Using Louvre::LPainter::drawTexture() in these cases would result in surfaces being displayed rotated, flipped, or clipped.

#### src/EOutput.cpp

```cpp
// Manually draw the cursor if hardware composition is not supported
if (!cursor()->hasHardwareSupport(this))
{
    p->drawTexture(
        cursor()->texture(),
        LRect(0, cursor()->texture()->sizeB()),
        cursor()->rect());
}
```

Finally, we check if hardware cursor compositing is available. Hardware cursor compositing is a feature found in many hardware systems that allows the display of a cursor on the output in a highly efficient manner, without the need to render it using OpenGL. If hardware composition is not supported, we proceed to render the current cursor using OpenGL. In this case the unique Louvre::LCursor instance provides us the current cursor texture and also the destination rect where we should render it with Louvre::LCursor::rect().

If you recompile and run, you should now see the surfaces coming back to life.

## Poor Performance 😥

While the rendering logic we've just implemented may appear straightforward, it's unfortunately quite inefficient. This inefficiency stems from the fact that we're repainting the entire output in each frame, including the background wallpaper and all the surfaces, even if only a small change occurs or parts of surfaces are hidden by others.

To create an efficient compositor that people would want to use, we should only repaint the regions that change in each frame. To achieve this, we need to consider several factors: the surfaces' opaque and translucent regions, the surfaces' damage region, and the outputs inter-frame damage.

Each surface has both translucent and opaque regions (Louvre::LSurface::translucentRegion() and Louvre::LSurface::opaqueRegion()), which contain lists of non-overlapping rectangles positioned relative to the surface's top-left corner. These regions can help us prevent rendering objects behind opaque regions.

Additionally, each surface has a damage region (Louvre::LSurface::damage()), similar to the opaque and translucent regions, which indicates which parts of the surface have changed and need repainting. This region is cleared after calling Louvre::LSurface::requestNextFrame().

Lastly, we need to keep track of the output damage. Typically, each output uses more than one framebuffer. While one framebuffer is being displayed on the screen, we render to the other buffer, and then we swap, repeating this process to prevent screen tearing.
So in order to handle damage tracking effectively, we would need to use Louvre::LOutput::currentBuffer(). This method provides the index of the buffer we're currently rendering to, which can range from [0] for single buffering to [0,1] for double buffering, or even [0,1,2] for triple buffering, depending on your setup.

You can use the Louvre::LRegion class to perform efficient boolean rectangle operations to calculate the region that needs repainting each frame. However, due to its complexity, we won't cover it here. You can refer to the [louvre-weston-clone](md_md__examples.html#weston) compositor example to see how it's done, but be warned that it can be quite complex.

## Good News! 😃

Fortunately, Louvre provides an excellent solution to this problem: the Louvre::LScene and Louvre::LView system. These classes simplify the process of efficiently rendering only the necessary regions, and we will explore them in the following chapter.

<a href="04.md">◀ Chapter 4: Compositor Initialization</a> || <a href="06.md"> Chapter 6: Rendering with LScene and LViews ▶</a>