# Chapter 6: Rendering with LScene and LViews {#tutorial-scene}

In the previous chapter, we explored the challenges of rendering efficiently using Louvre::LPainter. In this chapter, we'll shift our focus to the Louvre::LScene and Louvre::LView system.

A single Louvre::LScene efficiently manages rendering across all outputs by overseeing multiple child and subchild Louvre::LView instances. You only need to define the stacking order, position, and size of each view, and Louvre::LScene automatically optimizes output rendering for you.

Louvre offers various specialized Louvre::LView types, and you can even create your own ones. These views cater to various purposes, including layer grouping, texture and surface display, solid color rectangles, and even nested scenes.

Each view can host multiple child views and has a single parent. Rendering order begins with the parent view and proceeds to its children (from bottom to top), adhering to the sequence defined in the children list.

### Layers

First, let's establish the structure of our scene. We will create five layers as follows:

1. **Background Layer**: This layer will host views that should always remain behind everything else, such as our wallpaper.
2. **Surfaces Layer**: Typically, this is where all client surfaces will reside.
3. **Overlay Layer**: Here, we'll position overlay UI elements, like a topbar.
4. **Fullscreen Layer**: Toplevel surfaces in fullscreen mode will be displayed here.
5. **Cursor Layer**: Here, we'll display a cursor view if hardware composition is not avaliable and drag & drop icons.

To implement this, we'll create our scene and layers within our `ECompositor` class, ensuring accessibility from any part of our code.

#### src/ECompositor.h

```cpp
#ifndef ECOMPOSITOR_H
#define ECOMPOSITOR_H
 
#include <LCompositor.h>
#include <LScene.h>
#include <LSceneView.h>
#include <LLayerView.h>
#include <LTextureView.h>
 
using namespace Louvre;
 
class ECompositor : public LCompositor
{
public:
    ECompositor();
 
    void initialized() override;

    // Virtual constructors
    LOutput *createOutputRequest(const void *params) override;

    // Scene and layers
    LScene scene;
    LLayerView backgroundLayer  { scene.mainView() };
    LLayerView surfacesLayer    { scene.mainView() };
    LLayerView overlayLayer     { scene.mainView() };
    LLayerView fullscreenLayer  { scene.mainView() };
    LLayerView cursorLayer      { scene.mainView() };
    LTextureView softwareCursor { nullptr, &cursorLayer };
};
 
#endif // ECOMPOSITOR_H
```

Here, we are defining our layers using Louvre::LLayerView and also creating an Louvre::LTextureView that we'll use to display the cursor when hardware composition is not available.

We add the layers to Louvre::LScene::mainView() in the order in which they will be rendered, from bottom to top, and initialize the `softwareCursor` view with a `nullptr` texture, automatically hiding it.

> Each view type has an Louvre::LView::mapped() property that the scene uses to determine whether it meets the criteria for being drawn. For instance, in the case of Louvre::LTextureView, having a non-nullptr texture is one of these conditions.

#### src/ECompositor.cpp

```cpp
#include <LSeat.h>
#include <LKeyboard.h>
#include "ECompositor.h"
#include "EOutput.h"
 
ECompositor::ECompositor()
{
    // Set black as the scene clear color
    scene.mainView()->setClearColor(0.f, 0.f, 0.f, 1.f);

    // This allow us to define a custom size to an LTextureView
    softwareCursor.enableDstSize(true);
}

// ...
```

Furthermore, we set the background color for the Louvre::LScene to black and finally, we call `softwareCursor.enableDstSize(true)` which allow us to specify a size to the `softwareCursor` view that is independent from its texture and buffer scale.


### Integrating LScene with LOutput

To enable our scene to render into our outputs, we need to "plug it" into them like this:

#### src/EOutput.cpp

```cpp
#include <LPainter.h>
#include <LTexture.h>
#include <LOpenGL.h>
#include <LLog.h>
#include <LDNDManager.h>
#include <LCompositor.h>
#include <LCursor.h>
#include <LSeat.h>
#include "EOutput.h"
#include "ECompositor.h"

EOutput::EOutput(const void *params) : LOutput(params) {}

void EOutput::initializeGL()
{
    ECompositor *comp { (ECompositor*)compositor() };
    comp->scene.handleInitializeGL(this);
    repaint();
}

void EOutput::moveGL()
{
    ECompositor *comp { (ECompositor*)compositor() };
    comp->scene.handleMoveGL(this);
    repaint();
}

void EOutput::resizeGL()
{
    ECompositor *comp { (ECompositor*)compositor() };
    comp->scene.handleResizeGL(this);
    repaint();
}

void EOutput::paintGL()
{
    ECompositor *comp { (ECompositor*)compositor() };
    comp->scene.handlePaintGL(this);
}

void EOutput::uninitializeGL()
{
    ECompositor *comp { (ECompositor*)compositor() };
    comp->scene.handleUninitializeGL(this);
}
```

It's crucial to plug the scene into all these events, as it needs to initialize resources before it can handle Louvre::LOutput::paintGL() events and release resources when an output is uninitialized.

If you recompile and run the compositor, you will only see a black background because we've only added layer views, which are invisible by themselves, and the cursor view doesn't have a texture yet.

You might have also observed that we constantly need to cast Louvre::LCompositor into our own `ECompositor` subclass, which can become cumbersome and verbose over time. To streamline this, let's create a global class called "G" to store quick access to global variables like that, eliminating the need for repetitive casting.

#### src/Global.h

```cpp
#ifndef G_H
#define G_H

#include <LCompositor.h>

using namespace Louvre;

class ECompositor;
class EOutput;

class G
{
public:
    static inline ECompositor *compositor()
    {
        return (ECompositor*)LCompositor::compositor();
    }

    static inline std::vector<EOutput*> &outputs()
    {
        return (std::vector<EOutput*>&)LCompositor::compositor()->outputs();
    }

    static LScene *scene();
};

#endif // G_H
```

#### src/Global.cpp

```cpp
#include "Global.h"
#include "ECompositor.h"

LScene *G::scene()
{
    return &compositor()->scene;
}
```

Now, we can easily access our compositor and scene from anywhere using `G::compositor()` and `G::scene()`, as well as the vector of initialized outputs with `G::outputs()`. You only need to include the `Global.h` header, of course.

### Wallpaper

Let's explore how to display our wallpaper in a more sophisticated manner. This time, we'll ensure the texture covers the output while maintaining its aspect ratio, rather than simply scaling it to the output size like before.

To display the wallpaper, we can use Louvre::LTextureView.

#### src/EOutput.h

```cpp
#ifndef EOUTPUT_H
#define EOUTPUT_H

#include <LOutput.h>
#include <LTextureView.h>
#include "ECompositor.h"
#include "Global.h"

using namespace Louvre;

class EOutput : public LOutput
{
public:
    EOutput(const void *params);

    void initializeGL() override;
    void moveGL() override;
    void resizeGL() override;
    void paintGL() override;
    void uninitializeGL() override;

    void updateWallpaper();
    LTextureView wallpaperView { nullptr, &G::compositor()->backgroundLayer };
};

#endif // EOUTPUT_H
```

Notice that we've replaced the wallpaper Louvre::LTexture pointer with an Louvre::LTextureView and added an `updateWallpaper()` method. Within this method, we will accurately crop and scale our wallpaper each time the output size changes.

#### src/EOutput.cpp

```cpp
#include <LPainter.h>
#include <LTexture.h>
#include <LOpenGL.h>
#include <LLog.h>
#include <LDNDManager.h>
#include <LCompositor.h>
#include <LCursor.h>
#include <LSeat.h>
#include <LTextureView.h>
#include "EOutput.h"
#include "Global.h"

EOutput::EOutput(const void *params) : LOutput(params)
{
    /* Texture views have a default nullptr translucent region, which means they are considered
     * completely translucent by the scene. However, setting an empty LRegion (not nullptr) indicates
     * to the scene that the view is completely opaque. */
    const LRegion emptyRegion;
    wallpaperView.setTranslucentRegion(&emptyRegion);
}

void EOutput::initializeGL()
{
    updateWallpaper();
    G::scene()->handleInitializeGL(this);
    repaint();
}

void EOutput::moveGL()
{
    wallpaperView.setPos(pos());
    G::scene()->handleMoveGL(this);
    repaint();
}

void EOutput::resizeGL()
{
    Int32 totalWidth { 0 };

    for (EOutput *o : G::outputs())
    {
        o->setPos(LPoint(totalWidth, 0));
        totalWidth += o->size().w();
    }

    updateWallpaper();
    G::scene()->handleResizeGL(this);
    repaint();
}

void EOutput::paintGL()
{
    G::scene()->handlePaintGL(this);
}

void EOutput::uninitializeGL()
{
    G::scene()->handleUninitializeGL(this);

    /* When a texture used by an LTextureView is destroyed,
     * the view's texture is automatically set to nullptr. */
    if (wallpaperView.texture())
        delete wallpaperView.texture();
}

void EOutput::updateWallpaper()
{
    if (wallpaperView.texture())
    {
        /* If the current wallpaper size is equal to the output size
         * then the output simply changed its scale */
        if (wallpaperView.texture()->sizeB() == sizeB())
        {
            wallpaperView.setBufferScale(scale());
            wallpaperView.setPos(pos());
            return;
        }

        delete wallpaperView.texture();
    }

    LTexture *originalWallpaper {
        LOpenGL::loadTexture(compositor()->defaultAssetsPath() / "wallpaper.png") };

    if (!originalWallpaper)
    {
        LLog::error("[louvre-example] Failed to load wallpaper.");
        return;
    }

    const LSize &originalSize { originalWallpaper->sizeB() };

    /* This section calculates the source wallpaper rect that needs to be copied and scaled to cover the entire output
     * while maintaining the image's aspect ratio. */

    // Source rect within the texture to clip
    LRect srcRect { 0 };

    // Calculate the texture's width if it were scaled to the output's height, while preserving the aspect ratio
    const Float32 scaledWidth { Float32(size().w() * originalSize.h()) / Float32(size().h()) };

    // If the scaled width is greater than or equal to the output's width, we clip the texture's left and right sides
    if (scaledWidth >= originalSize.w())
    {
        srcRect.setW(originalSize.w());
        srcRect.setH((originalSize.w() * size().h()) / size().w());
        srcRect.setY((originalSize.h() - srcRect.h()) / 2);
    }
    // Otherwise, we clip the texture's top and bottom
    else
    {
        srcRect.setH(originalSize.h());
        srcRect.setW((originalSize.h() * size().w()) / size().h());
        srcRect.setX((originalSize.w() - srcRect.w()) / 2);
    }

    // Copy the srcRect of the original wallpaper and scale it to match the output buffer size
    wallpaperView.setTexture(originalWallpaper->copyB(sizeB(), srcRect));

    // Set the buffer scale of wallpaperView to match the output scale
    wallpaperView.setBufferScale(scale());

    // Delete the original wallpaper texture since we are using the scaled copy
    delete originalWallpaper;
    wallpaperView.setPos(pos());
}
```

This is indeed a lot of information to digest, so let's break it down step by step. 

First, in the constructor, we use an empty Louvre::LRegion (not the same as `nullptr`) to set the `wallpaperView` translucent region, which means the entire view will be considered opaque. If we were to pass `nullptr`, the entire view would be translucent, which is the default value. We make it opaque intentionally because otherwise the background would unnecessary be repainted each time the wallpaper is repainted, consuming additional processing power.\n

In Louvre::LOutput::initializeGL() we call our new `updateWallpaper()` method, where we carry out the following steps:

1. We delete the previous wallpaper if there was any.
2. We load the original wallpaper image again.
3. We create a cropped and scaled version using the Louvre::LTexture::copyB() method to ensure it covers the entire output while maintaining the wallpaper's original aspect ratio.
4. We update the wallpaper buffer scale and position.

It's important to note that each Louvre::LTextureView has a default buffer scale of 1. Since we are setting the buffer size of the texture to match the output buffer size, we also need to ensure their scales match. This is because Louvre::LScene internally uses Louvre::LPainter for rendering, and it expects the position and size of each view to be in surface coordinates. If the output is HIDPi and we assign it a scale of 2, while keeping the wallpaper view at a scale of 1, it would appear twice as big. This may sound counterintuitive, but think of it this way, if both the wallpaper and the output have the same scale, then no transformations are required by Louvre::LPainter to render the view on the output.

Alternatively, starting from Louvre version 1.2, you can achieve the same cropping effect by specifying a custom source rect for Louvre::LTextureView (refer to Louvre::LTextureView::setSrcRect()). However, creating a cropped and scaled version of the texture that fits perfectly on the screen can enhance the compositor's performance, especially in cases where the texture is larger than the screen resolution. 

In Louvre::LOutput::moveGL(), we simply set the wallpaper position equal to the output position. 

In Louvre::LOutput::resizeGL(), we re-arrange our initialized outputs and call `updateWallpaper()` again so that it gets resized to match the new output resolution or scale.

Finally, in Louvre::LOutput::uninitializeGL(), we check if the wallpaper view has a texture attached and delete it. 

It's worth noting that a texture is not automatically deleted when a view using it is deleted. This is because you can use the same texture in multiple views at a time. However, when a texture that is currently in use by texture views is deleted, these views' textures are automatically set to `nullptr`, consequently hiding them.

Try recompiling and running the compositor again, and you should see your wallpaper displayed.

> Remember that you need to run `meson setup build --wipe` each time you add or remove source files.

### Topbar

Now, let's create a topbar that will serve as the designated space for displaying minimized window thumbnails.\n
Instead of opting for a straightforward Louvre::LSolidColorView to display a solid color rectangle, we're taking a more versatile approach by introducing a custom class called `ETopbar`.

First, let's define the top bar's height in a macro within the `Global.h` header for easy access, as we'll be using it in various parts of our code.

#### src/Global.h

```cpp
#ifndef G_H
#define G_H

#include <LCompositor.h>

#define TOPBAR_HEIGHT 32

// ...

#endif // G_H
```

I'm using 32, but feel free to adjust it to your preference.

#### src/ETopbar.h

```cpp
#ifndef ETOPBAR_H
#define ETOPBAR_H

#include <LSolidColorView.h>
#include "ECompositor.h"
#include "Global.h"

using namespace Louvre;

class EOutput;

class ETopbar : public LObject
{
public:
    ETopbar(EOutput *output);
    ~ETopbar();

    void destroyThumbnails();
    void copyThumbnailsFromAnotherTopbar();
    void updateGUI();

    EOutput *output;
    LSolidColorView view {
        0.f,                            // Red
        0.f,                            // Green
        0.f,                            // Blue
        0.8f,                           // Alpha
        &G::compositor()->overlayLayer  // Parent view
    };
};

#endif // ETOPBAR_H
```

In this class, we define an `EOutput` object as a constructor parameter, which we will store in the `output` variable. This will enable us to precisely adjust the top bar's size and position relative to its associated output when needed (each output will have its own topbar).

Furthermore, we've incorporated a black and semi translucent Louvre::LSolidColorView to visually represent the top bar within the scene. As the name implies, Louvre::LSolidColorView is designed for displaying solid color rectangles.

We've also introduced an `updateGUI()` method, responsible for dynamically managing the top bar's position, size, and ordering child thumbnail elements as necessary.

Finally the `destroyThumbnails()` and `copyThumbnailsFromAnotherTopbar()` methods are self explanatory.

#### src/ETopbar.cpp

```cpp
#include "ETopbar.h"
#include "EOutput.h"
#include "Global.h"

ETopbar::ETopbar(EOutput *output) : output(output) {}

ETopbar::~ETopbar()
{
    destroyThumbnails();
}

void ETopbar::destroyThumbnails()
{
    while (!view.children().empty())
        delete view.children().back();
}

void ETopbar::copyThumbnailsFromAnotherTopbar()
{
    updateGUI();
}

void ETopbar::update()
{
    view.setSize(output->size().w(), TOPBAR_HEIGHT);
    view.setPos(output->pos());
    output->repaint();
}
```

In the constructor, we're straightforwardly storing the output pointer.

In `destroyThumbnails()` and destructor, we destroy the children views of the topbar that represent the thumbnails of minimized surfaces (we will cover that later).

In `copyThumbnailsFromAnotherTopbar()`, for now, all we do is call `updateGUI()`. Later, we will see how to copy the thumbnails.

In `updateGUI()`, we ensure that the top bar's width matches the output's width, and we set its height to `TOPBAR_HEIGHT`. We also position it at the same coordinates as the output (we will adjust thumbnails later).

Now we only need to create it from its respective output:

#### src/EOutput.h

```cpp
#ifndef EOUTPUT_H
#define EOUTPUT_H

#include <LOutput.h>
#include <LTextureView.h>
#include "ECompositor.h"
#include "ETopbar.h"
#include "Global.h"

using namespace Louvre;

class ETopbar;

class EOutput : public LOutput
{
public:

    // ...

    ETopbar topbar { this };
};

#endif // EOUTPUT_H
```

And ensure it is appropriately adjusted whenever the output geometry changes.

#### src/EOutput.cpp

```cpp
// ...

#include "ETopbar.h"

// ...

void EOutput::initializeGL()
{
    // ...

    topbar.copyThumbnailsFromAnotherTopbar();
    G::scene()->handleInitializeGL(this);
    
    // ...
}

void EOutput::moveGL()
{
    // ...

    topbar.updateGUI();
    G::scene()->handleMoveGL(this);

    // ...
}

void EOutput::resizeGL()
{
    // ...

    topbar.updateGUI();
    G::scene()->handleResizeGL(this);

    // ...
}

// ...

void EOutput::uninitializeGL()
{
    // ...

    topbar.destroyThumbnails();
}

// ...
```

We also make sure to destroy any thumbnail added to the top bar of surfaces that remain alive after an output is uninitialized.

With these steps in place, you should now be able to admire your elegant black topbar after recompiling and running your compositor.

<img src="https://lh3.googleusercontent.com/pw/ADCreHdSqkWstdqyTsfb1kMmhZwZi_uoydlPmxAhwM5IZ4PdHR3H6xdA2Ahyd8yaKAarLmYlbtzjxaoJomO_mbzvdYeAx8nb9DA-hDvC0ULoyKmyrJh4wII=w2400"/>

### Rendering the Cursor

We have yet to address the scenario in which hardware cursor compositing is not supported. To display the cursor in such cases, we can set the `softwareCursor` texture to Louvre::LCursor::texture() and its size and position to Louvre::LCursor::rect().

As cursor hardware composition can vary across different outputs, we should enable or disable the `softwareCursor` view within each Louvre::LOutput::paintGL() event, just before invoking Louvre::LScene::handlePaintGL(), like this:

#### src/EOutput.cpp

```cpp
//...

void EOutput::paintGL()
{
    // Check if hw cursor is supported
    if (cursor()->hasHardwareSupport(this))
        G::compositor()->softwareCursor.setTexture(nullptr);
    else
    {
        G::compositor()->softwareCursor.setTexture(cursor()->texture());
        G::compositor()->softwareCursor.setPos(cursor()->rect().pos());
        G::compositor()->softwareCursor.setDstSize(cursor()->rect().size());
        G::compositor()->softwareCursor.setVisible(cursor()->visible());
    }

    G::scene()->handlePaintGL(this);
}

//...
```

We perform a check to determine whether the output supports hardware cursor compositing. If it does, we hide the `softwareCursor` view by setting its texture to `nullptr`. Otherwise, we adjust its texture, position, size, and visibility to match those of Louvre::LCursor.

### LScene Internal Behaviour

You've probably noticed that every time a parameter of a view is changed, it doesn't necessarily mean that the view is immediately rendered. You can modify the parameters as many times as you want, but it's only when Louvre::LScene::handlePaintGL() is called that the scene analyzes all the views and exclusively paints the parts that have changed. This means that if, for example, you assign the same parameter repeatedly with the same value to a view, the screen might never be repainted, and the rendering thread might remain blocked (because there is nothing new to render). On the other hand, whenever you change a parameter, the views call Louvre::LOutput::repaint() on all the outputs they are currently visible, which eventually invokes Louvre::LScene::handlePaintGL().

### Performing Advanced Rendering

A question you might also be asking yourself is why it is necessary to manually plug all the scene `handle..GL` methods in each event, and why Louvre doesn't do it automatically for you. The reason for this is that Louvre::LScene specializes only in 2D rendering, but there may be occasions when you want to perform 3D rendering or add sophisticated effects/animations. In those cases, you can perform rendering using your own methods before or after calling Louvre::LScene::handlePaintGL(), or even not calling it at all.

That's an alternative. If you want to exclusively use the scene and view system, you have also the option to create your own customized views and even build a complete toolkit around them. To do so, you'd need to create your own subclasses of Louvre::LView and override various of its virtual methods, such as Louvre::LView::paintEvent(), which is invoked by Louvre::LScene or Louvre::LSceneView when they request the view to repaint one of its regions.

For more information on creating your own views, refer to the Louvre::LView documentation.

### Surfaces

Rendering surfaces introduces a bit more complexity as we must keep track of when they are created and monitor changes in their stacking order. In the upcoming chapter, we will delve deeply into surfaces, exploring how to render them using Louvre::LSurfaceView, and handle specific events, such as detecting when they are minimized. See you there!

<a href="05.md">◀ Chapter 5: Rendering with LPainter</a> || <a href="07.md"> Chapter 7: Surfaces ▶</a>