# Chapter 7: Surfaces {#tutorial-surfaces}

You may already have an idea of what a surface is, but let's clarify it. Think of a surface as a canvas created by a Wayland client, which the compositor can render on the screen. The client can request the compositor to use it as a cursor, a drag & drop icon, a toplevel window, and so on. This specific use assigned by the client is known as the surface role. Each surface role provides rules on how the compositor should position, stack, and interact with the surface. Refer to the Louvre::LSurface documentation for additional information on the available roles.

You can determine the assigned role of a surface using Louvre::LSurface::roleId() and access it through Louvre::LSurface::role(). Alternatively, you can directly cast it using methods like Louvre::LSurface::cursorRole(), Louvre::LSurface::dndIcon(), Louvre::LSurface::toplevel(), etc.\n 

> All surface roles are derived from the Louvre::LBaseSurfaceRole class.

For example, to check if a surface has a toplevel role, you can use Louvre::LSurface::toplevel(), which may return an Louvre::LToplevelRole instance or `nullptr` if it has a different role or none at all. We will delve into the details of each specific role in future chapters, so please be patient.

### Position

As mentioned earlier, each role provides information on how the compositor should position the surface. Typically, this position is either a local offset relative to a parent surface or an offset relative to a position of your choice. For instance, toplevel surfaces for example, representing typical app windows with a title bar, close, minimize, and maximize buttons, often have window decorations such as shadows. 
When positioning a toplevel window, you want to place it relative to its top-left corner but ignore its decorations. This is why toplevels offer a window geometry rect (Louvre::LToplevelRole::windowGeometry()) that specifies which part of its texture belongs to the window and which part constitutes decorations. 

<center>
<img style="width:25vw" src="https://lh3.googleusercontent.com/pw/AIL4fc9WuCZOWGGC6v7ZepWA7VOWygycmNFEngEHX7J9ODdmzm1Qs4au1KYPMRJ_102tfvOXFBcL-Kwghm0bSKbd4U6der3p94l1drpn3ONqb9yfv-uFgHQ=w2400"/>
</center>

The role doesn't dictate where to position the toplevel but provides the offset necessary for correct positioning.

To set the position of a surface, you can use Louvre::LSurface::setPos(). If you then call Louvre::LSurface::pos(), it will return the same position. However, if you call Louvre::LSurface::rolePos() and the surface has a toplevel role, it will return the position you assigned minus the decoration offset. In this case, Louvre::LSurface::rolePos() internally invokes Louvre::LToplevelRole::rolePos().
Louvre implements the default position logic for each role, but you have the option to override it if needed. However, it's worth noting that certain positioning rules can be challenging to implement, such as Louvre::LPopupRole::rolePos().

Now, let's consider another example: the subsurface role (Louvre::LSubsurfaceRole). A subsurface is always a child of another surface and is positioned relative to the parent surface's position. In this case, the position set with Louvre::LSurface::setPos() is disregarded, and Louvre::LSurface::rolePos() simply returns the parent surface's position plus the offset defined by the client (check Louvre::LSubsurfaceRole::rolePos()).

In summary, employ Louvre::LSurface::setPos() to define the surface position without applying any additional role position rules. And use Louvre::LSurface::rolePos() to retrieve the actual position when rendering its texture.

### Size and Scale

A surface's size is determined by its texture size and its scale. When you use Louvre::LSurface::sizeB(), it returns the texture size, disregarding the scale. Conversely, Louvre::LSurface::size() provides the size in surface coordinates, which is the texture size divided by the scale.

Just as we've seen with outputs, a client may render the surface content into a HiDPI buffer when displayed on a HiDPI output. In such cases, it informs us that it's using a scale of 2 (or greater), allowing us to render it properly.

> Since Louvre 1.2.0, the Viewporter and Fractional Scale protocols have been introduced. With these protocols, clients are now allowed to define a custom rect of their texture to use as a source and scale it to any arbitrary size, thereby removing the direct relationship of Louvre::LSurface::size() with the buffer size and scale. However, this does not affect how you need to handle their rendering. You can continue using Louvre::LSurface::srcRect() and Louvre::LSurface::size() as always. Additionally, properties such as input, opaque, translucent, and damage regions are automatically scaled to the specified Louvre::LSurface::size() by Louvre.

### Stacking Order

As we've seen before, the Louvre::LCompositor class maintains a list of all client surfaces (Louvre::LCompositor::surfaces()) that defines their stacking order.\n
Certain surface roles require us to adhere to their provided stacking order, such as Louvre::LSubsurfaceRole and Louvre::LPopupRole.\n
As illustrated in the following image, you can observe a cluster of popups, with each one neatly stacked on top of its parent popup.

<center>
<img style="width:35vw" src="https://lh3.googleusercontent.com/6caGayutKKWqndpd6ogno2lPw8XGELxnFums4gvkWZKOJYO0762yVG3mHLrc1rw63r1eEJabEdW9F5AA2BDTFCtpB_hiPlTY4FkKlHfH1B-2MdLvXCD6RuwZOZvhOl6EhydtsOYGPw=w2400"/>
</center>

You have the flexibility to modify this order tho, for example, by calling Louvre::LSurface::raise(). This action brings the specified surface to the top while ensuring that the stacking order of its role (or other surfaces' roles) is respected, so it may raise other surfaces as well. 

Additionally, you can detect when the order of a surface changes in the list, overriding the Louvre::LSurface::orderChanged() virtual method.

To determine which surface is positioned behind or in front of a particular surface, you can use Louvre::LSurface::prevSurface() and Louvre::LSurface::nextSurface(). However, be careful, as these methods might return `nullptr` if the surface is at the beginning or end of the surfaces list.

> You should never call Louvre::LSurface::raise() while iterating through the Louvre::LCompositor::surfaces() list, as doing so could result in a segfault.

## Rendering Surfaces

Hoping that all that theory didn't bore you to death, it's now time to delve into something more enjoyable – rendering the surfaces in our scene. To achieve this, we'll make use of the Louvre::LSurfaceView. This view inherits all the properties of the surface, like Louvre::LSurface::rolePos() for positioning, Louvre::LSurface::size() to determine its size, and other attributes like opaque/translucent regions, damage, and more.

So, let's get started by creating our very own Louvre::LSurface subclass, which we will of course name `ESurface`.

#### src/ESurface.h

```cpp
#ifndef ESURFACE_H
#define ESURFACE_H

#include <LSurface.h>
#include <LSurfaceView.h>
#include "ECompositor.h"
#include "Global.h"

using namespace Louvre;

class ESurface : public LSurface
{
public:
    ESurface(const void *params);

    void roleChanged() override;
    void orderChanged() override;

    LSurfaceView view { (LSurface*) this, &G::compositor()->surfacesLayer };
};

#endif // ESURFACE_H
```

Here, we're overriding the Louvre::LSurface::roleChanged() event because we want to exclude certain roles from rendering, such as the cursor role.

Additionally, we're overriding Louvre::LSurface::orderChanged(), which we will use for updating the views stacking order.

Lastly, we've introduced an Louvre::LSurfaceView for this surface, adding it to the surfaces layer.

#### src/ESurface.cpp

```cpp
#include <LCursor.h>
#include "ESurface.h"

ESurface::ESurface(const void *params) : LSurface(params)
{
    // Detaches its position from its parent view
    view.enableParentOffset(false);
}

void ESurface::roleChanged()
{
    // Hide cursor surfaces because we use LCursor
    if (cursorRole())
        view.setVisible(false);

    // Move drag & drop icons to the cursor layer so they always
    // appear on top of other views
    else if (dndIcon())
    {
        // Ensure it is positioned behind 'softwareCursor'
        view.setParent(&G::compositor()->cursorLayer);
        view.insertAfter(nullptr, false);
        setPos(cursor()->pos());
    }
}

void ESurface::orderChanged()
{
    ESurface *prev { (ESurface*)prevSurface() };

    /* Surfaces are ordered in a single list, however, our views
     * can be children of the surface, cursor, or fullscreen layers.
     * Therefore, if 'prev' has a different parent, we need to continue searching. */

    while (prev && prev->view.parent() != view.parent())
        prev = (ESurface*)prev->prevSurface();

    if (prev)
        view.insertAfter(&prev->view, false);

    // If there is no prev surface with the same parent, insert it at the beginning of the current parent's children list
    else
        view.insertAfter(nullptr, false);
}
```

In the constructor, we initialize the Louvre::LSurface parent class and disable the Louvre::LView::parentOffsetEnabled() property for the view, which is enabled by default. Disabling it means that when the scene retrieves the position of a view using Louvre::LView::pos(), the position we assigned won't be adjusted by the parent view's position. The position of the Louvre::LSurfaceView, as mentioned earlier, is inherited from the surface role's position, which is already global. Therefore, we disable the parent offset to prevent any additional positioning adjustments.

Within the Louvre::LSurface::roleChanged() event we hide the views that get assigned the cursor role (remember that we are using Louvre::LCursor instead). We also move drag & drop icon views to the cursor layer so that they always appear on top of the rest.

And in Louvre::LSurface::orderChanged(), we update the view stacking order with Louvre::LView::insertAfter(). This inserts the view on top of the surface that precedes it in the surfaces list. If there's no previous surface, we pass `nullptr`, which places the view at the beginning of its parent's children list.\n
You might also notice the `false` second parameter. This setting prevents the view from changing its parent if the view we are inserting it after belongs to another parent.

Once more, to make Louvre use our subclass, we need to override the Louvre::LCompositor::createSurfaceRequest() virtual constructor:

#### src/ECompositor.h

```cpp
// ...

// Virtual constructors
LOutput *createOutputRequest(const void *params) override;
LSurface *createSurfaceRequest(const void *params) override;

// ...
```

#### src/ECompositor.cpp

```cpp
// ...

#include "ESurface.h"

// ...

LSurface *ECompositor::createSurfaceRequest(const void *params)
{
    return new ESurface(params);
}
```

The `params` argument simply contains the resources required for Louvre to correctly construct each Louvre::LSurface instance.

If you recompile and run the compositor now, you should see the application windows coming back to life. Also, don't forget that you can press `F1` to launch `weston-terminal` and `Ctrl + Shift + Esc` to exit the compositor.

Oh, and as you can clearly see in the screenshot, our windows just love to play hide-and-seek behind the topbar. It adds a certain charm, wouldn't you agree? 🥴

<img src="https://lh3.googleusercontent.com/pw/ADCreHcHJvjtARnb8Vdm09fqb4kknHowY_-SlVlCXsYB1A3RKupGuCslPBDm7aMZXiv5nqVG-jFrXdUSueHWPDUO2cM4UzRs9ncCwCLsgk20eBEcEbAx8mQ=w2400"/>

We will address how to fix that later. In the meantime, I'll share a tip for checking if Louvre::LScene is drawing efficiently.\n
It's a good practice to use this trick occasionally to ensure it's working correctly, as there could be something causing inefficient rendering.

To do this, let's go to the `paintGL()` implementation in `EOutput.cpp` and clear the output screen before calling Louvre::LScene::handlePaintGL().

#### src/ECompositor.cpp

```cpp
// ...

void EOutput::paintGL()
{
    // ...

    painter()->clearScreen();
    G::scene()->handlePaintGL(this);

    LRegion damage;
    damage.addRect(rect());
    setBufferDamage(&damage);
}

// ...
```

> We also need to call Louvre::LOutput::setBufferDamage() with a region encompassing the entire output rect. This serves as a hint to the graphic backend, informing it about the regions of the framebuffer that have been modified during the `paintGL()` event. This allows it to selectively copy those regions in certain scenarios, especially in multi-GPU systems where one of the GPUs cannot directly render to its own connectors (outputs) due to the inability to import surface textures, as it lacks [DMA](https://en.wikipedia.org/wiki/Direct_memory_access) support. Louvre::LScene automatically calls this method with damage generated during Louvre::LScene::handlePaintGL(). However, because we called Louvre::LPainter::clearScreen() from outside, we need to manually include the entire damaged area.

Recompile and run the compositor, then launch an app like `gedit`. You'll notice that only the parts of the window that change, such as when you hover the mouse over a button or move a window, are rendered. However, please note that `weston-terminal` is a poor example of a client because it damages its entire surface, even if only a small part of it changes.

<img src="https://lh3.googleusercontent.com/pw/ADCreHeYAYKPI2HYT_TC_-lVn_Z3TEYbaGzKinQjWJWGTSEU_ls0ZH4qx8wysZybtha5pLx6UG-5tSF0NZqtTXUtm8xBkeqNtGnsd21D2A7kW-U3x2WilTo=w2400" />

If you consistently observe that the entire screen or a great portion of it is being rendered in every frame, it is probable that there is an issue in your implementation. This might occur for example if you are frequently altering an output's position, adjusting a wallpaper's position, or manually introducing damage to the scene using methods like Louvre::LSceneView::damageAll() or Louvre::LSceneView::addDamage().

Delete the added lines to restore your compositor to its normal state.

### Displaying Minimized Windows

Let's continue with how to present minimized windows on our topbar. You can modify the minimized state of any surface, regardless of its role, using Louvre::LSurface::setMinimized() and get its current state with Louvre::LSurface::minimized(). Additionally, you can listen to changes in this property by overriding the Louvre::LSurface::minimizedChanged() virtual method.

> When you minimize a surface, all of its child surfaces are also minimized.

Clients can only request to minimize toplevels tho through Louvre::LToplevelRole::setMinimizedRequest(). Typically, this request occurs when a user interacts with the toplevel, such as clicking the minimize button or using a shortcut like `Ctrl + M`. In these cases, the client itself initiates the request. Interestingly, they can only request to minimize a surface but not to unminimize it (check the [XDG Shell protocol](https://wayland.app/protocols/xdg-shell#xdg_toplevel:request:set_minimized)).

Various methods exist to display a thumbnail of a surface in the topbar view. One approach could be to create an additional view for the surface or its texture, positioning it within the topbar view, and then applying scaling using Louvre::LTextureView::setDstSize() or Louvre::LView::setScalingVector(). However, this approach can present challenges, especially with clients like Firefox, which use multiple surface layers for a single window. For instance, Firefox employs a toplevel surface for window decorations and a child subsurface for displaying browser content. Managing and scaling each surface properly in such cases, especially with significant downscaling, can lead to precision loss and a less than ideal appearance.

Another recommended approach, which I'll guide you through, involves creating a temporary Louvre::LSceneView. Each Louvre::LSceneView renders its content in its own framebuffer, allowing you to render all the necessary surfaces into that framebuffer and then create a downscaled copy to display in the topbar. Let's see how to achieve this.

To begin, we'll define the padding, margin, and height of the thumbnails in a macro located within the `Global.h` header. In addition, we will introduce an auxiliary method to facilitate the automatic casting of the Louvre::LCompositor::surfaces() lists into our custom subtype. Furthermore, we'll implement another method for relocating a surface view and all its child surfaces into another view, which will be used multiple times in the subsequent sections.

#### src/Global.h

```cpp
#ifndef G_H
#define G_H

#include <LCompositor.h>

#define TOPBAR_HEIGHT 32

#define TOPBAR_PADDING 4
#define THUMBNAIL_MARGIN 4
#define THUMBNAIL_HEIGHT (TOPBAR_HEIGHT - 2 * TOPBAR_PADDING)

using namespace Louvre;

class ECompositor;
class EOutput;
class ESurface;

class G
{
public:
    
    // ...

    static inline std::list<ESurface*> &surfaces()
    {
        return (std::list<ESurface*>&)LCompositor::compositor()->surfaces();
    }

    static void reparentSurfaceAndChildrenViews(ESurface *surface, LView *newParent, bool subsurfaceRoleOnly = false);
};

#endif // G_H
```

We define the thumbnails height as the topbar height minus twice the topbar padding.

#### src/Global.cpp

```cpp
#include "Global.h"
#include "ECompositor.h"
#include "ESurface.h"

// ...

void G::reparentSurfaceAndChildrenViews(ESurface *surface, LView *newParent, bool subsurfaceRoleOnly)
{
    surface->view.setParent(newParent);

    // The children of a surface are always positioned after their parent in the LCompositor::surfaces() list.
    ESurface *next { surface };

    if (subsurfaceRoleOnly)
    {
        while ((next = (ESurface*)next->nextSurface()))
            if (next->isSubchildOf(surface) && next->subsurface())
                next->view.setParent(newParent);
    }
    else
        while ((next = (ESurface*)next->nextSurface()))
            if (next->isSubchildOf(surface))
                next->view.setParent(newParent);
}
```
> Note that if the `subsurfaceRoleOnly` parameter is set to `true`, only child surfaces with the Louvre::LSubsurfaceRole are moved into the `newParent`. This is because surfaces with the Louvre::LSubsurfaceRole are typically thought to be composited with their parent surface as if they were one. On the other hand, child popups and toplevels typically represent context menus and dialog windows respectively, so we want to exclude those when generating our thumbnails.

Now, let's proceed to create an Louvre::LTextureView subclass to serve as our thumbnail views, which we'll name `EThumbnail`.

#### src/EThumbnail.h

```cpp
#ifndef ETHUMBNAIL_H
#define ETHUMBNAIL_H

#include <LTextureView.h>

using namespace Louvre;

class ETopbar;
class ESurface;

class EThumbnail : public LTextureView
{
public:
    EThumbnail(ETopbar *topbar, ESurface *surface);
    ETopbar  *topbar;
    ESurface *surface;
};

#endif // ETHUMBNAIL_H
```

It accepts two arguments: the topbar in which it will be embedded and the minimized surface it represents.

#### src/EThumbnail.cpp

```cpp
#include "EThumbnail.h"
#include "ESurface.h"
#include "ETopbar.h"

EThumbnail::EThumbnail(ETopbar *topbar, ESurface *surface) :
    LTextureView(surface->thumbnailTexture, &topbar->view),
    topbar(topbar),
    surface(surface)
{
    surface->thumbnailViews.push_back(this);
    enableParentOpacity(false);
    setBufferScale(2);
    topbar->updateGUI();
}

EThumbnail::~EThumbnail()
{
    /* This is a handy function used for efficiently removing an element from a
     * vector where order isn't essential, employing the back-swap approach */
    LVectorRemoveOneUnordered(surface->thumbnailViews, this);

    /* Before calling updateGUI(), it's essential to remove it from the children
     * list of the topbar view to ensure proper ordering of the remaining thumbnails */
    setParent(nullptr);

    topbar->updateGUI();
}
```

In the constructor, we set the surface thumbnail as the view texture (we'll cover how to create it shortly) and add it to the topbar view. Additionally, we add the item to the surface thumbails vector (we'll create it shortly too), set its buffer scale to 2 because the thumbnail textures will be high DPI, and we also disable the Louvre::LView::parentOpacityEnabled() property, which prevents the views opacity to be affected by the topbar's opacity.

> By default, all views have their Louvre::LView::opacity() set to 1.0 and the Louvre::LView::parentOpacityEnabled() property enabled. This means that their opacities are multiplied by the opacity of their parent views. To prevent thumbnails from appearing translucent due to the semi-translucent top bar, we have disabled this feature.

When the thumbnail is destroyed, we remove the item from the surface vector and the topbar view. This way, when `updateGUI()` is called, the remaining thumbnails will be properly re-ordered. Further clarification on these details will become apparent as we implement the remaining components.

> A view is automatically removed from its parent's children list when destroyed. Conversely, its `Louvre::LView::parent()` is automatically set to `nullptr` when the parent is destroyed.

In the `updateGUI()` method of `ETopbar`, we now need to arrange the thumbnails from left to right, spacing them apart by a distance of `THUMBNAIL_MARGIN`.

#### src/ETopbar.cpp

```cpp
#include "ETopbar.h"
#include "EOutput.h"
#include "EThumbnail.h"
#include "Global.h"

// ...

void ETopbar::updateGUI()
{
    view.setSize(output->size().w(), TOPBAR_HEIGHT);
    view.setPos(output->pos());

    Int32 x { TOPBAR_PADDING };
    for (EThumbnail *thumbnail : (std::list<EThumbnail*>&)view.children())
    {
        thumbnail->setPos(x, TOPBAR_PADDING);
        x += thumbnail->size().w() + THUMBNAIL_MARGIN;
    }

    output->repaint();
}
```

It's important to note that all views have the Louvre::LView::parentOffsetEnabled() property enabled by default (as we already discussed), which means we're positioning the thumbnail views relative to the topbar view position.\n 
At the moment, we haven't encountered any issues with positioning views because they have all been children of the layers defined in `ECompositor`, which, by default, have a position of (0, 0).

Now, considering that the user of our compositor may have more than one screen, and we aim to display thumbnails in the top bar of each screen, we need to address the scenario where they are hotplugged on the fly. The straightforward approach is to simply copy the thumbnails from an already existing top bar, so lets implement that in `copyThumbnailsFromAnotherTopbar()`:

#### src/ETopbar.cpp

```cpp
// ...

void ETopbar::copyThumbnailsFromAnotherTopbar()
{
    for (EOutput *o : G::outputs())
    {
        if (o == output)
            continue;

        for (EThumbnail *item : (std::list<EThumbnail *>&)o->topbar.view.children())
            new EThumbnail(this, item->surface);

        break;
    }

    updateGUI();
}

// ...
```

Now, let's see how to create the thumbnails:

#### src/ESurface.h

```cpp
#ifndef ESURFACE_H
#define ESURFACE_H

#include <LSurface.h>
#include <LSurfaceView.h>
#include "ECompositor.h"
#include "Global.h"

class EThumbnail;
class EOutput;

using namespace Louvre;

class ESurface : public LSurface
{
public:
    ESurface(const void *params);
    ~ESurface();

    void roleChanged() override;
    void orderChanged() override;
    void minimizedChanged() override;

    // Output where the surface is currently most visible
    EOutput *primaryOutput();

    // Take a snapshot of the surface
    LTexture *capture(const LSize &dstSizeB);

    LSurfaceView view { (LSurface*) this, &G::compositor()->surfacesLayer };

    // Single texture shared by all thumbnail views
    LTexture *thumbnailTexture { nullptr };

    // Thumbnail views, one for each output's topbar
    std::vector<EThumbnail*> thumbnailViews;

    // The output where the surface was most visible before being minimized
    EOutput *outputBeforeMinimize { nullptr };

    // The relative % surface position to outputBeforeMinimize
    LPointF relativePosBeforeMinimize;
};

#endif // ESURFACE_H
```

In the `ESurface` class, several additions have been made:

Firstly, we've overridden the Louvre::LSurface::minimizedChanged() virtual method to listen for changes in the minimized state of a surface.

We've also introduced a new method called `primaryOutput()`, which will be used to determine the output in which the surface is most visible before it's minimized. Additionally, we'll store the position of the surface relative to that output as a percentage. This will allow us to restore the surface's position later even if the arrangement of outputs changes during the time it's minimized.

The `capture()` method will be used to render both the surface and its subsurfaces (Louvre::LSubsurfaceRole) into a texture, which will serve as our thumbnail.

Lastly, the `thumbnailViews` vector will store all the minimized views for this surface, as we intend to create one for each initialized output (topbar).

#### Destructor

In the destructor, we simply destroy the thumbnail texture and views associated with this surface. Remember that in the `EThumbnail` class destructor, we remove the item from the `thumbnailViews` vector.

#### src/ESurface.cpp

```cpp
#include <LCursor.h>
#include "ESurface.h"
#include "EThumbnail.h"


// ...

ESurface::~ESurface()
{
    while (!thumbnailViews.empty())
        delete thumbnailViews.back();

    if (thumbnailTexture)
        delete thumbnailTexture;
}
 
// ...
```

#### primaryOutput()

In this method, we calculate the area generated by intersecting the surface rect with each output rect and return the one where the area is the largest. 
If there is no intersection with any of them, we return `nullptr`.

```cpp
// ..

#include "EOutput.h"

// ...

EOutput *ESurface::primaryOutput()
{
    EOutput    *bestOutput   { nullptr };
    Int32       bestArea     { 0 };
    const LRect surfaceRect  { rolePos(), size() };

    /* Calculate the area of the surface intersected
     * with each output and return the one with the largest area */
    for (EOutput *output : G::outputs())
    {
        // We use LRegion to intersect both rects
        LRegion tmpRegion;
        tmpRegion.addRect(surfaceRect);
        tmpRegion.clip(output->rect());

        const LBox &extents { tmpRegion.extents() };
        const Int32 area { (extents.x2 - extents.x1) * (extents.y2 - extents.y1) };

        if (area > bestArea)
        {
            bestArea = area;
            bestOutput = output;
        }
    }

    return bestOutput;
}
 
// ...
```

#### capture()

In the `capture()` method, we create a temporary Louvre::LSceneView with a framebuffer size twice that of the surface and a scale factor of 2. This ensures the entire surface is captured and appears sharp on both low and high DPI displays. We deliberately refrain from assigning a parent to the scene view because its purpose is solely for offscreen rendering and does not need to be displayed in our main scene.

Instead of relocating the surface view and any subsurface views to the scene view's position, we position the scene view at the location of the surface view. This is feasible because, as a reminder, we have disabled the parent offset property for all surface views.

Subsequently, we designate the surface and subsurface views as children of the scene view using the auxiliary function defined in `Global.h`.

To render the views into the scene view framebuffer, we simply call Louvre::LSceneView::render().

Finally, we return the views to the surfaces layer and create a copy of the rendered scene view texture scaled to `dstSizeB`. For this, we use Louvre::LTexture::copyB(), allowing us to achieve high-quality downsampling.

#### src/ESurface.cpp

```cpp
// ...

LTexture *ESurface::capture(const LSize &dstSizeB)
{
    LSceneView tmpSceneView {
        size() * 2, // Framebuffer size : x2 the surface size to make it look good on HiDPI displays.
        2,          // Scaling factor   : Affects how the children views are rendered
        nullptr     // Parent view      : Not required because we are using it for offscreen rendering
    };

    /* Instead of moving each view to the scene view, we move the scene to the views position.
     * This is why disabling parent offset is required. */
    tmpSceneView.setPos(rolePos());

    G::reparentSurfaceAndChildrenViews(this, &tmpSceneView, true);
    tmpSceneView.render();
    G::reparentSurfaceAndChildrenViews(this, &G::compositor()->surfacesLayer, true);

    return tmpSceneView.texture()->copyB(
        dstSizeB, // Destination size in buffer coordiantes
        LRect(),  // Passing an empty src rect means copying the entire texture
        true      // Enable high quality downscaling
    );
}

// ...
```

#### minimizedChanged()

In Louvre::LSurface::minimizedChanged(), we take the following actions:

```cpp
// ...

#include <LSeat.h>
#include <LKeyboard.h>
#include <LPointer.h>

// ...

void ESurface::minimizedChanged()
{
    /* When a surface is minimized, all its children are also minimized
     * we only want to display toplevels in the topbar so we ignore the rest */
    if (!toplevel())
        goto toggleVisibility;

    if (minimized())
    {
        // We don't want the surface to continue receiving input events while minimized
        if (hasPointerFocus())
            seat()->pointer()->setFocus(nullptr);

        if (hasKeyboardFocus())
            seat()->keyboard()->setFocus(nullptr);

        /* Save the current surface position relative to the output position as a percentage so we can restore
         * it later even if the outputs arrangement changes or the given output is no longer available */
        if ( (outputBeforeMinimize = primaryOutput()) )
        {
            const LPointF localPos { pos() - outputBeforeMinimize->pos() };
            relativePosBeforeMinimize = localPos / LSizeF(outputBeforeMinimize->size());
        }

        /* In case the surface is not visible on any output, we select the first available output and position
         * the surface at 1/4 of the output as a fallback */
        else if (!G::outputs().empty())
        {
            outputBeforeMinimize = G::outputs().front();
            relativePosBeforeMinimize = LPointF(0.25f, 0.25f);
        }
        else
        {
            setMinimized(false);
            return;
        }

        const LSize thumbnailSizeB {
            2 * (THUMBNAIL_HEIGHT * size().w()) / size().h(),
            2 * THUMBNAIL_HEIGHT
        };

        thumbnailTexture = capture(thumbnailSizeB);

        for (EOutput *output : G::outputs())
            new EThumbnail(&output->topbar, this);
    }
    else
    {
        // If nullptr, then it was uninitialized while the surface was minimized
        if (!outputBeforeMinimize)
        {
            if (!G::outputs().empty())
            {
                outputBeforeMinimize = G::outputs().front();
                relativePosBeforeMinimize = LPointF(0.25f, 0.25f);
            }
        }

        // Bye bye thumbnails
        while (!thumbnailViews.empty())
            delete thumbnailViews.front();

        delete thumbnailTexture;
        thumbnailTexture = nullptr;

        // Restore back the previous unminimized position
        setPos(outputBeforeMinimize != nullptr ? outputBeforeMinimize->pos() + (relativePosBeforeMinimize * outputBeforeMinimize->size()) : LPoint(0, TOPBAR_HEIGHT));
        outputBeforeMinimize = nullptr;
        raise();
    }

toggleVisibility:
    view.setVisible(!minimized());
}

// ...
```

1. First, we check whether the surface is a toplevel. If it isn't, we skip it to ensure that only toplevel surfaces appear minimized in our topbar.

2. When Louvre::LSurface::minimized() returns `true`, indicating that the surface is minimized we:

   a. Determine the output where the surface is currently most visible using `primaryOutput()`. This could yield `nullptr` if the surface is not visible in any output, although it's an unusual scenario. In such cases we fallback to the first available output.

   b. Calculate the current surface position relative to the output and as a percentage of its size. This enables us to restore the surface's position when unminimizing it. If the surface wasn't visible in any output, we opt for the first initialized one and set the position to 1/4 of its size.

   c. Use `capture()` to create the thumbnail texture.

   d. Create a thumbnail view for each topbar (all sharing the same texture).

3. In the alternative case, when the surface is unminimized we:

   a. Destroy the thumbnail views and texture.

   b. Restore the view's prev position.

   c. Raise the surface.

The final step we need to take is to set the `outputBeforeMinimize` variable to another output if it is uninitialized while the surface is minimized. This ensures that the surface correctly handles unminimization in Louvre::LSurface::minimizedChanged() and prevents our compositor from crashing.

So let's create a new method in `EOutput` named `rescueViewsFromVoid()`:

#### src/EOutput.h

```cpp
// ...

class EOutput : public LOutput
{
public:

    // ...

    void rescueViewsFromVoid();

    // ...
};

// ...
```

#### src/EOutput.cpp

```cpp
// ...

#include "ESurface.h"

// ...

void EOutput::rescueViewsFromVoid()
{
    // Search another initialized output
    const auto isDiff = [this](const EOutput *output) { return this != output; };
    const auto it { std::find_if(G::outputs().begin(), G::outputs().end(), isDiff) };
    EOutput *fallbackOutput { it != G::outputs().end() ? *it : nullptr };

    for (ESurface *surface : G::surfaces())
        if (surface->outputBeforeMinimize == this)
            surface->outputBeforeMinimize = fallbackOutput;
}

// ...

void EOutput::uninitializeGL()
{
    rescueViewsFromVoid();

    // ...
}

// ...
```

> Note that if there were zero initialized outputs, in Louvre::LSurface::minimizedChanged(), we have already addressed the situation where a surface is unminimized, and `outputBeforeMinimize` is `nullptr`.

After recompiling and running the compositor, you should now be able to minimize windows, and their thumbnails will appear on the top bar. Additionally, if you hotplug an output on the fly, you will observe the previously minimized surfaces automatically appearing on its respective top bar.

<img src="https://lh3.googleusercontent.com/pw/ADCreHdClzn_FgEv67oYw4Hxmln3JkiUk9YRwITs1xIswKqiKOnMUvKNll6hTnIQJHw8W8Cd9YvJheZMqgYWXSBRzAo_YI_TP-KEp7zi8TdUp8rebLAaM3A=w2400"/>

Also, take the time to appreciate our dear biologist Humberto Maturana, may he rest in peace ❤️.

One more thing you may have noticed is that we can only minimize windows but not unminimize them yet 😄. In the next chapter, we'll delve into handling pointer events, which encompass input from devices like a mouse, touchpad, and more. We'll also explore how to unminimize the windows by clicking on their thumbnails.

Until next time!

<a href="06.md">◀ Chapter 6: Rendering with LScene and LViews</a> || <a href="08.md"> Chapter 8: Pointer Events ▶</a>