# Chapter 8: Pointer Events {#tutorial-events}

In this chapter, we will delve into the intricacies of managing pointer events, which encompass actions from a mouse or touchpad, including button clicks, movement, and scrolling. All pointer events can be effectively managed through the Louvre::LPointer class. This class serves as an interface utilized by the input backend, which, in turn, triggers numerous of its virtual methods.

Furthermore, this class allows you to reroute these events to client surfaces, offering a flexible and controlled event-handling mechanism.

### Security

As you may already be aware, Wayland presents a significant advancement in security compared to X11, particularly concerning input events. This is due to the fact that in Wayland, the compositor is the exclusive process with direct access to system input events (unless your system is configured in an unsafe way). Consequently, it can selectively and securely transmit these events to clients, thus thwarting potential security vulnerabilities. This approach effectively safeguards against threats like information theft, which could be exploited by malware keyloggers, providing a more robust security framework.

### Sending Events to Clients

In Louvre, pointer events can be directed to a single client surface at a time. To achieve this, you must initially set the pointer's focus to a specific surface using one of the variants of Louvre::LPointer::setFocus(). Once the focus is established, you can then use any of the other event-sending methods, such as Louvre::LPointer::sendMoveEvent(), Louvre::LPointer::sendButtonEvent(), or Louvre::LPointer::sendAxisEvent(). These methods will automatically redirect the events to the currently focused surface.

If you subsequently call Louvre::LPointer::setFocus() again, pointing to a different surface, the previous surface will automatically lose focus. To remove focus from all surfaces, you can pass `nullptr`. Additionally, when a client destroys a surface that currently holds the pointer's focus, the focus is automatically unset as well. You can ascertain which surface currently has pointer focus with Louvre::LPointer::focus(), which may return `nullptr` if no surface currently holds it.

Without further delay, let's proceed to override the default way that Louvre handles pointer events. To achieve this, we'll create a subclass of Louvre::LPointer named `EPointer`:

#### src/EPointer.h

```cpp
#ifndef EPOINTER_H
#define EPOINTER_H

#include <LPointer.h>

using namespace Louvre;

class EPointer : public LPointer
{
public:
    EPointer(const void *params);

    void pointerMoveEvent(Float32 x, Float32 y, bool isAbsolute) override;
    void pointerButtonEvent(Button button, ButtonState state) override;
    void pointerAxisEvent(Float64 axisX, Float64 axisY, Int32 discreteX, Int32 discreteY, AxisSource source) override;
};

#endif // EPOINTER_H
```

Here, we're overriding all input event methods of Louvre::LPointer, each serving a distinct purpose:

* Louvre::LPointer::pointerMoveEvent: Notifies us of absolute or relative pointer movement.

* Louvre::LPointer::pointerButtonEvent: Informs us when a mouse or touchpad button is pressed or released. The Louvre::LPointer::Button argument specifies which button, and Louvre::LPointer::ButtonState indicates the state (pressed or released).

* Louvre::LPointer::pointerAxisEvent: Reports scrolling events, which can involve continuous or discrete axis components depending on the source. For example, two-finger scrolling employs continuous axis components, while a mouse scroll wheel uses discrete ones.

#### src/EPointer.cpp

```cpp
#include <LCursor.h>
#include <LDNDManager.h>
#include <LSeat.h>
#include <LOutput.h>
#include <LSurface.h>
#include <LDNDIconRole.h>
#include <LToplevelRole.h>
#include <LKeyboard.h>
#include <LCompositor.h>
#include "EPointer.h"

EPointer::EPointer(const void *params) : LPointer(params) {}

void EPointer::pointerMoveEvent(Float32 x, Float32 y, bool isAbsolute)
{
    if (isAbsolute)
        cursor()->setPos(x, y);
    else
        cursor()->move(x, y);

    // Repaint outputs that intersect with the cursor if hardware composition is not supported.
    cursor()->repaintOutputs(true);

    // Update the drag & drop icon position
    if (seat()->dndManager()->icon())
    {
        seat()->dndManager()->icon()->surface()->setPos(cursor()->pos());
        seat()->dndManager()->icon()->surface()->repaintOutputs();
    }

    if (resizingToplevel())
    {
        updateResizingToplevelSize(cursor()->pos());
        return;
    }

    if (movingToplevel())
    {
        updateMovingToplevelPos(cursor()->pos());

        movingToplevel()->surface()->repaintOutputs();

        if (movingToplevel()->maximized())
            movingToplevel()->configure(movingToplevel()->pendingStates() &~ LToplevelRole::Maximized);

        return;
    }

    // If we are in a drag & drop session, we call setDraggingSurface(nullptr)
    // to prevent the current surface from retaining focus.
    if (seat()->dndManager()->dragging())
        setDraggingSurface(nullptr);

    // If a surface had the left pointer button held down
    if (draggingSurface())
    {
        sendMoveEvent();
        return;
    }

    // Find the first surface under the cursor
    LSurface *surface { surfaceAt(cursor()->pos()) };

    if (surface)
    {
        if (focus() == surface)
            sendMoveEvent();
        else
            setFocus(surface);
    }
    else
    {
        setFocus(nullptr);
        cursor()->useDefault();
        cursor()->setVisible(true);
    }
}

void EPointer::pointerButtonEvent(Button button, ButtonState state)
{
    if (state == Released && button == Left)
        seat()->dndManager()->drop();

    if (!focus())
    {
        LSurface *surface { surfaceAt(cursor()->pos()) };

        if (surface)
        {
            seat()->keyboard()->setFocus(surface);
            setFocus(surface);
            sendButtonEvent(button, state);

            if (!surface->popup())
                dismissPopups();
        }
        else
        {
            seat()->keyboard()->setFocus(nullptr);
            dismissPopups();
        }

        return;
    }

    sendButtonEvent(button, state);

    if (button != Left)
        return;

    // Left button pressed
    if (state == Pressed)
    {
        // We save the pointer focus surface to continue sending events to it even when the cursor
        // is outside of it (while the left button is being held down)
        setDraggingSurface(focus());

        if (!seat()->keyboard()->focus() || !focus()->isSubchildOf(seat()->keyboard()->focus()))
            seat()->keyboard()->setFocus(focus());

        if (focus()->toplevel() && !focus()->toplevel()->activated())
            focus()->toplevel()->configure(focus()->toplevel()->pendingStates() | LToplevelRole::Activated);

        if (!focus()->popup())
            dismissPopups();

        if (focus() == compositor()->surfaces().back())
            return;

        if (focus()->parent())
            focus()->topmostParent()->raise();
        else
            focus()->raise();
    }
    // Left button released
    else
    {
        stopResizingToplevel();
        stopMovingToplevel();

        // We stop sending events to the surface on which the left button was being held down
        setDraggingSurface(nullptr);

        if (!focus()->inputRegion().containsPoint(cursor()->pos() - focus()->rolePos()))
        {
            setFocus(nullptr);
            cursor()->useDefault();
            cursor()->setVisible(true);
        }
    }
}

void EPointer::pointerAxisEvent(Float64 axisX, Float64 axisY, Int32 discreteX, Int32 discreteY, AxisSource source)
{
    // Invert the scroll axis for natural scrolling
    sendAxisEvent(-axisX, -axisY, -discreteX, -discreteY, source);
}
```

Stay calm! Here, we are currently using the default implementation provided by Louvre. I'll explain what is happening shortly. But first, let's proceed to override the Louvre::LPointer virtual constructor (Louvre::LCompositor::createPointerRequest()).

#### src/ECompositor.h

```cpp
    // ...

    // Virtual constructors
    LOutput *createOutputRequest(const void *params) override;
    LSurface *createSurfaceRequest(const void *params) override;
    LPointer *createPointerRequest(const void *params) override;

    // ...
```

#### src/ECompositor.cpp

```cpp

// ...

#include "EPointer.h"

// ...

LPointer *ECompositor::createPointerRequest(const void *params)
{
    return new EPointer(params);
}
```

If you recompile and run the compositor, you won't observe any changes because we've implemented the same logic that Louvre uses by default to handle each Louvre::LPointer event. 

Now, let's delve deeper into what each method is accomplishing:

#### src/EPointer.cpp

```cpp
void EPointer::pointerMoveEvent(Float32 x, Float32 y, bool isAbsolute)
{
    if (isAbsolute)
        cursor()->setPos(x, y);
    else
        cursor()->move(x, y);

    // Repaint outputs that intersect with the cursor if hardware composition is not supported.
    cursor()->repaintOutputs(true);

    // Update the drag & drop icon position
    if (seat()->dndManager()->icon())
    {
        seat()->dndManager()->icon()->surface()->setPos(cursor()->pos());
        seat()->dndManager()->icon()->surface()->repaintOutputs();
    }

    // ...
}
```

Here, we first update the cursor position. Following this, we invoke repaint on all outputs where the cursor is visible if hardware cursor composition is not available (check Louvre::LCursor::repaintOutputs()). This ensures that the cursor is appropriately repainted using OpenGL.

Additionally, we conduct a check to determine if a drag & drop session is currently in progress. If so, we match the position of the dragged icon with the cursor position, ensuring that it follows the cursor.

#### src/EPointer.cpp

```cpp
void EPointer::pointerMoveEvent(Float32 x, Float32 y, bool isAbsolute)
{
    // ...

    if (resizingToplevel())
    {
        updateResizingToplevelSize(cursor()->pos());
        return;
    }

    if (movingToplevel())
    {
        updateMovingToplevelPos(cursor()->pos());

        movingToplevel()->surface()->repaintOutputs();

        if (movingToplevel()->maximized())
            movingToplevel()->configure(movingToplevel()->pendingStates() &~ LToplevelRole::Maximized);

        return;
    }

    // If we are in a drag & drop session, we call setDraggingSurface(nullptr)
    // to prevent the current surface from retaining focus.
    if (seat()->dndManager()->dragging())
        setDraggingSurface(nullptr);

    // If a surface had the left pointer button held down
    if (draggingSurface())
    {
        sendMoveEvent();
        return;
    }

    // ...
}
```

In this section, we handle interactive resizing or movement of toplevel windows, elaborated further in [Chapter 10: Toplevels](10.md). 

We also employ Louvre::LPointer::setDraggingSurface() and Louvre::LPointer::draggingSurface() to maintain the focus when the left pointer button is initially pressed within the focused surface and then moved outside. This is done for preserving the surface focus during actions like text selection, preventing annoying focus loss when the cursor moves outside the surface. It's important to note that this is unrelated to drag & drop sessions. If a drag and drop session is active, we unset Louvre::LPointer::draggingSurface() to ensure that each surface beneath the cursor receives pointer focus. 

Furthermore, we use Louvre::LPointer::sendMoveEvent() to transmit the event to the focused surface. The client anticipates the position to be in local surface coordinates. In this scenario, we opt for the variant of Louvre::LPointer::sendMoveEvent() that requires no arguments, as it automatically computes the local surface position using Louvre::LSurface::rolePos() and the cursor's position. Alternatively, you have the option to use other variats to manually specify the local surface position.

#### src/EPointer.cpp

```cpp
void EPointer::pointerMoveEvent(Float32 x, Float32 y, bool isAbsolute)
{
    // ...

    // Find the first surface under the cursor
    LSurface *surface { surfaceAt(cursor()->pos()) };

    if (surface)
    {
        if (focus() == surface)
            sendMoveEvent();
        else
            setFocus(surface);
    }
    else
    {
        setFocus(nullptr);
        cursor()->useDefault();
        cursor()->setVisible(true);
    }
}
```

Finally, if this method was not previously returned, we perform the following actions:

1. Recheck which surface is currently under the cursor using Louvre::LPointer::surfaceAt(). This method returns the first mapped surface beneath the cursor based on the Louvre::LSurface::inputRegion(). It's important to note that certain surfaces, such as Louvre::LCursorRole or Louvre::LDNDIconRole, lack an input region and are therefore never returned by this method. Additionally, it returns `nullptr` if no surface is found.

2. If we discover a surface under the cursor, we assign it focus. If the surface is the same as the currently focused one, we send a move event.

3. If no surface is found under the cursor, we unset pointer focus from all surfaces and restore the default cursor texture. Details on how clients can request changes to cursor texture and visibility will be covered later in [Chapter 13: Cursor](13.md).

Now lets move to Louvre::LPointer::pointerButtonEvent():

#### src/EPointer.cpp

```cpp
void EPointer::pointerButtonEvent(Button button, ButtonState state)
{
    if (state == Released && button == Left)
        seat()->dndManager()->drop();

    // ...
}
```

First, if the left pointer button is released, we invoke Louvre::LDNDManager::drop(). This action initiates a drop operation if a drag & drop session is in progress. Importantly, calling it when no session is active is a harmless no-op.

#### src/EPointer.cpp

```cpp
void EPointer::pointerButtonEvent(Button button, ButtonState state)
{
    // ...

    if (!focus())
    {
        LSurface *surface { surfaceAt(cursor()->pos()) };

        if (surface)
        {
            seat()->keyboard()->setFocus(surface);
            setFocus(surface);
            sendButtonEvent(button, state);

            if (!surface->popup())
                dismissPopups();
        }
        else
        {
            seat()->keyboard()->setFocus(nullptr);
            dismissPopups();
        }

        return;
    }

    // ...
}
```

Then, in the absence of a currently focused surface, we search for one using Louvre::LPointer::surfaceAt() once again.

If a surface is discovered, we assign it both pointer and keyboard focus and transmit the button event. Additionally, if the surface is not a popup, we dismiss them. This is employed to address the issue of certain buggy clients failing to close popups, an undesired behavior.

If no surface is found under the cursor, we remove keyboard focus and once again we dismiss any popups.

#### src/EPointer.cpp

```cpp
void EPointer::pointerButtonEvent(Button button, ButtonState state)
{
    // ...

    sendButtonEvent(button, state);

    if (button != Left)
        return;

    // Left button pressed
    if (state == Pressed)
    {
        // We save the pointer focus surface to continue sending events to it even when the cursor
        // is outside of it (while the left button is being held down)
        setDraggingSurface(focus());

        if (!seat()->keyboard()->focus() || !focus()->isSubchildOf(seat()->keyboard()->focus()))
            seat()->keyboard()->setFocus(focus());

        if (focus()->toplevel() && !focus()->toplevel()->activated())
            focus()->toplevel()->configure(focus()->toplevel()->pendingStates() | LToplevelRole::Activated);

        if (!focus()->popup())
            dismissPopups();

        if (focus() == compositor()->surfaces().back())
            return;

        if (focus()->parent())
            focus()->topmostParent()->raise();
        else
            focus()->raise();
    }
    // Left button released
    else
    {
        stopResizingToplevel();
        stopMovingToplevel();

        // We stop sending events to the surface on which the left button was being held down
        setDraggingSurface(nullptr);

        if (!focus()->inputRegion().containsPoint(cursor()->pos() - focus()->rolePos()))
        {
            setFocus(nullptr);
            cursor()->useDefault();
            cursor()->setVisible(true);
        }
    }
}
```

If a surface has pointer focus, we send the button event. If the button isn't the left one, we return since we mainly use the left button for actions.

When the left button is pressed over a surface:
- We set it as the dragging surface, maintaining focus.
- Assign keyboard focus.
- If it's a toplevel, we make it active, highlighting its decorations.
- If not a popup, we dismiss popups.
- Finally, we bring it to the front, along with its parent surface, if applicable, stacking it above others.

When the left button is released:
- We halt interactive toplevel resizing or movement.
- Unset the dragging surface.
- Check if the cursor is over the surface's input region. If not, we clear pointer focus and restore the cursor to its default state.

Well, let's all take a deep breath and shift our focus to the seemingly simpler Louvre::LPointer::pointerAxisEvent().

#### src/EPointer.cpp

```cpp
void EPointer::pointerAxisEvent(Float64 axisX, Float64 axisY, Int32 discreteX, Int32 discreteY, AxisSource source)
{
    // Invert the scroll axis for natural scrolling
    sendAxisEvent(-axisX, -axisY, -discreteX, -discreteY, source);
}
```

In this section, we're simply routing the scroll event to the surface that currently holds focus.

Additionally, it's worth noting that we've inverted the axis values. This adjustment is made to accommodate natural scrolling. However, feel free to customize it to your preference.

## Managing Pointer Events with LScene

The detailed breakdown of each event handling was provided solely for educational purposes, enabling you to grasp event management concepts. However, similar to the approach we explored with Louvre::LOutput, the Louvre::LScene class also offers methods to handle pointer events in a similar manner.

Using Louvre::LScene presents an additional advantage: it allows you to listent to per-view events. To integrate Louvre::LScene with Louvre::LPointer, you simply need to "plug it" like this:

#### src/EPointer.cpp

```cpp
#include <LScene.h>
#include "EPointer.h"
#include "Global.h"

EPointer::EPointer(const void *params) : LPointer(params) {}

void EPointer::pointerMoveEvent(Float32 x, Float32 y, bool isAbsolute)
{
    G::scene()->handlePointerMoveEvent(x, y, isAbsolute);
}

void EPointer::pointerButtonEvent(Button button, ButtonState state)
{
    G::scene()->handlePointerButtonEvent(button, state);
}

void EPointer::pointerAxisEvent(Float64 axisX, Float64 axisY, Int32 discreteX, Int32 discreteY, AxisSource source)
{
    G::scene()->handlePointerAxisEvent(-axisX, -axisY, -discreteX, -discreteY, source);
}
```

After recompiling and running the compositor, everything should continue to function as expected, with one minor change. If you happen to move the cursor outside of a surface, you'll notice that the cursor does not revert to its default state.\n
To address this issue, we can implement the following solution:

#### src/EPointer.cpp

```cpp
#include <LScene.h>
#include <LCursor.h>
#include <LView.h>
#include "EPointer.h"
#include "Global.h"

// ...

void EPointer::pointerMoveEvent(Float32 x, Float32 y, bool isAbsolute)
{
    LView *view { G::scene()->handlePointerMoveEvent(x, y, isAbsolute) };

    if ((!view || view->type() != LView::Type::Surface) && !resizingToplevel() && !movingToplevel())
    {
        cursor()->useDefault();
        cursor()->setVisible(true);
    }
}

// ...
```

The Louvre::LScene::handlePointerMoveEvent() method returns the first view located under the cursor or `nullptr` if no view is found.\n
In cases where either no view is found or the identified view is not of type Louvre::LSurfaceView, we take action to restore the cursor to its default state.
Additionally, we avoid restoring the cursor when a toplevel window is currently being resized or moved. This precaution is taken because the client may intend to display a specific cursor, such as a resize edge cursor or a hand-dragging cursor, during these operations.

### Restoring Minimized Surfaces

By using Louvre::LScene for handling pointer events, we can now detect when the user clicks on the topbar thumbnails. To achieve this, we need to enable input events for each thumbnail view using Louvre::LView::enableInput().

> By default, all view types, except for Louvre::LSurfaceView, have input disabled. This means they are ignored during pointer events handling, enabling views behind them to receive those events instead.

Additionally, we must override their Louvre::LView::pointerButtonEvent() virtual method:

#### src/EThumbnail.h

```cpp
// ...

class EThumbnail : public LTextureView
{
public:

    // ...

    void pointerButtonEvent(LPointer::Button button, LPointer::ButtonState state) override;

    // ...
};

// ...
```

#### src/EThumbnail.cpp

```cpp
// ...

EThumbnail::EThumbnail(ETopbar *topbar, ESurface *surface) :
    LTextureView(surface->thumbnailTexture, &topbar->view),
    topbar(topbar),
    surface(surface)
{
    // ...

    enableInput(true);    
}

// ...

void EThumbnail::pointerButtonEvent(LPointer::Button button, LPointer::ButtonState state)
{
    if (button == LPointer::Button::Left && state == LPointer::ButtonState::Released)
        surface->setMinimized(false);
}
```

Now, clicking on a thumbnail view using the left pointer button results in the corresponding surface being unminimized—simple and efficient.

You can also explore other per-view pointer events, such as when the cursor enters or leaves the view, scrolling events, and more. For detailed information, please refer to the documentation of Louvre::LView.

### Disabling Wayland Event Handling

If you need to handle pointer events differently from how Louvre::LScene does, while still receiving per-view events, you can disable the Wayland event handling of Louvre::LScene using Louvre::LScene::enableHandleWaylandPointerEvents(). This allows you to continue receiving per-view pointer events while giving you the flexibility to manually send these events to clients, as we demonstrated earlier in this chapter.

In the upcoming chapter, we'll explore the realm of handling keyboard events. Until then, see you there!

<a href="07.md">◀ Chapter 7: Surfaces</a> || <a href="09.md"> Chapter 9: Keyboard Events ▶</a>