# Chapter 9: Keyboard Events {#tutorial-keyboard}

Handling keyboard events closely resembles the process for managing pointer events. You begin by assigning keyboard focus to a surface, and from that point on, all keyboard events are directed to that surface. Keyboard event management is facilitated through the Louvre::LKeyboard class and is quite simpler compared to handling pointer events.

There are two key virtual methods you can override:

* Louvre::LKeyboard::keyEvent: This event is generated each time a key is pressed or released.

* Louvre::LKeyboard::keyModifiersEvent: This event is generated whenever the keyboard modifiers change.

## Keyboard Mapping

As discussed in [Chapter 4: Compositor Initialization](04.md), you can set the XKB keymap using the Louvre::LKeyboard::setKeymap() method. The key codes generated by the input backend correspond to the raw codes defined in the [`<linux/input-event-codes.h>`](https://github.com/torvalds/linux/blob/master/include/uapi/linux/input-event-codes.h) header. These codes may represent different symbols across various keyboards and can change based on the current modifier state. This is where keymaps come into play, allowing for the correct interpretation of these raw key codes.

You can convert a raw key code into an XKB symbol using the Louvre::LKeyboard::keySymbol() method and ascertain the state of a modifier with Louvre::LKeyboard::isModActive().
The XKB key symbols are defined in the [`<xkbcommon/xkbcommon-keysyms.h>`](https://github.com/xkbcommon/libxkbcommon/blob/master/include/xkbcommon/xkbcommon-keysyms.h) header.

## Keyboard Repeat Rate

In scenarios where you hold down a key, such as when typing in a text editor, clients typically initiate key repeats after a specific delay. You have the ability to control both the delay and the rate of these repeats using the Louvre::LKeyboard::setRepeatInfo() method.

Let's create our own Louvre::LKeyboard subclass named `EKeyboard`:

#### src/EKeyboard.h

```cpp
#ifndef EKEYBOARD_H
#define EKEYBOARD_H

#include <LKeyboard.h>

using namespace Louvre;

class EKeyboard : public LKeyboard
{
public:
    EKeyboard(const void *params);

    void keyEvent(UInt32 keyCode, KeyState keyState) override;
    void keyModifiersEvent(UInt32 depressed, UInt32 latched, UInt32 locked, UInt32 group) override;
};

#endif // EKEYBOARD_H
```

#### src/EKeyboard.cpp

```cpp
#include <linux/input-event-codes.h>
#include <LCompositor.h>
#include <LSurface.h>
#include <LClient.h>
#include <LCursor.h>
#include <LOutput.h>
#include <LLauncher.h>
#include <LSeat.h>
#include <LDNDManager.h>
#include <unistd.h>
#include "EKeyboard.h"

EKeyboard::EKeyboard(const void *params) : LKeyboard(params) {}

void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    sendKeyEvent(keyCode, keyState);

    const bool L_CTRL      { isKeyCodePressed(KEY_LEFTCTRL) };
    const bool L_SHIFT     { isKeyCodePressed(KEY_LEFTSHIFT) };
    const bool mods        { isKeyCodePressed(KEY_LEFTALT) && L_CTRL };
    const xkb_keysym_t sym { keySymbol(keyCode) };

    if (keyState == Released)
    {
        if (keyCode == KEY_F1 && !mods)
            LLauncher::launch("weston-terminal");

        else if (L_CTRL && (sym == XKB_KEY_q || sym == XKB_KEY_Q))
        {
            if (focus())
                focus()->client()->destroy();
        }

        else if (L_CTRL && (sym == XKB_KEY_m || sym == XKB_KEY_M))
        {
            if (focus() && focus()->toplevel() && !focus()->toplevel()->fullscreen())
                focus()->setMinimized(true);
        }

        // Screenshot
        else if (L_CTRL && L_SHIFT && keyCode == KEY_3)
        {
            if (cursor()->output() && cursor()->output()->bufferTexture(0))
            {
                std::filesystem::path path { getenvString("HOME") };

                if (path.empty())
                    return;

                path /= "Desktop/Louvre_Screenshoot_";

                char timeString[32];
                const auto now { std::chrono::system_clock::now() };
                const auto time { std::chrono::system_clock::to_time_t(now) };
                std::strftime(timeString, sizeof(timeString), "%Y-%m-%d %H:%M:%S.png", std::localtime(&time));

                path += timeString;

                cursor()->output()->bufferTexture(0)->save(path);
            }
        }

        else if (keyCode == KEY_ESC && L_CTRL && L_SHIFT)
        {
            compositor()->finish();
            return;
        }
        else if (L_CTRL && !L_SHIFT)
            seat()->dndManager()->setPreferredAction(LDNDManager::Copy);
        else if (!L_CTRL && L_SHIFT)
            seat()->dndManager()->setPreferredAction(LDNDManager::Move);
        else if (!L_CTRL && !L_SHIFT)
            seat()->dndManager()->setPreferredAction(LDNDManager::NoAction);
    }

    // Key pressed
    else
    {
        // CTRL sets Copy as the preferred action in drag & drop session
        if (L_CTRL)
            seat()->dndManager()->setPreferredAction(LDNDManager::Copy);

        // SHIFT sets the Move as the preferred action in drag & drop session
        else if (L_SHIFT)
            seat()->dndManager()->setPreferredAction(LDNDManager::Move);
    }
}

void EKeyboard::keyModifiersEvent(UInt32 depressed, UInt32 latched, UInt32 locked, UInt32 group)
{
    sendModifiersEvent(depressed, latched, locked, group);
}
```

Once again we are using the default implementation provided by Louvre. We will delve into the details of what it does shortly. However, before we do that, let's override the Louvre::LCompositor::createKeyboardRequest() virtual constructor:

#### src/ECompositor.h

```cpp
    // ...

    // Virtual constructors
    LOutput *createOutputRequest(const void *params) override;
    LSurface *createSurfaceRequest(const void *params) override;
    LPointer *createPointerRequest(const void *params) override;
    LKeyboard *createKeyboardRequest(const void *params) override;

    // ...
```

#### src/ECompositor.cpp

```cpp
// ...

#include "EKeyboard.h"

// ...

LKeyboard *ECompositor::createKeyboardRequest(const void *params)
{
    return new EKeyboard(params);
}
```

Let's take a closer look at what's happening in Louvre::LKeyboard::keyEvent():

#### src/EKeyboard.cpp

```cpp
void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    sendKeyEvent(keyCode, keyState);

    const bool L_CTRL      { isKeyCodePressed(KEY_LEFTCTRL) };
    const bool L_SHIFT     { isKeyCodePressed(KEY_LEFTSHIFT) };
    const bool mods        { isKeyCodePressed(KEY_LEFTALT) && L_CTRL };
    const xkb_keysym_t sym { keySymbol(keyCode) };

    // ...
}
```

First, we simply send the key event to the currently focused surface, set with Louvre::LKeyboard::setFocus() as we discussed in the previous chapter. 

> It's worth noting that we send the raw key codes to the client and not XKB symbols. Clients use the same keymap we set in [Chapter 4: Compositor Initialization](04.md) which is automatically sent to them when they connect to the compositor.

Next, we use raw key codes to detect whether the left `Ctrl`, `Shift`, and `Alt` keys are pressed. We are not using the Louvre::LKeyboard::isModActive() method because it would also return `true` if, for example, we press the right `Shift` or `Alt` keys. The Louvre::LKeyboard::isKeyCodePressed() method allows us to determine if a raw key code is pressed. Additionally, we can obtain a list of all currently pressed keys using Louvre::LKeyboard::pressedKeys().

#### src/EKeyboard.cpp

```cpp
void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    // ...

    if (keyState == Released)
    {
        if (keyCode == KEY_F1 && !mods)
            LLauncher::launch("weston-terminal");

        // ...
    }

    // ...
}
```

Then, when the `F1` key is released, we launch [weston-terminal](https://gitlab.freedesktop.org/wayland/weston).

We also verify that both the left `Ctrl` and `Alt` keys are not pressed during this process. This check is necessary because when you press `Ctrl + Alt + F[1,...,10]`, the system switches to another TTY (if libseat is enabled).

#### src/EKeyboard.cpp

```cpp
void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    // ...

    if (keyState == Released)
    {
        // ...

        else if (L_CTRL && (sym == XKB_KEY_q || sym == XKB_KEY_Q))
        {
            if (focus())
                focus()->client()->destroy();
        }

        // ...
    }

    // ...
}
```

If we press `Ctrl + Q`, we invoke Louvre::LClient::destroy() to the client owner of the currently keyboard-focused surface. This action essentially disconnects the client from the compositor.\n
It's important to note that this approach is somewhat abrupt, as it doesn't provide the client an opportunity, for example, to display a "Save before exit" dialog. A more graceful alternative to consider is using Louvre::LToplevelRole::close() instead.

#### src/EKeyboard.cpp

```cpp
void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    // ...

    if (keyState == Released)
    {
        // ...

        else if (L_CTRL && (sym == XKB_KEY_m || sym == XKB_KEY_M))
        {
            if (focus() && focus()->toplevel() && !focus()->toplevel()->fullscreen())
                focus()->setMinimized(true);
        }

        // ...
    }

    // ...
}
```

When `Ctrl + M` is pressed, we minimize the currently focused surface.

#### src/EKeyboard.cpp

```cpp
void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    // ...

    if (keyState == Released)
    {
        // ...

        // Screenshot
        else if (L_CTRL && L_SHIFT && keyCode == KEY_3)
        {
            if (cursor()->output() && cursor()->output()->bufferTexture(0))
            {
                std::filesystem::path path { getenvString("HOME") };

                if (path.empty())
                    return;

                path /= "Desktop/Louvre_Screenshoot_";

                char timeString[32];
                const auto now { std::chrono::system_clock::now() };
                const auto time { std::chrono::system_clock::to_time_t(now) };
                std::strftime(timeString, sizeof(timeString), "%Y-%m-%d %H:%M:%S.png", std::localtime(&time));

                path += timeString;

                cursor()->output()->bufferTexture(0)->save(path);
            }
        }

        // ...
    }

    // ...
}
```

When `Ctrl + Shift + 3` is pressed, we capture a screenshot of the output where the cursor is currently located and save it to your desktop directory.

#### src/EKeyboard.cpp

```cpp
void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    // ...

    if (keyState == Released)
    {
        // ...

        else if (keyCode == KEY_ESC && L_CTRL && L_SHIFT)
        {
            compositor()->finish();
            return;
        }

        // ...
    }

    // ...
}
```

If `Ctrl + Shift + Esc` is pressed, we exit the compositor using Louvre::LCompositor::finish().

#### src/EKeyboard.cpp

```cpp
void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    // ...

    if (keyState == Released)
    {
        // ...

        else if (L_CTRL && !L_SHIFT)
            seat()->dndManager()->setPreferredAction(LDNDManager::Copy);
        else if (!L_CTRL && L_SHIFT)
            seat()->dndManager()->setPreferredAction(LDNDManager::Move);
        else if (!L_CTRL && !L_SHIFT)
            seat()->dndManager()->setPreferredAction(LDNDManager::NoAction);
    }

    // Key press
    else
    {
        // CTRL sets Copy as the preferred action in drag & drop session
        if (L_CTRL)
            seat()->dndManager()->setPreferredAction(LDNDManager::Copy);

        // SHIFT sets the Move as the preferred action in drag & drop session
        else if (L_SHIFT)
            seat()->dndManager()->setPreferredAction(LDNDManager::Move);
    }
}
```

Lastly, during a drag & drop operation, we have the option to select our preferred action. Typically, if you are dragging a file to another directory, pressing `Ctrl` would indicate the action to copy the file, while pressing `Shift` would indicate the action to move the file. If neither `Ctrl` nor `Shift` is pressed, we unset the preferred action and leave it to the clients to decide.

The implementation of Louvre::LKeyboard::keyModifiersEvent() is straightforward and doesn't require much explanation. We are simply sending the updated keyboard modifier states to the client.

## Handling Keyboard Events with LScene

Similar to pointer events, we can also delegate keyboard event handling to Louvre::LScene, which provides the added benefit of emitting per-view keyboard events. To enable Louvre::LScene for this purpose, simply "plug it" like this:

#### src/EKeyboard.cpp

```cpp
#include <LScene.h>
#include "EKeyboard.h"
#include "Global.h"

EKeyboard::EKeyboard(const void *params) : LKeyboard(params) {}

void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    G::scene()->handleKeyEvent(keyCode, keyState);
}

void EKeyboard::keyModifiersEvent(UInt32 depressed, UInt32 latched, UInt32 locked, UInt32 group)
{
    G::scene()->handleKeyModifiersEvent(depressed, latched, locked, group);
}
```

Ahh, clean and beautiful.

### Customizing LScene Event Handling

Just like with pointer events, you have the option to disable the Wayland events handling while still receiving per-view events by using Louvre::LScene::enableHandleWaylandKeyboardEvents(). The scene by default also implements the other functionality we saw previously, such as launching [weston-terminal](https://gitlab.freedesktop.org/wayland/weston), exiting the compositor, saving screenshots, etc. You can disable these features using Louvre::LScene::enableAuxKeyboardImplementation().

> Throughout the tutorial, we don't recommend disabling these features because doing so might leave you locked inside your compositor without a shortcut to terminate it or access to a terminal to kill its process. This could potentially force you to restart your machine.

### Native Input Events

Louvre also offers the ability to monitor native input backend events through the Louvre::LSeat::nativeInputEvent() virtual method. The parameter of this method contains an opaque data structure that can be cast to either a [struct libinput_event](https://wayland.freedesktop.org/libinput/doc/latest/api/structlibinput__event.html) or [XEvent](https://www.x.org/releases/X11R7.6/doc/libX11/specs/libX11/libX11.html), depending on whether the libinput or X11 input backend is in use. Use Louvre::LSeat::inputBackendId() to determine which input backend is in use.

> As of Louvre v1.0.0, the X11 input backend is considered obsolete and is no longer maintained. Therefore, you can be confident that this method exclusively provides [struct libinput_event](https://wayland.freedesktop.org/libinput/doc/latest/api/structlibinput__event.html) events.

Using this method allows you to tap into a more extensive range of events, including touch events and multi-finger pointer gestures, among others. For a practical demonstration, refer to the [louvre-views](md_md__examples.html#views) example which handles three-finger swipe events for workspace switching.

Please note that touch and pointer gesture events are planned for inclusion in the upcoming Louvre 2.0.0 release, so your patience is appreciated.

In the next chapter, we will explore how to manage toplevel surfaces and prevent them from positioning under the topbar.

<a href="08.md">◀ Chapter 8: Pointer Events</a> || <a href="10.md"> Chapter 10: Toplevels ▶</a>