# Chapter 10: Toplevels {#tutorial-toplevels}

As previously discussed, toplevel surfaces represent the typical windows containing a title bar and buttons for close, minimize, and maximize actions. They are perhaps the most common type of surfaces encountered in a compositor during everyday use.

<center>
<img style="max-width:45vw" src="https://lh3.googleusercontent.com/PvNaxLgkjaPryJ8W_P3bDOccQu1m-zNEcI7aH_R8WggzylV5LQZtuzLTUSImThDI8IVsAI9DERF4cwvSqPAEAyjsihHuPCQlZAbvu33iMC2iXvpTswZC3RBNDyKm1YEWDnnKeCn2Qw=w2400"/>
</center>

Toplevels can either exist independently or serve as children of other toplevels, such as dialogs do. The functionality of toplevels is primarily managed within the Louvre::LToplevelRole class. In this chapter, we will learn how to interact with them and handle their various requests.

### State Flags

Toplevel surfaces can exhibit different states, which are listed in the Louvre::LToplevelRole::State flags enum. Among these states, the most common ones include:

* Louvre::LToplevelRole::Activated: When this flag is set, the toplevel's decorations are highlighted above others. Only one toplevel can be active at a time.

* Louvre::LToplevelRole::Maximized: When this flag is enabled, the toplevel typically removes its decorations and occupies a significant portion of the screen.

* Louvre::LToplevelRole::Fullscreen: Enabling this flag typically results in the toplevel removing its decorations and completely occupying the entire screen, always staying displayed above other surfaces (except for its child surfaces).

* Louvre::LToplevelRole::Resizing: This flag indicates an ongoing interactive resizing session, usually initiated when the user drags one of its corners.

> Toplevels can also request to be minimized, but this is not considered a toplevel state.

### Current and Pending States

To set the state flags of a toplevel, you can select from various Louvre::LToplevelRole::configure() variants, which also allow you to specify the desired size. If you provide a size of (0,0), the client is at liberty to determine its own dimensions. You can retrieve the current toplevel state flags and size using the Louvre::LToplevelRole::states() and Louvre::LToplevelRole::size() properties.

It's crucial to recognize that the state of a toplevel doesn't immediately change upon calling Louvre::LToplevelRole::configure(). You must await the respective property listener to detect the actual state change. For instance, if you configure a toplevel with the Louvre::LToplevelRole::Maximized flag, wait for the Louvre::LToplevelRole::maximizedChanged() event to ensure proper positioning. Alternatively, monitor changes in any state using Louvre::LToplevelRole::statesChanged().

As mentioned earlier, you can access the current size and state flags of the toplevel with Louvre::LToplevelRole::size() and Louvre::LToplevelRole::states(), respectively. Meanwhile, Louvre::LToplevelRole::pendingSize() and Louvre::LToplevelRole::pendingStates() represent the values from the last Louvre::LToplevelRole::configure() that has not yet been acknowledged.

Once a configuration is acknowledged by the client (when the current state and size actually change), the values of pending state and size are updated to match the current state and size. It's important to note that you will likely call Louvre::LToplevelRole::configure() multiple times before these changes are acknowledged by the client. Therefore, using the pending size and state flags when configuring is advisable to avoid undesired results stemming from potential race conditions.

### Window Geometry

The size of a toplevel is not determined by its surface size but by its window geometry (`Louvre::LToplevelRole::windowGeometry().size()` or `Louvre::LToplevelRole::size()`). The window geometry specifies the rect within the surface that encompasses the actual toplevel window, excluding its decorative elements like shadows. This is the size you should take into account when configuring it.

<center>
<img style="max-width:25vw;min-width:400px" src="https://lh3.googleusercontent.com/pw/AIL4fc9WuCZOWGGC6v7ZepWA7VOWygycmNFEngEHX7J9ODdmzm1Qs4au1KYPMRJ_102tfvOXFBcL-Kwghm0bSKbd4U6der3p94l1drpn3ONqb9yfv-uFgHQ=w2400"/>
</center>

### Decorations

By default, clients handle their own toplevel decorations. However, some applications, like Qt applications, implement the [XDG Decoration](https://wayland.app/protocols/xdg-decoration-unstable-v1) protocol, allowing the compositor to take charge of drawing the decorations.

It's worth noting that this protocol is still somewhat experimental, and only a limited number of clients support it. Therefore, for the purpose of this tutorial, we will focus on client-side decorations. If you're interested in exploring server-side decorations, please refer to the [louvre-views](md_md__examples.html#views) example, which creates macOS-style decorations for toplevels that support it.

### Interactive Events

Clients can request to initiate interactive moving or resizing of toplevels, typically triggered by the user clicking on its top bar or one of its corners. We will explore how to handle these requests shortly.

Let's begin by crafting our own Louvre::LToplevelRole subclass, which we'll name `EToplevel`:

#### src/EToplevel.h

```cpp
#ifndef ETOPLEVEL_H
#define ETOPLEVEL_H

#include <LToplevelRole.h>

using namespace Louvre;

class ESurface;

class EToplevel : public LToplevelRole
{
public:
    EToplevel(const void *params);

    inline ESurface *surf() const
    {
        return (ESurface*) surface();
    };

    void configureRequest() override;
    void setMinimizedRequest() override;
};

#endif // ETOPLEVEL_H
```

Here, we're overriding the Louvre::LToplevelRole::configureRequest() virtual method. This method is invoked when the toplevel is created and every time it's mapped after being previously unmapped. Within this method, we should call Louvre::LToplevelRole::configure() to properly set the initial state and size of the toplevel.

Additionally, we've overridden Louvre::LToplevelRole::setMinimizedRequest(), where we will allow the client to minimize the toplevel only if it's not in fullscreen mode. Finally, we've created an auxiliary method to perform the casting of Louvre::LSurface into our custom `ESurface` type.

#### src/EToplevel.cpp

```cpp
#include "EToplevel.h"
#include "ESurface.h"

EToplevel::EToplevel(const void *params) : LToplevelRole(params) {}

void EToplevel::configureRequest()
{
    // Request the client to draw its own window decorations
    setDecorationMode(ClientSide);

    // Configure the toplevel with (0,0) so that the client can decide the size
    configure(LSize(0, 0), pendingStates() | Activated);
}

void EToplevel::setMinimizedRequest()
{
    if (!fullscreen())
        surf()->setMinimized(true);
}
```

In this case, we're requesting the client to use client-side decorations. Additionally, we're configuring the toplevel with a size of (0,0), allowing it to determine its own size. Furthermore, we're adding the Louvre::LToplevelRole::Activated state using the "|" operator. If you're not yet familiar with working with bit flags, you can refer to [this post](https://dietertack.medium.com/using-bit-flags-in-c-d39ec6e30f08) by Dieter Tack for a detailed guide on handling them.

Additionally, don't overlook its virtual constructor.

#### src/ECompositor.h

```cpp
    // ...

    // Virtual constructors
    LOutput *createOutputRequest(const void *params) override;
    LSurface *createSurfaceRequest(const void *params) override;
    LPointer *createPointerRequest(const void *params) override;
    LKeyboard *createKeyboardRequest(const void *params) override;
    LToplevelRole *createToplevelRoleRequest(const void *params) override;

    // ...
```

#### src/ECompositor.h

```cpp
// ...

#include "EToplevel.h"

// ...

LToplevelRole *ECompositor::createToplevelRoleRequest(const void *params)
{
    return new EToplevel(params);
}
```

After recompiling and running the compositor, you should observe that everything remains unchanged. To explore further, consider configuring the toplevels with different sizes and state flags to observe their responses.

You may have also noticed that upon creation, all toplevels are automatically centered on the screen. This behavior is a result of the default implementation of Louvre::LSurface::mappingChanged(), which centers them each time they are remapped.

To modify this behavior, let's override it and introduce a boolean variable in our `ESurface` class. This variable will ensure that the toplevels are centered only during their initial mapping.

#### src/ESurface.h

```cpp
// ...

class EToplevel;

class ESurface : public LSurface
{
public:

    inline EToplevel *tl()
    {
        return (EToplevel*)toplevel();
    };

    // ...

    void mappingChanged() override;

    // ...

    bool firstMapping { true };
};
```

We initialize `firstMapping` as `true`, and after the initial mapping, we set it to `false`.

#### src/ESurface.cpp

```cpp
// ...

#include "EToplevel.h"

// ...

void ESurface::mappingChanged()
{
    if (mapped() && firstMapping)
    {
        firstMapping = false;

        // We only center toplevels the first time they are mapped
        if (tl())
        {
            // Use the output where the cursor is currently located
            LOutput *output { cursor()->output() };

            // If output == nullptr it means there is no initialized output
            if (!output)
                return;

            // Restrict the area to the output size - the topbar area
            const LSize availableSize { output->size() - LSize(0, TOPBAR_HEIGHT) };

            // Center the toplevel
            setPos( (output->pos() + LPoint(0, TOPBAR_HEIGHT)) + (availableSize - tl()->size()) / 2);

            // Prevent positioning it under the topbar (when the toplevel height is greater than dstSize.h())
            if (pos().y() < output->pos().y() + TOPBAR_HEIGHT)
                setY(output->pos().y() + TOPBAR_HEIGHT);

            output->repaint();
        }

        raise();
    }

    /* Note: LSurfaceViews are automatically hidden when their surfaces are unmapped */

    compositor()->repaintAllOutputs();
}

// ...
```

Louvre::LSurface::mappingChanged() is invoked whenever there's a change in the Louvre::LSurface::mapped() property. In contrast to the default approach, we have improved it by considering the topbar's height when centering the toplevel. When a toplevel's size exceeds the output size minus the topbar region, we reposition it downward to prevent it from appearing beneath the topbar.

### Interactive Toplevel Movement

Clients have the capability to request the initiation of interactive toplevel movement sessions, typically triggered when a user drags a toplevel window's titlebar. This request is managed within the Louvre::LToplevelRole::startMoveRequest() virtual method. Louvre also provides auxiliary methods for handling these sessions through the Louvre::LPointer class.

To initiate an interactive toplevel movement session, use Louvre::LPointer::startMovingToplevel(). This method accepts several arguments, including the target toplevel window you intend to move and the defined constraint area that specifies where the toplevel can be positioned. 

The toplevel's position is then updated by invoking Louvre::LPointer::updateMovingToplevelPos() within a pointer move event, as we previously demonstrated in [Chapter 8: Pointer Events](08.md), taking the cursor's position into consideration. 

Lastly, to conclude the session, we call Louvre::LPointer::stopMovingToplevel() when releasing the left pointer button as we also saw.

> You are not obligated to employ the Louvre::LPointer auxiliary methods for managing interactive moving toplevel sessions. Instead, you can manually update their positions using Louvre::LSurface::setPos(), as previously explained.

Let's customize the default behavior of Louvre::LToplevelRole::startMoveRequest() to prevent toplevel windows from being positioned under the topbar.

#### src/EToplevel.h

```cpp
// ...

class EToplevel : public LToplevelRole
{
public:
    
    // ...

    void startMoveRequest() override;
};

// ...
```

#### src/EToplevel.cpp

```cpp
#include <LSeat.h>
#include <LPointer.h>
#include <LCursor.h>
#include "ESurface.h"
#include "EToplevel.h"
#include "Global.h"

// ...

void EToplevel::startMoveRequest()
{
    if (!fullscreen() && !maximized() && surf()->hasPointerFocus())
        seat()->pointer()->startMovingToplevel(
            this,                    // The toplevel being dragged
            cursor()->pos(),         // The initial cursor position
            LPointer::EdgeDisabled,  // Left edge constraint
            TOPBAR_HEIGHT);          // Top edge constraint
}
```

Here, we initiate the moving session only if the toplevel has pointer focus and is not in fullscreen mode or maximized. Additionally, we impose a constraint on the top edge position by assigning it the `TOPBAR_HEIGHT` macro as the value for the T (top) argument.

It's worth noting that, by default, all edges are disabled. You have the option to disable a specific edge restriction by using the Louvre::LPointer::EdgeDisabled value.

Upon recompiling and running the compositor again, you'll observe that toplevel windows can no longer be positioned under the topbar. However, if you resize them, they still can... let's explore how to address this issue.

### Interactive Toplevel Resizing

Much like interactive movement, clients can initiate a resizing session by requesting it through Louvre::LToplevelRole::startResizeRequest(). This is commonly triggered when users drag one of the corners or edges of a toplevel window.

The Louvre::LPointer class also provides auxiliary methods to manage these toplevel resizing sessions. The process begins with a call to Louvre::LPointer::startResizingToplevel(). As the cursor position changes, we invoke Louvre::LPointer::updateResizingToplevelSize(), and each time the toplevel's size changes (handled within Louvre::LToplevelRole::geometryChanged()), we update its position using Louvre::LPointer::updateResizingToplevelPos(). Lastly, upon releasing the left pointer button (implemented in [Chapter 8: Pointer Events](08.md)), we call Louvre::LPointer::stopResizingToplevel() to conclude the resizing session.

> It's important to note that both moving and resizing sessions will automatically stop if the toplevel is destroyed. Therefore, you don't need to be overly cautious about handling these situations yourself; the library takes care of this for you.

The Louvre::LPointer auxiliary methods internally configure the toplevel with the Louvre::LToplevelRole::Resizing flag during the session. If you want to handle toplevel resizing sessions manually, you should configure them with that flag and unset it when it finishes.

Let's customize the default behavior of Louvre::LToplevelRole::startResizeRequest() to prevent toplevel windows from being positioned under the topbar.

#### src/EToplevel.h

```cpp
// ...

class EToplevel : public LToplevelRole
{
public:
    
    // ...

    void startResizeRequest(ResizeEdge edge) override;
    void geometryChanged() override;
};

// ...
```

#### src/EToplevel.cpp

```cpp
// ...

void EToplevel::startResizeRequest(ResizeEdge edge)
{
    if (!fullscreen() && !maximized() && surf()->hasPointerFocus())
        seat()->pointer()->startResizingToplevel(
            this,                   // The toplevel being resized
            edge,                   // Edge or corner being dragged
            cursor()->pos(),        // Initial cursor position
            LSize(0, 0),            // Minimum allowed size
            LPointer::EdgeDisabled, // Left edge constraint
            TOPBAR_HEIGHT);         // Top edge constraint
}

void EToplevel::geometryChanged()
{
    /* When the toplevel size changes during interactive resizing
     * we need to update its position depending on the dragged edge/corner */
    if (resizing())
        updateResizingPos();
}
```

Just like with the interactive moving session, we ignore the request if the toplevel is in fullscreen mode, maximized or if its surface has no pointer focus.

As for Louvre::LPointer::startResizingToplevel(), we pass it the toplevel being resized, the edge or corner being dragged, the current cursor position, the minimum toplevel size `LSize(0,0)` and also the positioning constraints for each toplevel size. In this case `TOPBAR_HEIGHT` for the top edge to prevent it from being positioned under the topbar.\n

If you recompile and run it again, you will notice that the toplevels no longer position under the topbar while being resized. 

<img src="https://lh3.googleusercontent.com/pw/ADCreHe9h6_f3uF5llrKa4nt-n88dwoIDXtS-QA67EFT6PVFtDHdxlE5dP74X46eTuJ14sI6F_iEuQtBnsPUn0YjRHCqO0ENrLd3lBWM0d1o8lvbHUMINjY=w2400"/>

> You might also observe that **weston-terminal** leaves an empty space between its toplevel window and the top bar when being resized against it. This behavior is due to **weston-terminal** using fixed size steps and is unrelated to Louvre. If you try another client, such as **gedit**, you'll notice that this issue doesn't occur.

Let's now discuss how to handle maximization requests.

### Handling Maximization Requests

Clients can request to maximize their toplevels by using Louvre::LToplevelRole::setMaximizedRequest() and unmaximize them using Louvre::LToplevelRole::unsetMaximizedRequest(). This is typically done in response to user actions such as clicking the maximize button or pressing a keyboard shortcut.

Within these requests, we can configure the toplevels with or without the Louvre::LToplevelRole::Maximized flag and specify the desired size. We then wait for the Louvre::LToplevelRole::maximizedChanged() or Louvre::LToplevelRole::statesChanged() events to ensure proper positioning.

> The compositor is free to ignore such requests.

Let's override these methods to configure the toplevels based on the output size, subtracting the topbar region.

#### src/EToplevel.h

```cpp
// ...

class EOutput;

class EToplevel : public LToplevelRole
{
public:
    
    // ...

    void statesChanged() override;
    bool updateGeometry(bool fallbackToNormalOnFail);

    void setMaximized();
    void setMaximizedRequest() override;
    void unsetMaximizedRequest() override;
    void maximizedChanged() override 
    { /* Using statesChanged() instead */ };

    void saveNormalState();

    EOutput *output         { nullptr };
    LPointF prevRelativePos { 200.f, 200.f};
    LSize prevSize          { 0, 0};
};

// ...
```

We also introduced some auxiliary variables and methods to store and recover the toplevel's state to its previous "normal" state before maximizing it.

#### src/EToplevel.cpp

```cpp
// ...

void EToplevel::saveNormalState()
{
    output = surf()->primaryOutput();

    if (output)
    {
        const LPointF localPos { surf()->pos() - output->pos() };
        prevRelativePos = localPos / LSizeF(output->size());
    }
    else if (!G::outputs().empty())
    {
        output = G::outputs().front();
        prevRelativePos = LSizeF(0.25f, 0.25f);
    }

    prevSize = size();
}

// ...
```

So first, in `saveNormalState()`, we will store the current primary output, position, and size of the toplevel before it is maximized or set to fullscreen, similar to what we did when minimizing surfaces.

#### src/EToplevel.cpp

```cpp
// ...

void EToplevel::setMaximizedRequest()
{
    if (pendingStates() & Maximized || !(surf()->hasKeyboardFocus() || surf()->hasPointerFocus()))
        return;

    setMaximized();
}

// ...
```

When a client requests to maximize the toplevel, we first check if it is already maximized or lacks pointer and keyboard focus, if so, we ignore the request.

#### src/EToplevel.cpp

```cpp
// ...

void EToplevel::setMaximized()
{
    const bool inNormalState = { output == nullptr };

    if (inNormalState)
        saveNormalState();

    // Halt if zero initialized outputs
    if (!output)
        return;

    const LSize targetSize { output->size() - LSize(0, TOPBAR_HEIGHT) };

    /* If the toplevel does not support the target size, we will configure it with that size regardless.
     * However, we will set only the activated flag, ensuring it expands solely in size, keeping its
     * decorations */
    if (!sizeInRange(targetSize))
    {
        if (inNormalState)
            output = nullptr;

        configure(targetSize, Activated);
        return;
    }

    configure(targetSize, Activated | Maximized);
}

// ...
```
Next, in our `setMaximized()` method, if the `output` variable is set, it indicates that `saveNormalState()` was previously called, and the toplevel is currently in fullscreen mode, eliminating the need to invoke it again.

Subsequently, we verify if the toplevel supports our target size (output size minus the topbar height). Each toplevel possesses an Louvre::LToplevelRole::minSize() and Louvre::LToplevelRole::maxSize(), which specify the range of sizes it can be configured with. If any component of these properties is 0, it implies no restrictions for that specific axis. Louvre::LToplevelRole::sizeInRange() serves as an auxiliary function to check if the toplevel supports a given size.

If the toplevel does not support the given size, we configure it with that size anyway but using only the Louvre::LToplevelRole::Activated flag. This may result in making it larger while preserving its decorations. 

Finally, if the size is supported, we configure it with the Louvre::LToplevelRole::Maximized flag, typically removing its decorations.

#### src/EToplevel.cpp

```cpp
// ...

void EToplevel::geometryChanged()
{
    /* When the toplevel size changes during interactive resizing
     * we need to update its position depending on the dragged edge/corner */
    if (resizing())
        updateResizingPos();
    else
        updateGeometry(true);
}

bool EToplevel::updateGeometry(bool fallbackToNormalOnFail)
{
    if (!output)
        return false;

    /* If the client did not adhere to our instructions, we may consider terminating it.
     * However, out of courtesy, we will instead reconfigure it back to its original
     * normal state. */

    if (maximized())
    {
        const LSize targetSize { output->size() - LSize(0, TOPBAR_HEIGHT) };

        if (fallbackToNormalOnFail && targetSize != size())
        {
            configure(prevSize, Activated);
            return false;
        }

        surf()->setPos(output->pos() + LPoint(0, TOPBAR_HEIGHT));
    }
 
    return true;
}

void EToplevel::statesChanged()
{
    if (!output)
        return;

    /* If output is set then the toplevel changed from maximized or fullscreen
     * to normal */
    if (!maximized() && !fullscreen())
    {
        G::reparentSurfaceAndChildrenViews(surf(), &G::compositor()->surfacesLayer);
        const LPoint prevPos { output->pos() + (prevRelativePos * output->size()) };
        surf()->setPos(prevPos);
        surf()->raise();
        output = nullptr;
        return;
    }

    if (!updateGeometry(true))
        return;

    if (maximized())
    {
        G::reparentSurfaceAndChildrenViews(surf(), &G::compositor()->surfacesLayer);
        surf()->raise();
    }
}

// ...
```

Instead of relying on Louvre::LToplevelRole::maximizedChanged() to listen for client configuration ACK, we use Louvre::LToplevelRole::statesChanged(). This notification occurs after all flags and the toplevel size are updated, proving more convenient for this scenario.

Here, we address all possible cases. The first scenario is when the toplevel transitions from fullscreen or maximized back to the normal state, requiring us to restore its previous normal position.

In the second scenario, when the window is maximized, we first verify whether the toplevel is configured with our predefined target size before proceeding using `updateGeometry()` (we will use this method to both update the toplevel's position and validate its size when `fallbackToNormalOnFail` is set to `true`). If it's not configured accordingly, we revert it to its normal state. If it is configured correctly, we position it at the top-left corner of the output, just below the top bar. This check is also performed each time there is a size change.

The rationale behind moving the surface view and its children to the surfaces layer will become clear shortly when we delve into handling fullscreen requests. 

#### src/EToplevel.cpp

```cpp
// ...

void EToplevel::unsetMaximizedRequest()
{
    if (~pendingStates() & Maximized || !(surf()->hasKeyboardFocus() || surf()->hasPointerFocus()))
        return;

    configure(prevSize, Activated);
}

// ...
```

Finally, when the client requests to unmaximize the toplevel, we check if it is already maximized and if it has pointer or keyboard focus before configuring it back to the normal state.

After recompiling and running the compositor, you'll observe that toplevels no longer maximize themselves behind the topbar, and upon unmaximizing, they return to their previous position and size.

<img src="https://lh3.googleusercontent.com/pw/ADCreHdWo70LhuTMAzHGuj-JukFE9xgmTmf-yj0YUeuMXbLmFXoffBvJtb49yDDQUu9XxK_GmUm2Jj7ct68y0A9FoaPtddBSH0DfOhIIwMsLXSYXU44c3JE=w2400"/>

I made a drawing of you, successfully creating your own compositor with Louvre.

### Handling Fullscreen Requests

In order to display toplevels in fullscreen mode, we'll create a new black Louvre::LSolidColorView and relocate the toplevel, along with its child surfaces, into this view. This approach is necessary because Louvre::LToplevelRole::maxSize() might be smaller than the output size. In such cases, it's important to center the toplevel on the screen to ensure a visually appealing appearance. If other surfaces were left in the background during fullscreen mode, it could appear somewhat unusual.

Just like with maximizing toplevels, we will store the current state before entering fullscreen mode and restore it when exiting fullscreen mode.

#### src/EToplevel.h

```cpp
// ...

#include <LSolidColorView.h>
#include "ECompositor.h"
#include "Global.h"

// ...

class EToplevel : public LToplevelRole
{
public:
    
    // ...

    void setFullscreen();
    void setFullscreenRequest(LOutput *preferredOutput) override;
    void unsetFullscreenRequest() override;
    void fullscreenChanged() override
    { /* Using statesChanged() instead */ };

    LSolidColorView fullscreenBackground {
        0.f, 0.f, 0.f, 1.f, // RGBA
        &G::compositor()->fullscreenLayer
    };

    // ...

};

// ...
```

#### src/EToplevel.cpp

```cpp
// ...

EToplevel::EToplevel(const void *params) : LToplevelRole(params)
{
    fullscreenBackground.setVisible(false);
}

// ...
```

We start by hiding the fullscreen background initially, considering that toplevels have no state upon creation. Additionally, there's no need to explicitly define its translucent region, as we did with the wallpaper texture view. Louvre::LScene automatically determines it based on the alpha (Louvre::LSolidColorView::opacity()) value we set to `1.f` (fully opaque).

#### src/EToplevel.cpp

```cpp
// ...

void EToplevel::setFullscreen()
{
    const bool inNormalState = { output == nullptr };

    if (inNormalState)
        saveNormalState();

    // Halt if zero initialized outputs
    if (!output)
        return;

    const LSize &targetSize { output->size() };

    /* Unlike maximizing, we can simply center the toplevel if its max size is smaller than the output size
     * on the other hand, if its min size is larger than the output size, then we simply ignore the request */
    if ( (minSize().w() > targetSize.w() && minSize().w() != 0 ) || (minSize().h() > targetSize.h() && minSize().h() != 0 ) )
    {
        if (inNormalState)
            output = nullptr;

        return;
    }

    configure(targetSize, Activated | Fullscreen);
}

void EToplevel::setFullscreenRequest(LOutput *preferredOutput)
{
    /* Clients can request to display the toplevel on a specific output
     * or pass nullptr if they have no preference. For simplicity we are
     * simply ignoring them */
    L_UNUSED(preferredOutput)

    if (pendingStates() & Fullscreen || !(surf()->hasKeyboardFocus() || surf()->hasPointerFocus()))
        return;

    setFullscreen();
}

// ...
```

When a client requests to set a toplevel into fullscreen mode, we adopt a similar approach to what we did for maximize requests. The `preferredOutput` parameter represents the client's preferred output for fullscreen mode, which can be `nullptr` if no preference is specified. For simplicity, we simply ignore it and use the output we have selected.

Unlike a maximize request, we configure it with the Louvre::LToplevelRole::Fullscreen flag and using the entire output size, even if its maximum size is smaller than the output. In such cases, we will simply center it on the screen.

> Adhering to the [XDG Shell protocol](https://wayland.app/protocols/xdg-shell#xdg_toplevel:request:set_maximized), we must always configure toplevels when clients request either setting or unsetting them in maximized or fullscreen mode. This applies even if for example a toplevel is already maximized, and the client seeks to maximize it again. If you do not explicitly configure the toplevels during these requests, Louvre will automatically handle the configuration for you, retaining the current pending states and size using `configure(pendingStates())`.

#### src/EToplevel.cpp

```cpp
// ...

bool EToplevel::updateGeometry(bool fallbackToNormalOnFail)
{
    // ..

    if (maximized())
    {
        // ...
    }
    else if (fullscreen())
    {
        const LSize &targetSize { output->size() };

        if (fallbackToNormalOnFail && (size().w() > targetSize.w() || size().h() > targetSize.h()))
        {
            configure(prevSize, Activated);
            return false;
        }

        fullscreenBackground.setPos(output->pos());
        fullscreenBackground.setSize(output->size());

        const LPoint centeredPos { output->pos() + output->size() / 2 - size() / 2 };
        surf()->setPos(centeredPos);
    }

    return true;
}

void EToplevel::statesChanged()
{
    fullscreenBackground.setVisible(fullscreen());

    if (!output)
        return;

    // ...

    if (!updateGeometry(true))
        return;

    if (maximized())
    {
        // ...
    }
    else if (fullscreen())
        G::reparentSurfaceAndChildrenViews(surf(), &fullscreenBackground);
}

// ...
```

Upon receiving the client's configuration acknowledgment, we toggle the visibility of the `fullscreenBackground`, update its size and position, and incorporate the toplevel surface view and its children into it. Additionally, we check if the toplevel size exceeds that of the screen, in such cases, we revert its configuration back to normal. Finally, we center the toplevel.

> An alternative approach could involve clipping the views to the `fullscreenBackground`. To implement this, you would need to enable parent clipping for all views using Louvre::LView::enableParentClipping(). 

#### src/EToplevel.cpp

```cpp
// ...

void EToplevel::unsetFullscreenRequest()
{
    if (~pendingStates() & Fullscreen || !(surf()->hasKeyboardFocus() || surf()->hasPointerFocus()))
        return;

    configure(prevSize, Activated);
}

// ...
```

Finally, we address unfullscreen requests in a manner similar to how we handled unmaximization requests.

After recompiling and running the compositor, you'll observe toplevels seamlessly transitioning into fullscreen mode and returning to their normal state when exiting fullscreen mode. Note: Clients usually let users toggle fullscreen mode by pressing `F1`.

### Managing Surface Parent Changes

As we relocate the fullscreen toplevels and their child surfaces into the `fullscreenBackground` view, it's essential to ensure that any newly created child surfaces are also relocated. To accomplish this, we can override the Louvre::LSurface::parentChanged() virtual method.

#### src/ESurface.h

```cpp
// ...

class ESurface : public LSurface
{
public:

    // ...

    void parentChanged() override;

    // ...
};

// ...
```

#### src/ESurface.cpp

```cpp
// ...

void ESurface::parentChanged()
{
    if (parent())
    {
        const ESurface *parentSurface = { (ESurface*)parent() };
        view.setParent(parentSurface->view.parent());
    }
    else
        view.setParent(&G::compositor()->surfacesLayer);

    repaintOutputs();
}

// ...
```

So, whenever a surface changes its parent, we place it within the same parent view as its parent. If the surface has no parent, we position it within the surfaces layer.

### Managing Output Uninitialization or Rearrangement

In a manner similar to our approach for minimizing surfaces, we also need to address situations where the output stored in `output` becomes uninitialized while the toplevel is in a maximized or fullscreen state. Additionally, we haven't yet covered the scenario where surfaces are no longer visible within any output after such an event occurs. To address these scenarios, let's create two new methods in `EOutput` and add a bit more code to `recoverViewsFromVoid()`.

#### src/EOutput.h

```cpp
// ...

class EOutput : public LOutput
{
public:

    // ...

    void updateToplevelsPos();
    void updateToplevelsSize();
};

// ...
```

#### src/EOutput.cpp

```cpp
// ...

#include "EToplevel.h"

// ...

void EOutput::moveGL()
{
    // ...

    updateToplevelsPos();
    G::scene()->handleMoveGL(this);

    // ...
}

void EOutput::resizeGL()
{
    // ...

    updateToplevelsSize();
    G::scene()->handleResizeGL(this);
    
    // ...
}

void EOutput::updateToplevelsPos()
{
    for (ESurface *surface : G::surfaces())
        if (surface->tl() && surface->tl()->output == this)
            surface->tl()->updateGeometry(false);
}

void EOutput::updateToplevelsSize()
{
    for (ESurface *surface : G::surfaces())
    {
        EToplevel *tl = { surface->tl() };

        if (tl && tl->output == this)
        {
            if (tl->pendingStates() & LToplevelRole::Maximized)
                tl->setMaximized();
            else
                tl->setFullscreen();
        }
    }
}

void EOutput::rescueViewsFromVoid()
{
    // Search another initialized output
    const auto isDiff = [this](const EOutput *output) { return this != output; };
    const auto it { std::find_if(G::outputs().begin(), G::outputs().end(), isDiff) };
    EOutput *fallbackOutput { it != G::outputs().end() ? *it : nullptr };

    for (ESurface *surface : G::surfaces())
    {
        if (surface->outputBeforeMinimize == this)
            surface->outputBeforeMinimize = fallbackOutput;
        else if (surface->tl() && surface->tl()->output == this)
        {
            surface->tl()->output = nullptr;
            surface->setPos(200, 200);
            G::reparentSurfaceAndChildrenViews(surface, &G::compositor()->surfacesLayer);
            surface->tl()->configure(surface->tl()->prevSize, LToplevelRole::Activated);
        }
        else if(surface->primaryOutput() == this)
            surface->setPos(200, 200);
    }
}
// ...
```

Now, when there is a change in the output position, we invoke `updateToplevelsPos()`, which, in turn, simply calls `updateGeometry(false)` on all toplevels that are maximized or in fullscreen mode on this output. It's essential to note that this time we pass `false` because our intention is solely to update their positions and not to validate if they are correctly configured.

In the event of an output size change, we call `updateToplevelsSize()`, which essentially reconfigures each toplevel to fullscreen mode or maximized state but using the new output size.

Finally, within `rescueViewsFromVoid()` (called from `Louvre::LOutput::uninitializeGL()`), we now revert the configuration of all toplevels that are maximized or in fullscreen mode on this output back to normal. Additionally, we relocate any surface that has this output as its primary output to the coordinates (200, 200), making them visible on the output positioned at (0, 0) (unless you are arranging outputs differently).

In the next chapter, we'll explore the handling of popup surfaces, which are notably simpler to manage than toplevels. See you there!

<a href="09.md">◀ Chapter 9: Keyboard Events</a> || <a href="11.md"> Chapter 11: Popups ▶</a>