# Chapter 12: Animations {#tutorial-animations}

In this chapter, we will provide a concise overview of the Louvre::LAnimation class, designed for creating time-based animations. Louvre::LAnimation consists of two essential callback functions:

* **OnUpdate**: This callback function is triggered just before an Louvre::LOutput::paintGL() event of any output.
* **OnFinish**: This callback function is triggered when the animation reaches its conclusion or when Louvre::LAnimation::stop() is called.

To create an animation, you can choose between two options: 
- **Louvre::LAnimation::oneShot()**: This is for creating animations which automatically start and clean up upon completion.
- **Louvre::LAnimation::LAnimation()**: This is used for creating reusable animations, which must be started manually with Louvre::LAnimation::start().

In this example, we will employ Louvre::LAnimation::oneShot() to implement a fade-out animation for toplevels or popups when they are destroyed.

Let's override the virtual destructors of Louvre::LToplevelRole and Louvre::LPopupRole and also create a new method called `fadeOutSurface()` within Louvre::LCompositor for generating this animation. 

#### src/ECompositor.h

```cpp
// ...

class ESurface;
 
class ECompositor : public LCompositor
{
public:
    
    // ...

    // Virtual destructors
    void destroyToplevelRoleRequest(LToplevelRole *toplevel) override;
    void destroyPopupRoleRequest(LPopupRole *popup) override;

    // Fade out surface animation
    void fadeOutSurface(ESurface *surface, UInt32 ms);

    // ...
};
```

Using these virtual destructors is a more secure approach compared to relying on the destructors of the classes themselves. This is due to the fact that virtual destructors are invoked well in advance of the class being fully destroyed. Consequently, many of its parameters and resources remain valid at that point.

#### src/ECompositor.cpp

```cpp
// ...

#include <LAnimation.h>

// ...

void ECompositor::destroyToplevelRoleRequest(LToplevelRole *toplevel)
{
    fadeOutSurface((ESurface*)toplevel->surface(), 400);
}

void ECompositor::destroyPopupRoleRequest(LPopupRole *popup)
{
    fadeOutSurface((ESurface*)popup->surface(), 50);
}

void ECompositor::fadeOutSurface(ESurface *surface, UInt32 ms)
{
    if (!surface)
        return;

    LTexture *capture = surface->capture(surface->size() * 2);
    LTextureView *fadeOutView = new LTextureView(capture, &cursorLayer);
    fadeOutView->setBufferScale(2);
    fadeOutView->enableParentOffset(false);
    fadeOutView->setPos(surface->rolePos());

    // Stack it below the software cursor view
    fadeOutView->insertAfter(nullptr);

    LAnimation::oneShot(ms,

    // On Update
    [this, fadeOutView](LAnimation *anim)
    {
        fadeOutView->setOpacity(1.f - anim->value());
        repaintAllOutputs();
    },

    // On Finish
    [this, fadeOutView](LAnimation *)
    {
        repaintAllOutputs();
        delete fadeOutView->texture();
        delete fadeOutView;
    });
}
```

Now, once a toplevel or popup is being destroyed, we call the `fadeOutSurface()` method. This method takes two parameters: the surface to animate and the duration of the animation in milliseconds.

Within this method, we take several steps:

1. We first check if the surface is `nullptr`. This check is essential because in some uncommon cases, the role may be destroyed when it's not yet attached to a surface. It's crucial always to verify this.

2. Next, we create a screenshot of the surface using the `ESurface::capture()` method we introduced earlier. Then, we generate a texture view to display the captured texture and insert it into the cursor layer. This ensures that it remains visible above other views, even if there is a toplevel in fullscreen mode.

3. Within the `onUpdate()` callback, we simply manipulate the view's opacity from 1.0 to 0.0 using the Louvre::LAnimation::value() property. This property returns a floating-point number ranging from 0.0 to 1.0, indicating the completion percentage of the animation. 

4. It's important to note that we also need to call Louvre::LOutput::repaint() on the outputs we are animating. Otherwise, the next `onUpdate()` callback may not be triggered.

5. Finally, within the `onFinish()` callback, we destroy the captured texture and the texture view. Since we are using Louvre::LAnimation::oneShot(), the animation is automatically destroyed upon completion.

If you recompile and run the compositor, you will notice that toplevels and popups now smoothly fade out when they are destroyed.

<img src="https://lh3.googleusercontent.com/pw/ADCreHdFDc8mTY0fxNtr42T-7ACP0wdINLsLKahQ_wQu8M4O112dJBOi5qUvb_sPADRAcN46uCBNASlF2j1tvYsO3fIZma0O8JtULThP9LoJ5VXqBxYDGLo=w2400"/>


In the upcoming chapter, we will delve into how to manage cursors using the Louvre::LCursor class. Bis bald!

<a href="11.md">◀ Chapter 11: Popups</a> || <a href="13.md"> Chapter 13: Cursor ▶</a>