# Chapter 15: Output Hotplugging and Seat {#tutorial-seat}

In our earlier discussion within [Chapter 4: Compositor Initialization](04.md), we explored the arrangement of outputs during the compositor initialization process. This chapter delves into the scenarios involving the connection and disconnection of outputs, and as a final addition, we will delve into the handling of TTY switch events.

Whenever a new output is connected, the Louvre::LSeat::outputPlugged() event is triggered. Within this method, you have the freedom to decide whether to initialize the output using Louvre::LCompositor::addOutput() or not. Similarly, when an output is no longer available, the Louvre::LSeat::outputUnplugged() event is triggered. If the given output was initialized while unplugged, it is automatically uninitialized after this event, even if you don't explicitly call Louvre::LCompositor::removeOutput().

Let's override these methods to see how to handle these situations:

#### src/ESeat.h

```cpp
// ...

class ESeat : public LSeat
{
public:
    // ...

    void outputPlugged(LOutput *output) override;
    void outputUnplugged(LOutput *output) override;
};

// ...
```

#### src/ESeat.cpp

```cpp
// ..

#include "EOutput.h"
#include "Global.h"

void ESeat::outputPlugged(LOutput *output)
{
    output->setScale(output->dpi() >= 200 ? 2 : 1);

    if (G::outputs().empty())
        output->setPos(LPoint(0,0));
    else
        output->setPos(G::outputs().back()->pos() + LPoint(G::outputs().back()->size().w(), 0));

    compositor()->addOutput(output);
    compositor()->repaintAllOutputs();
}

void ESeat::outputUnplugged(LOutput *output)
{
    compositor()->removeOutput(output);

    Int32 totalWidth { 0 };

    for (EOutput *o : G::outputs())
    {
        o->setPos(LPoint(totalWidth, 0));
        totalWidth += o->size().w();
    }

    compositor()->repaintAllOutputs();
}
```

When a new output is plugged, our approach is to adjust its scale factor, position it at the rightmost side of the already initialized outputs, and initialize it using Louvre::LCompositor::addOutput().

And when an output is unplugged, we first manually uninitialize it just for removing it from Louvre::LCompositor::outputs(). Afterward, we re-arrange the still-initialized outputs.

For a firsthand experience of these changes, recompile and run the compositor, and experiment with plugging and unplugging outputs on the fly to observe how the compositor reacts.

### TTY Switching

Throughout this tutorial, we've frequently referred to the Louvre::LSeat class, and you may be curious about its purpose. In the realm of Linux/Wayland, the concept of a "seat" represents a collection of input and output devices. Think of it as a physical "seat" where a user is situated, providing access to their keyboard, mouse, screen, and more. This concept becomes particularly important in scenarios like educational institutions where multiple students can share the same machine, while each having their dedicated monitor, keyboard, and so on. This setup is commonly known as a multi-seat configuration.

However, in the most typical scenario, a single machine is equipped with a single set of input and output devices. In such cases, Linux still allows you to run multiple user sessions concurrently but in different virtual terminals (TTYs), which you can switch between using shortcuts like `Ctrl + Shift + [F1, F2, ..., FN]` or through the `chvt` command. This capability enables you to have multiple desktop environments running simultaneously or, in the context of Louvre, run multiple compositors on different TTYs.

Each of these desktop environments, compositors, etc., needs to interact with input devices and render content on the screen. However, if there is only one physical screen, this simultaneous interaction would be chaotic. Linux device drivers are designed to allow only one process to manage them at a given time. For example, in the DRM/KMS API, the concept of the DRM master exists, and only one process can be the DRM master, thereby controlling the screen content.

Since there are multiple TTYs, there needs to be a way to negotiate these permissions during TTY switches. This is where seat services and libraries like [libseat](https://github.com/kennylevinsen/seatd) come into play. Programs that integrate with seat services enable other processes to take ownership of the seat, and Louvre is no exception.

By default, Louvre allows users to switch between TTYs using the methods mentioned earlier or by invoking the Louvre::LSeat::setTTY() method. When the user switches to another TTY, the compositor can no longer listen to input events or draw content on screens. You have the option to disable this feature by setting the **LOUVRE_ENABLE_LIBSEAT** environment variable to 0. When the compositor is run in this manner, it won't allow any other process to take ownership of the seat, and TTY switching won't be possible.

With this feature enabled, when the user is in another TTY, there is a possibility for other processes to modify resources used by the compositor, such as the hardware cursor plane, CRTCs, among others. Louvre automatically restores the previous state when resuming the session. However, the Louvre::LScene inter-frame damage calculation may go out of sync. To appropriately manage such cases and keep track of TTY switches, you can override the virtual method Louvre::LSeat::enabledChanged().

#### src/ESeat.h

```cpp
// ...

class ESeat : public LSeat
{
public:
    // ...

    // TTY switching
    void enabledChanged() override;
};

// ...
```

#### src/ESeat.cpp

```cpp
// ..

#include <LSceneView.h>
#include <LScene.h>
#include <LCursor.h>

// ...

void ESeat::enabledChanged()
{
    if (!enabled())
        return;

    /* Synchronize LScene's output inter-frame damage tracking by performing
     * full damage on all outputs. */
    for (EOutput *output : G::outputs())
    {
        G::scene()->mainView()->damageAll(output);
        output->repaint();
    }
}
```

Here, each time the seat is re-enabled, indicating a user's return from another session, we apply damage to all the outputs within our scene. This precautionary measure helps prevent the occurrence of odd visual glitches.

<a href="14.md">◀ Chapter 14: Clipboard and DND</a> || <a href="16.md"> Chapter 16: Gamma Correction ▶</a>