# Chapter 16: Gamma Correction {#tutorial-gamma}

Since version 1.2.0, Louvre now offers support for applying custom gamma correction curves to outputs. This feature proves valuable, especially for graphic designers seeking precise representation and translation of their works across various devices, or for implementing functionalities such as adjusting screen temperature based on the current time.

To configure gamma correction curves for an output, you can use Louvre::LOutput::setGamma(), which accepts an Louvre::LGammaTable as an argument. The Louvre::LGammaTable comprises an array of Louvre::UInt16 values representing the components of each curve (red, green, and blue). When creating a table, its size must match the gamma size of the output to which it will be applied, obtainable through Louvre::LOutput::gammaSize(). Keep in mind that if an output doesn't support gamma correction, it returns 0. It also returns 0 if the output is uninitialized, so it's advisable to set a gamma table only after the output is initialized.

> By default, Louvre assigns linear gamma curves each time an output is initialized. 

Let's have some fun and experiment creating a custom gamma table that inverts the colors displayed on the screen.

#### src/EOutput.cpp

```cpp
// ...

#include <LGammaTable.h>

// ...

void EOutput::initializeGL()
{
    const UInt32 gSize { gammaSize() };

    if (gSize > 0)
    {
        LGammaTable gamma { gSize };

        UInt16 *r { gamma.red()   };
        UInt16 *g { gamma.green() };
        UInt16 *b { gamma.blue()  };

        const Float64 n { (Float64)gSize - 1 };

        for (UInt32 i = 0; i < gSize; i++)
        {
            const Float64 val = 1.0 - i/n;
            r[i] = g[i] = b[i] = (UInt16)(UINT16_MAX * val);
        }

        setGamma(&gamma);
    }
    else
        LLog::debug("[louvre-example] Output %s doesn't support gamma correction.", name());

    // ...
}

// ...
```

Upon recompiling and running, you should observe something like this on outputs that support gamma correction:

<img src="https://lh3.googleusercontent.com/pw/ABLVV84-HXAmJ5CXMpVFMdbIUlpc70MeQD3HTFYvDW3piVYv28u6rmdW2GcetZMFnoxtZ9E8aMEnWswl7eouc4HZXcHUUPUnjoZnwgGjF9i_m-HXMiIS4qk=w2400"/>

Now, this is, of course, just a demonstration of how to manually configure each RGB curve. The Louvre::LGammaTable also provides an auxiliary method (Louvre::LGammaTable::fill()) that can set the table values for you, accepting parameters such as gamma, contrast, and brightness.

### Wlr Gamma Control

Certain Wayland clients like [Redshift](http://jonls.dk/redshift/) or [GammaStep](https://gitlab.com/chinstrap/gammastep) can request to set the gamma table through the [Wlr Gamma Control](https://wayland.app/protocols/wlr-gamma-control-unstable-v1) protocol. These clients are commonly employed to automatically adjust the "temperature" of screens, applying a cooler tone during the day and a warmer tone during the night.

For security reasons, this feature is disabled by default. You should only permit well-known clients to modify the gamma tables, and the mechanism to identify them is left to your discretion.
Let's be a bit trusting for now and allow any client to assign the gamma tables. To do this, first, remove the code we just added and override the Louvre::LOutput::setGammaRequest() virtual method.

#### src/EOutput.h

```cpp
// ...

class EOutput : public LOutput
{
public:

    // ...

    void setGammaRequest(LClient *client, const LGammaTable *gamma) override;

    // ...
;
};


// ...
```

#### src/EOutput.cpp

```cpp
// ...

void EOutput::setGammaRequest(LClient *client, const LGammaTable *gamma)
{
    /* Note: This is quite unsafe, you should only permit well-known
     * clients to set gamma correction curves. */
    L_UNUSED(client)
    
    setGamma(gamma);
}

// ...
```

This method is invoked each time a client requests to set the gamma table for this particular output. If the gamma parameter is set to `nullptr`, it indicates that the client no longer wishes to control gamma or has been disconnected. It's important to note that passing `nullptr` to Louvre::LOutput::setGamma() restores the default linear gamma curves.

Now, explore launching applications such as Redshift or GammaStep to observe their effects. Pretty neat, isn't it?

In the next chapter, we will delve into the intricacies of fractional scaling and discuss techniques to achieve optimal results and performance while using it. Chao chao!

<a href="15.md">◀ Chapter 15: Output Hotplugging and Seat</a> || <a href="17.md"> Chapter 17: Fractional Scaling ▶</a>