# Chapter 17: Fractional Scaling {#tutorial-scaling}

We briefly touched upon it in [Chapter 4: Compositor Initialization](04.md) that, starting from version 1.2.0, Louvre::LOutput::setScale() now supports fractional scales. While this doesn't alter how rendering is done, it is handled differently internally by Louvre. To comprehend its functioning, please read @ref Scaling "this section" of the Louvre::LOutput documentation before proceeding.

Assuming that you've gone through it, you should now be aware that fractional scaling may result in unsightly aliasing artifacts, particularly noticeable on low DPI displays, especially on rendered fonts. This is why Louvre enables oversampling by default when using fractional scales, which significantly reduces these artifacts but may impact performance. For instance, here is a rather drastic example with two captures of a font being rendered, one with oversampling (at the top) and the other without (at the bottom).

<center>
<img style="width:65%" src="https://lh3.googleusercontent.com/pw/ABLVV85j3HjCdqdHsTczA6sLH76wouruXqYlORoRbeOL0qNmPyoOoiinwWctALJgxnBoLZeaRfEVdWpJOKTSBBPZklM_N1vDt-i0HYnL2LedBfABVh4EARY=w2400"/>
</center>

You may observe that even with the drastic scaling factor, the one using oversampling remains quite readable and appears smoother, whereas the other appears somewhat unintelligible, resembling something only understandable to Russians.

## Performance

Instead of directly using fractional scales, it is advisable to first check if your outputs support different native modes (Louvre::LOutput::modes()) with the desired resolution and use integer scale factors whenever possible, as oversampling is always disabled when using integer scales.

If that is not the case, when using fractional scales and oversampling, you can still achieve good performance if you are using Louvre::LScene for rendering or manually specifying the damage generated during the rendering of a frame with Louvre::LOutput::setBufferDamage() like the *louvre-weston-clone* example does. However, in certain scenarios, such as displaying a video or a game in fullscreen mode, this approach might impact performance, as the entire screen is rendered twice—first into the larger "oversampling" framebuffer and then scaled down to the actual screen framebuffer. In such cases, you could disable oversampling with Louvre::LOutput::enableFractionalOversampling(). Let's implement that.

> Note: Louvre::LScene automatically damages the entire screen when a fractional scale is used by an output and oversampling is toggled. This is necessary to maintain synchronization in its inter-frame damage tracking.

First, let's add some keyboard shortcuts to increase and decrease the scaling factor by 0.25 and toggle oversampling.

#### src/EKeyboard.cpp

```cpp
// ...

#include <LOutput.h>
#include "EKeyboard.h"

// ...

void EKeyboard::keyEvent(UInt32 keyCode, KeyState keyState)
{
    if (keyState == Released)
    {
        const bool L_META       { isKeyCodePressed(KEY_LEFTMETA) };
        const bool L_SHIFT      { isKeyCodePressed(KEY_LEFTSHIFT) };
        LOutput *cursorOutput   { cursor()->output() };

        if (L_META && L_SHIFT && cursorOutput)
        {
            if (keyCode == KEY_UP && cursorOutput->fractionalScale() < 3.f)
                cursorOutput->setScale(cursorOutput->fractionalScale() + 0.25f);
            else if (keyCode == KEY_DOWN && cursorOutput->fractionalScale() > 0.25f)
                cursorOutput->setScale(cursorOutput->fractionalScale() - 0.25f);
            else if (keyCode == KEY_O)
                cursorOutput->enableFractionalOversampling(!cursorOutput->fractionalOversamplingEnabled());
        }
    }

    G::scene()->handleKeyEvent(keyCode, keyState);
}

// ...
```

Now, by pressing `Super + Shift + [Up/Down Arrow]`, the output where the cursor is currently located will increase or decrease its scaling factor, and `Super + Shift + O` will allow you to toggle oversampling on and off.

Experiment a bit with different scaling factors and toggling oversampling to observe any noticeable differences.

Instead of toggling oversampling manually, let's remove the shortcut for that and add a new method on `EOutput` that returns the first available fullscreen toplevel on the given output or `nullptr` otherwise.

#### src/EOutput.h

```cpp
// ...

class EToplevel;

class EOutput : public LOutput
{
public:

    // ...

    EToplevel *findFullscreenToplevel() const;

    // ...
};

// ...
```

#### src/EOutput.cpp

```cpp
// ...


void EOutput::paintGL()
{
    // ...

    EToplevel *fullscreenToplevel { findFullscreenToplevel() };
    enableFractionalOversampling( fullscreenToplevel == nullptr );
    G::scene()->handlePaintGL(this);
}

EToplevel *EOutput::findFullscreenToplevel() const
{
    for (ESurface *surface : G::surfaces())
        if (surface->tl() && surface->tl()->fullscreen() && surface->tl()->output == this)
            return surface->tl();

    return nullptr;
}

// ...
```

So, now, each time a toplevel is in fullscreen mode on an output, oversampling will be disabled and then re-enabled when there are no fullscreen toplevels. 

> Just to reiterate, when using integer scales, oversampling is always disabled, even if you call `enableFractionalOversampling(true)`.

In the next chapter, we will explore how to toggle VSync so that our gaming users' eyes can experience a bit of bleeding if they desire it.

<a href="16.md">◀ Chapter 16: Gamma Correction</a> || <a href="18.md"> Chapter 18: Tearing Control ▶</a>