# Chapter 18: Tearing Control {#tutorial-tearing}

Louvre defaults to using double buffering when rendering. This implies that each output utilizes two framebuffers: one displayed on the screen, and another where the subsequent frame is rendered. These buffers are then swapped, with the last rendered framebuffer now being displayed on the screen, and the other used for rendering, and so forth.

## VSync

When VSync is enabled, the framebuffer swapping occurs during a specific hardware event known as a vblank event. A vblank, or vertical blank, typically occurs after a fixed period determined by the refresh rate of the current output mode. It happens when the hardware responsible for displaying your framebuffers on a display starts scanning out the current framebuffer and displaying it on the screen.

## Tearing

If you swap the framebuffers at any other time, you may notice that half or part of the screen displays the current framebuffer, while the other displays the previous one. This artifact is known as tearing, demonstrated in the following image:

<img style="width:100%" src="https://lh3.googleusercontent.com/2ousoWwxnVGvFX5bT6ual2G8UUbhUOJ21mK1UQmthPNM-7XfracRlL5GCYBQTzt4Os28eKO_FzC6BS-rasiNngvTMI9lEdET0ItKrI2wK_9IwSDaF-hNGkTMI6gVlL0m4ENDJYbckw">

## Toggling VSync

In a typical scenario, you'd likely want VSync to be enabled. However, some gaming enthusiasts may prefer to disable it to achieve a faster screen response time and gain a few more FPS.

Once again, starting from version 1.2.0, Louvre supports toggling VSync on outputs. This can be accomplished through Louvre::LOutput::enableVSync(). Disabling it is not universally supported and depends on your GPU hardware and drivers. To check if an output supports it, you can use Louvre::LOutput::hasVSyncControlSupport(). In cases where it is unsupported, VSync is always enabled.

## Refresh Rate Limit

When you disable VSync on a specific output, Louvre defaults to limiting its refresh rate to twice the native refresh rate determined by its current mode. This precaution is taken because disabling the limit could unnecessarily consume processing power and even diminish the overall performance of your system.

To set a custom refresh rate limit when VSync is disabled, you can use Louvre::LOutput::setRefreshRateLimit(). If you pass a value of -1, the limit is disabled. If you pass a value of 0, the limit is set to double the actual refresh rate of your output. Any other value represents the desired limit in Hz. 

## Client Control

Clients using the [Tearing Control](https://wayland.app/protocols/tearing-control-v1) protocol can communicate their preference for enabling or disabling VSync for a particular surface to the compositor. To check if a surface prefers VSync, you can use Louvre::LSurface::preferVSync() and be notified when this property changes through the Louvre::LSurface::preferVSyncChanged() virtual method. For clients not using this protocol, Louvre::LSurface::preferVSync() always returns `true`.

Now, let's take a somewhat lazy approach and reuse our previous `findFullscreenToplevel()` in Louvre::LOutput::paintGL() to toggle VSync based on the preference of the first fullscreen toplevel surface found on a given output.

#### src/EOutput.cpp

```cpp
// ...

void EOutput::paintGL()
{
    // ...

    EToplevel *fullscreenToplevel { findFullscreenToplevel() };
    enableFractionalOversampling( fullscreenToplevel == nullptr);
    enableVSync(fullscreenToplevel == nullptr ? true : fullscreenToplevel->surf()->preferVSync());
    G::scene()->handlePaintGL(this);
}

// ...
```

To test if this is working, try launching `$ weston-simple-egl -t` (the `-t` argument makes the client prefer tearing), and then press `F11` to turn it into fullscreen mode. If you pay attention, you should be able to observe the tearing effect, and the logging messages from *weston-simple-egl* should indicate a refresh rate approximately double that of your screen's current mode. If not, your monitor likely lacks VSync control support.

<img src="https://lh3.googleusercontent.com/pw/ABLVV84Qz7gTHDzk8h2jh_b0oHRnXi2ydfs2WzvAlCfhUH_SWWho6wCdw9_DmaXSKWmbvXl3283yfMPHU_kCCClkJoAMrRwn6c1nBVBoYOzNT1bT1bMxr7A=w2400"/>

Also, try experimenting with different values using Louvre::LOutput::setRefreshRateLimit(). For instance, using `-1` to disable the limit, but keep in mind that this might be harsh on your eyes.

### Congratulations!

We've covered the most crucial aspects of building a compositor with Louvre, and you may already be prepared to let your creativity soar.

If you encounter any bugs, have suggestions, questions, want to share your creations or contribute to the development of Louvre, please don't hesitate to get in touch. You can find our contact information on the [Contact](md_md__contact.html) page.


<a href="17.md">◀ Chapter 17: Fractional Scaling</a>