{
  lib,
  stdenv,
  stdenvAdapters,
  pkg-config,
  pkgconf,
  makeWrapper,
  cmake,
  aquamarine,
  binutils,
  cairo,
  epoll-shim,
  git,
  glaze-hyprland,
  gtest,
  hyprcursor,
  hyprgraphics,
  hyprland-protocols,
  hyprland-guiutils,
  hyprlang,
  hyprutils,
  hyprwayland-scanner,
  hyprwire,
  libGL,
  libdrm,
  libexecinfo,
  libinput,
  libxkbcommon,
  libuuid,
  libgbm,
  muparser,
  pango,
  pciutils,
  re2,
  systemd,
  tomlplusplus,
  udis86-hyprland,
  wayland,
  wayland-protocols,
  wayland-scanner,
  xorg,
  xwayland,
  debug ? false,
  withTests ? false,
  enableXWayland ? true,
  withSystemd ? lib.meta.availableOn stdenv.hostPlatform systemd,
  wrapRuntimeDeps ? true,
  version ? "git",
  commit,
  revCount,
  date,
  # deprecated flags
  enableNvidiaPatches ? false,
  nvidiaPatches ? false,
  hidpiXWayland ? false,
  legacyRenderer ? false,
  withHyprtester ? false,
}: let
  inherit (builtins) foldl' readFile;
  inherit (lib.asserts) assertMsg;
  inherit (lib.attrsets) mapAttrsToList;
  inherit (lib.lists) flatten concatLists optional optionals;
  inherit (lib.strings) makeBinPath optionalString cmakeBool trim;
  fs = lib.fileset;

  adapters = flatten [
    stdenvAdapters.useMoldLinker
    (lib.optional debug stdenvAdapters.keepDebugInfo)
  ];

  customStdenv = foldl' (acc: adapter: adapter acc) stdenv adapters;
in
  assert assertMsg (!nvidiaPatches) "The option `nvidiaPatches` has been removed.";
  assert assertMsg (!enableNvidiaPatches) "The option `enableNvidiaPatches` has been removed.";
  assert assertMsg (!hidpiXWayland) "The option `hidpiXWayland` has been removed. Please refer https://wiki.hypr.land/Configuring/XWayland";
  assert assertMsg (!legacyRenderer) "The option `legacyRenderer` has been removed. Legacy renderer is no longer supported.";
  assert assertMsg (!withHyprtester) "The option `withHyprtester` has been removed. Hyprtester is always built now.";
    customStdenv.mkDerivation (finalAttrs: {
      pname = "hyprland${optionalString debug "-debug"}";
      inherit version withTests;

      src = fs.toSource {
        root = ../.;
        fileset =
          fs.intersection
          # allows non-flake builds to only include files tracked by git
          (fs.gitTracked ../.)
          (fs.unions (flatten [
            ../assets/hyprland-portals.conf
            ../assets/install
            ../hyprctl
            ../hyprland.pc.in
            ../LICENSE
            ../protocols
            ../src
            ../start
            ../systemd
            ../VERSION
            (fs.fileFilter (file: file.hasExt "1") ../docs)
            (fs.fileFilter (file: file.hasExt "conf" || file.hasExt "in") ../example)
            (fs.fileFilter (file: file.hasExt "sh") ../scripts)
            (fs.fileFilter (file: file.name == "CMakeLists.txt") ../.)
            (optional withTests [../tests ../hyprtester])
          ]));
      };

      postPatch = ''
        # Fix hardcoded paths to /usr installation
        sed -i "s#/usr#$out#" src/render/OpenGL.cpp

        # Remove extra @PREFIX@ to fix some paths
        sed -i "s#@PREFIX@/##g" hyprland.pc.in
        sed -i "s#@PREFIX@/##g" example/hyprland.desktop.in
      '';

      env = {
        GIT_COMMITS = revCount;
        GIT_COMMIT_DATE = date;
        GIT_COMMIT_HASH = commit;
        GIT_DIRTY = if (commit == "") then "clean" else "dirty";
        GIT_TAG = "v${trim (readFile "${finalAttrs.src}/VERSION")}";
      };

      depsBuildBuild = [
        pkg-config
      ];

      nativeBuildInputs = [
        hyprwayland-scanner
        hyprwire
        makeWrapper
        cmake
        pkg-config
      ];

      outputs = [
        "out"
        "man"
        "dev"
      ];

      buildInputs = concatLists [
        [
          aquamarine
          cairo
          git
          glaze-hyprland
          gtest
          hyprcursor
          hyprgraphics
          hyprland-protocols
          hyprlang
          hyprutils
          hyprwire
          libdrm
          libGL
          libinput
          libuuid
          libxkbcommon
          libgbm
          muparser
          pango
          pciutils
          re2
          tomlplusplus
          udis86-hyprland
          wayland
          wayland-protocols
          wayland-scanner
          xorg.libXcursor
        ]
        (optionals customStdenv.hostPlatform.isBSD [epoll-shim])
        (optionals customStdenv.hostPlatform.isMusl [libexecinfo])
        (optionals enableXWayland [
          xorg.libxcb
          xorg.libXdmcp
          xorg.xcbutilerrors
          xorg.xcbutilrenderutil
          xorg.xcbutilwm
          xwayland
        ])
        (optional withSystemd systemd)
      ];

      strictDeps = true;

      cmakeBuildType =
        if debug
        then "Debug"
        else "RelWithDebInfo";

      # we want as much debug info as possible
      dontStrip = debug;

      cmakeFlags = mapAttrsToList cmakeBool {
        "BUILT_WITH_NIX" = true;
        "NO_XWAYLAND" = !enableXWayland;
        "LEGACY_RENDERER" = legacyRenderer;
        "NO_SYSTEMD" = !withSystemd;
        "CMAKE_DISABLE_PRECOMPILE_HEADERS" = true;
        "NO_UWSM" = !withSystemd;
        "NO_HYPRPM" = true;
        "TRACY_ENABLE" = false;
        "WITH_TESTS" = withTests;
      };

      preConfigure = ''
        substituteInPlace hyprtester/CMakeLists.txt --replace-fail \
          "\''${CMAKE_CURRENT_BINARY_DIR}" \
          "${placeholder "out"}/bin"
      '';

      postInstall = ''
        ${optionalString wrapRuntimeDeps ''
          wrapProgram $out/bin/Hyprland \
            --suffix PATH : ${makeBinPath [
            binutils
            hyprland-guiutils
            pciutils
            pkgconf
          ]}
        ''}

        ${optionalString withTests ''
          install hyprtester/pointer-warp -t $out/bin
          install hyprtester/pointer-scroll -t $out/bin
          install hyprland_gtests -t $out/bin
          install hyprtester/child-window -t $out/bin
        ''}
      '';

      passthru.providedSessions = ["hyprland"];

      meta = {
        homepage = "https://github.com/hyprwm/Hyprland";
        description = "Dynamic tiling Wayland compositor that doesn't sacrifice on its looks";
        license = lib.licenses.bsd3;
        platforms = lib.platforms.linux;
        mainProgram = "Hyprland";
      };
    })
