!! Copyright (C) 2002-2011 M. Marques, A. Castro, A. Rubio, G. Bertsch, M. Oliveira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

! ---------------------------------------------------------
!
!> Reads a mesh function from file filename, and puts it into ff. If
!! the map argument is passed, the subroutine will reorder the values
!! in the file according to it, missing values will be filled with
!! zeros. (For the moment this is only implemented for the obf format.)
!!
!! On output, ierr signals how everything went:
!! ierr > 0 => Error. The function ff was not read: \n
!!              1 : illegal filename (must have ".obf" or ".ncdf" extension). \n
!!              2 : file could not be successfully opened. \n
!!              3 : file opened, but error reading. \n
!!              4 : The number of points/mesh dimensions do not coincide. \n
!!              5 : Format or NetCDF error (one or several warnings are written) \n
!! ierr = 0 => Success. \n
!! ierr < 0 => Success, but some kind of type conversion was necessary. The value
!!             of ierr is then: \n
!!             -1 : function in file is real, sp. \n
!!             -2 : function in file is complex, sp. \n
!!             -3 : function in file is real, dp. \n
!!             -4 : function in file is complex, dp. \n
! ---------------------------------------------------------

subroutine X(io_function_input)(filename, namespace, space, mesh, ff, ierr, map)
  character(len=*),      intent(in)    :: filename
  type(namespace_t),     intent(in)    :: namespace
  class(space_t),        intent(in)    :: space
  type(mesh_t),          intent(in)    :: mesh
  R_TYPE,                intent(inout) :: ff(:)
  integer,               intent(out)   :: ierr
  integer(int64), optional, intent(in) :: map(:)

  R_TYPE, allocatable :: ff_global(:)

  PUSH_SUB_WITH_PROFILE(X(io_function_input))

  ASSERT(ubound(ff, dim = 1) == mesh%np .or. ubound(ff, dim = 1) == mesh%np_part)


  select case(trim(io_get_extension(filename)))
  case ('obf')
    if(mesh%parallel_in_domains) then
      SAFE_ALLOCATE(ff_global(1:1))

      if (mpi_grp_is_root(mesh%mpi_grp)) then
        SAFE_DEALLOCATE_A(ff_global)
        SAFE_ALLOCATE(ff_global(1:mesh%np_global))
        call X(io_function_input_obf)(filename, namespace, space, mesh, ff_global, ierr, map)
      end if

      ! Only root knows if the file was successfully read.
      ! Now, it tells everybody else
      call mesh%mpi_grp%bcast(ierr, 1, MPI_INTEGER, 0)

      ! Only scatter, when successfully read the file(s).
      if (ierr <= 0) then
        call par_vec_scatter(mesh%pv, 0, ff, ff_global)
      end if

      SAFE_DEALLOCATE_A(ff_global)
    else
      call X(io_function_input_obf)(filename, namespace, space, mesh, ff, ierr, map)
    end if

  case ('ncdf')
#if defined(HAVE_NETCDF)
    call X(io_function_input_ncdf)(filename, namespace, space, mesh, ff, ierr, map)
#else
    ierr = 1
    message(1) = "Octopus was compiled without NetCDF support."
    call messages_fatal(1, namespace=namespace)
#endif
  case ('csv')
    call X(io_function_input_csv)(filename, namespace, space, mesh, ff, ierr)
  case default
    ierr = 1
    message(1) = "Illegal filename (must have '.csv', '.obf' or '.ncdf' extension)."
    call messages_warning(1, namespace=namespace)
  end select

  POP_SUB_WITH_PROFILE(X(io_function_input))

end subroutine X(io_function_input)


#if defined(HAVE_NETCDF)

!> @brief Reads a mesh function from a NetCDF file filename, and puts it into ff.
!
subroutine X(io_function_input_ncdf)(filename, namespace, space, mesh, ff, ierr, map)
  character(len=*),      intent(in)    :: filename  !< Name of the file to read
  type(namespace_t),     intent(in)    :: namespace !< Namespace
  class(space_t),        intent(in)    :: space     !< Space definition
  type(mesh_t),          intent(in)    :: mesh      !< The mesh
  R_TYPE,                intent(inout) :: ff(:)     !< The function to read
  integer,               intent(out)   :: ierr      !< Error code
  integer(int64), optional, intent(in) :: map(:)    !< Optional: Map to reorder the function

  type(cube_t) :: cube
  type(cube_function_t) :: cf

  character(len=512) :: file
  integer :: ncid, status
  integer :: function_kind
#if defined(R_TCOMPLEX)
  type(cube_function_t) :: re, im
#endif

  PUSH_SUB(X(io_function_input_ncdf))

  function_kind = X(output_kind)*kind(ff(1)) ! +4 for real, single; +8 for real, double;
  ! -4 for complex, single, -8 for real, double

  ASSERT(.not. present(map))
  file = io_workpath(filename, namespace)
  status = nf90_open(trim(file), NF90_WRITE, ncid)
  if (status /= NF90_NOERR) then
    ierr = 2
  else
    call cube_init(cube, mesh%idx%ll, namespace, space, mesh%spacing, mesh%coord_system)
    call cube_init_cube_map(cube, mesh)
    call X(cube_function_alloc_RS)(cube, cf)
#if defined(R_TCOMPLEX)
    call dcube_function_alloc_RS(cube, re)
    call dcube_function_alloc_RS(cube, im)
    call read_netcdf(namespace)
    cf%zRS = re%dRS + M_zI*im%dRS
    call X(cube_to_mesh) (cube, cf, mesh, ff)
    call dcube_function_free_RS(cube, re)
    call dcube_function_free_RS(cube, im)
#else
    call read_netcdf(namespace)
    call X(cube_to_mesh)(cube, cf, mesh, ff)
#endif
    call X(cube_function_free_RS)(cube, cf)
    call cube_end(cube)
  end if

  POP_SUB(X(io_function_input_ncdf))

contains

  ! ---------------------------------------------------------
  subroutine read_netcdf(namespace)
    type(namespace_t), intent(in)    :: namespace

    integer :: data_id, data_im_id, dim_data_id(3), ndim(3), xtype, file_kind
    real(real64), allocatable :: xx(:, :, :)

    PUSH_SUB(X(io_function_input_global).read_netcdf)

    !Inquire about dimensions
    if (status == NF90_NOERR) then
      status = nf90_inq_dimid (ncid, "dim_1", dim_data_id(1))
      call ncdf_error('nf90_inq_dimid', status, file, namespace, ierr)
    end if

    if (status == NF90_NOERR) then
      status = nf90_inq_dimid (ncid, "dim_2", dim_data_id(2))
      call ncdf_error('nf90_inq_dimid', status, file, namespace, ierr)
    end if

    if (status == NF90_NOERR) then
      status = nf90_inq_dimid (ncid, "dim_3", dim_data_id(3))
      call ncdf_error('nf90_inq_dimid', status, file, namespace, ierr)
    end if

    if (status == NF90_NOERR) then
      status = nf90_inquire_dimension (ncid, dim_data_id(1), len = ndim(3))
      call ncdf_error('nf90_inquire_dimension', status, file, namespace, ierr)
    end if
    if (status == NF90_NOERR) then
      status = nf90_inquire_dimension (ncid, dim_data_id(2), len = ndim(2))
      call ncdf_error('nf90_inquire_dimension', status, file, namespace, ierr)
    end if
    if (status == NF90_NOERR) then
      status = nf90_inquire_dimension (ncid, dim_data_id(3), len = ndim(1))
      call ncdf_error('nf90_inquire_dimension', status, file, namespace, ierr)
    end if
    if ((ndim(1) /= cube%rs_n_global(1)) .or. &
      (ndim(2) /= cube%rs_n_global(2)) .or. &
      (ndim(3) /= cube%rs_n_global(3))) then
      ierr = 12
      POP_SUB(X(io_function_input_global).read_netcdf)
      return
    end if

    if (status == NF90_NOERR) then
      status = nf90_inq_varid (ncid, "rdata", data_id)
      call ncdf_error('nf90_inq_varid', status, file, namespace, ierr)
    end if
    status = nf90_inq_varid(ncid, "idata", data_im_id)
    if (status == 0) then
      file_kind = -1
    else
      file_kind = 1
    end if
    status = 0

    if (status == NF90_NOERR) then
      status = nf90_inquire_variable (ncid, data_id, xtype = xtype)
      call ncdf_error('nf90_inquire_variable', status, file, namespace, ierr)
    end if

    if (xtype == NF90_FLOAT) then
      file_kind = file_kind*4
    else
      file_kind = file_kind*8
    end if
    if (file_kind /= function_kind) then
      select case (file_kind)
      case (4)
        ierr = -1
      case (-4)
        ierr = -2
      case (8)
        ierr = -3
      case (-8)
        ierr = -4
      end select
    end if

    SAFE_ALLOCATE(xx(1:cube%rs_n_global(3), 1:cube%rs_n_global(2), 1:cube%rs_n_global(1)))
#if defined(R_TCOMPLEX)
    if (status == NF90_NOERR) then
      select case (space%dim)
      case (1)
        status = nf90_get_var (ncid, data_id, xx(1, 1, :))
      case (2)
        status = nf90_get_var (ncid, data_id, xx(1, :, :))
      case (3)
        status = nf90_get_var (ncid, data_id, xx)
      end select
      call transpose3(xx, re%dRS)
      call ncdf_error('nf90_get_var', status, file, namespace, ierr)
    end if
    if (file_kind<0) then
      if (status == NF90_NOERR) then
        select case (space%dim)
        case (1)
          status = nf90_get_var (ncid, data_im_id, xx(1, 1, :))
        case (2)
          status = nf90_get_var (ncid, data_im_id, xx(1, :, :))
        case (3)
          status = nf90_get_var (ncid, data_im_id, xx)
        end select
        call transpose3(xx, im%dRS)
        call ncdf_error('nf90_get_var', status, file, namespace, ierr)
      end if
    else
      im%dRS = M_ZERO
    end if
#else
    if (status == NF90_NOERR) then
      select case (space%dim)
      case (1)
        status = nf90_get_var (ncid, data_id, xx(1, 1, :))
      case (2)
        status = nf90_get_var (ncid, data_id, xx(1, :, :))
      case (3)
        status = nf90_get_var (ncid, data_id, xx)
      end select
      call transpose3(xx, cf%dRS)
      call ncdf_error('nf90_get_var', status, file, namespace, ierr)
    end if
#endif
    SAFE_DEALLOCATE_A(xx)

    status = nf90_close(ncid)
    POP_SUB(X(io_function_input_global).read_netcdf)
  end subroutine read_netcdf

end subroutine X(io_function_input_ncdf)
#endif

! ---------------------------------------------------------
!> @brief Reads a mesh function from a obf file filename, and puts it into ff.
!
subroutine X(io_function_input_obf)(filename, namespace, space, mesh, ff, ierr, map)
  character(len=*),      intent(in)    :: filename    !< Name of the file to read
  type(namespace_t),     intent(in)    :: namespace   !< Namespace
  class(space_t),        intent(in)    :: space       !< Space definition
  type(mesh_t),          intent(in)    :: mesh        !< The mesh
  R_TYPE,                intent(inout) :: ff(:)       !< The function to read
  integer,               intent(out)   :: ierr        !< Error code
  integer(int64), optional, intent(in) :: map(:)      !< Optional: Map to reorder the function

  integer(int64) :: np, ip, file_size
  R_TYPE,       allocatable :: read_ff(:)

  PUSH_SUB(X(io_function_input_obf))

  if (present(map)) then

    call io_binary_get_info(io_workpath(filename, namespace), np, file_size, ierr)

    if (ierr == 0) then
      SAFE_ALLOCATE(read_ff(1:np))

      call io_binary_read(io_workpath(filename, namespace), np, read_ff, ierr)
      call profiling_count_transfers(np, read_ff(1))

      if (ierr == 0) then
        ff(1:mesh%np_global) = M_ZERO
        do ip = 1, min(np, ubound(map, dim=1, kind=int64))
          if (map(ip) > 0) ff(map(ip)) = read_ff(ip)
        end do
      end if

      SAFE_DEALLOCATE_A(read_ff)
    end if

  else
    call io_binary_read(io_workpath(filename, namespace), mesh%np_global, ff, ierr)
    call profiling_count_transfers(mesh%np_global, ff(1))
  end if

  POP_SUB(X(io_function_input_obf))

end subroutine X(io_function_input_obf)

!> @brief Reads a mesh function from a csv file filename, and puts it into ff.
!!
!! Note that this function can only read real-valued CSV files. If the function
!! is complex, the real part will be read and the imaginary part will be set to zero.
!!
!! For parallel calculations, using domain decomposition, the whole function is read
!! on all nodes, but only the part corresponding to the local domain is copied to ff.
!
subroutine X(io_function_input_csv)(filename, namespace, space, mesh, ff, ierr)
  character(len=*),      intent(in)    :: filename   !< Name of the file to read
  type(namespace_t),     intent(in)    :: namespace  !< Namespace
  class(space_t),        intent(in)    :: space      !< Space definition
  type(mesh_t),          intent(in)    :: mesh       !< The mesh
  R_TYPE,                intent(inout) :: ff(:)      !< The function to read
  integer,               intent(out)   :: ierr       !< Error code

  integer :: ii, jj, kk
  integer(int64) :: dims(3)
  real(real64), allocatable :: x_in(:, :)
  real(real64), allocatable :: x_out(:, :)
  type(cube_t) :: cube
  type(cube_function_t) :: cf

  real(real64), allocatable :: read_ff(:) !< The data read from the file
  real(real64), allocatable :: real_ff(:) !< The interpolated data

  PUSH_SUB(X(io_function_input_csv))

  ierr = -3 ! Error code for real valued csv files.

  ! Fo csv input we can only read real valued files, and copy later to the complex function, if needed.

  select type (box => mesh%box)
  type is (box_parallelepiped_t)
  class default
    message(1) = "Box shape must be parallelepiped when a .csv file is used."
    call messages_fatal(1, namespace=namespace)
  end select

  call cube_init(cube, mesh%idx%ll, namespace, space, mesh%spacing, mesh%coord_system)
  call cube_init_cube_map(cube, mesh)
  call dcube_function_alloc_RS(cube, cf)

  call io_csv_get_info(io_workpath(filename, namespace), dims, ierr)

  if (ierr /= 0) then
    write(message(1), '("Could not read file ",A," (error code",I2,")")') trim(io_workpath(filename, namespace)), ierr
    call messages_fatal(1, namespace=namespace)
  end if

  SAFE_ALLOCATE(read_ff(1:dims(1)*dims(2)*dims(3)))
  SAFE_ALLOCATE(real_ff(1:dims(1)*dims(2)*dims(3)))
  call dread_csv(io_workpath(filename, namespace), dims(1)*dims(2)*dims(3), read_ff, ierr)

#ifdef T_COMPLEX
  message(1) = "Reading real-valued csv file into complex valued wavefunction."
  call messages_info(1, namespace=namespace)
#endif

  if (ierr /= 0) then
    message(1) = "Could not read file "//trim(io_workpath(filename, namespace))//""
    call messages_fatal(1, namespace=namespace)
  end if

  select case(space%dim)

  case (1)

    SAFE_ALLOCATE(x_in(1:dims(1), 1))
    SAFE_ALLOCATE(x_out(1:cube%rs_n_global(1), 1))

    do ii = 1, int(dims(1))
      x_in(ii,:) = (/ real(ii-1, real64) *(real(cube%rs_n_global(1)-1, real64)/(dims(1)-1)) /)
    end do

    do ii = 1, cube%rs_n_global(1)
      x_out(ii,1) = real(ii-1, real64)
    end do

    call dmf_interpolate_points(1, int(dims(1)), x_in(:,:),&
      read_ff, cube%rs_n_global(1), x_out(:,:), real_ff)

    SAFE_DEALLOCATE_A(x_in)
    SAFE_DEALLOCATE_A(x_out)

  case (2)
    SAFE_ALLOCATE(x_in(1:(dims(1)*dims(2)), 1:2))
    SAFE_ALLOCATE(x_out(1:(cube%rs_n_global(1)*cube%rs_n_global(2)), 1:2))

    do ii = 1, int(dims(2))
      do jj = 1, int(dims(1))
        x_in((ii-1)*dims(1) + jj,:) = &
          (/ real(jj-1, real64) *(real(cube%rs_n_global(1)-1, real64)/(dims(1)-1)),&
          real(ii-1, real64) *(real(cube%rs_n_global(2)-1, real64)/(dims(2)-1)) /)
      end do
    end do

    do ii = 1, cube%rs_n_global(2)
      do jj = 1, cube%rs_n_global(1)
        x_out((ii-1)*cube%rs_n_global(1) + jj,:) = (/ real(jj-1, real64) , real(ii-1, real64)  /)
      end do
    end do
    call dmf_interpolate_points(2, int(dims(1)*dims(2)), x_in(:,:),&
      read_ff, cube%rs_n_global(1)*cube%rs_n_global(2), x_out(:,:), real_ff)

    SAFE_DEALLOCATE_A(x_in)
    SAFE_DEALLOCATE_A(x_out)

  case (3)

    SAFE_ALLOCATE(x_in(1:(dims(1)*dims(2)*dims(3)), 1:3))
    SAFE_ALLOCATE(x_out(1:(cube%rs_n_global(1)*cube%rs_n_global(2)*cube%rs_n_global(3)), 1:3))

    do ii = 1, int(dims(3))
      do jj = 1, int(dims(2))
        do kk = 1, int(dims(1))
          x_in((ii-1)*dims(1)*dims(2) + (jj-1)*dims(1) + kk,:) = &
            (/ real(kk-1, real64) *(real(cube%rs_n_global(1)-1, real64)/(dims(1)-1)) , &
            real(jj-1, real64) *(real(cube%rs_n_global(2)-1, real64)/(dims(2)-1)) , &
            real(ii-1, real64) *(real(cube%rs_n_global(3)-1, real64)/(dims(3)-1)) /)
        end do
      end do
    end do

    do ii = 1, cube%rs_n_global(3)
      do jj = 1, cube%rs_n_global(2)
        do kk = 1, cube%rs_n_global(1)
          x_out((ii-1)*cube%rs_n_global(1)*cube%rs_n_global(2) + (jj-1)*cube%rs_n_global(1) + kk,:) = &
            (/ real(kk-1, real64) , real(jj-1, real64) , real(ii-1, real64)  /)
        end do
      end do
    end do
    call dmf_interpolate_points(3, int(dims(1)*dims(2)*dims(3)), x_in(:,:),&
      read_ff, cube%rs_n_global(1)*cube%rs_n_global(2)*cube%rs_n_global(3), x_out(:,:), real_ff)

    SAFE_DEALLOCATE_A(x_in)
    SAFE_DEALLOCATE_A(x_out)

  end select


  cf%dRS = reshape(real_ff, (/ cube%rs_n_global(1), cube%rs_n_global(2), cube%rs_n_global(3) /))

  !Skip the reallocation to a smaller array if the mesh is not parallel as mesh%np < mesh%np_global.

  call dcube_to_mesh(cube, cf, mesh, real_ff(1:mesh%np))
  call dcube_function_free_RS(cube, cf)
  call cube_end(cube)

  ff = real_ff(1:mesh%np)

  SAFE_DEALLOCATE_A(real_ff)

  POP_SUB(X(io_function_input_csv))

end subroutine X(io_function_input_csv)


! ---------------------------------------------------------
subroutine X(io_function_output_vector)(how, dir, fname, namespace, space, mesh, ff, unit, ierr, &
  pos, atoms, grp, root)
  integer(int64),                intent(in)  :: how
  character(len=*),           intent(in)  :: dir
  character(len=*),           intent(in)  :: fname
  type(namespace_t),          intent(in)  :: namespace
  class(space_t),             intent(in)  :: space
  class(mesh_t),              intent(in)  :: mesh
  R_TYPE, contiguous, target, intent(in)  :: ff(:, :)
  type(unit_t),               intent(in)  :: unit
  integer,                    intent(out) :: ierr
  real(real64),     optional, intent(in)  :: pos(:,:)
  type(atom_t),     optional, intent(in)  :: atoms(:)
  type(mpi_grp_t),  optional, intent(in)  :: grp !< the group that shares the same data, must contain the domains group
  integer,          optional, intent(in)  :: root !< which process is going to write the data

  integer :: ivd
  integer(int64) :: how_seq
  character(len=MAX_PATH_LEN) :: full_fname
  type(cube_t) :: cube
  type(cube_function_t), allocatable :: cf(:)
  character(len=MAX_PATH_LEN) :: filename
  real(real64) :: dk(3)
  integer :: root_
  logical :: i_am_root

  PUSH_SUB(X(io_function_output_vector))

  ASSERT(space%dim < 10)

  how_seq = bitand(how, not(OPTION__OUTPUTFORMAT__VTK)) ! first do everything except VTK

  ! everything except VTK
  do ivd = 1, space%dim
    full_fname = trim(fname)//'-'//index2axis(ivd)
    call X(io_function_output)(how_seq, dir, full_fname, namespace, space, mesh, &
      ff(:, ivd), unit, ierr, pos=pos, atoms=atoms, grp=grp, root=root)
  end do

  if (bitand(how, OPTION__OUTPUTFORMAT__VTK) /= 0) then
    ! write vtk files only from one rank
    i_am_root = .true.
    root_ = optional_default(root, 0)
    if (mesh%parallel_in_domains) then
      i_am_root = (mesh%mpi_grp%rank == root_)
    end if
    if (present(grp)) then
      i_am_root = i_am_root .and. (grp%rank == root_)
    end if
    ! now the VTK output
    call cube_init(cube, mesh%idx%ll, namespace, space, mesh%spacing, mesh%coord_system)
    call cube_init_cube_map(cube, mesh)
    SAFE_ALLOCATE(cf(1:space%dim))

    ! get global cube function for each dimension
    do ivd = 1, space%dim
      call X(cube_function_alloc_RS) (cube, cf(ivd))
      call X(mesh_to_cube) (mesh, ff(:, ivd), cube, cf(ivd))
    end do

    filename = trim(dir)//'/'//trim(fname)//".vtk"
    do ivd = 1, space%dim
      dk(ivd)= units_from_atomic(units_out%length, mesh%spacing(ivd))
    end do

    if (i_am_root) then
      call X(vtk_out_cf_vector)(filename, namespace, fname, ierr, cf, space%dim, cube, dk, unit)
    end if

    do ivd = 1, space%dim
      call X(cube_function_free_RS)(cube, cf(ivd))
    end do
    call cube_end(cube)
  end if

  POP_SUB(X(io_function_output_vector))
end subroutine X(io_function_output_vector)

! ---------------------------------------------------------
subroutine X(io_function_output_vector_BZ)(how, dir, fname, namespace, space, kpt, kpoints, ff, &
  unit, ierr, grp, root)
  integer(int64),                intent(in)  :: how
  character(len=*),           intent(in)  :: dir
  character(len=*),           intent(in)  :: fname
  type(namespace_t),          intent(in)  :: namespace
  class(space_t),             intent(in)  :: space
  type(distributed_t),        intent(in)  :: kpt
  type(kpoints_t),            intent(in)  :: kpoints
  R_TYPE,           target,   intent(in)  :: ff(:, :)
  type(unit_t),               intent(in)  :: unit
  integer,                    intent(out) :: ierr
  type(mpi_grp_t),  optional, intent(in)  :: grp  !< the group that shares the same data, must contain the domains group
  integer,          optional, intent(in)  :: root !< which process is going to write the data

  integer :: ivd
  character(len=MAX_PATH_LEN) :: full_fname
  R_TYPE, pointer :: ff_global(:, :)
  logical :: i_am_root
  integer :: root_
  type(MPI_Comm) :: comm
  type(mpi_grp_t) :: grp_

  PUSH_SUB(X(io_function_output_vector_BZ))

  ASSERT(space%dim < 10)

  ierr = 0
  ASSERT( ubound(ff, 1) - lbound(ff, dim = 1) + 1 == kpt%end - kpt%start + 1)

  root_ = optional_default(root, 0)
  if (present(grp)) then
    call mpi_grp_copy(grp_, grp)
  else
    call mpi_grp_init(grp_, comm)
  end if

  i_am_root = grp%rank == root_

  if (kpt%parallel) then
    SAFE_ALLOCATE(ff_global(1:kpoints%reduced%npoints, 1:space%dim))
    ff_global(1:kpoints%reduced%npoints, 1:space%dim) = R_TOTYPE(M_ZERO)

    do ivd = 1, space%dim
      ff_global(kpt%start:kpt%end, ivd) = ff(lbound(ff, 1):ubound(ff, 1), ivd)
    end do
    call comm_allreduce(grp_, ff_global)
  else
    ff_global => ff
  end if

  if (i_am_root) then
    if (how /= 0) then !perhaps there is nothing left to do
      do ivd = 1, space%dim
        full_fname = trim(fname)//'-'//index2axis(ivd)

        call X(io_function_output_global_BZ)(how, dir, full_fname, namespace, kpoints, &
          ff_global(:, ivd), unit, ierr)
      end do
    end if

  end if

  ! I have to broadcast the error code
  call grp_%bcast(ierr, 1, MPI_INTEGER, 0)

  if (kpt%parallel) then
    SAFE_DEALLOCATE_P(ff_global)
  else
    nullify(ff_global)
  end if

  POP_SUB(X(io_function_output_vector_BZ))
end subroutine X(io_function_output_vector_BZ)


!> @brief Top-level IO routine for functions defined on the mesh.
!!
!! Outputs functions defined on the mesh to several file and output formats:
!! * DX
!! * XCRYSDEN
!! * CUBE
!! * VTK
!! * NETCDF
!!
!! Distinguishes between output formats that require a whole cube grid, and those that need
!! a whole mesh.
subroutine X(io_function_output) (how, dir, fname, namespace, space, mesh, ff, unit, ierr, pos, atoms, grp, root)
  integer(int64),             intent(in)  :: how       !< Defines the output format in bitwise representation
  !           representing all valid permutations of file format, and data format. For example "cube + axis_z"
  character(len=*),           intent(in)  :: dir       !< Output directory
  character(len=*),           intent(in)  :: fname     !< Output file, minus extension
  type(namespace_t),          intent(in)  :: namespace
  class(space_t),             intent(in)  :: space
  class(mesh_t),              intent(in)  :: mesh
  R_TYPE, contiguous, target, intent(in)  :: ff(:)
  type(unit_t),               intent(in)  :: unit
  integer,                    intent(out) :: ierr
  real(real64),     optional, intent(in)  :: pos(:,:)
  type(atom_t),     optional, intent(in)  :: atoms(:)
  type(mpi_grp_t),  optional, intent(in)  :: grp !< the group that shares the same data, must contain the domains group
  integer,          optional, intent(in)  :: root !< which process is going to write the data

  logical :: i_am_root
  integer :: root_
  R_TYPE, pointer :: ff_global(:)
  logical :: need_global_cube, need_global_mesh
  type(cube_t) :: cube
  type(cube_function_t) :: cf

  PUSH_SUB(X(io_function_output))

  ASSERT(present(pos) .eqv. present(atoms))

  ! first check if we need a global cube or global mesh function
  need_global_cube = (bitand(how, OPTION__OUTPUTFORMAT__DX) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__XCRYSDEN) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__CUBE) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__VTK) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__NETCDF) /= 0)

  need_global_mesh = (bitand(how, OPTION__OUTPUTFORMAT__BINARY) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__AXIS_X) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__AXIS_Y) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__AXIS_Z) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__PLANE_X) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Y) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Z) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_XY) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_XZ) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_YZ) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__MESH_INDEX) /= 0) .or. &
    (bitand(how, OPTION__OUTPUTFORMAT__MATLAB) /= 0)

  if (need_global_cube) then
    ! write files depending on cube data only from one rank
    i_am_root = .true.
    root_ = optional_default(root, 0)
    if (mesh%parallel_in_domains) then
      i_am_root = (mesh%mpi_grp%rank == root_)
    end if
    if (present(grp)) then
      i_am_root = i_am_root .and. (grp%rank == root_)
    end if

    call cube_init(cube, mesh%idx%ll, namespace, space, mesh%spacing, mesh%coord_system, fft_library=FFTLIB_NONE)
    call cube_init_cube_map(cube, mesh)
    call X(cube_function_alloc_RS) (cube, cf)
    ! mesh_to_cube gathers the full cube function on all ranks, although we
    ! only need it on the root rank here. This should be improved at some point.
    ! See gitlab issue #1027
    call X(mesh_to_cube)(mesh, ff, cube, cf)
    if (i_am_root) then
      call X(io_cf_output_global)(how, dir, fname, namespace, space, mesh, cube, cf, unit, ierr, &
        pos=pos, atoms=atoms)
    end if
    call X(cube_function_free_RS)(cube, cf)
    call cube_end(cube)
  end if

  if (need_global_mesh) then
    ! get global mesh function
    ierr = 0
    ASSERT(ubound(ff, dim = 1) == mesh%np .or. ubound(ff, dim = 1) == mesh%np_part)

    i_am_root = .true.
    root_ = optional_default(root, 0)
    if (mesh%parallel_in_domains) then
      i_am_root = (mesh%mpi_grp%rank == root_)
      if (bitand(how, OPTION__OUTPUTFORMAT__BOUNDARY_POINTS) /= 0) then
        call messages_not_implemented("OutputFormat = boundary_points with domain parallelization", namespace=namespace)
        SAFE_ALLOCATE(ff_global(1:mesh%np_part_global))
        ! FIXME: needs version of par_vec_gather that includes boundary points. See ticket #127
      else
        SAFE_ALLOCATE(ff_global(1:mesh%np_global))
      end if

      !note: here we are gathering data that we won`t write if grp is
      !present, but to avoid it we will have to find out all if the
      !processes are members of the domain line where the root of grp
      !lives
      call par_vec_gather(mesh%pv, root_, ff, ff_global)
    else
      ff_global => ff
    end if

    if (present(grp)) then
      i_am_root = i_am_root .and. (grp%rank == root_)
    end if

    if (i_am_root) then
      call X(io_function_output_global)(how, dir, fname, namespace, space, mesh, ff_global, unit, ierr)
    end if

    if (mesh%parallel_in_domains) then
      SAFE_DEALLOCATE_P(ff_global)
    else
      nullify(ff_global)
    end if
  end if

  POP_SUB(X(io_function_output))
end subroutine X(io_function_output)

! ---------------------------------------------------------
subroutine X(io_function_output_global) (how, dir, fname, namespace, space, mesh, ff, unit, ierr)
  integer(int64),                intent(in)  :: how
  character(len=*),           intent(in)  :: dir, fname
  type(namespace_t),          intent(in)  :: namespace
  class(space_t),             intent(in)  :: space
  type(mesh_t),               intent(in)  :: mesh
  R_TYPE,                     intent(in)  :: ff(:)  !< (mesh%np_global or mesh%np_part_global)
  type(unit_t),               intent(in)  :: unit
  integer,                    intent(out) :: ierr

  character(len=512) :: filename
  character(len=20)  :: mformat, mformat2, mfmtheader
  integer            :: iunit, jj
  integer(int64)        :: np_max, ip
  real(real64)       :: x0

  call profiling_in(TOSTRING(X(DISK_WRITE)))
  PUSH_SUB(X(io_function_output_global))

  call io_mkdir(dir, namespace)

! Define the format
  mformat    = '(99es23.14E3)'
  mformat2   = '(i12,99es34.24E3)'
  mfmtheader = '(a,a10,5a23)'

  ASSERT(ubound(ff, dim=1, kind=int64) >= mesh%np_global)

  np_max = mesh%np_global
  ! should we output boundary points?
  if (bitand(how, OPTION__OUTPUTFORMAT__BOUNDARY_POINTS) /= 0) then
    if (ubound(ff, dim=1, kind=int64) >= mesh%np_part_global) then
      np_max = mesh%np_part_global
    else
      write(message(1),'(2a)') trim(fname), ': not outputting boundary points; they are not available'
      call messages_warning(1, namespace=namespace)
      ! FIXME: in this case, one could allocate an array of the larger size and apply boundary conditions
    end if
  end if

  if (bitand(how, OPTION__OUTPUTFORMAT__BINARY)     /= 0) call out_binary()
  if (bitand(how, OPTION__OUTPUTFORMAT__AXIS_X)     /= 0) call out_axis (1, 2, 3) ! x ; y=0,z=0
  if (bitand(how, OPTION__OUTPUTFORMAT__AXIS_Y)     /= 0) call out_axis (2, 1, 3) ! y ; x=0,z=0
  if (bitand(how, OPTION__OUTPUTFORMAT__AXIS_Z)     /= 0) call out_axis (3, 1, 2) ! z ; x=0,y=0
  if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_X)    /= 0) call out_plane(1, 2, 3) ! x=0; y; z;
  if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Y)    /= 0) call out_plane(2, 1, 3) ! y=0; x; z;
  if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Z)    /= 0) call out_plane(3, 1, 2) ! z=0; x; y;
  if (bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_XY)    /= 0) call out_integrate_plane(1, 2, 3) ! \int dx dy; z;
  if (bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_XZ)    /= 0) call out_integrate_plane(1, 3, 2) ! \int dx dz; y;
  if (bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_YZ)    /= 0) call out_integrate_plane(2, 3, 1) ! \int dy dz; x;
  if (bitand(how, OPTION__OUTPUTFORMAT__MESH_INDEX) /= 0) call out_mesh_index()

  if (bitand(how, OPTION__OUTPUTFORMAT__MATLAB) /= 0) then
#if defined(R_TCOMPLEX)
    do jj = 1, 3 ! re, im, abs
#else
    do jj = 1, 1 ! only real part
#endif
      if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_X) /= 0) call out_matlab(how, 1, 2, 3, jj) ! x=0; y; z;
      if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Y) /= 0) call out_matlab(how, 2, 1, 3, jj) ! y=0; x; z;
      if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Z) /= 0) call out_matlab(how, 3, 1, 2, jj) ! z=0; x; y;
    end do
    if (bitand(how, OPTION__OUTPUTFORMAT__MESHGRID) /= 0) then
      do jj = 4, 5 ! meshgrid
        if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_X) /= 0) call out_matlab(how, 1, 2, 3, jj) ! x=0; y; z;
        if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Y) /= 0) call out_matlab(how, 2, 1, 3, jj) ! y=0; x; z;
        if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Z) /= 0) call out_matlab(how, 3, 1, 2, jj) ! z=0; x; y;
      end do
    end if
  end if

  POP_SUB(X(io_function_output_global))
  call profiling_out(TOSTRING(X(DISK_WRITE)))

contains

  ! ---------------------------------------------------------
  subroutine out_binary()
    character(len=512) :: workdir

    PUSH_SUB(X(io_function_output_global).out_binary)

    workdir = io_workpath(dir, namespace)
    call io_binary_write(trim(workdir)//'/'//trim(fname)//'.obf', np_max, ff, ierr)

    call profiling_count_transfers(np_max, ff(1))
    POP_SUB(X(io_function_output_global).out_binary)
  end subroutine out_binary


  ! ---------------------------------------------------------
  subroutine out_axis(d1, d2, d3)
    integer, intent(in) :: d1, d2, d3

    integer :: ixvect(space%dim)
    real(real64)   :: xx(space%dim)
    R_TYPE  :: fu
    logical :: in_line

    PUSH_SUB(X(io_function_output_global).out_axis)

    filename = trim(dir)//'/'//trim(fname)//"."//index2axis(d2)//"=0,"//index2axis(d3)//"=0"
    iunit = io_open(filename, namespace, action='write')

    write(iunit, mfmtheader, iostat=ierr) '#', index2axis(d1), 'Re', 'Im'
    do ip = 1, np_max
      call mesh_global_index_to_coords(mesh, ip, ixvect)

      select case (space%dim)
      case (1)
        in_line = .true.
      case (2)
        in_line = ixvect(d2) == 0
      case default
        in_line = ixvect(d2) == 0  .and. ixvect(d3) == 0
      end select

      if (in_line) then
        xx = units_from_atomic(units_out%length, mesh_x_global(mesh, ip))
        fu = units_from_atomic(unit, ff(ip))
        write(iunit, mformat, iostat=ierr) xx(d1), fu
      end if
    end do

    call io_close(iunit)
    POP_SUB(X(io_function_output_global).out_axis)
  end subroutine out_axis


  ! ---------------------------------------------------------
  subroutine out_plane(d1, d2, d3)
    integer, intent(in) :: d1, d2, d3

    integer :: ix, iy, iz, jdim
    integer :: ixvect(space%dim)
    integer :: ixvect_test(space%dim)
    real(real64)   :: xx(space%dim)
    R_TYPE  :: fu

    PUSH_SUB(X(io_function_output_global).out_plane)

    filename = trim(dir)//'/'//trim(fname)//"."//index2axis(d1)//"=0"
    iunit = io_open(filename, namespace, action='write')

    write(iunit, mfmtheader, iostat=ierr) '#', index2axis(d2), index2axis(d3), 'Re', 'Im'

! here we find the indices for coordinate 0 along all directions apart from d2
! and d3, to get a plane. Do the same as for ix, but with all the other
! dimensions.

    ixvect = 1
    do jdim = 1, space%dim
      if (jdim == d2 .or. jdim == d3) cycle

      do ix = mesh%idx%nr(1, jdim), mesh%idx%nr(2, jdim)
! NOTE: MJV: how could this return anything but ix=0? Answ: if there is a shift in origin
        ixvect_test = 1
        ixvect_test(jdim) = ix
        ip = mesh_global_index_from_coords(mesh, ixvect_test)
        if (ip /= 0) then
          call mesh_global_index_to_coords(mesh, ip, ixvect_test)
          if (ixvect_test(jdim) == 0) exit
        end if
      end do
      ixvect(jdim) = ix
    end do ! loop over dimensions

    ! have found ix such that coordinate d1 is 0 for this value of ix
    ! ixvect is prepared for all dimensions apart from d2 and d3

    do iy = mesh%idx%nr(1, d2), mesh%idx%nr(2, d2)
      write(iunit, mformat, iostat=ierr)
      do iz = mesh%idx%nr(1, d3), mesh%idx%nr(2, d3)

        ixvect(d2) = iy
        ixvect(d3) = iz
        ip = mesh_global_index_from_coords(mesh, ixvect)

        if (ip <= np_max .and. ip > 0) then
          xx = units_from_atomic(units_out%length, mesh_x_global(mesh, ip))
          fu = units_from_atomic(unit, ff(ip))
          write(iunit, mformat, iostat=ierr)  &
            xx(d2), xx(d3), fu
        end if
      end do
    end do

    call io_close(iunit)

    POP_SUB(X(io_function_output_global).out_plane)
  end subroutine out_plane

  ! ---------------------------------------------------------
  subroutine out_integrate_plane(d1, d2, d3)
    integer, intent(in) :: d1, d2, d3

    integer :: ix, iy, iz, np
    integer :: ixvect(space%dim)
    real(real64)   :: xx(space%dim), zz
    R_TYPE  :: fu

    PUSH_SUB(X(io_function_output_global).out_integrate_plane)

    filename = trim(dir)//'/'//trim(fname)//".int_d"//index2axis(d1)//"d"//index2axis(d2)
    iunit = io_open(filename, namespace, action='write')

    write(iunit, mfmtheader, iostat=ierr) '#', index2axis(d3), 'Re', 'Im'

    do iz = mesh%idx%nr(1, d3), mesh%idx%nr(2, d3)

      fu = R_TOTYPE(M_ZERO)
      np = 0
      ixvect(d3) = iz
      do ix = mesh%idx%nr(1, d1), mesh%idx%nr(2, d1)
        do iy = mesh%idx%nr(1, d2), mesh%idx%nr(2, d2)

          ixvect(d1) = ix
          ixvect(d2) = iy
          ip = mesh_global_index_from_coords(mesh, ixvect)

          if (ip <= np_max .and. ip > 0) then
            xx = units_from_atomic(units_out%length, mesh_x_global(mesh, ip))
            if (mesh%use_curvilinear) then
              fu = fu + units_from_atomic(unit, ff(ip))*mesh%vol_pp(ip)
            else
              fu = fu + units_from_atomic(unit, ff(ip))
            end if
            zz = xx(d3)
            np = np + 1
          end if
        end do
      end do

      fu = fu*mesh%coord_system%surface_element(d3)
      if(np > 0 ) write(iunit, mformat, iostat=ierr) zz, fu
    end do

    call io_close(iunit)

    POP_SUB(X(io_function_output_global).out_integrate_plane)
  end subroutine out_integrate_plane


  ! ---------------------------------------------------------
  subroutine out_matlab(how, d1, d2, d3, out_what)
    integer(int64), intent(in) :: how
    integer, intent(in) :: d1, d2, d3, out_what

    integer :: ix, iy, record_length
    integer :: min_d2, min_d3, max_d2, max_d3
    real(real64), allocatable :: out_vec(:)
    real(real64) :: xx(space%dim)
    R_TYPE  :: fu

    PUSH_SUB(X(io_function_output_global).out_matlab)

    min_d2 = mesh%idx%nr(1, d2) + mesh%idx%enlarge(d2)
    max_d2 = mesh%idx%nr(2, d2) - mesh%idx%enlarge(d2)
    min_d3 = mesh%idx%nr(1, d3) + mesh%idx%enlarge(d3)
    max_d3 = mesh%idx%nr(2, d3) - mesh%idx%enlarge(d3)

    if (bitand(how, OPTION__OUTPUTFORMAT__BOUNDARY_POINTS) /= 0) then
      min_d2 = mesh%idx%nr(1, d2)
      max_d2 = mesh%idx%nr(2, d2)
      min_d3 = mesh%idx%nr(1, d3)
      max_d3 = mesh%idx%nr(2, d3)
    end if

    select case (out_what)
    case (1:3)
      filename = trim(dir)//'/'//trim(fname)//"."//index2axis(d1)//"=0.matlab"
#if defined(R_TCOMPLEX)
      filename = trim(filename)//"."//trim(index2label(out_what))
#endif
    case (4)
      filename = trim(dir)//'/meshgrid.'//index2axis(d1)//"=0."//trim(index2axis(d3)) ! meshgrid d3
    case (5)
      filename = trim(dir)//'/meshgrid.'//index2axis(d1)//"=0."//trim(index2axis(d2)) ! meshgrid d2
    end select

    record_length = (max_d3 - min_d3 + 1)*23  ! 23 because of F23.14 below
    iunit = io_open(filename, namespace, action='write', recl=record_length)


    SAFE_ALLOCATE(out_vec(min_d3:max_d3))

    do ix = min_d2, max_d2

      out_vec = M_ZERO

      do iy = min_d3, max_d3

        select case (d1)
        case (1)
          ip = mesh_global_index_from_coords(mesh, [ 0, ix, iy])    ! plane_x
        case (2)
          ip = mesh_global_index_from_coords(mesh, [ix,  0, iy])    ! plane_y
        case (3)
          ip = mesh_global_index_from_coords(mesh, [ix, iy,  0])    ! plane_z
        end select

        select case (out_what)
        case (4)
          xx(:) = mesh_x_global(mesh, ip)
          out_vec(iy) = xx(d2)      ! meshgrid d2 (this is swapped wrt.
        case (5)
          xx(:) = mesh_x_global(mesh, ip)
          out_vec(iy) = xx(d3)      ! meshgrid d3  to the filenames)
        end select

        if (ip < 1 .or. ip > np_max) cycle

        fu = units_from_atomic(unit, ff(ip))

        select case (out_what)
        case (1)
          out_vec(iy) = R_REAL(fu)  ! real part
        case (2)
          out_vec(iy) = R_AIMAG(fu) ! imaginary part
        case (3)
          out_vec(iy) = abs(fu)   ! absolute value
        end select

      end do

      ! now we write to the disk
      write(iunit,'(32767f23.14)') (out_vec(iy), iy = min_d3, max_d3)

    end do

    SAFE_DEALLOCATE_A(out_vec)
    call io_close(iunit)

    POP_SUB(X(io_function_output_global).out_matlab)
  end subroutine out_matlab


  ! ---------------------------------------------------------
  subroutine out_mesh_index()
    real(real64) :: xx(space%dim)
    R_TYPE :: fu

    integer :: idir

    PUSH_SUB(X(io_function_output_global).out_mesh_index)

    iunit = io_open(trim(dir)//'/'//trim(fname)//".mesh_index", namespace, action='write')

    write(iunit, mfmtheader, iostat=ierr) '#', 'Index', 'x', 'y', 'z', 'Re', 'Im'
    xx = mesh_x_global(mesh, 1_int64)
    x0 = xx(1)
    if (ierr == 0) write(iunit, mformat, iostat=ierr)

    do ip = 1, np_max
      xx = mesh_x_global(mesh, ip)
      if (ierr == 0 .and. abs(x0 - xx(1)) <= M_EPSILON) then
        write(iunit, mformat, iostat=ierr)      ! write extra lines for gnuplot grid mode
        x0 = xx(1)
      end if
      fu = units_from_atomic(unit, ff(ip))
      if (ierr == 0) write(iunit, mformat2, iostat=ierr) ip, &
        (units_from_atomic(units_out%length, xx(idir)), idir = 1, 3), fu
    end do

    if (ierr == 0) write(iunit, mformat, iostat=ierr)
    call io_close(iunit)

    POP_SUB(X(io_function_output_global).out_mesh_index)
  end subroutine out_mesh_index

end subroutine X(io_function_output_global)

! ---------------------------------------------------------
subroutine X(io_cf_output_global) (how, dir, fname, namespace, space, mesh, cube, cf, unit, ierr, pos, atoms)
  integer(int64),                intent(in)  :: how
  character(len=*),           intent(in)  :: dir, fname
  type(namespace_t),          intent(in)  :: namespace
  class(space_t),             intent(in)  :: space
  type(mesh_t),               intent(in)  :: mesh
  type(cube_t),               intent(in)  :: cube
  type(cube_function_t),      intent(in)  :: cf
  type(unit_t),               intent(in)  :: unit
  integer,                    intent(out) :: ierr
  real(real64),     optional, intent(in)  :: pos(:,:)
  type(atom_t),     optional, intent(in)  :: atoms(:)

  character(len=512) :: filename
  character(len=20)  :: mformat, mformat2, mfmtheader
  integer            :: iunit

  call profiling_in(TOSTRING(X(DISK_WRITE)))
  PUSH_SUB(X(io_cf_output_global))

  ASSERT(present(pos) .eqv. present(atoms))

  call io_mkdir(dir, namespace)

! Define the format
  mformat    = '(99es23.14E3)'
  mformat2   = '(i12,99es34.24E3)'
  mfmtheader = '(a,a10,5a23)'

  if (bitand(how, OPTION__OUTPUTFORMAT__DX)         /= 0) call out_dx()
  if (bitand(how, OPTION__OUTPUTFORMAT__XCRYSDEN)   /= 0) then
    call out_xcrysden(.true.)
#ifdef R_TCOMPLEX
    call out_xcrysden(.false.)
#endif
  end if
  if (bitand(how, OPTION__OUTPUTFORMAT__CUBE)       /= 0) call out_cube()
#if defined(HAVE_NETCDF)
  if (bitand(how, OPTION__OUTPUTFORMAT__NETCDF)     /= 0) call out_netcdf(namespace)
#endif
  if (bitand(how, OPTION__OUTPUTFORMAT__VTK) /= 0) call out_vtk()

  POP_SUB(X(io_cf_output_global))
  call profiling_out(TOSTRING(X(DISK_WRITE)))

contains

  ! ---------------------------------------------------------
  !> Writes real and imaginary parts
  subroutine out_dx()
    integer :: ix, iy, iz, idir
    real(real64)   :: offset(3)
    character(len=40) :: nitems

    PUSH_SUB(X(io_cf_output_global).out_dx)

    ! Along periodic dimensions the offset is -1/2 in reduced coordinates, as
    ! our origin is at the center of the cell instead of being at the corner.
    offset(1:space%dim) = cube%latt%red_to_cart(spread(-M_HALF, 1, space%dim))
    do idir = space%periodic_dim + 1, 3
      offset(idir) = -(cube%rs_n_global(idir) - 1)/2*cube%spacing(idir)
    end do
    offset = units_from_atomic(units_out%length, offset)

    ! just for nice formatting of the output
    write(nitems,*) cube%rs_n_global(1)*cube%rs_n_global(2)*cube%rs_n_global(3)
    nitems=trim(adjustl(nitems))

    iunit = io_open(trim(dir)//'/'//trim(fname)//".dx", namespace, action='write')

    write(iunit, '(a,3i7)') 'object 1 class gridpositions counts', cube%rs_n_global(1:3)
    write(iunit, '(a,3f12.6)') ' origin', offset(1:3)
    write(iunit, '(a,3f12.6)') ' delta ', (units_from_atomic(units_out%length, &
      cube%spacing(1)*cube%latt%rlattice_primitive(idir, 1)), idir = 1, 3)
    write(iunit, '(a,3f12.6)') ' delta ', (units_from_atomic(units_out%length, &
      cube%spacing(2)*cube%latt%rlattice_primitive(idir, 2)), idir = 1, 3)
    write(iunit, '(a,3f12.6)') ' delta ', (units_from_atomic(units_out%length, &
      cube%spacing(3)*cube%latt%rlattice_primitive(idir, 3)), idir = 1, 3)
    write(iunit, '(a,3i7)') 'object 2 class gridconnections counts', cube%rs_n_global(1:3)
#if defined(R_TREAL)
    write(iunit, '(a,a,a)') 'object 3 class array type float rank 0 items ', nitems, ' data follows'
#else
    write(iunit, '(a,a,a)') 'object 3 class array type float category complex rank 0 items ', nitems, ' data follows'
#endif
    do ix = 1, cube%rs_n_global(1)
      do iy = 1, cube%rs_n_global(2)
        do iz = 1, cube%rs_n_global(3)
          write(iunit,'(2f25.15)') units_from_atomic(unit, cf%X(RS)(ix, iy, iz))
        end do
      end do
    end do
    write(iunit, '(a)') 'object "regular positions regular connections" class field'
    write(iunit, '(a)') ' component "positions" value 1'
    write(iunit, '(a)') ' component "connections" value 2'
    write(iunit, '(a)') ' component "data" value 3'
    write(iunit, '(a)') 'end'

    call io_close(iunit)

    POP_SUB(X(io_cf_output_global).out_dx)
  end subroutine out_dx


  ! ---------------------------------------------------------
  !> see http://local.wasp.uwa.edu.au/~pbourke/dataformats/cube/
  !! Writes only real part
  subroutine out_cube()

    integer :: ix, iy, iz, idir, idir2, iatom
    real(real64)   :: offset(3)
    character(len=20) :: fmt

    PUSH_SUB(X(io_cf_output_global).out_cube)

    ! Along periodic dimensions the offset is -1/2 in reduced coordinates, as
    ! our origin is at the center of the cell instead of being at the corner.
    offset(1:space%dim) = cube%latt%red_to_cart(spread(-M_HALF, 1, space%dim))
    do idir = space%periodic_dim + 1, 3
      offset(idir) = -(cube%rs_n_global(idir) - 1)/2*cube%spacing(idir)
    end do

    iunit = io_open(trim(dir)//'/'//trim(fname)//".cube", namespace, action='write')

    write(iunit, '(2a)') 'Generated by octopus ', trim(conf%version)
    write(iunit, '(4a)') 'git: ', trim(conf%git_commit), " configuration: ",  trim(conf%config_time)
    if (present(pos)) then
      write(iunit, '(i5,3f12.6)') size(pos, dim=2), offset(1:3)
    else
      write(iunit, '(i5,3f12.6)') 0, offset(1:3)
    end if

    ! According to http://gaussian.com/cubegen/, cube files are always in atomic units.
    ! Although the specifications mention a way of indicating Angstrom units,
    ! this is only for Gaussian input files, not for output files.
    do idir = 1, 3
      write(iunit, '(i5,3f12.6)') cube%rs_n_global(idir), &
        (cube%spacing(idir)*cube%latt%rlattice_primitive(idir2, idir), idir2 = 1, 3)
    end do
    if (present(pos) .and. present(atoms)) then
      do iatom = 1, size(pos, dim=2)
        write(iunit, '(i5,4f12.6)') int(atoms(iatom)%species%get_z()),  M_ZERO, pos(:, iatom)
      end do
    end if

    do ix = 1, cube%rs_n_global(1)
      do iy = 1, cube%rs_n_global(2)
        do iz = 1, cube%rs_n_global(3), 6

          if (iz + 6 - 1 <= cube%rs_n_global(3)) then
            write(iunit,'(6e15.6e3)') R_REAL(cf%X(RS)(ix, iy, iz:iz + 6 - 1))
          else
            write(fmt, '(a,i1,a)') '(', cube%rs_n_global(3) - iz + 1, 'e15.6e3)'
            write(iunit, trim(fmt)) R_REAL(cf%X(RS)(ix, iy, iz:cube%rs_n_global(3)))
          end if

        end do
      end do
    end do

    call io_close(iunit)

    POP_SUB(X(io_cf_output_global).out_cube)
  end subroutine out_cube


  ! ---------------------------------------------------------
  !> For format specification see:
  !! http://www.xcrysden.org/doc/XSF.html#__toc__11
  !! XCrySDen can only read 3D output, though it could be
  !! extended to plot a function on a 2D plane.
  !! Writes real part unless write_real = false and called in complex version
  subroutine out_xcrysden(write_real)
    logical, intent(in) :: write_real

    integer :: ix, iy, iz, idir2, ix2, iy2, iz2, my_n(3), ll(3), idir
    character(len=80) :: fname_ext
    type(lattice_vectors_t) :: latt

    PUSH_SUB(X(io_cf_output_global).out_xcrysden)

#ifdef R_TCOMPLEX
    if (write_real) then
      fname_ext = trim(fname) // '.real'
    else
      fname_ext = trim(fname) // '.imag'
    end if
#else
    fname_ext = trim(fname)
#endif

    if (space%dim /= 3 .and. space%dim /= 2) then
      write(message(1), '(a)') 'Cannot output function '//trim(fname_ext)//' in XCrySDen format except in 2D or 3D.'
      call messages_warning(1, namespace=namespace)
      POP_SUB(X(io_cf_output_global).out_xcrysden)
      return
    end if

    ! Note that XCrySDen uses "general" not "periodic" grids
    ! mesh%idx%ll is "general" in aperiodic directions,
    ! but "periodic" in periodic directions.
    ! Making this assignment, the output grid is entirely "general"
    my_n(1:space%periodic_dim) = mesh%idx%ll(1:space%periodic_dim) + 1
    my_n(space%periodic_dim + 1:space%dim) = mesh%idx%ll(space%periodic_dim + 1:space%dim)
    if (space%dim == 2) my_n(3) = 1


    iunit = io_open(trim(dir)//'/'//trim(fname_ext)//".xsf", namespace, action='write')

    if (present(pos) .and. present(atoms)) then
      ! To avoid passing as an argument latt, we use the one of the cube.
      ! As it is always 3D, we need to create a temporary lattice vectors to the right dimension
      latt = lattice_vectors_t(namespace, space, cube%latt%rlattice(1:space%dim, 1:space%dim))
      call write_xsf_geometry(iunit, space, latt, pos, atoms, mesh)
    end if

    write(iunit, '(a,i1,a)') 'BEGIN_BLOCK_DATAGRID_', space%dim, 'D'
    write(iunit, '(4a)') 'units: coords = ', trim(units_abbrev(units_out%length)), &
      ', function = ', trim(units_abbrev(unit))
    write(iunit, '(a,i1,a)') 'BEGIN_DATAGRID_', space%dim, 'D_function'
    write(iunit, '(3i7)') my_n(1:space%dim)
    write(iunit, '(a)') '0.0 0.0 0.0'

    do idir = 1, space%dim
      write(iunit, '(3f12.6)') (units_from_atomic(units_out%length, &
        cube%latt%rlattice(idir2, idir)), idir2 = 1, 3)
    end do

    ll(1:space%dim) = mesh%idx%ll(1:space%dim)
    ll(space%dim+1:3) = 0

    do iz = 1, my_n(3)
      do iy = 1, my_n(2)
        do ix = 1, my_n(1)
          ! this is about "general" grids also
          if (ix == ll(1) + 1) then
            ix2 = 1
          else
            ix2 = ix
          end if

          if (iy == ll(2) + 1) then
            iy2 = 1
          else
            iy2 = iy
          end if

          if (iz == ll(3) + 1) then
            iz2 = 1
          else
            iz2 = iz
          end if

#ifdef R_TCOMPLEX
          if (.not. write_real) then
            write(iunit,'(2f25.15)') aimag(units_from_atomic(unit, cf%X(RS)(ix2, iy2, iz2)))
          else
            write(iunit,'(2f25.15)') real(units_from_atomic(unit, cf%X(RS)(ix2, iy2, iz2)), real64)
          end if
#else
          write(iunit,'(2f25.15)') units_from_atomic(unit, cf%X(RS)(ix2, iy2, iz2))
#endif
        end do
      end do
    end do

    write(iunit, '(a,i1,a)') 'END_DATAGRID_', space%dim, 'D'
    write(iunit, '(a,i1,a)') 'END_BLOCK_DATAGRID_', space%dim, 'D'

    call io_close(iunit)

    POP_SUB(X(io_cf_output_global).out_xcrysden)
  end subroutine out_xcrysden


#if defined(HAVE_NETCDF)
  ! ---------------------------------------------------------
  subroutine out_netcdf(namespace)
    type(namespace_t), intent(in)  :: namespace

    PUSH_SUB(X(io_cf_output_global).out_netcdf)

    filename = io_workpath(trim(dir)//'/'//trim(fname)//".ncdf", namespace)

    call X(out_cf_netcdf)(filename, ierr, cf, cube, space, &
      units_from_atomic(units_out%length, mesh%spacing), .true., unit, namespace)

    POP_SUB(X(io_cf_output_global).out_netcdf)
  end subroutine out_netcdf

#endif /*defined(HAVE_NETCDF)*/

  ! ---------------------------------------------------------
  subroutine out_vtk()
    real(real64) :: dk(3), pnt(3)
    integer :: i, i1, i2, i3
    real(real64), allocatable :: points(:,:,:,:)

    PUSH_SUB(X(io_cf_output_global).out_vtk)

    ASSERT(space%dim == 3)

    do i = 1, 3
      dk(i)= units_from_atomic(units_out%length, mesh%spacing(i))
    end do

    filename = trim(dir)//'/'//trim(fname)//".vtk"

    if (cube%latt%nonorthogonal) then
      ! non-orthogonal grid
      SAFE_ALLOCATE(points(1:cube%rs_n_global(1),1:cube%rs_n_global(2),1:cube%rs_n_global(3),1:3))

      do i1 =1 , cube%rs_n_global(1)
        do i2 =1 , cube%rs_n_global(2)
          do i3 =1 , cube%rs_n_global(3)
            pnt(1:3) =(/cube%Lrs(i1, 1),cube%Lrs(i2, 2),cube%Lrs(i3, 3)/)
            points(i1,i2,i3, 1:3) = matmul(cube%latt%rlattice_primitive(1:3,1:3), pnt(1:3))
          end do
        end do
      end do

      call X(vtk_out_cf_structured)(filename, namespace, fname, ierr, cf, cube, unit, points)
      SAFE_DEALLOCATE_A(points)
    else
      !Ordinary grid
      call X(vtk_out_cf)(filename, namespace, fname, ierr, cf, cube, dk(:), unit)
    end if

    POP_SUB(X(io_cf_output_global).out_vtk)
  end subroutine out_vtk

end subroutine X(io_cf_output_global)


! ---------------------------------------------------------
subroutine X(io_function_output_global_BZ) (how, dir, fname, namespace, kpoints, ff, unit, ierr)
  integer(int64),                intent(in)  :: how
  character(len=*),           intent(in)  :: dir, fname
  type(namespace_t),          intent(in)  :: namespace
  type(kpoints_t),            intent(in)  :: kpoints
  R_TYPE,                     intent(in)  :: ff(:)  !< (st%nik)
  type(unit_t),               intent(in)  :: unit
  integer,                    intent(out) :: ierr

  character(len=512) :: filename
  character(len=20)  :: mformat, mformat2, mfmtheader
  integer            :: iunit, np_max

  call profiling_in(TOSTRING(X(DISK_WRITE)))
  PUSH_SUB(X(io_function_output_global_BZ))

  call io_mkdir(dir, namespace)

! Define the format
  mformat    = '(99es23.14E3)'
  mformat2   = '(i12,99es34.24E3)'
  mfmtheader = '(a,a10,5a23)'

  ASSERT(ubound(ff, dim = 1) >= kpoints%reduced%npoints)

  np_max = kpoints%reduced%npoints

  if (bitand(how, OPTION__OUTPUTFORMAT__BINARY)   /= 0) call messages_not_implemented("Output_KPT with format binary", namespace=namespace)
  if (bitand(how, OPTION__OUTPUTFORMAT__AXIS_X)   /= 0) call messages_not_implemented("Output_KPT with format axis x", namespace=namespace)
  if (bitand(how, OPTION__OUTPUTFORMAT__AXIS_Y)   /= 0) call messages_not_implemented("Output_KPT with format axis y", namespace=namespace)
  if (bitand(how, OPTION__OUTPUTFORMAT__AXIS_Z)   /= 0) call messages_not_implemented("Output_KPT with format axis z", namespace=namespace)
  if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_X)  /= 0) call out_plane(1, 2, 3) ! x=0; y; z;
  if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Y)  /= 0) call out_plane(2, 1, 3) ! y=0; x; z;
  if (bitand(how, OPTION__OUTPUTFORMAT__PLANE_Z)  /= 0) call out_plane(3, 1, 2) ! z=0; x; y;
  if (bitand(how, OPTION__OUTPUTFORMAT__DX)       /= 0) call messages_not_implemented("Output_KPT with format dx", namespace=namespace)
  if (bitand(how, OPTION__OUTPUTFORMAT__XCRYSDEN) /= 0) call messages_not_implemented("Output_KPT with format xcrysden", namespace=namespace)
  if (bitand(how, OPTION__OUTPUTFORMAT__CUBE)     /= 0) call messages_not_implemented("Output_KPT with format cube", namespace=namespace)

  if (bitand(how, OPTION__OUTPUTFORMAT__MATLAB) /= 0) call messages_not_implemented("Output_KPT with format matlab", namespace=namespace)

#if defined(HAVE_NETCDF)
  if (bitand(how, OPTION__OUTPUTFORMAT__NETCDF)  /= 0) call messages_not_implemented("Output_KPT with format netcdf", namespace=namespace)
#endif
  if (bitand(how, OPTION__OUTPUTFORMAT__VTK) /= 0) call out_vtk()

  POP_SUB(X(io_function_output_global_BZ))
  call profiling_out(TOSTRING(X(DISK_WRITE)))


contains
  ! ---------------------------------------------------------
  subroutine out_plane(d1, d2, d3)
    integer, intent(in) :: d1, d2, d3

    integer :: ik, dim
    real(real64)   :: kk(kpoints%reduced%dim)
    R_TYPE  :: fu

    PUSH_SUB(X(io_function_output_global_BZ).out_plane)

    filename = trim(dir)//'/'//trim(fname)//"."//index2axisBZ(d1)//"=0"
    iunit = io_open(filename, namespace, action='write')

#ifdef R_TCOMPLEX
    write(iunit, mfmtheader, iostat=ierr) '#', index2axisBZ(d2), index2axisBZ(d3), 'Re', 'Im'
#else
    write(iunit, mfmtheader, iostat=ierr) '#', index2axisBZ(d2), index2axisBZ(d3), 'Re'
#endif

    dim = kpoints%reduced%dim

    do ik = 1, kpoints%reduced%npoints
      kk(1:dim) = units_from_atomic(units_out%length**(-1), &
        kpoints%reduced%point1BZ(1:dim,ik))
      if (abs(kk(d1)) < 1.0e-6_real64) then
        fu = units_from_atomic(unit, ff(ik))
        write(iunit, mformat, iostat=ierr) kk(d2), kk(d3), fu

      end if
    end do

    call io_close(iunit)

    POP_SUB(X(io_function_output_global_BZ).out_plane)
  end subroutine out_plane


  ! ---------------------------------------------------------

  subroutine out_vtk()
    integer :: npath, nik, axis(3)
    real(real64), allocatable :: kout(:,:)

    PUSH_SUB(X(io_function_output_global_BZ).out_vtk)

    filename = trim(dir)//'/'//trim(fname)//".vtk"

    ! In case the user used also a k-point path, we ignore it
    npath = kpoints%nkpt_in_path()
    axis(1:3) = kpoints%nik_axis(1:3)
    nik = product(kpoints%nik_axis(1:3))

    if (nik /= kpoints%reduced%npoints - npath) then
      call messages_not_implemented("Output_KPT with format vtk without KPointsGrid")
    end if

    SAFE_ALLOCATE(kout(1:3, 1:nik))
    kout(:, :) = units_from_atomic(units_out%length**(-1), kpoints%reduced%point1BZ(:,1:nik))
    call X(vtk_out_bz_structured)(filename, namespace, fname, ierr, kpoints%nik_axis, ff, unit, kout)
    SAFE_DEALLOCATE_A(kout)

    POP_SUB(X(io_function_output_global_BZ).out_vtk)
  end subroutine out_vtk

end subroutine X(io_function_output_global_BZ)

#if defined(HAVE_NETCDF)
 ! ---------------------------------------------------------
 !>  Writes a cube_function in netcdf format
 ! ---------------------------------------------------------
subroutine X(out_cf_netcdf)(filename, ierr, cf, cube, space, spacing, transpose, unit, namespace)
  character(len=*),      intent(in) :: filename        !< the file name
  integer,               intent(out):: ierr            !< error message
  type(cube_function_t), intent(in) :: cf              !< the cube_function to be written
  type(cube_t),          intent(in) :: cube            !< the underlying cube mesh
  class(space_t),        intent(in) :: space           !< the spatial dimensions
  real(real64),          intent(in) :: spacing(:)      !< the mesh spacing already converted to units_out
  logical,               intent(in) :: transpose       !< whether we want the function cf(x,y,z) to be saved as cf(z,y,x)
  type(unit_t),          intent(in) :: unit            !< unit of data in cf
  type(namespace_t),     intent(in) :: namespace

  integer :: ncid, status, data_id, pos_id, dim_min
  integer :: dim_data_id(3), dim_pos_id(2)

  real(real32) :: pos(2, 3)
  real(real64), allocatable :: xx(:, :, :)

#if defined(R_TCOMPLEX)
  integer :: data_im_id
#endif

  PUSH_SUB(X(out_cf_netcdf))

  ierr = 0

  status = nf90_create(trim(filename), NF90_CLOBBER, ncid)
  if (status /= NF90_NOERR) then
    ierr = 2
    POP_SUB(X(out_cf_netcdf))
    return
  end if

  ! dimensions
  if (status == NF90_NOERR) then
    status = nf90_def_dim (ncid, "dim_1", cube%rs_n_global(3), dim_data_id(1))
    call ncdf_error('nf90_def_dim', status, filename, namespace, ierr)
  end if

  if (status == NF90_NOERR) then
    status = nf90_def_dim (ncid, "dim_2", cube%rs_n_global(2), dim_data_id(2))
    call ncdf_error('nf90_def_dim', status, filename, namespace, ierr)
  end if

  if (status == NF90_NOERR) then
    status = nf90_def_dim (ncid, "dim_3", cube%rs_n_global(1), dim_data_id(3))
    call ncdf_error('nf90_der_dim', status, filename, namespace, ierr)
  end if

  if (status == NF90_NOERR) then
    status = nf90_def_dim (ncid, "pos_1", 2, dim_pos_id(1))
    call ncdf_error('nf90_def_dim', status, filename, namespace, ierr)
  end if

  if (status == NF90_NOERR) then
    status = nf90_def_dim (ncid, "pos_2", 3, dim_pos_id(2))
    call ncdf_error('nf90_def_dim', status, filename, namespace, ierr)
  end if

  dim_min = 3 - space%dim + 1

  if (status == NF90_NOERR) then
    status = nf90_def_var (ncid, "rdata", NF90_DOUBLE, dim_data_id(dim_min:3), data_id)
    call ncdf_error('nf90_def_var', status, filename, namespace, ierr)
  end if
#if defined(R_TCOMPLEX)
  if (status == NF90_NOERR) then
    status = nf90_def_var (ncid, "idata", NF90_DOUBLE, dim_data_id(dim_min:3), data_im_id)
    call ncdf_error('nf90_def_var', status, filename, namespace, ierr)
  end if
#endif
  if (status == NF90_NOERR) then
    status = nf90_def_var (ncid, "pos", NF90_FLOAT,  dim_pos_id,  pos_id)
    call ncdf_error('nf90_def_var', status, filename, namespace, ierr)
  end if

  ! attributes
  if (status == NF90_NOERR) then
    status = nf90_put_att (ncid, data_id, "field", "rdata, scalar")
    call ncdf_error('nf90_put_att', status, filename, namespace, ierr)
  end if
  if (status == NF90_NOERR) then
    status = nf90_put_att (ncid, data_id, "positions", "pos, regular")
    call ncdf_error('nf90_put_att', status, filename, namespace, ierr)
  end if
#if defined(R_TCOMPLEX)
  if (status == NF90_NOERR) then
    status = nf90_put_att (ncid, data_im_id, "field", "idata, scalar")
    call ncdf_error('nf90_put_att', status, filename, namespace, ierr)
  end if
  if (status == NF90_NOERR) then
    status = nf90_put_att (ncid, data_im_id, "positions", "pos, regular")
    call ncdf_error('nf90_put_att', status, filename, namespace, ierr)
  end if
#endif

  ! end definitions
  status = nf90_enddef (ncid)

  ! FIXME: needs explicit casts to single-precision. Why single-precision anyway?
  ! data
  pos(:,:) = M_ZERO
  pos(1, 1:space%dim) = &
    real ( - (cube%rs_n_global(1:space%dim) - 1)/2*spacing(1:space%dim), real32)
  pos(2, 1:space%dim) = real (spacing(1:space%dim), real32)

  if (status == NF90_NOERR) then
    status = nf90_put_var (ncid, pos_id, pos(:,:))
    call ncdf_error('nf90_put_var', status, filename, namespace, ierr)
  end if

  SAFE_ALLOCATE(xx(1:cube%rs_n_global(3), 1:cube%rs_n_global(2), 1:cube%rs_n_global(1)))
#if defined(R_TCOMPLEX)
  if (status == NF90_NOERR) then
    if (transpose) then
      call transpose3(real(cf%X(RS), real64), xx)
    else
      xx = real(cf%X(RS), real64)
    end if
    xx = units_from_atomic(unit, xx)
    call write_variable(ncid, data_id, status, space, xx)
    call ncdf_error('nf90_put_var', status, filename, namespace, ierr)
  end if
  if (status == NF90_NOERR) then
    if (transpose) then
      call transpose3(aimag(cf%X(RS)), xx)
    else
      xx = aimag(cf%X(RS))
    end if
    xx = units_from_atomic(unit, xx)
    call write_variable(ncid, data_im_id, status, space, xx)
    call ncdf_error('nf90_put_var', status, filename, namespace, ierr)
  end if
#else
  if (status == NF90_NOERR) then
    if (transpose) then
      call transpose3(cf%X(RS), xx)
    else
      xx=cf%X(RS)
    end if
    xx = units_from_atomic(unit, xx)
    call write_variable(ncid, data_id, status, space, xx)
    call ncdf_error('nf90_put_var', status, filename, namespace, ierr)
  end if
#endif
  SAFE_DEALLOCATE_A(xx)

  ! close
  status = nf90_close(ncid)

  POP_SUB(X(out_cf_netcdf))

contains

  ! ---------------------------------------------------------
  subroutine write_variable(ncid, data_id, status, space, xx)
    integer,       intent(in)  :: ncid, data_id
    integer,       intent(out) :: status
    class(space_t),intent(in)  :: space
    real(real64),  intent(in)  :: xx(:,:,:)

    PUSH_SUB(X(out_cf_netcdf).write_variable)

    select case (space%dim)
    case (1)
      status = nf90_put_var (ncid, data_id, xx(1,1,:))
    case (2)
      status = nf90_put_var (ncid, data_id, xx(1,:,:))
    case (3)
      status = nf90_put_var (ncid, data_id, xx)
    end select

    POP_SUB(X(out_cf_netcdf).write_variable)
  end subroutine write_variable

end subroutine X(out_cf_netcdf)

#endif /*defined(HAVE_NETCDF)*/

 ! ---------------------------------------------------------
subroutine X(io_function_output_supercell) (how, dir, fname, mesh, space, latt, ff, centers, supercell, unit, ierr, &
  namespace, pos, atoms, grp, root, is_global, extra_atom)
  integer(8),                 intent(in)  :: how
  character(len=*),           intent(in)  :: dir
  character(len=*),           intent(in)  :: fname
  class(mesh_t),              intent(in)  :: mesh
  class(space_t),             intent(in)  :: space
  type(lattice_vectors_t),    intent(in)  :: latt
  R_TYPE, contiguous, target, intent(in)  :: ff(:,:)
  real(real64),               intent(in)  :: centers(:,:)
  integer,                    intent(in)  :: supercell(:)
  type(unit_t),               intent(in)  :: unit
  integer,                    intent(out) :: ierr
  type(namespace_t),          intent(in)  :: namespace
  real(real64),     optional, intent(in)  :: pos(:,:)
  type(atom_t),     optional, intent(in)  :: atoms(:)
  type(mpi_grp_t),  optional, intent(in)  :: grp !< the group that shares the same data, must contain the domains group
  integer,          optional, intent(in)  :: root !< which process is going to write the data
  logical,          optional, intent(in)  :: is_global !< Input data is mesh%np_global? And, thus, it has not be gathered
  real(real64),     optional, intent(in)  :: extra_atom(:)

  integer :: Nreplica
  logical :: is_global_
  logical :: i_am_root
  integer :: root_, irep
  R_TYPE, pointer, contiguous :: ff_global(:, :)
  type(mpi_grp_t) :: grp_

  PUSH_SUB(X(io_function_output_supercell))

  ASSERT(present(pos) .eqv. present(atoms))

  ierr = 0
  is_global_ = optional_default(is_global, .false.)
  if (is_global_) then
    ASSERT(ubound(ff, dim=1, kind=int64) == mesh%np_global .or. ubound(ff, dim=1, kind=int64) == mesh%np_part_global)
  else
    ASSERT(ubound(ff, dim = 1) == mesh%np .or. ubound(ff, dim = 1) == mesh%np_part)
  end if

  Nreplica = product(supercell(1:space%dim))

  i_am_root = .true.
  call mpi_grp_init(grp_, MPI_COMM_UNDEFINED)
  root_ = optional_default(root, 0)
  if(mesh%parallel_in_domains) then
    call mpi_grp_copy(grp_, mesh%mpi_grp)
    i_am_root = (mesh%mpi_grp%rank == root_)
    if (.not. is_global_) then
      if(bitand(how, OPTION__OUTPUTFORMAT__BOUNDARY_POINTS) /= 0) then
        call messages_not_implemented("OutputFormat = boundary_points with domain parallelization")
        SAFE_ALLOCATE(ff_global(1:mesh%np_part_global, 1:Nreplica))
        ! FIXME: needs version of vec_gather that includes boundary points. See ticket #127
      else
        SAFE_ALLOCATE(ff_global(1:mesh%np_global, 1:Nreplica))
      end if

      !note: here we are gathering data that we won`t write if grp is
      !present, but to avoid it we will have to find out all if the
      !processes are members of the domain line where the root of grp
      !lives
      do irep = 1, Nreplica
        call par_vec_gather(mesh%pv, root_, ff(:,irep), ff_global(:,irep))
      end do
    else
      ff_global => ff
    end if
  else
    ff_global => ff
  end if

  if(present(grp)) then
    i_am_root = i_am_root .and. (grp%rank == root_)
    call mpi_grp_copy(grp_, grp)
  end if

  if(i_am_root) then
    call X(io_function_output_global_supercell)(how, dir, fname, mesh, space, latt, ff_global, centers, supercell, &
      unit, ierr, namespace, pos = pos, atoms=atoms, extra_atom=extra_atom)
  end if
  ! I have to broadcast the error code
  if (.not. is_global_) call grp_%bcast(ierr, 1, MPI_INTEGER, 0)

  if(mesh%parallel_in_domains .and. .not. is_global_) then
    SAFE_DEALLOCATE_P(ff_global)
  else
    nullify(ff_global)
  end if

  POP_SUB(X(io_function_output_supercell))
end subroutine X(io_function_output_supercell)

! ---------------------------------------------------------
subroutine X(io_function_output_global_supercell) (how, dir, fname, mesh, space, latt, ff, centers, &
  supercell, unit, ierr, namespace, pos, atoms, extra_atom)
  integer(8),                 intent(in)  :: how
  character(len=*),           intent(in)  :: dir, fname
  type(mesh_t),               intent(in)  :: mesh
  class(space_t),             intent(in)  :: space
  type(lattice_vectors_t),    intent(in)  :: latt
  R_TYPE, contiguous,         intent(in)  :: ff(:,:)  !< (mesh%np_global or mesh%np_part_global)
  real(real64),               intent(in)  :: centers(:,:)
  integer,                    intent(in)  :: supercell(:)
  type(unit_t),               intent(in)  :: unit
  integer,                    intent(out) :: ierr
  type(namespace_t),          intent(in)  :: namespace
  real(real64),     optional, intent(in)  :: pos(:,:)
  type(atom_t),     optional, intent(in)  :: atoms(:)
  real(real64),     optional, intent(in)  :: extra_atom(:)

  character(len=512) :: filename
  character(len=20)  :: mformat, mformat2, mfmtheader
  integer            :: iunit, idir
  integer(int64)        :: ip, np_max
  integer            :: Nreplica

  call profiling_in("DISK_WRITE")
  PUSH_SUB(X(io_function_output_global_supercell))

  call io_mkdir(dir)

  ! Define the format
  mformat    = '(99es23.14E3)'
  mformat2   = '(i12,99es34.24E3)'
  mfmtheader = '(a,a10,5a23)'

  ASSERT(how > 0)
  ASSERT(ubound(ff, dim=1, kind=int64) >= mesh%np_global)

  Nreplica = product(supercell(1:space%dim))

  np_max = mesh%np_global
  ! should we output boundary points?
  if(bitand(how, OPTION__OUTPUTFORMAT__BOUNDARY_POINTS) /= 0) then
    if(ubound(ff, dim=1, kind=int64) >= mesh%np_part_global) then
      np_max = mesh%np_part_global
    else
      write(message(1),'(2a)') trim(fname), ': not outputting boundary points; they are not available'
      call messages_warning(1)
      ! FIXME: in this case, one could allocate an array of the larger size and apply boundary conditions
    endif
  endif

  if(bitand(how, OPTION__OUTPUTFORMAT__BINARY)     /= 0) call messages_not_implemented("Output supercell with format binary")
  if(bitand(how, OPTION__OUTPUTFORMAT__AXIS_X)     /= 0) call out_axis(1, 2, 3)
  if(bitand(how, OPTION__OUTPUTFORMAT__AXIS_Y)     /= 0) call out_axis(2, 1, 3)
  if(bitand(how, OPTION__OUTPUTFORMAT__AXIS_Z)     /= 0) call out_axis(3, 1, 2)
  if(bitand(how, OPTION__OUTPUTFORMAT__PLANE_X)    /= 0) call out_plane(1, 2, 3) ! x=0; y; z;
  if(bitand(how, OPTION__OUTPUTFORMAT__PLANE_Y)    /= 0) call out_plane(2, 1, 3) ! y=0; x; z;
  if(bitand(how, OPTION__OUTPUTFORMAT__PLANE_Z)    /= 0) call out_plane(3, 1, 2) ! z=0; x; y;
  if(bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_XY)    /= 0) call messages_not_implemented("Output supercell with integrated format")
  if(bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_XZ)    /= 0) call messages_not_implemented("Output supercell with integrated format")
  if(bitand(how, OPTION__OUTPUTFORMAT__INTEGRATE_YZ)    /= 0) call messages_not_implemented("Output supercell with integrated format")
  if(bitand(how, OPTION__OUTPUTFORMAT__MESH_INDEX) /= 0) call messages_not_implemented("Output supercell with mesh index")
  if(bitand(how, OPTION__OUTPUTFORMAT__DX)         /= 0) call messages_not_implemented("Output supercell with DX format")
  if(bitand(how, OPTION__OUTPUTFORMAT__XCRYSDEN)   /= 0) then
    call out_xcrysden(.true.)
#ifdef R_TCOMPLEX
    call out_xcrysden(.false.)
#endif
  end if
  if(bitand(how, OPTION__OUTPUTFORMAT__CUBE)       /= 0) call messages_not_implemented("Output supercell with cube format")

  if(bitand(how, OPTION__OUTPUTFORMAT__MATLAB) /= 0) then
    call messages_not_implemented("Output supercell with matlab format")
  end if

#if defined(HAVE_NETCDF)
  if(bitand(how, OPTION__OUTPUTFORMAT__NETCDF)     /= 0) call messages_not_implemented("Output supercell with Netcdf format")
#endif
  if(bitand(how, OPTION__OUTPUTFORMAT__VTK) /= 0) call messages_not_implemented("Output supercell with VTK format")

  POP_SUB(X(io_function_output_global_supercell))
  call profiling_out("DISK_WRITE")

contains
  ! ---------------------------------------------------------
  subroutine out_axis(d1, d2, d3)
    integer, intent(in) :: d1, d2, d3

    integer :: ixvect(space%dim), irep
    real(real64)   :: xx(space%dim)
    R_TYPE  :: fu

    PUSH_SUB(X(io_function_output_global_supercell).out_axis)

    filename = trim(dir)//'/'//trim(fname)//"."//index2axis(d2)//"=0,"//index2axis(d3)//"=0"
    iunit = io_open(filename, action='write')

    write(iunit, mfmtheader, iostat=ierr) '#', index2axis(d1), 'Re', 'Im'

    do irep = 1, Nreplica
      if(abs(centers(d2,irep)) > M_EPSILON .or. abs(centers(d3,irep)) > M_EPSILON ) cycle
      do ip = 1, np_max
        call index_to_coords(mesh%idx, ip, ixvect)

        if(ixvect(d2)==0.and.ixvect(d3)==0) then
          xx = units_from_atomic(units_out%length, mesh_x_global(mesh, ip)) + centers(1:space%dim, irep)
          fu = units_from_atomic(unit, ff(ip, irep))
          write(iunit, mformat, iostat=ierr) xx(d1), fu
        end if
      end do
    end do

    call io_close(iunit)
    POP_SUB(X(io_function_output_global_supercell).out_axis)
  end subroutine out_axis


  ! ---------------------------------------------------------
  subroutine out_plane(d1, d2, d3)
    integer, intent(in) :: d1, d2, d3

    integer :: ix, iy, iz, jdim, irep
    integer :: ixvect(space%dim)
    integer :: ixvect_test(space%dim)
    real(real64)   :: xx(space%dim)
    R_TYPE  :: fu

    PUSH_SUB(X(io_function_output_global_supercell).out_plane)

    filename = trim(dir)//'/'//trim(fname)//"."//index2axis(d1)//"=0"
    iunit = io_open(filename, action='write')

    write(iunit, mfmtheader, iostat=ierr) '#', index2axis(d2), index2axis(d3), 'Re', 'Im'

! here we find the indices for coordinate 0 along all directions apart from d2
! and d3, to get a plane. Do the same as for ix, but with all the other
! dimensions.

    do irep = 1, Nreplica
      if(abs(centers(d1, irep)) > M_EPSILON) cycle
      ixvect=1
      do jdim=1, space%dim
        if (jdim==d2 .or. jdim==d3) cycle

        do ix = mesh%idx%nr(1, jdim), mesh%idx%nr(2, jdim)
! NOTE: MJV: how could this return anything but ix=0? Answ: if there is a shift in origin
          ixvect_test = 1
          ixvect_test(jdim) = ix
          ip = mesh_global_index_from_coords(mesh, ixvect_test)
          if(ip /= 0) then
            call index_to_coords(mesh%idx, ip, ixvect_test)
            if(ixvect_test(jdim) == 0) exit
          end if
        end do
        ixvect(jdim) = ix
      end do ! loop over dimensions

      ! have found ix such that coordinate d1 is 0 for this value of ix
      ! ixvect is prepared for all dimensions apart from d2 and d3

      do iy = mesh%idx%nr(1, d2), mesh%idx%nr(2, d2)
        write(iunit, mformat, iostat=ierr)
        do iz = mesh%idx%nr(1, d3), mesh%idx%nr(2, d3)

          ixvect(d2) = iy
          ixvect(d3) = iz
          ip = mesh_global_index_from_coords(mesh, ixvect)

          if(ip <= np_max .and. ip > 0) then
            xx = units_from_atomic(units_out%length, mesh_x_global(mesh, ip)+centers(1:space%dim, irep))
            fu = units_from_atomic(unit, ff(ip, irep))
            write(iunit, mformat, iostat=ierr)  &
              xx(d2), xx(d3), fu
          end if
        end do
      end do

    end do !irep

    call io_close(iunit)

    POP_SUB(X(io_function_output_global_supercell).out_plane)
  end subroutine out_plane

  ! ---------------------------------------------------------
  !> For format specification see:
  !! http://www.xcrysden.org/doc/XSF.html#__toc__11
  !! XCrySDen can only read 3D output, though it could be
  !! extended to plot a function on a 2D plane.
  !! Writes real part unless write_real = false and called in complex version
  subroutine out_xcrysden(write_real)
    logical, intent(in) :: write_real

    integer :: ix, iy, iz, idir2, my_n(3)
    integer :: ixs, iys, izs, isx, isy, isz, my_n_s(3)
    integer :: irep
    real(real64) :: lattice_vectors(3,3)
    type(cube_t) :: cube
    type(cube_function_t), allocatable :: cf(:)
    character(len=80) :: fname_ext

    PUSH_SUB(X(io_function_output_global_supercell).out_xcrysden)

#ifdef R_TCOMPLEX
    if(write_real) then
      fname_ext = trim(fname) // '.real'
    else
      fname_ext = trim(fname) // '.imag'
    end if
#else
    fname_ext = trim(fname)
#endif

    if(space%dim /= 3 .and. space%dim /= 2) then
      write(message(1), '(a)') 'Cannot output function '//trim(fname_ext)//' in XCrySDen format except in 2D or 3D.'
      call messages_warning(1)
      return
    end if

    ! put values in a nice cube
    call cube_init(cube, mesh%idx%ll, namespace, space, mesh%spacing, mesh%coord_system)
    call cube_init_cube_map(cube, mesh)
    SAFE_ALLOCATE(cf(1:Nreplica))
    do irep = 1, Nreplica
      call X(cube_function_alloc_RS) (cube, cf(irep))
      call X(mesh_to_cube) (mesh, ff(:,irep), cube, cf(irep))
    end do

    ! Note that XCrySDen uses "general" not "periodic" grids
    ! mesh%idx%ll is "general" in aperiodic directions,
    ! but "periodic" in periodic directions.
    ! Making this assignment, the output grid is entirely "general"
    !
    !Here we do not put the +1 for my_n
    my_n(1:space%periodic_dim) = mesh%idx%ll(1:space%periodic_dim)
    my_n(space%periodic_dim + 1:space%dim) = mesh%idx%ll(space%periodic_dim + 1:space%dim)
    if(space%dim == 2) my_n(3) = 1

    my_n_s(1:space%periodic_dim) = mesh%idx%ll(1:space%periodic_dim)*supercell(1:space%periodic_dim) +1
    my_n_s(space%periodic_dim + 1:space%dim) = mesh%idx%ll(space%periodic_dim + 1:space%dim) &
      * supercell(space%periodic_dim + 1:space%dim)
    if(space%dim == 2) my_n_s(3) = 1

    ! This differs from mesh%sb%rlattice if it is not an integer multiple of the spacing
    do idir = 1, 3
      do idir2 = 1, 3
        lattice_vectors(idir2, idir) = mesh%spacing(idir) * (my_n_s(idir) - 1) * latt%rlattice_primitive(idir2, idir)
      end do
    end do

    iunit = io_open(trim(dir)//'/'//trim(fname_ext)//".xsf", action='write')

    call write_xsf_geometry_supercell(iunit, space, latt, pos, atoms, mesh, centers, supercell, extra_atom)

    write(iunit, '(a,i1,a)') 'BEGIN_BLOCK_DATAGRID_', space%dim, 'D'
    write(iunit, '(4a)') 'units: coords = ', trim(units_abbrev(units_out%length)), &
      ', function = ', trim(units_abbrev(unit))
    write(iunit, '(a,i1,a)') 'BEGIN_DATAGRID_', space%dim, 'D_function'
    write(iunit, '(3i7)') my_n_s(1:space%dim)
    write(iunit, '(a)') '0.0 0.0 0.0'

    do idir = 1, space%dim
      write(iunit, '(3f12.6)') (units_from_atomic(units_out%length, &
        lattice_vectors(idir2, idir)), idir2 = 1, 3)
    end do

    do izs = 1, my_n_s(3)
      do iys = 1, my_n_s(2)
        do ixs = 1, my_n_s(1)

          ! this is about "general" grids also
          if (ixs == my_n_s(1)) then
            ix = 1
            isx = supercell(1)-1
          else
            ix = modulo (ixs-1, my_n(1)) + 1
            isx = (ixs-ix)/my_n(1)
          end if

          if (iys == my_n_s(2)) then
            iy = 1
            isy = supercell(2)-1
          else
            iy = modulo (iys-1, my_n(2)) + 1
            isy = (iys-iy)/my_n(2)
          end if

          if (izs == my_n_s(3)) then
            iz = 1
            isz = supercell(3)-1
          else
            iz = modulo (izs-1, my_n(3)) + 1
            isz = (izs-iz)/my_n(3)
          end if

          irep = isx*supercell(2)*supercell(3) + isy*supercell(3) + isz + 1

#ifdef R_TCOMPLEX
          if(.not. write_real) then
            write(iunit,'(1f25.15)') aimag(units_from_atomic(unit, cf(irep)%zRS(ix, iy, iz)))
          else
            write(iunit,'(1f25.15)') real(units_from_atomic(unit, cf(irep)%zRS(ix, iy, iz)), real64)
          end if
#else
          write(iunit,'(1f25.15)') units_from_atomic(unit, cf(irep)%dRS(ix, iy, iz))
#endif
        end do
      end do
    end do

    write(iunit, '(a,i1,a)') 'END_DATAGRID_', space%dim, 'D'
    write(iunit, '(a,i1,a)') 'END_BLOCK_DATAGRID_', space%dim, 'D'

    call io_close(iunit)

    do irep = 1, Nreplica
      call X(cube_function_free_RS)(cube, cf(irep))
    end do
    call cube_end(cube)
    SAFE_DEALLOCATE_A(cf)

    POP_SUB(X(io_function_output_global_supercell).out_xcrysden)
  end subroutine out_xcrysden

end subroutine X(io_function_output_global_supercell)


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
