!! Copyright (C) 2020-2023 M. Oliveira
!!
!! This Source Code Form is subject to the terms of the Mozilla Public
!! License, v. 2.0. If a copy of the MPL was not distributed with this
!! file, You can obtain one at https://mozilla.org/MPL/2.0/.
!!
#include "global.h"

!> @brief This module defines the abstract interaction_t class, and some
!> auxiliary classes for interactions.
!!
!! When executing two or more coupled iterative algorithms there needs to be a
!! mechanism for the algorithms to exchange the information that couples
!! them. In this framework, this is done by the interaction class and we call
!! the information to be exchanged "couplings". An interaction connects two
!! systems, which is the name given to the class that executes the algorithms
!! and stores their internal state (see the system_t class for more details
!! about systems).
!!
!! Although the exchange of couplings between two systems usually happens in
!! both directions, this is not always a symmetric operation. Therefore, the
!! choice was made to split interactions between two systems into
!! uni-directional interactions. This in turn means that the interactions have a
!! well defined source (causing the interaction) and a target system, which is
!! affected by the interaction:
!!
!! \dot
!!    digraph interaction {
!!     "system A" -> "system B" [label=" A -> B   "];
!!     "system B" -> "system A" [label=" B -> A   "];
!!    }
!! \enddot
!!
!! The flow of information goes from the source to the target and the
!! interaction is "owned" by the target. That means that, in the way systems and
!! interactions are implemented, the target is aware of the interactions that
!! affect it, but the source does not store any information about the
!! interactions it needs to provide information to. Instead, during the
!! algorithm execution, the source will receive requests to provide specific
!! quantities it knows about to an interaction. Those quantities are the
!! couplings.
!!
!! In some cases, systems can require information at specific iterations that
!! does not come from the execution of another algorithm. For example, this
!! information could be a prescribed mathematical expression or data read from a
!! file. Because of this, it is convenient to introduce the concept of
!! "interaction partner." Interaction partners are entities that can provide
!! couplings to an interaction. All systems are interaction partners, but not
!! all interaction partners need to be systems. Interaction partners are
!! implemented in the interaction_partner_t class and, because all systems are
!! interaction partners, the system_t class is an extension of this class.
!!
!! Often, the couplings a system requires from an interaction partner need to be
!! combined with information from the system itself in some way before it is
!! used. For example, to calculate the gravitational force acting on a planet
!! due to the Sun, one needs the positions and masses of both the planet and the
!! Sun. Because of this, besides handling the retrieval of the couplings from
!! the partner, the interaction class also provides a
!! interaction_oct_m::interaction_update deferred method that can be used to
!! combine the couplings with information from the system.
!!
!! When updating an interaction, the system will request this to be done for a
!! specific iteration. The interaction then needs to get the couplings of the
!! partner for that requested iteration. Because system and partner might
!! increment their iterations at a different pace, the couplings might not
!! always be available for the requested iteration. Different schemes are
!! implemented to handle these cases. These are called timing methods. There are
!! currently two of these methods implemented in the framwork. The first one is
!! the "exact" timing. In that case the couplings must be at the requested
!! iteration or the algorithm will refuse to proceed. The other one is the
!! "retarded" timing, where the previous partner iteration closest to the
!! requested iteration is used. Because the couplings used by the interaction
!! might be out-of-sync with the partner information, the interaction needs to
!! store a copy of them.
!!
!! The process of updating an interaction is then a four step process:
!!
!!  1. Request the partner to update the couplings (see system_oct_m::system_update_couplings).
!!  2. Check if the couplings are at the requested iteration or at a suitable
!!     iteration, depending on the timing method. If not, repeat step 1 until
!!     they are.
!!  3. If necessary, request the partner to copy the couplings to the
!!     interaction.
!!  4. Combine the couplings with the information from the system to upate the
!!     interaction.
!!
!! Note that, from the point of view of this framework, it is perfectly possible
!! to have a system interacting with itself. Because these "self-interactions"
!! often need to be treated in a special way, a flag is provided in the
!! interaction_t class to signal those cases.
!!
!! Interactions can only be created after all systems are created.
!! This is done by the call to the interactions_factory_t::create_interactions() function in run.F90:
!! \dontinclude{} main/run.F90
!! \snippet{lineno} main/run.F90 create_interactions
!
module interaction_oct_m
  use debug_oct_m
  use global_oct_m
  use interaction_surrogate_oct_m
  use interaction_partner_oct_m
  use iteration_counter_oct_m
  use linked_list_oct_m
  use multisystem_debug_oct_m
  use namespace_oct_m
  use profiling_oct_m
  use quantity_oct_m
  implicit none

  private
  public ::                 &
    interaction_t,          &
    interaction_end,        &
    interaction_list_t,     &
    interaction_iterator_t

  integer, parameter, public :: &
    TIMING_EXACT    = 1,        &
    TIMING_RETARDED = 2

  !> @brief abstract interaction class
  !!
  !! This class handles the exchange of couplings from an interaction partner to
  !! a system. It can then combine those couplings with information from the
  !! system itself to calculate the interaction as needed by the system. This
  !! means that the interaction needs to know a list of couplings it requires
  !! from the partner and a list of quantities needed from the system.
  !!
  !! This class has no explicit constructor, but objects are generated using interactions_factory_create().
  !
  type, extends(interaction_surrogate_t), abstract :: interaction_t
    ! General information about the interaction
    type(iteration_counter_t)     :: iteration         !< Iteration counter storing the iteration at which the interaction was last updated.
    character(len=:), allocatable :: label             !< label of an interaction, used for debug and restart
    logical                       :: intra_interaction !< Is this an interaction of a system with itself?
    integer                       :: timing            !< parameter to determine if the interaction should use the couplings
    !!                                                    at the exact requested iteration or if retardation is allowed
    real(real64)                  :: energy !< Energy associated with the interaction.

    ! The interaction requires access to some quantities from a system to be evaluated.
    character(len=:), allocatable :: system_quantities(:) !< Identifiers of the quantities needed from the system

    ! Information regarding the partner
    class(interaction_partner_t), pointer     :: partner => NULL()
    character(len=:),             allocatable :: couplings_from_partner(:) !< Identifiers of the quantities needed from the partner
    logical                                   :: partner_couplings_up_to_date = .false. !< Have the partner couplings been updated yet?

  contains
    procedure(interaction_calculate), deferred :: calculate
    procedure(interaction_calculate_energy), deferred :: calculate_energy
    procedure :: update => interaction_update !< @copydoc interaction_oct_m::interaction_update
    procedure :: update_partner_couplings => interaction_update_partner_couplings !< @copydoc interaction_oct_m::interaction_update_partner_couplings
    procedure :: restart_read => interaction_restart_read   !< @copydoc interaction_oct_m::interaction_restart_read
    procedure :: restart_write => interaction_restart_write !< @copydoc interaction_oct_m::interaction_restart_write
  end type interaction_t

  abstract interface
    !> All interactions need to implement the following deferred method, which
    !> takes information form the interaction partner and system to calculate
    !> the interaction
    subroutine interaction_calculate(this)
      import interaction_t
      class(interaction_t),              intent(inout) :: this
    end subroutine interaction_calculate

    !> All interactions need to implement the following deferred method, which
    !> takes information form the interaction partner and system to calculate
    !> the interaction energy
    subroutine interaction_calculate_energy(this)
      import interaction_t
      class(interaction_t),              intent(inout) :: this
    end subroutine interaction_calculate_energy
  end interface


  !> These classes extend the list and list iterator to make an interaction list
  type, extends(linked_list_t) :: interaction_list_t
    private
  contains
    procedure :: add => interaction_list_add_node
  end type interaction_list_t

  !> These class extend the list and list iterator to make an interaction list
  type, extends(linked_list_iterator_t) :: interaction_iterator_t
    private
  contains
    procedure :: get_next => interaction_iterator_get_next
  end type interaction_iterator_t

contains

  ! ---------------------------------------------------------
  !> Try to update all the couplings needed from the partner to update the interaction
  subroutine interaction_update_partner_couplings(this, requested_iteration)
    class(interaction_t),              intent(inout) :: this
    class(iteration_counter_t),        intent(in)    :: requested_iteration

    type(event_handle_t) :: debug_handle

    PUSH_SUB(interaction_update_partner_couplings)

    if (this%partner_couplings_up_to_date) then
      ! Couplings have already been updated
      POP_SUB(interaction_update_partner_couplings)
      return
    end if

    if (.not. allocated(this%couplings_from_partner)) then
      ! No couplings to update
      this%partner_couplings_up_to_date = .true.
      POP_SUB(interaction_update_partner_couplings)
      return
    end if

    debug_handle = multisystem_debug_write_event_in(system_namespace = this%partner%namespace, &
      event = event_function_call_t("system_update_exposed_quantities"), &
      requested_iteration = requested_iteration, &
      interaction_iteration = this%iteration)

    ! Update all the necessary couplings that are updated on demand
    call this%partner%update_on_demand_quantities(this%couplings_from_partner, requested_iteration, &
      retardation_allowed = this%timing == TIMING_RETARDED)

    ! Check the status of all the couplings and take the appropriate actions
    select case (this%partner%check_couplings_status(this%couplings_from_partner, requested_iteration))
    case (COUPLINGS_ON_TIME)
      ! Update was successful and we need to copy the couplings to the interaction
      this%partner_couplings_up_to_date = .true.
      call this%partner%copy_quantities_to_interaction(this)

    case (COUPLINGS_AHEAD_IN_TIME)
      ! Update is successful, only if retarded interaction is allowed. If that
      ! is the case, the interaction should use the couplings it already has, so
      ! there is no need to copy them to the interaction
      this%partner_couplings_up_to_date = (this%timing == TIMING_RETARDED)

    case (COUPLINGS_BEHIND_IN_TIME)
      ! For now we will mark this case as unsuccessful, but it could be
      ! successful if interpolation was allowed
      this%partner_couplings_up_to_date = .false.

    case (COUPLINGS_UNDEFINED)
      ! Partner couplings are in an undefined state (some are ahead of the
      ! requested iteration and some are at the requested iteration). This
      ! should never happen!
      ASSERT(.false.)
    end select

    call multisystem_debug_write_event_out(debug_handle, update=this%partner_couplings_up_to_date, &
      requested_iteration = requested_iteration, &
      interaction_iteration = this%iteration)

    POP_SUB(interaction_update_partner_couplings)

  end subroutine interaction_update_partner_couplings

  ! ---------------------------------------------------------
  !> Update the interaction to the requested_iteration.
  subroutine interaction_update(this, requested_iteration)
    class(interaction_t),              intent(inout) :: this
    class(iteration_counter_t),        intent(in)    :: requested_iteration

    type(event_handle_t) :: debug_handle

    PUSH_SUB(interaction_update)

    ! We should only try to update the interaction if it is not yet at the requested iteration
    ASSERT(.not. (this%iteration == requested_iteration))

    debug_handle = multisystem_debug_write_event_in(event = event_function_call_t("interaction_update"), &
      extra="target: "//trim(this%label)//"-"//trim(this%partner%namespace%get()), &
      interaction_iteration = this%iteration,      &
      requested_iteration = requested_iteration)

    call this%calculate()

    ! Set new interaction iteration
    call this%iteration%set(requested_iteration)
    call multisystem_debug_write_marker(event = event_iteration_update_t( "interaction", &
      trim(this%label)//"-"//trim(this%partner%namespace%get()), &
      this%iteration, "set"))

    ! After leaving this routine, the partner quantities will need to updated
    ! again for the next interaction update
    this%partner_couplings_up_to_date = .false.

    call multisystem_debug_write_event_out(debug_handle, update = .true.,  &
      interaction_iteration = this%iteration,  &
      requested_iteration = requested_iteration)

    POP_SUB(interaction_update)
  end subroutine interaction_update

  ! ---------------------------------------------------------
  subroutine interaction_end(this)
    class(interaction_t),              intent(inout) :: this

    PUSH_SUB(interaction_end)

    if (allocated(this%couplings_from_partner)) then
      deallocate(this%couplings_from_partner)
    end if
    nullify(this%partner)

    if (allocated(this%system_quantities)) then
      deallocate(this%system_quantities)
    end if

    if (allocated(this%label)) then
      deallocate(this%label)
    end if

    POP_SUB(interaction_end)
  end subroutine interaction_end

  ! ---------------------------------------------------------
  !> @brief read restart information
  !!
  !! return .true. on success.
  logical function interaction_restart_read(this, namespace)
    class(interaction_t),              intent(inout) :: this
    type(namespace_t),                 intent(in)    :: namespace

    PUSH_SUB(interaction_restart_read)

    interaction_restart_read = this%iteration%restart_read('restart_iteration_interaction_'//trim(this%label), &
      namespace)

    POP_SUB(interaction_restart_read)
  end function interaction_restart_read

  ! ---------------------------------------------------------
  subroutine interaction_restart_write(this, namespace)
    class(interaction_t),              intent(inout) :: this
    type(namespace_t),                 intent(in)    :: namespace

    PUSH_SUB(interaction_restart_write)

    call this%iteration%restart_write('restart_iteration_interaction_'//trim(this%label), namespace)

    POP_SUB(interaction_restart_write)
  end subroutine interaction_restart_write

  ! ---------------------------------------------------------
  subroutine interaction_list_add_node(this, interaction)
    class(interaction_list_t)                 :: this
    class(interaction_t),              target :: interaction

    PUSH_SUB(interaction_list_add_node)

    call this%add_ptr(interaction)

    POP_SUB(interaction_list_add_node)
  end subroutine interaction_list_add_node

  ! ---------------------------------------------------------
  function interaction_iterator_get_next(this) result(interaction)
    class(interaction_iterator_t),     intent(inout) :: this
    class(interaction_t),              pointer       :: interaction

    PUSH_SUB(interaction_iterator_get_next)

    select type (ptr => this%get_next_ptr())
    class is (interaction_t)
      interaction => ptr
    class default
      ASSERT(.false.)
    end select

    POP_SUB(interaction_iterator_get_next)
  end function interaction_iterator_get_next

end module interaction_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
