!! Copyright (C) 2020-24 M. Oliveira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

!> @brief This module defines the abstract class for the interaction factory.
!!
module interactions_factory_abst_oct_m
  use interaction_oct_m
  use interaction_partner_oct_m
  use namespace_oct_m
  implicit none

  private
  public ::                         &
    interactions_factory_options_t, &
    interactions_factory_abst_t

  integer, parameter, public ::   &
    NO_PARTNERS   = -1,            &
    ALL_PARTNERS  = -2,            &
    ONLY_PARTNERS = -3,            &
    ALL_EXCEPT    = -4


  !> @brief type for storing options to be used when creating a given interaction
  !!
  !! There are four possible modes:
  !!  - NO_PARTNERS: do not create this interaction with any partner. This effectively turns the interaction off.
  !!  - ALL_PARTNERS: create this interaction with any partner that supports it.
  !!  - ONLY_PARTNERS: create this interaction only with a given list of partners.
  !!  - ALL_EXCEPT: create this interaction with all partners that support it, excluding the given list of partners.
  !
  type interactions_factory_options_t
    integer :: mode                                               !< interaction creation mode
    integer :: timing                                             !< timing strategy for coupling update
    character(len=MAX_NAMESPACE_LEN), allocatable :: partners(:)  !< list of partner names to be used with ONLY_PARTNERS and ALL_EXCEPT modes.
  end type interactions_factory_options_t

  !> @brief abstract class for interaction factories
  type, abstract :: interactions_factory_abst_t
  contains
    procedure(interactions_factory_abst_create), deferred :: create
    !< @copydoc interactions_factory_abst_create
    procedure(interactions_factory_abst_options), deferred, nopass :: options
    !< @copydoc interactions_factory_abst_options
  end type interactions_factory_abst_t

  abstract interface
    !> @brief Interface for the function to create a specific interaction.
    !!
    !! @note As this function requires explicit knowlegde on the interaction, it is
    !! implemented in the concrete factory class (e.g. interactions_factory_create())
    !
    function interactions_factory_abst_create(this, type, partner) result(interaction)
      import :: interactions_factory_abst_t
      import interaction_partner_t
      import interaction_t
      class(interactions_factory_abst_t),         intent(in)    :: this
      integer,                                    intent(in)    :: type
      class(interaction_partner_t),       target, intent(inout) :: partner
      class(interaction_t),                       pointer       :: interaction
    end function interactions_factory_abst_create

    !> @brief Interface for the function to get the interaction options for a given namespace
    !!
    !! @note As this function requires explicit knowlegde on the interactions, it is
    !! implemented in the concrete factory class (e.g. interactions_factory_options())
    !
    function interactions_factory_abst_options(namespace, interactions) result(options)
      use namespace_oct_m
      import :: interactions_factory_options_t
      type(namespace_t),                    intent(in)  :: namespace
      integer,                              intent(in)  :: interactions(:)
      type(interactions_factory_options_t) :: options(size(interactions))
    end function interactions_factory_abst_options
  end interface

end module interactions_factory_abst_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
