!! Copyright (C) 2019-2020 M. Oliveira, Heiko Appel
!! Copyright (C) 2021 S. Ohlmann
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

!> @brief This module implements the basic mulsisystem class, a container system for other systems
!!
module multisystem_basic_oct_m
  use debug_oct_m
  use global_oct_m
  use io_oct_m
  use multisystem_oct_m
  use namespace_oct_m
  use system_oct_m
  use system_factory_abst_oct_m
  implicit none

  private
  public ::               &
    multisystem_basic_t

  !> @brief Container class for lists of system_oct_m::system_t
  !!
  !! The multisystem_oct_m::multisystem_basic_t class is a special kind of system_oct_m::system_t, which can contain
  !! other multisystems, allowing a hierarchy of grouped systems.
  type, extends(multisystem_t) :: multisystem_basic_t

  contains
    final :: multisystem_basic_finalizer
  end type multisystem_basic_t

  interface multisystem_basic_t
    procedure multisystem_basic_constructor
  end interface multisystem_basic_t

contains

  ! ---------------------------------------------------------------------------------------
  !> @brief initialize a basic multisystem class
  !!
  !! This routine creates the systems contained in the basic multisystem container.
  !!
  recursive function multisystem_basic_constructor(namespace, names, types, factory) result(system)
    type(namespace_t),            intent(in) :: namespace
    class(system_factory_abst_t), intent(in) :: factory
    character(len=128),           intent(in) :: names(:)
    integer,                      intent(in) :: types(:)
    class(multisystem_basic_t),   pointer    :: system

    integer :: is
    class(system_t), pointer :: sys

    PUSH_SUB(multisystem_basic_constructor)

    allocate(system)
    system%namespace = namespace

    ! No interaction directly supported by this system (but classes that extend
    ! it can add their own)
    allocate(system%supported_interactions(0))
    allocate(system%supported_interactions_as_partner(0))

    do is = 1, size(names)
      ! Create folder to store system files.
      ! Needs to be done before creating the system as this in turn might create subfolders.
      call io_mkdir(names(is), namespace=namespace)

      ! Create system
      sys => factory%create(namespace_t(names(is), parent=system%namespace), types(is))

      ! Add system to list of systems
      call system%list%add(sys)
    end do

    POP_SUB(multisystem_basic_constructor)
  end function multisystem_basic_constructor

  ! ---------------------------------------------------------
  recursive subroutine multisystem_basic_finalizer(this)
    type(multisystem_basic_t), intent(inout) :: this

    PUSH_SUB(multisystem_basic_finalizer)

    call multisystem_end(this)

    POP_SUB(multisystem_basic_finalizer)
  end subroutine multisystem_basic_finalizer

end module multisystem_basic_oct_m
