!! Copyright (C) 2024 N. Tancogne-Dejean
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

!>@brief This module implements unit tests for the mixing methods
module mix_tests_oct_m
  use debug_oct_m
  use derivatives_oct_m
  use global_oct_m
  use, intrinsic :: iso_fortran_env
  use messages_oct_m
  use mesh_oct_m
  use mix_oct_m
  use namespace_oct_m
  use profiling_oct_m
  use space_oct_m
  use types_oct_m

  implicit none

  private
  public ::                     &
    mix_tests_run

contains

  subroutine mix_tests_run()
    integer :: iter, max_iter
    real(real64), parameter :: tol = 1e-10_real64
    real(real64), parameter :: solution = 2.0134438978154419300_real64
    real(real64) :: rhoin(1,1), rhoout(1,1)
    type(mix_t) :: smix
    type(mixfield_t), pointer :: mixfield
    type(derivatives_t) :: der
    type(mesh_t), target :: mesh
    type(space_t) :: space

    max_iter = 1000

    ! Fake initializations due to current code design
    der%dim = 1
    der%mesh => mesh
    der%mesh%use_curvilinear = .false.
    der%mesh%volume_element = M_ONE
    der%mesh%parallel_in_domains = .false.
    space = space_t(global_namespace)

    ! Initialization
    call mix_init(smix, global_namespace, space, der, 1, 1, func_type_ = TYPE_FLOAT)
    call mix_get_field(smix, mixfield)

    !Starting point
    rhoin = M_ONE

    write(message(1),'(a, f20.13)') "Starting point : ", rhoin
    call messages_info(1)

    do iter = 1, max_iter
      call mixfield_set_vin(mixfield, rhoin)
      call test_function(rhoin(1,1), rhoout(1,1))

      write(message(1),'(a, i3, a, f20.13)') "At iter. ", iter, " the value is ", rhoout
      call messages_info(1)

      ! Check convergence
      if(abs(rhoout(1,1)-rhoin(1,1)) < tol) exit

      call mixfield_set_vout(mixfield, rhoout)
      call mixing(global_namespace, smix)
      call mixfield_get_vnew(mixfield, rhoin)
      call mixfield_set_vin(mixfield, rhoin)
    end do

    message(1) = ''
    if (iter < max_iter) then
      write(message(2), '(a,f25.15,a,i3,a)') "Computed fixed point       ", rhoout, " after ", iter, " iterations."
    else
      write(message(2), '(a,f25.15,a,i3,a)') "Failure: fixed point       ", rhoout, " after ", iter, " iterations."
    end if
    write(message(3),'(a, f25.15, a )') "Numerical fixed point should be ", solution, " ."
    call messages_info(3)

  end subroutine mix_tests_run

  !> A simple test function
  !!
  !! The solution is 2.0134438978154419300, from Mathematica
  subroutine test_function(x, fx)
    real(real64), intent(in)  :: x
    real(real64), intent(out) :: fx

    fx = sin(x) + atan(x)

  end subroutine test_function

end module mix_tests_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
