!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module xc_oct_m
  use comm_oct_m
  use debug_oct_m
  use derivatives_oct_m
  use distributed_oct_m
  use electron_space_oct_m
  use exchange_operator_oct_m
  use global_oct_m
  use grid_oct_m
  use io_oct_m
  use io_function_oct_m
  use iso_c_binding
  use, intrinsic :: iso_fortran_env
  use kpoints_oct_m
  use lalg_basic_oct_m
  use lalg_adv_oct_m
  use libvdwxc_oct_m
  use math_oct_m
  use mesh_oct_m
  use mesh_function_oct_m
  use messages_oct_m
  use mpi_oct_m
  use namespace_oct_m
  use parser_oct_m
  use poisson_oct_m
  use profiling_oct_m
  use space_oct_m
  use states_elec_oct_m
  use states_elec_dim_oct_m
  use unit_system_oct_m
  use xc_f03_lib_m
#ifdef HAVE_LIBXC_FUNCS
  use xc_f03_funcs_m
#endif
  use xc_fbe_oct_m
  use xc_functional_oct_m
  use xc_interaction_oct_m

  implicit none

  private
  public ::             &
    xc_t,               &
    xc_init,            &
    xc_end,             &
    xc_write_info,      &
    xc_get_vxc,         &
    xc_get_nc_vxc,      &
    xc_get_fxc,         &
    xc_get_kxc,         &
    xc_is_orbital_dependent, &
    xc_is_not_size_consistent, &
    xc_is_energy_functional, &
    family_is_mgga,     &
    family_is_mgga_with_exc, &
    family_is_hybrid,   &
    in_family

  ! A Structure that contains the quantities needed to compute the functionals
  type internal_quantities_t
    real(real64), pointer     :: rho(:,:)       ! A pointer to the full density

    real(real64), allocatable :: dens(:,:)      ! Density (in the local frame of the magnetization)
    real(real64), allocatable :: gdens(:,:,:)   ! Gradient of the density
    real(real64), allocatable :: ldens(:,:)     ! Laplacian of the density
    real(real64), allocatable :: tau(:,:)       ! Kinetic energy density
  end type internal_quantities_t


  type xc_t
    private
    integer,               public :: family              !< the families present
    integer,               public :: flags               !<flags of the xc functional
    integer,               public :: kernel_family
    type(xc_functional_t), public :: functional(2,2)     !< (FUNC_X,:) => exchange,    (FUNC_C,:) => correlation
    !!                                                      (:,1) => unpolarized, (:,2) => polarized

    type(xc_functional_t), public :: kernel(2,2)
    real(real64),          public :: kernel_lrc_alpha  !< long-range correction alpha parameter for kernel in solids

    real(real64), public   :: cam_omega                !< Cam coefficients omega, alpha, beta
    real(real64), public   :: cam_alpha                !< amount of EXX to add for the hybrids
    real(real64), public   :: cam_beta
    real(real64), public   :: cam_ext(3)               !< external CAM parameters to add for the hybrids

    logical         :: use_gi_ked               !< should we use the gauge-independent kinetic energy density?

    integer :: xc_density_correction
    logical :: xcd_optimize_cutoff
    real(real64)   :: xcd_ncutoff
    logical :: xcd_minimum
    logical :: xcd_normalize
    logical :: parallel

    type(internal_quantities_t) :: quantities
  end type xc_t

  real(real64), parameter :: tiny      = 1.0e-12_real64

  integer, parameter :: &
    LR_NONE = 0,        &
    LR_X    = 1

contains

  ! ---------------------------------------------------------
  subroutine xc_write_info(xcs, iunit, namespace)
    type(xc_t),                  intent(in) :: xcs
    integer,           optional, intent(in) :: iunit
    type(namespace_t), optional, intent(in) :: namespace

    integer :: ifunc

    PUSH_SUB(xc_write_info)

    write(message(1), '(a)') "Exchange-correlation:"
    call messages_info(1, iunit=iunit, namespace=namespace)

    do ifunc = FUNC_X, FUNC_C
      call xc_functional_write_info(xcs%functional(ifunc, 1), iunit, namespace)
    end do

    if (abs(xcs%cam_alpha + xcs%cam_beta) > M_EPSILON) then
      write(message(1), '(1x)')
      write(message(2), '(a,f8.5)') "Exact exchange mixing = ", xcs%cam_alpha
      write(message(3), '(a,f8.5)') "Exact exchange for short-range beta = ", xcs%cam_beta
      write(message(4), '(a,f8.5)') "Exact exchange range-separate omega = ", xcs%cam_omega
      call messages_info(4, iunit=iunit, namespace=namespace)
    end if


    POP_SUB(xc_write_info)
  end subroutine xc_write_info


  ! ---------------------------------------------------------
  subroutine xc_init(xcs, namespace, ndim, periodic_dim, nel, x_id, c_id, xk_id, ck_id, hartree_fock, ispin)
    type(xc_t),        intent(out) :: xcs
    type(namespace_t), intent(in)  :: namespace
    integer,           intent(in)  :: ndim
    integer,           intent(in)  :: periodic_dim
    real(real64),      intent(in)  :: nel
    integer,           intent(in)  :: x_id
    integer,           intent(in)  :: c_id
    integer,           intent(in)  :: xk_id
    integer,           intent(in)  :: ck_id
    logical,           intent(in)  :: hartree_fock
    integer,           intent(in)  :: ispin

    integer :: isp, xc_major, xc_minor, xc_micro
    logical :: ll
    type(block_t) :: blk

    PUSH_SUB(xc_init)

    call xc_f03_version(xc_major, xc_minor, xc_micro)

    xcs%family = 0
    xcs%flags  = 0
    xcs%kernel_family = 0

    call parse()

    !we also need XC functionals that do not depend on the current
    !get both spin-polarized and unpolarized

    ! TODO: check that nel should not be spin polarized here
    do isp = 1, 2

      call xc_functional_init(xcs%functional(FUNC_X, isp), namespace, x_id, ndim, nel, isp)
      call xc_functional_init(xcs%functional(FUNC_C, isp), namespace, c_id, ndim, nel, isp)

      call xc_functional_init(xcs%kernel(FUNC_X, isp), namespace, xk_id, ndim, nel, isp)
      call xc_functional_init(xcs%kernel(FUNC_C, isp), namespace, ck_id, ndim, nel, isp)

    end do

    xcs%family = ior(xcs%family, xcs%functional(FUNC_X,1)%family)
    xcs%family = ior(xcs%family, xcs%functional(FUNC_C,1)%family)

    xcs%flags = ior(xcs%flags, xcs%functional(FUNC_X,1)%flags)
    xcs%flags = ior(xcs%flags, xcs%functional(FUNC_C,1)%flags)

    xcs%kernel_family = ior(xcs%kernel_family, xcs%kernel(FUNC_X,1)%family)
    xcs%kernel_family = ior(xcs%kernel_family, xcs%kernel(FUNC_C,1)%family)


    if (xc_is_not_size_consistent(xcs, namespace) .and. periodic_dim > 0) then
      message(1) = "Cannot perform a periodic calculation with a functional"
      message(2) = "that depends on the number of electrons."
      call messages_fatal(2, namespace=namespace)
    end if

    ! Take care of hybrid functionals (they appear in the correlation functional)
    xcs%cam_omega = M_ZERO
    xcs%cam_alpha = M_ZERO
    xcs%cam_beta = M_ZERO

    ll =  (hartree_fock) &
      .or.(xcs%functional(FUNC_X,1)%id == XC_OEP_X) &
      .or. family_is_hybrid(xcs)
    if (ll) then
      if ((xcs%functional(FUNC_X,1)%id /= 0).and.(xcs%functional(FUNC_X,1)%id /= XC_OEP_X)) then
        message(1) = "You cannot use an exchange functional when performing"
        message(2) = "a Hartree-Fock calculation or using a hybrid functional."
        call messages_fatal(2, namespace=namespace)
      end if

      if (periodic_dim == ndim) then
        call messages_experimental("Fock operator (Hartree-Fock, OEP, hybrids) in fully periodic systems", namespace=namespace)
      end if

      ! get the mixing coefficient for hybrids
      if (family_is_hybrid(xcs)) then
        if( any(abs(xcs%cam_ext) > M_EPSILON) ) call set_hybrid_params(xcs,namespace)

        call xc_f03_hyb_cam_coef(xcs%functional(FUNC_C,1)%conf, xcs%cam_omega, &
          xcs%cam_alpha, xcs%cam_beta)
        call xc_f03_hyb_cam_coef(xcs%functional(FUNC_C,2)%conf, xcs%cam_omega, &
          xcs%cam_alpha, xcs%cam_beta)
      else
        ! we are doing Hartree-Fock plus possibly a correlation functional
        xcs%cam_omega = M_ZERO
        xcs%cam_alpha = M_ONE
        xcs%cam_beta = M_ZERO
      end if

      ! reset certain variables
      xcs%functional(FUNC_X,1)%family = XC_FAMILY_OEP
      xcs%functional(FUNC_X,1)%id     = XC_OEP_X
      xcs%functional(FUNC_X,2)%family = XC_FAMILY_OEP
      xcs%functional(FUNC_X,2)%id     = XC_OEP_X
      if (.not. hartree_fock) then
        xcs%family             = ior(xcs%family, XC_FAMILY_OEP)
      end if
    end if

    if (in_family(xcs%family, [XC_FAMILY_LCA])) then
      call messages_not_implemented("LCA current functionals", namespace) ! not even in libxc!
    end if

    call messages_obsolete_variable(namespace, 'MGGAimplementation')
    call messages_obsolete_variable(namespace, 'CurrentInTau', 'XCUseGaugeIndependentKED')

    if (xcs%functional(FUNC_X, 1)%id == XC_MGGA_X_TB09 .and. periodic_dim /= 3) then
      message(1) = "mgga_x_tb09 functional can only be used for 3D periodic systems"
      call messages_fatal(1, namespace=namespace)
    end if

    if (family_is_mgga(xcs%family) .or. family_is_nc_mgga(xcs%family)) then
      !%Variable XCUseGaugeIndependentKED
      !%Type logical
      !%Default yes
      !%Section Hamiltonian::XC
      !%Description
      !% If true, when evaluating the XC functional, a term including the (paramagnetic or total) current
      !% is added to the kinetic-energy density such as to make it gauge-independent.
      !% Applies only to meta-GGA (and hybrid meta-GGA) functionals.
      !%End
      call parse_variable(namespace, 'XCUseGaugeIndependentKED', .true., xcs%use_gi_ked)
    end if

    POP_SUB(xc_init)

  contains

    subroutine parse()

      PUSH_SUB(xc_init.parse)

      ! the values of x_id,  c_id, xk_id, and c_id are read outside the routine

      !%Variable XCKernelLRCAlpha
      !%Type float
      !%Default 0.0
      !%Section Hamiltonian::XC
      !%Description
      !% Set to a non-zero value to add a long-range correction for solids to the kernel.
      !% This is the <math>\alpha</math> parameter defined in S. Botti <i>et al.</i>, <i>Phys. Rev. B</i>
      !% 69, 155112 (2004). The <math>G = G^\prime = 0</math> term <math>-\alpha/q^2</math> is taken
      !% into account by introducing an additional pole to the polarizability (see R. Stubner
      !% <i>et al.</i>, <i>Phys. Rev. B</i> 70, 245119 (2004)). The rest of the terms are included by
      !% multiplying the Hartree term by <math>1 - \alpha / 4 \pi</math>. The use of non-zero
      !% <math>\alpha</math> in combination with <tt>HamiltonianVariation</tt> = <tt>V_ext_only</tt>
      !% corresponds to account of only the <math>G = G^\prime = 0</math> term.
      !% Applicable only to isotropic systems. (Experimental)
      !%End

      call parse_variable(namespace, 'XCKernelLRCAlpha', M_ZERO, xcs%kernel_lrc_alpha)
      if (abs(xcs%kernel_lrc_alpha) > M_EPSILON) then
        call messages_experimental("Long-range correction to kernel", namespace=namespace)
      end if

      !%Variable XCDensityCorrection
      !%Type integer
      !%Default none
      !%Section Hamiltonian::XC::DensityCorrection
      !%Description
      !% This variable controls the long-range correction of the XC
      !% potential using the <a href=http://arxiv.org/abs/1107.4339>XC density representation</a>.
      !%Option none 0
      !% No correction is applied.
      !%Option long_range_x 1
      !% The correction is applied to the exchange potential.
      !%End
      call parse_variable(namespace, 'XCDensityCorrection', LR_NONE, xcs%xc_density_correction)

      if (xcs%xc_density_correction /= LR_NONE) then
        call messages_experimental('XC density correction', namespace=namespace)

        if(ispin /= UNPOLARIZED) then
          call messages_not_implemented('XCDensityCorrection with SpinComponents /= unpolarized', namespace)
        end if

        !%Variable XCDensityCorrectionOptimize
        !%Type logical
        !%Default true
        !%Section Hamiltonian::XC::DensityCorrection
        !%Description
        !% When enabled, the density cutoff will be
        !% optimized to replicate the boundary conditions of the exact
        !% XC potential. If the variable is set to no, the value of
        !% the cutoff must be given by the <tt>XCDensityCorrectionCutoff</tt>
        !% variable.
        !%End
        call parse_variable(namespace, 'XCDensityCorrectionOptimize', .true., xcs%xcd_optimize_cutoff)

        !%Variable XCDensityCorrectionCutoff
        !%Type float
        !%Default 0.0
        !%Section Hamiltonian::XC::DensityCorrection
        !%Description
        !% The value of the cutoff applied to the XC density.
        !%End
        call parse_variable(namespace, 'XCDensityCorrectionCutoff', M_ZERO, xcs%xcd_ncutoff)

        !%Variable XCDensityCorrectionMinimum
        !%Type logical
        !%Default true
        !%Section Hamiltonian::XC::DensityCorrection
        !%Description
        !% When enabled, the cutoff optimization will
        !% return the first minimum of the <math>q_{xc}</math> function if it does
        !% not find a value of -1 (<a href=http://arxiv.org/abs/1107.4339>details</a>).
        !% This is required for atoms or small
        !% molecules, but may cause numerical problems.
        !%End
        call parse_variable(namespace, 'XCDensityCorrectionMinimum', .true., xcs%xcd_minimum)

        !%Variable XCDensityCorrectionNormalize
        !%Type logical
        !%Default true
        !%Section Hamiltonian::XC::DensityCorrection
        !%Description
        !% When enabled, the correction will be
        !% normalized to reproduce the exact boundary conditions of
        !% the XC potential.
        !%End
        call parse_variable(namespace, 'XCDensityCorrectionNormalize', .true., xcs%xcd_normalize)

      end if

      !%Variable ParallelXC
      !%Type logical
      !%Default true
      !%Section Execution::Parallelization
      !%Description
      !% When enabled, additional parallelization
      !% will be used for the calculation of the XC functional.
      !%End
      call messages_obsolete_variable(namespace, 'XCParallel', 'ParallelXC')
      call parse_variable(namespace, 'ParallelXC', .true., xcs%parallel)


      !%Variable HybridCAMParameters
      !%Type block
      !%Section Hamiltonian::XC
      !%Description
      !% This variable specifies the <math>\alpha, \beta, \omega</math> for CAM-type
      !% hybrid functionals. Defaults are zero.
      !%End
      xcs%cam_ext = M_ZERO

      if(parse_block(namespace, 'HybridCamParameters', blk) == 0) then
        do isp = 1, size(xcs%cam_ext)
          call parse_block_float(blk, 0, isp - 1, xcs%cam_ext(isp))
        end do
        call parse_block_end(blk)
      end if

      if( any(abs(xcs%cam_ext) > M_EPSILON) ) then
        write(message(1), '(1x)')
        write(message(2), '(a,f8.5)') "Info: Setting external cam_alpha = ", xcs%cam_ext(1)
        write(message(3), '(a,f8.5)') "Info: Setting external cam_beta  = ", xcs%cam_ext(2)
        write(message(4), '(a,f8.5)') "Info: Setting external cam_omega = ", xcs%cam_ext(3)
        call messages_info(4, namespace=namespace)
      end if

      POP_SUB(xc_init.parse)
    end subroutine parse

  end subroutine xc_init


  ! ---------------------------------------------------------
  subroutine xc_end(xcs)
    type(xc_t), intent(inout) :: xcs

    integer :: isp

    PUSH_SUB(xc_end)

    do isp = 1, 2
      call xc_functional_end(xcs%functional(FUNC_X, isp))
      call xc_functional_end(xcs%functional(FUNC_C, isp))
      call xc_functional_end(xcs%kernel(FUNC_X, isp))
      call xc_functional_end(xcs%kernel(FUNC_C, isp))
    end do
    xcs%family = 0
    xcs%flags  = 0

    POP_SUB(xc_end)
  end subroutine xc_end

  ! ---------------------------------------------------------
  !>@brief Is the xc family orbital dependent
  !!
  !! Note that at the moment this routine consider all mGGAs to be orbital dependent
  !! while it should be formally only the one having a dependence in vtau
  logical pure function xc_is_orbital_dependent(xcs)
    type(xc_t), intent(in) :: xcs

    xc_is_orbital_dependent = family_is_hybrid(xcs) .or. &
      in_family(xcs%functional(FUNC_X,1)%family, [XC_FAMILY_OEP]) .or. &
      in_family(xcs%family, [XC_FAMILY_MGGA, XC_FAMILY_NC_MGGA])

  end function xc_is_orbital_dependent

  ! ---------------------------------------------------------
  !>@brief Is the xc function part of the GGA family
  pure logical function family_is_gga(family, only_collinear)
    integer,           intent(in) :: family
    logical, optional, intent(in) :: only_collinear

    if(optional_default(only_collinear, .false.)) then
      family_is_gga = in_family(family, [XC_FAMILY_GGA, XC_FAMILY_HYB_GGA, &
        XC_FAMILY_MGGA, XC_FAMILY_HYB_MGGA, XC_FAMILY_LIBVDWXC])
    else
      family_is_gga = in_family(family, [XC_FAMILY_GGA, XC_FAMILY_HYB_GGA, &
        XC_FAMILY_MGGA, XC_FAMILY_HYB_MGGA, XC_FAMILY_LIBVDWXC, XC_FAMILY_NC_MGGA])
    end if
  end function  family_is_gga

  !----------------------------------------------------------------------
  !>@brief Is the xc family internally supported by Octopus
  !!
  !! This is used internally to check if we need to call libxc or not
  !! OEP functionals are treated separately.
  pure logical function family_is_supported(family)
    integer, intent(in) :: family

    family_is_supported = in_family(family, [XC_FAMILY_LDA, XC_FAMILY_HYB_LDA, XC_FAMILY_GGA, XC_FAMILY_HYB_GGA, &
      XC_FAMILY_MGGA, XC_FAMILY_HYB_MGGA, XC_FAMILY_LIBVDWXC])
  end function  family_is_supported

  ! ---------------------------------------------------------
  !>@brief Is the xc function part of the mGGA family
  pure logical function family_is_mgga(family, only_collinear)
    integer, optional, intent(in) :: family
    logical, optional, intent(in) :: only_collinear

    if(optional_default(only_collinear, .false.)) then
      family_is_mgga = in_family(family, [XC_FAMILY_MGGA, XC_FAMILY_HYB_MGGA])
    else
      family_is_mgga = in_family(family, [XC_FAMILY_MGGA, XC_FAMILY_HYB_MGGA, XC_FAMILY_NC_MGGA])
    end if
  end function family_is_mgga

  ! ---------------------------------------------------------
  !>@brief Is the xc function part of the mGGA family with an energy functional
  logical pure function family_is_mgga_with_exc(xcs)
    type(xc_t), intent(in) :: xcs

    integer :: ixc

    family_is_mgga_with_exc = .false.
    do ixc = 1, 2
      if (in_family(xcs%functional(ixc, 1)%family, [XC_FAMILY_MGGA, XC_FAMILY_HYB_MGGA, XC_FAMILY_NC_MGGA]) &
        .and. xc_functional_is_energy_functional(xcs%functional(ixc, 1))) then
        family_is_mgga_with_exc = .true.
      end if
    end do
  end function family_is_mgga_with_exc

  !>@brief Returns true if the functional is an hybrid functional
  logical pure function family_is_hybrid(xcs)
    type(xc_t), intent(in) :: xcs

    integer :: ixc

    family_is_hybrid = .false.
    do ixc = 1, 2
      if (in_family(xcs%functional(ixc, 1)%family, [XC_FAMILY_HYB_LDA, XC_FAMILY_HYB_GGA, XC_FAMILY_HYB_MGGA])) then
        family_is_hybrid = .true.
      end if
    end do
  end function family_is_hybrid

  pure logical function in_family(family, xc_families)
    integer, intent(in) :: family
    integer, intent(in) :: xc_families(:)

    in_family = bitand(family, sum(xc_families)) /= 0
  end function in_family

  !  ---------------------------------------------------------
  !> make a local copy with the correct memory order for libxc
  subroutine copy_global_to_local(global, local, n_block, nspin, ip)
    real(real64),  intent(in)  :: global(:,:)
    real(real64),  intent(out) :: local(:,:)
    integer,       intent(in)  :: n_block
    integer,       intent(in)  :: nspin
    integer,       intent(in)  :: ip

    integer :: ib, is

    PUSH_SUB(copy_global_to_local)

    do is = 1, nspin
      !$omp parallel do
      do ib = 1, n_block
        local(is, ib) = global(ib + ip - 1, is)
      end do
    end do

    POP_SUB(copy_global_to_local)
  end subroutine copy_global_to_local

  ! ---------------------------------------------------------
  subroutine copy_local_to_global(local, global, n_block, spin_channels, ip)
    real(real64),   intent(in)    :: local(:,:)
    real(real64),   intent(inout) :: global(:,:)
    integer,        intent(in)    :: n_block
    integer,        intent(in)    :: spin_channels
    integer,        intent(in)    :: ip

    integer :: ib, is

    PUSH_SUB(xc_compute_vxc.copy_local_to_global)

    do is = 1, spin_channels
      !$omp parallel do
      do ib = 1, n_block
        global(ib + ip - 1, is) = global(ib + ip - 1, is) + local(is, ib)
      end do
    end do

    POP_SUB(xc_compute_vxc.copy_local_to_global)
  end subroutine copy_local_to_global

  ! ---------------------------------------------------------
  !>@brief Sets external parameters for some hybrid functionals
  subroutine set_hybrid_params(xcs,namespace)
    type(namespace_t), intent(in)     :: namespace
    type(xc_t),        intent(inout)  :: xcs

    real(real64) :: parameters(3)

    PUSH_SUB(set_hybrid_params)

    parameters =  xcs%cam_ext
    xcs%cam_alpha = parameters(1)

    select case(xcs%functional(FUNC_C,1)%id)
    case(XC_HYB_GGA_XC_PBEH, XC_HYB_LDA_XC_LDA0) ! original PBE0/LDA0 in libxc
      ! If the value is negative, we use the default value of PBE0/LDA0
      if(parameters(1) < M_ZERO) parameters(1) = 0.25_real64

      call xc_f03_func_set_ext_params(xcs%functional(FUNC_C,1)%conf, parameters)
      call xc_f03_func_set_ext_params(xcs%functional(FUNC_C,2)%conf, parameters)

      write(message(1), '(a,f6.3,a)') 'Info: Setting mixing parameter (' , parameters(1) ,').'
      call messages_info(1)

    case(XC_HYB_GGA_XC_CAM_PBEH, XC_HYB_LDA_XC_CAM_LDA0)
      xcs%cam_beta = parameters(2)
      xcs%cam_omega = parameters(3)
      ! check parameters
      call xc_f03_hyb_cam_coef(xcs%functional(FUNC_C,1)%conf, xcs%cam_omega, &
        xcs%cam_alpha, xcs%cam_beta)
      call xc_f03_hyb_cam_coef(xcs%functional(FUNC_C,2)%conf, xcs%cam_omega, &
        xcs%cam_alpha, xcs%cam_beta)
      write(message(1), '(a,f6.3,a)') 'Info: Setting alpha parameter (' , xcs%cam_alpha ,').'
      write(message(2), '(a,f6.3,a)') 'Info: Setting  beta parameter (' , xcs%cam_beta ,').'
      write(message(3), '(a,f6.3,a)') 'Info: Setting omega parameter (' , xcs%cam_omega ,').'
      call messages_info(3)

    case default
      ASSERT(.false.)
    end select

    POP_SUB(set_hybrid_params)
  end subroutine set_hybrid_params

  ! ---------------------------------------------------------
  !>@brief Is one of the x or c functional is not size consistent
  logical function xc_is_not_size_consistent(xcs, namespace)
    type(xc_t),        intent(in) :: xcs
    type(namespace_t), intent(in) :: namespace

    xc_is_not_size_consistent = xc_functional_is_not_size_consistent(xcs%functional(FUNC_X,1), namespace) &
      .or. xc_functional_is_not_size_consistent(xcs%functional(FUNC_C,1), namespace)
  end function xc_is_not_size_consistent

  ! ---------------------------------------------------------
  !>@brief Is one of the x or c functional is not an energy functional
  logical pure function xc_is_energy_functional(xcs)
    type(xc_t), intent(in)  :: xcs
    xc_is_energy_functional = xc_functional_is_energy_functional(xcs%functional(FUNC_X,1)) &
      .or. xc_functional_is_energy_functional(xcs%functional(FUNC_C,1))
  end function xc_is_energy_functional

#include "xc_vxc_inc.F90"
#include "xc_fxc_inc.F90"
#include "xc_kxc_inc.F90"

#include "xc_vxc_nc_inc.F90"

end module xc_oct_m


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
