!! Copyright (C) 2020-2021 N. Tancogne-Dejean
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

!> This routines is similar to xc_get_vxc but for noncollinear functionals, which are not implemented in libxc
subroutine xc_get_nc_vxc(gr, xcs, st, kpoints, space, namespace, rho, vxc, ex, ec, vtau, ex_density, ec_density)
  type(grid_t),                  intent(in)    :: gr
  type(xc_t), target,            intent(inout) :: xcs           !< Details about the xc functional used
  type(states_elec_t),           intent(in)    :: st            !< State of the system (wavefunction,eigenvalues...)
  type(kpoints_t),               intent(in)    :: kpoints
  type(space_t),                 intent(in)    :: space
  type(namespace_t),             intent(in)    :: namespace
  real(real64),                  intent(in)    :: rho(:, :)     !< Electronic density
  real(real64), contiguous,      intent(inout) :: vxc(:,:)      !< XC potential
  real(real64), optional,        intent(inout) :: ex            !< Exchange energy.
  real(real64), optional,        intent(inout) :: ec            !< Correlation energy.
  real(real64), optional,        intent(inout) :: vtau(:,:)     !< Derivative wrt (two times kinetic energy density)
  real(real64), optional, target, intent(out)  :: ex_density(:) !< The exchange energy density
  real(real64), optional, target, intent(out)  :: ec_density(:) !< The correlation energy density

  PUSH_SUB_WITH_PROFILE(xc_get_nc_vxc)

  call xc_nc_update_internal_quantities(gr, xcs, st, kpoints, space, namespace, rho, &
    xcs%quantities)

  call xc_compute_vxc_nc(gr, xcs, st, namespace, xcs%quantities, &
    vxc, ex, ec, vtau, ex_density, ec_density)

  call xc_release_internal_quantities(xcs%quantities)

  POP_SUB_WITH_PROFILE(xc_get_nc_vxc)
end subroutine xc_get_nc_vxc

! -----------------------------------------------------
!> Given a functional, we are updating/transfering/computing the needed internal quantities
!> that are later needed to perform an actual call to xc_compute_vxc
! -----------------------------------------------------
subroutine xc_nc_update_internal_quantities(gr, xcs, st, kpoints, space, namespace, rho, quantities)
  type(grid_t),                intent(in)    :: gr              !< Discretization and the derivative operators and details
  type(xc_t), target,          intent(inout) :: xcs             !< Details about the xc functional used
  type(states_elec_t),         intent(in)    :: st              !< State of the system (wavefunction,eigenvalues...)
  type(kpoints_t),             intent(in)    :: kpoints
  type(space_t),               intent(in)    :: space
  type(namespace_t),           intent(in)    :: namespace
  real(real64), target,               intent(in)    :: rho(:,:)
  type(internal_quantities_t), intent(inout) :: quantities       !< Internal quantities

  integer :: families, ip, ixc
  type(xc_functional_t), pointer :: functl(:)
  logical :: mgga

  PUSH_SUB(xc_nc_update_internal_quantities)

  functl => xcs%functional(:, 1)

  ! is there anything to do ?
  families = XC_FAMILY_NC_LDA + XC_FAMILY_NC_MGGA
  if(bitand(xcs%family, families) == 0) then
    POP_SUB(xc_nc_update_internal_quantities)
    return
  end if

  do ixc = 1, 2
    if(functl(ixc)%family /= XC_FAMILY_NONE) then
      ASSERT(bitand(functl(ixc)%flags, XC_FLAGS_HAVE_VXC) /= 0)
    end if
  end do

  ! initialize a couple of handy variables
  mgga = family_is_nc_mgga(xcs%family)

  SAFE_ALLOCATE(quantities%dens(1:gr%np, 1:4))
  !$omp parallel do
  do ip = 1, gr%np
    quantities%dens(ip, 1) = max(rho(ip, 1), M_ZERO)
    quantities%dens(ip, 2) = max(rho(ip, 2), M_ZERO)
    quantities%dens(ip, 3) = rho(ip, 3)
    quantities%dens(ip, 4) = rho(ip, 4)
  end do

  ! Not used for noncollinear functionals
  nullify(quantities%rho)

  if(mgga) then
    SAFE_ALLOCATE(quantities%gdens(1:gr%np, 1:space%dim, 1:4))
    SAFE_ALLOCATE(quantities%tau(1:gr%np, 1:4))
    SAFE_ALLOCATE(quantities%ldens(1:gr%np, 1:4))

    if(allocated(st%rho_core)) then
      call messages_not_implemented("MGGA with nonlinear core correction", namespace=namespace)
    end if

    if (xcs%use_gi_ked) then
      call states_elec_calc_quantities(gr, st, kpoints, .true., gi_kinetic_energy_density = quantities%tau, &
        density_gradient = quantities%gdens, density_laplacian = quantities%ldens)
    else
      call states_elec_calc_quantities(gr, st, kpoints, .true., kinetic_energy_density = quantities%tau, &
        density_gradient = quantities%gdens, density_laplacian = quantities%ldens)
    end if
  end if

  POP_SUB(xc_nc_update_internal_quantities)
end subroutine xc_nc_update_internal_quantities



!> This routines is similar to xc_update_internal_quantities but for noncollinear functionals, which are not implemented in libxc
subroutine xc_compute_vxc_nc(gr, xcs, st, namespace, quantities, vxc, ex, ec, vtau, ex_density, ec_density)
  type(grid_t),                intent(in)    :: gr
  type(xc_t), target,          intent(inout) :: xcs             !< Details about the xc functional used
  type(states_elec_t),         intent(in)    :: st              !< State of the system (wavefunction,eigenvalues...)
  type(namespace_t),           intent(in)    :: namespace
  type(internal_quantities_t), intent(inout) :: quantities       !< Internal quantities
  real(real64), contiguous,    intent(inout) :: vxc(:,:)        !< XC potential
  real(real64), optional,             intent(inout) :: ex              !< Exchange energy.
  real(real64), optional,             intent(inout) :: ec              !< Correlation energy.
  real(real64), contiguous, optional, intent(inout) :: vtau(:,:)       !< Derivative wrt (two times kinetic energy density)
  real(real64), optional, target,     intent(out)   :: ex_density(:)   !< The exchange energy density
  real(real64), optional, target,     intent(out)   :: ec_density(:)   !< The correlation energy density


  integer, parameter :: N_BLOCK_MAX = 10000
  integer :: n_block

  ! Local blocks (with the correct memory order for libxc):
  !  Input quantities
  real(real64), allocatable :: l_dens(:,:)     ! Density
  real(real64), allocatable :: l_sigma(:,:)    ! Modulus squared of the gradient of the density
  real(real64), allocatable :: l_ldens(:,:)    ! Laplacian of the density
  real(real64), allocatable :: l_tau(:,:)      ! Kinetic energy density
  !  Energy
  real(real64), allocatable :: l_zk(:)
  !  First order (functional) derivatives
  real(real64), allocatable :: l_dedd(:,:)
  real(real64), allocatable :: l_vsigma(:,:)
  real(real64), allocatable :: l_dedldens(:,:)
  real(real64), allocatable :: l_dedtau(:,:)

  !  Energies
  real(real64), pointer :: ex_per_vol(:)  ! Exchange energy per unit volume
  real(real64), pointer :: ec_per_vol(:)  ! Correlation energy per unit volume
  !  First order (functional) derivatives
  real(real64), allocatable :: dedd(:,:)      ! Derivative of the exchange or correlation energy wrt the density
  real(real64), allocatable :: dedgd(:,:,:)   ! Derivative of the exchange or correlation energy wrt the gradient of the density
  real(real64), allocatable :: dedldens(:,:)  ! Derivative of the exchange or correlation energy wrt the laplacian of the density

  integer :: ib, ip, isp, families, ixc, idir, ipstart, ipend
  real(real64)   :: energy(1:2)
  logical :: mgga, mgga_withexc
  logical :: calc_energy
  type(xc_functional_t), pointer :: functl(:)
  type(distributed_t) :: distribution

  PUSH_SUB(xc_compute_vxc_nc)

  nullify(ex_per_vol)
  nullify(ec_per_vol)

  ASSERT(present(ex) .eqv. present(ec))
  calc_energy = present(ex) .or. present(ex_density) .or. present(ec_density)

  functl => xcs%functional(:, 1)

  ! is there anything to do ?
  families = XC_FAMILY_NC_LDA + XC_FAMILY_NC_MGGA
  if(bitand(xcs%family, families) == 0) then
    POP_SUB(xc_compute_vxc_nc)
    return
  end if

  do ixc = 1, 2
    if(functl(ixc)%family /= XC_FAMILY_NONE) then
      ASSERT(bitand(functl(ixc)%flags, XC_FLAGS_HAVE_VXC) /= 0)
    end if
  end do

  ! initialize a couple of handy variables
  mgga = family_is_nc_mgga(xcs%family)
  mgga_withexc = family_is_mgga_with_exc(xcs)
  if(mgga_withexc) then
    ASSERT(present(vtau))
  end if

  call nc_lda_init()
  if(mgga) then
    call nc_gga_init()
    call nc_mgga_init()
  end if

  if(xcs%parallel) then
    call distributed_init(distribution, gr%np, st%st_kpt_mpi_grp%comm)
    ipstart = distribution%start
    ipend = distribution%end
  else
    ipstart = 1
    ipend = gr%np
  end if

  call local_allocate()

  space_loop: do ip = ipstart, ipend, N_BLOCK_MAX

    call space_loop_init(ip, ipend, n_block)

    ! Calculate the potential/gradient density in local reference frame.
    functl_loop: do ixc = FUNC_X, FUNC_C

      if(functl(ixc)%family == XC_FAMILY_NONE) cycle

      if(calc_energy .and. xc_functional_is_energy_functional(functl(ixc))) then
        ! we get the xc energy and potential
        select case(functl(ixc)%family)
          !Requires development of a noncollinear LDA functional.
        case(XC_FAMILY_NC_LDA)
          ASSERT(.false.)

        case(XC_FAMILY_NC_MGGA)
          call nc_mgga_exc_vxc(functl(ixc), namespace, n_block, l_dens, l_sigma, l_ldens, l_tau, &
            l_dedd, l_vsigma, l_dedldens, l_dedtau, l_zk)

        case default
          cycle
        end select

      else ! we just get the potential
        l_zk(:) = M_ZERO

        select case(functl(ixc)%family)
          !Requires development of a noncollinear LDA functional.
        case(XC_FAMILY_NC_LDA)
          ASSERT(.false.)

        case(XC_FAMILY_NC_MGGA)
          call nc_mgga_exc_vxc(functl(ixc), namespace, n_block, l_dens, l_sigma, l_ldens, l_tau, &
            l_dedd, l_vsigma, l_dedldens, l_dedtau)

        case default
          cycle
        end select

      end if

      if(calc_energy) then
        if(functl(ixc)%type == XC_EXCHANGE) then
          do ib = 1, n_block
            ex_per_vol(ib + ip - 1) = ex_per_vol(ib + ip - 1) + sum(l_dens(1:2, ib))*l_zk(ib)
          end do
        else
          do ib = 1, n_block
            ec_per_vol(ib + ip - 1) = ec_per_vol(ib + ip - 1) + sum(l_dens(1:2, ib))*l_zk(ib)
          end do
        end if
      end if

      call copy_local_to_global(l_dedd, dedd, n_block, 4, ip)

      if(family_is_mgga(functl(ixc)%family)) then
        do ib = 1, n_block
          dedgd(ib+ip-1,:,1) = dedgd(ib+ip-1,:,1) + M_TWO*l_vsigma(1, ib)*quantities%gdens(ib + ip - 1,:,1) &
            + l_vsigma(4, ib)*quantities%gdens(ib + ip - 1,:,2)
          dedgd(ib+ip-1,:,2) = dedgd(ib+ip-1,:,2) + M_TWO*l_vsigma(2, ib)*quantities%gdens(ib + ip - 1,:,2) &
            + l_vsigma(4, ib)*quantities%gdens(ib + ip - 1,:,1)
          dedgd(ib+ip-1,:,3) = dedgd(ib+ip-1,:,3) + l_vsigma(3, ib)*quantities%gdens(ib + ip - 1,:,3)
          dedgd(ib+ip-1,:,4) = dedgd(ib+ip-1,:,4) + l_vsigma(3, ib)*quantities%gdens(ib + ip - 1,:,4)
        end do
      end if

      if(family_is_mgga(functl(ixc)%family)) then
        call copy_local_to_global(l_dedldens, dedldens, n_block, 4, ip)
        if(xc_functional_is_energy_functional(functl(ixc))) &
          call copy_local_to_global(l_dedtau, vtau, n_block, 4, ip)
      end if


    end do functl_loop
  end do space_loop

  call local_deallocate()

  ! calculate the energy, we do the integrals directly so when the data
  ! is fully distributed we do not need to allgather first
  if(present(ex) .or. present(ec)) then

    energy(1:2) = M_ZERO

    if(gr%use_curvilinear) then
      do ip = ipstart, ipend
        energy(1) = energy(1) + ex_per_vol(ip)*gr%vol_pp(ip)
        energy(2) = energy(2) + ec_per_vol(ip)*gr%vol_pp(ip)
      end do
    else
      do ip = ipstart, ipend
        energy(1) = energy(1) + ex_per_vol(ip)
        energy(2) = energy(2) + ec_per_vol(ip)
      end do
    end if

    energy(1:2) = energy(1:2)*gr%volume_element

    if(xcs%parallel) then
      call comm_allreduce(st%dom_st_kpt_mpi_grp, energy)
    else if(gr%parallel_in_domains) then
      call comm_allreduce(gr%mpi_grp, energy)
    end if

    ex = ex + energy(1)
    ec = ec + energy(2)
  end if

  if(xcs%parallel) then
    if(distribution%parallel) then
      call profiling_in(TOSTRING(X(XC_GATHER)))

      do isp = 1, 4
        call distributed_allgather(distribution, dedd(:, isp))
      end do

      if(mgga) then
        do idir = 1, gr%der%dim
          do isp = 1, 4
            call distributed_allgather(distribution, dedgd(:, idir, isp))
          end do
        end do

        do isp = 1, 4
          call distributed_allgather(distribution, dedldens(:, isp))
          if(mgga_withexc) then
            call distributed_allgather(distribution, vtau(:, isp))
          end if
        end do
      end if

      call profiling_out(TOSTRING(X(XC_GATHER)))
    end if

    call distributed_end(distribution)
  end if

  ! this has to be done in inverse order
  if(mgga) then
    call nc_mgga_process()
    call nc_gga_process()
  end if
  call nc_lda_process()

  ! clean up allocated memory
  call nc_lda_end()
  if(mgga) then
    call nc_gga_end()
    call nc_mgga_end()
  end if

  POP_SUB(xc_compute_vxc_nc)

contains

  ! ---------------------------------------------------------
  subroutine space_loop_init(ip, np, nblock)
    integer, intent(in)  :: ip
    integer, intent(in)  :: np
    integer, intent(out) :: nblock

    PUSH_SUB(xc_get_nc_vxc.space_loop_init)

    !Resize the dimension of the last block when the number of the mesh points
    !it is not a perfect divisor of the dimension of the blocks.
    nblock = min(np - ip + 1, N_BLOCK_MAX)

    ! make a local copy with the correct memory order for libxc
    call copy_global_to_local(quantities%dens, l_dens, nblock, 4, ip)

    if(mgga) then
      do ib = 1, nblock
        l_sigma(1, ib) = sum(quantities%gdens(ib + ip - 1, 1:gr%der%dim, 1)**2)
        l_sigma(2, ib) = sum(quantities%gdens(ib + ip - 1, 1:gr%der%dim, 2)**2)
        l_sigma(3, ib) = sum(quantities%gdens(ib + ip - 1, 1:gr%der%dim, 3)**2 + quantities%gdens(ib + ip - 1, 1:gr%der%dim, 4)**2)
        l_sigma(4, ib) = sum(quantities%gdens(ib + ip - 1, 1:gr%der%dim, 1)*quantities%gdens(ib + ip - 1, 1:gr%der%dim, 2))
      end do

      call copy_global_to_local(quantities%tau, l_tau, nblock, 4, ip)

      call copy_global_to_local(quantities%ldens, l_ldens, nblock, 4, ip)
    end if

    POP_SUB(xc_get_nc_vxc.space_loop_init)
  end subroutine space_loop_init

  ! ---------------------------------------------------------
  !> Takes care of the initialization of the LDA part of the functionals
  !!   *) allocates density and dedd, and their local variants
  !!   *) calculates the density taking into account nlcc and non-collinear spin
  subroutine nc_lda_init()

    PUSH_SUB(xc_get_nc_vxc.nc_lda_init)

    ! allocate some general arrays
    if(calc_energy) then
      if(present(ex_density)) then
        ex_per_vol => ex_density
      else
        SAFE_ALLOCATE(ex_per_vol(1:gr%np))
      end if
      if(present(ec_density)) then
        ec_per_vol => ec_density
      else
        SAFE_ALLOCATE(ec_per_vol(1:gr%np))
      end if
      ex_per_vol = M_ZERO
      ec_per_vol = M_ZERO
    end if

    SAFE_ALLOCATE(dedd(1:gr%np_part, 1:4))
    dedd(1:gr%np, 1:4) = M_ZERO

    POP_SUB(xc_get_nc_vxc.nc_lda_init)
  end subroutine nc_lda_init

  ! ---------------------------------------------------------
  !> deallocate variables allocated in nc_lda_init
  subroutine nc_lda_end()
    PUSH_SUB(xc_get_nc_vxc.nc_lda_end)

    if(.not. present(ex_density)) then
      SAFE_DEALLOCATE_P(ex_per_vol)
    end if
    if(.not. present(ec_density)) then
      SAFE_DEALLOCATE_P(ec_per_vol)
    end if
    SAFE_DEALLOCATE_A(dedd)

    POP_SUB(xc_get_nc_vxc.nc_lda_end)
  end subroutine nc_lda_end

  ! ---------------------------------------------------------
  !> calculates the LDA part of vxc
  subroutine nc_lda_process()
    integer :: is

    PUSH_SUB(xc_get_nc_vxc.nc_lda_process)

    do is = 1, 4
      call lalg_axpy(gr%np, M_ONE, dedd(:,is), vxc(:,is))
    end do

    POP_SUB(xc_get_nc_vxc.nc_lda_process)
  end subroutine nc_lda_process

  ! ---------------------------------------------------------
  !> initialize GGAs
  !!   *) allocates gradient of the density (gdens), dedgd, and its local variants
  subroutine nc_gga_init()
    PUSH_SUB(xc_get_nc_vxc.nc_gga_init)

    SAFE_ALLOCATE(dedgd(1:gr%np_part, 1:gr%der%dim, 1:4))
    dedgd = M_ZERO

    POP_SUB(xc_get_nc_vxc.nc_gga_init)
  end subroutine nc_gga_init


  ! ---------------------------------------------------------
  !> cleans up memory allocated in gga_init
  subroutine nc_gga_end()
    PUSH_SUB(xc_get_nc_vxc.nc_gga_end)

    SAFE_DEALLOCATE_A(dedgd)

    POP_SUB(xc_get_nc_vxc.nc_gga_end)
  end subroutine nc_gga_end

  ! ---------------------------------------------------------
  !> calculates the GGA contribution to vxc
  subroutine nc_gga_process()
    integer :: is
    real(real64), allocatable :: gf(:)

    PUSH_SUB(xc_get_nc_vxc.nc_gga_process)

    ! subtract the divergence of the functional derivative of Exc with respect to
    ! the gradient of the density.
    SAFE_ALLOCATE(gf(1:gr%np))
    do is = 1, 4
      call dderivatives_div(gr%der, dedgd(:, :, is), gf(:))
      call lalg_axpy(gr%np, -M_ONE, gf, dedd(:, is))
    end do
    SAFE_DEALLOCATE_A(gf)

    POP_SUB(xc_get_nc_vxc.nc_gga_process)
  end subroutine nc_gga_process

  ! ---------------------------------------------------------
  !> initialize meta-GGAs
  !!   *) allocate the kinetic-energy density, dedtau, and local variants
  subroutine nc_mgga_init()
    PUSH_SUB(xc_get_nc_vxc.nc_mgga_init)

    ! allocate variables
    SAFE_ALLOCATE(dedldens(1:gr%np_part, 1:4))
    dedldens = M_ZERO

    POP_SUB(xc_get_nc_vxc.nc_mgga_init)
  end subroutine nc_mgga_init

  ! ---------------------------------------------------------
  !> clean up memory allocated in nc_mgga_init
  subroutine nc_mgga_end()
    PUSH_SUB(xc_get_nc_vxc.nc_mgga_end)

    SAFE_DEALLOCATE_A(dedldens)

    POP_SUB(xc_get_nc_vxc.nc_mgga_end)
  end subroutine nc_mgga_end

  ! ---------------------------------------------------------
  !> calculate the mgga contribution to vxc
  subroutine nc_mgga_process()
    integer :: is
    real(real64), allocatable :: lf(:)

    PUSH_SUB(xc_get_nc_vxc.nc_mgga_process)

    ! add the Laplacian of the functional derivative of Exc with respect to
    ! the laplacian of the density.

    SAFE_ALLOCATE(lf(1:gr%np))
    do is = 1, 4
      call dderivatives_lapl(gr%der, dedldens(:, is), lf)
      call lalg_axpy(gr%np, M_ONE, lf, dedd(:, is))
    end do
    SAFE_DEALLOCATE_A(lf)

    POP_SUB(xc_get_nc_vxc.nc_mgga_process)
  end subroutine nc_mgga_process


  ! ---------------------------------------------------------
  !> THREADSAFE (no SAFE ALLOCATE or PUSH/POP SUB)
  subroutine local_allocate()

    allocate(l_dens(1:4, 1:N_BLOCK_MAX))
    allocate(l_zk(1:N_BLOCK_MAX))

    if(mgga) then
      allocate(l_sigma   (1:4, 1:N_BLOCK_MAX))
      allocate(l_tau     (1:4, 1:N_BLOCK_MAX))
      allocate(l_ldens   (1:4, 1:N_BLOCK_MAX))
      allocate(l_dedd    (1:4, 1:N_BLOCK_MAX))
      allocate(l_vsigma  (1:4, 1:N_BLOCK_MAX))
      allocate(l_dedldens(1:4, 1:N_BLOCK_MAX))
      allocate(l_dedtau  (1:4, 1:N_BLOCK_MAX))
    end if

  end subroutine local_allocate

  ! ---------------------------------------------------------
  !> THREADSAFE (no SAFE ALLOCATE or PUSH/POP SUB)
  subroutine local_deallocate()

    deallocate(l_dens)
    deallocate(l_zk)
    deallocate(l_dedd)

    if(mgga) then
      deallocate(l_sigma)
      deallocate(l_tau)
      deallocate(l_ldens)
      deallocate(l_vsigma)
      deallocate(l_dedldens)
      deallocate(l_dedtau)
    end if

  end subroutine local_deallocate

end subroutine xc_compute_vxc_nc


! -----------------------------------------------------
!>@brief Returns true is the functional is a noncollinear functional
pure logical function family_is_nc_mgga(family)
  integer, intent(in) :: family

  family_is_nc_mgga = bitand(family, XC_FAMILY_NC_MGGA) /= 0
end function family_is_nc_mgga

! -----------------------------------------------------
!> Interface between the generic call to get the energy and potential and the calls to each specific functionals.
subroutine  nc_mgga_exc_vxc(functl, namespace, n_block, l_dens, l_sigma, l_ldens, l_tau, &
  l_dedd, l_vsigma, l_deddldens, l_dedtau, l_zk)
  type(xc_functional_t), intent(in):: functl
  type(namespace_t), intent(in)    :: namespace
  integer,           intent(in)    :: n_block
  real(real64),             intent(in)    :: l_dens(:,:)     ! Density
  real(real64),             intent(in)    :: l_sigma(:,:)    ! Modulus squared of the gradient of the density
  real(real64),             intent(in)    :: l_ldens(:,:)    ! Laplacian of the density
  real(real64),             intent(in)    :: l_tau(:,:)      ! Kinetic energy density
  real(real64),             intent(out)   :: l_dedd(:,:)     ! Derivative of the energy versus l_dens
  real(real64),             intent(out)   :: l_vsigma(:,:)   ! Derivative of the energy versus l_sigma
  real(real64),             intent(out)   :: l_deddldens(:,:)! Derivative of the energy versus the l_dens
  real(real64),             intent(out)   :: l_dedtau(:,:)   ! Derivative of the energy versus l_tau
  real(real64), optional,   intent(out)   :: l_zk(:)         ! Energy density

  integer :: ib
  real(real64) :: gamma
  real(real64) :: my_sigma(4)
  logical :: explicit

  PUSH_SUB(nc_mgga_exc_vxc)

  if(present(l_zk)) then
    ASSERT(xc_functional_is_energy_functional(functl))
  end if

  select case(functl%id)
  case(XC_MGGA_X_NC_BR, XC_MGGA_X_NC_BR_1, XC_MGGA_X_NC_BR_EXPLICIT)

    if(functl%id == XC_MGGA_X_NC_BR .or. functl%id == XC_MGGA_X_NC_BR_EXPLICIT) then
      gamma = 0.8_real64
    else
      gamma = M_ONE
    end if

    explicit = .false.
    if (functl%id == XC_MGGA_X_NC_BR_EXPLICIT) explicit = .true.

    do ib = 1, n_block

      ! We enforce Schwartz inequality
      my_sigma = l_sigma(:,ib)
      my_sigma(1) = min(my_sigma(1), M_FOUR*l_dens(1,ib)*l_tau(1,ib))
      my_sigma(2) = min(my_sigma(2), M_FOUR*l_dens(2,ib)*l_tau(2,ib))
      my_sigma(3) = min(my_sigma(3), (l_tau(1,ib) + l_tau(2,ib))*(l_dens(1,ib)+l_dens(2,ib)))
      ! We can also use the fact that |grad n_upup +/- grad n_dndn |^2 > 0
      my_sigma(4) = min(my_sigma(4), M_HALF*(my_sigma(1)+my_sigma(2)))
      my_sigma(4) = max(my_sigma(4), -M_HALF*(my_sigma(1)+my_sigma(2)))

      if(present(l_zk)) then
        call nc_br_vxc_exc(l_dens(:, ib), my_sigma, l_ldens(:, ib), l_tau(:, ib), gamma, explicit, &
          l_dedd(:,ib), l_vsigma(:,ib), l_deddldens(:,ib), l_dedtau(:,ib), l_zk(ib))
      else
        call nc_br_vxc_exc(l_dens(:, ib), my_sigma, l_ldens(:, ib), l_tau(:, ib), gamma, explicit, &
          l_dedd(:,ib), l_vsigma(:,ib), l_deddldens(:,ib), l_dedtau(:,ib))
      end if
    end do

  case(XC_MGGA_C_NC_CS)

    do ib = 1, n_block

      ! We enforce Schwartz inequality
      my_sigma = l_sigma(:,ib)
      my_sigma(1) = min(my_sigma(1), M_FOUR*l_dens(1,ib)*l_tau(1,ib))
      my_sigma(2) = min(my_sigma(2), M_FOUR*l_dens(2,ib)*l_tau(2,ib))
      my_sigma(3) = min(my_sigma(3), (l_tau(1,ib) + l_tau(2,ib))*(l_dens(1,ib)+l_dens(2,ib)))
      ! We can also use the fact that |grad n_upup +/- grad n_dndn |^2 > 0
      my_sigma(4) = min(my_sigma(4), M_HALF*(my_sigma(1)+my_sigma(2)))
      my_sigma(4) = max(my_sigma(4), -M_HALF*(my_sigma(1)+my_sigma(2)))

      if(present(l_zk)) then
        call nc_cs_vxc_exc(l_dens(:, ib), my_sigma, l_ldens(:, ib), l_tau(:, ib), &
          l_dedd(:,ib), l_vsigma(:,ib), l_deddldens(:,ib), l_dedtau(:,ib), l_zk(ib))
      else
        call nc_cs_vxc_exc(l_dens(:, ib), my_sigma, l_ldens(:, ib), l_tau(:, ib), &
          l_dedd(:,ib), l_vsigma(:,ib), l_deddldens(:,ib), l_dedtau(:,ib))
      end if

    end do

  case default
    message(1) = "Unsupported noncollinear functional"
    call messages_fatal(1, namespace=namespace)
  end select

  POP_SUB(nc_mgga_exc_vxc)
end subroutine nc_mgga_exc_vxc

 ! -----------------------------------------------------
 !>@brief Computes the local curvature of the exchange-hole and get the corresponding values of x and b
 !!
 !! This allows to compute the local Coulomb potential and the energy
 !! The exchange potential is finally constructed from the potential
 !
 !! The energy functional is defined in Tancogne-Dejean et al., PRB 107, 165111 (2023)
 !! The energy function is defined by Eq. 7, and Eq. 8, with the gauge-invariant KED defined by Eq. 6.
 !! The exchange-hole curvature is given by Eq. 5
subroutine nc_br_vxc_exc(l_dens, l_sigma, l_ldens, l_tau, gamma, explicit, l_dedd, l_vsigma, l_dedldens, l_dedtau, l_zk)
  real(real64),             intent(in)    :: l_dens(:)     ! Density
  real(real64),             intent(in)    :: l_sigma(:)    ! Modulus squared of the gradient of the density
  real(real64),             intent(in)    :: l_ldens(:)    ! Laplacian of the density
  real(real64),             intent(in)    :: l_tau(:)      ! Kinetic energy density
  real(real64),             intent(in)    :: gamma
  logical,                  intent(in)    :: explicit      ! True if we use the explicit inversion of x(y)
  real(real64),             intent(out)   :: l_dedd(:)     ! Derivative of the energy versus l_dens
  real(real64),             intent(out)   :: l_vsigma(:)   ! Derivative of the energy versus l_sigma
  real(real64),             intent(out)   :: l_dedldens(:) ! Derivative of the energy versus the l_dens
  real(real64),             intent(out)   :: l_dedtau(:)   ! Derivative of the energy versus l_tau
  real(real64), optional,   intent(out)   :: l_zk          ! Energy density

  real(real64) :: l_ontop, l_curv, xx_BR, l_charge
  real(real64) :: cnst, U_BR, P_BR, dUdx
  real(real64) :: dtop_dn(4), dcurv_dn(2)
  real(real64), parameter :: tol_den = 1e-15_real64
  real(real64), parameter :: tol_x = 1e-7_real64
  !This tolerance here seems important. I checked using Matlab that
  !the error using the Taylor expansion is lower than 1e-8 compared to the
  !full formula below this threshold. However, for x below this threshold, the
  !full formula become unstable and the error grows. NTD
  real(real64), parameter :: tol_dUdx = 1e-4_real64
  real(real64) :: tauW, my_tau

  l_dedd(1:4)     = M_ZERO
  l_vsigma(1:4)   = M_ZERO
  l_dedldens(1:4) = M_ZERO
  l_dedtau(1:4)   = M_ZERO

  l_charge = sum(l_dens(1:2))

  ! In this case, there is no charge
  if(l_charge < tol_den) return

  ! We compute the value of the on-top effective exchange hole
  l_ontop = (sum(l_dens(1:2)**2) + M_TWO*sum(l_dens(3:4)**2))/l_charge

  ! We also compute of the curvature and we normalize it properly
  ! Off-diagonal term of the curvature, that belongs to the SU(2) gauge invariant KED
  ! First we get 1/2(n\tau+\tau n), with \tau the U(1) gauge-invariant KED
  my_tau = l_dens(1)*l_tau(1) + l_dens(2)*l_tau(2) + M_TWO*(l_dens(3)*l_tau(3) + l_dens(4)*l_tau(4))

  ! Now adding the part for the SU(2) gauge invariance
  my_tau = my_tau - (l_dens(3)*l_ldens(3) + l_dens(4)*l_ldens(4))
  my_tau = my_tau + M_HALF*(-l_sigma(3) + l_sigma(4) + l_dens(1)*l_ldens(2) + l_dens(2)*l_ldens(1))
  my_tau = my_tau / l_charge

  ! von Weizsaecker kinetic energy density
  tauW = M_FOURTH*(l_sigma(1)+l_sigma(2)+M_TWO*l_sigma(4))/l_charge

  ! We enforce here that \tau-\tau^W > 0
  ! This is commented as this seems to give a different result than libxc5 if used
  ! my_tau = max(my_tau, tauW)

  ! \nabla^2 n - 2\gamma (\tau - (\nabla n\dotp\nabla n)/4)
  l_curv = (sum(l_ldens(1:2)) - M_TWO*gamma*(my_tau - tauW)) / 6.0_real64

  ! We get the value of x for up and down components
  if (explicit) then
    xx_BR = nc_br_get_x_explicit(l_ontop, l_curv)
  else
    xx_BR = nc_br_get_x(l_ontop, l_curv)
  end if

  if(xx_BR < M_ZERO) then
    message(1) = "Newton-Raphson produced a negative x value"
    call messages_fatal(1)
  end if

  cnst = M_TWO * (M_PI*l_ontop)**M_THIRD

  ! We construct the enengy density
  if (xx_BR > tol_x) then
    U_BR = -( M_ONE - exp(-xx_BR)*(M_ONE + M_HALF*xx_BR) ) &
      / xx_BR * exp(xx_BR*M_THIRD) * cnst
  else ! Taylor expansion at x=0
    U_BR = -cnst * (M_HALF + xx_BR/6.0_real64 - xx_BR**2/18.0_real64)
  end if

  ! Energy density
  if(present(l_zk)) then
    l_zk = U_BR*M_HALF
  end if

  ! Minus sign cancels with the next equation
  if(xx_BR > tol_dUdx) then
    dUdx = (xx_BR**2 + M_TWO*xx_BR + exp(xx_BR)*(xx_BR-M_THREE) + M_THREE) &
      * exp(-xx_BR*M_TWOTHIRD) / (M_THREE * xx_BR**2)
  else ! Taylor expansion at x=0
    dUdx = M_ONE/6.0_real64 - xx_BR/9.0_real64
  end if
  dUdx = dUdx * cnst

  P_BR = M_HALF * l_charge * dUdx * (xx_BR-M_TWO)*xx_BR/(xx_BR**2-M_TWO*xx_BR+M_THREE) &
    * M_THREE * M_HALF / l_ontop

  ! Derivative of the on-top density wrt the density
  dtop_dn(1:2) = (M_TWO*l_dens(1:2) - l_ontop) / l_charge
  ! Here dtop_dn(3,4) are Re/Im of the derivative wrt n_{dn up}, not derivative wrt Re/Im(n_{updn})
  dtop_dn(3:4) = M_TWO*l_dens(3:4)/l_charge

  ! Derivative of the curvature wrt the density
  dcurv_dn(1:2) = - M_THIRD * gamma * tauW / l_charge

  ! Derivative of the energy wrt the density
  l_dedd(1:2) = M_HALF * (M_ONE + M_THIRD*l_charge/l_ontop * dtop_dn(1:2) )*U_BR &
    + P_BR*(M_FIVE*M_THIRD*dtop_dn(1:2) - l_ontop / l_curv * dcurv_dn(1:2))

  l_dedd(3:4) = M_THIRD*l_dens(3:4)*U_BR/l_ontop + P_BR*(M_FIVE*M_THIRD*dtop_dn(3:4))

  P_BR = -P_BR * l_ontop / l_curv / 6.0_real64

  ! Derivative of the energy wrt sigma
  l_vsigma(1:2) = M_HALF * gamma * P_BR / l_charge
  l_vsigma(3) = M_ZERO
  l_vsigma(4) = gamma * P_BR / l_charge

  ! Derivative of the energy wrt the laplacian of the density
  l_dedldens(1:2) = P_BR

  ! Derivative of the energy wrt the kinetic energy
  P_BR = -M_TWO * gamma * P_BR / l_charge

  l_dedd(1) = l_dedd(1) + P_BR*(l_tau(1) + M_HALF*l_ldens(2) - my_tau)
  l_dedd(2) = l_dedd(2) + P_BR*(l_tau(2) + M_HALF*l_ldens(1) - my_tau)
  l_dedd(3:4) = l_dedd(3:4) + P_BR*(l_tau(3:4) - M_HALF*l_ldens(3:4))

  l_vsigma(3) = l_vsigma(3) - M_HALF*P_BR
  l_vsigma(4) = l_vsigma(4) + M_HALF*P_BR

  l_dedldens(1) = l_dedldens(1) + P_BR*M_HALF*l_dens(2)
  l_dedldens(2) = l_dedldens(2) + P_BR*M_HALF*l_dens(1)
  l_dedldens(3:4) = -M_HALF*P_BR*l_dens(3:4)

  l_dedtau(1:4) = P_BR * l_dens(1:4)

end subroutine nc_br_vxc_exc

! -----------------------------------------------------
!> Computes the coefficient x from the local density and the local curvature
!! of the exchange hole, see Eq. B7 in PRB 107, 165111 (2023)
function nc_br_get_x(ldens, lcurv) result(br_x)
  real(real64),             intent(in)    :: ldens      ! On-top exchange hole
  real(real64),             intent(in)    :: lcurv      ! Exchange-hole curvature
  real(real64)                            :: br_x

  real(real64) :: rhs
  real(real64), parameter :: tol = 1e-11_real64

  !Reduced curvature
  rhs = M_TWOTHIRD * M_PI ** M_TWOTHIRD * (ldens ** (M_FIVE*M_THIRD)) / (lcurv+M_TINY)
  if(abs(rhs) < tol) rhs = sign(tol, rhs)

  br_x = nc_br_rtsafe(rhs, tol)

end function nc_br_get_x

! -----------------------------------------------------
!>@brief Computes the coefficient x from the local density and the local curvature
!! of the exchange hole, see Eq. B7 in PRB 107, 165111 (2023)
function nc_br_get_x_explicit(ldens, lcurv) result(br_x)
  real(real64),             intent(in)    :: ldens      ! On-top exchange hole
  real(real64),             intent(in)    :: lcurv      ! Exchange-hole curvature

  real(real64)                            :: br_x

  real(real64) :: rhs
  real(real64), parameter :: tol = 1e-11_real64
  ! Variables and parameters for the explicit inversion
  ! See Appendix of Proynov et al., Chem. Phys. Letters 455, 103 (2008)
  real(real64) :: gy, p1, p2
  real(real64), parameter :: a1 = 1.5255251812009530_real64
  real(real64), parameter :: a2 = 0.4576575543602858_real64
  real(real64), parameter :: a3 = 0.4292036732051034_real64
  real(real64), parameter :: b = 2.085749716493756_real64
  real(real64), parameter :: b0 =  0.4771976183772063_real64
  real(real64), parameter :: b1 = -1.7799813494556270_real64
  real(real64), parameter :: b2 =  3.8433841862302150_real64
  real(real64), parameter :: b3 = -9.5912050880518490_real64
  real(real64), parameter :: b4 =  2.1730180285916720_real64
  real(real64), parameter :: b5 = -30.425133851603660_real64
  real(real64), parameter :: c0 =  0.7566445420735584_real64
  real(real64), parameter :: c1 = -2.6363977871370960_real64
  real(real64), parameter :: c2 =  5.4745159964232880_real64
  real(real64), parameter :: c3 = -12.657308127108290_real64
  real(real64), parameter :: c4 =  4.1250584725121360_real64
  real(real64), parameter :: c5 = -30.425133957163840_real64
  real(real64), parameter :: d0 = 0.00004435009886795587_real64
  real(real64), parameter :: d1 = 0.58128653604457910_real64
  real(real64), parameter :: d2 = 66.742764515940610_real64
  real(real64), parameter :: d3 = 434.26780897229770_real64
  real(real64), parameter :: d4 = 824.7765766052239000_real64
  real(real64), parameter :: d5 = 1657.9652731582120_real64
  real(real64), parameter :: e0 = 0.00003347285060926091_real64
  real(real64), parameter :: e1 = 0.47917931023971350_real64
  real(real64), parameter :: e2 = 62.392268338574240_real64
  real(real64), parameter :: e3 = 463.14816427938120_real64
  real(real64), parameter :: e4 = 785.2360350104029000_real64
  real(real64), parameter :: e5 = 1657.962968223273000000_real64


  !Reduced curvature
  rhs = M_TWOTHIRD * M_PI ** M_TWOTHIRD * (ldens ** (M_FIVE*M_THIRD)) / (lcurv+M_TINY)
  if(abs(rhs) < tol) rhs = sign(tol, rhs)

  ! Formulas are given in the Appendix of Proynov et al., Chem. Phys. Letters 455, 103 (2008)
  if (rhs <= M_ZERO) then ! Eqs. A3 and A4
    gy = -atan(a1 * rhs + a2) + a3
    p1 = c0 + rhs * (c1 + rhs * (c2 + rhs * (c3 + rhs * (c4 + c5 * rhs))))
    p2 = b0 + rhs * (b1 + rhs * (b2 + rhs * (b3 + rhs * (b4 + b5 * rhs))))
  else ! Eqs. A5 and A6
    gy = asinh(M_ONE / (b*rhs)) + M_TWO
    p1 = d0 + rhs * (d1 + rhs * (d2 + rhs * (d3 + rhs * (d4 + d5 * rhs))))
    p2 = e0 + rhs * (e1 + rhs * (e2 + rhs * (e3 + rhs * (e4 + e5 * rhs))))
  end if
  br_x = gy * p1 / p2 !Eq. A2

end function nc_br_get_x_explicit

! -----------------------------------------------------
!> This is inspired by the safe Newton-Raphson method from numerical recipies
!! This function returns the value of x that fulfill the equation
!! \f$x exp(-2/3*x)/(x-2) = rhs\f$.
real(real64) function nc_br_rtsafe(rhs, tol) result(br_x)
  real(real64), intent(in) :: rhs
  real(real64), intent(in) :: tol

  integer, parameter :: maxit = 500
  real(real64) :: ff, dff, emx, xl, xh, dx, dxold, temp
  integer :: it

  ! For small rhs, the solution is trivially 2
  if (abs(rhs) < 5e-11_real64) then
    br_x = M_TWO
    return
  end if

  ! Select the correct branch for the function
  if (rhs < M_ZERO) then
    xh = M_ZERO
    xl = M_TWO
  else
    xh = M_TWO
    xl = M_TWO + M_ONE/rhs
  end if

  dxold = abs(xh-xl)
  dx = dxold
  br_x = M_HALF*(xl+xh)

  ! Calculate the function and its derivative
  emx = exp(-M_TWOTHIRD*br_x)
  ff = br_x * emx / rhs - (br_x-M_TWO)
  dff = emx * ( M_ONE - M_TWOTHIRD * br_x) / rhs - M_ONE

  do it = 1, maxit
    if (((br_x-xh)*dff-ff) * ((br_x-xl)*dff-ff) > M_ZERO & ! Bisect if Newton out of range,
      .or. abs(M_TWO*ff) > abs(dxold*dff)) then            ! or not decreasing fast enough
      dxold = dx
      dx = M_HALF*(xh-xl)
      br_x = xl+dx
      if (abs(xl - br_x) < M_TINY) return !! Change in root is negligible
    else                    ! Newton step acceptable. Take it.
      dxold = dx
      dx = ff/dff
      temp = br_x
      br_x = max(br_x - dx, M_ZERO)
      if (abs(temp - br_x) < M_TINY) return
    end if
    if (abs(dx) < tol) return ! Convergence criterion

    ! Calculate the function and its derivative
    emx = exp(-M_TWOTHIRD*br_x)
    ff = br_x * emx / rhs - (br_x-M_TWO)
    dff = emx * ( M_ONE - M_TWOTHIRD * br_x) / rhs - M_ONE
    if(ff < M_ZERO) then ! Maintain the bracket on the root
      xl = br_x
    else
      xh = br_x
    end if
  end do

  message(1) = "Newton-Raphson did not converged"
  call messages_fatal(1)

end function nc_br_rtsafe

!--------------------------------------------------------------------------------------------------------
!>@brief  Implements the correlation energy functional defined in Tancogne-Dejean et al., PRB 107, 165111 (2023)
subroutine nc_cs_vxc_exc(l_dens, l_sigma, l_ldens, l_tau, l_dedd, l_vsigma, l_dedldens, l_dedtau, l_zk)
  real(real64),             intent(in)    :: l_dens(:)     ! Density
  real(real64),             intent(in)    :: l_sigma(:)    ! Modulus squared of the gradient of the density
  real(real64),             intent(in)    :: l_ldens(:)    ! Laplacian of the density
  real(real64),             intent(in)    :: l_tau(:)      ! Kinetic energy density
  real(real64),             intent(out)   :: l_dedd(:)     ! Derivative of the energy versus l_dens
  real(real64),             intent(out)   :: l_vsigma(:)   ! Derivative of the energy versus l_sigma
  real(real64),             intent(out)   :: l_dedldens(:) ! Derivative of the energy versus the l_dens
  real(real64),             intent(out)   :: l_dedtau(:)   ! Derivative of the energy versus l_tau
  real(real64), optional,   intent(out)   :: l_zk          ! Energy density

  real(real64) :: l_charge, tauW, tauHF, gamma, weight, denom, exp_term, rho_third
  real(real64) :: dgammadd(4)
  real(real64), parameter :: tol_den = 1e-15_real64
  real(real64), parameter :: aa = 0.04918_real64
  real(real64), parameter :: bb = 0.132_real64
  real(real64), parameter :: cc = 0.2533_real64
  real(real64), parameter :: dd = 0.349_real64

  PUSH_SUB(nc_cs_vxc_exc)

  l_dedd(1:4)     = M_ZERO
  l_vsigma(1:4)   = M_ZERO
  l_dedldens(1:4) = M_ZERO
  l_dedtau(1:4)   = M_ZERO

  l_charge = sum(l_dens(1:2))

  ! In this case, there is no charge
  if(l_charge < tol_den) return

  ! von Weizsaecker kinetic energy density
  tauW = M_ONE/8.0_real64*((l_sigma(1)+l_sigma(2)+M_TWO*l_sigma(4))/(l_charge) - l_ldens(1)-l_ldens(2))

  ! HF kinetic energy density defined by Eq. 15
  tauHF = (l_dens(1)*(l_tau(1)-M_FOURTH*l_ldens(1)) + l_dens(2)*(l_tau(2)-M_FOURTH*l_ldens(2)) &
    + M_TWO * (l_dens(3)*(l_tau(3)-M_FOURTH*l_ldens(3))+l_dens(4)*(l_tau(4)-M_FOURTH*l_ldens(4))))/l_charge

  gamma = M_ONE - ((l_dens(1)-l_dens(2))**2 + M_FOUR*(l_dens(3)**2+l_dens(4)**2))/(l_charge**2)

  rho_third = l_charge**M_THIRD
  denom = (M_ONE + dd / rho_third)
  exp_term = exp(-cc / rho_third)
  weight = -aa * gamma / denom * bb * exp_term /(rho_third**2)

  ! Energy density, see Eq. 14
  ! Note that l_zk is multiplied by the density later
  if(present(l_zk)) then
    l_zk = -aa * gamma * (M_ONE + bb/(rho_third**5)*(tauHF-M_TWO*tauW)*exp_term) / denom
  end if

  dgammadd(1) = M_TWO*(M_ONE-gamma)/l_charge - M_TWO * (l_dens(1)-l_dens(2))/(l_charge**2)
  dgammadd(2) = M_TWO*(M_ONE-gamma)/l_charge + M_TWO * (l_dens(1)-l_dens(2))/(l_charge**2)
  dgammadd(3:4) = -M_FOUR * l_dens(3:4)/(l_charge**2)

  ! Derivative of the energy wrt the density
  l_dedd(1:4) = -aa* dgammadd(1:4) * (l_charge + bb/(rho_third**2)*(tauHF-M_TWO*tauW) * exp_term)/ denom
  l_dedd(1:2) = l_dedd(1:2) - aa * gamma / denom * (M_ONE &
    - M_TWOTHIRD*bb*(tauHF-M_TWO*tauW)* exp_term/(rho_third**5) &
    + bb*cc*(tauHF-M_TWO*tauW) * exp_term/(M_THREE*l_charge**2))
  l_dedd(1:2) = l_dedd(1:2) - aa*dd*gamma*(l_charge + bb/(rho_third**2)*(tauHF-M_TWO*tauW) * exp_term) &
    / (M_THREE*(denom*rho_third**2)**2)
  l_dedd(1:2) = l_dedd(1:2) + weight*( (l_tau(1:2)-M_FOURTH*l_ldens(1:2) - tauHF) &
    + M_FOURTH*(l_sigma(1)+l_sigma(2)+M_TWO*l_sigma(4))/l_charge)/l_charge
  l_dedd(3:4) = l_dedd(3:4) + weight*(l_tau(3:4)-M_FOURTH*l_ldens(3:4))/l_charge

  ! Derivative of the energy wrt sigma
  l_vsigma(1:2) = weight * ( -M_FOURTH/l_charge)
  l_vsigma(3) = M_ZERO
  l_vsigma(4) = weight * ( -M_HALF/l_charge)

  ! Derivative of the energy wrt the laplacian of the density
  l_dedldens(1:2) = weight*(-M_FOURTH*l_dens(1:2)/l_charge + M_FOURTH)
  l_dedldens(3:4) = weight*(-M_FOURTH*l_dens(3:4)/l_charge)

  ! Derivative of the energy wrt the kinetic energy
  l_dedtau(1:4) = weight * l_dens(1:4) / l_charge

  POP_SUB(nc_cs_vxc_exc)
end subroutine nc_cs_vxc_exc


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
