# Chapter 3: Setting Up the Project {#tutorial-setup}

In this chapter, we will configure our project using the [Meson build system](https://mesonbuild.com). To get started, we'll create a new directory, which should include a `meson.build` file and a `src` subdir with a `main.cpp` file like this.

```
project_dir
|-- meson.build
|-- src
    |-- main.cpp
```

#### meson.build

```bash
project(
    'Louvre Example',
    'c','cpp',
    version : '0.1.0',
    meson_version: '>= 0.56.0',
    default_options: [
        'warning_level=2',
        'cpp_std=c++20',
    ]
)
 
louvre_dep = dependency('Louvre')
 
# List C/C++ source files stored in ./src
sources = run_command('find', './src', '-type', 'f', '-name', '*[.c,.cpp,.h,.hpp]', check : false).stdout().strip().split('\n')
 
executable(
    'louvre-example',
    sources : sources,
    dependencies : [
        louvre_dep,
])
```

This `meson.build` file will automatically link to the required libs and scan the `project_dir/src` directory for available C/C++ source code files of our project.

#### src/main.cpp

```cpp
#include <LLauncher.h>
#include <LCompositor.h>
#include <LLog.h>

using namespace Louvre;

int main()
{
    setenv("LOUVRE_DEBUG", "4", 0);

    LLauncher::startDaemon();

    LCompositor compositor;

    if (!compositor.start())
    {
        LLog::fatal("[louvre-example] Failed to start compositor.");
        return 1;
    }

    while (compositor.state() != LCompositor::Uninitialized)
        compositor.processLoop(-1);

    return 0;
}
```

In this main file, we first enable all Louvre debugging messages (see Louvre::LLog) and create a daemon with Louvre::LLauncher::startDaemon(). This daemon is optional and allows us to launch applications safely while the compositor is running (we will use it to launch `weston-terminal` later). Subsequently, we create an Louvre::LCompositor instance and launch it using Louvre::LCompositor::start(). As long as the compositor remains active, we manage Wayland and backend events through the Louvre::LCompositor::processLoop() method. This method operates similarly to the [poll()](https://man7.org/linux/man-pages/man2/poll.2.html) function, with the "-1" argument indicating that it will block indefinitely until an event occurs. Additionally, if necessary, you can obtain a pollable file descriptor using Louvre::LCompositor::fd().

### Building the Project

Let's build the project to see Louvre's default behavior. To configure it, run these commands in your terminal:

```bash
$ cd project_dir
$ meson setup build
```

After successful configuration, you'll see confirmation of the Louvre library detection, like this:

```bash
Build type: native build
Project name: Louvre Example
Project version: 0.1.0
C compiler for the host machine: cc (gcc 11.4.0 "cc (Ubuntu 11.4.0-1ubuntu1~22.04) 11.4.0")
C linker for the host machine: cc ld.bfd 2.38
C++ compiler for the host machine: c++ (gcc 11.4.0 "c++ (Ubuntu 11.4.0-1ubuntu1~22.04) 11.4.0")
C++ linker for the host machine: c++ ld.bfd 2.38
Host machine cpu family: x86_64
Host machine cpu: x86_64
Library Louvre found: YES
Build targets in project: 1

Found ninja-1.10.1 at /usr/bin/ninja
```

If you face configuration issues and don't see the expected Louvre library confirmation, make sure you have all required tools and libraries installed. Refer to the [Downloads](md_md__downloads.html) section for detailed build instructions. 

Now, follow these commands to build the compositor:

```bash
cd build
meson compile
```

After executing these commands, you should find a new executable file named `louvre-example` within the `build` directory.

This compositor should let you launch `weston-terminal` by pressing `F1` and terminate itself by using the key combination `Ctrl + Shift + Esc`. To run it, switch to a free TTY and execute it from there. You should see something like this:

![Louvre Compositor](https://lh3.googleusercontent.com/pw/AIL4fc97hD995n2SkAxjZuwS_Lh8zdv_4SojJP_0UL25rLOMpXYjkyT-Qsf656HlBLYnLNQfSCA6O5BlEdzyCzNfDNGWErf7i9U5zpmO6xWOMOTpbK3B88o=w2400)

As you've seen, Louvre handles numerous tasks by default behind the scenes. In the next chapter, we'll explore how to customize these defaults and take control of the compositor's initialization process.

<a href="02.md">◀ Chapter 2: Quick Overview </a> || <a href="04.md"> Chapter 4: Compositor Initialization ▶</a>