# Chapter 11: Popups {#tutorial-popups}

Popup surfaces are short-lived surfaces typically employed for displaying tooltips or context menus. They are always children of a toplevel or other popups.

<center>
<img style="width:35vw" src="https://lh3.googleusercontent.com/6caGayutKKWqndpd6ogno2lPw8XGELxnFums4gvkWZKOJYO0762yVG3mHLrc1rw63r1eEJabEdW9F5AA2BDTFCtpB_hiPlTY4FkKlHfH1B-2MdLvXCD6RuwZOZvhOl6EhydtsOYGPw=w2400"/>
</center>

### Window Geometry

Similar to toplevels, the size of a popup is not determined by its surface size but by its window geometry (accessible through Louvre::LPopupRole::windowGeometry()). The window geometry defines the rectangular area within the surface that represents the actual popup window, excluding any decorative elements like shadows. This is the size you should consider when configuring a popup.

<center>
<img style="max-width:25vw;min-width:400px" src="https://lh3.googleusercontent.com/pw/AIL4fc9WuCZOWGGC6v7ZepWA7VOWygycmNFEngEHX7J9ODdmzm1Qs4au1KYPMRJ_102tfvOXFBcL-Kwghm0bSKbd4U6der3p94l1drpn3ONqb9yfv-uFgHQ=w2400"/>
</center>

### Positioning

Popups involve complex rules for their positioning relative to their parent surface. These rules dictate how popups should be repositioned or scaled when their geometries extend beyond a specified bounding rect, such as an output.

All these positioning rules are defined in their Louvre::LPositioner instance which can be accessed through Louvre::LPopupRole::positioner(). If you wish to understand how these work, please refer to the Louvre::LPositioner documentation.

Fortunately, Louvre simplifies this, implementing these rules by default in Louvre::LPopupRole::rolePos(). This leaves you with the straightforward task of defining the bounding area where the popup should be positioned. In our case, we should set the bounding area equal to the current output size minus the topbar region, or equal to the entire output size if the popup is a child of a fullscreen toplevel. 

Let's explore how to handle this by creating our custom Louvre::LPopupRole subclass called `EPopup`.

#### src/EPopup.h

```cpp
#ifndef EPOPUP_H
#define EPOPUP_H

#include <LPopupRole.h>

using namespace Louvre;

class ESurface;

class EPopup : public LPopupRole
{
public:
    EPopup(const void *params);

    // Cast LSurface to ESurface
    inline ESurface *surf() const
    {
        return (ESurface*) surface();
    };

    void configureRequest() override;
    void grabSeatRequest(Protocols::Wayland::GSeat *seatGlobal) override;
};

#endif // EPOPUP_H
```

#### src/EPopup.cpp

```cpp
#include <protocols/Wayland/GSeat.h>
#include <LSeat.h>
#include <LKeyboard.h>
#include <LCursor.h>
#include <LPositioner.h>
#include <LOutput.h>
#include "ECompositor.h"
#include "ESurface.h"
#include "EPopup.h"
#include "Global.h"

EPopup::EPopup(const void *params) : LPopupRole(params) {}

void EPopup::configureRequest()
{
    // A bounding rect with zero area disable constraints
    LRect boundingRect { 0 };

    // The cursor is always positioned within an output unless there are zero initialized outputs
    if (cursor()->output())
    {
        // Ensure the Popup stays within the boundaries of the current output where the cursor is positioned
        boundingRect = cursor()->output()->rect();

        // If the popup is child of a fullscreen toplevel we can ignore the topbar area
        if (surf()->view.parent() == &G::compositor()->surfacesLayer)
            boundingRect += LRect(0, TOPBAR_HEIGHT, 0, -TOPBAR_HEIGHT);
    }

    setPositionerBounds(boundingRect);

    /* Calculate the relative position of the Popup with respect to its parent position,
     * calling rolePos() also determines LPositioner::unconstrainedSize() */
    const LPoint localParentPos { rolePos() - surface()->parent()->pos() };
    configure(LRect(localParentPos, positioner().unconstrainedSize()));
}

void EPopup::grabSeatRequest(Protocols::Wayland::GSeat *seatGlobal)
{
    /* The grabSeatRequest() is only triggered if the request has been
     * originated from some client event, such as a pointer click or key press*/
    seat()->keyboard()->setGrabbingSurface(surface(), seatGlobal->keyboardResource());
}
```

I'll explain what's happening in a moment, but first, let's override its virtual constructor.

#### src/ECompositor.h

```cpp
// ...

class ECompositor : public LCompositor
{
public:

    // ...

    LPopupRole *createPopupRoleRequest(const void *params) override;

    // ...
};
```

#### src/ECompositor.cpp

```cpp
// ...

#include "EPopup.h"

// ...

LPopupRole *ECompositor::createPopupRoleRequest(const void *params)
{
    return new EPopup(params);
}
```
Similar to what we saw with toplevels, popups also request to be configured the first time they are created, accomplished through Louvre::LPopupRole::configureRequest().

We configure popups using Louvre::LPopupRole::configure(). However, unlike toplevels, popups don't have state flags, but they do require configuration of both their position relative to the parent surface and their size.

#### src/EPopup.cpp

```cpp
void EPopup::configureRequest()
{
    // If the bounding rect has zero area, constraints are disabled
    LRect boundingRect { 0 };

    // The cursor is always positioned within an output unless there are zero initialized outputs
    if (cursor()->output())
    {
        // Ensure the popup stays within the boundaries of the current output where the cursor is positioned
        boundingRect = cursor()->output()->rect();

        // If the popup is a child of a fullscreen toplevel, we can ignore the top bar area
        if (surf()->view.parent() == &G::compositor()->surfacesLayer)
            boundingRect += LRect(0, TOPBAR_HEIGHT, 0, -TOPBAR_HEIGHT);
    }

    setPositionerBounds(boundingRect);

    /* Calculate the relative position of the Popup with respect to its parent position,
     * calling rolePos() also determines LPositioner::unconstrainedSize() */
    const LPoint localParentPos { rolePos() - surface()->parent()->pos() };
    configure(LRect(localParentPos, positioner().unconstrainedSize()));
}
```

So, within Louvre::LPopupRole::configureRequest(), we begin by defining the boundaries within which the popup can be positioned. If it's a child of a fullscreen toplevel, we set the bounding rect equal to the output rect, otherwise, we subtract the topbar region. To determine the position, we call Louvre::LPopupRole::rolePos() which returns the popup global position and then we subtract its parent's position to calculate the local parent offset.

Moreover, calling Louvre::LPopupRole::rolePos() also stores the size the popup should have to remain unconstrained (in case it is constrained), which can be accessed with Louvre::LPositioner::unconstrainedSize(). 

Finally, we configure the popup using the calculated local parent position and unconstrained size.

### Explicit Grab

Popups can also request to be created with an explicit keyboard grab through Louvre::LPopupRole::grabSeatRequest(). This means that all keyboard events should be redirected to that surface while it is active. If the popup is dismissed, the grab should be returned to the parent surface (if it is also a popup previously created with a keyboard grab).

> This request is only triggered if it originated from a keyboard or pointer event. If you deny the grab, the popup will be automatically dismissed.

#### src/EPopup.cpp

```cpp
void EPopup::grabSeatRequest(Protocols::Wayland::GSeat *seatGlobal)
{
    /* The grabSeatRequest() is only triggered if the request has been
     * originated from some client event, such as a pointer click or key press*/
    seat()->keyboard()->setGrabbingSurface(surface(), seatGlobal->keyboardResource());
}
```

The Louvre::LKeyboard class includes a utility method for setting a keyboard grab on a surface (Louvre::LKeyboard::setGrabbingSurface()). During a keyboard grab, no other surface can acquire keyboard focus, rendering Louvre::LKeyboard::setFocus() a no-op.

The Louvre::LKeyboard::setGrabbingSurface() method accepts two arguments: the target surface that will receive the keyboard grab and the native [wl_keyboard](https://wayland.app/protocols/wayland#wl_keyboard) resource used by the client (clients can bind multiple times to the [wl_seat](https://wayland.app/protocols/wayland#wl_seat) global, and create more than one [wl_keyboard](https://wayland.app/protocols/wayland#wl_keyboard) resource).

> Classes located in `<protocols/...>`, such as `<protocols/Wayland/GSeat.h>`, represent low-level Wayland interfaces/resources of protocols implemented by Louvre. While they can be used for sending or handling low-level Wayland events/requests, it's not recommended, as doing so could potentially disrupt the higher-level functioning of Louvre.

### Dismissing Popups

As you may recall from previous chapters, we dismissed all popups, for example when clicking on a surface that is not a popup or when no surface is clicked, using Louvre::LPointer::dismissPopups(). To dismiss a specific popup, you can use Louvre::LPopupRole::dismiss(). This method will dismiss the specified popup along with any child or subchild popups, in reverse order of their creation.

> We have briefly discussed other surface roles like Louvre::LSubsurfaceRole, Louvre::LCursorRole, and Louvre::LDNDIconRole. However, we won't go into extensive detail about these because there is limited practical value in subclassing them and overriding their virtual methods. The functionality they provide is already efficiently handled by the classes we've explored so far and the ones that will be covered in the upcoming chapters.

In the next chapter, we will delve into creating animations using the Louvre::LAnimation class. Until then, happy coding!

<a href="10.md">◀ Chapter 10: Toplevels</a> || <a href="12.md"> Chapter 12: Animations ▶</a>