# Chapter 13: Cursor {#tutorial-cursor}

As we have already seen in previous chapters, Louvre features the Louvre::LCursor class, which simplifies cursor rendering on outputs. We have also discussed that certain graphic backends support hardware compositing, providing an efficient means of rendering the cursor without the need for using OpenGL. In this chapter, we will discuss how clients request to change the cursor, and we will also explore how to load cursors from themes installed on your system.

### Texture and Hotspot

To set the cursor, you can use Louvre::LCursor::setTextureB(). This method requires two arguments: the texture you intend to use as the cursor and its hotspot, which is defined in buffer coordinates. The hotspot indicates the precise location within the texture where the cursor actually points to. For example, if the cursor was a cross, the hotspot would be placed at the center of the texture.

### Position and Size

To adjust the cursor position, you can use Louvre::LCursor::setPos(), which takes global surface coordinates. Alternatively, you can use Louvre::LCursor::move(), which moves the cursor by a delta (dx, dy).

If you move the cursor to a position where there is no output, it will be automatically relocated to the edge of the closest output. You can access the output where the cursor is currently pointing to with Louvre::LCursor::output() and retrieve all the outputs intersected by the cursor rect using Louvre::LCursor::intersectedOutputs().

When hardware compositing is available, the cursor position is automatically adjusted to the hotspot. In cases where OpenGL rendering is required, the hotspot is also included in Louvre::LCursor::rect(), defining the on-screen area where you should render the cursor.

However, the cursor's size is not determined by the dimensions of its texture. To set the cursor's size, you must use Louvre::LCursor::setSize(). This method automatically scales the texture and the hotspot proportionally to the specified size. As a result, even if you change the texture, the cursor always maintains the same size. The default cursor size is (24, 24) in surface coordinates.

> It's important to note that hardware cursor planes have limited size, guaranteed to be at least (64, 64) in buffer coordinates. If you set the cursor size to dimensions that are too large, it may appear clipped when using hardware composition. In such cases, you should consider rendering it with OpenGL instead. Given that we typically work with scale factors that are not greater than 2, the maximum size you should use when using hardware composition is (32, 32).

### Default Cursor

Louvre provides a default fallback arrow cursor texture that you can load by calling Louvre::LCursor::useDefault(). This automatically restores the default texture and hotspot without alterig the current cursor size and position.

<center>
    <img width="74px" height="74px" src="https://lh3.googleusercontent.com/MSUUg3LSS6lYtpyLnKzbECf9eeZeFscmnLGJLRCdADwcjjcVd4xT07AMvQoHUTGptJFzY4tZrQ3IdLKyEbM_O0WyWYk8Pvc-Jf8xZHXoFUkFo2RRYTP8zN_LeOhsvIc6SlsO83TJUw=w2400"><br><br>
    <i>Louvre's default cursor</i>
</center>

If you wish to replace the default texture and hotspot with your own, you have the option to do so using Louvre::LCursor::replaceDefaultB().

### Client Requests

Clients can request to set the cursor texture and hotspot through Louvre::LPointer::setCursorRequest(). To do this, they create an Louvre::LCursorRole and pass it as an argument. Alternatively, this parameter can be `nullptr`, indicating that the client wants to hide the cursor.

Let's proceed to override that method to observe how Louvre handles it by default:

#### src/EPointer.h

```cpp
#ifndef EPOINTER_H
#define EPOINTER_H

#include <LPointer.h>

using namespace Louvre;

class EPointer : public LPointer
{
public:
    
    // ...

    void setCursorRequest(LCursorRole *cursorRole) override;
};

#endif // EPOINTER_H
```

#### src/EPointer.cpp

```cpp
#include <LCursorRole.h>
#include <LSurface.h>

// ...

void EPointer::setCursorRequest(LCursorRole *cursorRole)
{
    if (cursorRole)
    {
        cursor()->setTextureB(
            cursorRole->surface()->texture(),
            cursorRole->hotspotB());

        cursor()->setVisible(true);
    }
    // If nullptr is provided, it indicates that the client intends to hide the cursor.
    else
        cursor()->setVisible(false);
}
```

This request is only triggered when the client has one of its surfaces with pointer focus or when it is the initiator of an active drag & drop session. Within this request, we examine the `cursorRole`. If it is `nullptr`, we hide the cursor. If it is not `nullptr`, we set the cursor to the texture of its corresponding surface and use the hotspot provided by the role.

Louvre also keeps track of the last request. You can access the last Louvre::LCursorRole requested with Louvre::LPointer::lastCursorRequest(), which may return `nullptr` if the client requested to hide it or if it was destroyed. To distinguish whether the last request was to hide it or the role was simply destroyed, you can use Louvre::LPointer::lastCursorRequestWasHide().

### Cursor Themes

Louvre also provides functionality to load [XCursor icon themes](https://wiki.archlinux.org/title/Cursor_themes) available on your system. XCursor icons are the standard way cursor icon themes are managed in Linux.

To load an XCursor, you can use Louvre::LXCursor::loadXCursorB(). This method takes three arguments: the cursor name, the theme name, and the preferred cursor size in buffer coordinates. If the cursor is found, it returns an instance of Louvre::LXCursor, which contains the cursor texture and hotspot. If no cursor is found, it returns `nullptr`. If the chosen size is not avaliable, it returns the variant with the closets size. Aditionally if you pass `nullptr` as the theme name it returns the first cursor matching the name from any theme.

Let's proceed to load an XCursor and set it as the default cursor. A suitable place to do this is within the Louvre::LCompositor::cursorInitialized() event, which is triggered once the Louvre::LCursor instance is initialized. We will also override Louvre::LCompositor::uninitialized(), which is invoked right before the compositor terminates, allowing us to destroy the loaded XCursor.

#### src/ECompositor.h

```cpp

// ...

class ECompositor : public LCompositor
{
public:
    
    // ...

    void uninitialized() override;
    void cursorInitialized() override;

    // Default cursor
    LXCursor *defaultCursor { nullptr };
 
    // ...
};
```

#### src/ECompositor.cpp

```cpp
#include <LXCursor.h>
#include <LCursor.h>

// ...

void ECompositor::uninitialized()
{
    if (defaultCursor)
    {
        delete defaultCursor;
        defaultCursor = nullptr;
    }
}

void ECompositor::cursorInitialized()
{
    defaultCursor = LXCursor::loadXCursorB("hand1");

    if (defaultCursor)
        cursor()->replaceDefaultB(defaultCursor->texture(), defaultCursor->hotspotB());
}

// ...
```

Within Louvre::LCompositor::cursorInitialized(), we load the "hand1" cursor and keep the other parameters at their default settings. If a cursor is found, it is designated as the default cursor.

From this point on, every time Louvre::LCursor::useDefault() is called, it will load the "hand1" cursor along with its hotspot. If, even after recompiling and running the compositor, you still see Louvre's default arrow cursor, it may be because none of your cursor themes include the "hand1" icon.

<img src="https://lh3.googleusercontent.com/pw/ADCreHcjsWuehhlro2cx8OaRX2rQCRAlpkHmRO12dP5VFKAXeLWvFXoCvjgwc5948_s_UZnuUFmGApMoV06HbvHeAZ-7ZBFRon-_WmSEKBvDqOnCk27ondk=w2400"/>

To better harmonize your compositor with your system's appearance, consider loading and setting the "arrow" icon as the default cursor instead of relying on Louvre's default.

If you wish to revert to the default Louvre cursor, simply pass `nullptr` as the texture in Louvre::LCursor::replaceDefaultB(). This action will restore the default Louvre cursor along with its preset hotspot. The Louvre's default cursor is also automatically restored if the texture you assigned is destroyed.

In the next chapter, we will explore how clients can request and exchange information through clipboard and drag & drop sessions. À bientôt!

<a href="12.md">◀ Chapter 12: Animations</a> || <a href="14.md"> Chapter 14: Clipboard and DND ▶</a>