# Chapter 14: Clipboard and DND {#tutorial-dnd}

The Wayland protocol encompasses multiple interfaces that enable clients to exchange information through selections (clipboard) or drag & drop sessions. The inner workings of these interfaces and their underlying complexities won't be extensively covered in this chapter. Instead, this chapter concentrates on practical aspects, deferring the intricate details to Louvre, as it efficiently manages these complexities on your behalf. If you desire a more in-depth understanding of these topics, I recommend referring to the Wayland protocol documentation or exploring the Louvre::LDataDevice, Louvre::LDataSource, and Louvre::LDataOffer classes for additional insights.

### Data Sharing

When clients need to share data with other clients, they create an Louvre::LDataSource. This data source comprises a list of MIME type strings that represent the data they intend to share. This data is subsequently shared with other clients in the form of an Louvre::LDataOffer when their surfaces obtain keyboard focus or during a drag and drop session. The receiving client can then request the source client to write the data for specific MIME types into file descriptors to retrieve the information.

As mentioned earlier, Louvre takes care of the data exchange process, offering you control over high-level functionality. This includes managing aspects like which clients can set the clipboard or initiate drag and drop sessions.

### Selections

In the context of Wayland and X11, the term "Selections" essentially corresponds to the clipboard functionality. Clients can make requests to set the clipboard using Louvre::LSeat::setSelectionRequest(). Within this method, you have the authority to decide whether to grant or deny the client's request to set the clipboard. To achieve this, you simply return `true` to allow the user to set the clipboard or `false` to decline it.

Let's create a custom Louvre::LSeat subclass named `ESeat` to manage such request:

#### src/ESeat.h

```cpp
#ifndef ESEAT_H
#define ESEAT_H

#include <LSeat.h>

using namespace Louvre;

class ESeat : public LSeat
{
public:
    ESeat(const void *params);

    // Set clipboard request
    virtual bool setSelectionRequest(LDataDevice *device) override;
};

#endif // ESEAT_H
```

#### src/ESeat.cpp

```cpp
#include <LDataDevice.h>
#include <LPointer.h>
#include <LKeyboard.h>
#include "ESeat.h"

ESeat::ESeat(const void *params) : LSeat(params) {}

bool ESeat::setSelectionRequest(LDataDevice *device)
{
    // Let the client set the clipboard only if one of its surfaces has pointer or keyboard focus
    return (pointer()->focus() && pointer()->focus()->client() == device->client()) ||
           (keyboard()->focus() && keyboard()->focus()->client() == device->client());
}
```

In essence, we permit the client to set the clipboard only when one of its surfaces currently holds pointer or keyboard focus. Once the clipboard is set, any surface that gains keyboard focus can request to read its contents, such as when a user presses `Ctrl + V`.

You can also access the current data selection source with Louvre::LSeat::dataSelection() and obtain the list of supported MIME types through Louvre::LDataSource::sources().

> The protocol's behavior results in the clipboard being unset when the source client is destroyed, which can lead to an inconvenient user experience since other applications can no longer request writing to their file descriptors. To address this issue, Louvre requests the source client to write the contents into its own file descriptors. This way, even when the source client is disconnected, users can still paste the contents into other applications. Louvre implements this behavior for a select set of common string-based MIME types only.

Lastly, remember to override its virtual constructor as well.

#### src/ECompositor.h

```cpp
class ECompositor : public LCompositor
{
public:

    // ...

    LSeat *createSeatRequest(const void *params) override;
    
    // ...
};
```

#### src/ECompositor.cpp

```cpp
#include "ESeat.h"

// ...

LSeat *ECompositor::createSeatRequest(const void *params)
{
    return new ESeat(params);
}
```

<img src="https://lh3.googleusercontent.com/pw/ADCreHdgbJtLy_zUvO4NO1osqHXHK1Ihe0whT9ELd9br8qTykn0yhxWQ7VRJYP8sD01h-o8AnvFXBj0oAQBFblnWOTycMf0KRmKK3ymDFgiAkO14GAJJ3HA=w2400"/>

### Drag & Drop

Drag and drop sessions are overseen by the Louvre::LDNDManager class, which can be accessed via Louvre::LSeat::dndManager(). Clients can initiate a session by making a request through Louvre::LDNDManager::startDragRequest(). If you wish to disallow the client from starting the session, you should call Louvre::LDNDManager::cancel().

Throughout a session, the data source is automatically dispatched to any surface that acquires pointer focus. To inform the destination client that a drop has been performed by the user, you should call Louvre::LDNDManager::drop(), as previously demonstrated in [Chapter 8: Pointer Events](08.md) when the left pointer button is released.

Clients also have the option to create an Louvre::LDNDIconRole for use as an icon during the session, which can be accessed through Louvre::LDNDManager::icon(). It's important to note that not all clients create an icon, so you should always check if it returns `nullptr`. We have also addressed the icon's position update in [Chapter 8: Pointer Events](08.md) during a pointer movement event.

Finally, in that chapter, we also discussed how to inform the clients about the preferred compositor action during a session, which can be to copy, move, or none for allowing the clients to decide.

Now, let's proceed to create our custom Louvre::LDNDManager subclass, named `EDNDManager`:

#### src/EDNDManager.h

```cpp
#ifndef EDNDMANAGER_H
#define EDNDMANAGER_H

#include <LDNDManager.h>

using namespace Louvre;

class EDNDManager : public LDNDManager
{
public:
    EDNDManager(const void *params);

    void startDragRequest() override;
};

#endif // EDNDMANAGER_H
```

#### src/EDNDManager.cpp

```cpp
#include <LSeat.h>
#include <LPointer.h>
#include <LKeyboard.h>
#include <LDataSource.h>
#include "EDNDManager.h"

EDNDManager::EDNDManager(const void *params) : LDNDManager(params) {}

void EDNDManager::startDragRequest()
{
    // Let the client start the session only if the origin surface has pointer focus
    if (origin()->hasPointerFocus())
        seat()->pointer()->setDraggingSurface(nullptr);
    else
        cancel();
}
```

#### src/ECompositor.h

```cpp
class ECompositor : public LCompositor
{
public:

    // ...

    LDNDManager *createDNDManagerRequest(const void *params) override;
    
    // ...
};
```

#### src/ECompositor.cpp

```cpp
#include "EDNDManager.h"

// ...

LDNDManager *ECompositor::createDNDManagerRequest(const void *params)
{
    return new EDNDManager(params);
}
```

In this scenario, we only permit the client to initiate a drag & drop session if the source surface has acquired pointer focus. The Louvre::LDNDManager::source() corresponds to the surface from which the DND operation originates.

You may have noticed that we are also calling `Louvre::LPointer::setDraggingSurface(nullptr)`. To refresh your memory, this auxiliary method serves to maintain pointer focus on a surface that is currently "dragged," such as when the user is selecting text. This prevents the surface from losing pointer focus if the cursor exits the surface. By setting it to `nullptr`, we ensure that the current surface does not retain pointer focus. Otherwise, during the drag & drop session, no other surface would receive pointer focus, and they would be unable to receive data offers.

<img src="https://lh3.googleusercontent.com/pw/ADCreHffCRJauTwh6Hg_oKaV2GBz87mJu-9iJFu_ZjowKiAqvbADV5dVCLIHshoVi_mXFHcDQTJdtrZvFBF2D7YiElihz2dNdDlf399nT_zdJfhdRm7cP9Y=w2400"/>

In the next chapter we'll delve into the handling of output hotplugging events and TTY session switching. Nos vemos!

<a href="13.md">◀ Chapter 13: Cursor</a> || <a href="15.md"> Chapter 15: Output Hotplugging and Seat ▶</a>