#!/usr/bin/env python3

import boto3
import argparse
import sys
import time
from datetime import datetime, timezone, timedelta
from concurrent.futures import ProcessPoolExecutor, as_completed
from botocore.exceptions import ClientError
from botocore.config import Config

def get_args():
    parser = argparse.ArgumentParser(
        description="Delete unfinished S3 multi-part uploads using multiple threads. "
            "Credentials can be provided via env vars AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY."
    )
    parser.add_argument("--bucket", required=True, help="S3 bucket name.")
    parser.add_argument("--minutes", type=int, required=True,
        help="Delete uploads older than this number of minutes.")
    parser.add_argument("--endpoint-url", type=str, default=None, help="Custom S3 endpoint URL.")
    parser.add_argument("--workers", type=int, default=4, help="Number of worker threads.")
    return parser.parse_args()

def init_worker():
    """
    Initializer for each worker process.
    We create a session/client HERE so each process has its own isolated connection pool.
    """
    global s3_client
    # Optimized config: Retries off (fail fast), max pool match worker (though 1 per process is fine)
    boto_config = Config(
        max_pool_connections=10,
        retries={'max_attempts': 2, 'mode': 'standard'}
    )
    # Re-create the client inside the process to ensure thread/fork safety
    s3_client = boto3.client('s3', endpoint_url=ENDPOINT_URL, config=boto_config)

def abort_upload_task(bucket, key, upload_id):
    """
    The actual task run by the worker. Uses the global client created in init_worker.
    """
    try:
        s3_client.abort_multipart_upload(
            Bucket=bucket,
            Key=key,
            UploadId=upload_id
        )
        return True
    except ClientError as e:
        return f"Error: {e}"

def main():
    args = get_args()

    # Pass endpoint to workers via global variable (simple way for multiprocessing)
    global ENDPOINT_URL
    ENDPOINT_URL = args.endpoint_url

    # 1. SCAN PHASE (Single Threaded)
    # We use a single client here just to list the work.
    main_s3 = boto3.client('s3', endpoint_url=args.endpoint_url)
    cutoff_date = datetime.now(timezone.utc) - timedelta(minutes=args.minutes)

    print(f"--- Scanning bucket '{args.bucket}' ---")
    paginator = main_s3.get_paginator('list_multipart_uploads')

    tasks = []
    try:
        for page in paginator.paginate(Bucket=args.bucket):
            if 'Uploads' in page:
                for upload in page['Uploads']:
                    if upload['Initiated'] < cutoff_date:
                        tasks.append((args.bucket, upload['Key'], upload['UploadId']))
    except ClientError as e:
        print(f"Error accessing bucket: {e}")
        sys.exit(1)

    total_tasks = len(tasks)
    print(f"Found {total_tasks} incomplete uploads to abort.")

    if total_tasks == 0:
        sys.exit(0)

    # 2. DELETE PHASE (Multi-Process)
    print(f"Starting deletion with {args.workers} Processes...")
    start_time = time.time()

    # ProcessPoolExecutor creates completely separate python processes
    # max_workers should typically be <= (CPU Cores * 2)
    with ProcessPoolExecutor(max_workers=args.workers, initializer=init_worker) as executor:
        futures = [executor.submit(abort_upload_task, *task) for task in tasks]

        success_count = 0
        for future in as_completed(futures):
            result = future.result()
            if result is True:
                success_count += 1

            if success_count % 100 == 0:
                print(f"Progress: {success_count}/{total_tasks}...", end='\r')

    duration = time.time() - start_time
    print(f"\n--- Done in {duration:.2f}s ({success_count / duration:.1f} deletes/sec) ---")

if __name__ == "__main__":
    main()
