/**
 * Copyright (c) 2022
 *    Marcus Britanicus (https://gitlab.com/marcusbritanicus)
 *    Abrar (https://gitlab.com/s96Abrar)
 *    rahmanshaber (https://gitlab.com/rahmanshaber)
 *
 * The DFL::Clipboard::Manager class manages the wayland clipboard.
 *
 * IMPL Class
 *  This class will read/write the different clipboard database files
 *  This is purely an implementational detail.
 *  This class can be modified/removed at anytime.
 **/

#include <algorithm>

#include "Table.hpp"

#include <wayqt/DataControl.hpp>

namespace DFLClip = DFL::Clipboard;

DFLClip::Table::Table( QString path, DFLClip::Type type ) {
    mType = type;

    if ( not path.endsWith( "/" ) ) {
        path += "/";
    }

    switch ( type ) {
        case Clipboard: {
            dbf = new QSettings( path + "Clipboard.ini", QSettings::IniFormat );
            break;
        }

        case Selection: {
            dbf = new QSettings( path + "Selection.ini", QSettings::IniFormat );
            break;
        }
    }

    /** Get the date-time of the entries */
    QStringList dateTimes = dbf->childGroups();

    /** Sort them in ascending order */
    dateTimes.sort();

    /** Reverse the list: we need the latest date-time first */
    std::reverse( dateTimes.begin(), dateTimes.end() );

    /** Read the entries */
    for ( QString dateTime: dateTimes ) {
        dbf->beginGroup( dateTime );

        if ( dbf->allKeys().count() ) {
            DFL::Clipboard::Entry *entry = new DFL::Clipboard::Entry( dateTime );
            for ( QString format: dbf->allKeys() ) {
                QByteArray hexData( dbf->value( format ).toByteArray() );
                entry->setData( format, QByteArray::fromHex( hexData ) );
            }
            mEntries << entry;
        }

        dbf->endGroup();
    }
}


DFLClip::Type DFLClip::Table::type() {
    return mType;
}


void DFLClip::Table::clear() {
    dbf->clear();
    mEntries.clear();
}


void DFLClip::Table::addEntry( WQt::MimeData mimeData ) {
    /**
     * Add new entries only if it does not already exist.
     * The user may do one or more of the following:
     *  1. Press Ctrl-C repeatedly.
     *  2. Copy a text that was previously copied
     *  3. The new entry is a subset of previous entry
     *  4. The previous entry is a subset of new entry
     * The last two cases are not handled currently. <== TODO
     *
     * To handle the first and the second cases, we check
     * if the entry already exists. If it doesn, then we
     * can simply promote that entry to the top.
     */

    DFL::Clipboard::Entry *entry = new DFL::Clipboard::Entry( QDateTime::currentDateTime(), mimeData );

    if ( mEntries.contains( entry ) ) {
        /** Get the index of the entry */
        int idx = mEntries.indexOf( entry );

        /** Promote the index to top */
        promoteToTop( idx );

        /** Promote stores the data, so get out */
        return;
    }

    /** Add the latest entry on top */
    mEntries.prepend( entry );

    /** Remove the outdated entries from the bottom */
    while ( mHistorySize < mEntries.count() ) {
        mEntries.removeLast();
    }

    /** Write the entries in to the settings file */
    storeEntries();
}


void DFLClip::Table::removeEntry( int index ) {
    /** Remove the entry at @index */
    mEntries.removeAt( index );

    /** Rewrite the db file */
    storeEntries();
}


void DFLClip::Table::replaceEntry( int index, WQt::MimeData mimeData ) {
    /** Create the entry from the given mimeData */
    DFL::Clipboard::Entry *oldEntry = mEntries.at( index );
    DFL::Clipboard::Entry *entry    = new DFL::Clipboard::Entry( oldEntry->dateTime(), mimeData );

    /** Remove the entry at @index */
    mEntries.replace( index, entry );

    /** Rewrite the db file */
    storeEntries();
}


void DFLClip::Table::promoteToTop( int index ) {
    /** If index is 0; ignore this request */
    if ( not index ) {
        return;
    }

    /** Take the entry at @index */
    DFL::Clipboard::Entry *entry = mEntries.takeAt( index );

    entry->refresh();

    /** Add the entry at the top */
    mEntries.prepend( entry );

    /** Rewrite the db file */
    storeEntries();
}


int DFLClip::Table::count() {
    return mEntries.count();
}


DFL::Clipboard::Entry *DFLClip::Table::entry( int idx ) {
    return mEntries.value( idx );
}


void DFLClip::Table::storeEntries() {
    /** If the user does not want store the history. */
    if ( not mStoreHistory ) {
        return;
    }

    /** This part needs improvements to handle large data */
    else {
        /** Reset the database */
        dbf->clear();

        /** Write each entry as a dedicated section */
        for ( int i = 0; i < mEntries.count(); i++ ) {
            DFL::Clipboard::Entry *entry = mEntries[ i ];
            dbf->beginGroup( entry->dateTime().toString( "yyyyMMdd-hhmmss.zzz" ) );
            for ( QString fmt: entry->formats() ) {
                dbf->setValue( fmt, entry->data( fmt ).toHex() );
            }
            dbf->endGroup();
        }
    }

    /** Ensure it's written to the disk */
    dbf->sync();
}


void DFLClip::Table::setStoreHistory( bool yes ) {
    mStoreHistory = yes;

    if ( not yes ) {
        dbf->clear();
    }
}


void DFLClip::Table::setHistorySize( quint64 size ) {
    mHistorySize = size;

    /** Remove the outdated entries from the bottom */
    while ( mHistorySize < mEntries.count() ) {
        mEntries.removeLast();
    }
}


/**
 * Entry
 */

DFL::Clipboard::Entry::Entry() {
    /** Do nothing here */
}


DFL::Clipboard::Entry::Entry( QString strDT ) {
    /** Set the date-time */
    mDateTime = QDateTime::fromString( strDT, "yyyyMMdd-hhmmss.zzz" );
}


DFL::Clipboard::Entry::Entry( QDateTime dt, WQt::MimeData mimeData ) {
    /** Store the date-time */
    mDateTime = dt;

    /** Store the data */
    mMimeData = mimeData;
}


DFL::Clipboard::Entry::Entry( const DFL::Clipboard::Entry& other ) {
    /** Set the date-time */
    mDateTime = other.mDateTime;

    /** Copy the data from @other */
    mMimeData = other.mMimeData;
}


bool DFL::Clipboard::Entry::isValid() {
    return mDateTime.isValid();
}


QDateTime DFL::Clipboard::Entry::dateTime() {
    return mDateTime;
}


QStringList DFL::Clipboard::Entry::formats() {
    return mMimeData.formats();
}


QByteArray DFL::Clipboard::Entry::data( QString format ) {
    return mMimeData.data( format );
}


void DFL::Clipboard::Entry::setData( QString format, QByteArray data ) {
    mMimeData.setData( format, data );
}


void DFL::Clipboard::Entry::clear() {
    mMimeData.clear();
}


void DFL::Clipboard::Entry::refresh() {
    mDateTime = QDateTime::currentDateTime();
}


DFL::Clipboard::Entry& DFL::Clipboard::Entry::operator=( const DFL::Clipboard::Entry& other ) {
    /** Copy the time and data from @other */

    mDateTime = other.mDateTime;
    mMimeData = other.mMimeData;
    return *this;
}


bool DFL::Clipboard::Entry::operator==( const DFL::Clipboard::Entry& other ) {
    return (mMimeData == other.mMimeData);
}


WQt::MimeData DFL::Clipboard::Entry::data() {
    return mMimeData;
}


WQt::MimeData DFL::Clipboard::Entry::data() const {
    return mMimeData;
}
