/**
 * Copyright (c) 2022-2024
 *    Marcus Britanicus (https://gitlab.com/marcusbritanicus)
 *    Abrar (https://gitlab.com/s96abrar)
 *    rahmanshaber (https://gitlab.com/rahmanshaber)
 *
 * DFL::Storage provides a C++ API for accessing devices and
 * partitions that are exposed by udisks2 via dbus.
 **/

#pragma once

#include <QtDBus>
#include <QString>
#include <QVariantMap>
#include <QStringList>

namespace DFL {
    namespace Storage {
        class Manager;
        class Volume;

        typedef QList<Volume *>                           Volumes;

        /** This map contains all the properties of a device or a block */
        typedef QMap<QString, QVariantMap>                UDisksProperties;

        /** this map contains a map of all the node (device/volume/manager) paths and their properties */
        typedef QMap<QDBusObjectPath, UDisksProperties>   UDisksNodeMap;
    }
}

class DFL::Storage::Manager : public QObject {
    Q_OBJECT

    public:
        explicit Manager( QObject *parent = nullptr );
        ~Manager();

        /**
         * Wait for the UDisks service to become ready.
         */
        bool waitForService( int timeout = -1 ) const;

        /**
         * This function does two things
         * 1. Retrieves the list of block_devices, and drives and emits volumeAdded/deviceAdded signals.
         * 2. Connect to InterfaceAdded/InterfaceRemoved DBus signals, and emits the above signals.
         * Make sure to connect to these signals before calling startMonitoring().
         */
        void startMonitoring();

        /**
         * Disconnect from InterfaceAdded/InterfaceRemoved signals.
         */
        void stopMonitoring();

        /**
         * Wait for ready
         * If @timeout == -1, wait forever.
         * @timeout is in ms.
         */
        bool waitForReady( int timeout = -1 );

        /**
         * Reload all the volumes and devices.
         * 1. This is a costly operation. Do not call repeatedly without reason.
         * 2. All the existing device and volume objects will be deleted as soon
         *    as reload() is called. It's user's responsiblity to make sure no active
         *    instance of these object exists.
         * 3. This will emit volumeAdded/deviceAdded singals for each volume/device
         *    that is retrieved. Make sure earlier instances are dispensed of.
         */
        void reload();

        /**
         * List all the drives. This will be volumes which are drives.
         */
        QStringList drives();

        /**
         * List all the volumes. This will include whole disks, containers and partitions.
         */
        QStringList volumes();

        /**
         * List volumes which contain a valid filesystem
         */
        QStringList validVolumes();

        /**
         * Retrieve the drive this volume belongs to.
         * If there is no volume which acts as the drive to a volume,
         * the current volume will be marked as the drive.
         */
        QString driveForVolume( const QString& path );

        /**
         * Retrieve the volume corresponding to the path.
         * If the path exists, a valid volume path will be returned.
         * If the path exists on a pseudo-fs (f.e. /sys, or /proc)
         * the underlying volume's path will be returned.
         * If the path is invalid, then an empty string is returned.
         */
        QString volumeForPath( const QString& path );

        /** Reference to the volume for the given path */
        QSharedPointer<DFL::Storage::Volume> volume( const QString& path );

        /** A device was added */
        Q_SIGNAL void deviceAdded( const QString& devicePath );

        /** A device was removed */
        Q_SIGNAL void deviceRemoved( const QString& devicePath );

        /** A volume was added (can be whole disk, partition or a container) */
        Q_SIGNAL void volumeAdded( const QString& volumePath );

        /** A volume was removed (can be whole disk, partition or a container) */
        Q_SIGNAL void volumeRemoved( const QString& volumePath );

        /** UDisks service is ready */
        Q_SIGNAL void serviceReady();

        /** We've listed all the connected volumes and devices */
        Q_SIGNAL void ready();

    private slots:
        void handleDeviceAdded( const QDBusObjectPath& objectPath );
        void handleDeviceRemoved( const QDBusObjectPath& objectPath );

    private:
        QDBusInterface *udisksInterface;

        QMap<QString, QSharedPointer<DFL::Storage::Volume> > volumesMap;

        // Factory method for creating a unique_ptr<Volume>
        static QSharedPointer<DFL::Storage::Volume> createVolume( const QString& volPath );
};


class DFL::Storage::Volume : public QObject {
    Q_OBJECT;

    public:

        /**
         * Get the path of this volume.
         */
        QString volumePath() const;

        /** Is this a whole drive (disk)? */
        bool isDrive() const;

        /** Is this a container? */
        bool isContainer() const;

        /** Does this have a valid FS? */
        bool hasValidFilesystem() const;

        /** Is this a swap partition? */
        bool isSwap() const;

        /** Refresh: Read the properties again */
        void refresh() const;

        /** Get all the properties */
        QStringList mountPoints() const;

        /** Get all the properties */
        UDisksProperties properties() const;

        /**
         * Sizes: available/used/total
         * While total size can be obtained even when
         * not mounted, available and used sizes can
         * be queries only when the parition is mounted.
         */
        uint64_t availableSize() const;
        uint64_t usedSize() const;
        uint64_t reservedSize() const;
        uint64_t totalSize() const;

        /**
         * Mount this device.
         * Typical options:
         *   - (fstype, tpye_name): Hint that the fs is of type @type_name
         *   - (as-user, uname): mount as user @uname. Use a valis name, and not a UID.
         *   - (x-udisks-auth, true): mount as calling user, failing which get authorisation
         * For more details, read the manual from https://storaged.org/
         * If the device is successfully mounted, @mounted signal will be emitted with the
         * mount point.
         * If the mount fails, @mountFailed signal will be emitted.
         */
        void mount( const QVariantMap& options = QVariantMap() );

        /** Unmount this device. */
        void unmount( const QVariantMap& options = QVariantMap() );

        /**
         * Power-off this device.
         * Note 1: Applicable only if isDrive() return true.
         * Note 2: This call will fail unless all the volumes are unmounted.
         */
        void powerOff();

        /**
         * Eject this device.
         * Note 1: Applicable only if isDrive() return true.
         * Note 2: This call will fail unless all the volumes are unmounted.
         */
        bool eject() const;

        /**
         * Is this device a removable device.
         * For example: USB, or a CD/DVD.
         * Note: Applicable only if isDrive() return true.
         */
        bool isRemovable() const;

        /**
         * Is this device an optical device.
         * This is a OR of MediaCompatibility and Optical
         * properties from the Drive interface.
         * Note: Applicable only if isDrive() return true.
         */
        bool isOptical() const;

        /**
         * Get the volumes of this drive
         * Note: Applicable only if isDrive() return true.
         */
        QStringList volumes() const;

        /** Mounted. Returns the mount point */
        Q_SIGNAL void mounted( const QString& );

        /** Mount failed. Returns the error message */
        Q_SIGNAL void mountFailed( const QString& );

        /** Unmounted */
        Q_SIGNAL void unmounted();

        /** Unmount failed. Returns the error message */
        Q_SIGNAL void unmountFailed( const QString& );

        /** Device powered-off. Emitted only for drives */
        Q_SIGNAL void poweredOff();

        /** Device powering-off failed. Emitted only for drives */
        Q_SIGNAL void powerOffFailed();

    private:
        /** Constructor */
        explicit Volume( const QString& objectPath );

        /** Desstructor */
        ~Volume();

        /**
         * This will be called by DFL::Storage::Manager when
         * the whole disk is used without a partition manager.
         */
        void markAsDrive();

        /** The path of this volume */
        QString mVolumePath;

        /** The path of device this volume belongs to */
        QString mDevicePath;

        /** This volume has been marked as a drive */
        bool mMarkedAsDrive = false;

        /** Properties */
        mutable UDisksProperties mProperties;

        /** FS sizes */
        uint64_t mFreeSize  = 0;
        uint64_t mUsedSize  = 0;
        uint64_t mResvSize  = 0;
        uint64_t mTotalSize = 0;

        std::unique_ptr<QDBusInterface> filesystemInterface;
        std::unique_ptr<QDBusInterface> partitionInterface;
        std::unique_ptr<QDBusInterface> ptableInterface;
        std::unique_ptr<QDBusInterface> blockInterface;
        std::unique_ptr<QDBusInterface> driveInterface;

        /** We want to be able to init and delete these from Manager */
        friend class DFL::Storage::Manager;
};

Q_DECLARE_METATYPE( DFL::Storage::UDisksProperties );
Q_DECLARE_METATYPE( DFL::Storage::UDisksNodeMap );
