/**
 * Copyright (c) 2022-2024
 *    Marcus Britanicus (https://gitlab.com/marcusbritanicus)
 *    Abrar (https://gitlab.com/s96abrar)
 *    rahmanshaber (https://gitlab.com/rahmanshaber)
 *
 * DFL::Storage provides a C++ API for accessing devices and
 * partitions that are exposed by udisks2 via dbus.
 **/

#include <QFileInfo>
#include <QIODevice>
#include <QXmlStreamReader>

#include "DFStorage.hpp"

#include <sys/statvfs.h>
#include <mntent.h>

#define UDISKS_SERVICE          "org.freedesktop.UDisks2"
#define BLOCK_INTERFACE         "org.freedesktop.UDisks2.Block"
#define DRIVE_INTERFACE         "org.freedesktop.UDisks2.Drive"
#define FILESYSTEM_INTERFACE    "org.freedesktop.UDisks2.Filesystem"
#define PTABLE_INTERFACE        "org.freedesktop.UDisks2.PartitionTable"
#define PART_INTERFACE          "org.freedesktop.UDisks2.Partition"

#define DBUS_PROPS_INTERFACE    "org.freedesktop.DBus.Properties"

// Manager Implementation
DFL::Storage::Manager::Manager( QObject *parent ) : QObject( parent ) {
    qRegisterMetaType<DFL::Storage::UDisksProperties>( "UDisksProperties" );
    qDBusRegisterMetaType<DFL::Storage::UDisksProperties>();

    qRegisterMetaType<DFL::Storage::UDisksNodeMap>( "UDisksNodeMap" );
    qDBusRegisterMetaType<DFL::Storage::UDisksNodeMap>();

    qRegisterMetaType<QByteArrayList>( "QByteArrayList" );
    qDBusRegisterMetaType<QByteArrayList>();

    udisksInterface = new QDBusInterface(
        UDISKS_SERVICE,
        "/org/freedesktop/UDisks2",
        "org.freedesktop.DBus.ObjectManager",
        QDBusConnection::systemBus(),
        this
    );

    if ( udisksInterface->isValid() == false ) {
        qCritical() << "Failed to connect to UDisks2 service:" << udisksInterface->lastError().message();
        return;
    }

    // Initialize the service watcher
    QDBusServiceWatcher serviceWatcher;
    serviceWatcher.setConnection( QDBusConnection::systemBus() );
    serviceWatcher.setWatchMode( QDBusServiceWatcher::WatchForOwnerChange );

    // Add the UDisks2 service to the watcher
    serviceWatcher.addWatchedService( UDISKS_SERVICE );

    // Connect the service owner change signal to a slot
    connect(
        &serviceWatcher, &QDBusServiceWatcher::serviceOwnerChanged, [ this ](const QString& serviceName, const QString&, const QString& newOwner) {
            // Disconnect all the signals. We will reconnect them shortly.
            disconnect();
            stopMonitoring();

            if ( ( serviceName == UDISKS_SERVICE ) && !newOwner.isEmpty() ) {
                emit serviceReady();
            }
        }
    );
}


DFL::Storage::Manager::~Manager() {
    stopMonitoring();

    if ( udisksInterface ) {
        delete udisksInterface;
        udisksInterface = nullptr;
    }
}


bool DFL::Storage::Manager::waitForService( int timeout ) const {
    /** The service is already online */
    if ( QDBusConnection::systemBus().interface()->isServiceRegistered( UDISKS_SERVICE ) ) {
        return true;
    }

    bool readyReceived = false;

    QEventLoop loop;

    /** Timer to stop the event loop on timeout */
    QTimer stopTimer;

    /** We need to run this only once */
    stopTimer.setSingleShot( true );

    connect(
        this, &DFL::Storage::Manager::ready, [ &loop, &stopTimer, &readyReceived ] () {
            /** We've caught this signal */
            readyReceived = true;

            /** Timer is not needed */
            stopTimer.stop();

            /** We're done blocking: quit the event loop */
            loop.quit();
        }
    );

    if ( timeout != -1 ) {
        connect(
            &stopTimer, &QTimer::timeout, [ &loop ] () {
                loop.quit();
            }
        );

        stopTimer.start( timeout );
    }

    /** We're gonna block here */
    loop.exec();

    return readyReceived;
}


void DFL::Storage::Manager::startMonitoring() {
    if ( !udisksInterface->isValid() ) {
        qWarning() << "UDisks2 interface is not valid.";
        return;
    }

    reload();

    QDBusConnection::systemBus().connect(
        UDISKS_SERVICE,
        "/org/freedesktop/UDisks2",
        "org.freedesktop.DBus.ObjectManager",
        "InterfacesAdded",
        this, SLOT(handleDeviceAdded(QDBusObjectPath))
    );

    QDBusConnection::systemBus().connect(
        UDISKS_SERVICE,
        "/org/freedesktop/UDisks2",
        "org.freedesktop.DBus.ObjectManager",
        "InterfacesRemoved",
        this, SLOT(handleDeviceRemoved(QDBusObjectPath))
    );
}


void DFL::Storage::Manager::stopMonitoring() {
    QDBusConnection::systemBus().disconnect(
        UDISKS_SERVICE,
        "/org/freedesktop/UDisks2",
        "org.freedesktop.DBus.ObjectManager",
        "InterfacesAdded",
        this, SLOT(handleDeviceAdded(QDBusObjectPath))
    );

    QDBusConnection::systemBus().disconnect(
        UDISKS_SERVICE,
        "/org/freedesktop/UDisks2",
        "org.freedesktop.DBus.ObjectManager",
        "InterfacesRemoved",
        this, SLOT(handleDeviceRemoved(QDBusObjectPath))
    );
}


bool DFL::Storage::Manager::waitForReady( int timeout ) {
    bool readyReceived = false;

    QEventLoop loop;

    /** Timer to stop the event loop on timeout */
    QTimer stopTimer;

    /** We need to run this only once */
    stopTimer.setSingleShot( true );

    connect(
        this, &DFL::Storage::Manager::ready, [ &loop, &stopTimer, &readyReceived ] () {
            /** We've caught this signal */
            readyReceived = true;

            /** Timer is not needed */
            stopTimer.stop();

            /** We're done blocking: quit the event loop */
            loop.quit();
        }
    );

    if ( timeout != -1 ) {
        connect(
            &stopTimer, &QTimer::timeout, [ &loop ] () {
                loop.quit();
            }
        );

        stopTimer.start( timeout );
    }

    /** We're gonna block here */
    loop.exec();

    return readyReceived;
}


void DFL::Storage::Manager::reload() {
    /**
     * Safe to call this.
     * Any existing references will keep the underlying object alive.
     * When the last reference goes out of scope, the raw pointer is deleted.
     */
    volumesMap.clear();

    QDBusPendingCall asyncCall = udisksInterface->asyncCall( "GetManagedObjects" );
    auto             watcher   = new QDBusPendingCallWatcher( asyncCall, this );

    connect(
        watcher, &QDBusPendingCallWatcher::finished, this, [ this ](QDBusPendingCallWatcher *watcher) {
            QDBusPendingReply<DFL::Storage::UDisksNodeMap> reply = *watcher;

            if ( reply.isError() ) {
                qWarning() << "Error fetching managed objects:" << reply.error().message();
            }

            else {
                /** Get the device and volume object paths */
                DFL::Storage::UDisksNodeMap managedObjects = reply.value();
                for (const auto& objectPath : managedObjects.keys() ) {
                    if ( objectPath.path().contains( "/block_devices/" ) ) {
                        QString volPath = objectPath.path();
                        volumesMap.insert( volPath, createVolume( volPath ) );
                        emit volumeAdded( volPath );
                    }
                }
            }

            watcher->deleteLater();

            emit ready();
        }
    );
}


QStringList DFL::Storage::Manager::drives() {
    QStringList drives;

    for( QString volPath: volumesMap.keys() ) {
        QString drivePath = driveForVolume( volPath );
        if ( drives.contains( drivePath ) == false ) {
            drives << volPath;
        }
    }

    return drives;
}


QStringList DFL::Storage::Manager::volumes() {
    return volumesMap.keys();
}


QStringList DFL::Storage::Manager::validVolumes() {
    QStringList validVols;

    for ( QString volPath: volumesMap.keys() ) {
        if ( volumesMap[ volPath ]->hasValidFilesystem() ) {
            validVols << volPath;
        }
    }

    return validVols;
}


QString DFL::Storage::Manager::driveForVolume( const QString& path ) {
    for( auto vol: volumesMap ) {
        if ( vol->volumes().contains( path ) ) {
            return vol->volumePath();
        }
    }

    QSharedPointer<DFL::Storage::Volume> vol = volume( path );
    vol->markAsDrive();

    return path;
}


QString DFL::Storage::Manager::volumeForPath( const QString& path ) {
    QString normPath = QString( path );

    if ( normPath.endsWith( "/" ) ) {
        normPath.truncate( normPath.length() - 1 );
    }

    QString volumePath;
    QString longest;
    for ( QString volPath: volumesMap.keys() ) {
        QSharedPointer<DFL::Storage::Volume> volume = volumesMap.value( volPath );
        for ( QString mtPt: volume->mountPoints() ) {
            if ( normPath.startsWith( mtPt ) && ( longest.length() < mtPt.length() ) ) {
                longest    = mtPt;
                volumePath = volPath;
            }
        }
    }

    return volumePath;
}


void DFL::Storage::Manager::handleDeviceAdded( const QDBusObjectPath& objectPath ) {
    if ( objectPath.path().contains( "/drives/" ) ) {
        emit deviceAdded( objectPath.path() );
    }

    else if ( objectPath.path().contains( "/block_devices/" ) ) {
        emit volumeAdded( objectPath.path() );
    }
}


void DFL::Storage::Manager::handleDeviceRemoved( const QDBusObjectPath& objectPath ) {
    if ( objectPath.path().contains( "/drives/" ) ) {
        emit deviceRemoved( objectPath.path() );
    }

    else if ( objectPath.path().contains( "/block_devices/" ) ) {
        emit volumeRemoved( objectPath.path() );
    }
}


QSharedPointer<DFL::Storage::Volume> DFL::Storage::Manager::volume( const QString& path ) {
    if ( volumesMap.contains( path ) ) {
        return volumesMap.value( path );
    }

    /** Return an empty shared-pointer of type Volume */
    return QSharedPointer<DFL::Storage::Volume>();
}


QSharedPointer<DFL::Storage::Volume> DFL::Storage::Manager::createVolume( const QString& volPath ) {
    return QSharedPointer<DFL::Storage::Volume>(
        new DFL::Storage::Volume( volPath ), []( DFL::Storage::Volume *ptr ) {
            delete ptr;
        }
    );
}


/**
 * DFL::Storage::Volume implementation
 * The constructor and the destructor are private.
 * Only DFL::Storage::Manager class can create or delete
 * instances of Volume.
 */

// Constructor
// Interfaces that will be used on /org/freedesktop/UDisks2/block_devices/....
// 1. org.freedesktop.UDisks2.Block
// 2. org.freedesktop.UDisks2.PartitionTable (For drives)
// 3. org.freedesktop.UDisks2.Partition
// 4. org.freedesktop.UDisks2.Filesystem
DFL::Storage::Volume::Volume( const QString& objectPath ) : QObject() {
    mVolumePath = objectPath;

    filesystemInterface = std::make_unique<QDBusInterface>(
        UDISKS_SERVICE,
        objectPath,
        "org.freedesktop.UDisks2.Filesystem",
        QDBusConnection::systemBus(),
        this
    );

    partitionInterface = std::make_unique<QDBusInterface>(
        UDISKS_SERVICE,
        objectPath,
        "org.freedesktop.UDisks2.Partition",
        QDBusConnection::systemBus(),
        this
    );

    blockInterface = std::make_unique<QDBusInterface>(
        UDISKS_SERVICE,
        objectPath,
        "org.freedesktop.UDisks2.Block",
        QDBusConnection::systemBus(),
        this
    );

    if ( blockInterface->isValid() ) {
        mDevicePath = blockInterface->property( "Drive" ).value<QDBusObjectPath>().path();

        driveInterface = std::make_unique<QDBusInterface>(
            UDISKS_SERVICE,
            mDevicePath,
            "org.freedesktop.UDisks2.Drive",
            QDBusConnection::systemBus(),
            this
        );

        ptableInterface = std::make_unique<QDBusInterface>(
            UDISKS_SERVICE,
            mVolumePath,
            PTABLE_INTERFACE,
            QDBusConnection::systemBus(),
            this
        );
    }
}


DFL::Storage::Volume::~Volume() {
}


QString DFL::Storage::Volume::volumePath() const {
    return mVolumePath;
}


bool DFL::Storage::Volume::isDrive() const {
    /** Forcibly marked as drive */
    if ( mMarkedAsDrive ) {
        return true;
    }

    /** Create the org.freedesktop.UDisks2.PartitionTable interface */
    QDBusInterface iface( UDISKS_SERVICE, mVolumePath, PTABLE_INTERFACE, QDBusConnection::systemBus() );

    /** Try to query a property an raise an error (if any) */
    QString ptType = iface.property( "Type" ).toString();

    /** There was no error => Interface exists and works -> Volume is a drive. */
    if ( iface.lastError().type() == QDBusError::NoError ) {
        return true;
    }

    return false;
}


bool DFL::Storage::Volume::isContainer() const {
    if ( !blockInterface || !blockInterface->isValid() ) {
        return false;
    }

    return blockInterface->property( "IsContainer" ).toBool();
}


bool DFL::Storage::Volume::hasValidFilesystem() const {
    if ( !blockInterface || !blockInterface->isValid() ) {
        return false;
    }

    return blockInterface->property( "IdType" ).toString().length();
}


bool DFL::Storage::Volume::isSwap() const {
    if ( !blockInterface || !blockInterface->isValid() ) {
        return false;
    }

    return blockInterface->property( "IdType" ).toString() == "swap";
}


void DFL::Storage::Volume::refresh() const {
    mProperties.clear();

    QDBusInterface propsIface( UDISKS_SERVICE, mVolumePath, DBUS_PROPS_INTERFACE, QDBusConnection::systemBus() );

    /** Block properties */
    QDBusReply<QVariantMap> reply = propsIface.call( "GetAll", BLOCK_INTERFACE );

    if ( reply.isValid() ) {
        mProperties.insert( BLOCK_INTERFACE, reply.value() );
    }

    /** Partition properties */
    reply = propsIface.call( "GetAll", PART_INTERFACE );

    if ( reply.isValid() ) {
        mProperties.insert( PART_INTERFACE, reply.value() );
    }

    /** FileSystem properties */
    reply = propsIface.call( "GetAll", FILESYSTEM_INTERFACE );

    if ( reply.isValid() ) {
        mProperties.insert( FILESYSTEM_INTERFACE, reply.value() );
    }

    /** Partition table properties */
    reply = propsIface.call( "GetAll", PTABLE_INTERFACE );

    if ( reply.isValid() ) {
        mProperties.insert( PTABLE_INTERFACE, reply.value() );
    }

    /** Properties from the drive interface */
    if ( isDrive() ) {
        QDBusInterface propsIface2( UDISKS_SERVICE, mDevicePath, DBUS_PROPS_INTERFACE, QDBusConnection::systemBus() );

        /** Block properties */
        reply = propsIface2.call( "GetAll", DRIVE_INTERFACE );

        if ( reply.isValid() ) {
            mProperties.insert( DRIVE_INTERFACE, reply.value() );
        }
    }
}


QStringList DFL::Storage::Volume::mountPoints() const {
    UDisksProperties props = properties();

    QByteArrayList array;

    /** Check if we have a valid "MountPoints" property */
    if ( props[ FILESYSTEM_INTERFACE ][ "MountPoints" ].isValid() ) {
        QDBusArgument mtPtsRaw = props[ FILESYSTEM_INTERFACE ][ "MountPoints" ].value<QDBusArgument>();
        mtPtsRaw >> array;
    }

    QStringList mtPts;
    for ( QByteArray mtPt: array ) {
        mtPts << QString( mtPt.data() );
    }

    return mtPts;
}


DFL::Storage::UDisksProperties DFL::Storage::Volume::properties() const {
    if ( mProperties.count() ) {
        return mProperties;
    }

    /** Retrieve the properties */
    refresh();

    return mProperties;
}


uint64_t DFL::Storage::Volume::availableSize() const {
    if ( mountPoints().count() ) {
        return mFreeSize;
    }

    return -1;
}


uint64_t DFL::Storage::Volume::usedSize() const {
    if ( mountPoints().count() ) {
        return mUsedSize;
    }

    return -1;
}


uint64_t DFL::Storage::Volume::reservedSize() const {
    if ( mountPoints().count() ) {
        return mResvSize;
    }

    return -1;
}


uint64_t DFL::Storage::Volume::totalSize() const {
    return mProperties[ FILESYSTEM_INTERFACE ][ "SIZE" ].toULongLong();
}


void DFL::Storage::Volume::mount( const QVariantMap& options ) {
    // Perform the asynchronous DBus call
    QDBusPendingCall pendingCall = filesystemInterface->asyncCall( "Mount", options );

    // Create a watcher to handle the reply
    QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher( pendingCall, this );

    // Connect the watcher's finished signal to a lambda or slot
    QObject::connect(
        watcher, &QDBusPendingCallWatcher::finished, this, [ this ](QDBusPendingCallWatcher *watcher) {
            // Check if the call was successful
            QDBusPendingReply<QString> reply = *watcher;

            if ( reply.isError() ) {
                // Handle the error
                qWarning() << "Mount failed:" << reply.error().message();
                emit mountFailed( reply.error().message() );
            }

            else {
                // Refresh the properties
                refresh();

                // Emit the mounted signal on success
                emit mounted( reply.value() );
            }

            // Clean up the watcher
            watcher->deleteLater();

            // Calculate the available/free/used size
            if ( mountPoints().length() ) {
                QString mtPt = mountPoints().at( 0 );
                struct statvfs stat;
                if ( statvfs( mtPt.toUtf8().constData(), &stat ) != 0 ) {
                    qDebug() << "Error: Failed to get filesystem statistics for " << mtPt;
                    return;
                }

                mTotalSize = stat.f_blocks * stat.f_frsize;
                mFreeSize  = stat.f_bavail * stat.f_frsize;
                mUsedSize  = mTotalSize - mFreeSize;
                mResvSize  = (stat.f_bfree - stat.f_bavail) * stat.f_frsize;
            }
        }
    );
}


void DFL::Storage::Volume::unmount( const QVariantMap& options ) {
    // Perform the asynchronous DBus call
    QDBusPendingCall pendingCall = filesystemInterface->asyncCall( "Unmount", options );

    // Create a watcher to handle the reply
    QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher( pendingCall, this );

    // Connect the watcher's finished signal to a lambda or slot
    QObject::connect(
        watcher, &QDBusPendingCallWatcher::finished, this, [ this ](QDBusPendingCallWatcher *watcher) {
            // Check if the call was successful
            QDBusPendingReply<void> reply = *watcher;

            if ( reply.isError() ) {
                // Handle the error
                qWarning() << "Unmount failed:" << reply.error().message();
                emit unmountFailed( reply.error().message() );
            }

            else {
                // Refresh the properties
                refresh();

                // Emit the mounted signal on success
                emit unmounted();
            }

            // Clean up the watcher
            watcher->deleteLater();
        }
    );
}


void DFL::Storage::Volume::powerOff() {
    if ( driveInterface == nullptr ) {
        return;
    }

    // Make the asynchronous D-Bus call
    QDBusPendingCall asyncCall = driveInterface->asyncCall("PowerOff");

    // Create a watcher to monitor the pending call
    QDBusPendingCallWatcher* watcher = new QDBusPendingCallWatcher(asyncCall, this);

    // Connect the watcher's finished signal to a lambda
    connect(
        watcher, &QDBusPendingCallWatcher::finished, this, [this, watcher]() {
            if ( watcher->reply().type() == QDBusMessage::ReplyMessage ) {
                emit poweredOff();
            }

            else {
                emit powerOffFailed();
            }

            // Clean up the watcher
            watcher->deleteLater();
        }
    );
}


bool DFL::Storage::Volume::eject() const {
    if ( driveInterface == nullptr ) {
        return false;
    }

    QDBusPendingReply<void> reply = driveInterface->asyncCall( "Eject" );

    reply.waitForFinished();
    return reply.isValid();
}


bool DFL::Storage::Volume::isRemovable() const {
    if ( driveInterface == nullptr ) {
        return false;
    }

    // Query the Removable property
    bool isRemovable = driveInterface->property( "Removable" ).toBool() || driveInterface->property( "Optical" ).toBool();

    // If not removable, check MediaCompatibility for CD/DVD/BluRay drives
    if ( !isRemovable ) {
        QStringList mediaTypes = driveInterface->property( "MediaCompatibility" ).toStringList();

        for (const QString& type : mediaTypes) {
            bool optical = type.contains( "optical", Qt::CaseInsensitive );
            bool cd      = type.contains( "cd", Qt::CaseInsensitive );
            bool dvd     = type.contains( "dvd", Qt::CaseInsensitive );
            bool bluray  = type.contains( "blu-ray", Qt::CaseInsensitive );

            if ( optical || cd || dvd || bluray ) {
                return true;
            }
        }
    }

    return isRemovable;
}


bool DFL::Storage::Volume::isOptical() const {
    if ( driveInterface == nullptr ) {
        return false;
    }

    // Query the Removable property
    bool isOptical = driveInterface->property( "Optical" ).toBool();

    // If not removable, check MediaCompatibility for CD/DVD/BluRay drives
    if ( !isOptical ) {
        QStringList mediaTypes = driveInterface->property( "MediaCompatibility" ).toStringList();

        for (const QString& type : mediaTypes) {
            bool optical = type.contains( "optical", Qt::CaseInsensitive );
            bool cd      = type.contains( "cd", Qt::CaseInsensitive );
            bool dvd     = type.contains( "dvd", Qt::CaseInsensitive );
            bool bluray  = type.contains( "blu-ray", Qt::CaseInsensitive );

            if ( optical || cd || dvd || bluray ) {
                return true;
            }
        }
    }

    return isOptical;
}


QStringList DFL::Storage::Volume::volumes() const {
    if ( !isDrive() ) {
        return QStringList();
    }

    if ( !driveInterface || !driveInterface->isValid() ) {
        return QStringList();
    }

    refresh();

    QDBusArgument arg = mProperties[ PTABLE_INTERFACE ][ "Partitions" ].value<QDBusArgument>();
    QList<QDBusObjectPath> parts;
    arg.beginArray();
    while (!arg.atEnd()) {
        QDBusObjectPath path;
        arg >> path;
        parts.append(path);
    }
    arg.endArray();

    QStringList volumePaths;
    for ( QDBusObjectPath volPath: parts ) {
        volumePaths << volPath.path();
    }

    /** This volume itself is the partition and the drive */
    if ( mMarkedAsDrive && volumePaths.isEmpty() ) {
        return { mVolumePath };
    }

    return volumePaths;
}


void DFL::Storage::Volume::markAsDrive() {
    mMarkedAsDrive = true;

    refresh();
}
