!! Copyright (C) 2023 F. Bonafé
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
#include "global.h"

module mxll_elec_coupling_oct_m
  use batch_ops_oct_m
  use debug_oct_m
  use derivatives_oct_m
  use global_oct_m
  use grid_oct_m
  use hamiltonian_elec_base_oct_m
  use, intrinsic :: iso_fortran_env
  use math_oct_m
  use mesh_oct_m
  use messages_oct_m
  use namespace_oct_m
  use parser_oct_m
  use profiling_oct_m
  use space_oct_m
  use states_elec_dim_oct_m
  use wfs_elec_oct_m

  implicit none

  private
  public ::                          &
    mxll_coupling_t,                 &
    mxll_coupling_init,              &
    mxll_coupling_calc,              &
    mxll_coupling_end,               &
    set_electric_quadrupole_pot,     &
    magnetic_dipole_coupling

  type :: mxll_coupling_t
    integer :: coupling_mode   !< Type of coupling between Maxwell and matter
    integer :: coupling_terms     !< Multipole expansion terms that will be added to the Hamiltonian
    integer :: dipole_field    !< Way to calculate the E field vector from space-dependent E field, for length gauge coupling
    real(real64), allocatable :: e_field(:,:)    !< Real-space E field from Maxwell solver (can be electric or vec pot ATM)
    real(real64), allocatable :: b_field(:,:)    !< Real-space B field from Maxwell solver
    real(real64), allocatable :: vec_pot(:,:)    !< Real-space A field from Maxwell solver
    real(real64), allocatable :: e_field_dip(:)    !< Electric field vector used for for dipole coupling
    real(real64), allocatable :: vec_pot_dip(:)    !< Electric field vector used for for dipole coupling
    real(real64), allocatable :: b_field_dip(:)  !< Magnetic field vector used for for magnetic dipole coupling
    real(real64), allocatable :: e_quadrupole_pot(:) !< Electric field quadrupole potential, if requested
    logical            :: calc_field_dip = .false.   !< If dipole field will be calculated
    logical            :: add_electric_dip = .false.   !< If electric dipole coupling is included
    logical            :: add_electric_quad = .false.   !<If electric quadrupole coupling is included
    logical            :: add_magnetic_dip = .false.   !< If magnetic dipole coupling is included
    logical            :: add_zeeman = .false.   !< If Zeeman term is included
    logical            :: add_non_uniform_vec_pot = .false. !< If A(r,t) should be included in the Pauli Hamiltonian
    real(real64) :: center_of_mass(1:3)    !< Center of mass of the ions
    integer :: center_of_mass_ip    !< Index of the center of mass of the ions
    integer :: center_of_mass_rankmin    !< Rank which contains the center_of_mass point
    logical :: test_equad = .false.
    type(derivatives_t), pointer, private :: der !< pointer to derivatives
    real(real64), private :: mass
  end type mxll_coupling_t

  ! See explanation in the MaxwellCouplingMode variable description.
  integer, public, parameter :: &
    NO_MAXWELL_COUPLING      = 0,  &
    LENGTH_GAUGE_DIPOLE      = 1,  &
    MULTIPOLAR_EXPANSION     = 2,  &
    VELOCITY_GAUGE_DIPOLE    = 3,  &
    FULL_MINIMAL_COUPLING    = 4

  integer, public, parameter :: &
    DIPOLE_AVERAGE    = 0,  &
    DIPOLE_AT_COM     = 1


contains

  ! ------------------------------------------------------------------------
  !> @brief Parse variables and initialize Maxwell coupling
  subroutine mxll_coupling_init(this, d, gr, namespace, mass)
    type(mxll_coupling_t),  intent(inout)  :: this
    type(states_elec_dim_t),intent(in)     :: d
    type(grid_t),   target, intent(in)     :: gr
    type(namespace_t),      intent(in)     :: namespace
    real(real64), intent(in) :: mass

    integer :: terms_default, fmc_default, multipolar_terms, pauli_terms

    PUSH_SUB(mxll_coupling_init)

    this%der => gr%der
    this%mass = mass

    !%Variable MaxwellCouplingMode
    !%Type integer
    !%Default none
    !%Section Hamiltonian
    !%Description
    !% This variable selects the level of light-matter coupling with electromagnetic fields from the Maxwell solver.
    !% Each option defines a coupling Hamiltonian <math> H_{int}</math>.
    !%Option none 0
    !% No coupling.
    !%Option length_gauge_dipole 1
    !% Dipole level in length gauge with transverse electric field E(t):
    !% <math> H_{int} = -r.E </math>
    !% The option MaxwellDipoleField should be set to define if <math>E</math> is calculated as an average at evaluated
    !% at the center of mass (by default, dipole coupling with the average electric field will be used).
    !%Option multipolar_expansion 2
    !% The option MultipoleExpansionTerms selects the terms to be included (electric dipole, electric quadrupole
    !% and/or magnetic dipole).
    !%Option velocity_gauge_dipole 3
    !% Coupling at dipole level in velocity gauge with transverse vector potential A(t).
    !% <math> H_{int} = 1/2m (2 A(t).p + A^2(t)) </math>
    !% The option MaxwellDipoleField should be set.
    !%Option full_minimal_coupling 4
    !% Full vector potential A(r,t) coupled to the momentum.
    !% <math> H_{int} = 1/2m (2 A(r,t).p + A^2(r,t)) </math>
    !%End
    call parse_variable(namespace, 'MaxwellCouplingMode', NO_MAXWELL_COUPLING, &
      this%coupling_mode)
    this%add_electric_dip = (this%coupling_mode == LENGTH_GAUGE_DIPOLE)

    if (this%coupling_mode == MULTIPOLAR_EXPANSION) then

      !%Variable MultipolarExpansionTerms
      !%Type flag
      !%Default electric_dipole + electric_quadrupole + magnetic_dipole
      !%Section Hamiltonian
      !%Description
      !% Terms to be included in the multipolar expansion.
      !% For this type of coupling to make sense, the E field has to be calculated at the center of mass (not averaged).
      !%Option electric_dipole bit(1)
      !% Adds electric dipole term in length gauge, uses transverse electric field E(t):
      !% <math> H_{int} = -e (r.E) </math>
      !%Option electric_quadrupole bit(2)
      !% Adds electric quadrupole term:
      !% <math> H_{int}  = \frac{1}{2} e (r . Q . r )] <\math>
      !% where Q is the outer product of gradient and electric field:  <math> Q_{ij} = \partial_i E_j |_{r=r_0} </math>
      !%Option magnetic_dipole bit(3)
      !% Adds magnetic dipole term:
      !% <math> H_{int}  = - i (e \hbar /2m) \sum_i (\vec{B}(r_0).(\vec{r} x \nabla)) </math>
      !%End
      terms_default = &
        OPTION__MULTIPOLAREXPANSIONTERMS__ELECTRIC_DIPOLE + &
        OPTION__MULTIPOLAREXPANSIONTERMS__ELECTRIC_QUADRUPOLE + &
        OPTION__MULTIPOLAREXPANSIONTERMS__MAGNETIC_DIPOLE

      call parse_variable(namespace, 'MultipolarExpansionTerms', terms_default, multipolar_terms)

      if (bitand(multipolar_terms, OPTION__MULTIPOLAREXPANSIONTERMS__ELECTRIC_DIPOLE) /= 0) then
        this%add_electric_dip = .true.
      end if
      if (bitand(multipolar_terms, OPTION__MULTIPOLAREXPANSIONTERMS__ELECTRIC_QUADRUPOLE) /=0) then
        this%add_electric_quad = .true.
      end if
      if (bitand(multipolar_terms, OPTION__MULTIPOLAREXPANSIONTERMS__MAGNETIC_DIPOLE) /=0) then
        this%add_magnetic_dip = .true.
      end if

    end if

    if (this%coupling_mode == FULL_MINIMAL_COUPLING) then

      !%Variable PauliHamiltonianTerms
      !%Type flag
      !%Default non_uniform_vector_potential + zeeman_term
      !%Section Hamiltonian
      !%Description
      !% Terms to be included in the Pauli Hamiltonnian.
      !%Option non_uniform_vector_potential bit(1)
      !% Adds non-uniform vector potential to the canonical momentum.
      !%Option zeeman_term bit(2)
      !% Adds the non-uniform Zeeman potential.
      !%End

      if (d%nspin == 2) then
        fmc_default = OPTION__PAULIHAMILTONIANTERMS__NON_UNIFORM_VECTOR_POTENTIAL + &
          OPTION__PAULIHAMILTONIANTERMS__ZEEMAN_TERM
      else
        fmc_default = OPTION__PAULIHAMILTONIANTERMS__NON_UNIFORM_VECTOR_POTENTIAL
      end if

      call parse_variable(namespace, 'PauliHamiltonianTerms', fmc_default, pauli_terms)
      if (bitand(pauli_terms, OPTION__PAULIHAMILTONIANTERMS__NON_UNIFORM_VECTOR_POTENTIAL) /= 0) then
        this%add_non_uniform_vec_pot = .true.
      end if
      if (bitand(pauli_terms, OPTION__PAULIHAMILTONIANTERMS__ZEEMAN_TERM) /=0) then
        this%add_zeeman = .true.
      end if

    end if

    !%Variable MaxwellDipoleField
    !%Type integer
    !%Default average
    !%Section Hamiltonian
    !%Description
    !% This variable selects the method to get the E field vector at the position of the system
    !% if Maxwell-matter coupling at dipole level within length gauge is done.
    !%Option average 0
    !% Transverse E field is averaged in the volume occupied by the matter system.
    !%Option center_of_mass 1
    !% Tranverse E field is evaluated at the center of mass of the matter system (at a single point).
    !%End
    call parse_variable(namespace, 'MaxwellDipoleField', DIPOLE_AVERAGE, &
      this%dipole_field)

    if (this%coupling_mode /= NO_MAXWELL_COUPLING) then
      SAFE_ALLOCATE(this%e_field(1:gr%np, 1:gr%box%dim))
      SAFE_ALLOCATE(this%b_field(1:gr%np, 1:gr%box%dim))
      SAFE_ALLOCATE(this%vec_pot(1:gr%np, 1:gr%box%dim))
      this%e_field = M_ZERO
      this%b_field = M_ZERO
      this%vec_pot = M_ZERO
      SAFE_ALLOCATE(this%e_field_dip(1:gr%box%dim))
      this%e_field_dip = M_ZERO
      SAFE_ALLOCATE(this%b_field_dip(1:gr%box%dim))
      this%b_field_dip = M_ZERO
      SAFE_ALLOCATE(this%vec_pot_dip(1:gr%box%dim))
      this%vec_pot_dip = M_ZERO
    end if

    if (this%coupling_mode == MULTIPOLAR_EXPANSION .and. this%add_electric_quad) then
      SAFE_ALLOCATE(this%e_quadrupole_pot(1:gr%np))
    end if

    call mxll_quadrupole_test_init(this, gr, namespace)

    POP_SUB(mxll_coupling_init)

  end subroutine mxll_coupling_init

  ! ------------------------------------------------------------------------
  !> @brief Initializes quadrupole test when requested.
  !> The test applies an electric field defined as E=(0.2 x, 0, 0)
  !> which produces a quadrupole potential that is harmonic: V=0.1 x^2.
  subroutine mxll_quadrupole_test_init(this, gr, namespace)
    type(mxll_coupling_t),  intent(inout)  :: this
    type(grid_t),           intent(in)     :: gr
    type(namespace_t),      intent(in)     :: namespace

    integer :: ip, rankmin
    real(real64) :: dmin

    PUSH_SUB(mxll_quadrupole_test_init)

    !%Variable MaxwellTestQuadrupole
    !%Type logical
    !%Default no
    !%Section Maxwell
    !%Description
    !% Override Maxwell field with linear E field for testing.
    !%End
    call parse_variable(namespace, 'MaxwellTestQuadrupole', .false., this%test_equad)

    if (this%test_equad) then
      SAFE_ALLOCATE(this%e_field(1:gr%np, 1:gr%box%dim))
      this%e_field = M_ZERO
      do ip = 1, gr%np
        this%e_field(ip,1) = 0.02_real64 * gr%x(ip,1)
      end do
      this%calc_field_dip = .true.
      this%center_of_mass(1:3) = M_ZERO
      this%center_of_mass_ip = mesh_nearest_point(gr, this%center_of_mass, dmin, rankmin)
      this%center_of_mass_rankmin = rankmin
      SAFE_ALLOCATE(this%e_quadrupole_pot(1:gr%np))
    end if

    POP_SUB(mxll_quadrupole_test_init)

  end subroutine mxll_quadrupole_test_init

  ! ------------------------------------------------------------------------
  !> @brief Add the Maxwell coupling to the electronic Hamiltonian
  subroutine mxll_coupling_calc(this, hm_base, mesh, d, space)
    type(mxll_coupling_t),         intent(inout) :: this
    type(hamiltonian_elec_base_t), intent(inout) :: hm_base
    type(mesh_t),                  intent(in)    :: mesh
    type(states_elec_dim_t),       intent(in)    :: d
    class(space_t),                intent(in)    :: space

    integer :: ip, ispin, idir

    PUSH_SUB(mxll_coupling_calc)

    ! coupling to Maxwell field
    select case (this%coupling_mode)

    case (LENGTH_GAUGE_DIPOLE, MULTIPOLAR_EXPANSION)

      if (this%add_electric_dip) then
        do ispin = 1, d%spin_channels
          do ip = 1, mesh%np
            ! The -1 sign is missing here. Check epot.F90 for the explanation.
            hm_base%potential(ip, ispin) = hm_base%potential(ip, ispin) + &
              sum(this%e_field_dip(1:space%dim) * (mesh%x(ip, 1:space%dim) - this%center_of_mass(1:space%dim)))
          end do
        end do
      end if

      if (this%add_electric_quad .and. this%calc_field_dip) then
        ! the calc_field_dip condition prevents calling the routine before initializing the interactions
        call set_electric_quadrupole_pot(this, mesh)
        do ispin = 1, d%spin_channels
          do ip = 1, mesh%np
            hm_base%potential(ip, ispin) = hm_base%potential(ip, ispin) + this%e_quadrupole_pot(ip)
          end do
        end do
      end if

      ! magnetic_dipole term is added in hamiltonian_elec_inc.F90

    case (VELOCITY_GAUGE_DIPOLE)
      call hm_base%allocate_field(mesh, FIELD_UNIFORM_VECTOR_POTENTIAL, .false.)
      ! The minus sign is here is for the wrong convention of Octopus for the phase
      hm_base%uniform_vector_potential(1:space%dim) = hm_base%uniform_vector_potential(1:space%dim) - &
        this%vec_pot_dip(1:space%dim)

    case (FULL_MINIMAL_COUPLING)
      call hm_base%allocate_field(mesh, FIELD_VECTOR_POTENTIAL, .false.)
      if (this%add_non_uniform_vec_pot) then
        do idir = 1, mesh%box%dim
          hm_base%vector_potential(idir, :) = hm_base%vector_potential(idir, :) + this%vec_pot(:, idir)
        end do
      end if
      if (this%add_zeeman) then
        call hm_base%allocate_field(mesh, FIELD_MAGNETIC_FIELD, .false.)
        hm_base%magnetic_field(:,:) = hm_base%magnetic_field(:,:) + this%b_field(:,:)
      end if

    end select

    POP_SUB(mxll_coupling_calc)

  end subroutine mxll_coupling_calc

  ! ------------------------------------------------------------------------
  !> @brief Finalize and deallocate Maxwell coupling arrays
  subroutine mxll_coupling_end(this)
    type(mxll_coupling_t), intent(inout) :: this

    PUSH_SUB(mxll_coupling_end)

    SAFE_DEALLOCATE_A(this%e_field)
    SAFE_DEALLOCATE_A(this%b_field)
    SAFE_DEALLOCATE_A(this%vec_pot)
    SAFE_DEALLOCATE_A(this%e_field_dip)
    SAFE_DEALLOCATE_A(this%b_field_dip)
    SAFE_DEALLOCATE_A(this%vec_pot_dip)
    SAFE_DEALLOCATE_A(this%e_quadrupole_pot)

    POP_SUB(mxll_coupling_end)
  end subroutine mxll_coupling_end


  ! ------------------------------------------------------------------------
  !> @brief Computes the electric quadrupole potential
  !! \f$H^{EQ} = \frac{1}{2} e [ (\vec{r} \cdot \mathbb{Q} \cdot \vec{r} )\phi \f$
  !! where \f$ \mathbb{Q} _{i j}=\left.\partial_i E_j^{\perp}\right|_{\vec{r}=\vec{r}_0} \f$
  subroutine set_electric_quadrupole_pot(this, mesh)
    type(mxll_coupling_t),    intent(inout)  :: this
    type(mesh_t),             intent(in)     :: mesh

    real(real64), allocatable   :: e_field_quadrupole_tensor(:,:), tmp_partial(:,:), this_e_field(:,:)
    real(real64)  :: r(3), tensor_dot_rr(3)
    integer :: ip, i, dims

    PUSH_SUB(set_electric_quadrupole_pot)

    call profiling_in('SET_ELECTRIC_QUADRUPOLE')

    SAFE_ALLOCATE(e_field_quadrupole_tensor(1:mesh%box%dim, 1:mesh%box%dim))
    SAFE_ALLOCATE(this_e_field(1:mesh%np_part, 1:mesh%box%dim))
    SAFE_ALLOCATE(tmp_partial(1:mesh%np, 1:mesh%box%dim))

    this_e_field(1:mesh%np,:) = this%e_field(1:mesh%np,:)

    e_field_quadrupole_tensor = M_ZERO
    dims = this%der%dim
    do i = 1, dims
      call dderivatives_grad(this%der, this_e_field(:, i), tmp_partial)
      if (mesh%mpi_grp%rank == this%center_of_mass_rankmin) then
        e_field_quadrupole_tensor(i, 1:dims) = tmp_partial(this%center_of_mass_ip, 1:dims)
      end if
    end do
    call mesh%allreduce(e_field_quadrupole_tensor)

    do ip = 1, mesh%np
      r(:) = mesh%x(ip,:) - this%center_of_mass(:)
      tensor_dot_rr = matmul(e_field_quadrupole_tensor, r)
      ! This term is +1/2 r.Q.r and not -1/2 r.Q.r (as e=+1 in Octopus)
      this%e_quadrupole_pot(ip) = M_HALF * dot_product(r, tensor_dot_rr)
    end do

    SAFE_DEALLOCATE_A(e_field_quadrupole_tensor)
    SAFE_DEALLOCATE_A(tmp_partial)
    SAFE_DEALLOCATE_A(this_e_field)

    call profiling_out('SET_ELECTRIC_QUADRUPOLE')
    POP_SUB(set_electric_quadrupole_pot)

  end subroutine set_electric_quadrupole_pot


  ! ------------------------------------------------------------------------
  !> @brief Computes the magnetic dipole term of the Hamiltonian.
  !! \f$H^{MD} = (-i e \hbar /2m) \sum_i (\vec{B}(r_0).(\vec{r} x \nabla)) \psi_i \f$
  !! This routine acually implements the equivalent expression
  !! (\vec{B}(r_0) \times \vec{r}) . (\nabla \psi)
  subroutine magnetic_dipole_coupling(this, psib, hpsib)
    type(mxll_coupling_t),  intent(in)    :: this
    type(wfs_elec_t),       intent(inout) :: psib
    type(wfs_elec_t),       intent(inout) :: hpsib

    integer             :: ip, idir
    real(real64)        :: rtmp(3)
    real(real64), allocatable  :: bxr(:,:)
    type(wfs_elec_t) :: gradb(this%der%dim)
    type(wfs_elec_t) :: bxr_dot_gradb

    PUSH_SUB(magnetic_dipole_coupling)

    SAFE_ALLOCATE(bxr(1:this%der%mesh%np,1:3))
    call profiling_in('MAGNETIC_DIPOLE')

    do ip = 1, this%der%mesh%np
      rtmp(:) = this%der%mesh%x(ip,:) - this%center_of_mass
      bxr(ip,:) = dcross_product(this%b_field_dip, rtmp)
    end do

    call zderivatives_batch_grad(this%der, psib, gradb)

    call hpsib%copy_to(bxr_dot_gradb)
    do idir = 1, this%der%dim
      call batch_mul(this%der%mesh%np, bxr(:,idir), gradb(idir), bxr_dot_gradb)
      call batch_axpy(this%der%mesh%np, M_zI*M_HALF/this%mass, bxr_dot_gradb, hpsib)
    end do

    call bxr_dot_gradb%end()
    do idir = 1, this%der%dim
      call gradb(idir)%end()
    end do
    SAFE_DEALLOCATE_A(bxr)

    call profiling_out('MAGNETIC_DIPOLE')
    POP_SUB(magnetic_dipole_coupling)
  end subroutine magnetic_dipole_coupling

end module mxll_elec_coupling_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
