!! Copyright (C) 2019-2024 M. Oliveira, H. Appel, N. Tancogne-Dejean
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module multisystem_run_oct_m
  use debug_oct_m
  use electrons_oct_m
  use global_oct_m
  use messages_oct_m
  use multisystem_basic_oct_m
  use multisystem_debug_oct_m
  use namespace_oct_m
  use parser_oct_m
  use profiling_oct_m
  use restart_oct_m
  use system_oct_m
  use td_oct_m
  use walltimer_oct_m

  implicit none

  private
  public :: multisystem_run

contains

  ! ---------------------------------------------------------
  subroutine multisystem_run(systems, from_scratch)
    class(multisystem_basic_t), intent(inout) :: systems
    logical,                    intent(in)    :: from_scratch

    logical :: trigger_restart, stop_code, stop_loop
    logical :: restart_read

    PUSH_SUB(multisystem_run)

    call multisystem_debug_init("debug/multisystem_execution.log", global_namespace, systems%grp)

    call messages_write('Info: Running Multi-system calculation')
    call messages_info(namespace=systems%namespace)

    ! Read restart files or set initial conditions
    call systems%init_iteration_counters()
    if (.not. from_scratch) then
      restart_read = systems%restart_read()
    else
      restart_read = .false.
    end if
    if (restart_read) then
      message(1) = "Successfully read restart data for all system."
      call messages_info(1, namespace=systems%namespace)
    else
      call systems%initialize()
    end if

    call systems%algorithm_start()

    call multisystem_debug_write_marker(systems%namespace, event_marker_t("multisystem_run_start"))

    ! The full loop
    stop_loop = .false.
    do while (.not. systems%algorithm_finished())
      ! Execute algorithm until next barrier
      call systems%execute_algorithm()

      ! determine cases in which to trigger writing restart files
      stop_code = clean_stop(systems%grp%comm) .or. walltimer_alarm(systems%grp%comm)
      trigger_restart = stop_code .or. restart_walltime_period_alarm(systems%grp%comm)

      if (trigger_restart .and. .not. stop_loop) then
        call systems%start_barrier(systems%next_time_on_largest_dt(), BARRIER_RESTART)
        stop_loop = stop_code
        trigger_restart = .false.
      end if
      if (systems%arrived_at_barrier(BARRIER_RESTART)) then
        call systems%restart_write()
        call systems%end_barrier(BARRIER_RESTART)
        if (stop_loop) exit
      end if
    end do

    if (systems%algorithm_finished()) then
      call systems%restart_write()
    end if

    call multisystem_debug_write_marker(systems%namespace, event_marker_t("multisystem_run_finish"))

    call systems%algorithm_finish()

    call multisystem_debug_end()

    POP_SUB(multisystem_run)
  end subroutine multisystem_run

end module multisystem_run_oct_m
