!! Copyright (C)  2024 N. Tancogne-Dejean
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module minimizer_scf_oct_m
  use algorithm_oct_m
  use debug_oct_m
  use global_oct_m
  use minimizer_algorithm_oct_m
  use iteration_counter_oct_m

  implicit none

  private
  public ::                       &
    minimizer_scf_t

  !> Implements a minimizer algorithm for SCF calculations
  type, extends(minimizer_algorithm_t) :: minimizer_scf_t
    private
  contains
    procedure :: do_operation => minimizer_scf_do_operation       !< @copydoc minimizer_scf_do_operation
  end type minimizer_scf_t

  interface minimizer_scf_t
    procedure minimizer_scf_constructor
  end interface minimizer_scf_t

  ! Specific SCF algorithmic operations identifiers
  character(len=ALGO_LABEL_LEN), public, parameter :: &
    GS_SCF_START        = 'GS_SCF_START',                 &
    GS_SCF_FINISH       = 'GS_SCF_FINISH',                &
    GS_SCF_ITERATION    = 'GS_SCF_ITERATION'

  ! Specific SCF algorithmic operations
  type(algorithmic_operation_t), public, parameter :: &
    OP_GS_START         = algorithmic_operation_t(GS_SCF_START,        'Starting ground state SCF'),  &
    OP_GS_FINISH        = algorithmic_operation_t(GS_SCF_FINISH,       'Finishing ground state SCF'), &
    OP_GS_SCF_ITERATION = algorithmic_operation_t(GS_SCF_ITERATION,    'SCF iteration for the electrons')



contains

  ! ---------------------------------------------------------
  function minimizer_scf_constructor() result(this)
    type(minimizer_scf_t), pointer    :: this

    PUSH_SUB(minimizer_scf_constructor)

    allocate(this)

    this%start_operation = OP_GS_START
    this%final_operation = OP_GS_FINISH

    call this%add_operation(OP_UPDATE_COUPLINGS)
    call this%add_operation(OP_UPDATE_INTERACTIONS)
    call this%add_operation(OP_GS_SCF_ITERATION)
    call this%add_operation(OP_ITERATION_DONE)
    call this%add_operation(OP_REWIND_ALGORITHM)

    this%algo_steps = 1


    POP_SUB(minimizer_scf_constructor)
  end function minimizer_scf_constructor

  !> @brief Try to perform one operation of the algorithm. Return .true. if sucessful.
  !!
  logical function minimizer_scf_do_operation(this, operation) result(done)
    class(minimizer_scf_t),        intent(inout) :: this
    type(algorithmic_operation_t), intent(in)    :: operation

    select case(operation%id)
    case (GS_SCF_ITERATION) ! If we are converged, this becomes a void operation
      if (this%finished()) done = .true.
    case default
      ! Nothing done yet here
      done = .false.
    end select

  end function minimizer_scf_do_operation


end module minimizer_scf_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
