!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module target_userdefined_oct_m
  use debug_oct_m
  use density_oct_m
  use global_oct_m
  use grid_oct_m
  use hamiltonian_elec_oct_m
  use io_oct_m
  use ions_oct_m
  use ion_dynamics_oct_m
  use, intrinsic :: iso_fortran_env
  use mesh_function_oct_m
  use messages_oct_m
  use namespace_oct_m
  use output_oct_m
  use output_low_oct_m
  use parser_oct_m
  use profiling_oct_m
  use target_low_oct_m
  use td_oct_m
  use space_oct_m
  use states_elec_oct_m
  use string_oct_m

  implicit none

  private
  public ::                    &
    target_init_userdefined,   &
    target_j1_userdefined,     &
    target_output_userdefined, &
    target_chi_userdefined


contains

  ! ----------------------------------------------------------------------
  !>
  subroutine target_init_userdefined(gr, namespace, tg, td)
    type(grid_t),      intent(in)    :: gr
    type(namespace_t), intent(in)    :: namespace
    type(target_t),    intent(inout) :: tg
    type(td_t),        intent(in)    :: td

    integer             :: no_states, ib, ip, idim, inst, inik, id, ist, ik
    type(block_t)       :: blk
    real(real64)        :: xx(1:gr%box%dim), rr, psi_re, psi_im
    complex(real64), allocatable  :: zpsi(:, :)

    PUSH_SUB(target_init_userdefined)

    message(1) =  'Info: Target is a user-defined state.'
    call messages_info(1, namespace=namespace)

    tg%move_ions = td%ions_dyn%ions_move()
    tg%dt = td%dt

    SAFE_ALLOCATE(zpsi(gr%np, 1:tg%st%d%dim))

    !%Variable OCTTargetUserdefined
    !%Type block
    !%Section Calculation Modes::Optimal Control
    !%Description
    !% Define a target state. Syntax follows the one of the <tt>UserDefinedStates</tt> block.
    !% Example:
    !%
    !% <tt>%OCTTargetUserdefined
    !% <br>&nbsp;&nbsp; 1 | 1 | 1 |  "exp(-r^2)*exp(-i*0.2*x)"
    !% <br>%</tt>
    !%
    !%End
    if (parse_block(namespace, 'OCTTargetUserdefined', blk) == 0) then

      no_states = parse_block_n(blk)
      do ib = 1, no_states
        call parse_block_integer(blk, ib - 1, 0, idim)
        call parse_block_integer(blk, ib - 1, 1, inst)
        call parse_block_integer(blk, ib - 1, 2, inik)

        ! read formula strings and convert to C strings
        do id = 1, tg%st%d%dim
          do ist = 1, tg%st%nst
            do ik = 1, tg%st%nik

              ! does the block entry match and is this node responsible?
              if (.not. (id == idim .and. ist == inst .and. ik == inik    &
                .and. tg%st%st_start <=  ist .and. tg%st%st_end >= ist)) cycle

              ! parse formula string
              call parse_block_string(                            &
                blk, ib - 1, 3, tg%st%user_def_states(id, ist, ik))
              ! convert to C string
              call conv_to_C_string(tg%st%user_def_states(id, ist, ik))

              do ip = 1, gr%np
                xx = gr%x(ip, :)
                rr = norm2(xx)

                ! parse user-defined expressions
                call parse_expression(psi_re, psi_im, &
                  gr%box%dim, xx, rr, M_ZERO, tg%st%user_def_states(id, ist, ik))
                ! fill state
                zpsi(ip, id) = cmplx(psi_re, psi_im, real64)
              end do

              ! normalize orbital
              call zmf_normalize(gr, tg%st%d%dim, zpsi)

              call states_elec_set_state(tg%st, gr, ist, ik, zpsi)

            end do
          end do
        end do
      end do
      call parse_block_end(blk)
      call density_calc(tg%st, gr, tg%st%rho)
    else
      call messages_variable_is_block(namespace, 'OCTTargetUserdefined')
    end if

    SAFE_DEALLOCATE_A(zpsi)

    POP_SUB(target_init_userdefined)
  end subroutine target_init_userdefined


  ! ----------------------------------------------------------------------
  subroutine target_output_userdefined(tg, namespace, space, gr, dir, ions, hm, outp)
    type(target_t),      intent(in) :: tg
    type(namespace_t),   intent(in) :: namespace
    class(space_t),      intent(in) :: space
    type(grid_t),        intent(in) :: gr
    character(len=*),    intent(in) :: dir
    type(ions_t),        intent(in) :: ions
    type(hamiltonian_elec_t), intent(in) :: hm
    type(output_t),      intent(in) :: outp
    PUSH_SUB(target_output_userdefined)

    call io_mkdir(trim(dir), namespace)
    call output_states(outp, namespace, space, trim(dir), tg%st, gr, ions, hm, -1)

    POP_SUB(target_output_userdefined)
  end subroutine target_output_userdefined
  ! ----------------------------------------------------------------------


  ! ----------------------------------------------------------------------
  !>
  real(real64) function target_j1_userdefined(tg, gr, psi) result(j1)
    type(target_t),      intent(in) :: tg
    type(grid_t),        intent(in) :: gr
    type(states_elec_t), intent(in) :: psi

    integer :: ik, ist
    complex(real64), allocatable :: zpsi(:, :), zst(:, :)

    PUSH_SUB(target_j1_userdefined)

    SAFE_ALLOCATE(zpsi(1:gr%np, 1:tg%st%d%dim))
    SAFE_ALLOCATE(zst(1:gr%np, 1:tg%st%d%dim))

    j1 = M_ZERO
    do ik = 1, psi%nik
      do ist = psi%st_start, psi%st_end

        call states_elec_get_state(psi, gr, ist, ik, zpsi)
        call states_elec_get_state(tg%st, gr, ist, ik, zst)

        j1 = j1 + psi%occ(ist, ik)*abs(zmf_dotp(gr, psi%d%dim, zpsi, zst))**2
      end do
    end do

    SAFE_DEALLOCATE_A(zpsi)
    SAFE_DEALLOCATE_A(zst)

    POP_SUB(target_j1_userdefined)
  end function target_j1_userdefined


  ! ----------------------------------------------------------------------
  !>
  subroutine target_chi_userdefined(tg, gr, psi_in, chi_out)
    type(target_t),      intent(in)    :: tg
    type(grid_t),        intent(in)    :: gr
    type(states_elec_t), intent(in)    :: psi_in
    type(states_elec_t), intent(inout) :: chi_out

    integer :: ik, ist
    complex(real64) :: olap
    complex(real64), allocatable :: zpsi(:, :), zst(:, :), zchi(:, :)

    PUSH_SUB(target_chi_userdefined)

    SAFE_ALLOCATE(zpsi(1:gr%np, 1:tg%st%d%dim))
    SAFE_ALLOCATE(zst(1:gr%np, 1:tg%st%d%dim))
    SAFE_ALLOCATE(zchi(1:gr%np, 1:tg%st%d%dim))

    do ik = 1, psi_in%nik
      do ist = psi_in%st_start, psi_in%st_end

        call states_elec_get_state(psi_in, gr, ist, ik, zpsi)
        call states_elec_get_state(tg%st, gr, ist, ik, zst)

        olap = zmf_dotp(gr, zst(:, 1), zpsi(:, 1))
        zchi(1:gr%np, 1:tg%st%d%dim) = olap*zst(1:gr%np, 1:tg%st%d%dim)

        call states_elec_set_state(chi_out, gr, ist, ik, zchi)

      end do
    end do

    SAFE_DEALLOCATE_A(zpsi)
    SAFE_DEALLOCATE_A(zst)
    SAFE_DEALLOCATE_A(zchi)

    POP_SUB(target_chi_userdefined)
  end subroutine target_chi_userdefined

end module target_userdefined_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
