!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

! ---------------------------------------------------------
subroutine output_states(outp, namespace, space, dir, st, gr, ions, hm, iter)
  type(output_t),           intent(in) :: outp
  type(namespace_t),        intent(in) :: namespace
  class(space_t),           intent(in) :: space
  character(len=*),         intent(in) :: dir
  type(states_elec_t),      intent(in) :: st
  type(grid_t),             intent(in) :: gr
  type(ions_t),             intent(in) :: ions
  type(hamiltonian_elec_t), intent(in) :: hm
  integer,                  intent(in) :: iter

  integer :: ik, ist, idim, idir, is, ierr, ip
  character(len=MAX_PATH_LEN) :: fname
  type(unit_t) :: fn_unit
  real(real64), allocatable :: dtmp(:), elf(:,:), polarization(:, :)
  complex(real64), allocatable :: ztmp(:)
  type(dos_t) :: dos

  PUSH_SUB(output_states)

  if (outp%what_now(OPTION__OUTPUT__DENSITY, iter)) then
    fn_unit = units_out%length**(-space%dim)
    do is = 1, st%d%nspin
      fname = get_filename_with_spin('density', st%d%nspin, is)
      call dio_function_output(outp%how(OPTION__OUTPUT__DENSITY), dir, fname, namespace, space, gr, &
        st%rho(:, is), fn_unit, ierr, pos=ions%pos, atoms=ions%atom, grp = st%dom_st_kpt_mpi_grp)
    end do

    if(allocated(st%frozen_rho)) then
      do is = 1, st%d%nspin
        fname = get_filename_with_spin('frozen_density', st%d%nspin, is)
        call dio_function_output(outp%how(OPTION__OUTPUT__DENSITY), dir, fname, namespace, space, gr, &
          st%frozen_rho(:, is), fn_unit, ierr, pos=ions%pos, atoms=ions%atom, grp = st%dom_st_kpt_mpi_grp)
      end do

    end if

  end if

  if (outp%what_now(OPTION__OUTPUT__POL_DENSITY, iter)) then
    fn_unit = units_out%length**(1 - space%dim)
    SAFE_ALLOCATE(polarization(1:gr%np, 1:space%dim))

    do is = 1, st%d%nspin
      do idir = 1, space%dim
        do ip = 1, gr%np
          polarization(ip, idir) = st%rho(ip, is)*gr%x(ip, idir)
        end do
      end do

      fname = get_filename_with_spin('dipole_density', st%d%nspin, is)
      call io_function_output_vector(outp%how(OPTION__OUTPUT__POL_DENSITY),&
        dir, fname, namespace, space, gr, polarization, fn_unit, ierr, &
        pos=ions%pos, atoms=ions%atom, grp = st%dom_st_kpt_mpi_grp)
    end do

    SAFE_DEALLOCATE_A(polarization)
  end if

  if (outp%what_now(OPTION__OUTPUT__WFS, iter)) then
    fn_unit = sqrt(units_out%length**(-space%dim))

    if (states_are_real(st)) then
      SAFE_ALLOCATE(dtmp(1:gr%np))
    else
      SAFE_ALLOCATE(ztmp(1:gr%np))
    end if

    do ist = st%st_start, st%st_end
      if (loct_isinstringlist(ist, outp%wfs_list)) then
        do ik = st%d%kpt%start, st%d%kpt%end
          do idim = 1, st%d%dim
            if (st%nik > 1) then
              if (st%d%dim > 1) then
                write(fname, '(a,i6.6,a,i5.5,a,i1)') 'wf-k', ik, '-st', ist, '-sp', idim
              else
                write(fname, '(a,i6.6,a,i5.5)')      'wf-k', ik, '-st', ist
              end if
            else
              if (st%d%dim > 1) then
                write(fname, '(a,i5.5,a,i1)')        'wf-st', ist, '-sp', idim
              else
                write(fname, '(a,i5.5)')             'wf-st', ist
              end if
            end if

            if (states_are_real(st)) then
              call states_elec_get_state(st, gr, idim, ist, ik, dtmp)
              call dio_function_output(outp%how(OPTION__OUTPUT__WFS), dir, fname, namespace, space, gr, dtmp, &
                fn_unit, ierr, pos=ions%pos, atoms=ions%atom)
            else
              call states_elec_get_state(st, gr, idim, ist, ik, ztmp)
              call zio_function_output(outp%how(OPTION__OUTPUT__WFS), dir, fname, namespace, space, gr, ztmp, &
                fn_unit, ierr, pos=ions%pos, atoms=ions%atom)
            end if
          end do
        end do
      end if
    end do

    SAFE_DEALLOCATE_A(dtmp)
    SAFE_DEALLOCATE_A(ztmp)
  end if

  if (outp%what_now(OPTION__OUTPUT__WFS_SQMOD, iter)) then
    fn_unit = units_out%length**(-space%dim)
    SAFE_ALLOCATE(dtmp(1:gr%np_part))
    if (states_are_complex(st)) then
      SAFE_ALLOCATE(ztmp(1:gr%np))
    end if
    do ist = st%st_start, st%st_end
      if (loct_isinstringlist(ist, outp%wfs_list)) then
        do ik = st%d%kpt%start, st%d%kpt%end
          do idim = 1, st%d%dim
            if (st%nik > 1) then
              if (st%d%dim > 1) then
                write(fname, '(a,i6.6,a,i5.5,a,i1)') 'sqm-wf-k', ik, '-st', ist, '-sp', idim
              else
                write(fname, '(a,i6.6,a,i5.5)')      'sqm-wf-k', ik, '-st', ist
              end if
            else
              if (st%d%dim > 1) then
                write(fname, '(a,i5.5,a,i1)')        'sqm-wf-st', ist, '-sp', idim
              else
                write(fname, '(a,i5.5)')             'sqm-wf-st', ist
              end if
            end if

            if (states_are_real(st)) then
              call states_elec_get_state(st, gr, idim, ist, ik, dtmp)
              dtmp(1:gr%np) = abs(dtmp(1:gr%np))**2
            else
              call states_elec_get_state(st, gr, idim, ist, ik, ztmp)
              dtmp(1:gr%np) = abs(ztmp(1:gr%np))**2
            end if
            call dio_function_output(outp%how(OPTION__OUTPUT__WFS_SQMOD), dir, fname, namespace, space, gr, &
              dtmp, fn_unit, ierr, pos=ions%pos, atoms=ions%atom)
          end do
        end do
      end if
    end do
    SAFE_DEALLOCATE_A(dtmp)
    SAFE_DEALLOCATE_A(ztmp)
  end if

  if (outp%what_now(OPTION__OUTPUT__KINETIC_ENERGY_DENSITY, iter)) then
    fn_unit = units_out%energy * units_out%length**(-space%dim)
    SAFE_ALLOCATE(elf(1:gr%np, 1:st%d%nspin))
    call states_elec_calc_quantities(gr, st, hm%kpoints, .false., kinetic_energy_density = elf)
    do is = 1, st%d%nspin
      fname = get_filename_with_spin('tau', st%d%nspin, is)
      call dio_function_output(outp%how(OPTION__OUTPUT__KINETIC_ENERGY_DENSITY), dir, trim(fname), namespace, space, &
        gr, elf(:,is), fn_unit, ierr, pos=ions%pos, atoms=ions%atom, grp = st%dom_st_kpt_mpi_grp)
    end do
    SAFE_DEALLOCATE_A(elf)
  end if

  if (outp%what_now(OPTION__OUTPUT__DOS, iter) .or. &
    outp%what_now(OPTION__OUTPUT__JDOS, iter)  .or. &
    outp%what_now(OPTION__OUTPUT__LDOS, iter)) then
    call dos_init(dos, namespace, st, hm%kpoints)
  end if

  if (outp%what_now(OPTION__OUTPUT__DOS, iter)) then
    call dos_write_dos(dos, trim(dir), st, gr%box, ions, gr, hm, namespace)
  end if

  if (outp%what_now(OPTION__OUTPUT__JDOS, iter)) then
    call dos_write_jdos(dos, trim(dir), st, gr%box, ions, gr, hm, namespace)
  end if

  if (outp%what_now(OPTION__OUTPUT__LDOS, iter)) then
    if (hm%kpoints%use_symmetries) then
      call messages_not_implemented("LDOS with k-point symmetries")
    end if
    call dos_write_ldos(dos, trim(dir), st, gr%box, ions, gr, outp%how(OPTION__OUTPUT__LDOS), namespace)
  end if

  if (outp%what_now(OPTION__OUTPUT__TPA, iter)) then
    call states_elec_write_tpa(trim(dir), namespace, space, gr, st)
  end if

  POP_SUB(output_states)

end subroutine output_states


! ---------------------------------------------------------
subroutine output_current_flow(outp, namespace, space, dir, gr, st, kpoints)
  type(output_t),       intent(in) :: outp
  type(namespace_t),    intent(in) :: namespace
  class(space_t),       intent(in) :: space
  character(len=*),     intent(in) :: dir
  type(grid_t),         intent(in) :: gr
  type(states_elec_t),  intent(in) :: st
  type(kpoints_t),      intent(in) :: kpoints

  integer :: iunit, ip, idir, rankmin
  real(real64)   :: flow, dmin
  real(real64), allocatable :: j(:, :, :)

  PUSH_SUB(output_current_flow)

  if (mpi_grp_is_root(mpi_world)) then

    call io_mkdir(dir, namespace)
    iunit = io_open(trim(dir)//'/'//'current-flow', namespace, action='write')

    select case (space%dim)
    case (3)
      write(iunit,'(a)')        '# Plane:'
      write(iunit,'(3a,3f9.5)') '# origin [', trim(units_abbrev(units_out%length)), '] = ', &
        (units_from_atomic(units_out%length, outp%plane%origin(idir)), idir = 1, 3)
      write(iunit,'(a,3f9.5)')  '# u = ', outp%plane%u(1), outp%plane%u(2), outp%plane%u(3)
      write(iunit,'(a,3f9.5)')  '# v = ', outp%plane%v(1), outp%plane%v(2), outp%plane%v(3)
      write(iunit,'(a,3f9.5)')  '# n = ', outp%plane%n(1), outp%plane%n(2), outp%plane%n(3)
      write(iunit,'(a, f9.5)')  '# spacing = ', units_from_atomic(units_out%length, outp%plane%spacing)
      write(iunit,'(a,2i4)')    '# nu, mu = ', outp%plane%nu, outp%plane%mu
      write(iunit,'(a,2i4)')    '# nv, mv = ', outp%plane%nv, outp%plane%mv

    case (2)
      write(iunit,'(a)')        '# Line:'
      write(iunit,'(3a,2f9.5)') '# origin [',  trim(units_abbrev(units_out%length)), '] = ', &
        (units_from_atomic(units_out%length, outp%line%origin(idir)), idir = 1, 2)
      write(iunit,'(a,2f9.5)')  '# u = ', outp%line%u(1), outp%line%u(2)
      write(iunit,'(a,2f9.5)')  '# n = ', outp%line%n(1), outp%line%n(2)
      write(iunit,'(a, f9.5)')  '# spacing = ', units_from_atomic(units_out%length, outp%line%spacing)
      write(iunit,'(a,2i4)')    '# nu, mu = ', outp%line%nu, outp%line%mu

    case (1)
      write(iunit,'(a)')        '# Point:'
      write(iunit,'(3a, f9.5)') '# origin [',  trim(units_abbrev(units_out%length)), '] = ', &
        units_from_atomic(units_out%length, outp%line%origin(1))

    end select
  end if

  if (states_are_complex(st)) then
    SAFE_ALLOCATE(j(1:gr%np, 1:space%dim, 1:st%d%nspin))
    call states_elec_calc_quantities(gr, st, kpoints, .false., paramagnetic_current = j)

    do idir = 1, space%dim
      do ip = 1, gr%np
        j(ip, idir, 1) = sum(j(ip, idir, 1:st%d%nspin))
      end do
    end do

    select case (space%dim)
    case (3)
      flow = mf_surface_integral(gr, j(:, :, 1), outp%plane)
    case (2)
      flow = mf_line_integral(gr, j(:, :, 1), outp%line)
    case (1)
      flow = j(mesh_nearest_point(gr, outp%line%origin(1:1), dmin, rankmin), 1, 1)
    end select

    SAFE_DEALLOCATE_A(j)
  else
    flow = M_ZERO
  end if

  if (mpi_grp_is_root(mpi_world)) then
    write(iunit,'(3a,e20.12)') '# Flow [', trim(units_abbrev(unit_one/units_out%time)), '] = ', &
      units_from_atomic(unit_one/units_out%time, flow)
    call io_close(iunit)
  end if

  POP_SUB(output_current_flow)
end subroutine output_current_flow

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
