!! Copyright (C) 2023  L. Konecny,  M. Lueders
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

!> @brief apply the ZORA to a batch of states psib
!!
!! @note This routine assumes
!!  - that hpsib already contains \f$ (\boldsymbol{p}^2 \Phi^\mathrm{ZORA}) \f$
!!  - that zora_oct_m::zora_update() had been called before.
!!
!! The routine makes use of the relation:
!! \f[ \boldsymbol{p} \frac{c^2}{2c^2 - V} \boldsymbol{p} \Phi^\mathrm{ZORA}
!!  = (\boldsymbol{p} \frac{c^2}{2c^2 - V})(\boldsymbol{p} \Phi^\mathrm{ZORA})
!!    + \frac{c^2}{2c^2 - V} (\boldsymbol{p}^2 \Phi^\mathrm{ZORA}) \f]
!
subroutine X(zora_apply_batch)(this, mesh, der, states_dim, psib, hpsib, set_bc)
  class(zora_t),               intent(in)    :: this        !< The ZORA object
  class(mesh_t),               intent(in)    :: mesh        !< mesh needs to be passed through
  type(derivatives_t),         intent(in)    :: der         !< the derivatives
  type(states_elec_dim_t),     intent(in)    :: states_dim  !< for accessing the spin index from the batch
  type(wfs_elec_t),    target, intent(inout) :: psib        !< input states
  type(wfs_elec_t),    target, intent(inout) :: hpsib       !< output states with ZORA applied
  logical,           optional, intent(in)    :: set_bc      !< flag whether to set the boundary conditions

  type(wfs_elec_t)            :: vgradb, gradb(mesh%box%dim)
  integer                     :: idir, ispin

  PUSH_SUB(X(zora_apply_batch))

  if ( this%zora_level == ZORA_NONE ) then
    POP_SUB(X(zora_apply_batch))
    return
  end if

  call psib%copy_to(vgradb)
  call batch_set_zero(vgradb)

  ispin = states_dim%get_spin_index(psib%ik)


  ! \frac{c^2}{2c^2 - V}  (\boldsymbol{p}^2 \Phi^\mathrm{ZORA})
  ! |---- this%pot ----| |------------- hpsib ---------------|
  call X(hamiltonian_elec_base_local_sub)(this%pot(:,:), mesh, states_dim, ispin, hpsib, vgradb)

  call X(derivatives_batch_grad)(der, psib, gradb, set_bc = set_bc, factor = -M_HALF/this%mass)

  ! (\boldsymbol{p} \frac{c^2}{2c^2 - V}) (\boldsymbol{p} \Phi^\mathrm{ZORA}) +
  ! |----------- this%grad_pot ---------| |------------- gradb ---------------|
  do idir = 1, mesh%box%dim
    call X(hamiltonian_elec_base_local_sub)(this%grad_pot(:,:,idir), mesh, states_dim, ispin, gradb(idir), vgradb)
  enddo

  ! SO_term = prefactor * sigma \cdot ( gscalar_pot \times p )
  !         = prefactor * ( sigma \times gscalar_pot ) \cdot p
  !           !------------- zora_soc ---------------| \cdot | gradb |
  if ( this%zora_level == ZORA_FULLY_RELATIVISTIC ) then

    do idir = 1, mesh%box%dim
      ! multiply by i to get p operator (-i grad), minus sign already applied above
      call batch_scal(mesh%np, M_zI, gradb(idir))
      call X(hamiltonian_elec_base_local_sub)(this%soc(:,:,idir), mesh, states_dim, ispin, gradb(idir), vgradb)
    enddo

  end if

  call vgradb%copy_data_to(mesh%np, hpsib)

  do idir = 1, mesh%box%dim
    call gradb(idir)%end()
  end do
  call vgradb%end()

  POP_SUB(X(zora_apply_batch))
end subroutine X(zora_apply_batch)
