!! Copyright (C) 2020 Heiko Appel
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module lorentz_force_oct_m
  use debug_oct_m
  use force_interaction_oct_m
  use global_oct_m
  use interaction_oct_m
  use interaction_partner_oct_m
  use messages_oct_m
  use namespace_oct_m
  use profiling_oct_m
  use quantity_oct_m

  implicit none

  private
  public ::                &
    lorentz_force_t

  !> Lorenz force between a systems of particles and an electromagnetic field.
  type, extends(force_interaction_t) :: lorentz_force_t
    private
    real(real64), pointer :: system_charge(:) => NULL() !< pointer to array storing the charges of the particles
    real(real64), pointer, public :: system_pos(:,:) => NULL() !< pointer to array storing the positions of the particles
    real(real64), pointer :: system_vel(:,:) => NULL() !< pointer to array storing the velocities of the particles

    real(real64), allocatable, public :: partner_e_field(:,:) !< E field generated by partner at the positions of the system particles
    real(real64), allocatable, public :: partner_b_field(:,:) !< B field generated by partner at the positions of the system particles

  contains
    procedure :: init => lorentz_force_init
    procedure :: calculate => lorentz_force_calculate
    procedure :: calculate_energy => lorentz_force_calculate_energy
    final :: lorentz_force_finalize
  end type lorentz_force_t

  interface lorentz_force_t
    module procedure lorentz_force_constructor
  end interface lorentz_force_t

contains

  ! ---------------------------------------------------------
  function lorentz_force_constructor(partner) result(this)
    class(interaction_partner_t), target, intent(inout) :: partner
    class(lorentz_force_t),               pointer       :: this

    PUSH_SUB(lorentz_force_constructor)

    allocate(this)

    this%label = "lorentz_force"

    this%partner => partner

    ! The Lorentz force needs the position, velocity and charge of the system
    this%system_quantities = [character(8) :: "position", "velocity", "charge"]

    ! The Lorenz force needs the E and B field of the interaction partner at the particle position
    this%couplings_from_partner = ["E field", "B field"]
    this%intra_interaction = .false. ! This interaction does not support intra-interactions


    POP_SUB(lorentz_force_constructor)
  end function lorentz_force_constructor

  ! ---------------------------------------------------------
  subroutine lorentz_force_init(this, dim, system_np, system_charge, system_pos, system_vel, namespace)
    class(lorentz_force_t),               intent(inout) :: this
    integer,                              intent(in)    :: dim !< number of dimensions in space
    integer,                              intent(in)    :: system_np  !< number of particles in the system that owns this interaction
    real(real64),                 target, intent(in)    :: system_charge(:)
    real(real64),                 target, intent(in)    :: system_pos(:,:)
    real(real64),                 target, intent(in)    :: system_vel(:,:)
    type(namespace_t),                    intent(in)    :: namespace

    PUSH_SUB(lorentz_force_init)

    message(1) = "Energies for Lorentz forces are not yet implemented, and currently set to 0."
    call messages_warning(1, namespace=namespace)

    this%dim = dim
    this%system_np = system_np
    this%energy = M_ZERO
    SAFE_ALLOCATE(this%force(dim, system_np))
    this%force = M_ZERO

    SAFE_ALLOCATE(this%partner_e_field(1:dim, 1:system_np))
    this%partner_e_field = M_ZERO
    SAFE_ALLOCATE(this%partner_b_field(1:dim, 1:system_np))
    this%partner_b_field = M_ZERO

    this%system_charge => system_charge
    this%system_pos => system_pos
    this%system_vel => system_vel

    POP_SUB(lorentz_force_init)
  end subroutine lorentz_force_init

  ! ---------------------------------------------------------
  subroutine lorentz_force_calculate(this)
    class(lorentz_force_t),             intent(inout) :: this

    integer :: ip

    PUSH_SUB(lorentz_force_calculate)

    ! Curl is defined only in 3D
    ASSERT(this%dim == 3)
    ASSERT(.not. this%intra_interaction)

    do ip = 1, this%system_np
      this%force(1, ip) = this%partner_e_field(1, ip) + &
        this%system_vel(2, ip)*this%partner_b_field(3, ip) - this%system_vel(3, ip)*this%partner_b_field(2, ip)

      this%force(2, ip) = this%partner_e_field(2, ip) + &
        this%system_vel(3, ip)*this%partner_b_field(1, ip) - this%system_vel(1, ip)*this%partner_b_field(3, ip)

      this%force(3, ip) = this%partner_e_field(3, ip) + &
        this%system_vel(1, ip)*this%partner_b_field(2, ip) - this%system_vel(2, ip)*this%partner_b_field(1, ip)

      this%force(1:3, ip) = this%force(1:3, ip)*this%system_charge(ip)
    end do

    POP_SUB(lorentz_force_calculate)
  end subroutine lorentz_force_calculate

  ! ---------------------------------------------------------
  subroutine lorentz_force_calculate_energy(this)
    class(lorentz_force_t),             intent(inout) :: this

    integer :: ip
    real(real64)   :: power

    PUSH_SUB(lorentz_force_calculate_energy)

    ! the rate at which the energy is transferred from the EM field to the particle
    ! the B field does not contribute any energy to the particles
    power = M_ZERO
    do ip = 1, this%system_np
      power = power - dot_product(this%system_vel(1:3,ip), this%force(1:3,ip))
    end do

    this%energy = M_ZERO
    !TODO: We need to implement the proper time integration of the power.
    !      However, to that end, we first need to implement another clock at which the energies are
    !      calculated.
    ! this%energy = this%energy + power*this%partner%iteration%get_time_step()

    POP_SUB(lorentz_force_calculate_energy)
  end subroutine lorentz_force_calculate_energy


  ! ---------------------------------------------------------
  subroutine lorentz_force_finalize(this)
    type(lorentz_force_t), intent(inout) :: this

    PUSH_SUB(lorentz_force_finalize)

    this%force = M_ZERO
    nullify(this%system_charge)
    nullify(this%system_pos)
    nullify(this%system_vel)
    SAFE_DEALLOCATE_A(this%force)
    SAFE_DEALLOCATE_A(this%partner_e_field)
    SAFE_DEALLOCATE_A(this%partner_b_field)

    call interaction_end(this)

    POP_SUB(lorentz_force_finalize)
  end subroutine lorentz_force_finalize

end module lorentz_force_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
