!! Copyright (C) 2009 X. Andrade
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

! -----------------------------------------------------------------------
!> This module contains interfaces for LAPACK routines
! -----------------------------------------------------------------------

module lapack_oct_m
  implicit none

  public ! only interfaces in this module

  !> computes the Cholesky factorization of a real symmetric
  !!  positive definite matrix A.
  !!
  !!  The factorization has the form
  !! \f[
  !!     A = U^T  U,  \mbox{ if UPLO} = 'U',
  !! \f]
  !! or
  !! \f[
  !!     A = L   L^T, \mbox{ if UPLO} = 'L',
  !! \f]
  !!  where U is an upper triangular matrix and L is lower triangular.
  !!
  !!  This is the block version of the algorithm, calling Level 3 BLAS.
  interface lapack_potrf
    subroutine dpotrf(uplo, n, a, lda, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: uplo
      integer,      intent(in)    :: n
      real(real64),      intent(inout) :: a
      integer,      intent(in)    :: lda
      integer,      intent(out)   :: info
    end subroutine dpotrf

    subroutine zpotrf(uplo, n, a, lda, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: uplo
      integer,      intent(in)    :: n
      complex(real64),   intent(inout) :: a
      integer,      intent(in)    :: lda
      integer,      intent(out)   :: info
    end subroutine zpotrf
  end interface lapack_potrf

  !>  Computes all the eigenvalues, and optionally, the eigenvectors
  !!  of a real generalized symmetric-definite eigenproblem, of the form
  !!  \f$Ax=(\lambda)Bx,  ABx=(\lambda)x, \mbox{ or } BAx=(\lambda)x \f$.
  !!  Here A and B are assumed to be symmetric and B is also
  !!  positive definite.
  interface lapack_sygv
    subroutine dsygv(itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1),      intent(in)    :: jobz, uplo
      integer,           intent(in)    :: itype, n, lda, ldb, lwork
      real(real64),      intent(inout) :: a, b    !< a(lda,n), b(ldb,n)
      real(real64),      intent(out)   :: w, work !< w(n), work(lwork)
      integer,           intent(out)   :: info
    end subroutine dsygv
  end interface lapack_sygv

  !> Same as lapack_sygv but using the divide and conquer algorithm
  interface lapack_sygvd
    subroutine dsygvd(itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, iwork, liwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1),      intent(in)    :: jobz, uplo
      integer,           intent(in)    :: itype, n, lda, ldb, lwork, liwork
      real(real64),      intent(inout) :: a, b    !< a(lda,n), b(ldb,n)
      real(real64),      intent(out)   :: w, work !< w(n), work(lwork)
      integer,           intent(out)   :: iwork, info !< iwork(liwork)
    end subroutine dsygvd
  end interface lapack_sygvd

  !>  Computes all the eigenvalues, and optionally, the eigenvectors
  !!  of a complex generalized Hermitian-definite eigenproblem, of the form
  !!  \f$Ax=(\lambda)Bx,  ABx=(\lambda)x, \mbox{ or } BAx=(\lambda)x \f$.
  !!  Here A and B are assumed to be Hermitian and B is also
  !!  positive definite.
  interface lapack_hegv
    subroutine zhegv(itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1),      intent(in)    :: jobz, uplo
      integer,           intent(in)    :: n, itype, lda, ldb, lwork
      complex(real64),   intent(inout) :: a, b     !< a(lda,n), b(ldb,n)
      real(real64),      intent(out)   :: w, rwork !< w(n), rwork(max(1,3*n-2))
      complex(real64),   intent(out)   :: work     !< work(lwork)
      integer,           intent(out)   :: info
    end subroutine zhegv
  end interface lapack_hegv

  !> Same as lapack_hegv but using the divide and conquer algorithm
  interface lapack_hegvd
    subroutine zhegvd(itype, jobz, uplo, n, a, lda, b, ldb, w, work, lwork, rwork, lrwork, iwork, liwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1),      intent(in)    :: jobz, uplo
      integer,           intent(in)    :: n, itype, lda, ldb, lwork, lrwork, liwork
      complex(real64),   intent(inout) :: a, b     !< a(lda,n), b(ldb,n)
      real(real64),      intent(out)   :: w, rwork !< w(n), rwork(lrwork)
      complex(real64),   intent(out)   :: work     !< work(lwork)
      integer,           intent(out)   :: iwork, info !< iwork(liwork)
    end subroutine zhegvd
  end interface lapack_hegvd

  !>  Computes for an \f$ N \times N \f$ complex nonsymmetric matrix A, the
  !!  eigenvalues and, optionally, the left and/or right eigenvectors.
  !!
  !!  The right eigenvector v(j) of A satisfies
  !! \f[
  !!                   A v(j) = \lambda(j)  v(j)
  !! \f]
  !!  where \f$ \lambda(j) \f$ is its eigenvalue.
  !!  The left eigenvector u(j) of A satisfies
  !! \f[
  !!                u(j)^H A = \lambda(j) u(j)^H
  !! \f]
  !!  where \f$ u(j)^H \f$ denotes the conjugate transpose of u(j).
  !!
  !!  The computed eigenvectors are normalized to have Euclidean norm
  !!  equal to 1 and largest component real.
  interface
    subroutine dgeev(jobvl, jobvr, n, a, lda, wr, wi, vl, ldvl, vr, ldvr, work, lwork, rwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: jobvl, jobvr
      integer,      intent(in)    :: n, lda, ldvl, ldvr, lwork
      real(real64),      intent(inout) :: a !< a(lda,n)
      real(real64),      intent(out)   :: wr, wi, vl, vr !< wr(n), wi(n), vl(ldvl,n), vl(ldvr,n)
      real(real64),      intent(out)   :: rwork !< rwork(max(1,2n))
      real(real64),      intent(out)   :: work  !< work(lwork)
      integer,      intent(out)   :: info
    end subroutine dgeev

    subroutine zgeev(jobvl, jobvr, n, a, lda, w, vl, ldvl, vr, ldvr, work, lwork, rwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: jobvl, jobvr
      integer,      intent(in)    :: n, lda, ldvl, ldvr, lwork
      complex(real64),   intent(inout) :: a !< a(lda,n)
      complex(real64),   intent(out)   :: w, vl, vr !< w(n), vl(ldvl,n), vl(ldvr,n)
      real(real64),      intent(out)   :: rwork !< rwork(max(1,2n))
      complex(real64),   intent(out)   :: work  !< work(lwork)
      integer,      intent(out)   :: info
    end subroutine zgeev
  end interface

  interface lapack_gesvx
    subroutine dgesvx(fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, &
      c, b, ldb, x, ldx, rcond, ferr, berr, work, iwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: fact, trans
      integer,      intent(in)    :: n, nrhs, lda, ldaf, ldb, ldx
      real(real64),     intent(inout) :: a, af, r, c, b      !< a(lda,n), af(ldaf,n), r(n), c(n), b(ldb,nrhs)
      integer,      intent(inout) :: ipiv                !< ipiv(n)
      real(real64),     intent(out)   :: x, ferr, berr, work !< x(ldx,nrhs), ferr(nrhs), berr(nrhs), work(4*n)
      real(real64),     intent(out)   :: rcond
      character(1), intent(inout) :: equed
      integer,      intent(out)   :: iwork               !< iwork(n)
      integer,      intent(out)   :: info
    end subroutine dgesvx

    subroutine zgesvx (fact, trans, n, nrhs, a, lda, af, ldaf, ipiv, equed, r, &
      c, b, ldb, x, ldx, rcond, ferr, berr, work, rwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: fact, trans
      integer,      intent(in)    :: n, nrhs, lda, ldaf, ldb, ldx
      complex(real64),  intent(inout) :: a, af, b            !< a(lda, n), af(ldaf, n), b(ldb, nrhs)
      real(real64),     intent(inout) :: r, c                !< r(n), c(n)
      integer,      intent(inout) :: ipiv                !< ipiv(n)
      real(real64),     intent(out)   :: ferr, berr          !< ferr(nrhs), berr(nrhs)
      real(real64),     intent(out)   :: rcond
      complex(real64),  intent(out)   :: x, work             !< x(ldx, nrhs), work(2*n)
      character(1), intent(inout) :: equed
      real(real64),     intent(out)   :: rwork               !< rwork(2*n)
      integer,      intent(out)   :: info
    end subroutine zgesvx
  end interface lapack_gesvx

  !>  Computes all eigenvalues and, optionally, eigenvectors of a
  !!  real symmetric matrix A.
  interface lapack_syev
    subroutine dsyev(jobz, uplo, n, a, lda, w, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: jobz, uplo
      integer,      intent(in)    :: n, lda, lwork
      real(real64),      intent(inout) :: a       !< a(lda,n)
      real(real64),      intent(out)   :: w, work !< w(n), work(lwork)
      integer,      intent(out)   :: info
    end subroutine dsyev
  end interface lapack_syev

  !>  Computes all eigenvalues and, optionally, eigenvectors of a
  !!  complex Hermitian matrix A.
  interface lapack_heev
    subroutine zheev(jobz, uplo, n, a, lda, w, work, lwork, rwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: jobz, uplo
      integer,      intent(in)    :: n, lda, lwork
      complex(real64),   intent(inout) :: a        !< a(lda,n)
      real(real64),      intent(out)   :: w, rwork !< w(n), rwork(max(1,3*n-2))
      complex(real64),   intent(out)   :: work     !< work(lwork)
      integer,      intent(out)   :: info
    end subroutine zheev
  end interface lapack_heev

  interface
    subroutine dsyevx(jobz, range, uplo, n, a, lda, &
      vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, iwork, ifail, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer,      intent(in)    :: n, lda, il, iu, ldz, lwork
      character(1), intent(in)    :: jobz, range, uplo
      integer,      intent(out)   :: m, iwork, ifail, info
      real(real64),     intent(in)    :: vl, vu, abstol
      real(real64),     intent(inout) :: a
      real(real64),     intent(out)   :: w, z, work
    end subroutine dsyevx

    subroutine zheevx(jobz, range, uplo, n, a, lda, &
      vl, vu, il, iu, abstol, m, w, z, ldz, work, lwork, iwork, ifail, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer,      intent(in)    :: n, lda, il, iu, ldz, lwork
      character(1), intent(in)    :: jobz, range, uplo
      integer,      intent(out)   :: m, iwork, ifail, info
      real(real64),     intent(in)    :: vl, vu, abstol
      real(real64),     intent(out)   :: w
      complex(real64),  intent(inout) :: a
      complex(real64),  intent(out)   :: z, work
    end subroutine zheevx
  end interface

  !>  Computes a QR factorization of a real \f$m \times n\f$ matrix A:
  !! \f[
  !!  A = Q R.
  !! \f]
  interface lapack_geqrf
    subroutine dgeqrf(m, n, a, lda, tau, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer, intent(in)    :: lda, lwork, m, n
      real(real64), intent(inout) :: a
      real(real64), intent(out)   :: tau, work
      integer, intent(out)   :: info
    end subroutine dgeqrf

    subroutine zgeqrf(m, n, a, lda, tau, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer, intent(in)       :: lda, lwork, m, n
      complex(real64), intent(inout) :: a
      complex(real64), intent(out)   :: tau, work
      integer, intent(out)      :: info
    end subroutine zgeqrf
  end interface lapack_geqrf

  !>  Generates an \f$ M \times N \f$ real matrix Q with orthonormal columns,
  !!  which is defined as the first N columns of a product of K elementary
  !!  reflectors of order M
  !!
  !! \f[
  !!        Q  =  H(1) H(2) . . . H(k)
  !! \f]
  !!
  !!  as returned by DGEQRF.
  interface lapack_orgqr
    subroutine dorgqr(m, n, k, a, lda, tau, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer, intent(in)    :: k, lda, lwork, m, n
      real(real64), intent(in)    :: tau
      real(real64), intent(inout) :: a
      real(real64), intent(out)   :: work
      integer, intent(out)   :: info
    end subroutine dorgqr

    subroutine zungqr(m, n, k, a, lda, tau, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer,    intent(in)    :: k, lda, lwork, m, n
      complex(real64), intent(in)    :: tau
      complex(real64), intent(inout) :: a
      complex(real64), intent(out)   :: work
      integer,    intent(out)   :: info
    end subroutine zungqr
  end interface lapack_orgqr

  !>  Computes selected eigenvalues, and optionally, eigenvectors
  !!  of a real generalized symmetric-definite eigenproblem, of the form
  !!  \f$Ax=(\lambda)Bx,  ABx=(\lambda)x, \mbox{ or } BAx=(\lambda)x \f$.  Here A
  !!  and B are assumed to be symmetric and B is also positive definite.
  !!  Eigenvalues and eigenvectors can be selected by specifying either a
  !!  range of values or a range of indices for the desired eigenvalues.
  interface lapack_sygvx
    subroutine dsygvx(itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, &
      m, w, z, ldz, work, lwork, iwork, ifail, info)
      use, intrinsic :: iso_fortran_env
      implicit none

      integer,             intent(in)    :: itype !< Specifies the problem: 1: A*x = l*B*x 2:  A*B*x = l*x 3: B*A*x = l*x
      character(len=1),    intent(in)    :: jobz  !< N: Compute eigenvalues only; V: Compute eigenvalues and eigenvectors.
      character(len=1),    intent(in)    :: range !< A: all eigenval V: all eigenval in (VL,VU] I: IL-th through IU-th eigenval
      character(len=1),    intent(in)    :: uplo  !< U: Upper triangle of A and B stored L: Lower triangle of A and B stored
      integer,             intent(in)    :: n     !< The order of the matrix pencil (A,B)
      real(real64),             intent(inout) :: a     !< a(:) On entry, the symmetric matrix A. On exit, destroyed
      integer,             intent(in)    :: lda   !< The leading dimension of the array A
      real(real64),             intent(inout) :: b     !< b(:)
      integer,             intent(in)    :: ldb
      real(real64),             intent(in)    :: vl
      real(real64),             intent(in)    :: vu
      integer,             intent(in)    :: il
      integer,             intent(in)    :: iu
      real(real64),             intent(in)    :: abstol
      integer,             intent(out)   :: m
      real(real64),             intent(out)   :: w
      real(real64),             intent(out)   :: z
      integer,             intent(in)    :: ldz
      real(real64),             intent(out)   :: work   !< work(:)
      integer,             intent(in)    :: lwork
      integer,             intent(out)   :: iwork  !< iwork(1:5*n)
      integer,             intent(out)   :: ifail  !< ifail(1:n)
      integer,             intent(out)   :: info
    end subroutine dsygvx
  end interface lapack_sygvx

  !>  Computes selected eigenvalues, and optionally, eigenvectors
  !!  of a complex generalized Hermitian-definite eigenproblem, of the form
  !!  \f$Ax=(\lambda)Bx,  ABx=(\lambda)x, \mbox{ or } BAx=(\lambda)x \f$.  Here A and
  !!  B are assumed to be Hermitian and B is also positive definite.
  !!  Eigenvalues and eigenvectors can be selected by specifying either a
  !!  range of values or a range of indices for the desired eigenvalues.
  interface lapack_hegvx
    subroutine zhegvx(itype, jobz, range, uplo, n, a, lda, b, ldb, vl, vu, il, iu, abstol, &
      m, w, z, ldz, work, lwork, rwork, iwork, ifail, info)
      use, intrinsic :: iso_fortran_env
      implicit none

      integer,             intent(in)    :: itype
      character(len=1),    intent(in)    :: jobz
      character(len=1),    intent(in)    :: range
      character(len=1),    intent(in)    :: uplo
      integer,             intent(in)    :: n
      complex(real64),          intent(inout) :: a
      integer,             intent(in)    :: lda
      complex(real64),          intent(inout) :: b
      integer,             intent(in)    :: ldb
      real(real64),             intent(in)    :: vl
      real(real64),             intent(in)    :: vu
      integer,             intent(in)    :: il
      integer,             intent(in)    :: iu
      real(real64),             intent(in)    :: abstol
      integer,             intent(out)   :: m
      real(real64),             intent(out)   :: w
      complex(real64),          intent(out)   :: z
      integer,             intent(in)    :: ldz
      complex(real64),          intent(out)   :: work
      integer,             intent(in)    :: lwork
      real(real64),             intent(out)   :: rwork !< rwork(1:7*n)
      integer,             intent(out)   :: iwork !< iwork(1:5*n)
      integer,             intent(out)   :: ifail !< ifail(1:n)
      integer,             intent(out)   :: info
    end subroutine zhegvx
  end interface lapack_hegvx

  interface lapack_gelss
    subroutine dgelss(m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer, intent(in)    :: m
      integer, intent(in)    :: n
      integer, intent(in)    :: nrhs
      real(real64),intent(inout) :: a
      integer, intent(in)    :: lda
      real(real64),intent(inout) :: b
      integer, intent(in)    :: ldb
      real(real64),intent(out)   :: s
      real(real64),intent(in)    :: rcond
      integer, intent(out)   :: rank
      real(real64),intent(out)   :: work
      integer, intent(in)    :: lwork
      integer, intent(out)   :: info
    end subroutine dgelss

    subroutine zgelss(m, n, nrhs, a, lda, b, ldb, s, rcond, rank, work, lwork, rwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer, intent(in)    :: m
      integer, intent(in)    :: n
      integer, intent(in)    :: nrhs
      complex(real64),   intent(inout) :: a
      integer, intent(in)    :: lda
      complex(real64),   intent(inout) :: b
      integer, intent(in)    :: ldb
      real(real64),intent(out)   :: s
      real(real64),intent(in)    :: rcond
      integer, intent(out)   :: rank
      complex(real64),   intent(out)   :: work
      integer, intent(in)    :: lwork
      real(real64),intent(out)   :: rwork
      integer, intent(out)   :: info
    end subroutine zgelss
  end interface lapack_gelss

  interface lapack_getrf
    subroutine dgetrf (m, n, a, lda, ipiv, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer,      intent(in)    :: m, n, lda
      real(real64),     intent(inout) :: a         !< a(lda, n)
      integer,      intent(out)   :: ipiv       !< ipiv(min(m,n)
      integer,      intent(out)   :: info
    end subroutine dgetrf

    subroutine zgetrf (m, n, a, lda, ipiv, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer,      intent(in)    :: m, n, lda
      complex(real64),  intent(inout) :: a         !< a(lda, n)
      integer,      intent(out)   :: ipiv       !< ipiv(min(m,n)
      integer,      intent(out)   :: info
    end subroutine zgetrf
  end interface lapack_getrf

  interface lapack_getri
    subroutine dgetri(n, a, lda, ipiv, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer,      intent(in)    :: n, lda, lwork
      real(real64),     intent(inout) :: a       !< a(lda, n)
      integer,      intent(in)    :: ipiv    !< ipiv(n)
      real(real64),     intent(out)   :: work    !< work(lwork)
      integer,      intent(out)   :: info
    end subroutine dgetri

    subroutine zgetri(n, a, lda, ipiv, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      integer,      intent(in)    :: n, lda, lwork
      complex(real64),  intent(inout) :: a       !< a(lda, n)
      integer,      intent(in)    :: ipiv    !< ipiv(n)
      complex(real64),  intent(out)   :: work    !< work(lwork)
      integer,      intent(out)   :: info
    end subroutine zgetri
  end interface lapack_getri

  interface lapack_sytrf
    subroutine dsytrf(uplo, n, a, lda, ipiv, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: uplo
      integer,      intent(in)    :: n, lda, lwork
      real(real64),     intent(inout) :: a
      integer,      intent(out)   :: ipiv
      real(real64),     intent(out)   :: work
      integer,      intent(out)   :: info
    end subroutine dsytrf

    subroutine zsytrf(uplo, n, a, lda, ipiv, work, lwork, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: uplo
      integer,      intent(in)    :: n, lda, lwork
      complex(real64),  intent(inout) :: a
      integer,      intent(out)   :: ipiv
      complex(real64),  intent(out)   :: work
      integer,      intent(out)   :: info
    end subroutine zsytrf
  end interface lapack_sytrf

  interface lapack_sytri
    subroutine dsytri (uplo, n, a, lda, ipiv, work, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: uplo
      integer,      intent(in)    :: n, lda
      real(real64),     intent(inout) :: a
      integer,      intent(in)    :: ipiv
      real(real64),     intent(out)   :: work
      integer,      intent(out)   :: info
    end subroutine dsytri

    subroutine zsytri (uplo, n, a, lda, ipiv, work, info)
      use, intrinsic :: iso_fortran_env
      implicit none
      character(1), intent(in)    :: uplo
      integer,      intent(in)    :: n, lda
      complex(real64),  intent(inout) :: a
      integer,      intent(in)    :: ipiv
      complex(real64),  intent(out)   :: work
      integer,      intent(out)   :: info
    end subroutine zsytri
  end interface lapack_sytri


end module lapack_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
