!! Copyright (C) 2024 Sebastian Ohlmann
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module propagator_exp_gauss2_oct_m
  use algorithm_oct_m
  use debug_oct_m
  use global_oct_m
  use propagator_oct_m

  implicit none

  private
  public ::                            &
    propagator_exp_gauss2_t

  !> Implements the an exponential RK scheme with Gauss collocation points, s=2
  !> see also Hochbruck, M. & Ostermann, A.: Exponential Runge–Kutta methods for parabolic
  !> problems. Applied Numerical Mathematics 53, 323–339 (2005).
  type, extends(propagator_t) :: propagator_exp_gauss2_t
    private
  end type propagator_exp_gauss2_t

  interface propagator_exp_gauss2_t
    procedure propagator_exp_gauss2_constructor
  end interface propagator_exp_gauss2_t

  !# doc_start exp_gauss2_propagation_operations
  ! Specific exponential mid-point propagation operations identifiers
  character(len=ALGO_LABEL_LEN), public, parameter :: &
    EXP_GAUSS2_START        = 'EXP_GAUSS2_START',             &
    EXP_GAUSS2_FINISH       = 'EXP_GAUSS2_FINISH',            &
    EXP_GAUSS2_EXTRAPOLATE  = 'EXP_GAUSS2_EXTRAPOLATE',      &
    EXP_GAUSS2_PROPAGATE    = 'EXP_GAUSS2_PROPAGATE'

  ! Specific exponential mid-point propagation operations
  type(algorithmic_operation_t), public, parameter :: &
    OP_EXP_GAUSS2_START        = algorithmic_operation_t(EXP_GAUSS2_START,        'Starting exponential with Gauss order 1'),  &
    OP_EXP_GAUSS2_FINISH       = algorithmic_operation_t(EXP_GAUSS2_FINISH,       'Finishing exponential with Gauss order 1'), &
    OP_EXP_GAUSS2_EXTRAPOLATE  = algorithmic_operation_t(EXP_GAUSS2_EXTRAPOLATE,  'Extrapolation step'), &
    OP_EXP_GAUSS2_PROPAGATE    = algorithmic_operation_t(EXP_GAUSS2_PROPAGATE,    'Propagation step for exponential midpoint')
  !# doc_end

contains

  ! ---------------------------------------------------------
  function propagator_exp_gauss2_constructor(dt) result(this)
    real(real64),           intent(in) :: dt
    type(propagator_exp_gauss2_t), pointer   :: this

    PUSH_SUB(propagator_exp_gauss2_constructor)

    allocate(this)

    this%predictor_corrector = .false.
    this%start_operation = OP_EXP_GAUSS2_START
    this%final_operation = OP_EXP_GAUSS2_FINISH

    call this%add_operation(OP_EXP_GAUSS2_EXTRAPOLATE)
    call this%add_operation(OP_EXP_GAUSS2_PROPAGATE)
    call this%add_operation(OP_UPDATE_COUPLINGS)
    call this%add_operation(OP_UPDATE_INTERACTIONS)
    call this%add_operation(OP_ITERATION_DONE)
    call this%add_operation(OP_REWIND_ALGORITHM)

    this%algo_steps = 1
    this%dt = dt

    POP_SUB(propagator_exp_gauss2_constructor)
  end function propagator_exp_gauss2_constructor

end module propagator_exp_gauss2_oct_m


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
