!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module target_ground_state_oct_m
  use debug_oct_m
  use global_oct_m
  use grid_oct_m
  use hamiltonian_elec_oct_m
  use io_oct_m
  use ions_oct_m
  use ion_dynamics_oct_m
  use, intrinsic :: iso_fortran_env
  use kpoints_oct_m
  use mesh_oct_m
  use mesh_function_oct_m
  use messages_oct_m
  use namespace_oct_m
  use output_oct_m
  use output_low_oct_m
  use profiling_oct_m
  use restart_oct_m
  use space_oct_m
  use states_elec_oct_m
  use states_elec_restart_oct_m
  use target_low_oct_m
  use td_oct_m

  implicit none

  private
  public ::                    &
    target_init_groundstate,   &
    target_j1_groundstate,     &
    target_output_groundstate, &
    target_chi_groundstate


contains

  ! ----------------------------------------------------------------------
  !>
  subroutine target_init_groundstate(mesh, namespace, space, tg, td, restart, kpoints)
    class(mesh_t),     intent(in)    :: mesh
    type(namespace_t), intent(in)    :: namespace
    class(space_t),    intent(in)    :: space
    type(target_t),    intent(inout) :: tg
    type(td_t),        intent(in)    :: td
    type(restart_t),   intent(in)    :: restart
    type(kpoints_t),   intent(in)    :: kpoints

    integer :: ierr

    PUSH_SUB(target_init_groundstate)

    message(1) =  'Info: Using Ground State for TargetOperator'
    call messages_info(1, namespace=namespace)

    call states_elec_load(restart, namespace, space, tg%st, mesh, kpoints, ierr)
    if (ierr /= 0) then
      message(1) = "Unable to read wavefunctions."
      call messages_fatal(1, namespace=namespace)
    end if

    tg%move_ions = td%ions_dyn%ions_move()
    tg%dt = td%dt

    POP_SUB(target_init_groundstate)
  end subroutine target_init_groundstate


  ! ----------------------------------------------------------------------
  subroutine target_output_groundstate(tg, namespace, space, gr, dir, ions, hm, outp)
    type(target_t),      intent(in) :: tg
    type(namespace_t),   intent(in) :: namespace
    class(space_t),      intent(in)    :: space
    type(grid_t),        intent(in) :: gr
    character(len=*),    intent(in) :: dir
    type(ions_t),        intent(in) :: ions
    type(hamiltonian_elec_t), intent(in) :: hm
    type(output_t),      intent(in) :: outp

    PUSH_SUB(target_output_groundstate)

    call io_mkdir(trim(dir), namespace)
    call output_states(outp, namespace, space, trim(dir), tg%st, gr, ions, hm, -1)

    POP_SUB(target_output_groundstate)
  end subroutine target_output_groundstate
  ! ----------------------------------------------------------------------


  ! ----------------------------------------------------------------------
  !>
  real(real64) function target_j1_groundstate(tg, gr, psi) result(j1)
    type(target_t),      intent(in) :: tg
    type(grid_t),        intent(in) :: gr
    type(states_elec_t), intent(in) :: psi

    integer :: ist, ik
    complex(real64), allocatable :: zpsi(:, :), zst(:, :)

    PUSH_SUB(target_j1_groundstate)

    SAFE_ALLOCATE(zpsi(1:gr%np, 1:tg%st%d%dim))
    SAFE_ALLOCATE(zst(1:gr%np, 1:tg%st%d%dim))

    j1 = M_ZERO

    do ik = 1, psi%nik
      do ist = psi%st_start, psi%st_end
        call states_elec_get_state(psi, gr, ist, ik, zpsi)
        call states_elec_get_state(tg%st, gr, ist, ik, zst)

        j1 = j1 + psi%occ(ist, ik)*abs(zmf_dotp(gr, psi%d%dim, zpsi, zst))**2

      end do
    end do

    SAFE_DEALLOCATE_A(zpsi)
    SAFE_DEALLOCATE_A(zst)

    POP_SUB(target_j1_groundstate)
  end function target_j1_groundstate


  ! ----------------------------------------------------------------------
  !>
  subroutine target_chi_groundstate(tg, gr, psi_in, chi_out)
    type(target_t),       intent(in)    :: tg
    type(grid_t),         intent(in)    :: gr
    type(states_elec_t),  intent(in)    :: psi_in
    type(states_elec_t),  intent(inout) :: chi_out

    integer :: ik, ist
    complex(real64) :: olap
    complex(real64), allocatable :: zpsi(:, :), zst(:, :), zchi(:, :)

    PUSH_SUB(target_chi_groundstate)

    SAFE_ALLOCATE(zpsi(1:gr%np, 1:tg%st%d%dim))
    SAFE_ALLOCATE(zst(1:gr%np, 1:tg%st%d%dim))
    SAFE_ALLOCATE(zchi(1:gr%np, 1:tg%st%d%dim))

    do ik = 1, psi_in%nik
      do ist = psi_in%st_start, psi_in%st_end

        call states_elec_get_state(psi_in, gr, ist, ik, zpsi)
        call states_elec_get_state(tg%st, gr, ist, ik, zst)

        olap = zmf_dotp(gr, zst(:, 1), zpsi(:, 1))
        zchi(1:gr%np, 1:tg%st%d%dim) = olap*zst(1:gr%np, 1:tg%st%d%dim)

        call states_elec_set_state(chi_out, gr, ist, ik, zchi)

      end do
    end do

    SAFE_DEALLOCATE_A(zpsi)
    SAFE_DEALLOCATE_A(zst)
    SAFE_DEALLOCATE_A(zchi)

    POP_SUB(target_chi_groundstate)
  end subroutine target_chi_groundstate

end module target_ground_state_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
