!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

! Some pieces glued from PW/w(0,1)gauss.f90 from PWSCF

#include "global.h"

module smear_oct_m
  use debug_oct_m
  use global_oct_m
  use, intrinsic :: iso_fortran_env
  use kpoints_oct_m
  use loct_math_oct_m
  use messages_oct_m
  use namespace_oct_m
  use parser_oct_m
  use profiling_oct_m
  use sort_oct_m
  use unit_oct_m
  use unit_system_oct_m
  use varinfo_oct_m

  implicit none

  private
  public ::                           &
    smear_t,                          &
    smear_init,                       &
    smear_copy,                       &
    smear_find_fermi_energy,          &
    smear_fill_occupations,           &
    smear_calc_entropy,               &
    smear_delta_function,             &
    smear_step_function,              &
    smear_entropy_function,           &
    smear_is_semiconducting,          &
    smear_write_info

  type smear_t
    private
    integer, public :: method       !< which smearing function to take
    real(real64),   public :: dsmear       !< the parameter defining this function
    real(real64)    :: dsmear_cond  !< the smearing for photodoping electrons in conduction bands
    real(real64),   public :: e_fermi      !< the Fermi energy
    real(real64)    :: e_fermi_cond !< the Fermi energy for photodoping electrons in conduction bands

    integer, public :: el_per_state !< How many electrons can we put in each state (1 or 2)
    real(real64),   public :: ef_occ       !< Occupancy of the level at the Fermi energy
    real(real64)    :: ef_occ_cond  !< Occupancy of the level at the Fermi energy for photodoping electrons
    logical, public :: integral_occs !< for fixed_occ, are they all integers?
    integer         :: MP_n         !< order of Methfessel-Paxton smearing
    integer         :: fermi_count  !< The number of occupied states at the fermi level
    integer         :: nik_factor   !< denominator, for treating k-weights as integers
    integer         :: nspins       !< = 2 if spin_polarized, else 1.

    ! PhotonDoping
    logical, public:: photodop         !< photodoping: two fermi energies
    real(real64)   :: nephotodop       !< number of photodoping electrons put in the conduction bands
    integer        :: photodop_bandmin !< the starting index of the conduction bands for photodoping

  end type smear_t

  integer, parameter, public ::       &
    SMEAR_SEMICONDUCTOR     = 1,      &
    SMEAR_FERMI_DIRAC       = 2,      &
    SMEAR_COLD              = 3,      &
    SMEAR_METHFESSEL_PAXTON = 4,      &
    SMEAR_SPLINE            = 5,      &
    SMEAR_FIXED_OCC         = 6

  real(real64), parameter :: TOL_SMEAR = 1e-6_real64

contains

  !--------------------------------------------------
  subroutine smear_init(this, namespace, ispin, fixed_occ, integral_occs, kpoints)
    type(smear_t),             intent(out) :: this
    type(namespace_t),         intent(in)  :: namespace
    integer,                   intent(in)  :: ispin
    logical,                   intent(in)  :: fixed_occ
    logical,                   intent(in)  :: integral_occs
    type(kpoints_t),           intent(in)  :: kpoints

    PUSH_SUB(smear_init)

    this%integral_occs = integral_occs

    !%Variable SmearingFunction
    !%Type integer
    !%Default semiconducting
    !%Section States
    !%Description
    !% This is the function used to smear the electronic occupations.
    !% It is ignored if the <tt>Occupations</tt> block is set.
    !%Option semiconducting 1
    !% Semiconducting occupations, <i>i.e.</i> the lowest lying states are occupied
    !% until no more electrons are left.
    !%Option fermi_dirac 2
    !% Simple Fermi-Dirac distribution. In this case, <tt>Smearing</tt> has
    !% the meaning of an electronic temperature. DN Mermin, <i>Phys. Rev.</i> <b>137</b>, A1441 (1965).
    !%Option cold_smearing 3
    !% N Marzari, D Vanderbilt, A De Vita, and MC Payne, <i>Phys. Rev. Lett.</i> <b>82</b>, 3296 (1999).
    !%Option methfessel_paxton 4
    !% M Methfessel and AT Paxton, <i>Phys. Rev. B</i> <b>40</b>, 3616 (1989).
    !% In this case, the variable <tt>SmearingMPOrder</tt> sets the order of the smearing.
    !% Occupations may be negative.
    !%Option spline_smearing 5
    !% Nearly identical to Gaussian smearing.
    !% JM Holender, MJ Gillan, MC Payne, and AD Simpson, <i>Phys. Rev. B</i> <b>52</b>, 967 (1995).
    !%End
    if (fixed_occ) then
      this%method = SMEAR_FIXED_OCC
    else
      call parse_variable(namespace, 'SmearingFunction', SMEAR_SEMICONDUCTOR, this%method)
      if (.not. varinfo_valid_option('SmearingFunction', this%method)) call messages_input_error(namespace, 'SmearingFunction')
      call messages_print_var_option('SmearingFunction', this%method, namespace=namespace)
    end if

    !%Variable Smearing
    !%Type float
    !%Default 0.1 eV
    !%Section States
    !%Description
    !% If <tt>Occupations</tt> is not set, <tt>Smearing</tt> is the
    !% smearing width used in the <tt>SmearingFunction</tt> to distribute the electrons
    !% among the existing states.
    !%End
    this%dsmear = 1e-14_real64
    if (this%method /= SMEAR_SEMICONDUCTOR .and. this%method /= SMEAR_FIXED_OCC) then
      call parse_variable(namespace, 'Smearing', 0.1_real64 / (M_TWO * P_Ry), this%dsmear, units_inp%energy)
    end if

    !%Variable PhotoDopingSmearing
    !%Type float
    !%Default 0.1 eV
    !%Section States
    !%Description
    !% If <tt>Occupations</tt> is not set, <tt>PhotoDopingSmearing</tt> is the
    !% smearing width used in the <tt>SmearingFunction</tt> to distribute the electrons in conduction bands
    !% among the existing states.
    !%End
    this%dsmear_cond = this%dsmear
    if (this%method /= SMEAR_SEMICONDUCTOR .and. this%method /= SMEAR_FIXED_OCC) then
      call parse_variable(namespace, 'PhotoDopingSmearing', 0.1_real64 / (M_TWO * P_Ry), this%dsmear_cond, units_inp%energy)
    end if

    !%Variable PhotoDopingNumElectrons
    !%Type float
    !%Default 0
    !%Section States
    !%Description
    !% This variable allows to set the number of valence electrons taken out to put
    !% into the conduction bands. The method follows Marini and Calandra, PRB 104, 144103 (2021).
    !%End
    call parse_variable(namespace, 'PhotoDopingNumElectrons', M_ZERO, this%nephotodop)

    if (this%nephotodop > M_MIN_OCC) then
      this%photodop=.true.
    else
      this%photodop=.false.
    endif


    !%Variable PhotoDopingBand
    !%Type integer
    !%Default 1
    !%Section States
    !%Description
    !% This variable specifies where the minium (starting) conduction band index for the photodoping electrons.
    !%End
    call parse_variable(namespace, 'PhotoDopingBand', 1, this%photodop_bandmin)


    call messages_obsolete_variable(namespace, 'ElectronicTemperature', 'Smearing')

    if (ispin == 1) then ! unpolarized
      this%el_per_state = 2
    else
      this%el_per_state = 1
    end if

    if (ispin == 2) then
      this%nspins = 2
    else
      this%nspins = 1
    end if

    if (this%method == SMEAR_SEMICONDUCTOR) then
      this%nik_factor = kpoints_kweight_denominator(kpoints)

      if (this%nik_factor == 0) then
        message(1) = "k-point weights in KPoints or KPointsReduced blocks must be rational numbers for semiconducting smearing."
        call messages_fatal(1, namespace=namespace)
      end if
    end if

    this%MP_n = 0
    if (this%method == SMEAR_METHFESSEL_PAXTON) then
      !%Variable SmearingMPOrder
      !%Type integer
      !%Default 1
      !%Section States
      !%Description
      !% Sets the order of the Methfessel-Paxton smearing function.
      !%End
      call parse_variable(namespace, 'SmearingMPOrder', 1, this%MP_n)
    end if

    POP_SUB(smear_init)
  end subroutine smear_init


  !--------------------------------------------------
  subroutine smear_copy(to, from)
    type(smear_t), intent(out) :: to
    type(smear_t), intent(in)  :: from

    PUSH_SUB(smear_copy)

    to%method       = from%method
    to%dsmear       = from%dsmear
    to%e_fermi      = from%e_fermi
    to%el_per_state = from%el_per_state
    to%ef_occ       = from%ef_occ
    to%MP_n         = from%MP_n
    to%fermi_count  = from%fermi_count
    to%nik_factor   = from%nik_factor

    POP_SUB(smear_copy)
  end subroutine smear_copy


  !--------------------------------------------------
  subroutine smear_find_fermi_energy(this, namespace, eigenvalues, occupations, &
    qtot, nik, nst, kweights)
    type(smear_t),     intent(inout) :: this
    type(namespace_t), intent(in)    :: namespace
    real(real64),      intent(in)    :: eigenvalues(:,:), occupations(:,:)
    real(real64),      intent(in)    :: qtot, kweights(:)
    integer,           intent(in)    :: nik, nst

    real(real64), parameter   :: tol = 1.0e-10_real64
    integer            :: ist, ik, iter, maxq, weight, sumq_int, sum_weight
    real(real64)       :: sumq_frac
    logical            :: conv
    real(real64),   allocatable :: eigenval_list(:)
    integer, allocatable :: k_list(:), reorder(:)
    integer            :: fermi_count_up, fermi_count_down

    PUSH_SUB(smear_find_fermi_energy)

    maxq = this%el_per_state * nst * this%nspins
    if (maxq - qtot <= -tol) then ! not enough states
      message(1) = 'Not enough states'
      write(message(2),'(6x,a,f12.6,a,i10)')'(total charge = ', qtot, &
        ' max charge = ', maxq
      call messages_fatal(2, namespace=namespace)
    end if

    conv = .true.
    if (this%method == SMEAR_FIXED_OCC) then ! Fermi energy: last of occupied states
      ist_cycle: do ist = nst, 1, -1
        if (any(occupations(ist, :) > M_MIN_OCC)) then
          this%e_fermi =  eigenvalues(ist, 1)
          this%ef_occ   = occupations(ist, 1) / this%el_per_state
          do ik = 2, nik
            if (eigenvalues(ist, ik) > this%e_fermi .and. occupations(ist, ik) > M_MIN_OCC) then
              this%e_fermi  = eigenvalues(ist, ik)
              this%ef_occ   = occupations(ist, ik) / this%el_per_state
            end if
          end do
          exit ist_cycle
        end if
      end do ist_cycle

    else if (this%method == SMEAR_SEMICONDUCTOR) then
      ! first we sort the eigenvalues
      SAFE_ALLOCATE(eigenval_list(1:nst * nik))
      SAFE_ALLOCATE(       k_list(1:nst * nik))
      SAFE_ALLOCATE(      reorder(1:nst * nik))

      iter = 1
      do ist = 1, nst
        do ik = 1, nik
          eigenval_list(iter) = eigenvalues(ist, ik)
          k_list(iter) = ik
          reorder(iter) = iter
          iter = iter + 1
        end do
      end do

      call sort(eigenval_list, reorder)

      sumq_int = floor(qtot) * this%nik_factor
      sumq_frac = qtot * this%nik_factor - sumq_int
      if ( sumq_frac + tol > this%nik_factor ) then
        sumq_int = sumq_int + this%nik_factor
        sumq_frac = sumq_frac - this%nik_factor
      end if
      if (sumq_frac < tol) sumq_frac = M_ZERO

      do iter = 1, nst * nik
        weight = int(kweights(k_list(reorder(iter))) * this%nik_factor + M_HALF)
        if (.not. weight > 0) cycle
        this%e_fermi = eigenval_list(iter)
        this%ef_occ  = (sumq_int + sumq_frac) / (weight * this%el_per_state)

        if (sumq_int - weight * this%el_per_state <= 0) then
          ! count how many occupied states are at the fermi level,
          ! this is required later to fill the states
          this%fermi_count = 1
          fermi_count_up = 1
          fermi_count_down = 1
          sum_weight = weight
          do
            if (iter - fermi_count_down < 1) exit
            if (abs(this%e_fermi - eigenval_list(iter - fermi_count_down)) > TOL_SMEAR) exit
            weight = int(kweights(k_list(reorder(iter - fermi_count_down))) * this%nik_factor + M_HALF)
            if (weight > M_EPSILON) then
              sumq_int = sumq_int + weight * this%el_per_state
              sum_weight = sum_weight + weight
            end if
            fermi_count_down = fermi_count_down + 1
          end do
          do
            if (iter + fermi_count_up > nst*nik) exit
            if (abs(this%e_fermi - eigenval_list(iter + fermi_count_up)) > TOL_SMEAR) exit
            weight = int(kweights(k_list(reorder(iter + fermi_count_up))) * this%nik_factor + M_HALF)
            if (weight > M_EPSILON) then
              sum_weight = sum_weight + weight
            end if
            fermi_count_up = fermi_count_up + 1
          end do
          this%fermi_count = fermi_count_up + fermi_count_down - 1
          this%ef_occ  = (sumq_int + sumq_frac) / (sum_weight * this%el_per_state)
          exit
        end if

        sumq_int = sumq_int - weight * this%el_per_state
      end do
      ASSERT(this%ef_occ < M_ONE + 10.0_real64*M_EPSILON)

      SAFE_DEALLOCATE_A(eigenval_list)
      SAFE_DEALLOCATE_A(k_list)
      SAFE_DEALLOCATE_A(reorder)

    else ! bisection
      if (this%photodop) then
        ! find fermi energy for valence electrons
        call bisection_find_fermi_energy(this, namespace, this%dsmear, tol, &
          eigenvalues, kweights, nik, qtot-this%nephotodop, 1, this%photodop_bandmin-1, &
          this%e_fermi, this%ef_occ)

        ! find fermi energy for conduction electrons
        call bisection_find_fermi_energy(this, namespace, this%dsmear_cond, tol, &
          eigenvalues, kweights, nik, this%nephotodop, this%photodop_bandmin, nst, &
          this%e_fermi_cond, this%ef_occ_cond)
      else
        call bisection_find_fermi_energy(this, namespace, this%dsmear, tol, &
          eigenvalues, kweights, nik, qtot, 1, nst, this%e_fermi, this%ef_occ)
      end if
    end if

    POP_SUB(smear_find_fermi_energy)
  end subroutine smear_find_fermi_energy

  ! ---------------------------------------------------------
  subroutine bisection_find_fermi_energy(this, namespace, dsmear_in, tol, &
    eigenvalues, kweights, nik, q_in, start_band, end_band, e_fermi, ef_occ)
    type(smear_t),     intent(inout) :: this
    type(namespace_t), intent(in)    :: namespace
    real(real64),      intent(in)    :: dsmear_in
    real(real64),      intent(in)    :: tol
    integer,           intent(in)    :: nik
    real(real64),      intent(in)    :: eigenvalues(:,:)
    real(real64),      intent(in)    :: kweights(:), q_in
    integer,           intent(in)    :: start_band, end_band
    real(real64),      intent(out)   :: e_fermi, ef_occ

    integer, parameter :: nitmax = 200
    integer            :: ist, ik, iter
    real(real64)       :: drange, dsmear, emin, emax, xx, sumq
    logical            :: conv

    PUSH_SUB(bisection_find_fermi_energy)

    dsmear = max(1e-14_real64, dsmear_in)
    drange = dsmear * sqrt(-log(tol * 0.01_real64))

    emin = minval(eigenvalues) - drange
    emax = maxval(eigenvalues) + drange

    do iter = 1, nitmax
      e_fermi = M_HALF * (emin + emax)
      sumq         = M_ZERO

      do ik = 1, nik
        do ist = start_band, end_band
          xx   = (e_fermi - eigenvalues(ist, ik))/dsmear
          sumq = sumq + kweights(ik) * this%el_per_state * &
            smear_step_function(this, xx)
        end do
      end do

      conv = (abs(sumq - q_in) <= tol)
      if (conv) exit

      if (sumq <= q_in) emin = e_fermi
      if (sumq >= q_in) emax = e_fermi

      ef_occ = smear_step_function(this, M_ZERO)
    end do

    if (.not. conv) then
      message(1) = 'Fermi: did not converge.'
      call messages_fatal(1, namespace=namespace)
    end if

    POP_SUB(bisection_find_fermi_energy)
  end subroutine bisection_find_fermi_energy

  ! ---------------------------------------------------------
  subroutine smear_fill_occupations(this, eigenvalues, occupations, nik, nst)
    type(smear_t),   intent(in)    :: this
    real(real64),    intent(in)    :: eigenvalues(:,:)
    real(real64),    intent(inout) :: occupations(:,:)
    integer,         intent(in)    :: nik, nst

    integer :: ik, ist, ifermi
    real(real64)   :: dsmear, xx, dsmear_cond

    PUSH_SUB(smear_fill_occupations)

    if (this%method == SMEAR_FIXED_OCC) then
      ! do nothing
    else if (this%method == SMEAR_SEMICONDUCTOR) then
      ASSERT(this%fermi_count > 0 .and. this%fermi_count <= nik*nst)

      ifermi = 0
      do ik = 1, nik
        do ist = 1, nst
          xx = eigenvalues(ist, ik) - this%e_fermi
          if (xx < -TOL_SMEAR) then
            occupations(ist, ik) = this%el_per_state
          else if (abs(xx) <= TOL_SMEAR .and. ifermi < this%fermi_count) then
            occupations(ist, ik) = this%ef_occ * this%el_per_state
            ifermi = ifermi + 1
          else
            occupations(ist, ik) = M_ZERO
          end if

        end do
      end do

    else
      dsmear = max(1e-14_real64, this%dsmear)
      if (this%photodop) then
        dsmear_cond = max(1e-14_real64, this%dsmear_cond)
        do ik = 1, nik
          do ist = 1, nst
            if (ist < this%photodop_bandmin) then
              ! valence electrons
              xx = (this%e_fermi - eigenvalues(ist, ik))/dsmear
            else
              ! conduction electrons
              xx = (this%e_fermi_cond - eigenvalues(ist, ik))/dsmear_cond
            end if
            occupations(ist, ik) = smear_step_function(this, xx) * this%el_per_state
          end do
        end do
      else
        do ik = 1, nik
          do ist = 1, nst
            xx = (this%e_fermi - eigenvalues(ist, ik))/dsmear
            occupations(ist, ik) = smear_step_function(this, xx) * this%el_per_state
          end do
        end do
      end if
    end if

    POP_SUB(smear_fill_occupations)
  end subroutine smear_fill_occupations


  !--------------------------------------------------
  real(real64) function smear_calc_entropy(this, eigenvalues, &
    nik, nst, kweights, occ) result(entropy)
    type(smear_t),  intent(inout) :: this
    real(real64),   intent(in)    :: eigenvalues(:,:)
    real(real64),   intent(in)    :: kweights(:)
    integer,        intent(in)    :: nik, nst
    real(real64),   intent(in)    :: occ(:, :) !< used if fixed_occ

    integer :: ist, ik
    real(real64) :: dsmear, xx, term, ff

    PUSH_SUB(smear_calc_entropy)

    entropy = M_ZERO

    if (this%method == SMEAR_FIXED_OCC .or. this%method == SMEAR_SEMICONDUCTOR) then
      ! Fermi-Dirac entropy, not quite the same as will be obtained with true smearing
      ! RM Wentzcovitch, JL Martins, and PB Allen, Phys. Rev. B 45, 11372 (1992) eqn (5)
      ! also N Marzari PhD thesis p 117, http://quasiamore.mit.edu/phd/Marzari_PhD.pdf
      do ik = 1, nik
        do ist = 1, nst
          ff = occ(ist, ik) / this%el_per_state
          if (ff > M_ZERO .and. ff  <  M_ONE) then
            term = ff * log(ff) + (1 - ff) * log (1 - ff)
          else ! we have semiconducting smearing, or perverse occupations as in Methfessel-Paxton
            term = M_ZERO
          end if
          entropy = entropy - kweights(ik) * this%el_per_state * term
        end do
      end do
    else
      dsmear = max(1e-14_real64, this%dsmear)

      do ik = 1, nik
        do ist = 1, nst
          if (eigenvalues(ist, ik) < HUGE(M_ONE)) then
            xx = (this%e_fermi - eigenvalues(ist, ik)) / dsmear
            entropy = entropy - kweights(ik) * this%el_per_state *  &
              smear_entropy_function(this, xx)
          end if
        end do
      end do
    end if

    POP_SUB(smear_calc_entropy)
  end function smear_calc_entropy


  ! ---------------------------------------------------------
  real(real64) function smear_delta_function(this, xx) result(deltaf)
    type(smear_t), intent(in) :: this
    real(real64),  intent(in) ::  xx

    real(real64), parameter :: maxarg = 200.0_real64
    real(real64) :: xp, arg, hd, hp, aa
    integer :: ii, ni

    ! no PUSH_SUB, called too often

    ! smear_delta_function is not defined for SMEAR_FIXED_OCC
    ASSERT(this%method /= SMEAR_FIXED_OCC)

    deltaf = M_ZERO
    select case (this%method)
    case (SMEAR_SEMICONDUCTOR)
      if (abs(xx) <= M_EPSILON) then
        deltaf = this%ef_occ
      end if

    case (SMEAR_FERMI_DIRAC)
      if (abs(xx) <= 36.0_real64) then
        deltaf = M_ONE / (M_TWO + exp(-xx) + exp(xx))
      end if

    case (SMEAR_COLD)
      xp  = xx - M_ONE / sqrt(M_TWO)
      arg = min(maxarg, xp**2)

      deltaf = exp(-arg) / sqrt(M_PI) * (M_TWO - sqrt(M_TWO) * xx)

    case (SMEAR_METHFESSEL_PAXTON)
      arg    = min(maxarg, xx**2)
      deltaf = exp(-arg) / sqrt(M_PI)

      if (this%MP_n > 0) then ! recursion
        hd = M_ZERO
        hp = exp(-arg)
        ni = 0
        aa = M_ONE / sqrt(M_PI)
        do ii = 1, this%MP_n
          hd = M_TWO * xx * hp - M_TWO * ni * hd
          ni = ni + 1
          aa = -aa / (M_FOUR * ii)
          hp = M_TWO * xx * hd - M_TWO * ni * hp
          ni = ni + 1
          deltaf = deltaf + aa * hp
        end do
      end if

    case (SMEAR_SPLINE)
      xp     = abs(xx) + M_ONE / sqrt(M_TWO)
      deltaf = sqrt(M_E) * xp * exp(-xp * xp)

    end select

  end function smear_delta_function


  ! ---------------------------------------------------------
  real(real64) function smear_step_function(this, xx) result(stepf)
    type(smear_t), intent(in) :: this
    real(real64),  intent(in) ::  xx

    real(real64), parameter :: maxarg = 200.0_real64
    real(real64) :: xp, arg, hd, hp, aa
    integer :: ii, ni

    PUSH_SUB(smear_step_function)

    ! smear_step_function is not defined for SMEAR_FIXED_OCC
    ASSERT(this%method /= SMEAR_FIXED_OCC)

    stepf = M_ZERO
    select case (this%method)
    case (SMEAR_SEMICONDUCTOR)
      if (xx > M_ZERO) then
        stepf = M_ONE
      else if (abs(xx) <= M_EPSILON) then
        stepf = this%ef_occ
      end if

    case (SMEAR_FERMI_DIRAC)
      if (xx > maxarg) then
        stepf = M_ONE
      else if (xx > -maxarg) then
        stepf = M_ONE / (M_ONE + exp(-xx))
      end if

    case (SMEAR_COLD)
      xp  = xx - M_ONE / sqrt(M_TWO)
      arg = min(maxarg, xp**2)

      stepf = M_HALF * loct_erf(xp) + &
        M_ONE / sqrt(M_TWO * M_PI) * exp(-arg) + M_HALF

    case (SMEAR_METHFESSEL_PAXTON)
      stepf = M_HALF * loct_erfc(-xx)

      if (this%MP_n > 0) then ! recursion
        hd = M_ZERO
        arg = min(maxarg, xx**2)
        hp = exp(-arg)
        ni = 0
        aa = M_ONE / sqrt(M_PI)
        do ii = 1, this%MP_n
          hd = M_TWO * xx * hp - M_TWO * ni * hd
          ni = ni + 1
          aa = -aa / (M_FOUR * ii)
          stepf = stepf - aa * hd
          hp = M_TWO * xx * hd - M_TWO * ni * hp
          ni = ni + 1
        end do
      end if

    case (SMEAR_SPLINE)
      if (xx <= M_ZERO) then
        xp = xx - M_ONE / sqrt(M_TWO)
        stepf = M_HALF * sqrt(M_E) * exp(-xp * xp)
      else
        xp = xx + M_ONE / sqrt(M_TWO)
        stepf = M_ONE - M_HALF * sqrt(M_E) * exp(-xp * xp)
      end if

    end select

    POP_SUB(smear_step_function)
  end function smear_step_function


  ! ---------------------------------------------------------
  !> This function is defined as \f$ \int_{-infty}^x y delta(y) dy \f$
  real(real64) function smear_entropy_function(this, xx) result(entropyf)
    type(smear_t), intent(in) :: this
    real(real64),  intent(in) ::  xx

    real(real64), parameter :: maxarg = 200.0_real64
    real(real64) :: xp, arg, hd, hp, hpm1, aa
    integer :: ii, ni

    PUSH_SUB(smear_entropy_function)

    ! smear_entropy_function is not defined for SMEAR_FIXED_OCC
    ASSERT(this%method /= SMEAR_FIXED_OCC)

    entropyf = M_ZERO
    select case (this%method)
    case (SMEAR_SEMICONDUCTOR)

    case (SMEAR_FERMI_DIRAC)
      if (abs(xx) <= 36.0_real64) then
        xp = M_ONE / (M_ONE + exp(-xx))
        entropyf = xp * log(xp) + (M_ONE - xp) * log(M_ONE - xp)
      end if

    case (SMEAR_COLD)
      xp  = xx - M_ONE / sqrt(M_TWO)
      arg = min(maxarg, xp**2)

      entropyf =  M_ONE / sqrt(M_TWO * M_PI) * xp * exp(-arg)

    case (SMEAR_METHFESSEL_PAXTON)
      arg = min(maxarg, xx**2)
      entropyf = -M_HALF * exp(-arg) / sqrt(M_PI)

      if (this%MP_n > 0) then ! recursion
        hd = M_ZERO
        hp = exp(-arg)
        ni = 0
        aa = M_ONE / sqrt(M_PI)
        do ii = 1, this%MP_n
          hd = M_TWO * xx * hp - M_TWO * ni * hd
          ni = ni + 1
          hpm1 = hp
          hp = M_TWO * xx * hd - M_TWO * ni * hp
          ni = ni + 1
          aa = -aa / (M_FOUR * ii)
          entropyf = entropyf - aa * (M_HALF * hp + hpm1 * ni)
        end do
      end if

    case (SMEAR_SPLINE)
      xp = abs(xx) + M_ONE / sqrt(M_TWO)
      entropyf = -sqrt(M_E) * (abs(xx) * exp(-xp * xp) / M_TWO + sqrt(M_PI) / M_FOUR * loct_erfc(xp))

    end select

    POP_SUB(smear_entropy_function)
  end function smear_entropy_function


  ! ---------------------------------------------------------
  logical pure function smear_is_semiconducting(this) result(answer)
    type(smear_t), intent(in) :: this

    answer = this%method == SMEAR_SEMICONDUCTOR

  end function smear_is_semiconducting

  subroutine smear_write_info(this, namespace, iunit)
    type(smear_t),     intent(in) :: this
    type(namespace_t), intent(in) :: namespace
    integer, optional, intent(in) :: iunit      !< optional file unit


    if (this%method /= SMEAR_SEMICONDUCTOR .and. this%method /= SMEAR_FIXED_OCC) then
      if (this%photodop) then
        write(message(1), '(a,f12.6,1x,a)') "Fermi energy (valence   ) = ", &
          units_from_atomic(units_out%energy, this%e_fermi), units_abbrev(units_out%energy)
        write(message(2), '(a,f12.6,1x,a)') "Fermi energy (conduction) = ", &
          units_from_atomic(units_out%energy, this%e_fermi_cond), units_abbrev(units_out%energy)
        call messages_info(2, iunit=iunit, namespace=namespace)
      else
        write(message(1), '(a,f12.6,1x,a)') "Fermi energy = ", &
          units_from_atomic(units_out%energy, this%e_fermi), units_abbrev(units_out%energy)
        call messages_info(1, iunit=iunit, namespace=namespace)
      end if
    end if

  end subroutine smear_write_info


end module smear_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
