!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module xc_functional_oct_m
  use debug_oct_m
  use global_oct_m
  use libvdwxc_oct_m
  use messages_oct_m
  use namespace_oct_m
  use parser_oct_m
  use pseudo_oct_m
  use xc_f03_lib_m
#ifdef HAVE_LIBXC_FUNCS
  use xc_f03_funcs_m
#endif

  implicit none

  ! Although the following file only contain comments, we include it here to make sure it exists.
  ! Otherwise the code might compile, but not run properly, as the variables documentations
  ! will be incomplete.
#include "functionals_list.F90"

  private
  public ::                     &
    xc_functional_t,                &
    xc_functional_init,             &
    xc_functional_end,              &
    xc_functional_write_info,       &
    xc_functional_is_not_size_consistent, &
    xc_functional_is_energy_functional, &
    xc_get_default_functional


  !> This adds to the constants defined in libxc. But since in that module
  !! the OEP functionals are not included, it is better to put it here.
  integer, public, parameter ::   &
    XC_OEP_X = 901,               &  !< Exact exchange
    XC_OEP_X_SLATER = 902,        &  !< Slater approximation to the exact exchange
    XC_OEP_X_FBE    = 903,        &  !< Exchange approximation based on the force balance equation
    XC_KS_INVERSION = 904,        &  !< inversion of Kohn-Sham potential
    XC_RDMFT_XC_M   = 905,        &  !< RDMFT Mueller functional
    XC_OEP_X_FBE_SL = 906,        &  !< Exchange approximation based on the force balance equation - Sturn-Liouville version
    XC_LDA_C_FBE    = 907,        &  !< LDA correlation based ib the force-balance equation
    XC_LDA_C_FBE_SL = 908,        &  !< LDA correlation based ib the force-balance equation - Sturm-Liouville version
    XC_HALF_HARTREE = 917,        &  !< half-Hartree exchange for two electrons (supports complex scaling)
    XC_VDW_C_VDWDF = 918,         &  !< vdw-df correlation from libvdwxc
    XC_VDW_C_VDWDF2 = 919,        &  !< vdw-df2 correlation from libvdwxc
    XC_VDW_C_VDWDFCX = 920,       &  !< vdw-df-cx correlation from libvdwxc
    XC_HYB_GGA_XC_MVORB_HSE06 = 921, &  !< Density-based mixing parameter of HSE06
    XC_HYB_GGA_XC_MVORB_PBEH = 922,  &  !< Density-based mixing parameter of PBE0
    XC_MGGA_X_NC_BR = 923,           &  !< Noncollinear version of the Becke-Roussel functional
    XC_MGGA_X_NC_BR_1 = 924,         &  !< Noncollinear version of the Becke-Roussel functional, gamma=1
    XC_MGGA_C_NC_CS = 925,           &  !< Noncollinear version of the Colle-Salvetti correlation functional
    XC_MGGA_X_NC_BR_EXPLICIT = 926      !< Noncollinear version of the Becke-Roussel functionalwith an explicit inversion of x(y), gamma = 0.8

  !> declaring 'family' constants for 'functionals' not handled by libxc
  !! careful not to use a value defined in libxc for another family!
  integer, public, parameter :: &
    XC_FAMILY_KS_INVERSION = 1024, &
    XC_FAMILY_RDMFT = 2048, &
    XC_FAMILY_LIBVDWXC = 4096, &
    XC_FAMILY_NC_LDA = 8192, &
    XC_FAMILY_NC_MGGA = 16384

  type xc_functional_t
    ! Components are public by default
    integer         :: family = XC_FAMILY_UNKNOWN !< LDA, GGA, etc.
    integer         :: type   = 0                 !< exchange, correlation, or exchange-correlation
    integer         :: id     = 0                 !< identifier

    integer         :: spin_channels = 0     !< XC_UNPOLARIZED | XC_POLARIZED
    integer         :: flags         = 0     !< XC_FLAGS_HAVE_EXC + XC_FLAGS_HAVE_VXC + ...

    logical, private :: from_libxc    = .false.

    type(xc_f03_func_t)               :: conf         !< the pointer used to call the library
    type(xc_f03_func_info_t), private :: info         !< information about the functional
    type(libvdwxc_t)                  :: libvdwxc     !< libvdwxc data for van der Waals functionals
  end type xc_functional_t

  integer, public, parameter :: LIBXC_C_INDEX = 1000

contains

  ! ---------------------------------------------------------
  subroutine xc_functional_init(functl, namespace, id, ndim, nel, spin_channels)
    type(xc_functional_t), intent(inout) :: functl
    type(namespace_t),     intent(in)    :: namespace
    integer,               intent(in)    :: id
    integer,               intent(in)    :: ndim
    real(real64),          intent(in)    :: nel
    integer,               intent(in)    :: spin_channels

    integer :: interact_1d
    real(real64)   :: alpha, parameters(2)

    PUSH_SUB(xc_functional_init)

    ! initialize structure
    functl%id = id
    functl%spin_channels = spin_channels

    if (functl%id == 0) then
      functl%family = XC_FAMILY_NONE
    else
      ! get the family of the functional
      functl%family = xc_f03_family_from_id(functl%id)
      ! this also ensures it is actually a functional defined by the linked version of libxc

      if (functl%family == XC_FAMILY_UNKNOWN) then

        select case (functl%id)
        case (XC_OEP_X, XC_OEP_X_SLATER, XC_OEP_X_FBE, XC_OEP_X_FBE_SL, XC_LDA_C_FBE_SL)
          functl%family = XC_FAMILY_OEP

        case (XC_KS_INVERSION)
          functl%family = XC_FAMILY_KS_INVERSION

        case (XC_HALF_HARTREE)
          call messages_experimental("half-Hartree exchange", namespace=namespace)
          functl%family = XC_FAMILY_LDA ! XXX not really

        case (XC_VDW_C_VDWDF, XC_VDW_C_VDWDF2, XC_VDW_C_VDWDFCX)
          functl%family = XC_FAMILY_LIBVDWXC
          !functl%flags = functl%flags + XC_FLAGS_HAVE_VXC + XC_FLAGS_HAVE_EXC

        case (XC_RDMFT_XC_M)
          functl%family = XC_FAMILY_RDMFT

        case (XC_HYB_GGA_XC_MVORB_HSE06, XC_HYB_GGA_XC_MVORB_PBEH)
          functl%family = XC_FAMILY_HYB_GGA

        case (XC_MGGA_X_NC_BR, XC_MGGA_X_NC_BR_1, XC_MGGA_X_NC_BR_EXPLICIT, XC_MGGA_C_NC_CS)
          functl%family = XC_FAMILY_NC_MGGA

        case default
          call messages_input_error(namespace, 'XCFunctional', 'Unknown functional')

        end select
      end if
    end if

    if (functl%family == XC_FAMILY_OEP) then
      functl%type = XC_EXCHANGE
      functl%flags = XC_FLAGS_1D + XC_FLAGS_2D + XC_FLAGS_3D

    else if (functl%family == XC_FAMILY_KS_INVERSION .or. functl%family == XC_FAMILY_RDMFT) then
      functl%type = XC_EXCHANGE_CORRELATION
      functl%flags = XC_FLAGS_1D + XC_FLAGS_2D + XC_FLAGS_3D

    else if (functl%family == XC_FAMILY_LIBVDWXC) then
      call xc_f03_func_init(functl%conf, XC_LDA_C_PW, spin_channels)
      functl%info = xc_f03_func_get_info(functl%conf)
      functl%type = xc_f03_func_info_get_kind(functl%info)
      functl%flags = xc_f03_func_info_get_flags(functl%info)
      ! Convert Octopus code for functional into corresponding libvdwxc code:
      call libvdwxc_init(functl%libvdwxc, namespace, functl%id - XC_VDW_C_VDWDF + 1)

    else if (functl%id == XC_HALF_HARTREE) then
      functl%type = XC_EXCHANGE_CORRELATION
      functl%flags = XC_FLAGS_1D + XC_FLAGS_2D + XC_FLAGS_3D

    else if(functl%id == XC_LDA_C_FBE) then
      functl%family = XC_FAMILY_LDA
      functl%type = XC_CORRELATION
      functl%flags = XC_FLAGS_HAVE_EXC + XC_FLAGS_HAVE_VXC + XC_FLAGS_3D

    else if (functl%id == XC_MGGA_X_NC_BR .or. functl%id == XC_MGGA_X_NC_BR_1 .or. functl%id == XC_MGGA_X_NC_BR_EXPLICIT) then
      functl%type = XC_EXCHANGE
      functl%flags = XC_FLAGS_HAVE_VXC + XC_FLAGS_HAVE_EXC + XC_FLAGS_3D

    else if(functl%id == XC_MGGA_C_NC_CS) then
      functl%type = XC_CORRELATION
      functl%flags = XC_FLAGS_HAVE_VXC + XC_FLAGS_HAVE_EXC + XC_FLAGS_3D

    else if (functl%family == XC_FAMILY_NONE) then
      functl%type = -1
      functl%flags = XC_FLAGS_1D + XC_FLAGS_2D + XC_FLAGS_3D

    else ! handled by libxc
      ! initialize
      functl%from_libxc = .true.

      !For the two MVORB functionals, we initialize libxc with the non-MVORB functionals
      select case (functl%id)
      case (XC_HYB_GGA_XC_MVORB_HSE06)
        call xc_f03_func_init(functl%conf, XC_HYB_GGA_XC_HSE06, spin_channels)

      case (XC_HYB_GGA_XC_MVORB_PBEH)
        call xc_f03_func_init(functl%conf, XC_HYB_GGA_XC_PBEH, spin_channels)

      case default
        call xc_f03_func_init(functl%conf, functl%id, spin_channels)
      end select
      functl%info     = xc_f03_func_get_info(functl%conf)
      functl%type     = xc_f03_func_info_get_kind(functl%info)
      functl%flags    = xc_f03_func_info_get_flags(functl%info)

      ! FIXME: no need to say this for kernel
      if (bitand(functl%flags, XC_FLAGS_HAVE_EXC) == 0) then
        message(1) = 'Specified functional does not have total energy available.'
        message(2) = 'Corresponding component of energy will just be left as zero.'
        call messages_warning(2, namespace=namespace)
      end if

      if (bitand(functl%flags, XC_FLAGS_HAVE_VXC) == 0) then
        message(1) = 'Specified functional does not have XC potential available.'
        message(2) = 'Cannot run calculations. Choose another XCFunctional.'
        call messages_fatal(2, namespace=namespace)
      end if
    end if

    ! Octopus-defined functionals need to provide the proper flags
    if (functl%family /= XC_FAMILY_NONE) then
      call xc_check_dimension(functl, ndim, namespace)
    end if

    ! Yukawa hybrids not supported yet
    if (bitand(functl%flags, XC_FLAGS_HYB_CAMY) /= 0) then
      call messages_not_implemented("Yukawa-range separated hybrids", namespace=namespace)
    end if

    ! VV10 correlations not supported yet
    if (bitand(functl%flags, XC_FLAGS_VV10) /= 0) then
      call messages_not_implemented("VV10 correlation functionals", namespace=namespace)
    end if

    ! FIXME: aren`t there other parameters that can or should be set?
    ! special parameters that have to be configured
    select case (functl%id)
      ! FIXME: aren`t there other Xalpha functionals?
    case (XC_LDA_C_XALPHA)

      !%Variable Xalpha
      !%Type float
      !%Default 1.0
      !%Section Hamiltonian::XC
      !%Description
      !% The parameter of the Slater X<math>\alpha</math> functional. Applies only for
      !% <tt>XCFunctional = xc_lda_c_xalpha</tt>.
      !%End
      call parse_variable(namespace, 'Xalpha', M_ONE, parameters(1))

      call xc_f03_func_set_ext_params(functl%conf, parameters(1))

      ! FIXME: doesn`t this apply to other 1D functionals?
    case (XC_LDA_X_1D_SOFT, XC_LDA_C_1D_CSC)
      !%Variable Interaction1D
      !%Type integer
      !%Default interaction_soft_coulomb
      !%Section Hamiltonian::XC
      !%Description
      !% When running in 1D, one has to soften the Coulomb interaction. This softening
      !% is not unique, and several possibilities exist in the literature.
      !%Option interaction_exp_screened 0
      !% Exponentially screened Coulomb interaction.
      !% See, <i>e.g.</i>, M Casula, S Sorella, and G Senatore, <i>Phys. Rev. B</i> <b>74</b>, 245427 (2006).
      !%Option interaction_soft_coulomb 1
      !% Soft Coulomb interaction of the form <math>1/\sqrt{x^2 + \alpha^2}</math>.
      !%End
      call messages_obsolete_variable(namespace, 'SoftInteraction1D_alpha', 'Interaction1D')
      call parse_variable(namespace, 'Interaction1D', OPTION__INTERACTION1D__INTERACTION_SOFT_COULOMB, interact_1d)

      !%Variable Interaction1DScreening
      !%Type float
      !%Default 1.0
      !%Section Hamiltonian::XC
      !%Description
      !% Defines the screening parameter <math>\alpha</math> of the softened Coulomb interaction
      !% when running in 1D.
      !%End
      call messages_obsolete_variable(namespace, 'SoftInteraction1D_alpha', 'Interaction1DScreening')
      call parse_variable(namespace, 'Interaction1DScreening', M_ONE, alpha)
      parameters(1) = real(interact_1d, real64)
      parameters(2) = alpha
      call xc_f03_func_set_ext_params(functl%conf, parameters(1))

    case (XC_GGA_X_LB)
      if (parse_is_defined(namespace, 'LB94_modified')) then
        call messages_obsolete_variable(namespace, 'LB94_modified')
      end if

      if (parse_is_defined(namespace, 'LB94_threshold')) then
        call messages_obsolete_variable(namespace, 'LB94_threshold')
      end if
    end select


    ! For functionals that depend on the number of electrons, we set the number of electrons
    ! This is currently the case of GGA_K_TFLW, LDA_X_2D_PRM, and LDA_X_RAE
    if (xc_functional_is_not_size_consistent(functl, namespace)) then
      ASSERT(xc_f03_func_info_get_n_ext_params(functl%info) == 1)
      parameters(1) = nel
      call xc_f03_func_set_ext_params(functl%conf, parameters(1))
      write(message(1), '(a,i1)') "Info: Setting the number of electrons for the functional for spin ", spin_channels
      call messages_info(1, namespace=namespace)
    end if

    POP_SUB(xc_functional_init)
  end subroutine xc_functional_init


  ! ---------------------------------------------------------
  subroutine xc_functional_end(functl)
    type(xc_functional_t), intent(inout) :: functl

    PUSH_SUB(xc_functional_end)

    if (functl%family /= XC_FAMILY_NONE .and. functl%family /= XC_FAMILY_OEP .and. &
      functl%family /= XC_FAMILY_KS_INVERSION .and. functl%id /= XC_HALF_HARTREE &
      .and. functl%id /= XC_LDA_C_FBE &
      .and. functl%family /= XC_FAMILY_NC_LDA .and. functl%family /= XC_FAMILY_NC_MGGA) then
      call xc_f03_func_end(functl%conf)
    end if

    if (functl%family == XC_FAMILY_LIBVDWXC) then
      call libvdwxc_end(functl%libvdwxc)
    end if

    POP_SUB(xc_functional_end)
  end subroutine xc_functional_end


  ! ---------------------------------------------------------
  !> @brief Write functional information
  subroutine xc_functional_write_info(functl, iunit, namespace)
    type(xc_functional_t),       intent(in) :: functl
    integer,           optional, intent(in) :: iunit
    type(namespace_t), optional, intent(in) :: namespace

    character(len=1024) :: reference
    character(len=120) :: family
    integer :: ii, ind, istart, iend

    PUSH_SUB(xc_functional_write_info)

    if (functl%family == XC_FAMILY_OEP) then
      ! this is handled separately

      select case (functl%id)
      case (XC_OEP_X)
        write(message(1), '(2x,a)') 'Exchange'
        write(message(2), '(4x,a)') 'Exact exchange'
        call messages_info(2, iunit=iunit, namespace=namespace)

      case(XC_OEP_X_SLATER)
        write(message(1), '(2x,a)') 'Exchange'
        write(message(2), '(4x,a)') 'Slater exchange'
        call messages_info(2, iunit=iunit, namespace=namespace)

      case (XC_OEP_X_FBE)
        write(message(1), '(2x,a)') 'Exchange'
        write(message(2), '(4x,a)') 'Force-based local exchange'
        write(message(3), '(4x,a)') '[1] Tancogne-Dejean et al., J. Chem. Phys. 160, 024103 (2024)'
        call messages_info(3, iunit=iunit, namespace=namespace)

      case (XC_OEP_X_FBE_SL)
        write(message(1), '(2x,a)') 'Exchange'
        write(message(2), '(4x,a)') 'Force-based local exchange - Sturm-Liouville'
        call messages_info(2, iunit=iunit, namespace=namespace)

      case (XC_LDA_C_FBE_SL)
        write(message(1), '(2x,a)') 'Correlation'
        write(message(2), '(4x,a)') 'Force-based local-density correlation - Sturm-Liouville'
        call messages_info(2, iunit=iunit, namespace=namespace)

      case default
        ASSERT(.false.)
      end select

    else if ( functl%family == XC_FAMILY_RDMFT) then
      select case (functl%id)
      case (XC_RDMFT_XC_M)
        write(message(1), '(2x,a)') 'RDMFT'
        write(message(2), '(4x,a)') 'Mueller functional'
        call messages_info(2, iunit=iunit, namespace=namespace)
      end select

    else if (functl%family == XC_FAMILY_KS_INVERSION) then
      ! this is handled separately
      select case (functl%id)
      case (XC_KS_INVERSION)
        write(message(1), '(2x,a)') 'Exchange-Correlation:'
        write(message(2), '(4x,a)') '  KS Inversion'
        call messages_info(2, iunit=iunit, namespace=namespace)
      end select

    else if (functl%family == XC_FAMILY_LIBVDWXC) then
      call libvdwxc_write_info(functl%libvdwxc, iunit=iunit)

    else if (functl%id == XC_MGGA_X_NC_BR) then
      write(message(1), '(2x,a)') 'Exchange'
      write(message(2), '(4x,a)') 'Noncollinear Becke-Roussel (MGGA)'
      write(message(3), '(4x,a)') '[1] N. Tancogne-Dejean, A. Rubio, and C. A. Ullrich, Phys. Rev. B 107, 165111 (2023)'
      call messages_info(3, iunit)

    else if (functl%id == XC_MGGA_X_NC_BR_1) then
      write(message(1), '(2x,a)') 'Exchange'
      write(message(2), '(4x,a)') 'Noncollinear Becke-Roussel, gamma = 1.0 (MGGA)'
      write(message(3), '(4x,a)') '[1] N. Tancogne-Dejean, A. Rubio, and C. A. Ullrich, Phys. Rev. B 107, 165111 (2023)'
      call messages_info(3, iunit)

    else if (functl%id == XC_MGGA_X_NC_BR_EXPLICIT) then
      write(message(1), '(2x,a)') 'Exchange'
      write(message(2), '(4x,a)') 'Noncollinear Becke-Roussel (MGGA) with explicit inversion of x(y)'
      write(message(3), '(4x,a)') '[1] N. Tancogne-Dejean, A. Rubio, and C. A. Ullrich, Phys. Rev. B 107, 165111 (2023)'
      write(message(4), '(4x,a)') '[2] E. Proynov et al., Chemical Physics Letters 455, 103 (2008)'
      call messages_info(4, iunit)

    else if (functl%id == XC_MGGA_C_NC_CS) then
      write(message(1), '(2x,a)') 'Correlation'
      write(message(2), '(4x,a)') 'Noncollinear Colle-Salvetti (MGGA)'
      write(message(3), '(4x,a)') '[1] N. Tancogne-Dejean, A. Rubio, and C. A. Ullrich, Phys. Rev. B 107, 165111 (2023)'
      call messages_info(3, iunit)


    else if (functl%id == XC_HALF_HARTREE) then
      write(message(1), '(2x,a)') 'Exchange-Correlation:'
      write(message(2), '(4x,a)') 'Half-Hartree two-electron exchange'
      call messages_info(2, iunit=iunit, namespace=namespace)

    else if(functl%id == XC_LDA_C_FBE) then
      write(message(1), '(2x,a)') 'Correlation'
      write(message(2), '(4x,a)') 'Force-based LDA correlation'
      write(message(3), '(4x,a)') '[1] N. Tancogne-Dejean, M. Penz, M. Ruggenthaler, A. Rubio, J. Phys. Chem. A 129, 3132 (2025)'
      call messages_info(3, iunit=iunit, namespace=namespace)

    else if (functl%family /= XC_FAMILY_NONE) then ! all the other families
      select case (functl%type)
      case (XC_EXCHANGE)
        write(message(1), '(2x,a)') 'Exchange'
      case (XC_CORRELATION)
        write(message(1), '(2x,a)') 'Correlation'
      case (XC_EXCHANGE_CORRELATION)
        write(message(1), '(2x,a)') 'Exchange-correlation'
      case (XC_KINETIC)
        call messages_not_implemented("kinetic-energy functionals", namespace=namespace)
      case default
        write(message(1), '(a,i6,a,i6)') "Unknown functional type ", functl%type, ' for functional ', functl%id
        call messages_fatal(1, namespace=namespace)
      end select

      select case (functl%family)
      case (XC_FAMILY_LDA)
        write(family,'(a)') "LDA"
      case (XC_FAMILY_GGA)
        write(family,'(a)') "GGA"
      case (XC_FAMILY_MGGA)
        write(family,'(a)') "MGGA"
      case (XC_FAMILY_LCA)
        call messages_not_implemented("Support of LCA functional", namespace=namespace)
      case (XC_FAMILY_HYB_LDA)
        write(family,'(a)') "Hybrid LDA"
      case (XC_FAMILY_HYB_GGA)
        write(family,'(a)') "Hybrid GGA"
      case (XC_FAMILY_HYB_MGGA)
        write(family,'(a)') "Hybrid MGGA"
      end select
      write(message(2), '(4x,4a)') trim(xc_f03_func_info_get_name(functl%info)), ' (', trim(family), ')'
      call messages_info(2, iunit=iunit, namespace=namespace)

      ii = 0
      do while(ii >= 0)
        ! Preventing End of record error as message(1) has length of 256, and libxc returns 1024 characters
        ind = ii + 1
        write(reference, '(a)') trim(xc_f03_func_reference_get_ref(xc_f03_func_info_get_references(functl%info, ii)))
        ! Some references are in fact notes, and references are void in this case
        if (len_trim(reference) == 0) cycle
        istart = 1
        iend = len(message(1))-10
        write(message(1), '(4x,a,i1,2a)') '[', ind, '] ', trim(reference(istart:iend))
        call messages_info(1, iunit=iunit, namespace=namespace)
        do while (iend < len_trim(reference))
          istart = iend
          iend = min(istart + len(message(1))-10, len_trim(reference))
          write(message(1), '(7x,a)') trim(reference(istart:iend))
          call messages_info(1, iunit=iunit, namespace=namespace)
        end do
      end do
    end if

    POP_SUB(xc_functional_write_info)
  end subroutine xc_functional_write_info


  !>@brief Returns the default functional given the one parsed from the pseudopotentials and the space dimensions
  integer function xc_get_default_functional(dim, pseudo_x_functional, pseudo_c_functional) result(default)
    integer, intent(in) :: dim
    integer, intent(in) :: pseudo_x_functional, pseudo_c_functional

    PUSH_SUB(xc_get_default_functional)

    default = 0

    if (pseudo_x_functional /= PSEUDO_EXCHANGE_ANY) then
      default = pseudo_x_functional
    else
      select case (dim)
      case (3)
        default = XC_LDA_X
      case (2)
        default = XC_LDA_X_2D
      case (1)
        default = XC_LDA_X_1D_SOFT
      end select
    end if

    ASSERT(default >= 0)

    if (pseudo_c_functional /= PSEUDO_CORRELATION_ANY) then
      default = default + LIBXC_C_INDEX*pseudo_c_functional
    else
      select case (dim)
      case (3)
        default = default + LIBXC_C_INDEX * XC_LDA_C_PZ_MOD
      case (2)
        default = default + LIBXC_C_INDEX * XC_LDA_C_2D_AMGB
      case (1)
        default = default + LIBXC_C_INDEX * XC_LDA_C_1D_CSC
      end select
    end if

    ASSERT(default >= 0)

    POP_SUB(xc_get_default_functional)
  end function xc_get_default_functional

  !>@brief Check that the selected functional is compatible with the space dimension
  subroutine xc_check_dimension(functl, ndim, namespace)
    type(xc_functional_t), intent(in) :: functl
    integer,               intent(in) :: ndim
    type(namespace_t),     intent(in) :: namespace

    logical :: ok

    PUSH_SUB(xc_check_dimension)

    ! Check that the functional is used for the correct space dimension
    ok = bitand(functl%flags, XC_FLAGS_1D) /= 0
    if (ndim == 1 .and. (.not. ok)) then
      message(1) = 'Cannot use the specified functionals in 1D.'
      call messages_fatal(1, namespace=namespace)
    end if

    ok = bitand(functl%flags, XC_FLAGS_2D) /= 0
    if (ndim == 2 .and. (.not. ok)) then
      message(1) = 'Cannot use the specified functionals in 2D.'
      call messages_fatal(1, namespace=namespace)
    end if

    ok = bitand(functl%flags, XC_FLAGS_3D) /= 0
    if (ndim == 3 .and. (.not. ok)) then
      message(1) = 'Cannot use the specified functionals in 3D.'
      call messages_fatal(1, namespace=namespace)
    end if

    POP_SUB(xc_check_dimension)
  end subroutine xc_check_dimension

  !> Does the functional depend on the number of electrons or not
  logical function xc_functional_is_not_size_consistent(functl, namespace)
    type(xc_functional_t), intent(in) :: functl
    type(namespace_t),     intent(in) :: namespace

    integer :: n_ext_params, ip
    character(len=128) :: ext_params_name

    xc_functional_is_not_size_consistent = .false.
    if (.not. functl%from_libxc) return

    PUSH_SUB(xc_functional_is_not_size_consistent)

    ! External parameters
    n_ext_params = xc_f03_func_info_get_n_ext_params(functl%info)
    do ip = 0, n_ext_params-1
      ext_params_name = xc_f03_func_info_get_ext_params_name(functl%info, ip)
      ! Internal parameters starts with '_'
      if (ext_params_name(1:1) == '_') cycle
      if (trim(xc_f03_func_info_get_ext_params_description(functl%info, ip)) == 'Number of electrons') then
        xc_functional_is_not_size_consistent = .true.
      end if
    end do

    ! We only support size inconsistent functionals with a single parameter at the moment
    if (xc_functional_is_not_size_consistent .and. n_ext_params > 1) then
      message(1) = 'The selected functional is currently not supported.'
      call messages_fatal(1, namespace=namespace)
    end if

    POP_SUB(xc_functional_is_not_size_consistent)
  end function xc_functional_is_not_size_consistent

  logical pure function xc_functional_is_energy_functional(functl)
    type(xc_functional_t), intent(in) :: functl

    xc_functional_is_energy_functional = bitand(functl%flags, XC_FLAGS_HAVE_EXC) /= 0
  end function xc_functional_is_energy_functional

end module xc_functional_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
