# The MIT License (MIT)
#
# Copyright (c) 2018 Nathan Osman
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

find_package(Qt5Core REQUIRED)

# Retrieve the absolute path to qmake and then use that path to find
# the windeployqt and macdeployqt binaries
get_target_property(_qmake_executable Qt5::qmake IMPORTED_LOCATION)
get_filename_component(_qt_bin_dir "${_qmake_executable}" DIRECTORY)

find_program(WINDEPLOYQT_EXECUTABLE windeployqt HINTS "${_qt_bin_dir}")
if(WIN32 AND NOT WINDEPLOYQT_EXECUTABLE)
    message(FATAL_ERROR "windeployqt not found")
endif()

find_program(MACDEPLOYQT_EXECUTABLE macdeployqt HINTS "${_qt_bin_dir}")
if(APPLE AND NOT MACDEPLOYQT_EXECUTABLE)
    message(FATAL_ERROR "macdeployqt not found")
endif()

# Add commands that copy the required Qt files to the same directory as the
# target after being built as well as including them in final installation
function(windeployqt target)

    # Run windeployqt immediately after build
    # Deploy all files in a deployqt sub directory so that we can copy it with
    # install() below
    # Define the plugindir otherwise plugins directories are copied directly in
    # deployqt/<plugin-type>/ instead of in deployqt/plugins/<plugin-type>/
    # and that causes QIcon to fail to load svg icons (at least with Qt 5.12.8)
    add_custom_command(TARGET ${target} POST_BUILD
        COMMAND "${CMAKE_COMMAND}" -E
            env PATH="${_qt_bin_dir}" "${WINDEPLOYQT_EXECUTABLE}"
                --no-angle
                --no-opengl-sw
                \"$<TARGET_FILE:${target}>\"
                --dir ${PROJECT_BINARY_DIR}/deployqt
                --plugindir ${PROJECT_BINARY_DIR}/deployqt/plugins
        COMMENT "Deploying Qt..."
    )

    # Install the deployed files so CPack pick them up
    # Use "deployqt/" with a trailing slash so that CMake does not create a
    # "deployqt" directory in our destination directory
    install(
        DIRECTORY ${PROJECT_BINARY_DIR}/deployqt/
        DESTINATION ${BIN_INSTALL_DIR}
    )
endfunction()

# Add commands that copy the required Qt files to the application bundle
# represented by the target.
function(macdeployqt target)
    execute_process(
        COMMAND ${_qmake_executable} -query QT_INSTALL_TRANSLATIONS
        OUTPUT_VARIABLE _qt_translations_dir
        OUTPUT_STRIP_TRAILING_WHITESPACE
    )

    add_custom_command(TARGET ${target} POST_BUILD
        COMMAND "${MACDEPLOYQT_EXECUTABLE}"
            \"$<TARGET_FILE_DIR:${target}>/../..\"
        COMMENT "Deploying Qt..."
    )
    install(
        DIRECTORY ${_qt_translations_dir}/
        DESTINATION $<TARGET_FILE_DIR:${target}>/../Translations
        FILES_MATCHING PATTERN "qtbase_*.qm"
    )
endfunction()

mark_as_advanced(WINDEPLOYQT_EXECUTABLE MACDEPLOYQT_EXECUTABLE)
