# syntax=docker/dockerfile:1.7
FROM debian:bookworm-slim AS build

RUN DEBIAN_FRONTEND=noninteractive apt-get -y update \
  && apt-get -y install \
  build-essential \
  cmake \
  g++ \
  gcc \
  gfortran \
  git \
  libatlas-base-dev \
  libblas-dev \
  libboost-dev \
  libcgal-dev \
  libelpa-dev \
  libetsf-io-dev \
  libfftw3-dev \
  libgmp-dev \
  libgsl-dev \
  liblapack-dev \
  libmetis-dev \
  libmpfr-dev \
  libnetcdff-dev \
  libnlopt-dev \
  libopenmpi-dev \
  libscalapack-mpi-dev \
  libyaml-dev \
  ninja-build \
  pkgconf \
  && rm -rf /var/lib/apt/lists/*

RUN sed -iE 's@${_IMPORT_PREFIX}/lib/@${_IMPORT_PREFIX}/lib/${CMAKE_LIBRARY_ARCHITECTURE}/@g' /usr/lib/cmake/scalapack-2.2.1.openmpi/scalapack-targets-none.cmake

WORKDIR /opt/octopus
COPY . .
RUN cmake --preset default --install-prefix /app -DOCTOPUS_OpenMP=On -DOCTOPUS_MPI=On -DOCTOPUS_APP_TESTS=Off -DOCTOPUS_ScaLAPACK=On
RUN cmake --build ./cmake-build-release
RUN cmake --install ./cmake-build-release


FROM debian:bookworm-slim AS deploy-base
RUN DEBIAN_FRONTEND=noninteractive apt-get -y update \
    && apt-get -y install --no-install-recommends --no-install-suggests \
    libatlas3-base \
    libblas3 \
    libelpa19 \
    libfftw3-bin \
    libgmp10 \
    libgsl27 \
    liblapack3 \
    libmetis5 \
    libmpfr6 \
    libnetcdff7 \
    libnlopt0 \
    libscalapack-openmpi2.2 \
    && rm -rf /var/lib/apt/lists/*

COPY --from=build /app /app

# allow root execution of mpirun
ENV OMPI_ALLOW_RUN_AS_ROOT=1 OMPI_ALLOW_RUN_AS_ROOT_CONFIRM=1
# https://github.com/open-mpi/ompi/issues/4948
ENV OMPI_MCA_btl_vader_single_copy_mechanism=none
# set number of OpenMP threads to 1 by default
ENV OMP_NUM_THREADS=1

ENV PATH="/app/bin:$PATH"


FROM deploy-base AS test-deployment
RUN octopus --version
RUN octopus --config

WORKDIR /test-octopus/recipe
COPY <<EOF inp
CalculationMode = recipe
EOF

RUN octopus

WORKDIR /test-octopus/h-atom
COPY <<EOF inp
CalculationMode = gs

%Coordinates
'H' | 0 | 0 | 0
%

Spacing = 0.43463
Radius = 7.5589
EOF

RUN octopus

WORKDIR /test-octopus/he
COPY <<EOF inp
CalculationMode = gs

Dimensions = 2
TheoryLevel = independent_particles

BoxShape = parallelepiped
Lsize = 8
Spacing = 0.1

%Output
  wfs | plane_z
%

%Species
  "helium" | species_user_defined | potential_formula | "-2/(1+x^2)^(1/2)-2/(1+y^2)^(1/2)+1/(1+(x-y)^2)^(1/2)" | valence | 1
%

%Coordinates
  "helium"| 0 | 0
%
EOF
RUN octopus

WORKDIR /test-octopus/he-mpi
COPY <<EOF inp
CalculationMode = gs

Dimensions = 2
TheoryLevel = independent_particles

BoxShape = parallelepiped
Lsize = 8
Spacing = 0.1

%Output
  wfs | plane_z
%

%Species
  "helium" | species_user_defined | potential_formula | "-2/(1+x^2)^(1/2)-2/(1+y^2)^(1/2)+1/(1+(x-y)^2)^(1/2)" | valence | 1
%

%Coordinates
  "helium"| 0 | 0
%
EOF
RUN mpirun -np 2 octopus


FROM deploy-base AS deploy
WORKDIR /io
CMD ["octopus"]
