!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module space_oct_m
  use debug_oct_m
  use global_oct_m
  use messages_oct_m
  use namespace_oct_m
  use parser_oct_m
  use profiling_oct_m

  implicit none

  private

  public ::                   &
    space_t


  type space_t
    ! Components are public by default
    integer :: dim
    integer :: periodic_dim = 0

  contains
    procedure :: is_periodic => space_is_periodic
    procedure :: has_mixed_periodicity => space_has_mixed_periodicity
    procedure :: write_info => space_write_info
    procedure :: short_info => space_short_info
  end type space_t

  interface space_t
    procedure space_constructor
  end interface space_t

contains

  !----------------------------------------------------------
  function space_constructor(namespace) result(this)
    type(space_t)                  :: this
    type(namespace_t),  intent(in) :: namespace

    integer, parameter :: default_ndim = 3

    PUSH_SUB(space_constructor)

    !%Variable Dimensions
    !%Type integer
    !%Section System
    !%Default 3
    !%Description
    !% <tt>Octopus</tt> can run in 1, 2 or 3 or more dimensions, depending on
    !% the value of this variable. Note that not all input variables may be
    !% available in all cases.
    !%End
    call parse_variable(namespace, 'Dimensions', default_ndim, this%dim)
    if (this%dim < 1) call messages_input_error(namespace, 'Dimensions')

    !%Variable PeriodicDimensions
    !%Type integer
    !%Default 0
    !%Section System
    !%Description
    !% Define how many directions are to be considered periodic. It has to be a number
    !% between zero and <tt>Dimensions</tt>.
    !%Option 0
    !% No direction is periodic (molecule).
    !%Option 1
    !% The <i>x</i> direction is periodic.
    !%Option 2
    !% The <i>x</i> and <i>y</i> directions are periodic.
    !%Option 3
    !% The <i>x</i>, <i>y</i>, and <i>z</i> directions are periodic.
    !%End
    call parse_variable(namespace, 'PeriodicDimensions', 0, this%periodic_dim)

    if ((this%periodic_dim < 0) .or. (this%periodic_dim > this%dim)) then
      call messages_input_error(namespace, 'PeriodicDimensions')
    end if

    POP_SUB(space_constructor)
  end function space_constructor

  !--------------------------------------------------------------
  logical pure function space_is_periodic(this)
    class(space_t), intent(in) :: this

    space_is_periodic = this%periodic_dim > 0

  end function space_is_periodic

  !--------------------------------------------------------------
  logical pure function space_has_mixed_periodicity(this)
    class(space_t), intent(in) :: this

    space_has_mixed_periodicity = this%periodic_dim > 0 .and. this%periodic_dim < this%dim

  end function space_has_mixed_periodicity

  !--------------------------------------------------------------
  subroutine space_write_info(this, namespace)
    class(space_t),    intent(in) :: this
    type(namespace_t), intent(in) :: namespace

    PUSH_SUB(space_write_info)

    call messages_print_with_emphasis(msg="Space", namespace=namespace)

    write(message(1), '(a,i1,a)') 'Octopus will run in ', this%dim, ' dimension(s).'
    write(message(2), '(a,i1,a)') 'Octopus will treat the system as periodic in ', this%periodic_dim, ' dimension(s).'
    call messages_info(2, namespace=namespace)

    call messages_print_with_emphasis(namespace=namespace)

    POP_SUB(space_write_info)
  end subroutine space_write_info

  !--------------------------------------------------------------
  character(len=40) function space_short_info(this) result(info)
    class(space_t), intent(in) :: this

    PUSH_SUB(space_short_info)

    write(info, '(a,i1,a,i1)') 'Dimensions = ', this%dim, '; PeriodicDimensions = ', this%periodic_dim

    POP_SUB(space_short_info)
  end function space_short_info

end module space_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
