!! Copyright (C) 2024 A. Buccheri
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
#include "global.h"

module mpi_distribute_oct_m
  use debug_oct_m
  use global_oct_m
  use messages_oct_m
  use profiling_oct_m
  implicit none
  private

  public :: mpi_displacements

  interface mpi_displacements
    module procedure mpi_displacements_int32, mpi_displacements_int64
  end interface mpi_displacements

contains

  !> @brief Compute the displacements required for (all)gather(v) and (all)scatter(v).
  subroutine mpi_displacements_int32(recvcounts, displs)
    integer, intent(in ) :: recvcounts(:) !< The number of elements that are received from each process
    integer, intent(out) :: displs(:)     !< Entry i specifies the displacement (relative to recvbuf)
    !                                        at which to place the incoming data from process i-1
    integer :: n_processes, i

    PUSH_SUB(mpi_displacements)

    n_processes = size(displs)
    ASSERT(size(recvcounts) == n_processes)

    displs(1) = 0
    do i = 2, n_processes
      displs(i) = displs(i-1) + recvcounts(i-1)
    enddo

    POP_SUB(mpi_displacements)
  end subroutine mpi_displacements_int32


  !> @brief Compute the displacements required for (all)gather(v) and (all)scatter(v).
  subroutine mpi_displacements_int64(recvcounts, displs)
    integer,        intent(in ) :: recvcounts(:) !< The number of elements that are received from each process
    integer(int64), intent(out) :: displs(:)     !< Entry i specifies the displacement (relative to recvbuf)
    !                                               at which to place the incoming data from process i-1
    integer :: n_processes, i

    PUSH_SUB(mpi_displacements)

    n_processes = size(displs)
    ASSERT(size(recvcounts) == n_processes)

    displs(1) = 0
    do i = 2, n_processes
      displs(i) = displs(i-1) + recvcounts(i-1)
    enddo

    POP_SUB(mpi_displacements)
  end subroutine mpi_displacements_int64

end module mpi_distribute_oct_m
