!! Copyright (C) 2024 M. Lueders
!!
!! This Source Code Form is subject to the terms of the Mozilla Public
!! License, v. 2.0. If a copy of the MPL was not distributed with this
!! file, You can obtain one at https://mozilla.org/MPL/2.0/.
!!

#include "global.h"

!> @brief This modules implements the dipole moment of the matter system.
!!
!! @note currently, this is only implemented for isolated systems.
!! This module can later be extended with proper treatment for periodic systems
!
module dipole_oct_m
  use debug_oct_m
  use global_oct_m
  use ions_oct_m
  use mesh_oct_m
  use messages_oct_m
  use profiling_oct_m
  use space_oct_m
  use states_elec_oct_m
  implicit none

  private
  public :: &
    dipole_t

  type :: dipole_t
    private

    integer      :: dim                    !< dimension of the dipole moment
    logical      :: periodic               !< if periodic, we need to use more complex code, which is not yet implemented
    real(real64), allocatable :: dipole(:) !< the total dipole moment of electrons and ions

  contains
    procedure :: init => dipole_init           !< @copydoc dipole_moment
    procedure :: calculate => dipole_calculate !< @copydoc dipole_calculate
    procedure :: get => dipole_get             !< @copydoc dipole_get
    procedure :: end => dipole_end
  end type dipole_t

contains

  !> @brief initialize the dipole moment
  !!
  !! Set the flag whether the system is periodic and set the moment to zero
  subroutine dipole_init(this, space)
    class(dipole_t), intent(out) :: this
    class(space_t),  intent(in)  :: space

    this%periodic = space%is_periodic()
    this%dim = space%dim

    SAFE_ALLOCATE(this%dipole(1:this%dim))
    this%dipole = M_ZERO

  end subroutine dipole_init

  !> @brief finalizer: release memory
  subroutine dipole_end(this)
    class(dipole_t), intent(inout) :: this

    SAFE_DEALLOCATE_A(this%dipole)

  end subroutine dipole_end

  !> @brief Calculate the dipole moment from the ions and the electronic states
  !!
  subroutine dipole_calculate(this, gr, ions, st)
    class(dipole_t),     intent(inout) :: this
    class(mesh_t),       intent(in)    :: gr
    type(ions_t),        intent(in)    :: ions
    type(states_elec_t), intent(in)    :: st

    PUSH_SUB(dipole_calculate)

    this%dipole = - ions%dipole() - st%dipole(gr)

    POP_SUB(dipole_calculate)
  end subroutine dipole_calculate

  !> @brief accedd the dipole moment
  function dipole_get(this) result(dipole)
    class(dipole_t), intent(in) :: this
    real(real64)                :: dipole(1:this%dim)

    PUSH_SUB(dipole_get)
    ASSERT(.not. this%periodic) ! remove this assertion once properly implemented
    dipole = this%dipole

    POP_SUB(dipole_get)
  end function

end module dipole_oct_m
